<?php

namespace ForgeSmith;

use WP_Block_Type_Registry;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly
final class Foundry {

	use FoundryInjectable;

	public array  $queries    = [];
	private array $allOptions = [];

	private array $schema = FoundrySchema::GLOBAL_STYLES;

	/**
	 * Empty constructor.
	 */
	public function __construct() {
	}

	public function init(): void {
		$this->overrideDefaultsFromThemJSON();
		$this->registerSettings();
		$this->generateOptions();
		$this->setupCustomFields();
		$this->setupEndpoints();
		// don't generate aesthetic things for REST requests...
		if ( ! defined( 'REST_REQUEST' ) || ! REST_REQUEST ) {
			$this->generateCSSVars();
			$this->generateCSSClasses();
			$this->generateImportFonts();
			$this->setupAnimationSettings();
			$this->registerShortcodes();
		}
	}

	private function registerSettings(): void {
		register_setting(
			'foundry_plugin_settings_page',
			'fndry_global_styles',
			[
				'type'              => 'object',
				'sanitize_callback' => [ $this, 'sanitizeSchema__GLOBAL_STYLES' ],
				'default'           => [
					'colors'     => [
						'primary'   => [
							[
								'hex'     => '#ED145B',
								'name'    => 'Raspberry',
								'key'     => '--fndry-color-raspberry',
								'fndryId' => 'fcp1',
							],
							[
								'hex'     => '#31498C',
								'name'    => 'Marian Blue',
								'key'     => '--fndry-color-marianBlue',
								'fndryId' => 'fcp2',
							],
							[
								'hex'     => '#A7B739',
								'name'    => 'Apple Green',
								'key'     => '--fndry-color-appleGreen',
								'fndryId' => 'fcp3',
							],
						],
						'secondary' => [
							[
								'hex'     => '#ED145B80',
								'name'    => 'Transparent Raspberry',
								'key'     => '--fndry-color-transparentRaspberry',
								'fndryId' => 'fcs3',
							],
							[
								'hex'     => '#31498C80',
								'name'    => 'Transparent Blue',
								'key'     => '--fndry-color-transparentBlue',
								'fndryId' => 'fcs4',
							],
							[
								'hex'     => '#A7B73980',
								'name'    => 'Transparent Green',
								'key'     => '--fndry-color-transparentGreen',
								'fndryId' => 'fc5',
							],
						],
						'neutral'   => [
							[ 'hex' => '#000', 'name' => 'Black', 'key' => '--fndry-color-black', 'fndryId' => 'fcn6' ],
							[ 'hex' => '#fff', 'name' => 'White', 'key' => '--fndry-color-white', 'fndryId' => 'fcn7' ],
							[
								'hex'     => '#C5CECE',
								'name'    => 'Grey',
								'key'     => '--fndry-color-grey',
								'fndryId' => 'fcn8',
							],
						],
					],
					'fonts'      => [
						'googleFontsEnabled'     => true,
						'googleFontsImportValue' => [
							[
								'family'   => 'Roboto Mono',
								'import'   => 'Roboto+Mono:ital,wght@0,100;0,300;0,500;0,700;1,500;1,600',
								'fallback' => 'monospace',
							],
							[
								'family'   => 'Roboto',
								'import'   => 'Roboto:ital,wght@0,300;0,500;1,300;1,400;1,500',
								'fallback' => 'sans-serif',
							],
						],
						'adobeFontsEnabled'      => false,
						'adobeFontsImportValue'  => [],
						'fontAwesomeEnabled'     => false,
					],
					'typography' => [
						'bodyStyle'    => [
							'color'       => 'fcn6',
							'family'      => 'var(--fndry-font-robotoMono)',
							'minFontSize' => '1rem',
							'line'        => 1.5,
							'spacing'     => '0rem',
							'weight'      => 500,
						],
						'headerStyles' => [
							[
								'fndryId'       => 'fth1',
								'name'          => 'Page Title',
								'key'           => 'fndry-text-pageTitle',
								'color'         => 'fcp1',
								'setAsDefault'  => true,
								'tag'           => 'h1',
								'family'        => 'var(--fndry-font-robotoMono)',
								'minFontSize'   => '3rem',
								'maxFontSize'   => '5rem',
								'line'          => 1.5,
								'spacing'       => '0rem',
								'weight'        => 500,
								'textUppercase' => false,
								'italic'        => false,
								'isResponsive'  => true,
							],
							[
								'fndryId'       => 'fth2',
								'name'          => 'Primary Headline',
								'key'           => 'fndry-text-primaryHeadline',
								'color'         => 'fcp2',
								'setAsDefault'  => true,
								'tag'           => 'h2',
								'family'        => 'var(--fndry-font-roboto)',
								'minFontSize'   => '2rem',
								'maxFontSize'   => '4rem',
								'line'          => 1.5,
								'spacing'       => '0rem',
								'weight'        => 600,
								'textUppercase' => false,
								'italic'        => false,
								'isResponsive'  => true,
							],
							[
								'fndryId'       => 'fth3',
								'name'          => 'Label',
								'key'           => 'fndry-text-label',
								'color'         => 'fcp3',
								'setAsDefault'  => true,
								'family'        => 'var(--fndry-font-robotoMono)',
								'minFontSize'   => '0.875rem',
								'line'          => 1.25,
								'spacing'       => '0rem',
								'kern'          => '1px',
								'weight'        => 900,
								'textUppercase' => true,
								'italic'        => false,
								'isResponsive'  => false,
							],
						],
						// because letter-spacing isn't included in the font shorthand, these should probably just
						// be generated as text utility classes in the head.
						'textStyles'   => [
							[
								'fndryId'       => 'ftt1',
								'name'          => 'Intro Copy',
								'key'           => 'fndry-text-introCopy', // -> .fndry-text prefix for utility classes
								'color'         => 'fcp2',
								'setAsDefault'  => true,
								'family'        => 'var(--fndry-font-robotoMono)',
								'minFontSize'   => '1.5rem',
								'isResponsive'  => false,
								'line'          => 1.5,
								'spacing'       => '0rem',
								'kern'          => '',
								'weight'        => 100,
								'textUppercase' => false,
								'italic'        => false,
							],
							[
								'fndryId'       => 'ftt2',
								'name'          => 'Legal',
								'key'           => 'fndry-text-legal',
								'color'         => 'fcn6',
								'setAsDefault'  => false,
								'family'        => 'var(--fndry-font-robotoMono)',
								'minFontSize'   => '1rem',
								'line'          => 1.25,
								'spacing'       => '0rem',
								'kern'          => '',
								'weight'        => 600,
								'textUppercase' => false,
								'italic'        => false,
								'isResponsive'  => false,
							],
						],
					],
					'shadows'    => [
						'boxShadows' => [
							[
								'fndryId' => 'fbs1',
								'name'    => 'Standard',
								'key'     => '--fndry-bshadow-standard',
								'color'   => 'fcs4',
								'x'       => '0',
								'y'       => '0.25em',
								'blur'    => '0.25em',
								'spread'  => '0',
								'inset'   => false,
							],
						],
					],
					'forms'      => [
						'labelStyle'           => 'ftt2',
						'textStyle'            => '', // empty means no style (i.e. use body cascade)
						'fieldBackgroundColor' => 'fcn7',
						'strokeColor'          => 'fcn2',
						'strokeWeight'         => '1px',
						'borderRadius'         => '1px',
						'buttonStyle'          => 'fb1',
					],
				],
				'show_in_rest'      => [
					'schema' => FoundrySchema::GLOBAL_STYLES,
				],
			]
		);

		register_setting(
			'foundry_plugin_settings_page',
			'fndry_custom_fields',
			[
				'type'         => 'object',
				//			'sanitize_callback' => [ $this, 'sanitizeSchema__CUSTOM_FIELDS' ],
				'show_in_rest' => [
					'schema' => FoundrySchema::CUSTOM_FIELDS,
				],
			]
		);

		register_setting(
			'foundry_plugin_settings_page',
			'fndry_animation_settings',
			[
				'type'         => 'object',
				//			'sanitize_callback' => [ $this, 'sanitizeSchema__ANIMATION' ],
				'show_in_rest' => [
					'schema' => FoundrySchema::ANIMATION,
				],
			]
		);

		register_setting(
			'foundry_plugin_settings_page',
			'fndry_general_settings',
			[
				'type'         => 'object',
				//			'sanitize_callback' => [ $this, 'sanitizeSchema__GENERAL' ],
				'default'      => [
					[
						'maps'        => [
							'googleMapsEnabled' => false,
							'mapsApiKey'        => '',
							'mapLocations'      => [
								[
									'fndryId' => 'fm1',
									'name'    => 'Main Location',
									'key'     => 'fndry-map-primary',
									'address' => '',
								],
							],
						],
						'gridSpacing' => [
							'gutter' => '1rem',
						],
					],
				],
				'show_in_rest' => [
					'schema' => FoundrySchema::GENERAL,
				],
			]
		);
		register_setting(
			'foundry_plugin_settings_page',
			'fndry_component_styles',
			[
				'type'              => 'object',
				'sanitize_callback' => [ $this, 'sanitizeSchema__COMPONENT_STYLES' ],
				'default'           => [
					'defaultLinkStyle' => [
						'hasUnderline' => false,
						'color'        => 'fcp2',
						'hoverColor'   => 'fcp1',
					],
					'buttons'          => [
						'buttonStyles' => [
							[
								// splice out "button" from the name when generating key
								'fndryId'         => 'fb1',
								'key'             => 'fndry-btn-primary',
								'name'            => 'Primary Button',
								'backgroundColor' => 'fcp2',
								'strokeColor'     => '', // if values are "0px", then save no value.
								'strokeWeight'    => '0',
								'borderRadius'    => '0.25rem',
								'btnPadding'      => [
									'top'    => '1em',
									'right'  => '1.5em',
									'bottom' => '1em',
									'left'   => '1.5em',
								],
								'fontFamily'      => 'var(--fndry-font-roboto)',
								'fontColor'       => 'fcn7',
								'fontSize'        => '1rem',
								'kern'            => '0.125em',
								'fontWeight'      => 400,
								'textUppercase'   => true,
								'italic'          => false,
								'setAsDefault'    => true,
								'hasIcon'         => false,
								'icon'            => [
									'type'     => 'custom',
									'id'       => 0,
									'faSize'   => '20px',
									'position' => 'right',
									'padding'  => '15px',
									'color'    => '',
								],
								'boxShadow'       => 'fbs1',
								'hasUnderline'    => false,
								'padding'         => [
									'x' => '1em',
									'y' => '0.5em',
								],
							],
							[
								'fndryId'         => 'fb2',
								'key'             => 'fndry-btn-secondary',
								'name'            => 'Secondary Button',
								'backgroundColor' => '',
								'strokeColor'     => '', // if values are "0px", then save no value.
								'strokeWeight'    => '',
								'borderRadius'    => '',
								'btnPadding'      => [
									'top'    => '1em',
									'right'  => '1.5em',
									'bottom' => '1em',
									'left'   => '1.5em',
								],
								'fontFamily'      => 'var(--fndry-font-roboto)',
								'fontColor'       => 'fcp1',
								'fontSize'        => '1rem',
								'kern'            => '0.0875em',
								'fontWeight'      => 700,
								'textUppercase'   => true,
								'italic'          => false,
								'setAsDefault'    => false,
								'hasIcon'         => false,
								'icon'            => [
									'type'     => 'custom',
									'id'       => 0,
									'faSize'   => '20px',
									'position' => 'right',
									'padding'  => '15px',
									'color'    => '',
								],
								'hasUnderline'    => false,
								'underline'       => [],
								'padding'         => [
									'x' => '1em',
									'y' => '0.5em',
								],
							],
						],
					],
				],
				'show_in_rest'      => [
					'schema' => FoundrySchema::COMPONENT_STYLES,
				],
			]
		);
	}

	private function generateOptions() {
		$general           = get_option( 'fndry_general_settings' ) ?: [];
		$globalStyles      = get_option( 'fndry_global_styles' ) ?: [];
		$compStyles        = get_option( 'fndry_component_styles' ) ?: [];
		$customFields      = get_option( 'fndry_custom_fields' ) ?: [];
		$animationSettings = get_option( 'fndry_animation_settings' ) ?: [];

		$this->allOptions = array_merge( $general,
			$globalStyles,
			$compStyles,
			$customFields,
			$animationSettings ) ?: [];
	}

	private function generateCSSVars() {
		$html = [];

		if ( isset( $this->allOptions['gridSettings'] ) ) {
			foreach ( $this->allOptions['gridSettings'] as $spacing ) {
				$html[] = "--fndry-gutter:$spacing;";
			}
		}
		if ( isset( $this->allOptions['colors'] ) ) {
			foreach ( $this->allOptions['colors'] as $palette ) {
				$colors = array_map( function ( $item ) {
					if ( empty( $item['key'] ) || empty( $item['hex'] ) ) {
						return null;
					}

					return "{$item['key']}:{$item['hex']}";
				}, $palette );
				array_push( $html, ...$colors );
			}
		}
		if ( isset( $this->allOptions['shadows']['boxShadows'] ) ) {
			foreach ( $this->allOptions['shadows']['boxShadows'] as $shadow ) {
				if ( ! is_array( $shadow ) || ! isset( $shadow['key'] ) ) {
					continue;
				}
				$tmp = [
					isset( $shadow['inset'] ) && $shadow['inset'] ? 'inset' : '',
					$shadow['x'] ?? '0',
					$shadow['y'] ?? '0',
					$shadow['blur'] ?? '0',
					$shadow['spread'] ?? '0',
					isset( $shadow['color'] ) && $shadow['color'] ? $this->maybeGetSettingKey( $shadow['color'] ) : '',
				];
				$str = join( ' ', $tmp );
				array_push( $html, "{$shadow['key']}:$str" );
			}
		}
		$duration = $this->allOptions['buttons']['transitionDuration'] ?? '125';
		$html[]   = "--fndry-global-transition-duration:{$duration}ms;";
		$delay    = $this->allOptions['buttons']['transitionDelay'] ?? '0';
		$html[]   = "--fndry-global-transition-delay:{$delay}ms;";

		$timingFunction = 'ease';
		if ( ! empty( $this->allOptions['buttons']['customTransitionTimingFunction'] ) ) {
			$timingFunction = $this->allOptions['buttons']['customTransitionTimingFunction'];
		} elseif ( ! empty( $this->allOptions['buttons']['transitionTimingFunction'] ) ) {
			$timingFunction = $this->allOptions['buttons']['transitionTimingFunction'];
		}

		$html[] = "--fndry-global-transition-timing-function:$timingFunction;";

		$this->injectInlineStyles( ":root{" . implode( ';', $html ) . "}" );
	}

	public function maybeGetSettingKey( $fieldName, $block = false ) {
		// check if we need to get by fndryID
		if ( ! $block ) {
			$attrVal = $fieldName;
		} else {
			$attrVal = $block->getAttribute( $fieldName );
		}

		// first condition is internal usage for generating inline css.
		if ( ! ! $fieldName && ! $block || ( isset( $block->block_type->attributes[ $fieldName ] ) && isset( $block->block_type->attributes[ $fieldName ]['isFndryId'] ) ) ) {
			$val = $this->getOptionById( $attrVal );
			if ( isset( $val['key'] ) ) {
				$key = $val['key'];
				// check if we need to format
				if ( strpos( $key, '--' ) === 0 ) {
					$attrVal = "var($key)";
				} else {
					// otherwise it's a key.
					$attrVal = $key;
				}
			}
		}

		return $attrVal;
	}

	public function getOptionById( $needle, $haystack = null ) {
		if ( $haystack === null ) {
			$haystack = $this->getOptions();
		}

		return fndry_recursive_search( $needle, 'fndryId', $haystack );
	}

	public function getOptions(): array {
		return $this->allOptions;
	}

	private function injectInlineStyles( $styles ) {
		add_action( 'enqueue_block_assets', function () use ( $styles ) {
			wp_add_inline_style( 'fndry-style', $styles );
		}, 99 );
		add_action( 'admin_enqueue_scripts', function () use ( $styles ) {
			wp_add_inline_style( 'fndry-settings', $styles );
		}, 99 );
	}

	private function generateCSSClasses() {
		global $pagenow;

		if ( is_admin() && ! in_array( $pagenow, [ 'post.php', 'site-editor.php', 'admin.php', 'post-new.php' ] ) ) {
			return;
		}

		$html = [];

		$editorPrefix = '';

		// differentiate between post editor view and settings page, as we dont want everything in settings page to get styled with customized css.
		is_admin() && in_array( $pagenow,
			[ 'post.php', 'site-editor.php', 'post-new.php' ] ) and $editorPrefix = '.editor-styles-wrapper';
		is_admin() && $pagenow === 'admin.php' and $editorPrefix = '.fndry-preview-wrapper';
		if ( isset( $this->allOptions['typography'] ) ) {
			$type = $this->allOptions['typography'];

			// do body style
			$bodyStyle = $type['bodyStyle'];
			if ( $bodyStyle ) {
				$tmp    = $this->doTypeProps( $bodyStyle, true );
				$tag    = is_admin() ? $editorPrefix : 'body';
				$html[] = "$tag{" . implode( "", array_filter( $tmp ) ) . "}";
				$html[] = $this->addMediaQuery( $bodyStyle, $tag );
			}

			$headerStyles = $type['headerStyles'] ?? [];
			$textStyles   = $type['textStyles'] ?? [];
			if ( $headerStyles || $textStyles ) {
				foreach ( [ ...$headerStyles, ...$textStyles ] as $style ) {
					$tmp = $this->doTypeProps( $style );
					//ISSET WILL ALWAYS BE FALSE CAUSE THERE'S NO setAsDefault AND tag PROPERTIES
					$defaultTag = isset( $style['setAsDefault'] ) && ! ! $style['setAsDefault'] && isset( $style['tag'] ) && ! ! $style['tag'] ? ",$editorPrefix {$style['tag']}:not([class*='fndry-text'])" : '';
					$selector   = "$editorPrefix .{$style['key']}$defaultTag";
					$html[]     = "$selector{" . implode( "", array_filter( $tmp ) ) . "}";
					$html[]     = $this->addMediaQuery( $style, $selector );
				}
			}
		}

		$defaultLinkStyle = $this->allOptions['defaultLinkStyle'] ?? false;

		if ( $defaultLinkStyle ) {
			$tmp = [];

			$tmp[] = "color:var(--linkColor, var(--color, " . ( isset( $defaultLinkStyle['color'] ) && $defaultLinkStyle['color'] ? $this->maybeGetSettingKey( $defaultLinkStyle['color'] ) : 'currentColor' ) . '))';
			isset( $defaultLinkStyle['fontFamily'] ) and $defaultLinkStyle['fontFamily'] and $tmp[] = "font-family: {$defaultLinkStyle['fontFamily']}";

			if ( isset( $defaultLinkStyle['hasUnderline'] ) && $defaultLinkStyle['hasUnderline'] ) {
				// if defaultLinkStyle has an underline, set it here. it can be overridden elsewhere.
				$tmp[]     = "text-decoration: var(--linkTextDecoration, underline)";
				$underline = $defaultLinkStyle['underline'] ?? false;
				if ( $underline ) {
					isset( $underline['strokeWeight'] ) and $underline['strokeWeight'] and $tmp[] = "text-decoration-thickness:" . $this->maybeGetSettingKey( $underline['strokeWeight'] );
					isset( $underline['offset'] ) and $underline['offset'] and $tmp[] = "text-underline-offset:" . $this->maybeGetSettingKey( $underline['offset'] );
				}
			} else {
				// otherwise no underline
				$tmp[] = "text-decoration: var(--linkTextDecoration, none)";
			}

			// setup for inline link stuff handled from header/text styles
			$tmp[] = 'background-position:0 100%';
			$tmp[] = 'background-repeat:no-repeat';
			$tmp[] = 'background-size:var(--linkBackgroundSize)';
			$tmp[] = 'background-image:var(--linkBackground)';
			$tmp[] = 'text-decoration-color:var(--linkUnderlineColor, currentColor)';

			! empty( $tmp ) and $html[] = "$editorPrefix a:not([class]){transition:var(--linkTransition, color 0.125s ease-in-out, text-decoration-color 0.125s ease-in-out);" . implode( ";",
					array_filter( $tmp ) ) . "}";

			// hovers
			$hoversTmp = [];
			isset( $defaultLinkStyle['hoverColor'] ) and $defaultLinkStyle['hoverColor'] and $hoversTmp[] = "color:var(--linkHoverColor, " . $this->maybeGetSettingKey( $defaultLinkStyle['hoverColor'] ) . ")";
			$hoversTmp[] = 'background-size:var(--linkHoverBackgroundSize)';
			$hoversTmp[] = 'text-decoration-color:var(--linkHoverUnderlineColor, currentColor)';

			isset( $defaultLinkStyle['hoverColor'] ) and $defaultLinkStyle['hoverColor'] and $html[] = "$editorPrefix a:not([class]):hover{" . implode( ';',
					array_filter( $hoversTmp ) ) . "}";
			isset( $defaultLinkStyle['activeColor'] ) and $defaultLinkStyle['activeColor'] and $html[] = "$editorPrefix a:not([class]):active{color:" . $this->maybeGetSettingKey( $defaultLinkStyle['activeColor'] ) . "}";
			isset( $defaultLinkStyle['visitedColor'] ) and $defaultLinkStyle['visitedColor'] and $html[] = "$editorPrefix a:not([class]):visited{color:" . $this->maybeGetSettingKey( $defaultLinkStyle['visitedColor'] ) . "}";
		}

		// btn and link should be condensed...
		$buttons = $this->allOptions['buttons']['buttonStyles'] ?? false;

		if ( $buttons ) {
			foreach ( $buttons as $btn ) {
				if ( ! is_array( $btn ) ) {
					continue;
				}

				$tmp = $this->doButtonProps( $btn );

				$defaultTag = isset( $btn['setAsDefault'] ) && $btn['setAsDefault'] ? ",$editorPrefix .fndry-btn-default" : '';

				$iconTmp = $this->doButtonIconProps( $btn );

				$defaultTransitions = $btn['transitions'] ?? false;

				if ( $defaultTransitions ) {
					$tmp[] = 'transition:' . join( ',', $this->generateCSSTransitionProps( $defaultTransitions ) );
				}

				isset( $btn['key'] ) and $html[] = "$editorPrefix .{$btn['key']}$defaultTag{" . implode( "",
						array_filter( $tmp ) ) . "}";
				if ( ! empty( $btn['hasIcon'] ) && ! empty( $btn['icon']['type'] ) && $btn['icon']['type'] === 'custom' ) {
					$defaultPseudo = isset( $btn['setAsDefault'] ) && $btn['setAsDefault'] ? ",$editorPrefix .fndry-btn-default::after" : '';
					! empty( $btn['key'] ) && $html[] = "$editorPrefix .{$btn['key']}::after$defaultPseudo{" . implode( "",
							array_filter( $iconTmp ) ) . "}";
				}

				$hoverStyle = $btn['hoverStyle'] ?? false;
				if ( $hoverStyle ) {
					$hoverTmp = $this->doButtonProps( $hoverStyle );
					// handle transitions
					if ( ! empty( $hoverStyle['transitions'] ) ) {
						$hoverTmp[] = 'transition:' . join( ',',
								$this->generateCSSTransitionProps( $hoverStyle['transitions'] ) );
					}
					$defaultHoverTag = $defaultTag ? "$editorPrefix$defaultTag:hover" : null;
					isset( $btn['key'] ) and $html[] = "$editorPrefix .{$btn['key']}:hover$defaultHoverTag{" . implode( "",
							array_filter( $hoverTmp ) ) . "}";
				}
			}
		}

		$links = $this->allOptions['links']['linkStyles'] ?? false;
		if ( $links ) {
			foreach ( $links as $link ) {
				if ( ! is_array( $link ) ) {
					continue;
				}
				$tmp = $this->doButtonProps( $link );

				$iconTmp = $this->doButtonIconProps( $link );

				$defaultTag    = isset( $link['setAsDefault'] ) ? ",$editorPrefix .fndry-link-default" : '';
				$defaultPseudo = isset( $btn['setAsDefault'] ) && $btn['setAsDefault'] ? ",$editorPrefix .fndry-link-default::after" : '';

				isset( $link['key'] ) and $html[] = "$editorPrefix .{$link['key']}$defaultTag{" . implode( "",
						array_filter( $tmp ) ) . "}";
				isset( $link['key'] ) and $link['hasIcon'] && $iconTmp && $html[] = "$editorPrefix .{$link['key']}::after$defaultPseudo{" . implode( "",
						array_filter( $iconTmp ) ) . "}";
			}
		}

		$this->injectInlineStyles( implode( '', $html ) );
	}

	private function doTypeProps( $prop, $isBody = false ) {
		if ( ! is_array( $prop ) ) {
			return [];
		}
		$tmp = [];
		// maybe the props should have all been the actual CSS property, but camelcase.
		// then we could have just foreach'd through em and converted em to kebab-case
		isset( $prop['color'] ) and $prop['color'] and $tmp[] = "color:var(--color, " . $this->maybeGetSettingKey( $prop['color'] ) . ");";
		isset( $prop['family'] ) and $prop['family'] and $tmp[] = "font-family:{$prop['family']};";
		isset( $prop['line'] ) and $prop['line'] and $tmp[] = "line-height:{$prop['line']};";
		isset( $prop['kern'] ) and $prop['kern'] and $tmp[] = "letter-spacing:{$prop['kern']};";
		isset( $prop['weight'] ) and $prop['weight'] and $tmp[] = "font-weight:{$prop['weight']};";
		isset( $prop['minFontSize'] ) and $prop['minFontSize'] and $tmp[] = "font-size:{$prop['minFontSize']};";
		isset( $prop['spacing'] ) and $prop['spacing'] and $tmp[] = $isBody ? "--fndry-base-p-spacing:{$prop['spacing']};" : "margin-bottom:{$prop['spacing']};";
		isset( $prop['textUppercase'] ) and $prop['textUppercase'] and $tmp[] = "text-transform:uppercase;";
		isset( $prop['italic'] ) and $prop['italic'] and $tmp[] = "font-style:italic;";

		if ( isset( $prop['isResponsive'] ) &&
		     $prop['isResponsive'] === true &&
		     $prop['minFontSize'] !== null &&
		     $prop['maxFontSize'] !== null ) {
			$min = $this->convertUnitsToPixels( $prop['minFontSize'] );
			$max = $this->convertUnitsToPixels( $prop['maxFontSize'] );
			//TODO this is hardcoded at 992 for now, but will need to become dynamic in the future i assume
			$vw = ( ( $max - $min ) / 992 ) * 100;
			isset( $prop['maxFontSize'] ) and $tmp[] = "font-size:calc({$prop['minFontSize']} + {$vw}vw);";
		}
		isset( $prop['linkColor'] ) and $prop['linkColor'] and $tmp[] = "--linkColor:" . $this->maybeGetSettingKey( $prop['linkColor'] ) . ";";
		isset( $prop['linkHoverColor'] ) and $prop['linkHoverColor'] and $tmp[] = "--linkHoverColor:" . $this->maybeGetSettingKey( $prop['linkHoverColor'] ) . ";";
		if ( isset( $prop['linkUnderline'] ) ) {
			isset( $prop['linkUnderlineWidth'] ) and $prop['linkUnderlineWidth'] and $tmp[] = "--linkUnderlineWidth:{$prop['linkUnderlineWidth']};";
			isset( $prop['linkUnderlineColor'] ) and $prop['linkUnderlineColor'] and $tmp[] = "--linkUnderlineColor:" . $this->maybeGetSettingKey( $prop['linkUnderlineColor'] ) . ";";
			isset( $prop['linkHoverUnderlineColor'] ) and $prop['linkHoverUnderlineColor'] and $tmp[] = "--linkHoverUnderlineColor:" . $this->maybeGetSettingKey( $prop['linkHoverUnderlineColor'] ) . ";";
			if ( isset( $prop['linkUnderlineAnimation'] ) and $prop['linkUnderlineAnimation'] ) {
				$tmp[] = "--linkBackground:linear-gradient(to right, var(--linkHoverUnderlineColor, currentColor), var(--linkHoverUnderlineColor, currentColor));";
				$tmp[] = "--linkBackgroundSize:0 var(--linkUnderlineWidth);";
				$tmp[] = "--linkTransition:color 0.25s, background-size 0.25s;";
				$tmp[] = "--linkHoverBackgroundSize:100% var(--linkUnderlineWidth);";
				$tmp[] = '--linkTextDecoration:none';
			} else {
				$tmp[] = '--linkTextDecoration:underline';
			}
		}

		return $tmp;
	}

	private function convertUnitsToPixels( $val ) {
		// convert to px value
		if ( strpos( $val, 'rem' ) ) {
			$unit = str_replace( 'rem', '', $val );

			return (int) $unit * 16;
		} elseif ( strpos( $val, 'px' ) ) {
			$unit = str_replace( 'px', '', $val );

			return (int) $unit;
		}
	}

	private function addMediaQuery( $prop, $tag ) {
		if ( ( ! isset( $prop['isResponsive'] ) || $prop['isResponsive'] === false ) || ! isset( $prop['maxFontSize'] ) || $prop['maxFontSize'] === null ) {
			return false;
		}

		//TODO this is hardcoded at 992 for now, but will need to become dynamic in the future i assume
		return "@media only screen and (min-width: 992px){ $tag{ font-size:{$prop['maxFontSize']}}}";
	}

	private function doButtonProps( $prop ) {
		if ( ! is_array( $prop ) ) {
			return [];
		}
		$tmp = [];
		// maybe the props should have all been the actual CSS property, but camelcase.
		// then we could have just foreach'd through em and converted em to kebab-case

		isset( $prop['backgroundColor'] ) && $prop['backgroundColor'] and $tmp[] = "background-color:" . $this->maybeGetSettingKey( $prop['backgroundColor'] ) . ";";

		isset( $prop['strokeColor'] ) && $prop['strokeColor'] and $tmp[] = "border-color:" . $this->maybeGetSettingKey( $prop['strokeColor'] ) . ";";

		if ( isset( $prop['strokeWeight'] ) ) {
			$prop['strokeWeight'] and $tmp[] = "border-width:{$prop['strokeWeight']};";
		}

		if ( isset( $prop['borderRadius'] ) ) {
			$prop['borderRadius'] and $tmp[] = "border-radius:{$prop['borderRadius']};";
		}

		if ( isset( $prop['btnPadding'] ) ) {
			$btnPadding = $prop['btnPadding'];
			$top        = $btnPadding['top'] ?? 0;
			$right      = $btnPadding['right'] ?? 0;
			$bottom     = $btnPadding['bottom'] ?? 0;
			$left       = $btnPadding['left'] ?? 0;

			$prop['btnPadding'] and $tmp[] = "padding:$top $right $bottom $left;";
		}

		isset( $prop['fontFamily'] ) && $prop['fontFamily'] and $tmp[] = "font-family:{$prop['fontFamily']};";
		isset( $prop['fontColor'] ) && $prop['fontColor'] and $tmp[] = "color:" . $this->maybeGetSettingKey( $prop['fontColor'] ) . ";";
		isset( $prop['fontSize'] ) && $prop['fontSize'] and $tmp[] = "font-size:{$prop['fontSize']};";
		isset( $prop['kern'] ) && $prop['kern'] and $tmp[] = "letter-spacing:{$prop['kern']};";
		isset( $prop['fontWeight'] ) && $prop['fontWeight'] and $tmp[] = "font-weight:{$prop['fontWeight']};";
		isset( $prop['textUppercase'] ) && $prop['textUppercase'] and $tmp[] = "text-transform:uppercase;";
		isset( $prop['italic'] ) && $prop['italic'] and $tmp[] = "font-style:italic;";

		if ( isset( $prop['hasIcon'] ) ) {
			isset( $prop['icon']['color'] ) && $prop['icon']['color'] and $tmp[] = "--fndry-btn-icon-color:" . $this->maybeGetSettingKey( $prop['icon']['color'] ) . ";";
			isset( $prop['hoverStyle']['iconColor'] ) && $prop['hoverStyle']['iconColor'] and $tmp[] = "--fndry-btn-icon-color-hover:" . $this->maybeGetSettingKey( $prop['hoverStyle']['iconColor'] ) . ";";
			isset( $prop['icon']['faSize'] ) && $prop['icon']['faSize'] and $tmp[] = "--fndry-btn-icon-size:{$prop['icon']['faSize']};";
			$padding = $prop['icon']['gap'] ?? $prop['icon']['padding'] ?? '';
			$padding and $tmp[] = "gap:{$padding};"; // todo - ensure default values are provided.
		}
		$underline = $prop['underline'] ?? false;

		isset( $prop['boxShadow'] ) && $prop['boxShadow'] && is_string( $prop['boxShadow'] ) and $tmp[] = "box-shadow:" . $this->maybeGetSettingKey( $prop['boxShadow'] ) . ";";

		isset( $prop['hasUnderline'] ) && $prop['hasUnderline'] and $underline and $tmp[] = "text-decoration:underline;text-decoration-color:" . ( ! empty( $underline['strokeColor'] ) ? $this->maybeGetSettingKey( $underline['strokeColor'] ) : 'currentColor' ) . ";text-decoration-thickness:{$underline['strokeWeight']};text-underline-offset:{$underline['offset']};";

		return $tmp;
	}

	private function doButtonIconProps( $btn ) {
		if ( empty( $btn['hasIcon'] ) || empty( $btn['icon'] ) ) {
			return [];
		}

		$icon = $btn['icon'];

		$iconTmp = [];

		if ( ( empty( $icon['type'] ) || $icon['type'] === 'custom' ) && ! empty( $icon['id'] ) ) {
			$img = wp_get_attachment_image_src( $icon['id'], 'thumbnail' );

			if ( $img !== false && isset( $icon['faSize'] ) ) {
				$iconTmp[] = "content:'';background-image:url({$img[0]});height:{$icon['faSize']};width:{$icon['faSize']};";
			}
		}

		if ( isset( $icon['position'] ) ) {
			$flexOrder = $icon['position'] === 'left' ? 'order:-1;' : '';
			$iconTmp[] = "{$icon['position']}:0;{$flexOrder}";
		}

		return $iconTmp;
	}

	private function generateCSSTransitionProps( $transitions ) {
		$transTemp = [];
		foreach ( $transitions as $transition ) {
			$props = $transition["transitionProperties"] ?? false;
			if ( ! $props ) {
				continue;
			}

			$dur    = ! empty( $transition['transitionDuration'] ) && (int) $transition['transitionDuration'] ? $transition['transitionDuration'] . 'ms' : '125ms';
			$delay  = ! empty( $transition['transitionDelay'] ) && (int) $transition['transitionDelay'] ? $transition['transitionDelay'] . 'ms' : '0ms';
			$timing = $transition['transitionTimingFunction'] ?? 'ease';
			foreach ( $props as $prop ) {
				$transTemp[] = "$prop $dur $delay $timing";
			}
		}

		return array_filter( $transTemp );
	}

	private function generateImportFonts() {
		if ( isset( $this->allOptions['fonts'] ) ) {
			$fonts = $this->allOptions['fonts'];

			if ( ! is_array( $fonts ) ) {
				return false;
			}

			if ( isset( $fonts['adobeFontsEnabled'] ) && isset( $fonts['adobeFontsProjectId'] ) && isset( $fonts['adobeFontsApiKey'] ) && is_array( $aFont = $fonts['adobeFontsImportValue'] ) ) {
				$vars = [];
				// we're not providing the user a means to customize fonts from the admin with typekit,
				// so we only care about the raw saved values.
				// 'family' => nicename, for display.
				// 'css_stack' => typekit provides us the proper css font-family property value, no need to dig through classifications
				foreach ( $aFont as $font ) {
					$camel  = $this->camelCase( $font['family'] );
					$vars[] = "--fndry-font-$camel:{$font['css_stack']};";
				}
				$import_url = 'https://use.typekit.net/' . $fonts['adobeFontsProjectId'] . '.css';
				$this->enqueueFonts( 'fndryAdobeFonts', $import_url, $vars );
			}

			if ( $fonts['googleFontsEnabled'] ) {
				$gFont = $fonts['googleFontsImportValue'];

				$imports = [];
				$vars    = [];
				foreach ( $gFont as $font ) {
					if ( empty( $font['import'] ) ) {
						continue;
					}
					$camel       = $this->camelCase( $font['family'] );
					$fallback    = $font['fallback'] ?? 'sans-serif';
					$imports[]   = 'family=' . $font['import'];
					$fallbackVar = in_array( $fallback,
						[ 'sans-serif', 'serif', 'monospace' ] ) ? "var(--fndry-$fallback)" : $fallback;
					$vars[]      = "--fndry-font-$camel:'{$font['family']}', $fallbackVar;";
				}

				$imports = implode( '&', array_filter( $imports ) );

				$import_url = 'https://fonts.googleapis.com/css2?' . $imports . '&display=swap';

				$this->enqueueFonts( 'fndryGoogleFonts', $import_url, $vars );
			}

			if ( $fonts['fontAwesomeEnabled'] ) {
				// do FA
			}
		}
	}

	public function camelCase( $string ) {
		if ( ! $string ) {
			return null;
		}
		$arr    = preg_split( "/[\s,-]+/", ucwords( strtolower( $string ), " \t\r\n\f\v'-_" ) );
		$arr[0] = strtolower( $arr[0] );

		return implode( '', $arr );
	}

	/**
	 * Enqueue versions are NULL because they're Google Font imports.
	 * see: https://core.trac.wordpress.org/ticket/49742
	 *
	 * @param string $handle
	 * @param string $import_url
	 * @param array  $vars
	 *
	 * @return void
	 */
	private function enqueueFonts( string $handle, string $import_url, array $vars ) {
		add_action( 'enqueue_block_assets', function () use ( $import_url, $vars, $handle ) {
			wp_enqueue_style(
				$handle,
				$import_url,
				[],
				null // phpcs:ignore   WordPress.WP.EnqueuedResourceParameters.MissingVersion
			);
			$this->injectInlineStyles( ":root{" . implode( '', $vars ) . "}" );
		} );
		add_action( 'admin_enqueue_scripts', function () use ( $import_url, $vars, $handle ) {
			wp_enqueue_style(
				$handle,
				$import_url,
				[],
				null // phpcs:ignore   WordPress.WP.EnqueuedResourceParameters.MissingVersion
			);
			$this->injectInlineStyles( ":root{" . implode( '', $vars ) . "}" );
		} );
	}

	private function setupCustomFields() {
		// Get our list of custom fields from settings
		if ( isset( $this->allOptions['manageCustomFields'] ) && count( $this->allOptions['manageCustomFields']['customFields'] ) > 0 ) {
			// Add the meta boxes
			add_action( 'add_meta_boxes', [ $this, 'setupCustomFieldsMetaBox' ] );

			// Add the save routines
			add_action( 'save_post', [ $this, 'saveCustomFieldsPostData' ] );

			// Add the API allowance
			add_action( 'rest_api_init', [ $this, 'customFieldRestAPI' ] );
		}
	}

	private function setupEndpoints() {
		add_action( 'rest_api_init', function () {
			/* register our locations route */
			register_rest_route( 'foundry/v1', 'typekit', [
				'methods'             => 'GET',
				'callback'            => [ $this, 'fetchAdobe' ],
				/* public */
				'permission_callback' => 'is_user_logged_in',
			] );
			register_rest_route( 'foundry/v1', 'post-types', [
				'methods'             => 'GET',
				'callback'            => [ $this, 'fetchPostTypes' ],
				/* public */
				'permission_callback' => 'is_user_logged_in',
			] );
			register_rest_route( 'foundry/v1', '/settings(?:/(?P<key>[a-zA-Z0-9_-]+))?', [
				'methods'             => 'GET',
				'callback'            => [ $this, 'getSettingsOption' ],
				/* public */
				'permission_callback' => 'is_user_logged_in',
				'args'                => [
					'key' => [
						'type'     => 'string',
						'required' => false,
					],
				],
			] );
		} );
	}

	private function setupAnimationSettings() {
		$parallax = $this->getOption( 'parallaxEnabled' );
		if ( $parallax ) {
			add_filter( 'body_class', function ( $classes ) {
				return array_merge( $classes, [ 'locomotive-enabled' ] );
			} );
		}
	}

	private function registerShortcodes() {
		add_shortcode( 'fndry-date', function ( $args ) {
			if ( ! empty( $args['format'] ) ) {
				$date = new \DateTime( 'now' );
				$date = $date->format( $args['format'] );
				if ( $date ) {
					return $date;
				}
			}

			return null;
		} );
	}

	// schema was a little too strict.

	public function getOption( $key ) {
		return fndry_recursive_search( false, $key, $this->getOptions() );
	}

	public function getSettingsOption( WP_REST_Request $request ) {
		$ourOptions = [
			'fndry_component_styles',
			'fndry_global_styles',
			'fndry_custom_fields',
			'fndry_animation_settings',
			'fndry_general_settings',
		];

		$key = $request->get_param( 'key' );
		// we're doin a specific one
		if ( in_array( $key, $ourOptions ) ) {
			return $this->sanitizeOption( $key );
//			return get_option( $key );
		} else {
			$result = [];
			foreach ( $ourOptions as $opt ) {
				$result[ $opt ] = get_option( $opt );
			}

			return $result;
		}
	}

	private function sanitizeOption( $key ) {
		$registered = get_registered_settings();
		if ( ! isset( $registered[ $key ] ) ) {
			return null;
		}

		$ourSchema = $registered[ $key ]['show_in_rest']['schema'];

		$ourOption = get_option( $key );

		// for the most part, we're just looking at stripping out bad vals

		return $this->restSanitizeValueFromSchema( $ourOption, $ourSchema );
	}

	public function sanitizeSchema__GLOBAL_STYLES( $value ) {
		return $this->restSanitizeValueFromSchema( $value, FoundrySchema::GLOBAL_STYLES );
	}

	public function sanitizeSchema__CUSTOM_FIELDS( $value ) {
		return $this->restSanitizeValueFromSchema( $value, FoundrySchema::CUSTOM_FIELDS );
	}

	public function sanitizeSchema__ANIMATION( $value ) {
		return $this->restSanitizeValueFromSchema( $value, FoundrySchema::ANIMATION );
	}

	public function sanitizeSchema__GENERAL( $value ) {
		return $this->restSanitizeValueFromSchema( $value, FoundrySchema::GENERAL );
	}

	public function sanitizeSchema__COMPONENT_STYLES( $value ) {
		return $this->restSanitizeValueFromSchema( $value, FoundrySchema::COMPONENT_STYLES );
	}

	private function restSanitizeValueFromSchema( $value, $args, $param = '' ) {
		if ( isset( $args['anyOf'] ) ) {
			$matching_schema = rest_find_any_matching_schema( $value, $args, $param );
			if ( is_wp_error( $matching_schema ) ) {
				return $matching_schema;
			}

			if ( ! isset( $args['type'] ) ) {
				$args['type'] = $matching_schema['type'];
			}

			$value = $this->restSanitizeValueFromSchema( $value, $matching_schema, $param );
		}

		if ( isset( $args['oneOf'] ) ) {
			$matching_schema = rest_find_one_matching_schema( $value, $args, $param );
			if ( is_wp_error( $matching_schema ) ) {
				return $matching_schema;
			}

			if ( ! isset( $args['type'] ) ) {
				$args['type'] = $matching_schema['type'];
			}

			$value = $this->restSanitizeValueFromSchema( $value, $matching_schema, $param );
		}

		$allowed_types = [ 'array', 'object', 'string', 'number', 'integer', 'boolean', 'null' ];

		if ( ! isset( $args['type'] ) ) {
			/* translators: %s: Parameter. */
			_doing_it_wrong( __FUNCTION__,
				esc_html( sprintf( 'The "type" schema keyword for %s is required.', $param ) ),
				'5.5.0' );
		}

		if ( is_array( $args['type'] ) ) {
			$best_type = rest_handle_multi_type_schema( $value, $args, $param );

			if ( ! $best_type ) {
				return null;
			}

			$args['type'] = $best_type;
		}

		if ( ! in_array( $args['type'], $allowed_types, true ) ) {
			_doing_it_wrong(
				__FUNCTION__,
				/* translators: 1: Parameter, 2: The list of allowed types. */
				esc_html( wp_sprintf( 'The "type" schema keyword for %1$s can only be one of the built-in types: %2$l.',
					$param,
					$allowed_types ) ),
				'5.5.0'
			);
		}

		if ( 'array' === $args['type'] ) {
			$value = rest_sanitize_array( $value );

			if ( ! empty( $args['items'] ) ) {
				foreach ( $value as $index => $v ) {
					$value[ $index ] = $this->restSanitizeValueFromSchema( $v,
						$args['items'],
						$param . '[' . $index . ']' );
				}
			}

			if ( ! empty( $args['uniqueItems'] ) && ! rest_validate_array_contains_unique_items( $value ) ) {
				/* translators: %s: Parameter. */
				return new WP_Error( 'rest_duplicate_items', sprintf( '%s has duplicate items.', $param ) );
			}

			return $value;
		}

		if ( 'object' === $args['type'] ) {
			$value = rest_sanitize_object( $value );
			foreach ( $value as $property => $v ) {
				if ( isset( $args['properties'][ $property ] ) ) {
					$value[ $property ] = $this->restSanitizeValueFromSchema( $v,
						$args['properties'][ $property ],
						$param . '[' . $property . ']' );
					continue;
				} else {
					unset( $value[ $property ] );
				}

				$pattern_property_schema = rest_find_matching_pattern_property_schema( $property, $args );
				if ( null !== $pattern_property_schema ) {
					$value[ $property ] = $this->restSanitizeValueFromSchema( $v,
						$pattern_property_schema,
						$param . '[' . $property . ']' );
					continue;
				}

				if ( isset( $args['additionalProperties'] ) ) {
					if ( false === $args['additionalProperties'] ) {
						unset( $value[ $property ] );
					} elseif ( is_array( $args['additionalProperties'] ) ) {
						$value[ $property ] = $this->restSanitizeValueFromSchema( $v,
							$args['additionalProperties'],
							$param . '[' . $property . ']' );
					}
				}
			}

			return $value;
		}

		if ( 'null' === $args['type'] ) {
			return null;
		}

		if ( 'integer' === $args['type'] ) {
			return (int) $value;
		}

		if ( 'number' === $args['type'] ) {
			return (float) $value;
		}

		if ( 'boolean' === $args['type'] ) {
			return rest_sanitize_boolean( $value );
		}

		// This behavior matches rest_validate_value_from_schema().
		if ( isset( $args['format'] )
		     && ( ! isset( $args['type'] ) || 'string' === $args['type'] || ! in_array( $args['type'],
					$allowed_types,
					true ) )
		) {
			switch ( $args['format'] ) {
				case 'hex-color':
					return (string) sanitize_hex_color( $value );

				case 'date-time':
					return sanitize_text_field( $value );

				case 'email':
					// sanitize_email() validates, which would be unexpected.
					return sanitize_text_field( $value );

				case 'uri':
					return sanitize_url( $value );

				case 'ip':
					return sanitize_text_field( $value );

				case 'uuid':
					return sanitize_text_field( $value );

				case 'text-field':
					return sanitize_text_field( $value );

				case 'textarea-field':
					return sanitize_textarea_field( $value );
			}
		}

		if ( 'string' === $args['type'] ) {
			return (string) $value;
		}

		return $value;
	}

	public function fetchAdobe( WP_REST_Request $request ) {
		$project_id = $request->get_param( 'projectId' );
		$api_key    = $request->get_param( 'apiKey' );

		if ( $project_id !== null && $api_key !== null ) {
			$api_url = 'https://typekit.com/api/v1/json/kits/' . $project_id . '?token=' . $api_key . '';

			if ( is_wp_error( ( $data = @wp_remote_get( $api_url ) ) ) ) {
				$error = error_get_last();
				// if it's a 404, then the project id is incorrect (because token is a query param)
				$status = '500';
				if ( str_contains( $error['message'], '404' ) ) {
					$status = '404';
					$error  = [
						'status'  => '404',
						'message' => 'Invalid Project ID',
						'data'    => $data,
					];
				} elseif ( str_contains( $error['message'], '401' ) ) {
					$status = '401';
					$error  = [
						'status'  => '401',
						'message' => 'Invalid API Key',
						'data'    => $data,
					];
				}

				return new WP_REST_Response( $error, $status );
			} else {
				return json_decode( wp_remote_retrieve_body( $data ) );
			}
		}

		return [ 'error' => 'missin some params there bud' ];
	}

	/* getters and setters for block store */

	public function fetchPostTypes( WP_REST_Request $request ) {
		return get_post_types( [ 'public' => true ], 'objects' );
	}

	/*
		exceptions:
		[blockName] => [className]
		- /list => ListBlock (list is reserved by php)
		- /foundry-map-block => MapBlock (adding foundry would be regressing imo)
		- /foundry-slideout => Slideout
		*/
	public function getOrConstructFoundryBlock( $attributes, $content = null, $wp_block = null ) {
		if ( is_admin() ) {
			return null;
		}

		$className = $this->getBlockClassName( $wp_block->name );
		if ( ! class_exists( $className ) ) {
			new WP_Error( '666', 'Missing block ID and RCB id!' );

			return null;
		}

		// RCB's assign block ID during their inner block parsing.
		if ( ! isset( $attributes['fndryBlockId'] ) ) {
			$attributes['fndryBlockId'] = uniqid( 'fbr-' );
		}

		$block = new $className( $attributes, $content, $wp_block );

		return $block->renderCallback( $wp_block->context );
	}

	public function getBlockClassName( $blockName ) {
		$phpClassName = null;
		$blockName    = $blockName ? explode( 'foundry/', $blockName )[1] : false;
		if ( ! $blockName ) {
			return null;
		}

		if ( $blockName === 'list' ) {
			$phpClassName = 'ListBlock';
		} elseif ( $blockName === 'foundry-map-block' ) {
			$phpClassName = 'MapBlock';
		} elseif ( $blockName === 'foundry-slideout' ) {
			$phpClassName = 'Slideout';
		} else {
			$phpClassName = str_replace( '-', '', ucwords( $blockName, '-' ) );
		}

		return "ForgeSmith\Blocks\\$phpClassName";
	}

	protected function registerBlock( $class, $json ) {
		if ( file_exists( $json ) ) {
			register_block_type_from_metadata( $json, [
				'render_callback'   => [ "ForgeSmith\Blocks\\$class", 'create' ],
				'skip_inner_blocks' => true,
			] );
		} else {
			new WP_Error( 500, 'No block registered at ' . $json );
		}
	}

	/**
	 * Registers blocks by class name as string, excluding namespace.
	 * This adds the class object to the forgeBlocks array, allowing the blocks to be modified through use
	 * of their own methods and functions.
	 *
	 * @param array $classes
	 */
	public function registerBlocks( array $classes = [] ) {
		foreach ( $classes as $class ) {
			$this->registerBlock( $class, REFOUNDRY_CORE_DIR . "src/blocks/$class/block.json" );
		}

		$otherClasses = apply_filters( 'foundry_register_blocks', null );

		if ( ! empty( $otherClasses ) && is_array( $otherClasses ) ) {
			foreach ( $otherClasses as $class => $json ) {
				$this->registerBlock( $class, $json );
			}
		}

		// instead of whitelisting only a specific set of blocks, we're just getting rid of problematic ones.
		add_filter( 'allowed_block_types_all', function ( $restrictedBlocks ) {
			$test = array_keys( WP_Block_Type_Registry::get_instance()->get_all_registered() );

			$verboten = [
				'core/navigation',
				'core/paragraph',
				'core/heading',
				'core/image',
				'core/query',
			];

			$cleanedCore = array_diff( $test, $verboten );

			if ( ! is_array( $restrictedBlocks ) && $restrictedBlocks === true ) {
				$restrictedBlocks = [];
			}

			return array_merge( $cleanedCore, $restrictedBlocks );
		}, 1, 1 );
	}

	public function setupCustomFieldsMetaBox() {
		// Right now this will show on all post types, in the future we could make it do specific post types
		add_meta_box(
			'fndry_custom_field',
			'Foundry Custom Fields',
			[ $this, 'customFieldMetaBoxDisplay' ],
			null,
			'side',
			'low',
			[
				'__block_editor_compatible_meta_box' => true,
				'__back_compat_meta_box'             => false,
			]
		);
	}

	public function saveCustomFieldsPostData( $post_id ) {
		if ( empty( $_REQUEST['fndry_custom_field'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['fndry_custom_field'] ) ),
				'fndry_custom_field' ) ) {
			return;
		}
		foreach ( $this->allOptions['manageCustomFields']['customFields'] as $custom_field ) {
			$field_slug = $custom_field['slug'];
			if ( array_key_exists( $field_slug, $_POST ) ) {
				update_post_meta(
					$post_id,
					$field_slug,
					filter_var( wp_unslash( $_POST[ $field_slug ] ), FILTER_SANITIZE_SPECIAL_CHARS )
				);
			}
		}
	}

	public function customFieldMetaBoxDisplay( $post ) {
		wp_create_nonce( 'fndry_custom_field' );
		foreach ( $this->allOptions['manageCustomFields']['customFields'] as $custom_field ) {
			if ( ! empty( $custom_field['slug'] ) && ! empty( $custom_field['label'] ) && ! empty( $custom_field['postType'] ) && in_array( $post->post_type,
					$custom_field['postType'] ) ) {
				$field_slug  = $custom_field['slug'];
				$field_label = $custom_field['label'];

				$value = get_post_meta( $post->ID, $field_slug, true );

				?>
                <label for="<?php echo esc_attr( $field_slug ) ?>"><?php echo esc_attr( $field_label ) ?></label><br />
                <input type="text" name="<?php echo esc_attr( $field_slug ) ?>"
                       id="<?php echo esc_attr( $field_slug ) ?>"
                       class="postbox" style="width:60%; border: 1px solid grey; padding: 5px; margin-top:10px;"
                       value="<?php echo esc_attr( $value ) ?>" />
                <br /><br />
				<?php
			}
		}
		wp_nonce_field( 'fndry_custom_field', 'fndry_custom_field' );
	}

	public function customFieldRestAPI() {
		foreach ( $this->allOptions['manageCustomFields']['customFields'] as $custom_field ) {
			$field_slug  = $custom_field['slug'] ?? false;
			$field_label = $custom_field['label'] ?? false;

			// ignore busted values
			if ( ! $field_slug && ! $field_label ) {
				continue;
			}

			foreach ( $custom_field['postType'] as $post_type ) {
				register_meta( 'post', $field_slug, [
					'object_subtype' => $post_type,
					'type'           => 'string',
					'description'    => $field_label,
					'single'         => true,
					'show_in_rest'   => true,
				] );
            };
		}
	}

	/**
	 * Checks the theme for a JSON file that provides default values.
	 * Currently, this is only applied around the time of the "init" action.
	 *
	 * @return void|WP_Error
	 */
	public function overrideDefaultsFromThemJSON() {
		$wp_theme = wp_get_theme();

		// check if theme file exists first.
		$theme_json_file = $wp_theme->get_file_path( 'refoundry.json' );
		if ( ! file_exists( $theme_json_file ) || ! is_readable( $theme_json_file ) ) {
			return;
		}

		// if we do have any empty options, get the theme json and override it.
		$settings       = get_registered_settings();
		$cachedSettings = wp_json_file_decode( $theme_json_file, [ 'associative' => true ] );

		// if the JSON file can't be decoded, abort!
		if ( ! $cachedSettings ) {
			// maybe also tell them something's wrong.
			return new WP_Error( 'bad_refoundry_json',
				"Malformed JSON - refoundry.json theme file found, but cannot be decoded." );
		}

		foreach (
			[
				'global_styles',
				'component_styles',
			] as $optionKey
		) {
			if ( ! empty( $settings["fndry_{$optionKey}"] ) && ! empty( $cachedSettings[ $optionKey ] ) ) {
				$setting   = $cachedSettings[ $optionKey ];
				$ourSchema = $settings["fndry_{$optionKey}"]['show_in_rest']['schema'];
				$sanitized = $this->restSanitizeValueFromSchema( $setting, $ourSchema );
				if ( ! empty( $sanitized ) ) {
					// if the decoded values from the JSON pass muster, then do the filter thing.
					add_filter( "default_option_fndry_{$optionKey}", function () use ( $sanitized ) {
						return $sanitized;
					} );
				}
			}
		}
	}
}
