<?php

  /**
   * File for our cool review ask in the header
   *
   * @category Child Plugin
   * @version v0.1.0
   * @since v0.1.0
   * @author iClyde <kontakt@iclyde.pl>
   */

  // Namespace
  namespace Inisev\Subs;

  // Disallow direct access
  if (defined('ABSPATH')) {

    /**
     * Main class for handling the Review
     */
    if (!class_exists('Inisev\Subs\Inisev_Review')) {
      class Inisev_Review {

        /**
         * Local variables
         */
        private $root; // __ROOT__ of plugin's root
        private $file; // __FILE__ of plugin's root
        private $slug; // Plugin's slug
        private $name; // Name displayed in banner
        private $days; // Days displayed in banner
        private $minutes; // Minutes displayed in banner
        private $debug = false; // If true it will display minutes
        private $minimum_time = (30 * 24 * 60 * 60); // Minimum time required to display (in seconds)
        // private $minimum_time = (3 * 60); // Minimum time required to display (in seconds)
        private $remind_time = '+14 days'; // Time when banner will be reminded
        private $time_between = '+2 days'; // Time when new banner can be displayed

        /**
         * Local URLs
         */
        private $root_url; // Root URL for plugin's dir
        private $assets_url; // Root URL for review assets
        private $review_url; // Review URL
        private $plugin_menu_url; // Plugin's settings menu
        public $option_name = '_irb_h_bn_review'; // Option name for this module
        public $using_since; // Check since user uses this plugin

        /**
         * __construct:
         * Compile some variables for "future use"
         * Such as slug of current plugin, root dir of plugin
         *
         * @param  string $root_file       __FILE__ of plugin's main file
         * @param  string $root_dir        __DIR__ of plugin's main file
         * @param  string $individual_slug Individual slug - mostly plugin's slug
         * @param  string $display_name    The name that will be displayed in the banner
         * @param  string $review_url      The URL that will be served as review one
         * @param  string $plugin_menu_url Plugin menu slug example.com/wp-admin/admin.php?page=<this slug here>
         * @return Inisev_Review           The review object
         */
        function __construct($root_file, $root_dir, $individual_slug, $display_name, $review_url, $plugin_menu_url) {

          $this->file = $root_file;
          $this->root = $root_dir;
          $this->slug = $individual_slug;
          $this->name = $display_name;

          $this->review_url = $review_url;
          $this->plugin_menu_url = admin_url('admin.php?page=' . $plugin_menu_url);

          $this->root_url = plugin_dir_url($this->file);
          $this->assets_url = $this->root_url . 'modules/review/assets/';
          $option_name = $this->option_name;
          $empty = ['users' => []];
          $empty[$individual_slug] = time();

          $data = get_option($option_name, false);

          if ($data != false && isset($data) && is_array($data)) {

            if (!array_key_exists($individual_slug, $data)) {

              $data[$individual_slug] = time();
              update_option($option_name, $data);

            }

            $this->using_since = $data;

          } else {

            if ($individual_slug == 'copy-delete-posts' && get_option('_cdp_review', false) != false) {
              if (isset(get_option('_cdp_review', false)['installed'])) {
                $empty[$individual_slug] = get_option('_cdp_review', false)['installed'];
              }
            }

            update_option($option_name, $empty);
            $this->using_since = $empty;

          }

          // Add handler for Ajax request
          if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] == 'POST') {

            // Check if slug is defined
            if (isset($_POST['slug']) && !empty($_POST['slug'])) {

              // Handle the request
              add_action('wp_ajax_inisev_review', [&$this, 'handle_review_action']);

            }

            // Stop for POST
            return;

          }

          add_action('wp_loaded', [&$this, 'init_review']);

        }

        /**
         * __asset - Loads assets
         *
         * @param  string $file path relative
         * @return string       file URL
         */
        private function __asset($file) {

          return $this->assets_url . $file;

        }

        /**
         * __dir_asset - Loads assets
         *
         * @param  string $file path relative
         * @return string       absolute path
         */
        private function __dir_asset($file) {

          return __DIR__ . '/assets' . '/' . $file;

        }

        /**
         * _asset - Loads assets and automatically echo
         *
         * @param  string $file path relative
         * @echo   string       file URL
         */
        private function _asset($file) {

          echo $this->assets_url . $file;

        }

        /**
         * _dir_asset - Loads assets and automatically echo
         *
         * @param  string $file path relative
         * @echo   string       absolute path
         */
        private function _dir_asset($file) {

          echo __DIR__ . '/assets' . '/' . $file;

        }

        /**
         * can_be_displayed - check if the banner should be displayed
         *
         * @return bool true if banner can be displayed
         */
        private function can_be_displayed() {
          
          global $pagenow;
          
          if ($pagenow == 'post.php' || $pagenow == 'profile.php' || $pagenow == "post-new.php") {
            return false;
          }

          $uid = get_current_user_id();
          if (!defined('IRB_H_CHECK_LOADED') && function_exists('get_current_user_id') && isset($uid) && intval($uid) > 0) {

            $since = intval($this->using_since[$this->slug]);
            $diff = time() - $since;

            if ($diff > $this->minimum_time) {

              $seconds = $diff;
              $minutes = intval($diff / 60);
              $hours = intval($minutes / 60);
              $days = intval($hours / 24);

              $this->days = $days;
              $this->minutes = $minutes;
              $data = $this->using_since;

              if (isset($data['users']) && isset($data['users'][$uid])) {

                if (isset($data['users'][$uid]['delay_between'])) {
                  if (time() <= intval($data['users'][$uid]['delay_between'])) {
                    return false;
                  }
                }

                if (isset($data['users'][$uid][$this->slug])) {
                  if (isset($data['users'][$uid][$this->slug]['remind'])) {
                    if (time() <= intval($data['users'][$uid][$this->slug]['remind'])) {
                      return false;
                    }
                  }
                  if (isset($data['users'][$uid][$this->slug]['dismiss'])) {
                    if ($data['users'][$uid][$this->slug]['dismiss'] == true || $data['users'][$uid][$this->slug]['dismiss'] == 'true') {
                      return false;
                    }
                  }
                }

              }

              define('IRB_H_CHECK_LOADED', true);
              return true;

            } else return false;

          } else return false;

        }

        /**
         * add_assets - adds required assests by the banner
         *
         * @return void
         */
        public function add_assets() {

          if (!defined('IRB_H_ASSETS_LOADED')) {
            define('IRB_H_ASSETS_LOADED', true);
            wp_enqueue_script('inisev-review-script', $this->__asset('js/script.js'), [], filemtime($this->__dir_asset('js/script.js')), true);
            wp_enqueue_style('inisev-review-style', $this->__asset('css/style.css'), [], filemtime($this->__dir_asset('css/style.css')));
            wp_localize_script('inisev-review-script', 'inisev_review_dismiss', [
              'nonce' => wp_create_nonce('inisev_review_dismiss'),
            ], true);
          }

        }

        /**
         * display_review - loads the HTML and prints it in the header only once
         *
         * @return void
         */
        public function display_review() {

          if (!defined('IRB_H_HTML_LOADED')) {
            define('IRB_H_HTML_LOADED', true);
            include_once __DIR__ . '/views/banner.php';
          }

        }

        /**
         * handle_review_action - Handles all POST actions
         *
         * @param  string $_POST['token'] - must be === 'irbh'
         * @param  string $_POST['slug'] - the unique slug
         * @param  string $_POST['mode'] - the unique action remind/dismiss
         *
         * @return json returns it to browser
         */
        public function handle_review_action() {

          if (check_ajax_referer('inisev_review_dismiss', 'nonce', false) === false) {
            return wp_send_json_error();
          }

          $slug = sanitize_text_field($_POST['slug']);
          $mode = sanitize_text_field($_POST['mode']);

          if (!empty($_POST['slug']) && isset($mode) && in_array($mode, ['dismiss', 'remind'])) {
            $option_name = $this->option_name;
            $data = get_option($option_name, false);
            if ($data != false) {

              $uid = get_current_user_id();

              if (!array_key_exists('users', $data)) $data['users'] = [];
              if (!array_key_exists($uid, $data['users'])) $data['users'][$uid] = [];
              if (!array_key_exists($slug, $data['users'][$uid])) $data['users'][$uid][$slug] = [];

              $data['users'][$uid]['delay_between'] = strtotime($this->time_between);

              if ($mode == 'remind') {
                $data['users'][$uid][$slug]['remind'] = strtotime($this->remind_time);
              }

              if ($mode == 'dismiss') {
                $data['users'][$uid][$slug]['dismiss'] = true;
              }

              update_option($option_name, $data);

              wp_send_json_success();

            } else wp_send_json_error();
          } else wp_send_json_error();

        }

        /**
         * init_review - initialization when the user is authenticated already
         *
         * @return void
         */
        public function init_review() {

          if ($this->can_be_displayed()) {
            add_action('admin_enqueue_scripts', [&$this, 'add_assets']);
            add_action('admin_notices', [&$this, 'display_review']);
          }

        }

      }
    }

  }
