<?php
/**
 * Plugin Name: Redesignee
 *  Description: Premium widgets for Elementor & Gutenberg.
 *  Version: 1.0.1
 *  Author: Redesignee
 *  License: GPLv2 or later
 *  Text Domain: redesignee
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

// 1. Admin Menu & Settings
function redesignee_add_admin_menu() {
    add_menu_page(
        'Redesignee Settings',
        'Redesignee',
        'manage_options',
        'redesignee-settings',
        'redesignee_settings_page_html',
        'dashicons-layout',
        100
    );
}
add_action('admin_menu', 'redesignee_add_admin_menu');

function redesignee_settings_page_html() {
    // Check user capabilities first
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    if ( isset( $_POST['redesignee_api_key'] ) ) {
        if ( ! check_admin_referer( 'redesignee_save_key' ) ) {
            wp_die( 'Security check failed' );
        }

        $input_key = sanitize_text_field( wp_unslash( $_POST['redesignee_api_key'] ) );

        // If key is empty, clear options
        if ( empty( $input_key ) ) {
            update_option( 'redesignee_api_key', '' );
            update_option( 'redesignee_is_premium', false );
            echo '<div class="notice notice-info"><p>License key removed. Reverted to Free Version.</p></div>';
        }
        else {
            // Validate Key against API
            $response = wp_remote_get( 'https://redesignee.com/api/squarespace-widgets', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $input_key,
                    'Referer'       => site_url()
                ],
                'timeout' => 15,
                'sslverify' => false
            ]);

            if ( is_wp_error( $response ) ) {
                // Connection Error
                echo '<div class="notice notice-error"><p>Connection failed: ' . esc_html( $response->get_error_message() ) . '</p></div>';
            }
            else {
                $code = wp_remote_retrieve_response_code( $response );

                if ( $code === 200 ) {
                    // SUCCESS
                    update_option( 'redesignee_api_key', $input_key );
                    update_option( 'redesignee_is_premium', true );
                    echo '<div class="notice notice-success"><p><strong>Success!</strong> Your Premium License is active.</p></div>';
                }
                elseif ( $code === 401 || $code === 403 ) {
                    // AUTH ERROR (Wrong Key or Domain not allowed)
                    update_option( 'redesignee_api_key', $input_key ); // Save it so user can fix it
                    update_option( 'redesignee_is_premium', false );

                    // SECURITY FIX: Safely retrieve, sanitize, and escape the Host
                    $host = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : 'your domain';

                    echo '<div class="notice notice-error">
                        <p><strong>Activation Failed</strong></p>
                        <p>Your API Key was rejected. Please ensure you have added <strong>' . esc_html( $host ) . '</strong> to your allowed domains at <a href="https://redesignee.com/user/settings" target="_blank">redesignee.com/user/settings</a>.</p>
                    </div>';
                }
                else {
                    // OTHER SERVER ERRORS
                    echo '<div class="notice notice-error"><p>Server returned error code: ' . esc_html( $code ) . '. Please try again later.</p></div>';
                }
            }
        }
    }

    // Retrieve Current Values
    $current_key = get_option( 'redesignee_api_key', '' );
    $is_premium  = get_option( 'redesignee_is_premium', false );
    ?>
    <div class="wrap">
        <h1>Redesignee Settings</h1>

        <form method="post">
            <?php wp_nonce_field( 'redesignee_save_key' ); ?>

            <table class="form-table">
                <tr>
                    <th scope="row">License Key</th>
                    <td>
                        <input type="text" name="redesignee_api_key" value="<?php echo esc_attr( $current_key ); ?>" class="regular-text">
                        <p class="description">Enter your Pro Key to unlock premium widgets.</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Status</th>
                    <td>
                        <?php if ( $is_premium ): ?>
                            <span class="dashicons dashicons-yes-alt" style="color: green; vertical-align: middle;"></span>
                            <strong style="color: green; vertical-align: middle;">Premium Active</strong>
                        <?php else: ?>
                            <?php if( !empty($current_key) ): ?>
                                <span class="dashicons dashicons-warning" style="color: #d63638; vertical-align: middle;"></span>
                                <strong style="color: #d63638; vertical-align: middle;">Invalid Key / Domain</strong>
                            <?php else: ?>
                                <strong style="color: #666;">Free Version</strong>
                            <?php endif; ?>
                        <?php endif; ?>
                    </td>
                </tr>
            </table>

            <?php submit_button( 'Save License' ); ?>
        </form>
    </div>
    <?php
}

// 2. Load Elementor Integration
function redesignee_load_elementor_support() {
    if ( did_action( 'elementor/loaded' ) ) {
        if ( file_exists( plugin_dir_path( __FILE__ ) . 'includes/elementor-integration.php' ) ) {
            require_once plugin_dir_path( __FILE__ ) . 'includes/elementor-integration.php';
        }
    }
}
add_action( 'plugins_loaded', 'redesignee_load_elementor_support' );

// --- HELPER: Enqueue Local Dependencies (Coloris, Trumbowyg) ---
function redesignee_enqueue_local_assets() {
    // Coloris
    wp_enqueue_style( 'redesignee-coloris-css', plugin_dir_url( __FILE__ ) . 'css/coloris.min.css', [], '1.0' );
    wp_enqueue_script( 'redesignee-coloris-js', plugin_dir_url( __FILE__ ) . 'js/coloris.min.js', [], '1.0', true );

    // Trumbowyg
    wp_enqueue_style( 'redesignee-trumbowyg-css', plugin_dir_url( __FILE__ ) . 'css/trumbowyg.min.css', [], '2.27.3' );
    // Trumbowyg depends on jQuery
    wp_enqueue_script( 'redesignee-trumbowyg-js', plugin_dir_url( __FILE__ ) . 'js/trumbowyg.min.js', [ 'jquery' ], '2.27.3', true );
}

// 3. Enqueue Editor Scripts (Elementor & Classic Admin)
function redesignee_enqueue_editor_scripts( $hook ) {
    global $pagenow;
    $is_edit_page = ( $pagenow == 'post.php' || $pagenow == 'post-new.php' );
    $is_elementor = did_action( 'elementor/loaded' );

    if ( ! $is_edit_page && ! $is_elementor ) {
        return;
    }

    wp_enqueue_media();

    // 1. Enqueue Dependencies Locally
    redesignee_enqueue_local_assets();

    // 2. Enqueue Main Editor
    wp_enqueue_script(
        'redesignee-editor-overlay',
        plugin_dir_url( __FILE__ ) . 'js/editor-overlay.js',
        [ 'jquery', 'redesignee-coloris-js', 'redesignee-trumbowyg-js' ],
        '1.5.1',
        true
    );

    $is_premium = get_option( 'redesignee_is_premium', false );
    $api_key    = get_option( 'redesignee_api_key', '' );

    wp_localize_script( 'redesignee-editor-overlay', 'RedesigneeConfig', [
        'isPremium' => $is_premium,
        'apiKey'    => $api_key,
        'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
        'assetsUrl' => plugin_dir_url( __FILE__ ) . 'assets/',
        'svgPath'   => plugin_dir_url( __FILE__ ) . 'assets/images/icons.svg'
    ]);
}
add_action( 'admin_enqueue_scripts', 'redesignee_enqueue_editor_scripts' );
add_action( 'elementor/editor/after_enqueue_scripts', 'redesignee_enqueue_editor_scripts' );


// 4. Gutenberg Block Registration
function redesignee_register_gutenberg_block() {
    // Register the Gutenberg Bridge Script
    wp_register_script(
        'redesignee-gutenberg-js',
        plugin_dir_url( __FILE__ ) . 'js/admin-gutenberg.js',
        array( 'wp-blocks', 'wp-element', 'wp-components', 'wp-editor' ),
        '1.0',
        true
    );

    // Register the Block
    register_block_type( 'redesignee/widget-block', array(
        'editor_script' => 'redesignee-gutenberg-js',
    ));
}
add_action( 'init', 'redesignee_register_gutenberg_block' );

// 5. Enqueue Assets specifically for Gutenberg Editor
function redesignee_enqueue_block_editor_assets() {
    // 1. Enqueue Dependencies Locally
    redesignee_enqueue_local_assets();

    // 2. Load main engine
    wp_enqueue_script(
        'redesignee-editor-overlay',
        plugin_dir_url( __FILE__ ) . 'js/editor-overlay.js',
        [ 'jquery', 'redesignee-coloris-js', 'redesignee-trumbowyg-js' ],
        '1.4.0',
        true
    );

    // 3. Pass Configuration to Script (Same as Elementor)
    $is_premium = get_option( 'redesignee_is_premium', false );
    $api_key    = get_option( 'redesignee_api_key', '' );

    wp_localize_script( 'redesignee-editor-overlay', 'RedesigneeConfig', [
        'isPremium' => $is_premium,
        'apiKey'    => $api_key,
        'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
        'assetsUrl' => plugin_dir_url( __FILE__ ) . 'assets/',
        'svgPath'   => plugin_dir_url( __FILE__ ) . 'assets/images/icons.svg'
    ]);

    wp_add_inline_style( 'wp-edit-blocks', '
        .rd-gutenberg-block { border: 1px dashed #764fe3; padding: 20px; background: #f9f9f9; }
        .rd-gutenberg-block .components-placeholder__label { color: #764fe3; }
    ' );
}
add_action( 'enqueue_block_editor_assets', 'redesignee_enqueue_block_editor_assets' );