(function($) {

    const RD_Config = {
        apiBase: 'https://redesignee.com',
        googleAPI: "https://www.googleapis.com/webfonts/v1/webfonts?key=AIzaSyCbs2mTuE2MgePWOYsti8T3NcFI5KazsHU"
    };

    window.Redesignee = window.Redesignee || {};
    window.Redesignee.currentSession = null;
    let googleFonts = [];
    let isColorisReady = false;
    let isRichTextReady = false;

    // ---------------------------------------------------------
    // 1. PUBLIC API
    // ---------------------------------------------------------
    window.Redesignee.openModal = function(options) {
        if ($('#redesignee-modal-wrapper').length === 0) initModalEnvironment();
        window.Redesignee.currentSession = options || {};

        loadAssets();
        prepareModalUI();

        const hasId = options.existingCode && options.existingCode.indexOf('data-widget-id') !== -1;
        if (options.mode === 'edit' && hasId) {
            loadEditView();
        } else {
            loadLibraryView();
        }

        $('#redesignee-modal-wrapper').css('display', 'flex').hide().fadeIn();
    };

    // ---------------------------------------------------------
    // 2. INIT & UTILS
    // ---------------------------------------------------------
    function initModalEnvironment() {
        buildDOM();
        bindGlobalEvents();
        if(googleFonts.length === 0) {
            $.ajax({ url: RD_Config.googleAPI, dataType: 'json', success: function(data) { googleFonts = data.items.map(f => f.family); } });
        }
    }

    function loadAssets() {
        // 1. Initialize Coloris (Assumes script is enqueued via PHP)
        if (window.Coloris) {
            isColorisReady = true;
            setTimeout(function() {
                safeInitColoris();
            }, 50);
        } else {
            console.warn('[RD] Coloris script not found. Make sure it is enqueued via PHP.');
        }

        // 2. Check Trumbowyg (Assumes script is enqueued via PHP)
        if ($.fn.trumbowyg) {
            isRichTextReady = true;
        }

        // 3. Load UI Font
        if ($('link[href*="Albert+Sans"]').length === 0) {
            $('head').append('<link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Albert+Sans:wght@400;600;700&family=Material+Icons&display=swap">');
        }
    }

    function safeInitColoris() {
        if (window.Coloris) {
            try {
                // 1. Only call init() if it exists (for newer Coloris versions)
                if (typeof Coloris.init === 'function') {
                    Coloris.init();
                }

                // 2. Configure Coloris (This applies the UI to the inputs)
                Coloris({
                    el: '.coloris',
                    themeMode: 'auto',
                    alpha: true,
                    wrap: true,
                    format: 'hex'
                });

                // 3. Force re-binding for dynamic inputs
                if (typeof Coloris.wrap === 'function') {
                    // Some older versions exposed this internal helper
                }

                console.log("Coloris initialized successfully.");
            } catch (e) {
                console.error("Coloris setup error:", e);
            }
        }
    }

    function slugify(text) {
        return (text || '').toString().toLowerCase().trim().replace(/\s+/g, '-').replace(/[^\w\-]+/g, '');
    }

    // ---------------------------------------------------------
    // 3. VIEW CONTROLLERS
    // ---------------------------------------------------------
    function loadLibraryView() {
        if(window.Redesignee.currentSession.onSave) window.Redesignee.currentSession.mode = 'add';
        $('#rd-editor-view').hide();
        $('#rd-library-view').css('display', 'flex');
        $('#rd-back-btn').hide();

        const $iframe = $('#live-preview');
        const doc = $iframe[0].contentDocument || $iframe[0].contentWindow.document;
        doc.open(); doc.write(''); doc.close();

        $('#customizer').empty();
        fetchLibrary();
    }

// ---------------------------------------------------------
    // 3. VIEW CONTROLLERS (FIXED)
    // ---------------------------------------------------------
    function loadEditView() {
        $('#rd-library-view').hide();
        $('#rd-editor-view').css('display', 'flex');
        $('#rd-back-btn').show();

        const btnText = (window.Redesignee.currentSession.onSave) ? 'Update Widget' : 'Get Code';
        $('#rd-save-btn').text(btnText);
        $('#customizer').html('<div style="padding:20px; color:#666;">Loading editor...</div>');

        let id = window.Redesignee.currentSession.widgetId;
        const existingCode = window.Redesignee.currentSession.existingCode;

        // Try to find ID in existing code if not set
        if (!id && existingCode) {
            const match = existingCode.match(/data-widget-id="([^"]+)"/);
            if (match && match[1]) id = match[1];
        }

        if (id) {
            // We still fetch assets to get the "structure" for new widgets,
            // but we prioritize existingCode if available.
            fetchWidgetData(id, function(assets) {

                let html, css, js;

                if (window.Redesignee.currentSession.mode === 'edit' && existingCode) {
                    // --- A. LOAD EXISTING (PRESERVE CUSTOM CODE) ---

                    // 1. Extract HTML (Clean script tags out)
                    html = cleanCodeForPreview(existingCode);

                    // 2. Extract CSS (Find content inside <style>)
                    const styleMatch = existingCode.match(/<style[^>]*>([\s\S]*?)<\/style>/i);
                    css = styleMatch ? styleMatch[1] : (assets.css || ''); // Fallback to API if empty

                    // 3. Extract JS (Find content inside <script>)
                    // We need to be careful not to grab the library script itself, usually widget JS is at the end
                    const scriptMatches = existingCode.match(/<script\b[^>]*>([\s\S]*?)<\/script>/gim);
                    if (scriptMatches && scriptMatches.length > 0) {
                        // Assuming the last script tag is the widget logic
                        const lastScript = scriptMatches[scriptMatches.length - 1];
                        js = lastScript.replace(/<script\b[^>]*>|<\/script>/gim, "");
                    } else {
                        js = assets.js || '';
                    }

                    // Store these in session so SaveWidget uses them
                    window.Redesignee.currentSession.loadedCSS = css;
                    window.Redesignee.currentSession.loadedJS = js;
                    window.Redesignee.currentSession.isExisting = true;

                } else {
                    // --- B. LOAD NEW (FROM API) ---
                    html = `<div class="redesignee-widget" data-widget-id="${id}" data-instance-id="${Date.now()}">${assets.html || ''}</div>`;

                    // Remove <style> tags from raw API CSS
                    css = (assets.css || '').replace(/<\/?style[^>]*>/gi, '');
                    js = assets.js || '';

                    // Scope the fresh API CSS to the initial temporary ID
                    const idMatch = html.match(/data-instance-id="([^"]+)"/);
                    if (idMatch && idMatch[1]) {
                        css = scopeCSS(css, idMatch[1]);
                    }

                    // Store in session
                    window.Redesignee.currentSession.loadedCSS = css;
                    window.Redesignee.currentSession.loadedJS = js; // We store Raw JS here, scoping happens on save
                    window.Redesignee.currentSession.isExisting = false;
                }

                initLocalCustomizer(html, window.Redesignee.currentSession.loadedCSS, js);
            });
        } else {
            loadLibraryView();
        }
    }
    // ---------------------------------------------------------
    // 4. PREVIEW ENGINE
    // ---------------------------------------------------------
    function initLocalCustomizer(html, css, js) {
        const $iframe = $('#live-preview');
        const doc = $iframe[0].contentDocument || $iframe[0].contentWindow.document;

        doc.open();
        doc.write('<!DOCTYPE html><html><head></head><body></body></html>');
        doc.close();

        const head = doc.head;
        const body = doc.body;

        const link = doc.createElement('link');
        link.id = 'rd-editor-icons';
        link.rel = 'stylesheet';
        link.href = 'https://fonts.googleapis.com/css?family=Material+Icons';
        head.appendChild(link);

        const fontLink = doc.createElement('link');
        fontLink.id = 'rd-preview-font';
        fontLink.rel = 'stylesheet';
        fontLink.href = 'https://fonts.googleapis.com/css2?family=Albert+Sans:wght@400;600;700&display=swap';
        head.appendChild(fontLink);

        const style = doc.createElement('style');
        style.textContent = css + `
            body { margin: 0; padding: 20px; font-family: 'Albert Sans', sans-serif; overflow-x: hidden; }
            [data-slide-template] { display: none !important; }
        `;
        head.appendChild(style);

        const wrapper = doc.createElement('div');
        wrapper.innerHTML = html;
        body.appendChild(wrapper);

        if(js) {
            const cleanJs = js.replace(/<script[^>]*>|<\/script>/gi, '');
            const script = doc.createElement('script');
            script.textContent = cleanJs;
            body.appendChild(script);
        }

        // Delay to ensure DOM is ready before attaching controls
        setTimeout(() => generateSidebarControls(doc), 200);
    }

    // ---------------------------------------------------------
    // 5. SIDEBAR GENERATOR
    // ---------------------------------------------------------
    function generateSidebarControls(doc) {
        const $sidebar = $('#customizer').empty();
        const $widget = $(doc.body).find('.redesignee-widget');

        if ($widget.length === 0) {
            $sidebar.html('<p>No widget found.</p>');
            return;
        }

        const $tabsNav = $(`
            <div class="rd-tabs-nav">
                <button class="rd-tab-link active" data-tab="content">Content</button>
                <button class="rd-tab-link" data-tab="style">Style</button>
            </div>
        `);
        const $contentTab = $('<div id="tab-content" class="rd-tab-pane active"></div>');
        const $styleTab = $('<div id="tab-style" class="rd-tab-pane"></div>');

        $sidebar.append($tabsNav, $contentTab, $styleTab);

        $tabsNav.find('button').on('click', function() {
            $('.rd-tab-link').removeClass('active'); $(this).addClass('active');
            $('.rd-tab-pane').removeClass('active'); $('#tab-' + $(this).data('tab')).addClass('active');
        });

        const editableAttrs = ['[data-text-editable]', '[data-richtext-editable]', '[data-video-url-editable]', '[data-link-editable]', '[data-color-editable]', '[data-background-color-editable]', '[data-font-size-editable]', '[data-family-editable]', '[data-extra-styles]', '[data-image-editable]', '[data-style-editable]'];

        // --- 1. STATIC CONTENT ---
        const $staticElements = $widget.find(editableAttrs.join(', ')).filter(function() {
            return $(this).closest('[data-slides-container]').length === 0;
        });

        const $extraStyleEls = $widget.find('[data-extra-styles]').addBack('[data-extra-styles]').filter(function() {
            return $(this).closest('[data-slides-container]').length === 0;
        });

        if ($staticElements.length > 0 || $extraStyleEls.length > 0) {
            const $staticContent = $('<div>');
            const $staticStyle = $('<div>');

            $staticElements.each(function() {
                generateStandardControls($(this), $staticContent, $staticStyle, false);
            });

            $extraStyleEls.each(function() {
                generateExtraStyleControls($(this), $staticStyle);
            });

            if($staticContent.children().length > 0) {
                $contentTab.append('<div class="rd-control-group"><div class="group-title">General Settings</div></div>');
                $contentTab.append($staticContent);
            }

            if($staticStyle.children().length > 0) {
                $styleTab.append('<div class="rd-control-group"><div class="group-title">Global Styles</div></div>');
                $styleTab.append($staticStyle);
            }
        }

        // --- 2. SLIDES ---
        const $slidesContainer = $widget.find('[data-slides-container]');
        if ($slidesContainer.length > 0) {
            const $slidesContentWrapper = $('<div class="rd-control-group"><h3>Items</h3></div>');
            const $slidesStyleWrapper = $('<div class="rd-control-group"><h3>Items Styling</h3></div>');

            $contentTab.append($slidesContentWrapper);
            $styleTab.append($slidesStyleWrapper);

            const renderSlides = () => {
                $slidesContentWrapper.find('.rd-accordion').remove();
                $slidesStyleWrapper.find('.rd-accordion').remove();
                $slidesContentWrapper.find('.rd-add-slide-btn').remove();

                const $contentAcc = $('<div class="rd-accordion"></div>');
                const $styleAcc = $('<div class="rd-accordion"></div>');

                $slidesContainer.find('[data-slide]').not('[data-slide-template]').each(function(i) {
                    const $slide = $(this);
                    const label = `Item ${i + 1}`;

                    const $cItem = createAccordionItem(label, true, () => {
                        if(confirm('Delete?')) { $slide.remove(); renderSlides(); }
                    });

                    const $sItem = createAccordionItem(label, false);

                    const nestedEditables = $slide.find(editableAttrs.join(', ')).filter(function() {
                        return $(this).closest('[data-slide]')[0] === $slide[0];
                    });

                    nestedEditables.each(function() {
                        generateStandardControls($(this), $cItem.find('.rd-acc-body'), $sItem.find('.rd-acc-body'), true);
                    });

                    generateExtraStyleControls($slide, $sItem.find('.rd-acc-body'));

                    if($cItem.find('.rd-acc-body').children().length === 0) {
                        $cItem.find('.rd-acc-body').append('<p class="rd-empty-msg">No editable content fields.</p>');
                    }
                    if($sItem.find('.rd-acc-body').children().length === 0) {
                        $sItem.find('.rd-acc-body').append('<p class="rd-empty-msg">No style options available.</p>');
                    }

                    $contentAcc.append($cItem);
                    $styleAcc.append($sItem);
                });

                $slidesContentWrapper.append($contentAcc);
                $slidesStyleWrapper.append($styleAcc);

                const $addBtn = $('<button class="rd-add-slide-btn">+ Add Item</button>');
                $addBtn.on('click', () => {
                    const $tpl = $slidesContainer.find('[data-slide-template]');
                    let $new = $tpl.length ? $tpl.clone().removeAttr('data-slide-template').attr('data-slide','').show() : $slidesContainer.find('[data-slide]').first().clone();
                    $new.css('display', '');
                    $slidesContainer.append($new);
                    renderSlides();
                });
                $slidesContentWrapper.append($addBtn);
            };
            renderSlides();
        }

        if ($styleTab.children().length === 0) {
            $styleTab.append('<p style="padding:15px;color:#999;">No style options available.</p>');
        }

        setTimeout(function() {
            safeInitColoris();
        }, 50);
    }

    function createAccordionItem(label, hasDelete, deleteCallback) {
        const $item = $(`
            <div class="rd-acc-item">
                <div class="rd-acc-header">
                    <span>${label}</span>
                    <div class="rd-acc-actions">
                        ${hasDelete ? '<span class="material-icons rd-delete-slide">delete</span>' : ''}
                        <span class="material-icons rd-chevron">expand_more</span>
                    </div>
                </div>
                <div class="rd-acc-body" style="display:none;"></div>
            </div>
        `);
        $item.find('.rd-acc-header').on('click', (e) => {
            if($(e.target).hasClass('rd-delete-slide')) return;
            $item.find('.rd-acc-body').slideToggle(200);
        });
        if(hasDelete) $item.find('.rd-delete-slide').on('click', deleteCallback);
        return $item;
    }

    function generateStandardControls($el, $contentParent, $styleParent, isInsideSlide) {
        const id = $el.attr('id') || `el-${Date.now()}-${Math.random().toString(36).substr(2, 5)}`;
        $el.attr('id', id);
        const label = $el.data('text-label') || 'Element';

        // CONTENT (Standard Text)
        if ($el.is('[data-text-editable]')) {
            const $ctrl = $(`<div class="rd-form-control"><label>${label}</label><textarea rows="2">${$el.text().trim()}</textarea></div>`);
            $ctrl.find('textarea').on('input', function() { $el.text($(this).val()); });
            $contentParent.append($ctrl);
        }

        // RICH TEXT EDITOR (FIXED) ---
        if ($el.is('[data-richtext-editable]')) {
            const $ctrl = $(`<div class="rd-form-control"><label>${label} (Rich Text)</label><textarea></textarea></div>`);
            $contentParent.append($ctrl);
            const $textarea = $ctrl.find('textarea');

            // Get initial HTML content
            const initialContent = $el.html();

            if($.fn.trumbowyg) {
                // Safety check for SVG Path to prevent JS errors
                const svgPath = (typeof window.RedesigneeConfig !== 'undefined' && window.RedesigneeConfig.svgPath)
                    ? window.RedesigneeConfig.svgPath
                    : '';

                // 1. Initialize
                $textarea.trumbowyg({
                    btns: [['bold', 'italic'], ['link'], ['unorderedList', 'orderedList'], ['viewHTML']],
                    svgPath: svgPath,
                    autogrow: true
                });

                // 2. Explicitly set content via API (Fixes empty editor bug)
                $textarea.trumbowyg('html', initialContent);

                // 3. Bind events (Adding 'tbwblur' ensures updates happen even if focus is lost)
                $textarea.on('tbwchange tbwblur', function() {
                    $el.html($textarea.trumbowyg('html'));
                });
            } else {
                // Fallback if Trumbowyg fails to load
                $textarea.val(initialContent);
                $textarea.on('input', function() { $el.html($(this).val()); });
            }
        }

        // VIDEO URL
        if ($el.is('[data-video-url-editable]')) {
            let val = $el.attr('src') || '';
            const ytMatch = val.match(/youtube\.com\/embed\/([-_a-zA-Z0-9]+)/);
            if(ytMatch) val = `https://youtu.be/${ytMatch[1]}`;
            const $ctrl = $(`<div class="rd-form-control"><label>${label}</label><input type="text" value="${val}"></div>`);
            $ctrl.find('input').on('input', function() {
                const currentEl = $el[0].ownerDocument.getElementById(id);
                if(currentEl) updateVideoURL($(this).val(), currentEl);
            });
            $contentParent.append($ctrl);
        }

        // IMAGE
        if ($el.is('[data-image-editable]')) {
            const $ctrl = $(`<div class="rd-form-control"><label>${label}</label><div style="display:flex;gap:5px;"><input type="text" value="${$el.attr('src')}" readonly style="background:#f9f9f9;color:#999;"><button class="rd-media-btn">Change</button></div></div>`);
            $ctrl.find('.rd-media-btn').on('click', (e) => {
                e.preventDefault();
                openWPMediaFrame((url) => { $el.attr('src', url); $ctrl.find('input').val(url); });
            });
            $contentParent.append($ctrl);
        }

        // LINKS
        if ($el.is('[data-link-editable]')) {
            const $ctrl = $(`<div class="rd-form-control"><label>Link</label><input type="text" value="${$el.attr('href')}"></div>`);
            $ctrl.find('input').on('input', function() { $el.attr('href', $(this).val()); });
            $contentParent.append($ctrl);
        }

        // STYLES
        if ($el.is('[data-color-editable], [data-background-color-editable], [data-font-size-editable], [data-family-editable]')) {
            const $group = $(`<div class="rd-style-subgroup"></div>`);
            $group.append(`<div class="rd-control-sublabel">${label}</div>`);

            if($el.is('[data-color-editable]')) {
                const val = $el.css('color');
                const $ctrl = createColorControl('Text Color', val, (v) => $el.css('color', v));
                $group.append($ctrl);
            }
            if($el.is('[data-background-color-editable]')) {
                const val = $el.css('background-color');
                const $ctrl = createColorControl('Background', val, (v) => $el.css('background-color', v));
                $group.append($ctrl);
            }
            if($el.is('[data-font-size-editable]')) {
                const val = $el.css('font-size');
                const $ctrl = $(`<div class="rd-form-control"><label>Font Size</label><input type="text" value="${val}"></div>`);
                $ctrl.find('input').on('input', function() { $el.css('font-size', $(this).val()); });
                $group.append($ctrl);
            }
            if($el.is('[data-family-editable]')) {
                let current = $el[0].style.fontFamily ? $el[0].style.fontFamily.replace(/['"]/g, '').split(',')[0].trim() : "";
                if(current && googleFonts.includes(current)) loadGoogleFontInIframe(current);

                const $ctrl = $(`<div class="rd-form-control"><label>Font Family</label><select class="form-control"><option value="">Default (Inherit)</option></select></div>`);
                const $sel = $ctrl.find('select');
                googleFonts.forEach(f => $sel.append(`<option value="${f}" ${f === current ? 'selected' : ''}>${f}</option>`));
                $sel.on('change', function() {
                    const val = $(this).val();
                    if(val) { $el.css('font-family', val); loadGoogleFontInIframe(val); } else { $el.css('font-family', ''); }
                });
                $group.append($ctrl);
            }
            $styleParent.append($group);
        }
    }
    function generateExtraStyleControls($el, $container) {
        if(!$el.is('[data-extra-styles]')) return;
        try {
            const extra = JSON.parse($el.attr('data-extra-styles'));
            Object.keys(extra).forEach(category => {
                const props = extra[category];
                if (typeof props === 'object') {
                    Object.entries(props).forEach(([key, val]) => {
                        const label = key.replace(/--/g,'').replace(/-/g,' ').replace(/rd /g, '').trim();
                        const isColor = val.includes('#') || val.includes('rgb') || val.includes('hsl');
                        let $ctrl;
                        if(isColor) {
                            $ctrl = createColorControl(label, val, (v) => $el[0].style.setProperty(key, v));
                        } else {
                            $ctrl = $(`<div class="rd-form-control"><label>${label}</label><input type="text" value="${val}"></div>`);
                            $ctrl.find('input').on('input', function() { $el[0].style.setProperty(key, $(this).val()); });
                        }
                        $container.append($ctrl);
                    });
                }
            });
        } catch(e) {}
    }

    function createColorControl(label, val, callback) {
        if (val && val.includes('rgba') && val.includes(', 0)')) val = '';
        const $ctrl = $(`<div class="rd-form-control"><label>${label}</label><div class="color-input-wrapper"><input type="text" class="coloris" value="${val}"></div></div>`);
        $ctrl.find('input').on('input change', function() { callback($(this).val()); });
        return $ctrl;
    }

    function updateVideoURL(newUrl, targetElement) {
        if (!targetElement) return;
        const parent = targetElement.parentNode;
        const doc = targetElement.ownerDocument;
        const ytRegex = /(?:https?:\/\/)?(?:www\.)?(?:youtube\.com\/(?:[^\/\n\s]+\/\S+\/|(?:v|e(?:mbed)?)\/|\S*?[?&]v=)|youtu\.be\/)([a-zA-Z0-9_-]{11})/;
        const match = newUrl.match(ytRegex);

        let newEl;
        if (match) {
            const vid = match[1];
            newEl = doc.createElement("iframe");
            newEl.src = `https://www.youtube.com/embed/${vid}?modestbranding=1&autoplay=1&mute=1&controls=0&loop=1&playlist=${vid}`;
            newEl.allow = "accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share";
            newEl.setAttribute("referrerpolicy", "strict-origin-when-cross-origin");
            newEl.allowFullscreen = true;
            newEl.frameBorder = "0";
        } else {
            newEl = doc.createElement("video");
            newEl.src = newUrl;
            newEl.playsInline = true; newEl.muted = true; newEl.autoplay = true; newEl.loop = true;
            newEl.innerHTML = "Your browser does not support the video tag.";
        }
        newEl.style.cssText = targetElement.style.cssText;
        newEl.style.width = "100%"; newEl.style.height = "100%";
        newEl.id = targetElement.id;
        newEl.className = targetElement.className;

        Array.from(targetElement.attributes).forEach(attr => {
            if(attr.name.startsWith('data-')) newEl.setAttribute(attr.name, attr.value);
        });
        newEl.setAttribute('data-video-url-editable', '');
        parent.replaceChild(newEl, targetElement);
    }

    function renderGrid(widgets, categories, userPlan) {
        const $grid = $('#rd-grid').empty();
        const $filters = $('#rd-filters').empty();

        $('#rd-status-bar').remove();
        let statusHtml = '';
        if (userPlan === 'free') {
            statusHtml = `<div id="rd-status-bar" class="rd-status-bar free"><span class="material-icons">info</span> Free Version &nbsp;&mdash;&nbsp; <a href="https://redesignee.com/premium" target="_blank">Upgrade to unlock all</a></div>`;
        } else {
            statusHtml = `<div id="rd-status-bar" class="rd-status-bar"><span class="material-icons">workspace_premium</span> Premium Member</div>`;
        }
        $('.rd-content').prepend(statusHtml);

        const $allBtn = $('<button class="rd-filter-btn active" data-cat="all">All Widgets</button>');
        $allBtn.on('click', function() {
            $('.rd-filter-btn').removeClass('active'); $(this).addClass('active');
            $('.sqs-widget-item').show();
            $('.rd-content').scrollTop(0);
        });
        $filters.append($allBtn);
        $filters.append('<hr style="margin:10px 0; border:0; border-top:1px solid #eee;">');

        categories.forEach(cat => {
            const catName = (typeof cat === 'string') ? cat : (cat.name || 'Category');
            const catId = cat.id || slugify(catName);
            const btn = $('<button class="rd-filter-btn" data-cat="' + catId + '">' + catName + '</button>');
            btn.on('click', function() {
                $('.rd-filter-btn').removeClass('active'); $(this).addClass('active');
                if($(this).data('cat') === 'all') { $('.sqs-widget-item').show(); }
                else {
                    $('.sqs-widget-item').hide();
                    $('.sqs-widget-item').each(function() { if($(this).data('cat') == catId) $(this).show(); });
                }
                $('.rd-content').scrollTop(0);
            });
            $filters.append(btn);
        });

        widgets.forEach(w => {
            let thumb = '<div style="padding:40px;color:#ccc;">Code</div>';
            if(w.demo_link) thumb = `<iframe src="${w.demo_link}" class="preview-frame" scrolling="no"></iframe>`;
            else if (w.thumbnail_url) thumb = `<img src="${w.thumbnail_url}" style="object-fit:contain;width:100%;height:100%;">`;

            const catId = w.category_id || 'uncategorized';

            let isLocked = (w.is_premium && userPlan === 'free');
            let badge = isLocked ? '<span class="rd-badge premium">Premium</span>' : '';
            let btnLabel = isLocked ? 'Upgrade' : 'Add';
            let itemClass = isLocked ? 'sqs-widget-item locked' : 'sqs-widget-item';

            const item = $(`
                <div class="${itemClass}" data-cat="${catId}">
                    <div class="sqs-widget-preview">
                        ${thumb}
                        ${badge}
                        <div class="sqs-widget-overlay">
                            <button class="btn-add-widget action-btn" data-id="${w.id}">${btnLabel}</button>
                        </div>
                    </div>
                    <div class="sqs-widget-info">${w.name}</div>
                </div>`);

            item.find('.action-btn').on('click', function() {
                if(isLocked) {
                    window.open('https://redesignee.com/pricing', '_blank');
                } else {
                    const widgetId = $(this).data('id');
                    if (window.Redesignee.currentSession.onSave) { $('#redesignee-modal-wrapper').fadeOut(); directInsert(widgetId); } else { window.Redesignee.currentSession.widgetId = widgetId; loadEditView(); }
                }
            });
            $grid.append(item);
        });
    }

    function cleanCodeForPreview(code) { return code.replace(/<script\b[^>]*>([\s\S]*?)<\/script>/gim, ""); }

    function fetchWidgetData(id, callback) {
        const apiKey = window.RedesigneeConfig ? window.RedesigneeConfig.apiKey : null;
        let url = apiKey ? `${RD_Config.apiBase}/api/widget-assets/${id}` : `${RD_Config.apiBase}/api/public-widget-assets/${id}`;
        let headers = apiKey ? { 'Authorization': 'Bearer ' + apiKey } : {};
        $.ajax({ url: url, headers: headers, success: callback });
    }

    function fetchLibrary() {
        if($('#rd-grid').children().length > 0) return;
        const apiKey = window.RedesigneeConfig ? window.RedesigneeConfig.apiKey : null;
        let url = apiKey ? `${RD_Config.apiBase}/api/squarespace-widgets` : `${RD_Config.apiBase}/api/public-squarespace-widgets`;
        let headers = apiKey ? { 'Authorization': 'Bearer ' + apiKey } : {};
        $.ajax({
            url: url, headers: headers,
            success: function(data) { renderGrid(data.widgets||[], data.categories||[], data.user_plan || 'free'); }
        });
    }

    function saveWidget() {
        const $iframe = $('#live-preview');
        const doc = $iframe[0].contentDocument;
        const $widget = $(doc.body).find('.redesignee-widget');

        // 1. Get the CURRENT Instance ID (Don't generate a new one!)
        // This preserves the link between your custom CSS and the HTML
        let instanceId = $widget.attr('data-instance-id');
        if (!instanceId) instanceId = generateUniqueId(); // Fallback

        const widgetId = window.Redesignee.currentSession.widgetId;

        // 2. Get content from memory (NOT API)
        // We use the CSS/JS we loaded at the start (which includes your manual edits)
        let currentCSS = window.Redesignee.currentSession.loadedCSS || '';
        let currentJS = window.Redesignee.currentSession.loadedJS || '';

        // 3. Process HTML (From Visual Editor)
        let finalHtml = $widget[0].outerHTML;

        // If this is a NEW widget, we need to apply scoping now.
        // If it's EXISTING, the CSS is likely already scoped to this ID.
        if (!window.Redesignee.currentSession.isExisting) {
            // For new widgets, we might need to re-scope JS if it wasn't done yet
            // But usually, we rely on the server assets for the first save.
            // To be safe, we can re-run scoping helper just to ensure ID matches
            currentJS = scopeJS(currentJS, finalHtml, instanceId);
        }

        // 4. Handle Fonts
        const fonts = [];
        $(doc.head).find('link[href*="fonts.googleapis"]').not('#rd-editor-icons, #rd-preview-font').each(function() {
            fonts.push(this.outerHTML);
        });

        const finalCode = `${fonts.join('\n')}\n<style>${currentCSS}</style>\n${finalHtml}\n<script>${currentJS}</script>`;

        if (window.Redesignee.currentSession.onSave) {
            window.Redesignee.currentSession.onSave(finalCode, widgetId);
        }
        $('#redesignee-modal-wrapper').fadeOut();
    }

    function directInsert(id) {
        fetchWidgetData(id, function(assets) {
            const uniqueId = generateUniqueId();

            // 1. Prepare HTML
            let rawHtml = `<div class="redesignee-widget" data-widget-id="${id}" data-instance-id="${uniqueId}">${assets.html || ''}</div>`;
            const scopedHtml = scopeHTML(rawHtml, uniqueId);

            // 2. Prepare CSS
            const rawCss = (assets.css || '').replace(/<\/?style[^>]*>/gi, '');
            const scopedCss = scopeCSS(rawCss, uniqueId);

            // 3. Prepare JS (NEW STEP)
            const scopedJs = scopeJS(assets.js || '', assets.html || '', uniqueId);

            window.Redesignee.currentSession.onSave(`<style>${scopedCss}</style>\n${scopedHtml}\n${scopedJs}`, id);
        });
    }

    function prepareModalUI() {
        const btnText = (window.Redesignee.currentSession.onSave) ? 'Update Widget' : 'Get Code';
        $('#rd-save-btn').text(btnText);
    }

    function openWPMediaFrame(callback) {
        if (typeof wp === 'undefined' || !wp.media) { alert('Media Library not available'); return; }
        const frame = wp.media({ title: 'Select Image', button: { text: 'Use Image' }, multiple: false });
        frame.on('select', () => { const att = frame.state().get('selection').first().toJSON(); callback(att.url); });
        frame.open();
    }

    function loadGoogleFontInIframe(font) {
        const doc = $('#live-preview')[0].contentDocument;
        const id = 'font-' + font.replace(/\s+/g, '-');
        if(!doc.getElementById(id)) {
            const link = doc.createElement('link'); link.id = id; link.rel = 'stylesheet';
            link.href = 'https://fonts.googleapis.com/css2?family=' + encodeURIComponent(font) + '&display=swap';
            doc.head.appendChild(link);
        }
    }

    function bindGlobalEvents() {
        $(document).on('click', '#rd-close-btn', function() { $('#redesignee-modal-wrapper').fadeOut(); });
        $(document).on('click', '#rd-back-btn', function() { loadLibraryView(); });
        $(document).on('click', '#rd-save-btn', function(e) { e.preventDefault(); saveWidget(); });
    }

    function buildDOM() {
        $('body').append(`
        <div id="redesignee-modal-wrapper" style="display:none;">
            <div class="rd-modal-window">
                <div class="rd-header"><div style="display:flex;align-items:center;gap:15px;"><h3>Redesignee</h3><button id="rd-back-btn" class="rd-back-btn">&larr; Back</button></div><button id="rd-close-btn" class="rd-close-btn">&times;</button></div>
                <div class="rd-body">
                    <div id="rd-library-view"><aside class="rd-sidebar" id="rd-filters">Loading...</aside><div class="rd-content"><div id="rd-grid" class="rd-grid"></div></div></div>
                    <div id="rd-editor-view"><div class="preview-area"><iframe id="live-preview"></iframe></div><aside class="sidebar-area"><div id="customizer"></div><div class="footer-area"><button id="rd-save-btn">Get Code</button></div></aside></div>
                </div>
            </div>
        </div>`);
    }

    if($('#rd-global-styles').length===0) {
        $('head').append(`<style id="rd-global-styles">
            #redesignee-modal-wrapper{position:fixed;top:0;left:0;width:100%;height:100%;z-index:99;background:rgba(0,0,0,0.85);display:none;justify-content:center;align-items:center;font-family:'Albert Sans',sans-serif}
            .rd-modal-window{width:95%;height:90%;background:#f4f6f8;border-radius:12px;display:flex;flex-direction:column;overflow:hidden;box-shadow:0 25px 50px rgba(0,0,0,0.5)}
            .rd-header{background:#fff;padding:15px 25px;border-bottom:1px solid #eee;display:flex;justify-content:space-between;align-items:center;height:60px}
            .rd-close-btn{border:none;background:transparent;font-size:24px;cursor:pointer;color:#888;display:flex;align-items:center;justify-content:center;width:32px;height:32px;border-radius:50%;transition:0.2s}
            .rd-close-btn:hover{background:#eee;color:#333}
            .rd-back-btn{border:1px solid #ddd;background:#fff;padding:6px 12px;border-radius:6px;cursor:pointer;font-size:13px;font-weight:500;display:flex;align-items:center;gap:5px;transition:0.2s}
            .rd-back-btn:hover{background:#f5f5f5;border-color:#ccc}
            .rd-body{display:flex;height:100%;overflow:hidden}
            #rd-library-view,#rd-editor-view{display:none;width:100%;height:100%}
            .rd-sidebar{width:240px;background:#fff;border-right:1px solid #eee;padding:20px;overflow-y:auto}
            .rd-content{flex:1;padding:30px;overflow-y:auto;background:#f9fafb}
            .rd-grid{display:grid;grid-template-columns:repeat(auto-fill,minmax(280px,1fr));gap:25px;padding-bottom:50px}
            .preview-area{flex:1;background:#eef2f5;display:flex;align-items:center;justify-content:center;padding:20px}
            #live-preview{width:100%;height:100%;border:1px solid #ddd;background:#fff;border-radius:8px}
            .sidebar-area{width:350px;background:#fff;border-left:1px solid #ddd;display:flex;flex-direction:column}
            #customizer{flex:1;overflow-y:auto;padding:0;padding-bottom:60px}
            .footer-area{padding:20px;border-top:1px solid #eee;background:#fff}
            #rd-save-btn{width:100%;background:#111;color:#fff;padding:14px;border:none;border-radius:6px;font-weight:600;cursor:pointer;transition:0.2s}
            #rd-save-btn:hover{background:#333}
            #customizer .rd-success-msg,#customizer h3:contains("Widget Saved"),#customizer p:contains("Copy the code"),#customizer textarea{display:none!important}
            .rd-tabs-nav{display:flex;border-bottom:1px solid #eee;background:#fff}
            .rd-tab-link{flex:1;background:none;border:none;padding:15px;font-weight:600;cursor:pointer;color:#666;border-bottom:2px solid transparent;font-family:inherit}
            .rd-tab-link.active{color:#764fe3;border-bottom-color:#764fe3;color:#764fe3}
            .rd-tab-pane{display:none;padding:20px}
            .rd-tab-pane.active{display:block}
            .rd-control-group{margin-bottom:20px}
            .group-title{font-size:12px;font-weight:700;color:#999;text-transform:uppercase;margin-bottom:10px}
            .rd-control-group h3 { font-size:16px; margin-bottom:15px; color:#333; }
            .rd-acc-item{border:1px solid #eee;border-radius:6px;margin-bottom:8px;background:#fff;overflow:hidden}
            .rd-acc-header{padding:12px;background:#f9f9f9;cursor:pointer;display:flex;justify-content:space-between;align-items:center;font-weight:600;font-size:14px}
            .rd-acc-header:hover{background:#f0f0f0}
            .rd-acc-header.active { background: #f0f0f0; }
            .rd-acc-actions{display:flex;gap:8px;align-items:center}
            .rd-delete-slide{color:#ff4d4d;cursor:pointer;font-size:18px}
            .rd-acc-body{padding:15px;border-top:1px solid #eee}
            .rd-add-slide-btn{width:100%;padding:10px;background:#f0ebff;color:#764fe3;border:1px dashed #764fe3;border-radius:6px;cursor:pointer;font-weight:600;margin-top:10px}
            .rd-form-control{margin-bottom:15px}
            .rd-form-control label{display:block;font-size:12px;font-weight:600;margin-bottom:5px;color:#555}
            .rd-form-control input,.rd-form-control textarea,.rd-form-control select{width:100%;padding:8px;border:1px solid #ddd;border-radius:4px;box-sizing:border-box;font-family:inherit}
            .rd-form-control select { height: auto; }
            .rd-media-btn{background:#eee;border:1px solid #ddd;padding:5px 10px;cursor:pointer;border-radius:4px;font-size:12px}
            .color-input-wrapper input{padding-left:35px}
            .clr-picker { z-index: 9999999 !important; }
            .sqs-widget-item{background:#fff;border-radius:8px;overflow:hidden;box-shadow:0 4px 10px rgba(0,0,0,0.05);position:relative}
            .sqs-widget-preview{width:100%;aspect-ratio:4/3;background:#f9f9f9;position:relative}
            .sqs-widget-preview iframe.preview-frame { width: 100%; height: 100%; border: none; pointer-events: none; }
            .sqs-widget-preview img{width:100%;height:100%;object-fit:contain}
            .sqs-widget-info{padding:15px;text-align:center;font-weight:600}
            .btn-add-widget{position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);background:#764fe3;color:#fff;border:none;padding:10px 20px;border-radius:20px;cursor:pointer;opacity:0;transition:opacity 0.2s}
            .sqs-widget-item:hover .btn-add-widget{opacity:1}
            .rd-filter-btn{display:block;width:100%;text-align:left;background:#fff;border:1px solid #eee;border-radius:8px;padding:10px 15px;margin-bottom:8px;font-size:14px;font-weight:500;cursor:pointer;font-family:inherit}
            .rd-filter-btn.active{background:#764fe3;color:#fff;border-color:#764fe3}
            .rd-control-sublabel { font-size:12px; font-weight:700; color:#333; margin-top:15px; margin-bottom:10px; padding-bottom:5px; border-bottom:1px solid #eee; text-transform: uppercase; letter-spacing: 0.5px; }
            .rd-status-bar { background: #f0ebff; color: #764fe3; padding: 12px; margin-bottom: 20px; border-radius: 8px; text-align: center; font-weight: 600; font-size: 14px; display: flex; align-items: center; justify-content: center; gap: 8px; }
            .rd-status-bar.free { background: #fff4e5; color: #b76e00; border: 1px solid #ffe0b2; }
            .rd-badge { position: absolute; top: 10px; right: 10px; padding: 4px 8px; border-radius: 4px; font-size: 11px; font-weight: 700; color: #fff; z-index: 2; text-transform: uppercase; }
            .rd-badge.premium { background: #111; }
            .sqs-widget-item.locked .sqs-widget-preview { opacity: 0.8; }
            .sqs-widget-item.locked .btn-add-widget { background: #333; }
            /* Rich Text Area CSS */
            .trumbowyg-box { margin-bottom: 15px; border: 1px solid #ddd; }
            .trumbowyg-button-pane { background: #f9f9f9; border-bottom: 1px solid #ddd; }
            .trumbowyg-editor { background: #fff; min-height: 100px; padding: 10px; font-size: 14px; }
            .rd-empty-msg { font-size: 13px; color: #999; font-style: italic; padding: 10px; text-align: center; }
        </style>`);
    }

    const isElementor = window.location.href.indexOf('action=elementor') !== -1 || $('body').hasClass('elementor-editor-active');
    const isAdmin = window.location.href.indexOf('wp-admin') !== -1;
    if (!isElementor && !isAdmin) {
        $(document).ready(function() {
            initModalEnvironment();
            $('body').append('<div id="redesignee-floating-trigger"><button id="redesignee-btn" style="background:#764fe3;color:#fff;border:none;border-radius:50px;padding:12px 24px;cursor:pointer;position:fixed;bottom:30px;right:30px;z-index:99999;">Add Widget</button></div>');
            $('body').on('click', '#redesignee-btn', function() { window.Redesignee.openModal({ mode: 'add' }); });
        });
    }


// ---------------------------------------------------------
    // 6. SCOPING HELPERS (Robust Regex Version)
    // ---------------------------------------------------------

    function generateUniqueId() {
        return Math.random().toString(36).substr(2, 9);
    }

    function scopeCSS(css, uniqueId) {
        const wrapperClass = `.redesignee-widget[data-instance-id="${uniqueId}"]`;

        // 1. Prefix every selector with the unique widget wrapper
        let scoped = css.replace(/([^\r\n,{}]+)(,(?=[^}]*{)|\s*{)/g, function(match, selector, end) {
            selector = selector.trim();

            if (/^(from|to|\d+%?)$/.test(selector)) {
                return match;
            }

            // Skip at-rules (keyframes, media, import, etc.)
            if (selector.startsWith('@') || selector.indexOf('%') > -1) return match;

            // FIX for :root -> Replace :root with the wrapper class itself
            if (selector.includes(':root')) {
                return selector.replace(':root', wrapperClass) + end;
            }

            // Standard selector scoping
            if (selector.indexOf('.redesignee-widget') === -1) {
                return `${wrapperClass} ${selector} ${end}`;
            }
            return match;
        });

        // 2. Suffix ID selectors (#my-id -> #my-id_UNIQUE)
        scoped = scoped.replace(/#([a-zA-Z0-9_-]+)(?=[.:,\s{>+~])/g, `#$1_${uniqueId}`);

        return scoped;
    }

    function scopeHTML(html, uniqueId) {
        let scoped = html;

        // 1. Force Clean Instance ID (Fixes the "doubled ID" bug)
        if (scoped.indexOf('data-instance-id=') !== -1) {
            scoped = scoped.replace(/data-instance-id="[^"]*"/, `data-instance-id="${uniqueId}"`);
        } else {
            scoped = scoped.replace('class="redesignee-widget"', `class="redesignee-widget" data-instance-id="${uniqueId}"`);
        }

        // 2. Refresh Element IDs (Strips old suffix, adds new)

        scoped = scoped.replace(/(\s)id="([a-zA-Z0-9-]+)(_[a-z0-9]+)?"/g, `$1id="$2_${uniqueId}"`);

        // 3. Refresh Links (href="#baseName")
        scoped = scoped.replace(/(\s)href="#([a-zA-Z0-9-]+)(_[a-z0-9]+)?"/g, `$1href="#$2_${uniqueId}"`);

        return scoped;
    }
    function scopeJS(jsCode, rawHtmlFromApi, uniqueId) {
        if (!jsCode) return '';
        let scopedJS = jsCode;

        // 1. Find all IDs defined in the original/fresh HTML
        const idRegex = /\bid="([a-zA-Z0-9_-]+)"/g;
        let match;
        const idsFound = [];

        while ((match = idRegex.exec(rawHtmlFromApi)) !== null) {
            idsFound.push(match[1]); // e.g. "rd-text-tabs-2"
        }

        // 2. Replace those strings in the JS
        idsFound.forEach(baseId => {
            const newId = `${baseId}_${uniqueId}`;

            scopedJS = scopedJS.split(`'${baseId}'`).join(`'${newId}'`);
            scopedJS = scopedJS.split(`"${baseId}"`).join(`"${newId}"`);
            scopedJS = scopedJS.split(`\`${baseId}\``).join(`\`${newId}\``);

            scopedJS = scopedJS.split(`'#${baseId}'`).join(`'#${newId}'`);
            scopedJS = scopedJS.split(`"#${baseId}"`).join(`"#${newId}"`);
            scopedJS = scopedJS.split(`\`#${baseId}\``).join(`\`#${newId}\``);
        });

        return scopedJS;
    }
})(jQuery);