<?php
/**
 * Main class for the Recently Viewed Products Popup plugin.
 * 
 */

 defined( 'ABSPATH' ) || exit;

class WRVPP_Main {

    protected $plugin_name;

    protected $version;

    public function __construct() {
        if ( ! defined( 'WRVPP_PLUGIN_VERSION' ) ) {
            define( 'WRVPP_PLUGIN_VERSION', '1.0.0' );
        }
        $this->version = WRVPP_PLUGIN_VERSION;
        $this->plugin_name = 'recently-viewed-products-popup-woo';

        $this->init();
    }

    public function init() {

        // Show the popup on specified pages.
        add_action( 'template_redirect', array( $this, 'show_popup' ) );

        // Register scripts and styles for the popup.
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

        // AJAX handler to fetch recently viewed products.
        add_action( 'wp_ajax_wrvpp_get_recently_viewed', array( $this, 'ajax_get_recently_viewed' ) );
        add_action( 'wp_ajax_nopriv_wrvpp_get_recently_viewed', array( $this, 'ajax_get_recently_viewed' ) );

        if ( is_admin() ) {
            require_once WRVPP_PLUGIN_DIR . 'includes/class-wrvpp-settings.php';
            // Initialize the settings page.
            $wrvpp_settings = new WRVPP_Settings();
            $wrvpp_settings->init();

            add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
        }
    }
    
    /**
     * Show the popup on specified WooCommerce pages.
     */
    public function show_popup() {
        if ( get_option( 'wrvpp_enable_popup', 'yes' ) !== 'yes' ) {
            return;
        }

        $display_on_pages = get_option( 'wrvpp_display_on', array( 'is_product' ) );
        if ( empty( $display_on_pages ) || ! is_array( $display_on_pages ) ) {
            return;
        }

        // Show on all pages
        if ( in_array( 'all_pages', $display_on_pages ) ) {
            add_action( 'wp_footer', array( $this, 'display_popup' ), 50 );
            return;
        }

        if ( in_array( 'is_product', $display_on_pages ) ) {
            add_action( 'woocommerce_after_single_product_summary', array( $this, 'display_popup' ), 50 );
        }

        if ( in_array( 'is_shop', $display_on_pages ) && is_shop() && ! is_product_category() ) {
            add_action( 'woocommerce_after_shop_loop', array( $this, 'display_popup' ), 50 );
        }

        if ( in_array( 'is_product_category', $display_on_pages ) && is_product_category() && ! is_shop() ) {
            add_action( 'woocommerce_after_shop_loop', array( $this, 'display_popup' ), 50 );
        }

        if ( in_array( 'is_cart', $display_on_pages ) ) {
            add_action( 'woocommerce_after_cart', array( $this, 'display_popup' ), 50 );
        }

    }
    
    /**
     * Enqueue scripts and styles for the popup.
     */
    public function enqueue_scripts() {
        $min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

        $product_id = ( is_product() && get_queried_object_id() ) ? get_queried_object_id() : 0;

        wp_register_script( $this->plugin_name . '-tracking', WRVPP_PLUGIN_URL . 'assets/js/wrvpp-tracking' . $min . '.js', array( 'jquery' ), $this->version, true );
        wp_localize_script( $this->plugin_name . '-tracking', 'wrvpp_tracking_params', array(
            'product_id'    => $product_id,
            'track_limit'   => get_option( 'wrvpp_limit_products', 15 ),
        ) );
        wp_register_script( 'slick', WRVPP_PLUGIN_URL . 'assets/js/slick.min.js', array( 'jquery' ), '1.8.1', true );
        wp_register_style( $this->plugin_name . '-style', WRVPP_PLUGIN_URL. 'assets/css/wrvpp-style' . $min . '.css', array(), $this->version );
        wp_register_script( $this->plugin_name . '-main', WRVPP_PLUGIN_URL . 'assets/js/wrvpp-main' . $min . '.js', array( 'jquery', 'slick' ), $this->version, true );
        wp_localize_script( $this->plugin_name . '-main', 'wrvpp_params', array(
            'ajax_url'      => admin_url( 'admin-ajax.php' ),
            'nonce'         => wp_create_nonce( 'wrvpp_nonce' ),
            'product_id'    => $product_id,
            'cookie_name'   => 'wrvpp_recently_viewed_products',
            'cookie_days'   => 30,
        ) );

        if ( $product_id && get_option( 'wrvpp_enable_popup', 'yes' ) === 'yes' ) {
            wp_enqueue_script( $this->plugin_name . '-tracking' );
        }
    
        if ( $this->is_popup_enabled() ) {
            wp_enqueue_script( $this->plugin_name . '-main' );
            wp_enqueue_script( 'slick' );
            wp_enqueue_style( $this->plugin_name . '-style' );
            // Enqueue WooCommerce styles on non-WooCommerce pages
            if ( ! is_woocommerce() && ! is_product() && ! is_product_category() && ! is_shop() && ! is_cart() ) {
                wp_enqueue_style( 'woocommerce-general' );
                wp_enqueue_style( 'woocommerce-layout' );
            }
        }
    }

    /**
     * Enqueue admin scripts and styles.
     */
    public function admin_enqueue_scripts( $hook ) {
        if ( 'woocommerce_page_wc-settings' !== $hook ) {
            return;
        }

        // Only load on the WRVPP settings section
        $tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : '';
        $section = isset( $_GET['section'] ) ? sanitize_text_field( wp_unslash( $_GET['section'] ) ) : '';

        if ( 'products' !== $tab || 'wrvpp' !== $section ) {
            return;
        }

        wp_enqueue_script( $this->plugin_name . '-admin', WRVPP_PLUGIN_URL . 'assets/js/wrvpp-admin-settings.js', array( 'jquery', 'select2' ), $this->version, true );
    }

    /**
     * Display the popup with recently viewed products.
     */
    public function display_popup() {

        if ( ! isset( $_COOKIE['wrvpp_recently_viewed_products'] ) ) {
            return;
        }

        $viewed_products = explode( '|', sanitize_text_field( wp_unslash( $_COOKIE['wrvpp_recently_viewed_products'] ) ) );
        $viewed_products =  array_filter( array_map( 'absint', $viewed_products ) );

        if ( ! empty( $viewed_products ) && is_product() ) {
            // If on a single product page, remove the current product from the viewed products.
            global $post;
            $viewed_products = array_diff( $viewed_products, array( $post->ID ) );
             
        } 

        if ( empty( $viewed_products ) ) {
            return;
        }
        
        // Query the products.
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'post__in' => $viewed_products,
            'posts_per_page' => -1,
            'orderby' => 'post__in',
        );

        $query = new WP_Query( $args );

        if ( ! $query->have_posts() ) {
            return;
        }

        ob_start();

        // Include the popup template.
        include WRVPP_PLUGIN_DIR . 'templates/popup-template.php';

        // Include the modal template.
        include WRVPP_PLUGIN_DIR . 'templates/modal-template.php';

        wp_reset_postdata();

        echo wp_kses_post( ob_get_clean() );
    }

    /**
     * Handle AJAX request to get recently viewed products.
     */
    public function ajax_get_recently_viewed() {
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'wrvpp_nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Invalid nonce.', 'recently-viewed-products-popup-woo' ) ) );
        }

        if ( ! isset( $_COOKIE['wrvpp_recently_viewed_products'] ) ) {
            $wrvpp_recently_viewed_products = isset( $_POST['viewed_products'] ) ? sanitize_text_field( wp_unslash( $_POST['viewed_products'] ) ) : '';
            if ( empty( $wrvpp_recently_viewed_products ) ) {
                wp_send_json_error( array( 'message' => __( 'No recently viewed products found.', 'recently-viewed-products-popup-woo' ) ) );
            }
        }

        $viewed_products = explode( '|', sanitize_text_field( wp_unslash( $_COOKIE['wrvpp_recently_viewed_products'] ) ) );
        $viewed_products = array_filter( array_map( 'absint', $viewed_products ) );

        if ( isset( $_POST['product_id'] ) && is_numeric( $_POST['product_id'] ) && in_array( absint( $_POST['product_id'] ), $viewed_products ) ) {
            $current_product_id = absint( $_POST['product_id'] );
            // Remove the current product from the viewed products.
            $viewed_products = array_diff( $viewed_products, array( $current_product_id ) );
        } 

        if ( empty( $viewed_products ) ) {
            wp_send_json_error( array( 'message' => __( 'No recently viewed products found.', 'recently-viewed-products-popup-woo' ) ) );
        }

        // Query the products.
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'post__in' => $viewed_products,
            'posts_per_page' => -1,
            'orderby' => 'post__in',
        );

        $query = new WP_Query( $args );

        if ( ! $query->have_posts() ) {
            wp_send_json_error( array( 'message' => __( 'No recently viewed products found.', 'recently-viewed-products-popup-woo' ) ) );
        }

        ob_start();
        include WRVPP_PLUGIN_DIR . 'templates/modal-template.php';
        $modal = ob_get_clean();

        ob_start();
        include WRVPP_PLUGIN_DIR . 'templates/popup-template.php';
        $popup = ob_get_clean();
        wp_reset_postdata();

        wp_send_json_success( 
            array(
                'popup' => $popup,
                'modal' => $modal
            )
        );
    }

    /**
     * Check if the popup is enabled and should be displayed on the current page.
     *
     * @return bool
     */
    public function is_popup_enabled() {
        $display_on_pages = get_option( 'wrvpp_display_on', array( 'is_product' ) );

        if ( get_option( 'wrvpp_enable_popup', 'yes' ) !== 'yes' ) {
            return false;
        }

        if ( empty( $display_on_pages ) || ! is_array( $display_on_pages ) ) {
            return false;
        }

        if ( in_array( 'all_pages', $display_on_pages ) ) {
            return true;
        }

        if ( ( in_array( 'is_product', $display_on_pages ) && is_product() ) ||
            ( in_array( 'is_shop', $display_on_pages ) && is_shop() ) ||
            ( in_array( 'is_product_category', $display_on_pages ) && is_product_category() ) ||
            ( in_array( 'is_cart', $display_on_pages ) && is_cart() ) ) {
            return true;
        }
        return false;
    }

}