<?php
/**
 * Plugin Name:       Real-Time Widget for Matomo
 * Plugin URI:        https://jrmora.com/widget-matomo-visitas-tiempo-real-wordpress-sin-plugins/
 * Description:       A lightweight, real-time dashboard widget for Matomo. Includes native desktop notifications and sound alerts.
 * Version:           1.4.2
 * Requires at least: 5.8
 * Tested up to:      6.9
 * Requires PHP:      7.4
 * Author:            JRMora
 * Author URI:        https://jrmora.com
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       real-time-widget-for-matomo
 * Domain Path:       /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class RTWFM_Realtime_Widget_Plugin {

    private $option_group = 'rtwfm_settings_group';
    private $option_name  = 'rtwfm_options';
    private $version      = '1.4.2';

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_plugin_page' ) );
        add_action( 'admin_init', array( $this, 'page_init' ) );
        add_action( 'wp_dashboard_setup', array( $this, 'add_dashboard_widget' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
    }

    /**
     * Enqueue JS and CSS assets.
     */
    public function enqueue_admin_assets( $hook ) {
        $is_dashboard = ( 'index.php' === $hook );
        $is_settings  = ( 'settings_page_real-time-widget-for-matomo' === $hook );

        if ( ! $is_dashboard && ! $is_settings ) {
            return;
        }

        if ( $is_dashboard ) {
            // CSS
            wp_enqueue_style( 
                'rtwfm-admin-style', 
                plugin_dir_url( __FILE__ ) . 'assets/css/mrw-admin.css', 
                array(), 
                $this->version 
            );

            // Chart.js
            wp_enqueue_script( 
                'chart-js', 
                plugin_dir_url( __FILE__ ) . 'assets/js/chart.umd.min.js', 
                array(), 
                '4.5.1', 
                true 
            );

            // Widget Logic
            wp_enqueue_script( 
                'rtwfm-widget-js', 
                plugin_dir_url( __FILE__ ) . 'assets/js/mrw-widget.js', 
                array( 'chart-js' ), 
                $this->version, 
                true 
            );

            $o = get_option( $this->option_name );
            $matomo_url = isset($o['matomo_url']) ? trailingslashit( str_replace('index.php', '', $o['matomo_url']) ) : '';
            
            wp_localize_script( 'rtwfm-widget-js', 'rtwfmData', array(
                'apiUrl'     => $matomo_url,
                'siteId'     => $o['id_site'] ?? '',
                'token'      => $o['token_auth'] ?? '',
                'confNotify' => isset($o['enable_notify']) ? (string)$o['enable_notify'] : '0',
                'confMode'   => $o['notify_mode'] ?? 'threshold',
                'confThresh' => $o['notify_threshold'] ?? 10,
                'confSound'  => isset($o['enable_sound']) ? (string)$o['enable_sound'] : '0',
                'iconUrl'    => get_site_icon_url(192) ? get_site_icon_url(192) : admin_url('images/wordpress-logo.svg'),
                'strings'    => array(
                    // General
                    'trafficHigh'  => __( '⚠️ High Traffic', 'real-time-widget-for-matomo' ),
                    'visitsActive' => __( 'Active visits:', 'real-time-widget-for-matomo' ),
                    'newVisit'     => __( 'New Visit', 'real-time-widget-for-matomo' ),
                    'navigating'   => __( 'Navigating...', 'real-time-widget-for-matomo' ),
                    'action'       => __( 'Action', 'real-time-widget-for-matomo' ),
                    'unknown'      => __( 'Unknown', 'real-time-widget-for-matomo' ),
                    'directEntry'  => __( 'Direct Entry', 'real-time-widget-for-matomo' ),
                    'updating'     => __( 'Updating...', 'real-time-widget-for-matomo' ),
                    'live'         => __( 'Live (Updated: ', 'real-time-widget-for-matomo' ),
                    'error'        => __( 'Error: ', 'real-time-widget-for-matomo' ),
                    'visits'       => __( 'Visits', 'real-time-widget-for-matomo' ),
                    'actions'      => __( 'Actions', 'real-time-widget-for-matomo' ),
                    'loading'      => __( 'Loading...', 'real-time-widget-for-matomo' ),
                    'yesterday0'   => __( 'Yesterday: 0', 'real-time-widget-for-matomo' ),
                    'yesterday'    => __( 'Yesterday:', 'real-time-widget-for-matomo' ),
                    'langCode'     => __( 'Lang Code', 'real-time-widget-for-matomo' ),
                    'refPrefix'    => __( 'Ref:', 'real-time-widget-for-matomo' ),
                    'campPrefix'   => __( 'Campaign:', 'real-time-widget-for-matomo' ),
                    'keywordUndef' => __( 'Keyword not defined', 'real-time-widget-for-matomo' ),
                    
                    // --- NUEVAS TRADUCCIONES PARA TOOLTIPS Y GRÁFICAS ---
                    'txtCountry'   => __( 'Country:', 'real-time-widget-for-matomo' ),
                    'txtRegion'    => __( 'Region:', 'real-time-widget-for-matomo' ),
                    'txtCity'      => __( 'City:', 'real-time-widget-for-matomo' ),
                    'txtLang'      => __( 'Language:', 'real-time-widget-for-matomo' ),
                    'txtIP'        => __( 'IP:', 'real-time-widget-for-matomo' ),
                    'txtID'        => __( 'ID:', 'real-time-widget-for-matomo' ),
                    'txtEvent'     => __( 'Event', 'real-time-widget-for-matomo' ),
                    'txtSearch'    => __( 'Internal Search:', 'real-time-widget-for-matomo' ),
                    'txtTime'      => __( 'Time on page:', 'real-time-widget-for-matomo' ),
                    // Gráfica de Rendimiento
                    'perfNetwork'  => __( 'Network', 'real-time-widget-for-matomo' ),
                    'perfServer'   => __( 'Server', 'real-time-widget-for-matomo' ),
                    'perfTransfer' => __( 'Transfer', 'real-time-widget-for-matomo' ),
                    'perfDomProc'  => __( 'DOM Proc.', 'real-time-widget-for-matomo' ),
                    'perfDomComp'  => __( 'DOM Comp.', 'real-time-widget-for-matomo' ),
                    'perfOnLoad'   => __( 'OnLoad', 'real-time-widget-for-matomo' )
                )
            ));
        }

        if ( $is_settings ) {
            wp_enqueue_script( 
                'rtwfm-settings-js', 
                plugin_dir_url( __FILE__ ) . 'assets/js/mrw-settings.js', 
                array(), 
                $this->version, 
                true 
            );

            wp_localize_script( 'rtwfm-settings-js', 'rtwfmSettingsParams', array(
                'testAudioUrl'       => plugin_dir_url(__FILE__) . 'alerta.mp3',
                'iconUrl'            => get_site_icon_url(192) ? get_site_icon_url(192) : admin_url('images/wordpress-logo.svg'),
                'strNoSupport'       => __( 'Your browser does not support notifications.', 'real-time-widget-for-matomo' ),
                'strTestTitle'       => __( 'Matomo Test', 'real-time-widget-for-matomo' ),
                'strTestBody'        => __( '✅ Notification active + Sound.', 'real-time-widget-for-matomo' ),
                'strPermissionGranted' => __( '✅ Permission granted.', 'real-time-widget-for-matomo' ),
                'strDenied'          => __( '❌ Permission denied. Check the lock icon in your address bar.', 'real-time-widget-for-matomo' )
            ));
        }
    }

    public function add_plugin_page() {
        add_options_page(
            __( 'Matomo Widget Settings', 'real-time-widget-for-matomo' ),
            __( 'Matomo Widget', 'real-time-widget-for-matomo' ),
            'manage_options',
            'real-time-widget-for-matomo',
            array( $this, 'create_admin_page' )
        );
    }

    public function create_admin_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Matomo Widget Configuration', 'real-time-widget-for-matomo' ); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields( $this->option_group );
                do_settings_sections( 'real-time-widget-for-matomo' );
                submit_button();
                ?>
            </form>
            <hr>
            <h3><?php esc_html_e( 'Diagnostics', 'real-time-widget-for-matomo' ); ?></h3>
            <p><?php esc_html_e( 'Click to test notifications and sound. Please check your volume.', 'real-time-widget-for-matomo' ); ?></p>
            
            <audio id="rtwfm-alert-audio" src="<?php echo esc_url( plugin_dir_url(__FILE__) . 'alerta.mp3' ); ?>" preload="auto" style="width:0; height:0; opacity:0; position:absolute; left:-9999px;"></audio>

            <button type="button" class="button button-secondary" onclick="rtwfmTestNotification()"><?php esc_html_e( '🔔 Test Alert & Sound', 'real-time-widget-for-matomo' ); ?></button>
        </div>
        <?php
    }

    public function page_init() {
        register_setting( $this->option_group, $this->option_name, array( $this, 'sanitize' ) );
        
        add_settings_section( 'rtwfm_conn_section', __( 'API Connection Data', 'real-time-widget-for-matomo' ), array( $this, 'section_conn_info' ), 'real-time-widget-for-matomo' );
        
        add_settings_field( 'matomo_url', __( 'Matomo URL', 'real-time-widget-for-matomo' ), array( $this, 'matomo_url_callback' ), 'real-time-widget-for-matomo', 'rtwfm_conn_section' );
        add_settings_field( 'id_site', __( 'Site ID', 'real-time-widget-for-matomo' ), array( $this, 'id_site_callback' ), 'real-time-widget-for-matomo', 'rtwfm_conn_section' );
        add_settings_field( 'token_auth', __( 'Auth Token', 'real-time-widget-for-matomo' ), array( $this, 'token_auth_callback' ), 'real-time-widget-for-matomo', 'rtwfm_conn_section' );
        
        add_settings_section( 'rtwfm_notif_section', __( 'Notifications', 'real-time-widget-for-matomo' ), array( $this, 'section_notif_info' ), 'real-time-widget-for-matomo' );
        
        add_settings_field( 'enable_notify', __( 'Enable', 'real-time-widget-for-matomo' ), array( $this, 'enable_notify_callback' ), 'real-time-widget-for-matomo', 'rtwfm_notif_section' );
        add_settings_field( 'notify_mode', __( 'Mode', 'real-time-widget-for-matomo' ), array( $this, 'notify_mode_callback' ), 'real-time-widget-for-matomo', 'rtwfm_notif_section' );
        add_settings_field( 'notify_threshold', __( 'Threshold', 'real-time-widget-for-matomo' ), array( $this, 'notify_threshold_callback' ), 'real-time-widget-for-matomo', 'rtwfm_notif_section' );
        add_settings_field( 'enable_sound', __( 'Sound', 'real-time-widget-for-matomo' ), array( $this, 'enable_sound_callback' ), 'real-time-widget-for-matomo', 'rtwfm_notif_section' );
        
        add_settings_section( 'rtwfm_other_section', __( 'Maintenance', 'real-time-widget-for-matomo' ), null, 'real-time-widget-for-matomo' );
        add_settings_field( 'clean_uninstall', __( 'Uninstallation', 'real-time-widget-for-matomo' ), array( $this, 'clean_uninstall_callback' ), 'real-time-widget-for-matomo', 'rtwfm_other_section' );
    }

    public function sanitize( $input ) {
        $new_input = array();
        $url = isset( $input['matomo_url'] ) ? esc_url_raw( $input['matomo_url'] ) : '';
        $url = str_replace('index.php', '', $url);
        $new_input['matomo_url'] = trailingslashit( $url );
        $new_input['id_site'] = isset( $input['id_site'] ) ? absint( $input['id_site'] ) : 1;
        $new_input['token_auth'] = isset( $input['token_auth'] ) ? sanitize_text_field( $input['token_auth'] ) : '';
        $new_input['clean_uninstall'] = isset( $input['clean_uninstall'] ) ? 1 : 0;
        $new_input['enable_notify'] = isset( $input['enable_notify'] ) ? 1 : 0;
        $new_input['notify_mode'] = isset( $input['notify_mode'] ) ? sanitize_text_field( $input['notify_mode'] ) : 'threshold';
        $new_input['notify_threshold'] = isset( $input['notify_threshold'] ) ? absint( $input['notify_threshold'] ) : 10;
        $new_input['enable_sound'] = isset( $input['enable_sound'] ) ? 1 : 0;
        return $new_input;
    }

    public function section_conn_info() { echo esc_html__( 'Enter your Matomo On-Premise installation data.', 'real-time-widget-for-matomo' ); }
    public function section_notif_info() { echo esc_html__( 'Native desktop alerts.', 'real-time-widget-for-matomo' ); }
    
    public function matomo_url_callback() { 
        $o = get_option( $this->option_name ); 
        echo '<input type="url" name="rtwfm_options[matomo_url]" value="' . esc_attr( $o['matomo_url'] ?? '' ) . '" class="regular-text" placeholder="https://..." />'; 
    }
    public function id_site_callback() { 
        $o = get_option( $this->option_name ); 
        echo '<input type="number" name="rtwfm_options[id_site]" value="' . esc_attr( $o['id_site'] ?? '1' ) . '" class="small-text" />'; 
    }
    public function token_auth_callback() { 
        $o = get_option( $this->option_name ); 
        echo '<input type="password" name="rtwfm_options[token_auth]" value="' . esc_attr( $o['token_auth'] ?? '' ) . '" class="regular-text" />'; 
    }
    
    public function enable_notify_callback() { 
        $o = get_option( $this->option_name ); 
        $val = isset( $o['enable_notify'] ) ? (int) $o['enable_notify'] : 0; 
        echo '<label><input type="checkbox" name="rtwfm_options[enable_notify]" value="1" ' . checked( 1, $val, false ) . ' /> ' . esc_html__( 'Enable', 'real-time-widget-for-matomo' ) . '</label>'; 
    }

    public function notify_mode_callback() { 
        $o = get_option( $this->option_name ); 
        $m = $o['notify_mode'] ?? 'threshold'; 
        ?> 
        <select name="rtwfm_options[notify_mode]">
            <option value="threshold" <?php selected($m,'threshold'); ?>><?php esc_html_e( 'By Threshold', 'real-time-widget-for-matomo' ); ?></option>
            <option value="every" <?php selected($m,'every'); ?>><?php esc_html_e( 'Every Visit', 'real-time-widget-for-matomo' ); ?></option>
        </select> 
        <?php 
    }
    public function notify_threshold_callback() { 
        $o = get_option( $this->option_name ); 
        echo '<input type="number" name="rtwfm_options[notify_threshold]" value="' . esc_attr( $o['notify_threshold'] ?? 10 ) . '" class="small-text" />'; 
    }
    
    public function enable_sound_callback() { 
        $o = get_option( $this->option_name ); 
        $val = isset( $o['enable_sound'] ) ? (int) $o['enable_sound'] : 0;
        echo '<label><input type="checkbox" name="rtwfm_options[enable_sound]" value="1" ' . checked( 1, $val, false ) . ' /> ' . esc_html__( 'Enable Sound', 'real-time-widget-for-matomo' ) . '</label>'; 
    }
    
    public function clean_uninstall_callback() { 
        $o = get_option( $this->option_name ); 
        $val = isset( $o['clean_uninstall'] ) ? (int) $o['clean_uninstall'] : 0;
        echo '<label><input type="checkbox" name="rtwfm_options[clean_uninstall]" value="1" ' . checked( 1, $val, false ) . ' /> ' . esc_html__( 'Clean DB on uninstall', 'real-time-widget-for-matomo' ) . '</label>'; 
    }

    public function add_dashboard_widget() { 
        wp_add_dashboard_widget( 
            'rtwfm_matomo_widget_plugin', 
            __( 'Matomo Statistics (Live)', 'real-time-widget-for-matomo' ), 
            array( $this, 'render_widget' ) 
        ); 
    }

    public function render_widget() {
        $o = get_option( $this->option_name );
        
        $settings_link = admin_url('options-general.php?page=real-time-widget-for-matomo');

        if ( empty( $o['matomo_url'] ) || empty( $o['token_auth'] ) ) {
            echo '<div style="padding:20px; color:#d63638;">' . sprintf( 
                /* translators: %s: Link to the settings page */
                esc_html__( '⚠️ Please configure the widget in %s.', 'real-time-widget-for-matomo' ), 
                '<a href="' . esc_url( $settings_link ) . '">' . esc_html__( 'Settings', 'real-time-widget-for-matomo' ) . '</a>' 
            ) . '</div>';
            return;
        }

        $matomo_url = trailingslashit( str_replace('index.php', '', $o['matomo_url']) );
        $id_site    = $o['id_site'];
        $token_auth = $o['token_auth'];
        
        $export_csv = $matomo_url . "index.php?module=API&method=VisitsSummary.get&idSite=$id_site&period=day&date=last30&format=CSV&token_auth=$token_auth";
        $export_img = $matomo_url . "index.php?module=API&method=ImageGraph.get&idSite=$id_site&apiModule=VisitsSummary&apiAction=get&token_auth=$token_auth&graphType=evolution&period=day&date=last30&width=800&height=400";
        
        $audio_url = plugin_dir_url(__FILE__) . 'alerta.mp3';
        ?>

        <div class="status-bar"><span class="loading-dot" id="m-dot"></span> <span id="m-status"><?php esc_html_e( 'Connecting...', 'real-time-widget-for-matomo' ); ?></span></div>
        
        <audio id="rtwfm-alert-audio" src="<?php echo esc_url($audio_url); ?>" preload="auto" style="width:0; height:0; opacity:0; position:absolute; left:-9999px;"></audio>
        
        <div class="rtwfm-js-wrapper">
            <div id="m-error" class="error-box"></div>
            
            <div class="hero-box">
                <span class="hero-num" id="val-live">--</span>
                <div class="hero-desc"><span id="val-live-vis">--</span> <?php esc_html_e( 'visits and', 'real-time-widget-for-matomo' ); ?> <span id="val-live-act">--</span> <?php esc_html_e( 'actions (last 3 min)', 'real-time-widget-for-matomo' ); ?></div>
            </div>
            
            <div class="kpi-grid">
                <div class="kpi-card"><h4><?php esc_html_e( 'Visits (Today)', 'real-time-widget-for-matomo' ); ?></h4><div class="kpi-val-group"><span class="val" id="val-uniq">--</span><span class="trend-badge" id="trend-uniq"></span></div></div>
                <div class="kpi-card"><h4><?php esc_html_e( 'Actions (Today)', 'real-time-widget-for-matomo' ); ?></h4><div class="kpi-val-group"><span class="val" id="val-page">--</span><span class="trend-badge" id="trend-page"></span></div></div>
            </div>

            <div class="graph-container">
                <div class="graph-header"><div class="graph-title-area"><span class="graph-title"><?php esc_html_e( 'Visits Chart', 'real-time-widget-for-matomo' ); ?></span><span id="btn-refresh-visits" class="btn-graph-refresh dashicons dashicons-update" title="<?php esc_attr_e( 'Update', 'real-time-widget-for-matomo' ); ?>"></span></div></div>
                <div class="chart-canvas-box"><canvas id="matomoChart"></canvas></div>
                <div class="graph-actions"><a href="<?php echo esc_url($export_csv); ?>" target="_blank" class="action-icon dashicons dashicons-media-spreadsheet" title="<?php esc_attr_e( 'Export CSV', 'real-time-widget-for-matomo' ); ?>"></a><a href="<?php echo esc_url($export_img); ?>" target="_blank" class="action-icon dashicons dashicons-format-image" title="<?php esc_attr_e( 'Download Image', 'real-time-widget-for-matomo' ); ?>"></a></div>
            </div>

            <div class="graph-container">
                <div class="graph-header"><div class="graph-title-area"><span class="graph-title"><?php esc_html_e( 'Performance Evolution (Seconds)', 'real-time-widget-for-matomo' ); ?></span><span id="btn-refresh-perf" class="btn-graph-refresh dashicons dashicons-update" title="<?php esc_attr_e( 'Update', 'real-time-widget-for-matomo' ); ?>"></span></div></div>
                <div class="chart-canvas-box"><canvas id="perfChart"></canvas></div>
            </div>

            <div class="log-container">
                <div class="log-header"><?php esc_html_e( 'Visit Log', 'real-time-widget-for-matomo' ); ?></div>
                <table class="summary-table">
                    <tr><td><?php esc_html_e( 'Last 24 hours', 'real-time-widget-for-matomo' ); ?></td><td id="val-24-v">--</td><td id="val-24-a">--</td></tr>
                    <tr><td><?php esc_html_e( 'Last 30 minutes', 'real-time-widget-for-matomo' ); ?></td><td id="val-30-v">--</td><td id="val-30-a">--</td></tr>
                </table>
                <div class="log-rows" id="log-content"><div style="padding:20px; text-align:center; color:#ccc;"><?php esc_html_e( 'Loading...', 'real-time-widget-for-matomo' ); ?></div></div>
            </div>
        </div>
        <?php
    }
}

new RTWFM_Realtime_Widget_Plugin();