<?php
/**
 * RankJet AI Slug Monitor
 * 
 * Automatically creates 301 redirects when post/page permalinks are changed.
 * Shows toast notifications in the editor to inform users.
 * 
 * @package RankJet_AI
 * @since 1.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Rankjet_Ai_Slug_Monitor {

    /**
     * Constructor - Register hooks
     */
    public function __construct() {
        // Only run if feature is enabled
        if (get_option('rankjet_auto_redirect_slug_change', 'yes') !== 'yes') {
            return;
        }

        // Hook into post update to detect slug changes
        add_action('post_updated', [$this, 'check_slug_change'], 10, 3);
        
        // Add admin notice for redirect creation
        add_action('admin_notices', [$this, 'show_redirect_notice']);
        
        // Enqueue editor script for toast notifications
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_editor_scripts']);
    }

    /**
     * Check if slug changed and create redirect
     */
    public function check_slug_change($post_id, $post_after, $post_before) {
        // Skip revisions and autosaves
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
            return;
        }

        // Skip if not published
        if ($post_after->post_status !== 'publish') {
            return;
        }

        // Skip if slug didn't change
        if ($post_before->post_name === $post_after->post_name) {
            return;
        }

        // Skip if old slug was empty (new post)
        if (empty($post_before->post_name)) {
            return;
        }

        // Build old and new URLs
        $post_type = get_post_type_object($post_after->post_type);
        if (!$post_type || !$post_type->public) {
            return;
        }

        // Get the old permalink
        $old_permalink = $this->get_old_permalink($post_id, $post_before);
        $new_permalink = get_permalink($post_id);

        // Don't create redirect if URLs are the same
        if ($old_permalink === $new_permalink) {
            return;
        }

        // Create the redirect
        $this->create_slug_redirect($post_id, $old_permalink, $new_permalink);
    }

    /**
     * Get the old permalink based on old slug
     */
    private function get_old_permalink($post_id, $post_before) {
        // Temporarily set the old slug
        global $wpdb;
        
        // Get current permalink structure
        $permalink = get_permalink($post_id);
        
        // Replace new slug with old slug in the URL
        $new_slug = get_post_field('post_name', $post_id);
        $old_slug = $post_before->post_name;
        
        // Simple replacement - works for most permalink structures
        $old_permalink = str_replace('/' . $new_slug . '/', '/' . $old_slug . '/', $permalink);
        
        // Handle trailing patterns
        if ($old_permalink === $permalink) {
            $old_permalink = str_replace('/' . $new_slug, '/' . $old_slug, $permalink);
        }

        return $old_permalink;
    }

    /**
     * Create redirect for slug change
     */
    private function create_slug_redirect($post_id, $old_url, $new_url) {
        // Extract path from URLs
        $old_path = wp_parse_url($old_url, PHP_URL_PATH);
        $new_path = wp_parse_url($new_url, PHP_URL_PATH);

        if (!$old_path || !$new_path) {
            return;
        }

        // Check if redirect handler exists
        if (!class_exists('Rankjet_Ai_Redirects')) {
            return;
        }

        $redirects = new Rankjet_Ai_Redirects();
        
        $result = $redirects->create_redirect([
            'source_url' => $old_path,
            'destination_url' => $new_path,
            'redirect_type' => '301',
            'notes' => sprintf(
                __('Auto-created: Slug changed for "%s" (Post ID: %d)', 'rankjet-ai'),
                get_the_title($post_id),
                $post_id
            ),
        ]);

        if (!is_wp_error($result)) {
            // Store flag for admin notice
            set_transient(
                'rankjet_slug_redirect_created_' . get_current_user_id(),
                [
                    'post_id' => $post_id,
                    'old_url' => $old_path,
                    'new_url' => $new_path,
                ],
                60
            );
        }
    }

    /**
     * Show admin notice when redirect is created
     */
    public function show_redirect_notice() {
        $transient_key = 'rankjet_slug_redirect_created_' . get_current_user_id();
        $data = get_transient($transient_key);

        if (!$data) {
            return;
        }

        delete_transient($transient_key);

        ?>
        <div class="notice notice-success is-dismissible rankjet-slug-redirect-notice">
            <p>
                <strong><?php _e('RankJet AI:', 'rankjet-ai'); ?></strong>
                <?php printf(
                    __('A 301 redirect was automatically created from %1$s to %2$s.', 'rankjet-ai'),
                    '<code>' . esc_html($data['old_url']) . '</code>',
                    '<code>' . esc_html($data['new_url']) . '</code>'
                ); ?>
                <a href="<?php echo admin_url('admin.php?page=rankjet-ai-redirects'); ?>">
                    <?php _e('Manage Redirects', 'rankjet-ai'); ?>
                </a>
            </p>
        </div>
        <?php
    }

    /**
     * Enqueue editor scripts for toast notifications
     */
    public function enqueue_editor_scripts() {
        // Add inline script for toast notification support
        $script = "
        (function() {
            // Check for pending redirect notification
            var notification = window.rankjetSlugRedirect || null;
            if (notification && wp.data && wp.data.dispatch) {
                wp.data.dispatch('core/notices').createNotice(
                    'success',
                    notification.message,
                    {
                        isDismissible: true,
                        type: 'snackbar',
                        actions: notification.url ? [{
                            label: '" . esc_js(__('View Redirects', 'rankjet-ai')) . "',
                            url: notification.url
                        }] : []
                    }
                );
            }
        })();
        ";

        // Check if we should show notification
        $transient_key = 'rankjet_slug_redirect_created_' . get_current_user_id();
        $data = get_transient($transient_key);

        if ($data) {
            wp_add_inline_script('wp-edit-post', "window.rankjetSlugRedirect = {
                message: '" . esc_js(sprintf(
                    __('RankJet AI: Redirect created from %s to %s', 'rankjet-ai'),
                    $data['old_url'],
                    $data['new_url']
                )) . "',
                url: '" . esc_js(admin_url('admin.php?page=rankjet-ai-redirects')) . "'
            };", 'before');
            wp_add_inline_script('wp-edit-post', $script);
            delete_transient($transient_key);
        }
    }
}
