<?php
/**
 * RankJet AI Redirects - Main Handler
 * 
 * Handles 301/302/410 redirects with regex support, loop detection,
 * and performance-optimized matching using flat-file cache.
 * 
 * @package RankJet_AI
 * @since 1.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Rankjet_Ai_Redirects {

    /**
     * Database table name (without prefix)
     */
    const TABLE_NAME = 'rankjet_redirects';

    /**
     * Maximum redirect hops to prevent infinite loops
     */
    const MAX_REDIRECT_HOPS = 5;

    /**
     * Cache instance
     */
    private $cache;

    /**
     * Admin page hook suffix
     */
    private $hook_suffix;

    /**
     * Constructor - Register hooks
     */
    public function __construct() {
        // Initialize cache handler
        $this->cache = new Rankjet_Ai_Redirect_Cache();

        // Hook into template_redirect at high priority (before 404)
        add_action('template_redirect', [$this, 'handle_redirect'], 1);

        // Admin menu
        add_action('admin_menu', [$this, 'add_redirects_menu'], 25);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_redirect_assets']);

        // AJAX handlers
        add_action('wp_ajax_rankjet_get_redirects', [$this, 'ajax_get_redirects']);
        add_action('wp_ajax_rankjet_save_redirect', [$this, 'ajax_save_redirect']);
        add_action('wp_ajax_rankjet_delete_redirect', [$this, 'ajax_delete_redirect']);
        add_action('wp_ajax_rankjet_bulk_action_redirects', [$this, 'ajax_bulk_action']);
        add_action('wp_ajax_rankjet_test_redirect', [$this, 'ajax_test_redirect']);
        add_action('wp_ajax_rankjet_run_setup', [$this, 'ajax_run_setup']);
        add_action('wp_ajax_rankjet_check_status', [$this, 'ajax_check_status']);
        add_action('wp_ajax_rankjet_save_settings', [$this, 'ajax_save_settings']);
    }

    /**
     * Add Redirects submenu page
     */
    public function add_redirects_menu() {
        $this->hook_suffix = add_submenu_page(
            'rankjet-ai',
            __('Redirects', 'rankjet-ai'),
            __('Redirects', 'rankjet-ai'),
            'manage_options',
            'rankjet-ai-redirects',
            [$this, 'render_redirects_page']
        );
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_redirect_assets($hook) {
        // Use robust check for page slug to prevent FOUC
        if (!isset($_GET['page']) || $_GET['page'] !== 'rankjet-ai-redirects') {
            return;
        }

        // Enqueue the build script
        $asset_file = include(RANKJET_AI_PLUGIN_DIR . 'build/index.asset.php');
        
        wp_enqueue_script(
            'rankjet-ai-redirects',
            RANKJET_AI_PLUGIN_URL . 'build/index.js',
            array_merge($asset_file['dependencies'], ['wp-element', 'wp-components', 'wp-api-fetch']),
            $asset_file['version'],
            true
        );

        wp_enqueue_style(
            'rankjet-ai-redirects-style', 
            RANKJET_AI_PLUGIN_URL . 'build/style-index.css', 
            ['wp-components'], 
            $asset_file['version']
        );

        // Check if tables exist
        $tables_exist = $this->tables_exist();
        
        // Get statistics only if tables exist
        $stats = ['total' => 0, 'active' => 0, 'hits_today' => 0];
        if ($tables_exist) {
            global $wpdb;
            $table = $wpdb->prefix . self::TABLE_NAME;
            
            $stats = [
                'total' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}"),
                'active' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE status = 'active'"),
                'hits_today' => (int) $wpdb->get_var($wpdb->prepare(
                    "SELECT SUM(hit_count) FROM {$table} WHERE DATE(last_accessed) = %s",
                    current_time('Y-m-d')
                )),
            ];
        }

        // Prepare settings data for React
        $settings_data = [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('rankjet_redirects_nonce'),
            'stats' => $stats,
            'plugin_version' => RANKJET_AI_VERSION,
            'auto_redirect_on_slug_change' => get_option('rankjet_auto_redirect_slug_change', 'yes'),
            'log_retention_days' => get_option('rankjet_404_log_retention', 30),
            'tables_exist' => $tables_exist,
        ];

        wp_localize_script('rankjet-ai-redirects', 'rankjetRedirectSettings', $settings_data);
    }

    /**
     * Render the redirects admin page
     */
    public function render_redirects_page() {
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline screen-reader-text"><?php _e('Redirects', 'rankjet-ai'); ?></h1>
            <hr class="wp-header-end">
            <div id="rankjet-redirects-root"></div>
        </div>
        <?php
    }
    
    /**
     * Check if database tables exist
     */
    public function tables_exist() {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;
        return !empty($wpdb->get_var("SHOW TABLES LIKE '{$table}'"));
    }

    /**
     * Handle redirect on template_redirect hook
     */
    public function handle_redirect() {
        // Check if module is enabled
        if (class_exists('Rankjet_Ai_Dashboard') && !Rankjet_Ai_Dashboard::is_module_enabled('redirects')) {
            return;
        }

        // Don't process in admin
        if (is_admin()) {
            return;
        }

        // Get request URI
        $request_uri = $this->get_request_uri();
        
        if (empty($request_uri)) {
            return;
        }

        // Find matching redirect from cache first
        $redirect = $this->find_matching_redirect($request_uri);
        
        if (!$redirect) {
            return;
        }

        // Update hit stats (async via shutdown hook to not slow down redirect)
        add_action('shutdown', function() use ($redirect) {
            $this->update_hit_stats($redirect['id']);
        });

        // Handle 410 Gone
        if ($redirect['type'] === '410') {
            status_header(410);
            nocache_headers();
            include(get_query_template('410') ?: get_query_template('404'));
            exit;
        }

        // Detect redirect loops
        if ($this->detect_loop($redirect['dest'], [$request_uri])) {
            // Log the loop and don't redirect
            error_log('RankJet AI: Redirect loop detected for ' . $request_uri);
            return;
        }

        // Perform redirect
        $status = (int) $redirect['type'];
        wp_redirect($redirect['dest'], $status);
        exit;
    }

    /**
     * Get the current request URI normalized
     */
    private function get_request_uri() {
        $uri = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
        
        // Remove query string
        $uri = strtok($uri, '?');
        
        // Handle subdirectory installation
        $site_path = parse_url(home_url(), PHP_URL_PATH);
        if ($site_path && $site_path !== '/') {
            $site_path = rtrim($site_path, '/');
            if (strpos($uri, $site_path) === 0) {
                $uri = substr($uri, strlen($site_path));
            }
        }
        
        // Normalize slashes
        $uri = '/' . ltrim($uri, '/');
        $uri = rtrim($uri, '/');
        
        // Empty path becomes /
        if ($uri === '') {
            $uri = '/';
        }
        
        return $uri;
    }

    /**
     * Find a matching redirect rule
     */
    private function find_matching_redirect($request_uri) {
        // Try cache first (fastest)
        $cached_rules = $this->cache->get_rules();
        
        if ($cached_rules !== false) {
            // Check exact matches first
            if (isset($cached_rules['exact'][$request_uri])) {
                return $cached_rules['exact'][$request_uri];
            }
            
            // Also check with trailing slash variants
            $uri_with_slash = rtrim($request_uri, '/') . '/';
            $uri_without_slash = rtrim($request_uri, '/');
            
            if (isset($cached_rules['exact'][$uri_with_slash])) {
                return $cached_rules['exact'][$uri_with_slash];
            }
            if (isset($cached_rules['exact'][$uri_without_slash])) {
                return $cached_rules['exact'][$uri_without_slash];
            }
            
            // Check regex patterns
            if (!empty($cached_rules['regex'])) {
                foreach ($cached_rules['regex'] as $pattern => $rule) {
                    if (@preg_match($pattern, $request_uri, $matches)) {
                        // Replace capture groups in destination
                        $destination = $rule['dest'];
                        foreach ($matches as $i => $match) {
                            if ($i > 0) {
                                $destination = str_replace('$' . $i, $match, $destination);
                            }
                        }
                        return [
                            'id' => $rule['id'],
                            'dest' => $destination,
                            'type' => $rule['type'],
                        ];
                    }
                }
            }
        }
        
        // Fallback to database query if cache miss
        return $this->find_redirect_from_db($request_uri);
    }

    /**
     * Find redirect from database (fallback)
     */
    private function find_redirect_from_db($request_uri) {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;
        
        // Check exact match first
        $redirect = $wpdb->get_row($wpdb->prepare(
            "SELECT id, destination_url, redirect_type, is_regex 
             FROM {$table} 
             WHERE status = 'active' 
             AND is_regex = 0 
             AND (source_url = %s OR source_url = %s OR source_url = %s)
             LIMIT 1",
            $request_uri,
            rtrim($request_uri, '/') . '/',
            rtrim($request_uri, '/')
        ));
        
        if ($redirect) {
            return [
                'id' => $redirect->id,
                'dest' => $redirect->destination_url,
                'type' => $redirect->redirect_type,
            ];
        }
        
        // Check regex patterns
        $regex_rules = $wpdb->get_results(
            "SELECT id, source_url, destination_url, redirect_type 
             FROM {$table} 
             WHERE status = 'active' AND is_regex = 1"
        );
        
        foreach ($regex_rules as $rule) {
            $pattern = $rule->source_url;
            // Ensure pattern has delimiters
            if ($pattern[0] !== '/' && $pattern[0] !== '#' && $pattern[0] !== '~') {
                $pattern = '/' . $pattern . '/';
            }
            
            if (@preg_match($pattern, $request_uri, $matches)) {
                $destination = $rule->destination_url;
                foreach ($matches as $i => $match) {
                    if ($i > 0) {
                        $destination = str_replace('$' . $i, $match, $destination);
                    }
                }
                return [
                    'id' => $rule->id,
                    'dest' => $destination,
                    'type' => $rule->redirect_type,
                ];
            }
        }
        
        return null;
    }

    /**
     * Detect redirect loops
     */
    private function detect_loop($destination, $visited = []) {
        if (count($visited) >= self::MAX_REDIRECT_HOPS) {
            return true;
        }
        
        // Normalize destination
        $dest_path = wp_parse_url($destination, PHP_URL_PATH);
        if (!$dest_path) {
            return false;
        }
        
        $dest_path = '/' . ltrim(rtrim($dest_path, '/'), '/');
        
        // Check if already visited
        if (in_array($dest_path, $visited)) {
            return true;
        }
        
        // Check if destination has its own redirect
        $visited[] = $dest_path;
        $next_redirect = $this->find_matching_redirect($dest_path);
        
        if ($next_redirect && $next_redirect['type'] !== '410') {
            return $this->detect_loop($next_redirect['dest'], $visited);
        }
        
        return false;
    }

    /**
     * Update hit statistics
     */
    private function update_hit_stats($redirect_id) {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;
        
        $wpdb->query($wpdb->prepare(
            "UPDATE {$table} SET hit_count = hit_count + 1, last_accessed = NOW() WHERE id = %d",
            $redirect_id
        ));
    }

    /**
     * Create a new redirect
     */
    public function create_redirect($data) {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;
        
        // Validate source URL
        $source = sanitize_text_field($data['source_url'] ?? '');
        
        // Normalize source: Strip home URL if present (fixing issue when user pastes full URL)
        $home_url = home_url();
        $home_url_clean = rtrim($home_url, '/');
        if (strpos($source, $home_url_clean) === 0) {
            $source = substr($source, strlen($home_url_clean));
        }
        
        if (empty($source)) {
            return new WP_Error('invalid_source', __('Source URL is required.', 'rankjet-ai'));
        }
        
        // Normalize source (add leading slash if not regex)
        $is_regex = !empty($data['is_regex']);
        if (!$is_regex && $source[0] !== '/') {
            $source = '/' . $source;
        }
        
        // Validate and normalize regex pattern
        if ($is_regex) {
            // Check if pattern handles full full-site-path stripping correctly (done above)
            
            // If pattern lacks delimiters, try to auto-wrap
            if (@preg_match($source, '') === false) {
                // Try wrapping with hash first (common for URLs)
                $wrapped = '#' . str_replace('#', '\#', $source) . '#';
                
                if (@preg_match($wrapped, '') !== false) {
                    $source = $wrapped;
                } else {
                    // Try wrapping with slash (standard)
                    $wrapped = '/' . str_replace('/', '\/', $source) . '/';
                    if (@preg_match($wrapped, '') !== false) {
                        $source = $wrapped;
                    } else {
                        return new WP_Error('invalid_regex', __('Invalid regex pattern.', 'rankjet-ai'));
                    }
                }
            }
        }
        
        // Check for duplicate source
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE source_url = %s",
            $source
        ));
        
        if ($exists) {
            return new WP_Error('duplicate', __('A redirect for this source URL already exists.', 'rankjet-ai'));
        }
        
        // Validate redirect type
        $type = $data['redirect_type'] ?? '301';
        if (!in_array($type, ['301', '302', '410'])) {
            $type = '301';
        }
        
        // Destination required for 301/302
        $destination = sanitize_text_field($data['destination_url'] ?? '');
        if ($type !== '410' && empty($destination)) {
            return new WP_Error('invalid_dest', __('Destination URL is required for 301/302 redirects.', 'rankjet-ai'));
        }
        
        // Check for loops before saving
        if ($type !== '410' && $this->would_create_loop($source, $destination)) {
            return new WP_Error('loop_detected', __('This redirect would create a redirect loop.', 'rankjet-ai'));
        }
        
        $result = $wpdb->insert($table, [
            'source_url' => $source,
            'destination_url' => $destination,
            'redirect_type' => $type,
            'is_regex' => $is_regex ? 1 : 0,
            'status' => $data['status'] ?? 'active',
            'created_by' => get_current_user_id(),
            'notes' => sanitize_textarea_field($data['notes'] ?? ''),
        ], ['%s', '%s', '%s', '%d', '%s', '%d', '%s']);
        
        if ($result === false) {
            return new WP_Error('db_error', __('Failed to create redirect.', 'rankjet-ai'));
        }
        
        $redirect_id = $wpdb->insert_id;
        
        // Regenerate cache
        $this->cache->regenerate();
        
        return $redirect_id;
    }

    /**
     * Check if a redirect would create a loop
     */
    private function would_create_loop($source, $destination) {
        // Normalize paths
        $source_path = '/' . ltrim(rtrim($source, '/'), '/');
        $dest_path = wp_parse_url($destination, PHP_URL_PATH);
        
        if (!$dest_path) {
            // External URL, no loop possible within our system
            return false;
        }
        
        $dest_path = '/' . ltrim(rtrim($dest_path, '/'), '/');
        
        // Direct loop
        if ($source_path === $dest_path) {
            return true;
        }
        
        // Check if destination redirects back to source
        return $this->detect_loop($destination, [$source_path]);
    }

    /**
     * Update an existing redirect
     */
    public function update_redirect($id, $data) {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;
        
        $update_data = [];
        $update_format = [];
        
        if (isset($data['source_url'])) {
            $source = sanitize_text_field($data['source_url']);
            
            // Normalize source: Strip home URL if present
            $home_url = home_url();
            $home_url_clean = rtrim($home_url, '/');
            if (strpos($source, $home_url_clean) === 0) {
                $source = substr($source, strlen($home_url_clean));
            }
            
            $is_regex = !empty($data['is_regex']);
            if ($is_regex && @preg_match($source, '') === false) {
                 // Try wrapping with hash first
                 $wrapped = '#' . str_replace('#', '\#', $source) . '#';
                 if (@preg_match($wrapped, '') !== false) {
                     $source = $wrapped;
                 } else {
                     $wrapped = '/' . str_replace('/', '\/', $source) . '/';
                     if (@preg_match($wrapped, '') !== false) {
                         $source = $wrapped;
                     } else {
                         return new WP_Error('invalid_regex', __('Invalid regex pattern.', 'rankjet-ai'));
                     }
                 }
            }

            $update_data['source_url'] = $source;
            $update_format[] = '%s';
        }
        
        if (isset($data['destination_url'])) {
            $update_data['destination_url'] = sanitize_text_field($data['destination_url']);
            $update_format[] = '%s';
        }
        
        if (isset($data['redirect_type'])) {
            $type = $data['redirect_type'];
            if (in_array($type, ['301', '302', '410'])) {
                $update_data['redirect_type'] = $type;
                $update_format[] = '%s';
            }
        }
        
        if (isset($data['is_regex'])) {
            $update_data['is_regex'] = $data['is_regex'] ? 1 : 0;
            $update_format[] = '%d';
        }
        
        if (isset($data['status'])) {
            $update_data['status'] = in_array($data['status'], ['active', 'inactive']) ? $data['status'] : 'active';
            $update_format[] = '%s';
        }
        
        if (isset($data['notes'])) {
            $update_data['notes'] = sanitize_textarea_field($data['notes']);
            $update_format[] = '%s';
        }
        
        if (empty($update_data)) {
            return new WP_Error('no_data', __('No data to update.', 'rankjet-ai'));
        }
        
        $result = $wpdb->update($table, $update_data, ['id' => $id], $update_format, ['%d']);
        
        if ($result === false) {
            return new WP_Error('db_error', __('Failed to update redirect.', 'rankjet-ai'));
        }
        
        // Regenerate cache
        $this->cache->regenerate();
        
        return true;
    }

    /**
     * Delete a redirect
     */
    public function delete_redirect($id) {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;
        
        $result = $wpdb->delete($table, ['id' => $id], ['%d']);
        
        if ($result === false) {
            return new WP_Error('db_error', __('Failed to delete redirect.', 'rankjet-ai'));
        }
        
        // Regenerate cache
        $this->cache->regenerate();
        
        return true;
    }

    /**
     * Get redirects with pagination and filtering
     */
    public function get_redirects($args = []) {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;
        
        $defaults = [
            'per_page' => 20,
            'page' => 1,
            'orderby' => 'created_at',
            'order' => 'DESC',
            'search' => '',
            'status' => '',
            'type' => '',
        ];
        
        $args = wp_parse_args($args, $defaults);
        
        $where = ['1=1'];
        $where_values = [];
        
        if (!empty($args['search'])) {
            $where[] = '(source_url LIKE %s OR destination_url LIKE %s)';
            $search = '%' . $wpdb->esc_like($args['search']) . '%';
            $where_values[] = $search;
            $where_values[] = $search;
        }
        
        if (!empty($args['status']) && in_array($args['status'], ['active', 'inactive'])) {
            $where[] = 'status = %s';
            $where_values[] = $args['status'];
        }
        
        if (!empty($args['type']) && in_array($args['type'], ['301', '302', '410'])) {
            $where[] = 'redirect_type = %s';
            $where_values[] = $args['type'];
        }
        
        $where_sql = implode(' AND ', $where);
        
        // Count total
        $count_sql = "SELECT COUNT(*) FROM {$table} WHERE {$where_sql}";
        if (!empty($where_values)) {
            $count_sql = $wpdb->prepare($count_sql, $where_values);
        }
        $total = (int) $wpdb->get_var($count_sql);
        
        // Get items
        $orderby = sanitize_sql_orderby($args['orderby'] . ' ' . $args['order']) ?: 'created_at DESC';
        $offset = ($args['page'] - 1) * $args['per_page'];
        
        $query = "SELECT * FROM {$table} WHERE {$where_sql} ORDER BY {$orderby} LIMIT %d OFFSET %d";
        $query_values = array_merge($where_values, [$args['per_page'], $offset]);
        
        $items = $wpdb->get_results($wpdb->prepare($query, $query_values));
        
        return [
            'items' => $items,
            'total' => $total,
            'pages' => ceil($total / $args['per_page']),
        ];
    }

    /**
     * AJAX: Get redirects
     */
    public function ajax_get_redirects() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $args = [
            'page' => isset($_GET['page']) ? (int) $_GET['page'] : 1,
            'per_page' => isset($_GET['per_page']) ? (int) $_GET['per_page'] : 20,
            'search' => isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '',
            'status' => isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '',
            'type' => isset($_GET['type']) ? sanitize_text_field($_GET['type']) : '',
            'orderby' => isset($_GET['orderby']) ? sanitize_text_field($_GET['orderby']) : 'created_at',
            'order' => isset($_GET['order']) ? sanitize_text_field($_GET['order']) : 'DESC',
        ];
        
        $result = $this->get_redirects($args);
        wp_send_json_success($result);
    }

    /**
     * AJAX: Save redirect (create/update)
     */
    public function ajax_save_redirect() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $data = [
            'source_url' => isset($_POST['source_url']) ? $_POST['source_url'] : '',
            'destination_url' => isset($_POST['destination_url']) ? $_POST['destination_url'] : '',
            'redirect_type' => isset($_POST['redirect_type']) ? $_POST['redirect_type'] : '301',
            'is_regex' => isset($_POST['is_regex']) ? filter_var($_POST['is_regex'], FILTER_VALIDATE_BOOLEAN) : false,
            'status' => isset($_POST['status']) ? $_POST['status'] : 'active',
            'notes' => isset($_POST['notes']) ? $_POST['notes'] : '',
            'sources' => isset($_POST['sources']) ? json_decode(stripslashes($_POST['sources']), true) : [], 
        ];
        
        $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
        
        if ($id > 0) {
            // Edit Mode
            if (!empty($data['sources']) && is_array($data['sources'])) {
                // Update the main ID with the FIRST source
                $first_source = $data['sources'][0];
                $update_data = $this->prepare_redirect_data_from_source($data, $first_source);
                $result = $this->update_redirect($id, $update_data);
                
                // Create remaining sources as NEW redirects
                for ($i = 1; $i < count($data['sources']); $i++) {
                    $sub_data = $this->prepare_redirect_data_from_source($data, $data['sources'][$i]);
                    $res = $this->create_redirect($sub_data);
                    if (is_wp_error($res) && !is_wp_error($result)) {
                        $result = $res;
                    }
                }
            } else {
                $result = $this->update_redirect($id, $data);
            }
        } else {
            // Create Mode
            if (!empty($data['sources']) && is_array($data['sources'])) {
                $result = true;
                foreach ($data['sources'] as $source) {
                    $sub_data = $this->prepare_redirect_data_from_source($data, $source);
                    $res = $this->create_redirect($sub_data);
                    if (is_wp_error($res)) {
                        $result = $res;
                    }
                }
            } else {
                $result = $this->create_redirect($data);
            }
        }
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        wp_send_json_success(['message' => __('Redirect saved.', 'rankjet-ai')]);
    }

    /**
     * Helper to prepare redirect data from source definition
     */
    private function prepare_redirect_data_from_source($base_data, $source_def) {
        $data = $base_data;
        $url = $source_def['url'] ?? '';
        $match = $source_def['match'] ?? 'exact';
        $ignore_case = !empty($source_def['ignore_case']);
        
        $is_regex = 0;
        $final_source = $url;
        
        switch ($match) {
             case 'regex':
                 $is_regex = 1;
                 break;
             case 'exact':
                  if ($ignore_case) {
                      $is_regex = 1;
                      $final_source = '#^' . preg_quote($url, '#') . '$#i';
                  } else {
                      $is_regex = 0;
                  }
                  break;
             case 'contains':
                  $is_regex = 1;
                  $final_source = '#' . preg_quote($url, '#') . '#' . ($ignore_case ? 'i' : '');
                  break;
             case 'starts_with':
                  $is_regex = 1;
                  $final_source = '#^' . preg_quote($url, '#') . '#' . ($ignore_case ? 'i' : '');
                  break; 
             case 'ends_with':
                  $is_regex = 1;
                  $final_source = '#' . preg_quote($url, '#') . '$#' . ($ignore_case ? 'i' : '');
                  break;
        }
        
        $data['source_url'] = $final_source;
        $data['is_regex'] = $is_regex;
        return $data;
    }

    /**
     * AJAX: Delete redirect
     */
    public function ajax_delete_redirect() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
        
        if ($id <= 0) {
            wp_send_json_error(['message' => 'Invalid ID']);
        }
        
        $result = $this->delete_redirect($id);
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        wp_send_json_success(['message' => __('Redirect deleted.', 'rankjet-ai')]);
    }

    /**
     * AJAX: Bulk action
     */
    public function ajax_bulk_action() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $action = isset($_POST['bulk_action']) ? sanitize_text_field($_POST['bulk_action']) : '';
        $ids = isset($_POST['ids']) ? array_map('intval', (array) $_POST['ids']) : [];
        
        if (empty($ids)) {
            wp_send_json_error(['message' => 'No items selected']);
        }
        
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;
        $id_placeholders = implode(',', array_fill(0, count($ids), '%d'));
        
        switch ($action) {
            case 'enable':
                $wpdb->query($wpdb->prepare(
                    "UPDATE {$table} SET status = 'active' WHERE id IN ({$id_placeholders})",
                    $ids
                ));
                break;
                
            case 'disable':
                $wpdb->query($wpdb->prepare(
                    "UPDATE {$table} SET status = 'inactive' WHERE id IN ({$id_placeholders})",
                    $ids
                ));
                break;
                
            case 'delete':
                $wpdb->query($wpdb->prepare(
                    "DELETE FROM {$table} WHERE id IN ({$id_placeholders})",
                    $ids
                ));
                break;
                
            default:
                wp_send_json_error(['message' => 'Invalid action']);
        }
        
        // Regenerate cache
        $this->cache->regenerate();
        
        wp_send_json_success(['message' => __('Bulk action completed.', 'rankjet-ai')]);
    }

    /**
     * AJAX: Test a redirect pattern
     */
    public function ajax_test_redirect() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $source = isset($_POST['source_url']) ? $_POST['source_url'] : '';
        $destination = isset($_POST['destination_url']) ? $_POST['destination_url'] : '';
        $is_regex = isset($_POST['is_regex']) ? (bool) $_POST['is_regex'] : false;
        $test_url = isset($_POST['test_url']) ? $_POST['test_url'] : '';
        
        if (empty($test_url)) {
            wp_send_json_error(['message' => 'Test URL is required']);
        }
        
        // Test the pattern
        if ($is_regex) {
            $pattern = $source;
            if ($pattern[0] !== '/' && $pattern[0] !== '#' && $pattern[0] !== '~') {
                $pattern = '/' . $pattern . '/';
            }
            
            if (@preg_match($pattern, $test_url, $matches)) {
                $result_dest = $destination;
                foreach ($matches as $i => $match) {
                    if ($i > 0) {
                        $result_dest = str_replace('$' . $i, $match, $result_dest);
                    }
                }
                wp_send_json_success([
                    'matches' => true,
                    'destination' => $result_dest,
                    'captures' => $matches,
                ]);
            } else {
                wp_send_json_success([
                    'matches' => false,
                    'message' => 'Pattern does not match the test URL',
                ]);
            }
        } else {
            // Exact match test
            $normalized_source = '/' . ltrim(rtrim($source, '/'), '/');
            $normalized_test = '/' . ltrim(rtrim($test_url, '/'), '/');
            
            if ($normalized_source === $normalized_test) {
                wp_send_json_success([
                    'matches' => true,
                    'destination' => $destination,
                ]);
            } else {
                wp_send_json_success([
                    'matches' => false,
                    'message' => 'Source URL does not match the test URL',
                ]);
            }
        }
    }

    /**
     * Install database tables
     */
    public static function install() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . self::TABLE_NAME;
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE {$table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            source_url VARCHAR(2048) NOT NULL,
            destination_url VARCHAR(2048) NOT NULL DEFAULT '',
            redirect_type ENUM('301', '302', '410') NOT NULL DEFAULT '301',
            is_regex TINYINT(1) NOT NULL DEFAULT 0,
            hit_count BIGINT(20) UNSIGNED NOT NULL DEFAULT 0,
            last_accessed DATETIME DEFAULT NULL,
            status ENUM('active', 'inactive') NOT NULL DEFAULT 'active',
            created_by BIGINT(20) UNSIGNED DEFAULT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            notes TEXT DEFAULT NULL,
            PRIMARY KEY (id),
            KEY source_url_hash (source_url(191)),
            KEY status (status),
            KEY redirect_type (redirect_type)
        ) {$charset_collate};";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Store DB version for future migrations
        update_option('rankjet_redirects_db_version', '1.0.0');
    }
    
    /**
     * AJAX: Run setup to create database tables
     */
    public function ajax_run_setup() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        // Install redirects table
        self::install();
        
        // Install 404 logs table
        if (class_exists('Rankjet_Ai_404_Monitor')) {
            Rankjet_Ai_404_Monitor::install();
        }

        // Set default options
        if (get_option('rankjet_auto_redirect_slug_change') === false) {
            add_option('rankjet_auto_redirect_slug_change', 'yes');
        }
        if (get_option('rankjet_404_log_retention') === false) {
            add_option('rankjet_404_log_retention', 30);
        }

        // Verify tables were created
        if ($this->tables_exist()) {
            wp_send_json_success([
                'message' => __('Setup completed successfully!', 'rankjet-ai'),
            ]);
        } else {
            wp_send_json_error([
                'message' => __('Failed to create database tables. Please check permissions.', 'rankjet-ai'),
            ]);
        }
    }

    /**
     * AJAX: Check status (tables exist)
     */
    public function ajax_check_status() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        wp_send_json_success([
            'tables_exist' => $this->tables_exist(),
        ]);
    }

    /**
     * AJAX: Save settings
     */
    public function ajax_save_settings() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $auto_redirect = isset($_POST['auto_redirect_on_slug_change']) ? sanitize_text_field($_POST['auto_redirect_on_slug_change']) : 'yes';
        $log_retention = isset($_POST['log_retention_days']) ? (int) $_POST['log_retention_days'] : 30;

        update_option('rankjet_auto_redirect_slug_change', $auto_redirect);
        update_option('rankjet_404_log_retention', $log_retention);

        wp_send_json_success(['message' => __('Settings saved successfully.', 'rankjet-ai')]);
    }
}
