<?php
/**
 * RankJet AI Redirect Import
 * 
 * Import redirects from competing plugins:
 * - Rank Math
 * - Yoast SEO Premium
 * - Redirection plugin
 * - CSV/JSON files
 * 
 * @package RankJet_AI
 * @since 1.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Rankjet_Ai_Redirect_Import {

    /**
     * Constructor - Register AJAX handlers
     */
    public function __construct() {
        add_action('wp_ajax_rankjet_import_redirects', [$this, 'ajax_import_redirects']);
        add_action('wp_ajax_rankjet_export_redirects', [$this, 'ajax_export_redirects']);
        add_action('wp_ajax_rankjet_detect_plugins', [$this, 'ajax_detect_plugins']);
    }

    /**
     * Detect which SEO/redirect plugins are active
     */
    public function detect_plugins() {
        $plugins = [];

        // Rank Math
        if ($this->has_rank_math_redirects()) {
            $count = $this->count_rank_math_redirects();
            $plugins[] = [
                'id' => 'rankmath',
                'name' => 'Rank Math',
                'count' => $count,
                'available' => $count > 0,
            ];
        }

        // Yoast SEO Premium
        if ($this->has_yoast_redirects()) {
            $count = $this->count_yoast_redirects();
            $plugins[] = [
                'id' => 'yoast',
                'name' => 'Yoast SEO Premium',
                'count' => $count,
                'available' => $count > 0,
            ];
        }

        // Redirection plugin
        if ($this->has_redirection_plugin()) {
            $count = $this->count_redirection_redirects();
            $plugins[] = [
                'id' => 'redirection',
                'name' => 'Redirection',
                'count' => $count,
                'available' => $count > 0,
            ];
        }

        return $plugins;
    }

    /**
     * Check if Rank Math redirects exist
     */
    private function has_rank_math_redirects() {
        global $wpdb;
        $table = $wpdb->prefix . 'rank_math_redirections';
        return $wpdb->get_var("SHOW TABLES LIKE '{$table}'") === $table;
    }

    /**
     * Count Rank Math redirects
     */
    private function count_rank_math_redirects() {
        global $wpdb;
        $table = $wpdb->prefix . 'rank_math_redirections';
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
    }

    /**
     * Check if Yoast redirects exist
     */
    private function has_yoast_redirects() {
        $redirects = get_option('wpseo_redirect');
        return !empty($redirects) && is_array($redirects);
    }

    /**
     * Count Yoast redirects
     */
    private function count_yoast_redirects() {
        $redirects = get_option('wpseo_redirect', []);
        return is_array($redirects) ? count($redirects) : 0;
    }

    /**
     * Check if Redirection plugin exists
     */
    private function has_redirection_plugin() {
        global $wpdb;
        $table = $wpdb->prefix . 'redirection_items';
        return $wpdb->get_var("SHOW TABLES LIKE '{$table}'") === $table;
    }

    /**
     * Count Redirection plugin redirects
     */
    private function count_redirection_redirects() {
        global $wpdb;
        $table = $wpdb->prefix . 'redirection_items';
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
    }

    /**
     * Import from Rank Math
     */
    public function import_from_rank_math() {
        global $wpdb;
        
        $source_table = $wpdb->prefix . 'rank_math_redirections';
        $target_table = $wpdb->prefix . 'rankjet_redirects';
        
        $redirects = $wpdb->get_results("SELECT * FROM {$source_table}");
        
        $imported = 0;
        $skipped = 0;
        $errors = [];

        foreach ($redirects as $redirect) {
            // Map Rank Math type to our type
            $type = '301';
            if ($redirect->header_code === '302' || $redirect->header_code === '307') {
                $type = '302';
            } elseif ($redirect->header_code === '410' || $redirect->header_code === '451') {
                $type = '410';
            }

            // Parse sources (Rank Math stores as serialized array)
            $sources = maybe_unserialize($redirect->sources);
            if (!is_array($sources)) {
                $sources = [['pattern' => $redirect->sources, 'comparison' => 'exact']];
            }

            foreach ($sources as $source) {
                $source_url = is_array($source) ? ($source['pattern'] ?? '') : $source;
                $is_regex = is_array($source) && isset($source['comparison']) && $source['comparison'] === 'regex';

                if (empty($source_url)) {
                    continue;
                }

                // Check for duplicate
                $exists = $wpdb->get_var($wpdb->prepare(
                    "SELECT id FROM {$target_table} WHERE source_url = %s",
                    $source_url
                ));

                if ($exists) {
                    $skipped++;
                    continue;
                }

                $result = $wpdb->insert($target_table, [
                    'source_url' => $source_url,
                    'destination_url' => $redirect->url_to,
                    'redirect_type' => $type,
                    'is_regex' => $is_regex ? 1 : 0,
                    'status' => $redirect->status === 'active' ? 'active' : 'inactive',
                    'created_by' => get_current_user_id(),
                    'notes' => __('Imported from Rank Math', 'rankjet-ai'),
                ], ['%s', '%s', '%s', '%d', '%s', '%d', '%s']);

                if ($result) {
                    $imported++;
                } else {
                    $errors[] = $source_url;
                }
            }
        }

        // Regenerate cache
        $cache = new Rankjet_Ai_Redirect_Cache();
        $cache->regenerate();

        return [
            'imported' => $imported,
            'skipped' => $skipped,
            'errors' => $errors,
        ];
    }

    /**
     * Import from Yoast SEO Premium
     */
    public function import_from_yoast() {
        global $wpdb;
        
        $target_table = $wpdb->prefix . 'rankjet_redirects';
        $redirects = get_option('wpseo_redirect', []);
        
        if (!is_array($redirects)) {
            return ['imported' => 0, 'skipped' => 0, 'errors' => []];
        }

        $imported = 0;
        $skipped = 0;
        $errors = [];

        foreach ($redirects as $old_url => $redirect) {
            $type = '301';
            if (isset($redirect['type'])) {
                if ($redirect['type'] === '302' || $redirect['type'] === '307') {
                    $type = '302';
                } elseif ($redirect['type'] === '410' || $redirect['type'] === '451') {
                    $type = '410';
                }
            }

            $destination = isset($redirect['url']) ? $redirect['url'] : '';
            
            // Check for duplicate
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$target_table} WHERE source_url = %s",
                $old_url
            ));

            if ($exists) {
                $skipped++;
                continue;
            }

            $result = $wpdb->insert($target_table, [
                'source_url' => $old_url,
                'destination_url' => $destination,
                'redirect_type' => $type,
                'is_regex' => 0,
                'status' => 'active',
                'created_by' => get_current_user_id(),
                'notes' => __('Imported from Yoast SEO', 'rankjet-ai'),
            ], ['%s', '%s', '%s', '%d', '%s', '%d', '%s']);

            if ($result) {
                $imported++;
            } else {
                $errors[] = $old_url;
            }
        }

        // Regenerate cache
        $cache = new Rankjet_Ai_Redirect_Cache();
        $cache->regenerate();

        return [
            'imported' => $imported,
            'skipped' => $skipped,
            'errors' => $errors,
        ];
    }

    /**
     * Import from Redirection plugin
     */
    public function import_from_redirection() {
        global $wpdb;
        
        $source_table = $wpdb->prefix . 'redirection_items';
        $target_table = $wpdb->prefix . 'rankjet_redirects';
        
        $redirects = $wpdb->get_results("SELECT * FROM {$source_table}");
        
        $imported = 0;
        $skipped = 0;
        $errors = [];

        foreach ($redirects as $redirect) {
            // Map action type
            $type = '301';
            if ($redirect->action_code === 302 || $redirect->action_code === 307) {
                $type = '302';
            } elseif ($redirect->action_code === 410) {
                $type = '410';
            }

            // Check for duplicate
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$target_table} WHERE source_url = %s",
                $redirect->url
            ));

            if ($exists) {
                $skipped++;
                continue;
            }

            // Determine if regex
            $is_regex = $redirect->regex == 1;

            $result = $wpdb->insert($target_table, [
                'source_url' => $redirect->url,
                'destination_url' => $redirect->action_data,
                'redirect_type' => $type,
                'is_regex' => $is_regex ? 1 : 0,
                'hit_count' => $redirect->last_count ?? 0,
                'status' => $redirect->status === 'enabled' ? 'active' : 'inactive',
                'created_by' => get_current_user_id(),
                'notes' => __('Imported from Redirection plugin', 'rankjet-ai'),
            ], ['%s', '%s', '%s', '%d', '%d', '%s', '%d', '%s']);

            if ($result) {
                $imported++;
            } else {
                $errors[] = $redirect->url;
            }
        }

        // Regenerate cache
        $cache = new Rankjet_Ai_Redirect_Cache();
        $cache->regenerate();

        return [
            'imported' => $imported,
            'skipped' => $skipped,
            'errors' => $errors,
        ];
    }

    /**
     * Import from CSV
     */
    public function import_from_csv($csv_content) {
        global $wpdb;
        $target_table = $wpdb->prefix . 'rankjet_redirects';

        $lines = explode("\n", $csv_content);
        $imported = 0;
        $skipped = 0;
        $errors = [];

        $header = null;

        foreach ($lines as $line_num => $line) {
            $line = trim($line);
            if (empty($line)) {
                continue;
            }

            $fields = str_getcsv($line);

            // First line is header
            if ($header === null) {
                $header = array_map('strtolower', array_map('trim', $fields));
                continue;
            }

            // Map fields
            $data = array_combine($header, $fields);
            
            $source = isset($data['source']) ? trim($data['source']) : (isset($data['source_url']) ? trim($data['source_url']) : '');
            $destination = isset($data['destination']) ? trim($data['destination']) : (isset($data['destination_url']) ? trim($data['destination_url']) : '');
            $type = isset($data['type']) ? trim($data['type']) : (isset($data['redirect_type']) ? trim($data['redirect_type']) : '301');

            if (empty($source)) {
                continue;
            }

            // Normalize type
            if (!in_array($type, ['301', '302', '410'])) {
                $type = '301';
            }

            // Check for duplicate
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$target_table} WHERE source_url = %s",
                $source
            ));

            if ($exists) {
                $skipped++;
                continue;
            }

            $result = $wpdb->insert($target_table, [
                'source_url' => $source,
                'destination_url' => $destination,
                'redirect_type' => $type,
                'is_regex' => 0,
                'status' => 'active',
                'created_by' => get_current_user_id(),
                'notes' => __('Imported from CSV', 'rankjet-ai'),
            ], ['%s', '%s', '%s', '%d', '%s', '%d', '%s']);

            if ($result) {
                $imported++;
            } else {
                $errors[] = $source;
            }
        }

        // Regenerate cache
        $cache = new Rankjet_Ai_Redirect_Cache();
        $cache->regenerate();

        return [
            'imported' => $imported,
            'skipped' => $skipped,
            'errors' => $errors,
        ];
    }

    /**
     * Import from JSON
     */
    public function import_from_json($json_content) {
        global $wpdb;
        $target_table = $wpdb->prefix . 'rankjet_redirects';

        $data = json_decode($json_content, true);
        if (!is_array($data)) {
            return ['imported' => 0, 'skipped' => 0, 'errors' => ['Invalid JSON']];
        }

        $imported = 0;
        $skipped = 0;
        $errors = [];

        foreach ($data as $item) {
            $source = isset($item['source_url']) ? trim($item['source_url']) : (isset($item['source']) ? trim($item['source']) : '');
            $destination = isset($item['destination_url']) ? trim($item['destination_url']) : (isset($item['destination']) ? trim($item['destination']) : '');
            $type = isset($item['redirect_type']) ? trim($item['redirect_type']) : (isset($item['type']) ? trim($item['type']) : '301');
            $is_regex = isset($item['is_regex']) ? (bool) $item['is_regex'] : false;

            if (empty($source)) {
                continue;
            }

            // Normalize type
            if (!in_array($type, ['301', '302', '410'])) {
                $type = '301';
            }

            // Check for duplicate
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$target_table} WHERE source_url = %s",
                $source
            ));

            if ($exists) {
                $skipped++;
                continue;
            }

            $result = $wpdb->insert($target_table, [
                'source_url' => $source,
                'destination_url' => $destination,
                'redirect_type' => $type,
                'is_regex' => $is_regex ? 1 : 0,
                'status' => 'active',
                'created_by' => get_current_user_id(),
                'notes' => isset($item['notes']) ? sanitize_textarea_field($item['notes']) : __('Imported from JSON', 'rankjet-ai'),
            ], ['%s', '%s', '%s', '%d', '%s', '%d', '%s']);

            if ($result) {
                $imported++;
            } else {
                $errors[] = $source;
            }
        }

        // Regenerate cache
        $cache = new Rankjet_Ai_Redirect_Cache();
        $cache->regenerate();

        return [
            'imported' => $imported,
            'skipped' => $skipped,
            'errors' => $errors,
        ];
    }

    /**
     * Export redirects to JSON
     */
    public function export_to_json() {
        global $wpdb;
        $table = $wpdb->prefix . 'rankjet_redirects';

        $redirects = $wpdb->get_results("SELECT * FROM {$table} ORDER BY created_at DESC");

        $export = [];
        foreach ($redirects as $redirect) {
            $export[] = [
                'source_url' => $redirect->source_url,
                'destination_url' => $redirect->destination_url,
                'redirect_type' => $redirect->redirect_type,
                'is_regex' => (bool) $redirect->is_regex,
                'status' => $redirect->status,
                'notes' => $redirect->notes,
            ];
        }

        return json_encode($export, JSON_PRETTY_PRINT);
    }

    /**
     * Export redirects to CSV
     */
    public function export_to_csv() {
        global $wpdb;
        $table = $wpdb->prefix . 'rankjet_redirects';

        $redirects = $wpdb->get_results("SELECT * FROM {$table} ORDER BY created_at DESC");

        $csv = "source_url,destination_url,redirect_type,is_regex,status,notes\n";

        foreach ($redirects as $redirect) {
            $csv .= sprintf(
                '"%s","%s","%s","%s","%s","%s"' . "\n",
                str_replace('"', '""', $redirect->source_url),
                str_replace('"', '""', $redirect->destination_url),
                $redirect->redirect_type,
                $redirect->is_regex ? 'yes' : 'no',
                $redirect->status,
                str_replace('"', '""', $redirect->notes ?? '')
            );
        }

        return $csv;
    }

    /**
     * AJAX: Detect available plugins
     */
    public function ajax_detect_plugins() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        wp_send_json_success(['plugins' => $this->detect_plugins()]);
    }

    /**
     * AJAX: Import redirects
     */
    public function ajax_import_redirects() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $source = isset($_POST['source']) ? sanitize_text_field($_POST['source']) : '';
        $result = [];

        switch ($source) {
            case 'rankmath':
                $result = $this->import_from_rank_math();
                break;

            case 'yoast':
                $result = $this->import_from_yoast();
                break;

            case 'redirection':
                $result = $this->import_from_redirection();
                break;

            case 'csv':
                if (isset($_FILES['file'])) {
                    $content = file_get_contents($_FILES['file']['tmp_name']);
                    $result = $this->import_from_csv($content);
                } elseif (isset($_POST['content'])) {
                    $result = $this->import_from_csv($_POST['content']);
                } else {
                    wp_send_json_error(['message' => 'No file uploaded']);
                }
                break;

            case 'json':
                if (isset($_FILES['file'])) {
                    $content = file_get_contents($_FILES['file']['tmp_name']);
                    $result = $this->import_from_json($content);
                } elseif (isset($_POST['content'])) {
                    $result = $this->import_from_json($_POST['content']);
                } else {
                    wp_send_json_error(['message' => 'No file uploaded']);
                }
                break;

            default:
                wp_send_json_error(['message' => 'Unknown import source']);
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Export redirects
     */
    public function ajax_export_redirects() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $format = isset($_GET['format']) ? sanitize_text_field($_GET['format']) : 'json';

        if ($format === 'csv') {
            $content = $this->export_to_csv();
            $filename = 'rankjet-redirects-' . date('Y-m-d') . '.csv';
            $content_type = 'text/csv';
        } else {
            $content = $this->export_to_json();
            $filename = 'rankjet-redirects-' . date('Y-m-d') . '.json';
            $content_type = 'application/json';
        }

        header('Content-Type: ' . $content_type);
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($content));
        
        echo $content;
        exit;
    }
}
