<?php
/**
 * RankJet AI Redirect Cache
 * 
 * Flat-file caching system for ultra-fast redirect matching.
 * Exports active redirect rules to a PHP array file that can be
 * included directly without database queries.
 * 
 * @package RankJet_AI
 * @since 1.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Rankjet_Ai_Redirect_Cache {

    /**
     * Cache directory path
     */
    private $cache_dir;

    /**
     * Cache file path
     */
    private $cache_file;

    /**
     * In-memory cache for current request
     */
    private static $rules_cache = null;

    /**
     * Constructor
     */
    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->cache_dir = $upload_dir['basedir'] . '/rankjet-cache';
        $this->cache_file = $this->cache_dir . '/redirects.php';
    }

    /**
     * Get redirect rules from cache
     * 
     * @return array|false Cached rules or false if cache miss
     */
    public function get_rules() {
        // Return in-memory cache if available
        if (self::$rules_cache !== null) {
            return self::$rules_cache;
        }

        // Check if cache file exists and is readable
        if (!file_exists($this->cache_file) || !is_readable($this->cache_file)) {
            // Try to regenerate cache
            $this->regenerate();
            
            if (!file_exists($this->cache_file)) {
                return false;
            }
        }

        // Include the cache file
        $rules = @include($this->cache_file);
        
        if (!is_array($rules)) {
            return false;
        }

        // Store in memory for this request
        self::$rules_cache = $rules;

        return $rules;
    }

    /**
     * Regenerate the cache file from database
     */
    public function regenerate() {
        global $wpdb;
        
        // Ensure cache directory exists
        if (!$this->ensure_cache_directory()) {
            error_log('RankJet AI: Failed to create cache directory');
            return false;
        }

        $table = $wpdb->prefix . 'rankjet_redirects';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") !== $table) {
            return false;
        }

        // Fetch all active redirects
        $redirects = $wpdb->get_results(
            "SELECT id, source_url, destination_url, redirect_type, is_regex 
             FROM {$table} 
             WHERE status = 'active' 
             ORDER BY is_regex ASC, id ASC"
        );

        $exact = [];
        $regex = [];

        foreach ($redirects as $redirect) {
            $rule = [
                'id' => (int) $redirect->id,
                'dest' => $redirect->destination_url,
                'type' => $redirect->redirect_type,
            ];

            if ($redirect->is_regex) {
                // Ensure pattern has delimiters
                $pattern = $redirect->source_url;
                if ($pattern[0] !== '/' && $pattern[0] !== '#' && $pattern[0] !== '~') {
                    $pattern = '/' . $pattern . '/';
                }
                $regex[$pattern] = $rule;
            } else {
                // Normalize source URL
                $source = '/' . ltrim(rtrim($redirect->source_url, '/'), '/');
                $exact[$source] = $rule;
            }
        }

        $rules = [
            'exact' => $exact,
            'regex' => $regex,
        ];

        // Generate cache file content
        $content = $this->generate_cache_content($rules);

        // Write to cache file
        $result = $this->write_cache_file($content);

        if ($result) {
            // Clear in-memory cache
            self::$rules_cache = null;
        }

        return $result;
    }

    /**
     * Generate PHP cache file content
     */
    private function generate_cache_content($rules) {
        $timestamp = current_time('Y-m-d H:i:s');
        $total_exact = count($rules['exact']);
        $total_regex = count($rules['regex']);

        $content = "<?php\n";
        $content .= "/**\n";
        $content .= " * RankJet AI Redirect Cache\n";
        $content .= " * Generated: {$timestamp}\n";
        $content .= " * Exact Rules: {$total_exact}\n";
        $content .= " * Regex Rules: {$total_regex}\n";
        $content .= " * \n";
        $content .= " * DO NOT EDIT - This file is auto-generated\n";
        $content .= " */\n\n";
        $content .= "defined('ABSPATH') || exit;\n\n";
        $content .= "return " . var_export($rules, true) . ";\n";

        return $content;
    }

    /**
     * Write content to cache file atomically
     */
    private function write_cache_file($content) {
        $temp_file = $this->cache_file . '.tmp';

        // Write to temp file first
        $written = file_put_contents($temp_file, $content, LOCK_EX);

        if ($written === false) {
            error_log('RankJet AI: Failed to write cache temp file');
            return false;
        }

        // Rename atomically
        if (!@rename($temp_file, $this->cache_file)) {
            // Fallback to copy + delete
            if (!@copy($temp_file, $this->cache_file)) {
                error_log('RankJet AI: Failed to copy cache file');
                @unlink($temp_file);
                return false;
            }
            @unlink($temp_file);
        }

        // Set proper permissions
        @chmod($this->cache_file, 0644);

        return true;
    }

    /**
     * Ensure cache directory exists
     */
    private function ensure_cache_directory() {
        if (!file_exists($this->cache_dir)) {
            if (!wp_mkdir_p($this->cache_dir)) {
                return false;
            }

            // Add index.php for security
            file_put_contents($this->cache_dir . '/index.php', "<?php\n// Silence is golden.");
            
            // Add .htaccess for extra protection
            file_put_contents($this->cache_dir . '/.htaccess', "deny from all");
        }

        return is_dir($this->cache_dir) && is_writable($this->cache_dir);
    }

    /**
     * Clear the cache
     */
    public function clear() {
        self::$rules_cache = null;

        if (file_exists($this->cache_file)) {
            @unlink($this->cache_file);
        }
    }

    /**
     * Get cache statistics
     */
    public function get_stats() {
        if (!file_exists($this->cache_file)) {
            return [
                'exists' => false,
                'size' => 0,
                'modified' => null,
                'exact_count' => 0,
                'regex_count' => 0,
            ];
        }

        $rules = $this->get_rules();

        return [
            'exists' => true,
            'size' => filesize($this->cache_file),
            'modified' => filemtime($this->cache_file),
            'exact_count' => $rules ? count($rules['exact']) : 0,
            'regex_count' => $rules ? count($rules['regex']) : 0,
        ];
    }
}
