<?php
/**
 * RankJet AI 404 Monitor
 * 
 * Logs 404 errors for analysis and provides AI-powered redirect suggestions.
 * Includes rate limiting to prevent spam and automatic log pruning.
 * 
 * @package RankJet_AI
 * @since 1.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Rankjet_Ai_404_Monitor {

    /**
     * Database table name (without prefix)
     */
    const TABLE_NAME = 'rankjet_404_logs';

    /**
     * Rate limit: minimum seconds between logging same URL
     */
    const RATE_LIMIT_SECONDS = 60;

    /**
     * Transient prefix for rate limiting
     */
    const RATE_LIMIT_PREFIX = 'rankjet_404_rate_';

    /**
     * AI suggestion handler
     */
    private $ai;

    /**
     * Constructor - Register hooks
     */
    public function __construct() {
        // Hook into template_redirect to log 404s
        add_action('template_redirect', [$this, 'log_404'], 99);
        
        // Schedule cleanup cron
        add_action('rankjet_404_cleanup', [$this, 'cleanup_old_logs']);
        
        if (!wp_next_scheduled('rankjet_404_cleanup')) {
            wp_schedule_event(time(), 'daily', 'rankjet_404_cleanup');
        }

        // AJAX handlers
        add_action('wp_ajax_rankjet_get_404_logs', [$this, 'ajax_get_logs']);
        add_action('wp_ajax_rankjet_resolve_404', [$this, 'ajax_resolve_404']);
        add_action('wp_ajax_rankjet_ignore_404', [$this, 'ajax_ignore_404']);
        add_action('wp_ajax_rankjet_delete_404', [$this, 'ajax_delete_404']);
        add_action('wp_ajax_rankjet_get_ai_suggestion', [$this, 'ajax_get_ai_suggestion']);
        add_action('wp_ajax_rankjet_bulk_action_404', [$this, 'ajax_bulk_action']);
    }

    /**
     * Log a 404 error
     */
    public function log_404() {
        if (!is_404()) {
            return;
        }

        // Don't log in admin
        if (is_admin()) {
            return;
        }

        $request_url = $this->get_request_url();
        
        // Skip if URL should be ignored
        if ($this->should_ignore($request_url)) {
            return;
        }

        // Rate limiting check
        if ($this->is_rate_limited($request_url)) {
            return;
        }

        $this->record_404($request_url);
    }

    /**
     * Get the current request URL
     */
    private function get_request_url() {
        $url = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
        return '/' . ltrim($url, '/');
    }

    /**
     * Check if URL should be ignored
     */
    private function should_ignore($url) {
        // Ignore patterns
        $ignore_patterns = [
            '/wp-admin',
            '/wp-login',
            '/wp-cron',
            '/wp-json',
            '/xmlrpc.php',
            '/favicon.ico',
            '/robots.txt',
            '/sitemap',
            '.xml',
            '.css',
            '.js',
            '.map',
            '.woff',
            '.woff2',
            '.ttf',
            '.eot',
            '.svg',
            '.png',
            '.jpg',
            '.jpeg',
            '.gif',
            '.webp',
            '.ico',
            '.pdf',
            '.zip',
            '.rar',
        ];

        $url_lower = strtolower($url);
        
        foreach ($ignore_patterns as $pattern) {
            if (strpos($url_lower, strtolower($pattern)) !== false) {
                return true;
            }
        }

        // Check custom ignore patterns from settings
        $custom_ignore = get_option('rankjet_404_ignore_patterns', '');
        if (!empty($custom_ignore)) {
            $patterns = array_filter(array_map('trim', explode("\n", $custom_ignore)));
            foreach ($patterns as $pattern) {
                if (strpos($url_lower, strtolower($pattern)) !== false) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Check if URL is rate limited
     */
    private function is_rate_limited($url) {
        $key = self::RATE_LIMIT_PREFIX . md5($url);
        
        if (get_transient($key)) {
            return true;
        }

        // Set rate limit
        set_transient($key, 1, self::RATE_LIMIT_SECONDS);
        
        return false;
    }

    /**
     * Record a 404 error in the database
     */
    private function record_404($request_url) {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;

        // Check if already logged
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT id, hit_count FROM {$table} WHERE request_url = %s",
            $request_url
        ));

        if ($existing) {
            // Update hit count
            $wpdb->update(
                $table,
                [
                    'hit_count' => $existing->hit_count + 1,
                    'last_hit' => current_time('mysql'),
                    'referrer_url' => isset($_SERVER['HTTP_REFERER']) ? substr($_SERVER['HTTP_REFERER'], 0, 2048) : null,
                    'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? substr($_SERVER['HTTP_USER_AGENT'], 0, 512) : null,
                ],
                ['id' => $existing->id],
                ['%d', '%s', '%s', '%s'],
                ['%d']
            );
        } else {
            // Insert new record
            $wpdb->insert($table, [
                'request_url' => substr($request_url, 0, 2048),
                'referrer_url' => isset($_SERVER['HTTP_REFERER']) ? substr($_SERVER['HTTP_REFERER'], 0, 2048) : null,
                'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? substr($_SERVER['HTTP_USER_AGENT'], 0, 512) : null,
                'ip_address' => $this->get_client_ip(),
                'hit_count' => 1,
                'first_hit' => current_time('mysql'),
                'last_hit' => current_time('mysql'),
            ], ['%s', '%s', '%s', '%s', '%d', '%s', '%s']);
        }
    }

    /**
     * Get client IP address
     */
    private function get_client_ip() {
        $ip_keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
        
        foreach ($ip_keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                // Handle comma-separated IPs (X-Forwarded-For)
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return '';
    }

    /**
     * Cleanup old logs based on retention setting
     */
    public function cleanup_old_logs() {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;
        
        $retention_days = (int) get_option('rankjet_404_log_retention', 30);
        if ($retention_days < 1) {
            $retention_days = 30;
        }

        $wpdb->query($wpdb->prepare(
            "DELETE FROM {$table} WHERE last_hit < DATE_SUB(NOW(), INTERVAL %d DAY)",
            $retention_days
        ));
    }

    /**
     * Get 404 logs with pagination and filtering
     */
    public function get_logs($args = []) {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;

        $defaults = [
            'per_page' => 20,
            'page' => 1,
            'orderby' => 'hit_count',
            'order' => 'DESC',
            'search' => '',
            'status' => '', // '', 'unresolved', 'ignored', 'resolved'
        ];

        $args = wp_parse_args($args, $defaults);

        $where = ['1=1'];
        $where_values = [];

        if (!empty($args['search'])) {
            $where[] = 'request_url LIKE %s';
            $where_values[] = '%' . $wpdb->esc_like($args['search']) . '%';
        }

        switch ($args['status']) {
            case 'unresolved':
                $where[] = 'is_resolved = 0 AND is_ignored = 0';
                break;
            case 'ignored':
                $where[] = 'is_ignored = 1';
                break;
            case 'resolved':
                $where[] = 'is_resolved = 1';
                break;
        }

        $where_sql = implode(' AND ', $where);

        // Count total
        $count_sql = "SELECT COUNT(*) FROM {$table} WHERE {$where_sql}";
        if (!empty($where_values)) {
            $count_sql = $wpdb->prepare($count_sql, $where_values);
        }
        $total = (int) $wpdb->get_var($count_sql);

        // Get items
        $orderby = sanitize_sql_orderby($args['orderby'] . ' ' . $args['order']) ?: 'hit_count DESC';
        $offset = ($args['page'] - 1) * $args['per_page'];

        $query = "SELECT * FROM {$table} WHERE {$where_sql} ORDER BY {$orderby} LIMIT %d OFFSET %d";
        $query_values = array_merge($where_values, [$args['per_page'], $offset]);

        $items = $wpdb->get_results($wpdb->prepare($query, $query_values));

        return [
            'items' => $items,
            'total' => $total,
            'pages' => ceil($total / $args['per_page']),
        ];
    }

    /**
     * AJAX: Get 404 logs
     */
    public function ajax_get_logs() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $args = [
            'page' => isset($_GET['page']) ? (int) $_GET['page'] : 1,
            'per_page' => isset($_GET['per_page']) ? (int) $_GET['per_page'] : 20,
            'search' => isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '',
            'status' => isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '',
            'orderby' => isset($_GET['orderby']) ? sanitize_text_field($_GET['orderby']) : 'hit_count',
            'order' => isset($_GET['order']) ? sanitize_text_field($_GET['order']) : 'DESC',
        ];

        $result = $this->get_logs($args);
        wp_send_json_success($result);
    }

    /**
     * AJAX: Mark 404 as resolved (and optionally create redirect)
     */
    public function ajax_resolve_404() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
        $redirect_to = isset($_POST['redirect_to']) ? sanitize_text_field($_POST['redirect_to']) : '';

        if ($id <= 0) {
            wp_send_json_error(['message' => 'Invalid ID']);
        }

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;

        // Get the 404 record
        $record = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $id
        ));

        if (!$record) {
            wp_send_json_error(['message' => '404 record not found']);
        }

        // Create redirect if destination provided
        if (!empty($redirect_to)) {
            $redirects = new Rankjet_Ai_Redirects();
            $redirect_result = $redirects->create_redirect([
                'source_url' => $record->request_url,
                'destination_url' => $redirect_to,
                'redirect_type' => '301',
                'notes' => __('Created from 404 Monitor', 'rankjet-ai'),
            ]);

            if (is_wp_error($redirect_result)) {
                wp_send_json_error(['message' => $redirect_result->get_error_message()]);
            }

            // Update 404 record with redirect ID
            $wpdb->update(
                $table,
                [
                    'is_resolved' => 1,
                    'suggested_redirect_id' => $redirect_result,
                ],
                ['id' => $id],
                ['%d', '%d'],
                ['%d']
            );
        } else {
            // Just mark as resolved without creating redirect
            $wpdb->update(
                $table,
                ['is_resolved' => 1],
                ['id' => $id],
                ['%d'],
                ['%d']
            );
        }

        wp_send_json_success(['message' => __('404 resolved.', 'rankjet-ai')]);
    }

    /**
     * AJAX: Ignore a 404
     */
    public function ajax_ignore_404() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

        if ($id <= 0) {
            wp_send_json_error(['message' => 'Invalid ID']);
        }

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;

        $wpdb->update(
            $table,
            ['is_ignored' => 1],
            ['id' => $id],
            ['%d'],
            ['%d']
        );

        wp_send_json_success(['message' => __('404 ignored.', 'rankjet-ai')]);
    }

    /**
     * AJAX: Delete a 404 record
     */
    public function ajax_delete_404() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

        if ($id <= 0) {
            wp_send_json_error(['message' => 'Invalid ID']);
        }

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;

        $wpdb->delete($table, ['id' => $id], ['%d']);

        wp_send_json_success(['message' => __('404 deleted.', 'rankjet-ai')]);
    }

    /**
     * AJAX: Get AI suggestion for a 404
     */
    public function ajax_get_ai_suggestion() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $url = isset($_POST['url']) ? sanitize_text_field($_POST['url']) : '';

        if (empty($url)) {
            wp_send_json_error(['message' => 'URL is required']);
        }

        // Initialize AI handler if not already
        if (!$this->ai) {
            $this->ai = new Rankjet_Ai_Redirect_AI();
        }

        $suggestions = $this->ai->suggest_redirect($url);

        wp_send_json_success(['suggestions' => $suggestions]);
    }

    /**
     * AJAX: Bulk action
     */
    public function ajax_bulk_action() {
        check_ajax_referer('rankjet_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $action = isset($_POST['bulk_action']) ? sanitize_text_field($_POST['bulk_action']) : '';
        $ids = isset($_POST['ids']) ? array_map('intval', (array) $_POST['ids']) : [];

        if (empty($ids)) {
            wp_send_json_error(['message' => 'No items selected']);
        }

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_NAME;
        $id_placeholders = implode(',', array_fill(0, count($ids), '%d'));

        switch ($action) {
            case 'ignore':
                $wpdb->query($wpdb->prepare(
                    "UPDATE {$table} SET is_ignored = 1 WHERE id IN ({$id_placeholders})",
                    $ids
                ));
                break;

            case 'unignore':
                $wpdb->query($wpdb->prepare(
                    "UPDATE {$table} SET is_ignored = 0 WHERE id IN ({$id_placeholders})",
                    $ids
                ));
                break;

            case 'delete':
                $wpdb->query($wpdb->prepare(
                    "DELETE FROM {$table} WHERE id IN ({$id_placeholders})",
                    $ids
                ));
                break;

            default:
                wp_send_json_error(['message' => 'Invalid action']);
        }

        wp_send_json_success(['message' => __('Bulk action completed.', 'rankjet-ai')]);
    }

    /**
     * Install database table
     */
    public static function install() {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            request_url VARCHAR(2048) NOT NULL,
            referrer_url VARCHAR(2048) DEFAULT NULL,
            user_agent VARCHAR(512) DEFAULT NULL,
            ip_address VARCHAR(45) DEFAULT NULL,
            hit_count BIGINT(20) UNSIGNED NOT NULL DEFAULT 1,
            first_hit DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            last_hit DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            is_ignored TINYINT(1) NOT NULL DEFAULT 0,
            is_resolved TINYINT(1) NOT NULL DEFAULT 0,
            suggested_redirect_id BIGINT(20) UNSIGNED DEFAULT NULL,
            ai_suggestion VARCHAR(2048) DEFAULT NULL,
            ai_confidence DECIMAL(5,4) DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY request_url_hash (request_url(191)),
            KEY is_resolved (is_resolved),
            KEY hit_count (hit_count)
        ) {$charset_collate};";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        update_option('rankjet_404_logs_db_version', '1.0.0');
    }
}
