<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Rankjet_Ai_Platform_Integration {

    private $api_base_url;
    private $api_token;
    private $website_id;

    public function __construct() {
        // Set the API base URL based on environment or setting
        $config_url = get_option('rankjet_platform_api_url');
        if (!empty($config_url)) {
            $this->api_base_url = rtrim($config_url, '/');
        } else {
            $this->api_base_url = defined('RANKJET_API_URL') ? RANKJET_API_URL : 'https://app.rankjet.ai/api/v1';
        }
        
        // Load saved credentials
        $this->api_token = get_option('rankjet_platform_api_token', '');
        $this->website_id = get_option('rankjet_platform_website_id', '');

        // Add admin hooks
        // add_action('admin_menu', [$this, 'add_platform_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('wp_ajax_rankjet_test_connection', [$this, 'test_connection']);
        add_action('wp_ajax_rankjet_sync_data', [$this, 'sync_data']);
        add_action('wp_ajax_rankjet_save_token', [$this, 'save_token_ajax']);
        
        // Add cron hooks for automatic sync
        add_action('rankjet_sync_to_platform', [$this, 'auto_sync_to_platform']);
        
        // Schedule cron if not already scheduled
        if (!wp_next_scheduled('rankjet_sync_to_platform')) {
            wp_schedule_event(time(), 'daily', 'rankjet_sync_to_platform');
        }
    }

    /**
     * Add platform integration menu
     */
    public function add_platform_menu() {
        add_submenu_page(
            'rankjet-ai',
            __('Integration', 'rankjet-ai'),
            __('Integration', 'rankjet-ai'),
            'manage_options',
            'rankjet-ai-integration',
            [$this, 'render_settings_page']
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('rankjet_platform_settings', 'rankjet_platform_api_token');
        register_setting('rankjet_platform_settings', 'rankjet_platform_website_id');
        register_setting('rankjet_platform_settings', 'rankjet_platform_api_url');
        register_setting('rankjet_platform_settings', 'rankjet_platform_auto_sync');
        register_setting('rankjet_platform_settings', 'rankjet_platform_last_sync');
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        // Enqueue the build script
        $asset_file = include( plugin_dir_path( __DIR__ ) . '../build/index.asset.php');
        
        wp_enqueue_script(
            'rankjet-ai-build',
            RANKJET_AI_PLUGIN_URL . 'build/index.js',
            array_merge($asset_file['dependencies'], ['wp-element', 'wp-components', 'wp-api-fetch']),
            $asset_file['version'],
            true
        );

        wp_enqueue_style('rankjet-ai-style', RANKJET_AI_PLUGIN_URL . 'build/index.css', ['wp-components'], $asset_file['version']);

        // Data for React
        $this->api_token = get_option('rankjet_platform_api_token', '');
        $this->website_id = get_option('rankjet_platform_website_id', '');
        
        $status = $this->check_connection_status();
        
        $settings_data = [
            'api_token' => $this->api_token,
            'website_id' => $this->website_id,
            'auto_sync' => get_option('rankjet_platform_auto_sync', 'yes'),
            'last_sync' => get_option('rankjet_platform_last_sync', ''),
            'connection_status' => $status,
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonces' => [
                'save_token' => wp_create_nonce('rankjet_save_token'),
                'test_connection' => wp_create_nonce('rankjet_test_connection'),
                'sync_data' => wp_create_nonce('rankjet_sync_data')
            ]
        ];

        wp_localize_script('rankjet-ai-build', 'rankjetSettings', $settings_data);
        
        ?>
        <div class="wrap">
            <div id="rankjet-settings-root"></div>
        </div>

        <?php
    }

    /**
     * Save token via AJAX
     */
    public function save_token_ajax() {
        check_ajax_referer('rankjet_save_token', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die();
        }

        $token = sanitize_text_field($_POST['token']);
        $website_id = sanitize_text_field($_POST['website_id']);
        // API URL is saved via normal form submission usually, but if updated during test/save flow:
        // We generally rely on the form save for URL.

        // Save the options
        update_option('rankjet_platform_api_token', $token);
        update_option('rankjet_platform_website_id', $website_id);

        // Check connection status for immediate UI update
        $status = $this->check_connection_status();
        $is_connected = isset($status['connected']) && $status['connected'] === true;

        wp_send_json_success([
            'message' => 'Token saved successfully',
            'connected' => $is_connected
        ]);
    }

    /**
     * Public method to check connection status
     */
    public function is_connected() {
        $status = $this->check_connection_status();
        return isset($status['connected']) && $status['connected'] === true;
    }

    /**
     * Check connection status
     */
    private function check_connection_status() {
        // Reload the token from database
        $this->api_token = get_option('rankjet_platform_api_token', '');
        $this->website_id = get_option('rankjet_platform_website_id', '');
        
        if (empty($this->api_token)) {
            return ['connected' => false];
        }

        $response = $this->make_api_request('/plugin/configuration', 'GET');
        
        if (is_wp_error($response)) {
            return ['connected' => false, 'error' => $response->get_error_message()];
        }

        return [
            'connected' => true,
            'website_name' => $response['data']['website']['name'] ?? '',
            'website_id' => $response['data']['website']['id'] ?? ''
        ];
    }

    /**
     * Test connection AJAX handler
     */
    public function test_connection() {
        check_ajax_referer('rankjet_test_connection', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die();
        }

        $token = sanitize_text_field($_POST['token']);
        
        if (empty($token)) {
            wp_send_json_error(['message' => 'API token is required']);
        }

        // Save the token temporarily for the request
        $this->api_token = $token;

        // First, register the site
        $register_response = $this->make_api_request('/plugin/register', 'POST', [
            'domain' => get_site_url(),
            'site_name' => get_bloginfo('name'),
            'wordpress_version' => get_bloginfo('version'),
            'plugin_version' => RANKJET_AI_VERSION,
            'admin_email' => get_option('admin_email'),
        ]);

        if (is_wp_error($register_response)) {
            wp_send_json_error(['message' => $register_response->get_error_message()]);
        }

        wp_send_json_success([
            'message' => 'Connection successful',
            'website_name' => $register_response['data']['website_name'] ?? get_bloginfo('name'),
            'website_id' => $register_response['data']['website_id'] ?? '',
        ]);
    }

    /**
     * Sync data AJAX handler
     */
    public function sync_data() {
        check_ajax_referer('rankjet_sync_data', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die();
        }

        $result = $this->perform_sync();

        if ($result['success']) {
            wp_send_json_success(['message' => $result['message']]);
        } else {
            wp_send_json_error(['message' => $result['message']]);
        }
    }

    /**
     * Perform synchronization
     */
    private function perform_sync() {
        if (empty($this->api_token)) {
            return ['success' => false, 'message' => 'API token not configured'];
        }

        // Gather data to sync
        $sync_data = $this->gather_sync_data();

        // Send to platform
        $response = $this->make_api_request('/plugin/sync', 'POST', [
            'type' => 'all',
            'data' => $sync_data
        ]);

        if (is_wp_error($response)) {
            return ['success' => false, 'message' => $response->get_error_message()];
        }

        // Update last sync time
        update_option('rankjet_platform_last_sync', current_time('mysql'));

        // Get data from platform
        $this->sync_from_platform();

        return ['success' => true, 'message' => 'Synchronization completed successfully'];
    }

    /**
     * Gather data for synchronization
     */
    private function gather_sync_data() {
        $data = [
            'pages' => [],
            'keywords' => [],
            'content' => []
        ];

        // Get all published pages and posts
        $posts = get_posts([
            'post_type' => ['post', 'page'],
            'post_status' => 'publish',
            'numberposts' => -1
        ]);

        foreach ($posts as $post) {
            // Get meta data
            $meta_title = get_post_meta($post->ID, 'rankjet_meta_title', true);
            $meta_description = get_post_meta($post->ID, 'rankjet_meta_description', true);
            $focus_keyword = get_post_meta($post->ID, 'rankjet_focus_keyword', true);

            $data['pages'][] = [
                'url' => get_permalink($post),
                'title' => !empty($meta_title) ? $meta_title : $post->post_title,
                'meta_description' => $meta_description,
                'h1' => $post->post_title,
                'word_count' => str_word_count(strip_tags($post->post_content)),
                'word_count' => str_word_count(strip_tags($post->post_content)),
                'status_code' => 200,
                'post_status' => $post->post_status,
                'planner_item_id' => get_post_meta($post->ID, '_rankjet_planner_item_id', true),
            ];

            if (!empty($focus_keyword)) {
                $data['keywords'][] = [
                    'keyword' => $focus_keyword,
                    'target_url' => get_permalink($post),
                ];
            }
        }

        return $data;
    }

    /**
     * Sync data from platform
     */
    private function sync_from_platform() {
        // Get keywords data
        $keywords_response = $this->make_api_request('/plugin/data', 'GET', [
            'type' => 'keywords',
            'limit' => 100
        ]);

        if (!is_wp_error($keywords_response) && !empty($keywords_response['data'])) {
            // Store keywords data for use in the plugin
            update_option('rankjet_platform_keywords', $keywords_response['data']);
        }

        // Get rankings data
        $rankings_response = $this->make_api_request('/plugin/data', 'GET', [
            'type' => 'rankings',
            'limit' => 100
        ]);

        if (!is_wp_error($rankings_response) && !empty($rankings_response['data'])) {
            update_option('rankjet_platform_rankings', $rankings_response['data']);
        }
    }

    /**
     * Auto sync to platform (cron job)
     */
    public function auto_sync_to_platform() {
        if (get_option('rankjet_platform_auto_sync') !== 'yes') {
            return;
        }

        $this->perform_sync();
    }

    /**
     * Make API request to platform
     */
    public function make_api_request($endpoint, $method = 'GET', $data = []) {
        $url = $this->api_base_url . $endpoint;
        
        $args = [
            'method' => $method,
            'headers' => [
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'X-API-Token' => $this->api_token,
            ],
            'timeout' => 30,
        ];

        if ($method === 'POST' && !empty($data)) {
            $args['body'] = json_encode($data);
        } elseif ($method === 'GET' && !empty($data)) {
            $url = add_query_arg($data, $url);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $decoded = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_decode_error', 'Invalid JSON response');
        }

        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code >= 400) {
            $error_message = $decoded['message'] ?? 'API request failed';
            return new WP_Error('api_error', $error_message);
        }

        return $decoded;
    }
}

