<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Rankjet_Ai_Page_Data_Handler {

    public function __construct() {
        // Register REST API endpoints
        add_action('rest_api_init', [$this, 'register_rest_routes']);
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        register_rest_route('rankjet/v1', '/page-data', [
            'methods' => 'POST',
            'callback' => [$this, 'get_page_data'],
            'permission_callback' => [$this, 'check_api_permission'],
        ]);

        register_rest_route('rankjet/v1', '/update-seo', [
            'methods' => 'POST',
            'callback' => [$this, 'update_page_seo'],
            'permission_callback' => [$this, 'check_api_permission'],
        ]);
        
        // Add pages-list endpoint for Page Explorer
        register_rest_route('rankjet/v1', '/pages-list', [
            'methods' => 'GET',
            'callback' => [$this, 'get_pages_list'],
            'permission_callback' => [$this, 'check_api_permission'],
        ]);

        register_rest_route('rankjet/v1', '/create-post', [
            'methods' => 'POST',
            'callback' => [$this, 'create_post'],
            'permission_callback' => [$this, 'check_api_permission'],
        ]);

    }

    /**
     * Check API permission
     */
    public function check_api_permission($request) {
        $api_token = $request->get_header('X-API-Token');
        $saved_token = get_option('rankjet_platform_api_token', '');
        
        // Log authentication attempt with full details
        error_log('=== RankJet API Auth Check ===');
        error_log('Received Token: ' . $api_token);
        error_log('Saved Token in WP: ' . $saved_token);
        error_log('Tokens Match: ' . ($api_token === $saved_token ? 'YES' : 'NO'));
        
        // Also check if the token might be stored differently
        if ($api_token !== $saved_token) {
            // Check if one is hashed and the other isn't
            error_log('Token mismatch details:');
            error_log('- Received length: ' . strlen($api_token));
            error_log('- Saved length: ' . strlen($saved_token));
            error_log('- Received starts with: ' . substr($api_token, 0, 10));
            error_log('- Saved starts with: ' . substr($saved_token, 0, 10));
        }
        
        return !empty($api_token) && !empty($saved_token) && $api_token === $saved_token;
    }

    /**
     * Get pages list for Page Explorer
     */
    public function get_pages_list($request) {
        $limit = $request->get_param('limit') ? intval($request->get_param('limit')) : 50;
        
        // Support explicit offset, fall back to page-based calculation
        if ($request->get_param('offset') !== null) {
            $offset = intval($request->get_param('offset'));
            // approximate page for calculation
            $page = floor($offset / $limit) + 1; 
        } else {
            $page = $request->get_param('page') ? intval($request->get_param('page')) : 1;
            $offset = ($page - 1) * $limit;
        }

        $pages_data = [];
        
        // count total posts
        $total_posts_query = new WP_Query([
            'post_type' => ['post', 'page'],
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids',
        ]);
        $total_posts = $total_posts_query->found_posts;
        $total_pages = ceil($total_posts / $limit);

        // Get paginated posts
        $posts = get_posts([
            'post_type' => ['post', 'page'],
            'post_status' => 'publish',
            'posts_per_page' => $limit,
            'offset' => $offset,
            'orderby' => 'modified', // Changed from 'date' to 'modified' to catch updates
            'order' => 'DESC'
        ]);

        foreach ($posts as $post) {
            // Get all the SEO data for each page
            $page_url = get_permalink($post->ID);
            $title = $this->get_page_title($post->ID);
            $description = $this->get_meta_description($post->ID);
            $schema = $this->get_schema_markup($post->ID);
            $headings = $this->get_page_headings($post->post_content);
            
            $pages_data[] = [
                'url' => $page_url,
                'title' => $title,
                'meta_description' => $description,
                'description' => $description, // Alias for compatibility
                'h1' => !empty($headings['h1']) ? $headings['h1'][0] : $post->post_title,
                'headings' => $headings,
                'schema' => $schema,
                'word_count' => str_word_count(wp_strip_all_tags($post->post_content)),
                'status_code' => 200,
                'content' => wp_trim_words(wp_strip_all_tags($post->post_content), 50),
                'links' => $this->get_page_links($post->post_content),
                'publish_date' => $post->post_date, // Add publish date
                'modified_date' => $post->post_modified, // Add modified date as well
                'post_type' => $post->post_type, // Add post type for reference
                'outline' => $this->get_page_structure($post->post_content), // Add sequential outline
                'tags' => $this->get_post_terms($post->ID, 'post_tag'),
                'categories' => $this->get_post_terms($post->ID, 'category'),
            ];
        }

        return rest_ensure_response([
            'success' => true,
            'pages' => $pages_data,
            'total' => $total_posts,
            'total_pages' => $total_pages,
            'current_page' => $page,
            'limit' => $limit
        ]);
    }

    /**
     * Create new post from RankJet platform
     */
    public function create_post($request) {
        error_log('=== RankJet: create_post called ===');
        
        $title = $request->get_param('title');
        $content = $request->get_param('content');
        $status = $request->get_param('status') ?: 'draft';
        $excerpt = $request->get_param('excerpt');
        $meta_description = $request->get_param('meta_description');
        $focus_keyword = $request->get_param('focus_keyword');
        $tags = $request->get_param('tags');
        
        error_log('RankJet: Received parameters:');
        error_log('- Title: ' . $title);
        error_log('- Content length: ' . strlen($content));
        error_log('- Status: ' . $status);
        
        // Validate required fields
        if (empty($title)) {
            error_log('RankJet ERROR: Missing title');
            return new WP_Error('missing_title', 'Title is required', ['status' => 400]);
        }

        if (empty($content)) {
            error_log('RankJet ERROR: Missing content');
            return new WP_Error('missing_content', 'Content is required', ['status' => 400]);
        }

        // Prepare post data
        $post_data = [
            'post_title' => sanitize_text_field($title),
            'post_content' => wp_kses_post($content),
            'post_status' => sanitize_text_field($status),
            'post_type' => 'post',
        ];

        // Add excerpt if provided
        if (!empty($excerpt)) {
            $post_data['post_excerpt'] = sanitize_textarea_field($excerpt);
        }

        error_log('RankJet: Inserting post...');
        
        // Insert the post
        $post_id = wp_insert_post($post_data, true);

        if (is_wp_error($post_id)) {
            error_log('RankJet ERROR: Failed to create post - ' . $post_id->get_error_message());
            return new WP_Error(
                'post_creation_failed',
                $post_id->get_error_message(),
                ['status' => 500]
            );
        }

        error_log('RankJet: Post created successfully. ID: ' . $post_id);

        // Add meta description
        if (!empty($meta_description)) {
            error_log('RankJet: Adding meta description');
            $this->update_meta_description($post_id, $meta_description);
        }

        // Add focus keyword
        if (!empty($focus_keyword)) {
            error_log('RankJet: Adding focus keyword');
            update_post_meta($post_id, '_rankjet_focus_keyword', sanitize_text_field($focus_keyword));
            
            // Also update for SEO plugins
            if (class_exists('RankMath')) {
                update_post_meta($post_id, 'rank_math_focus_keyword', sanitize_text_field($focus_keyword));
            }
            if (function_exists('YoastSEO')) {
                update_post_meta($post_id, '_yoast_wpseo_focuskw', sanitize_text_field($focus_keyword));
            }
        }

        // Add tags if provided
        if (!empty($tags) && is_array($tags)) {
            error_log('RankJet: Adding ' . count($tags) . ' tags');
            wp_set_post_tags($post_id, $tags, false);
        }

        // Add planner item id if provided
        $planner_item_id = $request->get_param('planner_item_id');
        if (!empty($planner_item_id)) {
            error_log('RankJet: Adding planner_item_id: ' . $planner_item_id);
            update_post_meta($post_id, '_rankjet_planner_item_id', sanitize_text_field($planner_item_id));
        }

        // Get post URL
        $post_url = get_permalink($post_id);
        $edit_url = get_edit_post_link($post_id, 'raw');

        $response = [
            'success' => true,
            'message' => 'Post created successfully!',
            'post_id' => $post_id,
            'post_url' => $post_url,
            'edit_url' => $edit_url,
            'post_status' => get_post_status($post_id),
        ];
        
        error_log('RankJet: Response: ' . print_r($response, true));
        error_log('=== RankJet: create_post completed ===');

        return rest_ensure_response($response);
    }

    /**
     * Get page data
     */
    public function get_page_data($request) {
        $url = $request->get_param('url');
        
        if (empty($url)) {
            return new WP_Error('missing_url', 'URL parameter is required', ['status' => 400]);
        }

        // Get post ID from URL
        $post_id = url_to_postid($url);
        
        if (!$post_id) {
            return new WP_Error('post_not_found', 'Page not found', ['status' => 404]);
        }

        $post = get_post($post_id);
        
        if (!$post) {
            return new WP_Error('post_not_found', 'Page not found', ['status' => 404]);
        }

        // Get page data
        $page_data = [
            'title' => $this->get_page_title($post_id),
            'description' => $this->get_meta_description($post_id),
            'images' => $this->get_page_images($post_id),
            'schema' => $this->get_schema_markup($post_id),
            'headings' => $this->get_page_headings($post->post_content),
            'content' => wp_strip_all_tags($post->post_content),
            'links' => $this->get_page_links($post->post_content),
            'url' => get_permalink($post_id),
            'publish_date' => $post->post_date, // Add publish date
            'last_modified' => $post->post_modified,
            'post_type' => $post->post_type, // Add post type
            'post_status' => get_post_status($post_id),
            'planner_item_id' => get_post_meta($post_id, '_rankjet_planner_item_id', true),
            'outline' => $this->get_page_structure($post->post_content), // Add sequential outline
            'tags' => $this->get_post_terms($post_id, 'post_tag'),
            'categories' => $this->get_post_terms($post_id, 'category'),
        ];

        return rest_ensure_response($page_data);
    }

    /**
     * Update page SEO data
     */
    public function update_page_seo($request) {
        error_log('=== RankJet: update_page_seo called ===');
        error_log('Request Method: ' . $_SERVER['REQUEST_METHOD']);
        error_log('Request URI: ' . $_SERVER['REQUEST_URI']);
        
        $url = $request->get_param('url');
        $updates = $request->get_param('updates');
        
        error_log('RankJet: Received parameters:');
        error_log('- URL: ' . $url);
        error_log('- Updates: ' . print_r($updates, true));
        
        if (empty($url)) {
            error_log('RankJet ERROR: Missing URL parameter');
            return new WP_Error('missing_url', 'URL parameter is required', ['status' => 400]);
        }

        // Get post ID from URL
        error_log('RankJet: Looking up post ID for URL: ' . $url);
        $post_id = url_to_postid($url);
        
        error_log('RankJet: Post ID found: ' . ($post_id ? $post_id : 'NONE'));
        
        if (!$post_id) {
            // Try alternative lookup methods
            error_log('RankJet: Trying alternative URL lookup methods...');
            
            // Try to find by slug
            $path = parse_url($url, PHP_URL_PATH);
            error_log('RankJet: URL path: ' . $path);
            
            $slug = basename($path);
            error_log('RankJet: Slug: ' . $slug);
            
            $posts = get_posts([
                'name' => $slug,
                'post_type' => ['post', 'page'],
                'post_status' => 'publish',
                'numberposts' => 1
            ]);
            
            if (!empty($posts)) {
                $post_id = $posts[0]->ID;
                error_log('RankJet: Found post by slug. ID: ' . $post_id);
            } else {
                error_log('RankJet ERROR: Post not found for URL: ' . $url);
                return new WP_Error('post_not_found', 'Page not found for URL: ' . $url, ['status' => 404]);
            }
        }

        $updated = [];
        error_log('RankJet: Starting updates for post ID: ' . $post_id);

        // Update title
        if (isset($updates['title'])) {
            error_log('RankJet: Updating title to: ' . $updates['title']);
            $this->update_page_title($post_id, $updates['title']);
            $updated[] = 'title';
            error_log('RankJet: Title updated successfully');
        }

        // Update meta description
        if (isset($updates['description'])) {
            error_log('RankJet: Updating description to: ' . $updates['description']);
            $this->update_meta_description($post_id, $updates['description']);
            $updated[] = 'description';
            error_log('RankJet: Description updated successfully');
        }

        // Update image alt texts
        if (isset($updates['images']) && is_array($updates['images'])) {
            error_log('RankJet: Updating ' . count($updates['images']) . ' image(s)');
            $this->update_image_alts($post_id, $updates['images']);
            $updated[] = 'images';
            error_log('RankJet: Images updated successfully');
        }

        // Update schema markup
        if (isset($updates['schema'])) {
            error_log('RankJet: Updating schema markup');
            $this->update_schema_markup($post_id, $updates['schema']);
            $updated[] = 'schema';
            error_log('RankJet: Schema updated successfully');
        }

        $response = [
            'success' => true,
            'updated' => $updated,
            'post_id' => $post_id,
            'message' => 'SEO data updated successfully',
        ];
        
        error_log('RankJet: Update complete. Response: ' . print_r($response, true));
        error_log('=== RankJet: update_page_seo completed ===');

        return rest_ensure_response($response);
    }

    /**
     * Get page title
     */
    private function get_page_title($post_id) {
        // Check for Yoast SEO
        if (function_exists('YoastSEO')) {
            $title = get_post_meta($post_id, '_yoast_wpseo_title', true);
            if (!empty($title)) {
                return $title;
            }
        }

        // Check for RankMath
        if (class_exists('RankMath')) {
            $title = get_post_meta($post_id, 'rank_math_title', true);
            if (!empty($title)) {
                return $title;
            }
        }

        // Check for All in One SEO
        if (function_exists('aioseo')) {
            $title = get_post_meta($post_id, '_aioseo_title', true);
            if (!empty($title)) {
                return $title;
            }
        }

        // Default to post title
        return get_the_title($post_id);
    }

    /**
     * Get meta description
     */
    private function get_meta_description($post_id) {
        // Check for Yoast SEO
        if (function_exists('YoastSEO')) {
            $desc = get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
            if (!empty($desc)) {
                return $desc;
            }
        }

        // Check for RankMath
        if (class_exists('RankMath')) {
            $desc = get_post_meta($post_id, 'rank_math_description', true);
            if (!empty($desc)) {
                return $desc;
            }
        }

        // Check for All in One SEO
        if (function_exists('aioseo')) {
            $desc = get_post_meta($post_id, '_aioseo_description', true);
            if (!empty($desc)) {
                return $desc;
            }
        }

        // Generate from excerpt or content
        $post = get_post($post_id);
        if ($post->post_excerpt) {
            return wp_trim_words($post->post_excerpt, 30);
        }

        return wp_trim_words(wp_strip_all_tags($post->post_content), 30);
    }

    /**
     * Get page images
     */
    private function get_page_images($post_id) {
        $images = [];
        $post = get_post($post_id);
        
        // Get featured image
        if (has_post_thumbnail($post_id)) {
            $thumbnail_id = get_post_thumbnail_id($post_id);
            $images[] = [
                'id' => $thumbnail_id,
                'src' => wp_get_attachment_url($thumbnail_id),
                'alt' => get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true),
                'title' => get_the_title($thumbnail_id),
                'type' => 'featured',
            ];
        }

        // Get images from content
        if ($post && $post->post_content) {
            preg_match_all('/<img[^>]+>/i', $post->post_content, $img_tags);
            
            foreach ($img_tags[0] as $img_tag) {
                $image = [];
                
                // Extract src
                preg_match('/src="([^"]+)"/i', $img_tag, $src_match);
                if (isset($src_match[1])) {
                    $image['src'] = $src_match[1];
                }
                
                // Extract alt
                preg_match('/alt="([^"]*)"/i', $img_tag, $alt_match);
                $image['alt'] = isset($alt_match[1]) ? $alt_match[1] : '';
                
                // Extract title
                preg_match('/title="([^"]*)"/i', $img_tag, $title_match);
                $image['title'] = isset($title_match[1]) ? $title_match[1] : '';
                
                $image['type'] = 'content';
                
                if (!empty($image['src'])) {
                    $images[] = $image;
                }
            }
        }

        return $images;
    }

    /**
     * Get schema markup - SIMPLE VERSION
     * Just returns the most relevant schema type without fetching the whole page
     */
    private function get_schema_markup($post_id) {
        // First, try to get from RankMath meta (since that's what you're using)
        if (class_exists('RankMath')) {
            $rm_schema = get_post_meta($post_id, 'rank_math_rich_snippet', true);
            
            if (!empty($rm_schema)) {
                // Map RankMath types to Schema.org types
                $schema_map = [
                    'article' => 'Article',
                    'book' => 'Book',
                    'course' => 'Course',
                    'event' => 'Event',
                    'jobposting' => 'JobPosting',
                    'local' => 'LocalBusiness',
                    'music' => 'MusicRecording',
                    'person' => 'Person',
                    'product' => 'Product',
                    'recipe' => 'Recipe',
                    'restaurant' => 'Restaurant',
                    'service' => 'Service',
                    'software' => 'SoftwareApplication',
                    'video' => 'VideoObject',
                ];
                
                $schema_type = isset($schema_map[$rm_schema]) ? $schema_map[$rm_schema] : 'WebPage';
                
                // Check for BlogPosting subtype
                if ($rm_schema === 'article') {
                    $article_type = get_post_meta($post_id, 'rank_math_schema_article_type', true);
                    if ($article_type === 'BlogPosting') {
                        $schema_type = 'BlogPosting';
                    }
                }
                
                return [
                    '@context' => 'https://schema.org',
                    '@type' => $schema_type,
                ];
            }
        }
        
        // Fallback based on post type
        $post_type = get_post_type($post_id);
        $default_type = 'WebPage';
        
        if ($post_type === 'post') {
            $default_type = 'Article';
        } elseif ($post_type === 'product') {
            $default_type = 'Product';
        }
        
        return [
            '@context' => 'https://schema.org',
            '@type' => $default_type,
        ];
    }

    /**
     * Get page headings
     */
    private function get_page_headings($content) {
        $headings = [
            'h1' => [],
            'h2' => [],
            'h3' => [],
            'h4' => [],
            'h5' => [],
            'h6' => [],
        ];

        // Extract all headings
        for ($i = 1; $i <= 6; $i++) {
            preg_match_all('/<h' . $i . '[^>]*>(.*?)<\/h' . $i . '>/i', $content, $matches);
            if (!empty($matches[1])) {
                $headings['h' . $i] = array_map('wp_strip_all_tags', $matches[1]);
            }
        }

        return $headings;
    }

    /**
     * Get page structure (sequential outline)
     */
    private function get_page_structure($content) {
        // Extract H1-H6 in order
        preg_match_all('/<(h[1-6])[^>]*>(.*?)<\/\1>/i', $content, $matches, PREG_SET_ORDER);
        
        $outline = [];
        
        foreach ($matches as $match) {
            $tag = strtolower($match[1]); // e.g., 'h2'
            $text = wp_strip_all_tags($match[2]);
            
            if (empty($text)) continue;
            
            $outline[] = [
                'tag' => $tag,
                'text' => $text,
            ];
        }
        
        return $outline;
    }

    /**
     * Update page title
     */
    private function update_page_title($post_id, $title) {
        error_log('RankJet: update_page_title for post ' . $post_id . ' with title: ' . $title);
        
        // Update for Yoast SEO
        if (function_exists('YoastSEO')) {
            error_log('RankJet: Updating Yoast SEO title');
            update_post_meta($post_id, '_yoast_wpseo_title', $title);
        } else {
            error_log('RankJet: Yoast SEO not found');
        }

        // Update for RankMath
        if (class_exists('RankMath')) {
            error_log('RankJet: Updating RankMath title');
            update_post_meta($post_id, 'rank_math_title', $title);
        } else {
            error_log('RankJet: RankMath not found');
        }

        // Update for All in One SEO
        if (function_exists('aioseo')) {
            error_log('RankJet: Updating AIOSEO title');
            update_post_meta($post_id, '_aioseo_title', $title);
        } else {
            error_log('RankJet: AIOSEO not found');
        }

        // Store in RankJet meta
        error_log('RankJet: Storing title in RankJet meta');
        $result = update_post_meta($post_id, '_rankjet_seo_title', $title);
        error_log('RankJet: Title update result: ' . ($result ? 'SUCCESS' : 'FAILED'));
        
        // Verify the update
        $stored_title = get_post_meta($post_id, '_rankjet_seo_title', true);
        error_log('RankJet: Verified stored title: ' . $stored_title);
    }

    /**
     * Update meta description
     */
    private function update_meta_description($post_id, $description) {
        error_log('RankJet: update_meta_description for post ' . $post_id);
        error_log('RankJet: New description: ' . $description);
        
        // Update for Yoast SEO
        if (function_exists('YoastSEO')) {
            error_log('RankJet: Updating Yoast SEO description');
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $description);
        } else {
            error_log('RankJet: Yoast SEO not found');
        }

        // Update for RankMath
        if (class_exists('RankMath')) {
            error_log('RankJet: Updating RankMath description');
            update_post_meta($post_id, 'rank_math_description', $description);
        } else {
            error_log('RankJet: RankMath not found');
        }

        // Update for All in One SEO
        if (function_exists('aioseo')) {
            error_log('RankJet: Updating AIOSEO description');
            update_post_meta($post_id, '_aioseo_description', $description);
        } else {
            error_log('RankJet: AIOSEO not found');
        }

        // Store in RankJet meta
        error_log('RankJet: Storing description in RankJet meta');
        $result = update_post_meta($post_id, '_rankjet_seo_description', $description);
        error_log('RankJet: Description update result: ' . ($result ? 'SUCCESS' : 'FAILED'));
        
        // Verify the update
        $stored_desc = get_post_meta($post_id, '_rankjet_seo_description', true);
        error_log('RankJet: Verified stored description: ' . substr($stored_desc, 0, 50) . '...');
    }

    /**
     * Update image alt texts
     */
    private function update_image_alts($post_id, $images) {
        foreach ($images as $image) {
            if (isset($image['id']) && isset($image['alt'])) {
                update_post_meta($image['id'], '_wp_attachment_image_alt', $image['alt']);
            }
        }
    }

    /**
     * Update schema markup
     */
    private function update_schema_markup($post_id, $schema) {
        // Store in RankJet meta
        update_post_meta($post_id, '_rankjet_schema_markup', json_encode($schema));
        
        // Update for RankMath if available
        if (class_exists('RankMath')) {
            update_post_meta($post_id, 'rank_math_schema', $schema);
        }
        }

    /**
     * Get page links
     */
    private function get_page_links($content) {
        $links = [];
        
        if (empty($content)) {
            return $links;
        }

        // Use DOMDocument to parse HTML
        $dom = new DOMDocument();
        
        // Suppress warnings for malformed HTML
        libxml_use_internal_errors(true);
        
        // Load HTML (utf-8 hack)
        $dom->loadHTML('<?xml encoding="utf-8" ?>' . $content, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        
        libxml_clear_errors();
        
        $anchors = $dom->getElementsByTagName('a');
        $home_url = home_url();
        $parsed_home = parse_url($home_url);
        $home_host = isset($parsed_home['host']) ? str_replace('www.', '', $parsed_home['host']) : '';

        foreach ($anchors as $anchor) {
            $href = $anchor->getAttribute('href');
            
            // Skip empty or special links
            if (empty($href) || strpos($href, '#') === 0 || strpos($href, 'mailto:') === 0 || strpos($href, 'tel:') === 0) {
                continue;
            }

            $is_internal = false;
            
            // Check if internal
            if (strpos($href, '/') === 0) {
                $is_internal = true; // Relative URL
                // Convert to absolute for consistency if needed, but keeping as is for now
                // $href = rtrim($home_url, '/') . $href;
            } else {
                $parsed_url = parse_url($href);
                if (isset($parsed_url['host'])) {
                    $host = str_replace('www.', '', $parsed_url['host']);
                    if ($host === $home_host) {
                        $is_internal = true;
                    }
                }
            }

            // Get anchor text
            $text = trim($anchor->textContent);
            
            // Limit text length
            if (strlen($text) > 255) {
                $text = substr($text, 0, 252) . '...';
            }

            $links[] = [
                'url' => $href,
                'anchor_text' => $text,
                'is_internal' => $is_internal
            ];
        }
        
        return $links;
    }


    /**
     * Get post terms (tags/categories)
     */
    private function get_post_terms($post_id, $taxonomy) {
        $terms = get_the_terms($post_id, $taxonomy);
        
        if (is_wp_error($terms) || empty($terms)) {
            return [];
        }
        
        return array_map(function($term) {
            return [
                'id' => $term->term_id,
                'name' => $term->name,
                'slug' => $term->slug
            ];
        }, $terms);
    }
}


// Initialize the handler
new Rankjet_Ai_Page_Data_Handler();
