<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Rankjet_Ai_Frontend_Schema {

    private $variables;

    public function __construct($variables_instance) {
        $this->variables = $variables_instance;
    }

    /**
     * Output Schema Logic (JSON-LD)
     */
    public function render() {
        // Check if meta tags display is enabled
        $display_meta = get_option('rankjet_display_meta_tags', 'yes');
        if ($display_meta !== 'yes') {
            return;
        }

        $post_id = get_the_ID();
        
        // Retrieval
        $schema_json = get_post_meta($post_id, '_rankjet_seo_schema', true);

        // Output Schema if exists
        if (!empty($schema_json) && strpos($schema_json, '{') !== false) {
             $schema_data = json_decode($schema_json, true);
             if (json_last_error() === JSON_ERROR_NONE) {
                 $processed_schema = $this->process_schema_recursively($schema_data, $post_id);
                 echo '<script type="application/ld+json" class="rank-jet-schema">' . json_encode($processed_schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . "</script>\n";
             }
        } else {
            // Default Schema Logic for ALL page types
            $default_type = 'Article'; // Fallback
            
            if (is_singular()) {
                $default_type = get_option('rankjet_post_default_schema_type', 'Article');
                if (is_page()) {
                    $default_type = get_option('rankjet_page_default_schema_type', 'Article');
                }
            } elseif (is_category() || is_tag() || is_tax()) {
                $default_type = 'CollectionPage';
            } elseif (is_author()) {
                $default_type = 'ProfilePage';
            } elseif (is_home() || is_front_page()) {
                $default_type = 'WebSite'; 
            }
            
            if ($default_type && $default_type !== 'none') {
                $default_schema_data = $this->generate_default_schema($default_type, $post_id);
                if ($default_schema_data) {
                    $processed_schema = $this->process_schema_recursively($default_schema_data, $post_id);
                    echo '<script type="application/ld+json" class="rank-jet-schema">' . json_encode($processed_schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . "</script>\n";
                }
            }
        }
        
        echo "<!-- /RankJet AI WordPress SEO -->\n";
    }

    /**
     * Generate Default Schema Data
     */
    private function generate_default_schema($type, $post_id) {
        $data = [
            '@context' => 'https://schema.org',
            '@type' => $type
        ];

        // Handle Product Types
        if ($type === 'Product') {
            $data['name'] = '%seo_title%';
            $data['description'] = '%seo_description%';
            $data['sku'] = ''; 
            $data['mpn'] = '';
            
            if (has_post_thumbnail($post_id)) {
                $data['image'] = '%post_thumbnail%';
            }

            $data['brand'] = [
                '@type' => 'Brand',
                'name' => ''
            ];

            $data['offers'] = [
                '@type' => 'Offer',
                'price' => '0.00',
                'priceCurrency' => 'USD',
                'priceValidUntil' => '%date(Y-12-31)%',
                'availability' => 'https://schema.org/InStock',
                'url' => '%url%'
            ];

            $data['aggregateRating'] = [
                '@type' => 'AggregateRating',
                'ratingValue' => '4.5',
                'reviewCount' => '10'
            ];
            
            // Structure Links
            $data['isPartOf'] = [
                '@type' => 'WebSite',
                '@id' => '%home_url%/#website',
                'url' => '%home_url%',
                'name' => '%sitename%'
            ];

            $data['mainEntityOfPage'] = [
               '@type' => 'WebPage',
               '@id' => '%url%/#webpage',
               'url' => '%url%'
            ];

            return $data;
        }

        // Handle Article Types
        if (in_array($type, ['Article', 'BlogPosting', 'NewsArticle'])) {
             // If type is Article but user selected specific sub-type in UI for defaults
             if ($type === 'Article') {
                 $subtype = get_option('rankjet_post_default_article_type', 'BlogPosting');
                 
                 // If page, use page default subtype
                 if (is_page($post_id)) {
                     $subtype = get_option('rankjet_page_default_article_type', 'BlogPosting');
                 }
                 
                 $data['@type'] = $subtype;
             }
             
             $data['headline'] = '%seo_title%';
             $data['description'] = '%seo_description%';
             $data['datePublished'] = '%date(c)%';
             $data['dateModified'] = '%modified(c)%';
             
             $data['author'] = [
                 '@type' => 'Person',
                 'name' => '%username%',
                 'url' => '%author_url%'
             ];

             $data['publisher'] = [
                '@type' => 'Organization',
                'name' => '%sitename%',
                'url' => '%home_url%',
                'logo' => [
                    '@type' => 'ImageObject',
                    'url' => '%site_logo%' 
                ]
             ];

             $data['isPartOf'] = [
                 '@type' => 'WebSite',
                 '@id' => '%home_url%/#website',
                 'url' => '%home_url%',
                 'name' => '%sitename%'
             ];

             $data['mainEntityOfPage'] = [
                '@type' => 'WebPage',
                '@id' => '%url%/#webpage',
                'url' => '%url%'
             ]; 

             if (has_post_thumbnail($post_id)) {
                 $data['image'] = [
                     '@type' => 'ImageObject',
                     'url' => '%post_thumbnail%'
                 ];
             }
             return $data;
        }

        // Basic Fallback for other types
        $data['name'] = '%seo_title%';
        $data['description'] = '%seo_description%';
        $data['url'] = '%url%';

        return $data;
    }

    /**
     * Recursively replace variables in Schema Data
     */
    private function process_schema_recursively($data, $post_id) {
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = $this->process_schema_recursively($value, $post_id);
            }
            return $data;
        } elseif (is_string($data)) {
            return $this->variables->replace_variables($data, $post_id);
        }
        return $data;
    }
}
