<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Rankjet_Ai_Frontend_Meta {

    private $variables;

    public function __construct($variables_instance) {
        $this->variables = $variables_instance;
    }

    /**
     * Override Document Title
     */
    public function seo_title($title) {
        if (is_singular()) {
            $post_id = get_the_ID();
            
            // Check for custom SEO title first
            $custom_title = get_post_meta($post_id, '_rankjet_seo_title', true);
            if (!empty($custom_title)) {
                return $custom_title;
            }
            
            // Use title template from settings
            $template = get_option('rankjet_post_title_template', '%title% %sep% %sitename%');
            
            // If page, use page template
            if (is_page()) {
                $template = get_option('rankjet_page_title_template', '%title% %sep% %sitename%');
            }

            $separator = get_option('rankjet_title_separator', '-');
            
            $post = get_post($post_id);
            if ($post) {
                $replacements = [
                    '%title%' => $post->post_title,
                    '%sep%' => ' ' . $separator . ' ',
                    '%sitename%' => get_bloginfo('name'),
                ];
                return str_replace(array_keys($replacements), array_values($replacements), $template);
            }

        } elseif (is_category()) {
            $template = get_option('rankjet_category_title_template', '%term_title% Archives %sep% %sitename%');
            $separator = get_option('rankjet_title_separator', '-');
            $term = get_queried_object();
            
            if ($term) {
                $replacements = [
                    '%term_title%' => $term->name,
                    '%sep%' => ' ' . $separator . ' ',
                    '%sitename%' => get_bloginfo('name'),
                ];
                return str_replace(array_keys($replacements), array_values($replacements), $template);
            }
        } elseif (is_tag()) {
            $template = get_option('rankjet_tag_title_template', '%term_title% Archives %sep% %sitename%');
            $separator = get_option('rankjet_title_separator', '-');
            $term = get_queried_object();
            
            if ($term) {
                $replacements = [
                    '%term_title%' => $term->name,
                    '%sep%' => ' ' . $separator . ' ',
                    '%sitename%' => get_bloginfo('name'),
                ];
                return str_replace(array_keys($replacements), array_values($replacements), $template);
            }
        }
        
        return $title; // Return default if nothing applies
    }

    /**
     * Output SEO Meta Tags (Non-Schema)
     */
    public function render() {
        // Check if meta tags display is enabled
        $display_meta = get_option('rankjet_display_meta_tags', 'yes');
        if ($display_meta !== 'yes') {
            // Still need to output title tag for the page to work
            echo '<title>' . esc_html(wp_get_document_title()) . '</title>' . "\n";
            return;
        }

        global $post;
        $post_id = get_the_ID();

        // Title Calculation
        // This will trigger our 'seo_title' filter if singular
        $final_title = wp_get_document_title();

        // 1. Data Retrieval (Singular only usually)
        $desc = '';
        
        if (is_singular()) {
            // Check for custom SEO description first
            $seo_desc = get_post_meta($post_id, '_rankjet_seo_description', true);
            
            if (!empty($seo_desc)) {
                $desc = $seo_desc;
            } else {
                // Use description template from settings
                $desc_template = get_option('rankjet_post_description_template', '%excerpt%');
                
                // If page, use page template
                if (is_page()) {
                    $desc_template = get_option('rankjet_page_description_template', '%excerpt%');
                }
                
                $excerpt = get_the_excerpt();
                
                if (empty($excerpt)) {
                    $content = get_post_field('post_content', $post_id);
                    $excerpt = wp_trim_words(strip_shortcodes(wp_strip_all_tags($content)), 25, '...');
                }
                
                $replacements = [
                    '%excerpt%' => $excerpt,
                    '%title%' => get_the_title($post_id),
                ];
                $desc = str_replace(array_keys($replacements), array_values($replacements), $desc_template);
            }
            
            // Clean up description
            $desc = strip_shortcodes(wp_strip_all_tags($desc));

        } elseif (is_category()) {
            $desc_template = get_option('rankjet_category_description_template', '%term_description%');
            $term = get_queried_object();
            
            if ($term) {
                $replacements = [
                    '%term_description%' => strip_tags(category_description() ?: ''),
                    '%term_title%' => $term->name,
                ];
                $desc = str_replace(array_keys($replacements), array_values($replacements), $desc_template);
            }
        } elseif (is_tag()) {
            $desc_template = get_option('rankjet_tag_description_template', '%term_description%');
            $term = get_queried_object();
            
            if ($term) {
                $replacements = [
                    '%term_description%' => strip_tags(term_description() ?: ''),
                    '%term_title%' => $term->name,
                ];
                $desc = str_replace(array_keys($replacements), array_values($replacements), $desc_template);
            }
        }

        $url = get_permalink();
        $site_name = get_bloginfo('name');
        $img_url = (is_singular() ? get_the_post_thumbnail_url($post_id, 'full') : '');
        $updated_time = get_the_modified_date('c');
        $published_time = get_the_date('c');
        $locale = get_locale();

        // Robots Logic
        // Check if site is public
        $public = get_option('blog_public');
        if ( '0' === $public ) {
            $robots = 'noindex, nofollow';
        } else {
            // Use default from settings
            $robots_default = get_option('rankjet_robots_meta_default', 'index, follow');
            
            // Strict check for index/follow directives
            $directives = array_map('trim', explode(',', $robots_default));
            
            // Add advanced directives for better SEO only if index and follow are explicitly present
            // and noindex is NOT present (to be safe)
            if (in_array('index', $directives) && in_array('follow', $directives) && !in_array('noindex', $directives)) {
                $robots = $robots_default . ', max-snippet:-1, max-video-preview:-1, max-image-preview:large';
            } else {
                $robots = $robots_default;
            }
        }

        // Start Output
        echo "\n<!-- Search Engine Optimization by RankJet AI - https://rankjet.ai/ -->\n";
        
        // Title Tag (Moved inside)
        echo '<title>' . esc_html($final_title) . '</title>' . "\n";

        // Robots Tag (Moved inside)
        echo '<meta name="robots" content="' . esc_attr($robots) . '" />' . "\n";

        // Description
        if (!empty($desc)) {
            echo '<meta name="description" content="' . esc_attr($desc) . '" />' . "\n";
        }
        
        // Canonical
        echo '<link rel="canonical" href="' . esc_url($url) . '" />' . "\n";

        // Open Graph (OG)
        echo '<meta property="og:locale" content="' . esc_attr($locale) . '" />' . "\n";
        echo '<meta property="og:type" content="article" />' . "\n";
        echo '<meta property="og:title" content="' . esc_attr($final_title) . '" />' . "\n";
        if (!empty($desc)) echo '<meta property="og:description" content="' . esc_attr($desc) . '" />' . "\n";
        echo '<meta property="og:url" content="' . esc_url($url) . '" />' . "\n";
        echo '<meta property="og:site_name" content="' . esc_attr($site_name) . '" />' . "\n";
        echo '<meta property="og:updated_time" content="' . esc_attr($updated_time) . '" />' . "\n";
        
        if ($img_url) {
            echo '<meta property="og:image" content="' . esc_url($img_url) . '" />' . "\n";
            echo '<meta property="og:image:secure_url" content="' . esc_url($img_url) . '" />' . "\n";
        }

        // Article Meta
        echo '<meta property="article:published_time" content="' . esc_attr($published_time) . '" />' . "\n";
        echo '<meta property="article:modified_time" content="' . esc_attr($updated_time) . '" />' . "\n";

        // Twitter Cards
        echo '<meta name="twitter:card" content="summary_large_image" />' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr($final_title) . '" />' . "\n";
        if (!empty($desc)) echo '<meta name="twitter:description" content="' . esc_attr($desc) . '" />' . "\n";
        
        if ($img_url) {
            echo '<meta name="twitter:image" content="' . esc_url($img_url) . '" />' . "\n";
        }
        
        // Note: Schema output removed from here
    }
}
