<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Rankjet_Ai_Editor {

    private $integration;
    private $content_service;

    public function __construct($integration) {
        $this->integration = $integration;
        $this->content_service = new Rankjet_Ai_Content_Service($this->integration);

        add_action('init', [$this, 'register_meta_fields']); // Register Meta
        add_action('add_meta_boxes', [$this, 'add_meta_boxes']);
        // add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']); // Removed as unused
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_block_assets']);
        
        // AJAX handlers
        add_action('wp_ajax_rankjet_editor_action', [$this, 'handle_ajax_action']);
        add_action('wp_ajax_rankjet_ai_get_suggestions', [$this, 'handle_get_suggestions']);
    }

    /**
     * Register Meta Fields for REST API
     */
    public function register_meta_fields() {
        register_meta('post', '_rankjet_focus_keyword', [
            'show_in_rest' => true,
            'single' => true,
            'type' => 'string',
            'auth_callback' => function() { return current_user_can('edit_posts'); }
        ]);

        register_meta('post', '_rankjet_seo_description', [
            'show_in_rest' => true,
            'single' => true,
            'type' => 'string',
            'auth_callback' => function() { return current_user_can('edit_posts'); }
        ]);

        // Originality check data
        register_meta('post', '_rankjet_originality_score', [
            'show_in_rest' => true,
            'single' => true,
            'type' => 'integer',
            'auth_callback' => function() { return current_user_can('edit_posts'); }
        ]);

        register_meta('post', '_rankjet_originality_data', [
            'show_in_rest' => true,
            'single' => true,
            'type' => 'string',
            'auth_callback' => function() { return current_user_can('edit_posts'); }
        ]);

        register_meta('post', '_rankjet_originality_checked_at', [
            'show_in_rest' => true,
            'single' => true,
            'type' => 'string',
            'auth_callback' => function() { return current_user_can('edit_posts'); }
        ]);

        // SEO Schema (JSON-LD)
        register_meta('post', '_rankjet_seo_schema', [
            'show_in_rest' => true,
            'single' => true,
            'type' => 'string',
            'auth_callback' => function() { return current_user_can('edit_posts'); }
        ]);
    }

    /**
     * Add Meta Boxes
     */
    public function add_meta_boxes() {
        // Meta box removed as it is replaced by the custom Gutenberg sidebar.
        // This prevents duplication in the editor UI.
    }

    /**
     * Enqueue Block Assets (Gutenberg)
     */
    public function enqueue_block_assets() {
        // Enqueue Components
        // Enqueue Components
        // Migrated to src/components: seo-checklist, readability-score, tone-of-voice, originality-check, content-score

        // Enqueue Block Assets (Gutenberg) - Migrated Components Removed

        // Enqueue Score Sidebar - MOVED TO BUILD/INDEX.JS
        // Legacy rankjet-score-sidebar-js removed.

        // Enqueue Toolbar - MOVED TO BUILD/INDEX.JS
        // Legacy rankjet-toolbar-js removed.

        // Enqueue Build (Modern Build System)
        $asset_file = include( plugin_dir_path( __FILE__ ) . '../../build/index.asset.php');
        wp_enqueue_script(
            'rankjet-ai-build',
            RANKJET_AI_PLUGIN_URL . 'build/index.js',
            array_merge($asset_file['dependencies'], ['jquery']),
            $asset_file['version'],
            true
        );

        // Use new CSS
        wp_enqueue_style('rankjet-ai-style', RANKJET_AI_PLUGIN_URL . 'build/index.css', [], $asset_file['version']);
        
        // Legacy Enqueues Removed:
        // wp_enqueue_style('rankjet-editor-css', RANKJET_AI_URL . 'assets/css/rankjet-editor.css', [], RANKJET_AI_VERSION);
        
        global $post;
        $post_id = $post ? $post->ID : 0;
        
        // Load stored originality data
        $stored_originality = null;
        if ($post_id) {
            $score = get_post_meta($post_id, '_rankjet_originality_score', true);
            $data_json = get_post_meta($post_id, '_rankjet_originality_data', true);
            $checked_at = get_post_meta($post_id, '_rankjet_originality_checked_at', true);
            if ($score !== '' && $score !== false) {
                $stored_originality = [
                    'score' => (int) $score,
                    'issues' => $data_json ? json_decode($data_json, true) : [],
                    'checked_at' => $checked_at
                ];
            }
        }

        // Load stored semantic analysis data
        $stored_semantic = null;
        if ($post_id) {
            $semantic_score = get_post_meta($post_id, '_rankjet_semantic_score', true);
            $semantic_data_json = get_post_meta($post_id, '_rankjet_semantic_data', true);
            $semantic_checked_at = get_post_meta($post_id, '_rankjet_semantic_checked_at', true);
            if ($semantic_score !== '' && $semantic_score !== false) {
                $stored_semantic = [
                    'score' => (int) $semantic_score,
                    'data' => $semantic_data_json ? json_decode($semantic_data_json, true) : [],
                    'checked_at' => $semantic_checked_at
                ];
            }
        }

        // Check if platform is integrated (API token exists)
        $api_token = get_option('rankjet_platform_api_token', '');
        $is_integrated = !empty($api_token);

        $data = [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('rankjet_editor_nonce'),
            'post_id' => $post_id,
            'icon_url' => RANKJET_AI_PLUGIN_URL . 'assets/images/rankjet-icon.png',
            'stored_originality' => $stored_originality,
            'stored_semantic' => $stored_semantic,
            'is_integrated' => $is_integrated,
            'settings_url' => admin_url('admin.php?page=rankjet-ai-integration'),
            'default_schema_type' => get_option('rankjet_post_default_schema_type', 'Article'),
            'default_article_type' => get_option('rankjet_post_default_article_type', 'BlogPosting')
        ];

        // If page, override defaults
        if ($post_id && get_post_type($post_id) === 'page') {
            $data['default_schema_type'] = get_option('rankjet_page_default_schema_type', 'Article');
            $data['default_article_type'] = get_option('rankjet_page_default_article_type', 'BlogPosting');
        }

        // Localize to content-score.js FIRST (it loads as a dependency, needs data early)
        wp_localize_script('rankjet-content-score', 'rankjetData', $data);
        
        // Also localize to sidebar(now build) and toolbar
        // wp_localize_script('rankjet-sidebar-js', ...); // Removed, handled by rankjet-ai-build
        wp_localize_script('rankjet-toolbar-js', 'rankjetData', $data);
        wp_localize_script('rankjet-ai-build', 'rankjetData', $data);
    }

    // enqueue_assets removed as unused

    // render_meta_box removed as unused

    /**
     * Handle AJAX Action
     */
    public function handle_ajax_action() {
        check_ajax_referer('rankjet_editor_nonce', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Permission denied');
        }

        $action_type = isset($_POST['action_type']) ? sanitize_text_field($_POST['action_type']) : '';
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if ($action_type === 'get_connections') {
            if (!$post_id) {
                wp_send_json_error('Invalid Post ID');
            }
            $url = get_permalink($post_id);
            // Handle drafts/previews if URL is not final?
            // Since plugin relies on mapped URL, we might need mapped_page_url from meta
            
            $response = $this->content_service->get_connections($url);
            
            if (is_wp_error($response)) {
                wp_send_json_error($response->get_error_message());
            }
            
            wp_send_json_success($response['data']);
        
        } elseif ($action_type === 'generate') {
            $type = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : ''; // title, meta, extend
            $prompt = isset($_POST['prompt']) ? sanitize_textarea_field($_POST['prompt']) : '';
            $context = isset($_POST['context']) && is_array($_POST['context']) ? $_POST['context'] : [];
            
            // Add server-side context
            $post = get_post($post_id);
            if (!$post) {
                 // Even if post not found, we might proceed with just provided context, or error out.
                 // For 'extend', we need content.
                 if ($type === 'extend') {
                     wp_send_json_error('Post not found');
                 }
            } else {
                if ($type === 'extend') {
                   $context['content'] = $post->post_content;
                }
                $context['article_title'] = get_the_title($post->ID); // cleaner than wrapping post_id again
            }
            
            $response = $this->content_service->generate_content($type, $context, $prompt);
            
            if (is_wp_error($response)) {
                wp_send_json_error($response->get_error_message());
            }
            
            wp_send_json_success($response['data']);
        
        } elseif ($action_type === 'check_originality') {
            $content = isset($_POST['content']) ? wp_kses_post($_POST['content']) : '';
            
            if (empty($content)) {
                wp_send_json_error('No content provided');
            }
            
            // Strip HTML for originality check
            $clean_content = wp_strip_all_tags($content);
            
            $response = $this->content_service->check_originality($clean_content);
            
            if (is_wp_error($response)) {
                wp_send_json_error($response->get_error_message());
            }
            
            // Save originality result to post meta
            if ($post_id && isset($response['data']['score'])) {
                update_post_meta($post_id, '_rankjet_originality_score', intval($response['data']['score']));
                update_post_meta($post_id, '_rankjet_originality_data', wp_json_encode($response['data']['issues'] ?? []));
                update_post_meta($post_id, '_rankjet_originality_checked_at', current_time('mysql'));
            }
            
            wp_send_json_success($response['data']);
        
        } elseif ($action_type === 'semantic_analysis') {
            $content = isset($_POST['content']) ? wp_kses_post($_POST['content']) : '';
            
            if (empty($content)) {
                wp_send_json_error('No content provided');
            }
            
            // Strip HTML for semantic analysis
            $clean_content = wp_strip_all_tags($content);
            
            // Check minimum word count
            $word_count = str_word_count($clean_content);
            if ($word_count < 50) {
                wp_send_json_error('Please add at least 50 words to run semantic analysis.');
            }
            
            $response = $this->content_service->analyze_semantics($clean_content);
            
            if (is_wp_error($response)) {
                wp_send_json_error($response->get_error_message());
            }
            
            // Save semantic analysis result to post meta (like originality)
            if ($post_id && isset($response['data']['semantic_score'])) {
                update_post_meta($post_id, '_rankjet_semantic_score', intval($response['data']['semantic_score']));
                update_post_meta($post_id, '_rankjet_semantic_data', wp_json_encode($response['data']));
                update_post_meta($post_id, '_rankjet_semantic_checked_at', current_time('mysql'));
            }
            
            wp_send_json_success($response['data']);
        }

        wp_die();
    }

    /**
     * Handle Get Suggestions AJAX
     */
    public function handle_get_suggestions() {
        check_ajax_referer('rankjet_editor_nonce', 'security');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $keyword = isset($_POST['keyword']) ? sanitize_text_field($_POST['keyword']) : '';
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;

        if (empty($keyword)) {
            wp_send_json_error(['message' => 'Please provide a keyword to get suggestions.']);
        }

        // Get post title for context
        $post_title = '';
        if ($post_id) {
            $post_title = get_the_title($post_id);
        }

        // Call API via Content Service (uses dedicated /plugin/keyword-ideas endpoint)
        $response = $this->content_service->get_keyword_ideas($keyword, $post_title);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => $response->get_error_message()]);
        }
        // If API returns valid data, use it
        // SaaS returns: { success: true, data: ["keyword1", "keyword2", ...] }
        if (!is_wp_error($response) && isset($response['data']) && is_array($response['data'])) {
             wp_send_json_success(['suggestions' => $response['data']]);
        }

        // Fallback to mock suggestions (when API unavailable or returns invalid data)
        $mock_suggestions = [
            $keyword . ' best practices',
            $keyword . ' guide',
            'How to ' . $keyword,
            $keyword . ' for beginners',
            $keyword . ' tips ' . date('Y')
        ];
        
        wp_send_json_success(['suggestions' => $mock_suggestions]);
    }
}
