<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * RankJet AI Titles & Meta Settings
 * 
 * Handles the Titles & Meta submenu page with settings for:
 * - General: Robots meta defaults, meta display toggle
 * - Posts: Title and description templates
 */
class Rankjet_Ai_Titles_Meta {

    public function __construct() {
        add_action('admin_menu', [$this, 'add_titles_meta_menu'], 20);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('wp_ajax_rankjet_save_titles_meta', [$this, 'save_settings_ajax']);
    }

    /**
     * Add Titles & Meta submenu page
     */
    public function add_titles_meta_menu() {
        add_submenu_page(
            'rankjet-ai',
            __('Titles & Meta', 'rankjet-ai'),
            __('Titles & Meta', 'rankjet-ai'),
            'manage_options',
            'rankjet-ai-titles-meta',
            [$this, 'render_settings_page']
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        // General Settings
        register_setting('rankjet_titles_meta_settings', 'rankjet_robots_meta_default', [
            'type' => 'string',
            'default' => 'index, follow',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        
        register_setting('rankjet_titles_meta_settings', 'rankjet_display_meta_tags', [
            'type' => 'string',
            'default' => 'yes',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        // Posts Settings
        register_setting('rankjet_titles_meta_settings', 'rankjet_post_title_template', [
            'type' => 'string',
            'default' => '%title% %sep% %sitename%',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        
        register_setting('rankjet_titles_meta_settings', 'rankjet_post_description_template', [
            'type' => 'string',
            'default' => '%excerpt%',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        // Separator setting
        register_setting('rankjet_titles_meta_settings', 'rankjet_title_separator', [
            'type' => 'string',
            'default' => '-',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        // Default Schema Settings
        register_setting('rankjet_titles_meta_settings', 'rankjet_post_default_schema_type', [
            'type' => 'string',
            'default' => 'Article',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        
        register_setting('rankjet_titles_meta_settings', 'rankjet_post_default_article_type', [
            'type' => 'string',
            'default' => 'BlogPosting',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        // Pages Settings
        register_setting('rankjet_titles_meta_settings', 'rankjet_page_title_template', [
            'type' => 'string',
            'default' => '%title% %sep% %sitename%',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        
        register_setting('rankjet_titles_meta_settings', 'rankjet_page_description_template', [
            'type' => 'string',
            'default' => '%excerpt%',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        register_setting('rankjet_titles_meta_settings', 'rankjet_page_default_schema_type', [
            'type' => 'string',
            'default' => 'Article',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        
        register_setting('rankjet_titles_meta_settings', 'rankjet_page_default_article_type', [
            'type' => 'string',
            'default' => 'BlogPosting',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        // Categories
        register_setting('rankjet_titles_meta_settings', 'rankjet_category_title_template', [
            'type' => 'string',
            'default' => '%term_title% Archives %sep% %sitename%',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        
        register_setting('rankjet_titles_meta_settings', 'rankjet_category_description_template', [
            'type' => 'string',
            'default' => '%term_description%',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        // Tags
        register_setting('rankjet_titles_meta_settings', 'rankjet_tag_title_template', [
            'type' => 'string',
            'default' => '%term_title% Archives %sep% %sitename%',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        
        register_setting('rankjet_titles_meta_settings', 'rankjet_tag_description_template', [
            'type' => 'string',
            'default' => '%term_description%',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
    }

    /**
     * AJAX handler for saving settings
     */
    public function save_settings_ajax() {
        check_ajax_referer('rankjet_titles_meta_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
            return;
        }

        // Sanitize and save settings
        $settings = [
            'rankjet_robots_meta_default' => sanitize_text_field($_POST['robots_meta_default'] ?? 'index, follow'),
            'rankjet_display_meta_tags' => sanitize_text_field($_POST['display_meta_tags'] ?? 'yes'),
            'rankjet_post_title_template' => sanitize_text_field($_POST['post_title_template'] ?? '%title% %sep% %sitename%'),
            'rankjet_post_description_template' => sanitize_text_field($_POST['post_description_template'] ?? '%excerpt%'),
            'rankjet_title_separator' => sanitize_text_field($_POST['title_separator'] ?? '-'),
            'rankjet_post_default_schema_type' => sanitize_text_field($_POST['post_default_schema_type'] ?? 'Article'),
            'rankjet_post_default_article_type' => sanitize_text_field($_POST['post_default_article_type'] ?? 'BlogPosting'),
            
            // Pages
            'rankjet_page_title_template' => sanitize_text_field($_POST['page_title_template'] ?? '%title% %sep% %sitename%'),
            'rankjet_page_description_template' => sanitize_text_field($_POST['page_description_template'] ?? '%excerpt%'),
            'rankjet_page_default_schema_type' => sanitize_text_field($_POST['page_default_schema_type'] ?? 'Article'),
            'rankjet_page_default_article_type' => sanitize_text_field($_POST['page_default_article_type'] ?? 'BlogPosting'),

            // Categories
            'rankjet_category_title_template' => sanitize_text_field($_POST['category_title_template'] ?? '%term_title% Archives %sep% %sitename%'),
            'rankjet_category_description_template' => sanitize_text_field($_POST['category_description_template'] ?? '%term_description%'),

            // Categories
            'rankjet_category_title_template' => sanitize_text_field($_POST['category_title_template'] ?? '%term_title% Archives %sep% %sitename%'),
            'rankjet_category_description_template' => sanitize_text_field($_POST['category_description_template'] ?? '%term_description%'),

            // Tags
            'rankjet_tag_title_template' => sanitize_text_field($_POST['tag_title_template'] ?? '%term_title% Archives %sep% %sitename%'),
            'rankjet_tag_description_template' => sanitize_text_field($_POST['tag_description_template'] ?? '%term_description%'),
        ];

        foreach ($settings as $key => $value) {
            update_option($key, $value);
        }

        wp_send_json_success(['message' => __('Settings saved successfully.', 'rankjet-ai')]);
    }

    /**
     * Render the settings page
     */
    public function render_settings_page() {
        // Enqueue the build script
        $asset_file = include(RANKJET_AI_PLUGIN_DIR . 'build/index.asset.php');
        
        wp_enqueue_script(
            'rankjet-ai-build',
            RANKJET_AI_PLUGIN_URL . 'build/index.js',
            array_merge($asset_file['dependencies'], ['wp-element', 'wp-components', 'wp-api-fetch']),
            $asset_file['version'],
            true
        );

        wp_enqueue_style(
            'rankjet-ai-style', 
            RANKJET_AI_PLUGIN_URL . 'build/index.css', 
            ['wp-components'], 
            $asset_file['version']
        );

        // Enqueue the component styles (Titles & Meta styles)
        wp_enqueue_style(
            'rankjet-ai-titles-meta-style', 
            RANKJET_AI_PLUGIN_URL . 'build/style-index.css', 
            [], 
            $asset_file['version']
        );

        // Prepare settings data for React
        $robots_default = get_option('rankjet_robots_meta_default', 'index, follow');
        $robots_parts = array_map('trim', explode(',', strtolower($robots_default)));
        
        $settings_data = [
            'robots_meta_default' => $robots_default,
            // Parse individual robot directives for React state initialization
            'robots_index' => in_array('noindex', $robots_parts) ? 'noindex' : 'index',
            'robots_follow' => in_array('nofollow', $robots_parts) ? 'nofollow' : 'follow',
            'robots_archive' => in_array('noarchive', $robots_parts) ? 'noarchive' : 'archive',
            'robots_snippet' => in_array('nosnippet', $robots_parts) ? 'nosnippet' : 'snippet',
            'display_meta_tags' => get_option('rankjet_display_meta_tags', 'yes'),
            'post_title_template' => get_option('rankjet_post_title_template', '%title% %sep% %sitename%'),
            'post_description_template' => get_option('rankjet_post_description_template', '%excerpt%'),
            'title_separator' => get_option('rankjet_title_separator', '-'),
            'post_default_schema_type' => get_option('rankjet_post_default_schema_type', 'Article'),
            'post_default_article_type' => get_option('rankjet_post_default_article_type', 'BlogPosting'),
            
            // Pages Data
            'page_title_template' => get_option('rankjet_page_title_template', '%title% %sep% %sitename%'),
            'page_description_template' => get_option('rankjet_page_description_template', '%excerpt%'),
            'page_default_schema_type' => get_option('rankjet_page_default_schema_type', 'Article'),
            'page_default_article_type' => get_option('rankjet_page_default_article_type', 'BlogPosting'),

            // Categories Data
            'category_title_template' => get_option('rankjet_category_title_template', '%term_title% Archives %sep% %sitename%'),
            'category_description_template' => get_option('rankjet_category_description_template', '%term_description%'),

            // Tags Data
            'tag_title_template' => get_option('rankjet_tag_title_template', '%term_title% Archives %sep% %sitename%'),
            'tag_description_template' => get_option('rankjet_tag_description_template', '%term_description%'),

            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('rankjet_titles_meta_nonce'),
            'site_name' => get_bloginfo('name'),
            'blog_public' => get_option('blog_public'), // '1' public, '0' private
            'plugin_version' => RANKJET_AI_VERSION,
            'custom_templates' => $this->get_custom_templates_data(),
        ];

        wp_localize_script('rankjet-ai-build', 'rankjetTitlesMetaSettings', $settings_data);
        
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline screen-reader-text"><?php _e('Titles & Meta', 'rankjet-ai'); ?></h1>
            <hr class="wp-header-end">
            <div id="rankjet-titles-meta-root"></div>
        </div>
        <?php
    }

    /**
     * Get default robots meta value
     */
    public static function get_robots_meta_default() {
        return get_option('rankjet_robots_meta_default', 'index, follow');
    }

    /**
     * Check if meta tags should be displayed
     */
    public static function should_display_meta_tags() {
        return get_option('rankjet_display_meta_tags', 'yes') === 'yes';
    }

    /**
     * Get processed post title with template
     */
    public static function get_processed_title($post_id) {
        $template = get_option('rankjet_post_title_template', '%title% %sep% %sitename%');
        $separator = get_option('rankjet_title_separator', '-');
        
        // Check for custom SEO title first
        $custom_title = get_post_meta($post_id, '_rankjet_seo_title', true);
        if (!empty($custom_title)) {
            return $custom_title;
        }
        
        $post = get_post($post_id);
        if (!$post) {
            return '';
        }

        $replacements = [
            '%title%' => $post->post_title,
            '%sep%' => ' ' . $separator . ' ',
            '%sitename%' => get_bloginfo('name'),
        ];

        return str_replace(array_keys($replacements), array_values($replacements), $template);
    }

    /**
     * Get processed post description with template
     */
    public static function get_processed_description($post_id) {
        $template = get_option('rankjet_post_description_template', '%excerpt%');
        
        // Check for custom SEO description first
        $custom_desc = get_post_meta($post_id, '_rankjet_seo_description', true);
        if (!empty($custom_desc)) {
            return $custom_desc;
        }
        
        $post = get_post($post_id);
        if (!$post) {
            return '';
        }

        $excerpt = get_the_excerpt($post_id);
        if (empty($excerpt)) {
            $excerpt = wp_trim_words(strip_shortcodes(wp_strip_all_tags($post->post_content)), 25, '...');
        }

        $replacements = [
            '%excerpt%' => $excerpt,
            '%title%' => $post->post_title,
        ];

        return str_replace(array_keys($replacements), array_values($replacements), $template);
    }

    /**
     * Get Custom Templates Data
     */
    private function get_custom_templates_data() {
        $templates = [];
        $query = new WP_Query([
            'post_type' => 'rankjet_schema_tpl',
            'posts_per_page' => -1,
            'post_status' => 'publish'
        ]);

        if ($query->have_posts()) {
            foreach ($query->posts as $post) {
                // Determine content: try meta first, fallback to content
                $json = get_post_meta($post->ID, '_rankjet_schema_json', true);
                if (empty($json)) {
                    $json = $post->post_content;
                }
                
                $data = json_decode($json, true);
                $type = isset($data['@type']) ? $data['@type'] : 'Custom';

                $templates[] = [
                    'id' => (string) $post->ID,
                    'title' => $post->post_title,
                    'type' => $type
                ];
            }
        }
        return $templates;
    }
}
