<?php
/**
 * RankJet AI Sitemap Admin Settings
 * 
 * Admin settings page for XML Sitemap configuration.
 * Follows the Titles & Meta UI pattern with React-based interface.
 * 
 * @package RankJet_AI
 * @since 1.0.8
 */

if (!defined('ABSPATH')) {
    exit;
}

class Rankjet_Ai_Sitemap_Admin {

    /**
     * Constructor - Register hooks
     */
    public function __construct() {
        add_action('admin_menu', [$this, 'add_sitemap_menu'], 25);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('wp_ajax_rankjet_save_sitemap_settings', [$this, 'save_settings_ajax']);
        add_action('wp_ajax_rankjet_clear_sitemap_cache', [$this, 'clear_cache_ajax']);
    }

    /**
     * Add Sitemap submenu page
     */
    public function add_sitemap_menu() {
        add_submenu_page(
            'rankjet-ai',
            __('XML Sitemap', 'rankjet-ai'),
            __('XML Sitemap', 'rankjet-ai'),
            'manage_options',
            'rankjet-ai-sitemap',
            [$this, 'render_settings_page']
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        // Main toggle
        register_setting('rankjet_sitemap_settings', 'rankjet_sitemap_enabled', [
            'type'              => 'string',
            'default'           => 'yes',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        // Post types
        register_setting('rankjet_sitemap_settings', 'rankjet_sitemap_include_posts', [
            'type'              => 'string',
            'default'           => 'yes',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        register_setting('rankjet_sitemap_settings', 'rankjet_sitemap_include_pages', [
            'type'              => 'string',
            'default'           => 'yes',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        register_setting('rankjet_sitemap_settings', 'rankjet_sitemap_include_custom_post_types', [
            'type'              => 'string',
            'default'           => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        // Taxonomies
        register_setting('rankjet_sitemap_settings', 'rankjet_sitemap_include_categories', [
            'type'              => 'string',
            'default'           => 'yes',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        register_setting('rankjet_sitemap_settings', 'rankjet_sitemap_include_tags', [
            'type'              => 'string',
            'default'           => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        register_setting('rankjet_sitemap_settings', 'rankjet_sitemap_include_custom_taxonomies', [
            'type'              => 'string',
            'default'           => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ]);

        // Changefreq settings
        $content_types = ['post', 'page', 'category', 'post_tag'];
        foreach ($content_types as $type) {
            register_setting('rankjet_sitemap_settings', "rankjet_sitemap_{$type}_changefreq", [
                'type'              => 'string',
                'default'           => 'weekly',
                'sanitize_callback' => 'sanitize_text_field'
            ]);
            register_setting('rankjet_sitemap_settings', "rankjet_sitemap_{$type}_priority", [
                'type'              => 'string',
                'default'           => '0.5',
                'sanitize_callback' => 'sanitize_text_field'
            ]);
        }
    }

    /**
     * AJAX handler for saving settings
     */
    public function save_settings_ajax() {
        check_ajax_referer('rankjet_sitemap_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
            return;
        }

        // Sanitize and save settings
        $settings = [
            'rankjet_sitemap_enabled'                  => sanitize_text_field($_POST['sitemap_enabled'] ?? 'yes'),
            'rankjet_sitemap_include_posts'            => sanitize_text_field($_POST['include_posts'] ?? 'yes'),
            'rankjet_sitemap_include_pages'            => sanitize_text_field($_POST['include_pages'] ?? 'yes'),
            'rankjet_sitemap_include_custom_post_types'=> sanitize_text_field($_POST['include_custom_post_types'] ?? 'no'),
            'rankjet_sitemap_include_categories'       => sanitize_text_field($_POST['include_categories'] ?? 'yes'),
            'rankjet_sitemap_include_tags'             => sanitize_text_field($_POST['include_tags'] ?? 'no'),
            'rankjet_sitemap_include_custom_taxonomies'=> sanitize_text_field($_POST['include_custom_taxonomies'] ?? 'no'),
            'rankjet_sitemap_post_changefreq'          => sanitize_text_field($_POST['post_changefreq'] ?? 'weekly'),
            'rankjet_sitemap_post_priority'            => sanitize_text_field($_POST['post_priority'] ?? '0.6'),
            'rankjet_sitemap_page_changefreq'          => sanitize_text_field($_POST['page_changefreq'] ?? 'monthly'),
            'rankjet_sitemap_page_priority'            => sanitize_text_field($_POST['page_priority'] ?? '0.7'),
            'rankjet_sitemap_category_changefreq'      => sanitize_text_field($_POST['category_changefreq'] ?? 'weekly'),
            'rankjet_sitemap_category_priority'        => sanitize_text_field($_POST['category_priority'] ?? '0.4'),
            'rankjet_sitemap_post_tag_changefreq'      => sanitize_text_field($_POST['post_tag_changefreq'] ?? 'weekly'),
            'rankjet_sitemap_post_tag_priority'        => sanitize_text_field($_POST['post_tag_priority'] ?? '0.3'),
        ];

        foreach ($settings as $key => $value) {
            update_option($key, $value);
        }

        // Flush rewrite rules and clear cache when settings change
        flush_rewrite_rules();
        
        // Clear sitemap cache
        if (class_exists('Rankjet_Ai_Sitemap')) {
            $sitemap = new Rankjet_Ai_Sitemap();
            $sitemap->clear_all_cache();
        }

        wp_send_json_success(['message' => __('Settings saved successfully.', 'rankjet-ai')]);
    }

    /**
     * AJAX handler for clearing sitemap cache
     */
    public function clear_cache_ajax() {
        check_ajax_referer('rankjet_sitemap_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
            return;
        }

        // Clear all sitemap caches
        if (class_exists('Rankjet_Ai_Sitemap')) {
            $sitemap = new Rankjet_Ai_Sitemap();
            $sitemap->clear_all_cache();
        }

        wp_send_json_success(['message' => __('Sitemap cache cleared successfully.', 'rankjet-ai')]);
    }

    /**
     * Render the settings page
     */
    public function render_settings_page() {
        // Enqueue the build script
        $asset_file = include(RANKJET_AI_PLUGIN_DIR . 'build/index.asset.php');
        
        wp_enqueue_script(
            'rankjet-ai-build',
            RANKJET_AI_PLUGIN_URL . 'build/index.js',
            array_merge($asset_file['dependencies'], ['wp-element', 'wp-components', 'wp-api-fetch']),
            $asset_file['version'],
            true
        );

        wp_enqueue_style(
            'rankjet-ai-style', 
            RANKJET_AI_PLUGIN_URL . 'build/index.css', 
            ['wp-components'], 
            $asset_file['version']
        );

        // Enqueue additional component styles
        wp_enqueue_style(
            'rankjet-ai-sitemap-style', 
            RANKJET_AI_PLUGIN_URL . 'build/style-index.css', 
            [], 
            $asset_file['version']
        );

        // Get custom post types
        $custom_post_types = get_post_types([
            'public'   => true,
            '_builtin' => false,
        ], 'objects');

        $cpt_list = [];
        foreach ($custom_post_types as $cpt) {
            $cpt_list[] = [
                'name'  => $cpt->name,
                'label' => $cpt->label,
            ];
        }

        // Get custom taxonomies
        $custom_taxonomies = get_taxonomies([
            'public'   => true,
            '_builtin' => false,
        ], 'objects');

        $taxonomy_list = [];
        foreach ($custom_taxonomies as $tax) {
            $taxonomy_list[] = [
                'name'  => $tax->name,
                'label' => $tax->label,
            ];
        }

        // Count stats for display
        $post_count = wp_count_posts('post');
        $page_count = wp_count_posts('page');
        $category_count = wp_count_terms(['taxonomy' => 'category', 'hide_empty' => true]);
        $tag_count = wp_count_terms(['taxonomy' => 'post_tag', 'hide_empty' => true]);

        // Prepare settings data for React
        $settings_data = [
            'sitemap_enabled'              => get_option('rankjet_sitemap_enabled', 'yes'),
            'include_posts'                => get_option('rankjet_sitemap_include_posts', 'yes'),
            'include_pages'                => get_option('rankjet_sitemap_include_pages', 'yes'),
            'include_custom_post_types'    => get_option('rankjet_sitemap_include_custom_post_types', 'no'),
            'include_categories'           => get_option('rankjet_sitemap_include_categories', 'yes'),
            'include_tags'                 => get_option('rankjet_sitemap_include_tags', 'no'),
            'include_custom_taxonomies'    => get_option('rankjet_sitemap_include_custom_taxonomies', 'no'),
            'post_changefreq'              => get_option('rankjet_sitemap_post_changefreq', 'weekly'),
            'post_priority'                => get_option('rankjet_sitemap_post_priority', '0.6'),
            'page_changefreq'              => get_option('rankjet_sitemap_page_changefreq', 'monthly'),
            'page_priority'                => get_option('rankjet_sitemap_page_priority', '0.7'),
            'category_changefreq'          => get_option('rankjet_sitemap_category_changefreq', 'weekly'),
            'category_priority'            => get_option('rankjet_sitemap_category_priority', '0.4'),
            'post_tag_changefreq'          => get_option('rankjet_sitemap_post_tag_changefreq', 'weekly'),
            'post_tag_priority'            => get_option('rankjet_sitemap_post_tag_priority', '0.3'),
            'ajax_url'                     => admin_url('admin-ajax.php'),
            'nonce'                        => wp_create_nonce('rankjet_sitemap_nonce'),
            'sitemap_url'                  => home_url('/sitemap_index.xml'),
            'site_name'                    => get_bloginfo('name'),
            'plugin_version'               => RANKJET_AI_VERSION,
            'custom_post_types'            => $cpt_list,
            'custom_taxonomies'            => $taxonomy_list,
            'stats'                        => [
                'posts'      => isset($post_count->publish) ? (int) $post_count->publish : 0,
                'pages'      => isset($page_count->publish) ? (int) $page_count->publish : 0,
                'categories' => is_wp_error($category_count) ? 0 : (int) $category_count,
                'tags'       => is_wp_error($tag_count) ? 0 : (int) $tag_count,
            ],
        ];

        wp_localize_script('rankjet-ai-build', 'rankjetSitemapSettings', $settings_data);
        
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline screen-reader-text"><?php _e('XML Sitemap', 'rankjet-ai'); ?></h1>
            <hr class="wp-header-end">
            <div id="rankjet-sitemap-root"></div>
        </div>
        <?php
    }
}
