<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * RankJet AI Schema Templates Handler
 * 
 * Handles storage and retrieval of global schema templates using a Custom Post Type.
 */
class Rankjet_Ai_Schema_Templates {

    const CPT = 'rankjet_schema_tpl';

    public function __construct() {
        add_action('init', [$this, 'register_cpt']);
        add_action('rest_api_init', [$this, 'register_routes']);
    }

    /**
     * Register Custom Post Type for Templates
     */
    public function register_cpt() {
        register_post_type(self::CPT, [
            'labels' => [
                'name' => 'Schema Templates',
                'singular_name' => 'Schema Template'
            ],
            'public' => false,
            'show_ui' => false, // Internal use only
            'show_in_rest' => false, // We'll use custom endpoints
            'supports' => ['title', 'custom-fields'], // Store JSON in meta or content? Content is easier for big JSON.
            'capability_type' => 'post',
            'map_meta_cap' => true,
        ]);
    }

    /**
     * Register REST API Routes
     */
    public function register_routes() {
        register_rest_route('rankjet/v1', '/schema-templates', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_templates'],
                'permission_callback' => function () {
                    return current_user_can('edit_posts');
                }
            ],
            [
                'methods' => 'POST',
                'callback' => [$this, 'create_template'],
                'permission_callback' => function () {
                    return current_user_can('edit_posts');
                }
            ]
        ]);

        register_rest_route('rankjet/v1', '/schema-templates/(?P<id>\d+)', [
            [
                'methods' => 'DELETE',
                'callback' => [$this, 'delete_template'],
                'permission_callback' => function () {
                    return current_user_can('edit_posts');
                }
            ],
            [
                'methods' => 'POST', // Use POST for update as well
                'callback' => [$this, 'update_template'],
                'permission_callback' => function () {
                    return current_user_can('edit_posts');
                }
            ]
        ]);
    }

    /**
     * Update an existing template
     */
    public function update_template($request) {
        $id = $request->get_param('id');
        $params = $request->get_json_params();
        
        $post = get_post($id);
        if (!$post || $post->post_type !== self::CPT) {
            return new WP_Error('invalid_template', 'Template not found', ['status' => 404]);
        }

        $title = isset($params['title']) ? sanitize_text_field($params['title']) : $post->post_title;
        $schema = $params['schema'] ?? null;

        $update_args = [
            'ID' => $id,
            'post_title' => $title,
        ];

        // If schema is provided, update content and meta
        if ($schema !== null) {
            $json_str = wp_json_encode($schema);
            $update_args['post_content'] = $json_str;
            update_post_meta($id, '_rankjet_schema_json', $json_str);
        }

        wp_update_post($update_args);

        return rest_ensure_response([
            'id' => $id,
            'title' => $title,
            'schema' => $schema ? $schema : json_decode($post->post_content)
        ]);
    }

    /**
     * Get all templates
     */
    public function get_templates($request) {
        $args = [
            'post_type' => self::CPT,
            'posts_per_page' => -1,
            'post_status' => 'publish'
        ];

        $query = new WP_Query($args);
        $templates = [];

        if ($query->have_posts()) {
            foreach ($query->posts as $post) {
                // Determine content: try meta first, fallback to content
                $json = get_post_meta($post->ID, '_rankjet_schema_json', true);
                if (empty($json)) {
                    $json = $post->post_content;
                }

                $templates[] = [
                    'id' => $post->ID,
                    'title' => $post->post_title,
                    'schema' => json_decode($json) // Return as object
                ];
            }
        }

        return rest_ensure_response($templates);
    }

    /**
     * Create/Save a template
     */
    public function create_template($request) {
        $params = $request->get_json_params();
        
        $title = sanitize_text_field($params['title'] ?? '');
        $schema = $params['schema'] ?? [];

        if (empty($title)) {
            return new WP_Error('missing_title', 'Template title is required', ['status' => 400]);
        }

        if (empty($schema)) {
            return new WP_Error('missing_schema', 'Schema data is required', ['status' => 400]);
        }

        // Store as JSON string
        $json_str = wp_json_encode($schema);

        $post_id = wp_insert_post([
            'post_type' => self::CPT,
            'post_title' => $title,
            'post_status' => 'publish',
            'post_content' => $json_str, // Store in content for backup/searchability
            'meta_input' => [
                '_rankjet_schema_json' => $json_str
            ]
        ]);

        if (is_wp_error($post_id)) {
            return $post_id;
        }

        return rest_ensure_response([
            'id' => $post_id,
            'title' => $title,
            'schema' => $schema
        ]);
    }

    /**
     * Delete a template
     */
    public function delete_template($request) {
        $id = $request->get_param('id');
        
        // Check post type to ensure we only delete templates
        $post = get_post($id);
        if (!$post || $post->post_type !== self::CPT) {
            return new WP_Error('invalid_template', 'Template not found', ['status' => 404]);
        }

        wp_delete_post($id, true);

        return rest_ensure_response(['deleted' => true, 'id' => $id]);
    }
}
