<?php
/**
 * RankJet AI Dashboard Module Controls
 * 
 * Provides a dashboard page for enabling/disabling feature modules
 * like Titles & Meta, XML Sitemap, Redirects, etc.
 * 
 * @package RankJet_AI
 * @since 1.0.8
 */

if (!defined('ABSPATH')) {
    exit;
}

class Rankjet_Ai_Dashboard {

    /**
     * Available modules configuration
     */
    private $modules = [];

    /**
     * Constructor
     */
    public function __construct() {
        // Defer module definition to init hook to avoid early translation loading
        add_action('init', [$this, 'define_modules']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('wp_ajax_rankjet_save_dashboard_modules', [$this, 'save_modules_ajax']);
    }

    /**
     * Define available modules (called on init hook)
     */
    public function define_modules() {
        $this->modules = [
            'titles_meta' => [
                'id'          => 'titles_meta',
                'title'       => __('Titles & Meta', 'rankjet-ai'),
                'description' => __('Manage meta titles, descriptions, and robots directives for all your content.', 'rankjet-ai'),
                'icon'        => 'dashicons-edit-large',
                'settings'    => 'rankjet-ai-titles-meta',
                'default'     => 'yes',
            ],
            'xml_sitemap' => [
                'id'          => 'xml_sitemap',
                'title'       => __('XML Sitemap', 'rankjet-ai'),
                'description' => __('Generate XML sitemaps to help search engines discover and index your content.', 'rankjet-ai'),
                'icon'        => 'dashicons-networking',
                'settings'    => 'rankjet-ai-sitemap',
                'default'     => 'yes',
            ],
            'redirects' => [
                'id'          => 'redirects',
                'title'       => __('Redirects', 'rankjet-ai'),
                'description' => __('Create and manage 301/302 URL redirects to preserve SEO when changing URLs.', 'rankjet-ai'),
                'icon'        => 'dashicons-randomize',
                'settings'    => 'rankjet-ai-redirects',
                'default'     => 'yes',
            ],
            '404_monitor' => [
                'id'          => '404_monitor',
                'title'       => __('404 Monitor', 'rankjet-ai'),
                'description' => __('Track 404 errors and missing pages to identify broken links on your site.', 'rankjet-ai'),
                'icon'        => 'dashicons-warning',
                'settings'    => 'rankjet-ai-redirects&view=404-monitor',
                'default'     => 'yes',
            ],
            'slug_monitor' => [
                'id'          => 'slug_monitor',
                'title'       => __('Slug Monitor', 'rankjet-ai'),
                'description' => __('Automatically create redirects when post or page slugs are changed.', 'rankjet-ai'),
                'icon'        => 'dashicons-admin-links',
                'settings'    => 'rankjet-ai-redirects',
                'default'     => 'yes',
            ],
            'ai_integration' => [
                'id'          => 'ai_integration',
                'title'       => __('AI Integration', 'rankjet-ai'),
                'description' => __('Connect to RankJet AI platform for advanced SEO analysis and content optimization.', 'rankjet-ai'),
                'icon'        => 'dashicons-cloud',
                'settings'    => 'https://app.rankjet.ai',
                'external'    => true,
                'default'     => 'yes',
            ],
        ];
    }

    /**
     * Register settings
     */
    public function register_settings() {
        foreach ($this->modules as $module) {
            register_setting('rankjet_dashboard_settings', 'rankjet_module_' . $module['id'], [
                'type'              => 'string',
                'default'           => $module['default'],
                'sanitize_callback' => 'sanitize_text_field'
            ]);
        }
    }

    /**
     * AJAX handler for saving module states
     */
    public function save_modules_ajax() {
        check_ajax_referer('rankjet_dashboard_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Unauthorized', 'rankjet-ai')]);
            return;
        }

        // Save each module state
        foreach ($this->modules as $module_key => $module) {
            $option_name = 'rankjet_module_' . $module['id'];
            $value = isset($_POST['modules'][$module['id']]) && $_POST['modules'][$module['id']] === 'yes' ? 'yes' : 'no';
            update_option($option_name, $value);
        }

        wp_send_json_success(['message' => __('Settings saved successfully.', 'rankjet-ai')]);
    }

    /**
     * Get modules data for React
     */
    public function get_modules_data() {
        // Ensure modules are defined (for instances created outside init hook)
        if (empty($this->modules)) {
            $this->define_modules();
        }
        
        $modules_data = [];
        
        foreach ($this->modules as $module) {
            $modules_data[] = [
                'id'          => $module['id'],
                'title'       => $module['title'],
                'description' => $module['description'],
                'icon'        => $module['icon'],
                'settings'    => isset($module['external']) && $module['external'] 
                    ? $module['settings'] 
                    : admin_url('admin.php?page=' . $module['settings']),
                'external'    => isset($module['external']) ? $module['external'] : false,
                'enabled'     => get_option('rankjet_module_' . $module['id'], $module['default']) === 'yes',
            ];
        }

        return $modules_data;
    }

    /**
     * Check if a module is enabled
     * 
     * @param string $module_id Module identifier
     * @return bool
     */
    public static function is_module_enabled($module_id) {
        return get_option('rankjet_module_' . $module_id, 'yes') === 'yes';
    }

    /**
     * Get all settings data for the dashboard React app
     */
    public function get_settings_data() {
        return [
            'modules'        => $this->get_modules_data(),
            'ajax_url'       => admin_url('admin-ajax.php'),
            'nonce'          => wp_create_nonce('rankjet_dashboard_nonce'),
            'plugin_version' => RANKJET_AI_VERSION,
            'site_name'      => get_bloginfo('name'),
        ];
    }
}
