<?php
/**
 * Plugin Name: RankBotAI – AI SEO Optimizer for RankMath & Yoast
 * Plugin URI: https://rankbotai.link
 * Description: AI-powered SEO optimizer that enhances Rank Math & Yoast SEO. Automated meta tags, focus keywords, content optimization, SEO scoring, and bulk processing — powered by advanced AI models, works 24/7.
 * Version: 1.2.0
 * Author: BuyReadySite.com
 * Author URI: https://buyreadysite.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: rankbotai-seo-optimizer
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

define('RANKBOT_VERSION', '1.2.0');
define('RANKBOT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('RANKBOT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('RANKBOT_API_URL', 'https://rankbotai.link');

if (!function_exists('rankbot_load_textdomain')) {
    function rankbot_load_textdomain(): void
    {
        $domain = 'rankbotai-seo-optimizer';

        // Extra safety: WordPress can return short locales (e.g. "uk") while our bundles may be full locales
        // (e.g. "uk_UA"). Load the best matching .mo explicitly if present.
        $locale = function_exists('determine_locale') ? determine_locale() : get_locale();
        $locale = is_string($locale) ? $locale : '';
        $locale = apply_filters('rankbot_ai_locale', $locale, $domain);

        $norm = str_replace('-', '_', (string) $locale);
        $aliases = [
            'uk' => 'uk_UA',
            'ru' => 'ru_RU',
            'de' => 'de_DE',
            'es' => 'es_ES',
            'pl' => 'pl_PL',
        ];

        $candidates = array_values(array_unique(array_filter([
            (string) $locale,
            (string) $norm,
            isset($aliases[$locale]) ? (string) $aliases[$locale] : '',
            isset($aliases[$norm]) ? (string) $aliases[$norm] : '',
        ])));

        foreach ($candidates as $loc) {
            $mofile = $domain . '-' . $loc . '.mo';
            $paths = [
                (defined('WP_LANG_DIR') ? WP_LANG_DIR : '') . '/plugins/' . $mofile,
                trailingslashit(RANKBOT_PLUGIN_DIR) . 'languages/' . $mofile,
            ];
            foreach ($paths as $path) {
                if (is_string($path) && $path !== '' && @is_readable($path)) {
                    load_textdomain($domain, $path);
                    return;
                }
            }
        }
    }
}

if (!function_exists('rankbot_is_debug_enabled')) {
    function rankbot_is_debug_enabled(): bool
    {
        // Allow enabling debug logs via wp-config.php without forcing global WP_DEBUG.
        if (defined('RANKBOT_DEBUG_LOG') && RANKBOT_DEBUG_LOG) {
            return true;
        }
        // Enable if WP debug is on, or via an explicit option toggle.
        if (defined('WP_DEBUG') && WP_DEBUG) {
            return true;
        }
        $opt = get_option('rankbot_debug_log', 'no');
        return is_string($opt) && $opt === 'yes';
    }
}

if (!function_exists('rankbot_log')) {
    /**
     * Log RankBot errors into WP debug log (wp-content/debug.log).
     *
     * Informational/debug logs were used during diagnostics and are intentionally
     * suppressed now. Only level=error is emitted.
     *
     * @param string $message
     * @param mixed  $context
     * @param string $level
     */
    function rankbot_log(string $message, $context = null, string $level = 'info'): void
    {
        // Only keep error logs.
        $level = is_string($level) ? strtolower(trim($level)) : 'info';
        if ($level !== 'error') {
            return;
        }

        $line = '[RankBot] [error] ' . $message;
        if ($context !== null) {
            $encoded = function_exists('wp_json_encode') ? wp_json_encode($context) : json_encode($context);
            if (is_string($encoded) && $encoded !== '') {
                $line .= ' ' . $encoded;
            }
        }
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log($line);

        /**
         * Fires when RankBot emits a debug log entry.
         *
         * Plugins/themes may hook this to forward logs to a preferred destination.
         *
         * @param string $message Log message.
         * @param mixed  $context Optional context.
         * @param string $level   Log level.
         */
        do_action('rankbot_ai_log', $message, $context, 'error');
    }
}

require_once RANKBOT_PLUGIN_DIR . 'includes/class-rankbot-api.php';
require_once RANKBOT_PLUGIN_DIR . 'includes/class-rankbot-admin.php';
require_once RANKBOT_PLUGIN_DIR . 'includes/class-rankbot-seo.php';

function rankbot_init()
{
    // Ensure any persisted/legacy API URL points to the canonical endpoint.
    $stored_api_url = get_option('rankbot_api_url');
    if (is_string($stored_api_url) && $stored_api_url !== '' && $stored_api_url !== RANKBOT_API_URL) {
        update_option('rankbot_api_url', RANKBOT_API_URL);
    }

    $api = new RankBot_API();
    new RankBot_Admin($api);

    // Frontend enhancements (schema, etc.)
    if (class_exists('RankBot_SEO') && method_exists('RankBot_SEO', 'init_frontend')) {
        RankBot_SEO::init_frontend();
    }

    // SEO plugin detection — show recommendation notice
    add_action('admin_notices', 'rankbot_seo_plugin_notice');
}

/**
 * Show admin notice recommending Rank Math or Yoast SEO if neither is active.
 */
function rankbot_seo_plugin_notice()
{
    // Only show on RankBot pages or plugins page
    $screen = get_current_screen();
    if (!$screen) return;
    $is_relevant = (
        strpos($screen->id, 'rankbot') !== false ||
        $screen->id === 'plugins'
    );
    if (!$is_relevant) return;

    // Check if any supported SEO plugin is active
    $has_seo = (
        defined('WPSEO_FILE') ||
        defined('RANK_MATH_FILE') ||
        defined('AIOSEO_VERSION')
    );
    if ($has_seo) return;

    // Check if user already dismissed
    $dismissed = get_user_meta(get_current_user_id(), 'rankbot_seo_notice_dismissed', true);
    if ($dismissed === 'yes') return;

    echo '<div class="notice notice-warning is-dismissible" id="rankbot-seo-notice" style="border-left-color: #6366f1;">';
    echo '<p><strong>RankBotAI Recommendation:</strong> For the best SEO results, install and activate ';
    echo '<a href="' . esc_url(admin_url('plugin-install.php?s=rank+math+seo&tab=search&type=term')) . '">Rank Math SEO</a> or ';
    echo '<a href="' . esc_url(admin_url('plugin-install.php?s=yoast+seo&tab=search&type=term')) . '">Yoast SEO</a>. ';
    echo 'RankBotAI works standalone but delivers maximum value when paired with a dedicated SEO plugin.</p>';
    echo '</div>';
    echo '<script>jQuery(document).on("click","#rankbot-seo-notice .notice-dismiss",function(){jQuery.post(ajaxurl,{action:"rankbot_dismiss_seo_notice",_wpnonce:"' . wp_create_nonce('rankbot_dismiss_seo') . '"})});</script>';
}

add_action('wp_ajax_rankbot_dismiss_seo_notice', function() {
    check_ajax_referer('rankbot_dismiss_seo', '_wpnonce');
    update_user_meta(get_current_user_id(), 'rankbot_seo_notice_dismissed', 'yes');
    wp_die();
});

add_action('plugins_loaded', 'rankbot_init');

// Load translations early.
add_action('plugins_loaded', 'rankbot_load_textdomain', 0);

// Note: textdomain loaded on plugins_loaded above. No duplicate init hook needed.

register_activation_hook(__FILE__, 'rankbot_create_tables');

function rankbot_create_tables() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();
    $table_name = $wpdb->prefix . 'rankbot_jobs';
    $bulk_table = $wpdb->prefix . 'rankbot_bulk_queue';

    $sql = "CREATE TABLE $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        job_id varchar(64) NOT NULL,
        site_key_hash varchar(64) DEFAULT '' NOT NULL,
        object_id bigint(20) UNSIGNED NOT NULL,
        object_type varchar(20) DEFAULT 'post' NOT NULL,
        action varchar(50) DEFAULT '' NOT NULL,
        status varchar(20) DEFAULT 'pending' NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        started_at datetime DEFAULT NULL,
        completed_at datetime DEFAULT NULL,
        duration_sec int(11) UNSIGNED DEFAULT NULL,
        PRIMARY KEY  (id),
        UNIQUE KEY job_id (job_id),
        KEY object_id (object_id),
        KEY site_key_hash (site_key_hash),
        KEY site_object (site_key_hash, object_id)
    ) $charset_collate;";

    $sql_bulk = "CREATE TABLE $bulk_table (
        id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        bulk_id varchar(64) NOT NULL,
        site_key_hash varchar(64) DEFAULT '' NOT NULL,
        object_id bigint(20) UNSIGNED NOT NULL,
        object_type varchar(20) DEFAULT 'post' NOT NULL,
        action varchar(50) DEFAULT '' NOT NULL,
        status varchar(20) DEFAULT 'queued' NOT NULL,
        job_id varchar(64) DEFAULT '' NOT NULL,
        attempts int(11) UNSIGNED DEFAULT 0 NOT NULL,
        next_attempt_at datetime DEFAULT NULL,
        last_error text,
        created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        PRIMARY KEY  (id),
        UNIQUE KEY bulk_object (bulk_id, object_id),
        KEY bulk_id (bulk_id),
        KEY site_key_hash (site_key_hash),
        KEY object_id (object_id),
        KEY job_id (job_id),
        KEY status (status),
        KEY next_attempt_at (next_attempt_at)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
    dbDelta($sql_bulk);
    
    // Attempt to run it immediately if plugin is arguably already active in this dev environment
    // but usually activation hook is enough. 
}

// Ensure table exists on update if needed
add_action('plugins_loaded', 'rankbot_update_db_check');
function rankbot_update_db_check() {
    if (get_option('rankbot_db_version') !== '1.4') {
        rankbot_create_tables();
        update_option('rankbot_db_version', '1.4');
    }
}
