<?php
if (!defined('ABSPATH')) {
    exit;
}

class RankBot_SEO {

    private static $did_output_rankbot_product_schema = false;

    public static function init_frontend() {
        if (is_admin()) return;

        $active = self::get_active_plugin();

        // Prefer extending SEO plugin schema (to avoid duplicate Product JSON-LD blocks).
        if ($active === 'rankmath') {
            add_filter('rank_math/json_ld', [__CLASS__, 'extend_rankmath_schema'], 20);
            // If RankMath is active, disable WooCommerce schema output for products to reduce duplicates.
            if (class_exists('WooCommerce')) {
                add_filter('woocommerce_structured_data_enabled', [__CLASS__, 'maybe_disable_wc_structured_data'], 20);
            }
            return;
        }

        if ($active === 'yoast') {
            // Yoast outputs schema graph pieces; extend Product node when present.
            add_filter('wpseo_schema_graph', [__CLASS__, 'extend_yoast_schema'], 20, 2);
            if (class_exists('WooCommerce')) {
                add_filter('woocommerce_structured_data_enabled', [__CLASS__, 'maybe_disable_wc_structured_data'], 20);
            }
            return;
        }

        // Generic fallback for ANY SEO plugin:
        // - Disable WooCommerce schema on product pages (common duplication source)
        // - Output RankBot's enhanced Product JSON-LD once on page load
        if (class_exists('WooCommerce')) {
            add_filter('woocommerce_structured_data_enabled', [__CLASS__, 'maybe_disable_wc_structured_data'], 20);
        }
        add_action('wp_head', [__CLASS__, 'output_rankbot_product_schema'], 99);
    }

    public static function maybe_disable_wc_structured_data($enabled) {
        // Only disable WC structured data on single product pages.
        if (function_exists('is_product') && is_product()) {
            return false;
        }
        return $enabled;
    }

    public static function output_rankbot_product_schema() {
        if (self::$did_output_rankbot_product_schema) return;
        if (!function_exists('is_product') || !is_product()) return;
        if (!function_exists('wc_get_product')) return;

        $product_id = (int) get_queried_object_id();
        if ($product_id <= 0) return;
        $product = wc_get_product($product_id);
        if (!$product) return;

        $schema = self::build_product_schema_additions($product);
        if (empty($schema)) return;

        // Make it a standalone JSON-LD block.
        // Note: other SEO plugins may also output Product JSON-LD. In that case, crawlers can still consume this
        // block as an additional Product description; we avoid WooCommerce schema duplication by disabling WC output.
        // Use JSON_HEX_TAG to escape </script> sequences and prevent XSS.
        echo "\n" . '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_HEX_TAG) . '</script>' . "\n";
        self::$did_output_rankbot_product_schema = true;
    }
    
    public static function get_active_plugin() {
        if (defined('WPSEO_FILE')) return 'yoast';
        if (defined('RANK_MATH_FILE')) return 'rankmath';
        if (defined('AIOSEO_VERSION')) return 'aioseo';
        return 'none';
    }

    public static function update_meta($post_id, $data) {
        $plugin = self::get_active_plugin();
        
        // Data format: ['keyword' => '...', 'title' => '...', 'description' => '...']
        
        switch($plugin) {
            case 'yoast':
                if(isset($data['keyword'])) update_post_meta($post_id, '_yoast_wpseo_focuskw', $data['keyword']);
                if(isset($data['title'])) update_post_meta($post_id, '_yoast_wpseo_title', $data['title']);
                if(isset($data['description'])) update_post_meta($post_id, '_yoast_wpseo_metadesc', $data['description']);
                break;
                
            case 'rankmath':
                if(isset($data['keyword'])) update_post_meta($post_id, 'rank_math_focus_keyword', $data['keyword']);
                if(isset($data['title'])) update_post_meta($post_id, 'rank_math_title', $data['title']);
                if(isset($data['description'])) update_post_meta($post_id, 'rank_math_description', $data['description']);
                break;
            
            case 'aioseo':
                if(isset($data['keyword'])) update_post_meta($post_id, '_aioseo_keywords', $data['keyword']);
                if(isset($data['title'])) update_post_meta($post_id, '_aioseo_title', $data['title']);
                if(isset($data['description'])) update_post_meta($post_id, '_aioseo_description', $data['description']);
                break;
                
            default:
                // If no plugin, maybe just save as standard custom fields for visibility
                if(isset($data['keyword'])) update_post_meta($post_id, 'rankbot_keyword', $data['keyword']);
                break;
        }
    }

    public static function update_term_meta($term_id, $data) {
        $plugin = self::get_active_plugin();
        
        switch($plugin) {
            case 'yoast':
                // Yoast stores term meta in wp_termmeta usually or options, but recently wp_termmeta with keys like:
                // wpseo_taxonomy_meta (serialized), usually easier to interact via API if available, but native meta keys exist too.
                // However, direct update to term meta for Yoast is tricky as it's often serialized in `wpseo_taxonomy_meta` or separate table `yoast_indexable`.
                // For simple implementation we try common keys, but reliable way is updating `_yoast_wpseo_...` equivalent for terms? 
                // Yoast actually uses `wpseo_taxonomy_meta` option or meta mostly.
                // NOTE: Proper Yoast Term update often requires using their surfaces/classes. 
                // We will try standard meta keys if they work in recent versions, otherwise might fail silently.
                if(isset($data['keyword'])) update_term_meta($term_id, 'wpseo_focuskw', $data['keyword']); 
                if(isset($data['title'])) update_term_meta($term_id, 'wpseo_title', $data['title']);
                if(isset($data['description'])) update_term_meta($term_id, 'wpseo_desc', $data['description']);
                break;
                
            case 'rankmath':
                // RankMath stores term meta in wp_termmeta
                if(isset($data['keyword'])) update_term_meta($term_id, 'rank_math_focus_keyword', $data['keyword']);
                if(isset($data['title'])) update_term_meta($term_id, 'rank_math_title', $data['title']);
                if(isset($data['description'])) update_term_meta($term_id, 'rank_math_description', $data['description']);
                break;
            
            case 'aioseo':
                 // AIOSEO term meta
                if(isset($data['title'])) update_term_meta($term_id, '_aioseo_title', $data['title']);
                if(isset($data['description'])) update_term_meta($term_id, '_aioseo_description', $data['description']);
                if(isset($data['keyword'])) update_term_meta($term_id, '_aioseo_keywords', $data['keyword']);
                break;
                
            default:
                if(isset($data['keyword'])) update_term_meta($term_id, 'rankbot_keyword', $data['keyword']);
                break;
        }
    }

    public static function extend_woocommerce_product_schema($markup, $product) {
        if (!is_array($markup) || !is_object($product) || !method_exists($product, 'get_id')) {
            return $markup;
        }

        $product_id = (int) $product->get_id();
        if ($product_id <= 0) return $markup;

        $url = get_permalink($product_id);
        $site_name = get_bloginfo('name');

        $additions = self::build_product_schema_additions($product);
        $markup = self::merge_schema_missing($markup, $additions);

        return $markup;
    }

    public static function extend_rankmath_schema($data) {
        if (!is_array($data) || !function_exists('is_product') || !is_product()) {
            return $data;
        }

        $product_id = get_queried_object_id();
        if (!$product_id) return $data;
        if (!function_exists('wc_get_product')) return $data;
        $product = wc_get_product($product_id);
        if (!$product) return $data;

        $additions = self::build_product_schema_additions($product);
        return self::traverse_and_extend_product_nodes($data, $additions);
    }

    public static function extend_yoast_schema($graph, $context) {
        if (!is_array($graph) || !function_exists('is_product') || !is_product()) {
            return $graph;
        }

        $product_id = get_queried_object_id();
        if (!$product_id) return $graph;
        if (!function_exists('wc_get_product')) return $graph;
        $product = wc_get_product($product_id);
        if (!$product) return $graph;

        $additions = self::build_product_schema_additions($product);
        return self::traverse_and_extend_product_nodes($graph, $additions);
    }

    private static function traverse_and_extend_product_nodes($data, $additions) {
        if (!is_array($data) || !is_array($additions)) {
            return $data;
        }

        foreach ($data as $k => $v) {
            if (is_array($v)) {
                // If this looks like a schema node
                $type = $v['@type'] ?? null;
                $types = is_array($type) ? $type : [$type];
                $types = array_filter(array_map('strval', $types));

                if (in_array('Product', $types, true)) {
                    $data[$k] = self::merge_schema_missing($v, $additions);
                } else {
                    $data[$k] = self::traverse_and_extend_product_nodes($v, $additions);
                }
            }
        }

        return $data;
    }

    private static function merge_schema_missing($base, $additions) {
        if (!is_array($base) || !is_array($additions)) return $base;

        foreach ($additions as $key => $val) {
            if (!array_key_exists($key, $base) || $base[$key] === '' || $base[$key] === null || (is_array($base[$key]) && empty($base[$key]))) {
                $base[$key] = $val;
                continue;
            }

            // Merge arrays for image/review to avoid overwriting existing schema
            if ($key === 'image' && is_array($base[$key]) && is_array($val)) {
                $merged = array_merge($base[$key], $val);
                // Some SEO plugins store complex image objects; dedupe only scalar values to avoid warnings.
                $merged = array_filter($merged, static function ($item) {
                    return is_string($item) || is_int($item) || is_float($item);
                });
                $merged = array_map('strval', $merged);
                $base[$key] = array_values(array_unique(array_filter($merged)));
            }
            if ($key === 'review' && is_array($base[$key]) && is_array($val)) {
                // Keep existing, append ours (dedupe is hard, keep minimal)
                $base[$key] = array_values(array_merge($base[$key], $val));
            }

            // Offers: add missing seller/currency/availability but don't replace prices
            if ($key === 'offers' && is_array($base[$key]) && is_array($val)) {
                $base[$key] = self::merge_schema_missing($base[$key], $val);
            }
        }

        return $base;
    }

    private static function build_product_schema_additions($product) {
        if (!is_object($product) || !method_exists($product, 'get_id')) return [];

        $product_id = (int) $product->get_id();
        $url = get_permalink($product_id);
        $site_name = get_bloginfo('name');

        $out = [
            '@context' => 'https://schema.org/',
            '@type' => 'Product',
        ];

        if (!empty($url)) {
            // Use the canonical URL as @id for better interoperability with different SEO plugins.
            $out['@id'] = $url;
            $out['url'] = $url;
            $out['mainEntityOfPage'] = $url;
        }

        $name = method_exists($product, 'get_name') ? (string) $product->get_name() : '';
        if ($name !== '') $out['name'] = $name;

        $desc = '';
        if (method_exists($product, 'get_short_description')) $desc = (string) $product->get_short_description();
        if ($desc === '' && method_exists($product, 'get_description')) $desc = (string) $product->get_description();
        $desc = trim(wp_strip_all_tags($desc));
        if ($desc !== '') $out['description'] = $desc;

        // Images: featured + gallery
        $images = [];
        if (method_exists($product, 'get_image_id')) {
            $fid = (int) $product->get_image_id();
            if ($fid) {
                $src = wp_get_attachment_url($fid);
                if ($src) $images[] = $src;
            }
        }
        if (method_exists($product, 'get_gallery_image_ids')) {
            foreach ((array) $product->get_gallery_image_ids() as $gid) {
                $src = wp_get_attachment_url((int) $gid);
                if ($src) $images[] = $src;
            }
        }
        $images = array_values(array_unique(array_filter($images)));
        if (!empty($images)) $out['image'] = $images;

        // SKU / identifiers
        $sku = method_exists($product, 'get_sku') ? (string) $product->get_sku() : '';
        if ($sku !== '') $out['sku'] = $sku;

        $maybe_mpn = get_post_meta($product_id, 'mpn', true);
        if (!$maybe_mpn) $maybe_mpn = get_post_meta($product_id, '_mpn', true);
        if (is_string($maybe_mpn) && trim($maybe_mpn) !== '') $out['mpn'] = trim($maybe_mpn);

        $gtin = get_post_meta($product_id, 'gtin', true);
        if (!$gtin) $gtin = get_post_meta($product_id, '_gtin', true);
        if (!$gtin) $gtin = get_post_meta($product_id, '_wpm_gtin_code', true);
        if (!$gtin) $gtin = get_post_meta($product_id, '_wc_gpf_gtin', true);
        if (is_string($gtin)) {
            $gtin = preg_replace('/\D+/', '', $gtin);
            if ($gtin !== '') {
                $len = strlen($gtin);
                if ($len === 8) $out['gtin8'] = $gtin;
                elseif ($len === 12) $out['gtin12'] = $gtin;
                elseif ($len === 13) $out['gtin13'] = $gtin;
                elseif ($len === 14) $out['gtin14'] = $gtin;
                else $out['gtin'] = $gtin;
            }
        }

        $brand = self::extract_product_brand_name($product_id, $product);
        if ($brand !== '') {
            $out['brand'] = ['@type' => 'Brand', 'name' => $brand];
        }

        $cat_terms = get_the_terms($product_id, 'product_cat');
        if (is_array($cat_terms) && !is_wp_error($cat_terms)) {
            $cats = [];
            foreach ($cat_terms as $t) {
                if (!empty($t->name)) $cats[] = $t->name;
            }
            $cats = array_values(array_unique(array_filter($cats)));
            if (!empty($cats)) $out['category'] = implode(', ', $cats);
        }

        // Weight
        if (method_exists($product, 'get_weight')) {
            $w = (string) $product->get_weight();
            if ($w !== '') {
                $out['weight'] = [
                    '@type' => 'QuantitativeValue',
                    'value' => (float) $w,
                    'unitText' => function_exists('wc_get_weight_unit') ? wc_get_weight_unit() : 'kg',
                ];
            }
        }

        // Offers (additive)
        $currency = function_exists('get_woocommerce_currency') ? get_woocommerce_currency() : '';
        $availability = self::schema_availability_for_product($product);
        $seller = [
            '@type' => 'Organization',
            'name' => $site_name,
            'url' => home_url('/'),
        ];

        if (method_exists($product, 'is_type') && $product->is_type('variable') && method_exists($product, 'get_variation_prices')) {
            $prices = $product->get_variation_prices(true);
            $price_values = isset($prices['price']) && is_array($prices['price']) ? array_values($prices['price']) : [];
            $price_values = array_filter(array_map('floatval', $price_values), function($v) { return $v >= 0; });
            if (!empty($price_values)) {
                $out['offers'] = [
                    '@type' => 'AggregateOffer',
                    'lowPrice' => min($price_values),
                    'highPrice' => max($price_values),
                    'priceCurrency' => $currency,
                    'offerCount' => count($price_values),
                    'availability' => $availability,
                    'url' => $url,
                    'seller' => $seller,
                ];
            }
        } else {
            $price = method_exists($product, 'get_price') ? (string) $product->get_price() : '';
            if ($price !== '') {
                $out['offers'] = [
                    '@type' => 'Offer',
                    'price' => (float) $price,
                    'priceCurrency' => $currency,
                    'availability' => $availability,
                    'url' => $url,
                    'seller' => $seller,
                ];
            }
        }

        // Ratings
        if (method_exists($product, 'get_rating_count') && method_exists($product, 'get_average_rating')) {
            $rating_count = (int) $product->get_rating_count();
            $avg = (float) $product->get_average_rating();
            if ($rating_count > 0 && $avg > 0) {
                $out['aggregateRating'] = [
                    '@type' => 'AggregateRating',
                    'ratingValue' => $avg,
                    'reviewCount' => $rating_count,
                    'bestRating' => 5,
                    'worstRating' => 1,
                ];
            }
        }

        // Reviews (up to 5 latest)
        $reviews = get_comments([
            'post_id' => $product_id,
            'status' => 'approve',
            'type' => 'review',
            'number' => 5,
        ]);
        if (is_array($reviews) && !empty($reviews)) {
            $rev = [];
            foreach ($reviews as $c) {
                if (!is_object($c)) continue;
                $rating = get_comment_meta($c->comment_ID, 'rating', true);
                $rating = is_string($rating) || is_numeric($rating) ? (float) $rating : 0;
                $body = trim(wp_strip_all_tags((string) ($c->comment_content ?? '')));
                if ($body === '') continue;

                $item = [
                    '@type' => 'Review',
                    'datePublished' => !empty($c->comment_date_gmt) ? gmdate('c', strtotime($c->comment_date_gmt)) : gmdate('c'),
                    'reviewBody' => $body,
                    'author' => [
                        '@type' => 'Person',
                        'name' => !empty($c->comment_author) ? (string) $c->comment_author : 'Anonymous',
                    ],
                ];
                if ($rating > 0) {
                    $item['reviewRating'] = [
                        '@type' => 'Rating',
                        'ratingValue' => $rating,
                        'bestRating' => 5,
                        'worstRating' => 1,
                    ];
                }
                $rev[] = $item;
            }
            if (!empty($rev)) $out['review'] = $rev;
        }

        return $out;
    }

    private static function schema_availability_for_product($product) {
        if (!is_object($product)) return 'https://schema.org/Discontinued';
        if (method_exists($product, 'is_on_backorder') && $product->is_on_backorder()) {
            return 'https://schema.org/BackOrder';
        }
        if (method_exists($product, 'is_in_stock') && $product->is_in_stock()) {
            return 'https://schema.org/InStock';
        }
        return 'https://schema.org/OutOfStock';
    }

    private static function extract_product_brand_name($product_id, $product) {
        $product_id = (int) $product_id;
        if ($product_id <= 0) return '';

        $tax_candidates = ['product_brand', 'brand'];
        foreach ($tax_candidates as $tax) {
            $terms = get_the_terms($product_id, $tax);
            if (is_array($terms) && !is_wp_error($terms) && !empty($terms)) {
                $t = $terms[0];
                if (is_object($t) && !empty($t->name)) return (string) $t->name;
            }
        }

        // Attribute taxonomy (pa_brand)
        $terms = get_the_terms($product_id, 'pa_brand');
        if (is_array($terms) && !is_wp_error($terms) && !empty($terms)) {
            $t = $terms[0];
            if (is_object($t) && !empty($t->name)) return (string) $t->name;
        }

        // Non-taxonomy product attribute named "brand"
        if (is_object($product) && method_exists($product, 'get_attributes')) {
            foreach ((array) $product->get_attributes() as $attr) {
                if (!is_object($attr) || !method_exists($attr, 'get_name')) continue;
                $name = (string) $attr->get_name();
                if (strtolower($name) !== 'brand') continue;

                if (method_exists($attr, 'is_taxonomy') && $attr->is_taxonomy()) {
                    $vals = wc_get_product_terms($product_id, $name, ['fields' => 'names']);
                    if (is_array($vals) && !empty($vals)) return (string) $vals[0];
                } elseif (method_exists($attr, 'get_options')) {
                    $opts = (array) $attr->get_options();
                    if (!empty($opts)) return (string) $opts[0];
                }
            }
        }

        return '';
    }
}
