<?php

class RankBot_API
{
    // Hardcoded API URL for the MVP
    private $api_url = 'https://rankbotai.link';
    private $option_key = 'rankbot_site_key';

    public function get_api_url()
    {
        // Enforce canonical URL; allow override via filter for advanced setups.
        $base = defined('RANKBOT_API_URL') ? (string) RANKBOT_API_URL : (string) $this->api_url;
        $base = rtrim($base, '/');
        return apply_filters('rankbot_api_url', $base);
    }

    private function detect_local_domain_host(): string
    {
        // Prefer current request host (works even if WP Home/Site URL wasn't updated after migration)
        $host = '';
        $raw_host = filter_input(INPUT_SERVER, 'HTTP_HOST', FILTER_UNSAFE_RAW);
        if (is_string($raw_host) && $raw_host !== '') {
            $host = sanitize_text_field(wp_unslash($raw_host));
            $host = preg_replace('/:\\d+$/', '', $host);
        }

        if (!$host) {
            $host = (string) wp_parse_url(home_url(), PHP_URL_HOST);
        }
        if (!$host) {
            $host = (string) wp_parse_url(site_url(), PHP_URL_HOST);
        }

        $host = strtolower(trim((string) $host));
        return $host;
    }

    private function normalize_domain_for_api(string $raw): string
    {
        $raw = trim($raw);
        if ($raw === '') return '';

        // If a full URL was provided, extract the host.
        if (strpos($raw, '://') !== false) {
            $host = (string) wp_parse_url($raw, PHP_URL_HOST);
            if ($host !== '') {
                return strtolower($host);
            }
        }

        // Strip path/query if present.
        $raw = preg_replace('#^https?://#i', '', $raw);
        $raw = preg_replace('#/.*$#', '', $raw);
        $raw = preg_replace('/:\\d+$/', '', $raw);
        return strtolower(trim($raw));
    }

    private function parse_json_response($response)
    {
        $code = wp_remote_retrieve_response_code($response);
        $body = (string) wp_remote_retrieve_body($response);
        $content_type = (string) wp_remote_retrieve_header($response, 'content-type');
        $location = (string) wp_remote_retrieve_header($response, 'location');

        $data = null;
        if ($body !== '') {
            $data = json_decode($body, true);
        }

        return [
            'code' => $code,
            'body' => $body,
            'data' => is_array($data) ? $data : null,
            'content_type' => $content_type,
            'location' => $location,
        ];
    }

    private function format_http_error(array $parsed, string $fallback)
    {
        $code = (int) ($parsed['code'] ?? 0);
        $location = (string) ($parsed['location'] ?? '');
        $content_type = (string) ($parsed['content_type'] ?? '');

        $data = $parsed['data'] ?? null;
        if (is_array($data) && isset($data['error']) && is_string($data['error']) && $data['error'] !== '') {
            return $data['error'];
        }

        $suffix = '';
        if ($location !== '') {
            $suffix .= ' Location: ' . $location . '.';
        }
        if ($content_type !== '') {
            $suffix .= ' Content-Type: ' . $content_type . '.';
        }

        // If debug logging is enabled, include a short, safe body preview.
        if (function_exists('rankbot_is_debug_enabled') && rankbot_is_debug_enabled()) {
            $body = (string) ($parsed['body'] ?? '');
            $preview = trim(wp_strip_all_tags($body));
            if (strlen($preview) > 240) {
                $preview = substr($preview, 0, 240) . '...';
            }
            if ($preview !== '') {
                $suffix .= ' Body: ' . $preview;
            }
        }

        if ($code > 0) {
            return $fallback . " (HTTP $code)." . $suffix;
        }

        return $fallback . '.' . $suffix;
    }

    private function build_default_headers(?string $bearer = null): array
    {
        $headers = [
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'User-Agent' => 'RankBotAI/' . (defined('RANKBOT_VERSION') ? RANKBOT_VERSION : '1.0.0') . '; ' . home_url('/'),
        ];
        if ($bearer) {
            $headers['Authorization'] = 'Bearer ' . $bearer;
        }
        return $headers;
    }

    public function get_key()
    {
        return get_option($this->option_key);
    }

    public function set_key($key)
    {
        update_option($this->option_key, $key);
    }

    public function reset_key()
    {
        delete_option($this->option_key);
    }

    public function register_site()
    {
        $url = rtrim($this->get_api_url(), '/') . '/api/v1/site/register';

        $domain = $this->detect_local_domain_host();
        if (!$domain) {
            $domain = $this->normalize_domain_for_api((string) site_url());
        }

        $body = [
            'domain' => $domain,
            'platform' => 'wordpress',
            'meta' => [
                'wp_version' => get_bloginfo('version'),
                'plugin_version' => RANKBOT_VERSION
            ]
        ];

        $response = wp_remote_post($url, [
            'body' => json_encode($body),
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'User-Agent' => 'RankBotAI/' . (defined('RANKBOT_VERSION') ? RANKBOT_VERSION : '1.0.0') . '; ' . home_url('/'),
            ],
            'timeout' => 15
        ]);

        if (is_wp_error($response)) {
            return ['error' => $response->get_error_message()];
        }

        $code = wp_remote_retrieve_response_code($response);
        $data = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200) {
            return ['error' => $data['error'] ?? 'Unknown error'];
        }

        if (isset($data['site_key'])) {
            $this->set_key($data['site_key']);
            return ['success' => true, 'balance' => $data['balance']['tokens'] ?? 0];
        }

        return ['error' => 'Invalid response'];
    }

    public function verify_connection()
    {
        $key = $this->get_key();
        if (!$key)
            return ['error' => 'No API Key found'];

        $url = rtrim($this->get_api_url(), '/') . '/api/v1/site/verify';

        if (function_exists('rankbot_log')) {
            rankbot_log('API verify_connection() request', ['url' => $url]);
        }

        $admin_email = '';
        if (function_exists('get_option')) {
            $admin_email = (string) get_option('admin_email', '');
        }
        if (function_exists('sanitize_email')) {
            $admin_email = sanitize_email($admin_email);
        }

        $response = wp_remote_post($url, [
            'body' => json_encode([
                'site_admin_email' => $admin_email,
            ]),
            'headers' => [
                'Authorization' => 'Bearer ' . $key,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'User-Agent' => 'RankBotAI/' . (defined('RANKBOT_VERSION') ? RANKBOT_VERSION : '1.0.0') . '; ' . home_url('/'),
            ],
            'redirection' => 5,
            'timeout' => 15
        ]);

        if (is_wp_error($response)) {
            return ['error' => $response->get_error_message()];
        }

        $parsed = $this->parse_json_response($response);
        $code = (int) ($parsed['code'] ?? 0);
        $data = $parsed['data'];

        if (function_exists('rankbot_log')) {
            rankbot_log('API verify_connection() response', [
                'status' => $code,
                'content_type' => (string) ($parsed['content_type'] ?? ''),
                'location' => (string) ($parsed['location'] ?? ''),
            ]);
        }

        if ($code === 200 && is_array($data)) {
            return ['success' => true, 'data' => $data];
        }

        return ['error' => $this->format_http_error($parsed, 'Verification failed')];
    }

    public function get_balance()
    {
        $status = $this->get_account_status();
        return $status['tokens'] ?? 0;
    }

    public function get_account_status()
    {
        $key = $this->get_key();
        if (!$key)
            return ['tokens' => 0, 'costs' => []];

        $url = rtrim($this->get_api_url(), '/') . '/api/v1/balance';

        // Align displayed costs with the selected model in plugin settings.
        $selected_model = get_option('rankbot_selected_model', '');
        $selected_model = is_string($selected_model) ? trim($selected_model) : '';
        if ($selected_model !== '') {
            $url = add_query_arg(['model_id' => $selected_model], $url);
        }

        if (function_exists('rankbot_log')) {
            rankbot_log('API get_account_status() request', ['url' => $url]);
        }

        $response = wp_remote_get($url, [
            'headers' => $this->build_default_headers($key),
            'redirection' => 5,
            'timeout' => 10
        ]);

        if (is_wp_error($response)) {
            if (function_exists('rankbot_log')) {
                rankbot_log('API get_account_status() WP_Error', ['message' => $response->get_error_message()], 'error');
            }
            return ['tokens' => 0, 'costs' => []];
        }

        $parsed = $this->parse_json_response($response);
        $code = (int) ($parsed['code'] ?? 0);
        $data = $parsed['data'];

        if (function_exists('rankbot_log')) {
            rankbot_log('API get_account_status() response', [
                'status' => $code,
                'content_type' => (string) ($parsed['content_type'] ?? ''),
                'location' => (string) ($parsed['location'] ?? ''),
            ]);
        }

        if ($code === 200 && is_array($data)) {
            return $data;
        }

        // Keep return type stable, but preserve more detail in debug logs.
        if (function_exists('rankbot_log')) {
            rankbot_log('API get_account_status() non-200/non-json', [
                'error' => $this->format_http_error($parsed, 'Balance request failed'),
            ], 'error');
        }

        return ['tokens' => 0, 'costs' => []];
    }

    public function get_plans()
    {
        $key = $this->get_key();
        if (!$key) return [];

        $url = rtrim($this->get_api_url(), '/') . '/api/v1/plans';

        if (function_exists('rankbot_log')) {
            rankbot_log('API get_plans() request', ['url' => $url]);
        }

        $response = wp_remote_get($url, [
            'headers' => $this->build_default_headers($key),
            'redirection' => 5,
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            if (function_exists('rankbot_log')) {
                rankbot_log('API get_plans() WP_Error', ['message' => $response->get_error_message()], 'error');
            }
            return [];
        }

        $parsed = $this->parse_json_response($response);
        $code = (int) ($parsed['code'] ?? 0);
        $data = $parsed['data'];

        if (function_exists('rankbot_log')) {
            rankbot_log('API get_plans() response', [
                'status' => $code,
                'content_type' => (string) ($parsed['content_type'] ?? ''),
                'location' => (string) ($parsed['location'] ?? ''),
            ]);
        }

        if ($code === 200 && is_array($data)) {
            return $data['plans'] ?? [];
        }

        if (function_exists('rankbot_log')) {
            rankbot_log('API get_plans() non-200/non-json', [
                'error' => $this->format_http_error($parsed, 'Plans request failed'),
            ], 'error');
        }

        return [];
    }


    public function get_models()
    {
        $key = $this->get_key();
        if (!$key) return [];

        $url = rtrim($this->get_api_url(), '/') . '/api/v1/models';

        if (function_exists('rankbot_log')) {
            rankbot_log('API get_models() request', ['url' => $url]);
        }

        $response = wp_remote_get($url, [
            'headers' => $this->build_default_headers($key),
            'redirection' => 5,
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            if (function_exists('rankbot_log')) {
                rankbot_log('API get_models() WP_Error', ['message' => $response->get_error_message()], 'error');
            }
            return [];
        }

        $parsed = $this->parse_json_response($response);
        $code = (int) ($parsed['code'] ?? 0);
        $data = $parsed['data'];

        if (function_exists('rankbot_log')) {
            rankbot_log('API get_models() response', [
                'status' => $code,
                'content_type' => (string) ($parsed['content_type'] ?? ''),
                'location' => (string) ($parsed['location'] ?? ''),
            ]);
        }

        if ($code === 200 && is_array($data)) {
            return $data['models'] ?? [];
        }

        if (function_exists('rankbot_log')) {
            rankbot_log('API get_models() non-200/non-json', [
                'error' => $this->format_http_error($parsed, 'Models request failed'),
            ], 'error');
        }

        return [];
    }

    public function get_history($page = 1, $limit = 20, $status = 'all', $type = 'all')
    {
        $key = $this->get_key();
        if (!$key) return [];
        
        $url = rtrim($this->get_api_url(), '/') . '/api/v1/history';
        $url = add_query_arg([
            'page' => $page,
            'limit' => $limit,
            'status' => $status,
            'type' => $type,
        ], $url);

        if (function_exists('rankbot_log')) {
            rankbot_log('API get_history() request', ['url' => $url]);
        }

        $response = wp_remote_get($url, [
            'headers' => $this->build_default_headers($key),
            'redirection' => 5,
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            if (function_exists('rankbot_log')) {
                rankbot_log('API get_history() WP_Error', ['message' => $response->get_error_message()], 'error');
            }
            return [];
        }

        $parsed = $this->parse_json_response($response);
        $code = (int) ($parsed['code'] ?? 0);
        $data = $parsed['data'] ?? [];

        if (function_exists('rankbot_log')) {
            rankbot_log('API get_history() response', [
                'status' => $code,
                'content_type' => (string) ($parsed['content_type'] ?? ''),
                'location' => (string) ($parsed['location'] ?? ''),
            ]);
        }

        if ($code !== 200 || !is_array($data)) {
            if (function_exists('rankbot_log')) {
                rankbot_log('API get_history() non-200/non-json', [
                    'error' => $this->format_http_error($parsed, 'History request failed'),
                ], 'error');
            }
            return [];
        }
        
        // Return full response (history + pagination + stats) if available
        if (isset($data['pagination'])) {
            return $data; 
        }

        return $data['history'] ?? [];
    }

    public function generate($action, $params = [])
    {
        $key = $this->get_key();
        if (!$key) return ['error' => 'Not connected'];
        
        $model = get_option('rankbot_selected_model', 'rankbot-ai-oss');

        $url = rtrim($this->get_api_url(), '/') . '/api/v1/generate';
        
        $body = [
            'action' => $action,
            'model' => $model,
            'params' => $params
        ];

        if (function_exists('rankbot_log')) {
            $ctx = [
                'action' => (string) $action,
                'model' => (string) $model,
            ];
            // Avoid logging big HTML; log only keys for params.
            if (is_array($params)) {
                $ctx['params_keys'] = array_values(array_slice(array_keys($params), 0, 50));
                if (isset($params['post_id'])) $ctx['post_id'] = (int) $params['post_id'];
            }
            rankbot_log('API generate() request', $ctx);
        }

        $timeout = (int) apply_filters('rankbot_api_generate_timeout', 90, $action, $params);
        if ($timeout < 5) {
            $timeout = 5;
        }

        $response = wp_remote_post($url, [
            'headers' => $this->build_default_headers($key),
            'body' => json_encode($body),
            // This endpoint is async and must return quickly with {status: queued, job_id: ...}
            'redirection' => 5,
            'timeout' => $timeout
        ]);

        if (is_wp_error($response)) {
            if (function_exists('rankbot_log')) {
                rankbot_log('API generate() WP_Error', [
                    'action' => (string) $action,
                    'error' => $response->get_error_message(),
                ], 'error');
            }
            return [
                'error' => $response->get_error_message(),
                'error_type' => 'wp_error',
                'error_code' => $response->get_error_code(),
                'http_code' => 0,
            ];
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $bodyRaw = wp_remote_retrieve_body($response);
        $data = json_decode($bodyRaw, true);

        if (function_exists('rankbot_log')) {
            rankbot_log('API generate() response', [
                'action' => (string) $action,
                'code' => (int) $code,
                'body_len' => is_string($bodyRaw) ? strlen($bodyRaw) : 0,
                'body_head' => is_string($bodyRaw) ? substr($bodyRaw, 0, 300) : '',
            ]);
        }

        if ($code !== 200) {
            if (function_exists('rankbot_log')) {
                rankbot_log('API generate() non-200', [
                    'action' => (string) $action,
                    'code' => (int) $code,
                    'error' => is_array($data) ? ($data['error'] ?? null) : null,
                ], 'error');
            }
            return [
                'error' => $data['error'] ?? 'API Error',
                'error_type' => 'http_error',
                'http_code' => (int) $code,
            ];
        }

        if (function_exists('rankbot_log')) {
            $ctx = ['action' => (string) $action];
            if (is_array($data)) {
                if (isset($data['status'])) $ctx['status'] = (string) $data['status'];
                if (isset($data['job_id'])) $ctx['job_id'] = (string) $data['job_id'];
                $ctx['keys'] = array_values(array_slice(array_keys($data), 0, 50));
            }
            rankbot_log('API generate() parsed', $ctx);
        }

        return $data;
    }

    public function check_job($job_id)
    {
        $key = $this->get_key();
        if (!$key) return ['error' => 'Not connected'];
        
        $url = rtrim($this->get_api_url(), '/') . '/api/v1/jobs/check';
        
        $body = ['job_id' => $job_id];

        if (function_exists('rankbot_log')) {
            rankbot_log('API check_job() request', ['job_id' => (string) $job_id]);
        }

        $timeout = (int) apply_filters('rankbot_api_check_job_timeout', 20, $job_id);
        if ($timeout < 5) {
            $timeout = 5;
        }

        $response = wp_remote_post($url, [
            'headers' => $this->build_default_headers((string) $key),
            'body' => wp_json_encode($body),
            'timeout' => $timeout,
            'redirection' => 5,
        ]);

        if (is_wp_error($response)) {
            if (function_exists('rankbot_log')) {
                rankbot_log('API check_job() WP_Error', [
                    'job_id' => (string) $job_id,
                    'error' => $response->get_error_message(),
                ], 'error');
            }
            return ['error' => $response->get_error_message()];
        }

        $parsed = $this->parse_json_response($response);
        $code = (int) ($parsed['code'] ?? 0);
        $data = $parsed['data'] ?? null;

        if (function_exists('rankbot_log')) {
            $ctx = [
                'job_id' => (string) $job_id,
                'code' => $code,
                'content_type' => (string) ($parsed['content_type'] ?? ''),
                'body_len' => is_string($parsed['body'] ?? null) ? strlen((string) $parsed['body']) : 0,
            ];
            if (is_array($data) && isset($data['status'])) {
                $ctx['status'] = (string) $data['status'];
                if ((string) $data['status'] === 'failed') {
                    if (isset($data['error'])) {
                        $err = is_string($data['error']) ? $data['error'] : wp_json_encode($data['error']);
                        $ctx['error'] = is_string($err) ? substr($err, 0, 300) : '';
                    }
                    if (isset($data['message']) && is_string($data['message'])) {
                        $ctx['message'] = substr((string) $data['message'], 0, 300);
                    }
                }
            }
            rankbot_log('API check_job() parsed', $ctx);
        }

        if ($code !== 200 || !is_array($data)) {
            return ['error' => $this->format_http_error($parsed, 'Job status check failed')];
        }

        if (!isset($data['status'])) {
            return ['error' => 'Invalid job status response'];
        }

        return $data;
    }

    // Fire-and-forget job trigger on the API host.
    // Uses WP HTTP with blocking=false so it won't slow admin UX.
    public function trigger_job_async(string $job_id, string $model_id = ''): void
    {
        if ($job_id === '') return;

        $url = rtrim($this->get_api_url(), '/') . '/api/v1/internal/run-job';
        $payload = ['job_id' => $job_id];
        if ($model_id !== '') {
            $payload['model_id'] = $model_id;
        }

        $args = [
            'headers' => [
                // No auth is required by the endpoint, but we send it anyway for future hardening.
                'Authorization' => 'Bearer ' . (string) $this->get_key(),
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($payload),
            // Non-blocking call; keep timeout low.
            'timeout' => 0.5,
            'blocking' => false,
        ];

        wp_remote_post($url, $args);
    }
}
