<?php

if (!defined('ABSPATH')) {
    exit;
}

// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
// phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnsupportedIdentifierPlaceholder

class RankBot_Admin
{
    // phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnsupportedIdentifierPlaceholder
    // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
    private $api;

    private function rankbot_format_duration(int $seconds): string
    {
        if ($seconds < 0) {
            $seconds = 0;
        }

        $h = (int) floor($seconds / 3600);
        $m = (int) floor(($seconds % 3600) / 60);
        $s = (int) ($seconds % 60);

        if ($h > 0) {
            return sprintf('%dh %dm', $h, $m);
        }
        if ($m > 0) {
            return sprintf('%dm %ds', $m, $s);
        }
        return sprintf('%ds', $s);
    }

    private function rankbot_history_storage_mode(): string
    {
        // Default: History page uses server-side /api/v1/history.
        // If a future setup stores history only in WordPress DB, allow enabling via filter.
        $mode = (string) apply_filters('rankbot_history_storage_mode', 'server');
        $mode = strtolower(trim($mode));
        return in_array($mode, ['server', 'wordpress'], true) ? $mode : 'server';
    }

    private function rankbot_site_key_hash(): string
    {
        $key = '';
        try {
            if (is_object($this->api) && method_exists($this->api, 'get_key')) {
                $key = (string) $this->api->get_key();
            }
        } catch (\Exception $e) {
            $key = '';
        }

        $key = trim($key);
        if ($key === '') {
            return '';
        }

        // Храним только хеш, а не сам API ключ.
        return sha1($key);
    }

    public function __construct($api)
    {
        $this->api = $api;
        add_action('admin_menu', [$this, 'add_menu_page']);
        add_action('admin_init', [$this, 'handle_actions']);
        add_action('admin_init', [$this, 'handle_incoming_webhook']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_styles']);
        add_action('edit_form_after_title', [$this, 'render_product_buttons_after_title']);
        add_action('add_meta_boxes', [$this, 'add_sidebar_metabox']);
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_gutenberg_rankbot_sidebar']);
        add_action('admin_bar_menu', [$this, 'add_admin_bar_balance'], 100);
        add_action('wp_ajax_rankbot_optimize', [$this, 'handle_ajax_optimize']);
        add_action('wp_ajax_rankbot_check_job', [$this, 'handle_ajax_check_job']);
        add_action('wp_ajax_rankbot_analyze_seo', [$this, 'handle_ajax_analyze_seo']);
        add_action('wp_ajax_rankbot_get_backups', [$this, 'handle_ajax_get_backups']);
        add_action('wp_ajax_rankbot_restore_backup', [$this, 'handle_ajax_restore_backup']);
        add_action('wp_ajax_rankbot_get_plans', [$this, 'handle_ajax_get_plans']);

        // SEO score backfill controls (Settings page)
        add_action('wp_ajax_rankbot_seo_recalc_start', [$this, 'handle_ajax_seo_recalc_start']);
        add_action('wp_ajax_rankbot_seo_recalc_status', [$this, 'handle_ajax_seo_recalc_status']);

        // Bulk processing
        add_action('wp_ajax_rankbot_estimate_cost', [$this, 'handle_ajax_estimate_cost']);
        add_action('wp_ajax_rankbot_bulk_start', [$this, 'handle_ajax_bulk_start']);
        add_action('wp_ajax_rankbot_bulk_overall_status', [$this, 'handle_ajax_bulk_overall_status']);
        add_action('wp_ajax_rankbot_bulk_items_status', [$this, 'handle_ajax_bulk_items_status']);
        add_action('wp_ajax_rankbot_bulk_count_active', [$this, 'handle_ajax_bulk_count_active']);
        add_action('wp_ajax_rankbot_bulk_stop_all', [$this, 'handle_ajax_bulk_stop_all']);
        add_action('wp_ajax_rankbot_clear_process', [$this, 'handle_ajax_clear_process']);
        add_action('wp_ajax_rankbot_clear_all_jobs', [$this, 'handle_ajax_clear_all_jobs']);
        add_action('wp_ajax_rankbot_bulk_status', [$this, 'handle_ajax_bulk_status']);
        add_action('wp_ajax_rankbot_optimize_term', [$this, 'handle_ajax_optimize_term']);

        // NOTE: Article Optimizer (rankbot_optimize_post) intentionally not exposed anymore.
        add_action('wp_ajax_rankbot_dismiss_job', [$this, 'handle_ajax_dismiss_job']);

        // Background job finalization (runs via WP-Cron)
        add_action('rankbot_poll_job', [$this, 'handle_cron_poll_job'], 10, 1);

        // Bulk task runner (runs via WP-Cron)
        add_action('rankbot_bulk_process_task', [$this, 'handle_cron_bulk_process_task'], 10, 1);

        // SEO score backfill runner (runs via WP-Cron)
        add_action('rankbot_seo_recalc_batch', [$this, 'handle_cron_seo_recalc_batch']);

        // Auto-start SEO backfill once (optional)
        add_action('admin_init', [$this, 'maybe_autostart_seo_recalc']);

        // Query helpers for Bulk Processing filters (safe: only applies when query var is set).
        add_filter('posts_join', [$this, 'rankbot_filter_no_keyword_posts_join'], 10, 2);
        add_filter('posts_where', [$this, 'rankbot_filter_no_keyword_posts_where'], 10, 2);
        add_filter('posts_distinct', [$this, 'rankbot_filter_no_keyword_posts_distinct'], 10, 2);

        // Dashboard Widget
        add_action('wp_dashboard_setup', [$this, 'register_dashboard_widget']);
    }

    /**
     * Enqueue admin styles for RankBotAI pages.
     */
    public function enqueue_admin_styles($hook)
    {
        // Load on RankBotAI pages and post edit screens
        $is_rankbot_page = (strpos($hook, 'rankbot') !== false);
        $is_edit_screen = in_array($hook, ['post.php', 'post-new.php', 'term.php', 'edit-tags.php'], true);
        
        if ($is_rankbot_page || $is_edit_screen) {
            $ver = defined('RANKBOT_VERSION') ? RANKBOT_VERSION : '1.0.0';

            wp_enqueue_style(
                'rankbot-admin-css',
                RANKBOT_PLUGIN_URL . 'assets/css/rankbot-admin.css',
                [],
                $ver
            );

            wp_enqueue_style(
                'rankbot-admin-views-css',
                RANKBOT_PLUGIN_URL . 'assets/css/rankbot-admin-views.css',
                ['rankbot-admin-css'],
                $ver
            );

            // Toast notifications (loaded on all RankBot admin + edit screens)
            wp_enqueue_script(
                'rankbot-toasts-js',
                RANKBOT_PLUGIN_URL . 'assets/js/rankbot-toasts.js',
                [],
                $ver,
                true
            );
        }

        // Bulk page: enqueue its dedicated CSS.
        if (strpos($hook, 'rankbot-bulk') !== false) {
            wp_enqueue_style(
                'rankbot-admin-bulk-css',
                RANKBOT_PLUGIN_URL . 'assets/css/rankbot-admin-bulk.css',
                ['rankbot-admin-css'],
                defined('RANKBOT_VERSION') ? RANKBOT_VERSION : '1.0.0'
            );
        }
    }

    private const RANKBOT_SEO_RECALC_STATE_OPTION = 'rankbot_seo_recalc_state';
    private const RANKBOT_SEO_RECALC_ENABLED_OPTION = 'rankbot_seo_recalc_enabled';
    private const RANKBOT_SEO_RECALC_LAST_AUTOSTART_OPTION = 'rankbot_seo_recalc_last_autostart_at';

    private function rankbot_get_focus_keyword_for_post(int $post_id): string
    {
        $candidates = [
            '_rankbot_focus_keyword',
            '_yoast_wpseo_focuskw',
            'rank_math_focus_keyword',
        ];

        foreach ($candidates as $key) {
            $val = get_post_meta($post_id, $key, true);
            if (!is_string($val)) {
                continue;
            }
            $kw = trim($val);
            if ($kw === '') {
                continue;
            }
            // RankMath can store comma-separated phrases
            if (str_contains($kw, ',')) {
                $kw = trim((string) explode(',', $kw)[0]);
            }
            if ($kw !== '') {
                return $kw;
            }
        }
        return '';
    }

    private function rankbot_normalize_text(string $text): string
    {
        $text = wp_strip_all_tags($text);
        $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        $text = mb_strtolower($text);
        $text = preg_replace('/\s+/u', ' ', $text);
        return trim((string) $text);
    }

    private function rankbot_text_contains_keyword(string $text, string $keyword): bool
    {
        $t = $this->rankbot_normalize_text($text);
        $k = $this->rankbot_normalize_text($keyword);
        if ($k === '' || $t === '') {
            return false;
        }
        return str_contains($t, $k);
    }

    private function rankbot_extract_seo_title_desc(int $post_id, WP_Post $post): array
    {
        $raw_title = '';
        $title_source = 'post_title';

        $yoast_title = get_post_meta($post_id, '_yoast_wpseo_title', true);
        if (is_string($yoast_title) && trim($yoast_title) !== '' && !str_contains($yoast_title, '%%')) {
            $raw_title = trim($yoast_title);
            $title_source = 'yoast';
        }

        if ($raw_title === '') {
            $rm_title = get_post_meta($post_id, 'rank_math_title', true);
            if (is_string($rm_title) && trim($rm_title) !== '' && !str_contains($rm_title, '%')) {
                $raw_title = trim($rm_title);
                $title_source = 'rankmath';
            }
        }

        if ($raw_title === '') {
            $aio_title = get_post_meta($post_id, '_aioseo_title', true);
            if (is_string($aio_title) && trim($aio_title) !== '' && !str_contains($aio_title, '#')) {
                $raw_title = trim($aio_title);
                $title_source = 'aioseo';
            }
        }

        if ($raw_title === '') {
            $raw_title = (string) $post->post_title;
            $title_source = 'post_title';
        }

        $raw_desc = '';
        $desc_source = 'excerpt/content';

        $yoast_desc = get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
        if (is_string($yoast_desc) && trim($yoast_desc) !== '' && !str_contains($yoast_desc, '%%')) {
            $raw_desc = trim($yoast_desc);
            $desc_source = 'yoast';
        }

        if ($raw_desc === '') {
            $rm_desc = get_post_meta($post_id, 'rank_math_description', true);
            if (is_string($rm_desc) && trim($rm_desc) !== '' && !str_contains($rm_desc, '%')) {
                $raw_desc = trim($rm_desc);
                $desc_source = 'rankmath';
            }
        }

        if ($raw_desc === '') {
            $aio_desc = get_post_meta($post_id, '_aioseo_description', true);
            if (is_string($aio_desc) && trim($aio_desc) !== '') {
                $raw_desc = trim($aio_desc);
                $desc_source = 'aioseo';
            }
        }

        if ($raw_desc === '') {
            $raw_desc = (string) $post->post_excerpt;
            if (trim($raw_desc) === '') {
                $raw_desc = mb_substr($this->rankbot_normalize_text((string) $post->post_content), 0, 180);
            }
            $desc_source = 'excerpt/content';
        }

        return [
            'title' => $raw_title,
            'title_source' => $title_source,
            'desc' => $raw_desc,
            'desc_source' => $desc_source,
        ];
    }

    private function rankbot_build_seo_analysis(int $post_id, string $keyword): array
    {
        $post = get_post($post_id);
        if (!$post instanceof WP_Post) {
            return [
                'score' => 0,
                'results' => [],
                'debug' => [
                    'title_source' => 'n/a',
                    'desc_source' => 'n/a',
                    'title_len' => 0,
                    'desc_len' => 0,
                    'title_preview' => '',
                    'desc_preview' => '',
                ],
            ];
        }

        $seo = $this->rankbot_extract_seo_title_desc($post_id, $post);
        $title = (string) $seo['title'];
        $desc = (string) $seo['desc'];
        $content_html = (string) $post->post_content;
        $content_text = $this->rankbot_normalize_text($content_html);

        // Very lightweight tokenization
        $words = preg_split('/\s+/u', $content_text, -1, PREG_SPLIT_NO_EMPTY);
        $word_count = is_array($words) ? count($words) : 0;

        $kw_norm = $this->rankbot_normalize_text($keyword);
        $occurrences = 0;
        if ($kw_norm !== '' && $content_text !== '') {
            $occurrences = substr_count($content_text, $kw_norm);
        }
        $density = ($word_count > 0) ? (($occurrences / $word_count) * 100) : 0;

        // Headings extraction
        $headings = [];
        if ($content_html !== '') {
            if (preg_match_all('/<(h[1-6])[^>]*>(.*?)<\/\1>/is', $content_html, $m)) {
                foreach ($m[2] as $h) {
                    $headings[] = $this->rankbot_normalize_text((string) $h);
                }
            }
        }
        $has_kw_in_headings = false;
        foreach ($headings as $h) {
            if ($this->rankbot_text_contains_keyword($h, $keyword)) {
                $has_kw_in_headings = true;
                break;
            }
        }

        // First part of content
        $first_chunk = '';
        if (is_array($words) && !empty($words)) {
            $first_chunk = implode(' ', array_slice($words, 0, 80));
        }
        $kw_early = $first_chunk !== '' ? $this->rankbot_text_contains_keyword($first_chunk, $keyword) : false;

        // Images + alts
        $img_count = 0;
        $img_kw_alt = false;
        if ($content_html !== '' && preg_match_all('/<img\b[^>]*>/i', $content_html, $imgs)) {
            $img_count = is_array($imgs[0]) ? count($imgs[0]) : 0;
            if ($img_count > 0) {
                foreach ($imgs[0] as $imgTag) {
                    if (preg_match('/\balt\s*=\s*(["\'])(.*?)\1/i', (string) $imgTag, $am)) {
                        if ($this->rankbot_text_contains_keyword((string) $am[2], $keyword)) {
                            $img_kw_alt = true;
                            break;
                        }
                    }
                }
            }
        }

        // Links
        $home = home_url('/');
        $has_internal = false;
        $has_external = false;
        if ($content_html !== '' && preg_match_all('/<a\b[^>]*href\s*=\s*(["\'])(.*?)\1[^>]*>/i', $content_html, $links)) {
            if (!empty($links[2]) && is_array($links[2])) {
                foreach ($links[2] as $href) {
                    $href = (string) $href;
                    if ($href === '' || str_starts_with($href, '#') || str_starts_with($href, 'mailto:') || str_starts_with($href, 'tel:')) {
                        continue;
                    }
                    if (str_starts_with($href, '/') || str_starts_with($href, $home)) {
                        $has_internal = true;
                    } elseif (preg_match('/^https?:\/\//i', $href)) {
                        $has_external = true;
                    }
                }
            }
        }

        $slug = (string) $post->post_name;
        $slug_contains_kw = $slug !== '' ? $this->rankbot_text_contains_keyword(str_replace('-', ' ', $slug), $keyword) : false;

        $post_type = (string) $post->post_type;
        $min_words = ($post_type === 'product') ? 150 : 300;
        $ok_words = (int) floor($min_words * 0.5);

        $checks = [];

        $add_check = function (string $group, string $label, int $score, int $maxPoints, string $message) use (&$checks) {
            $status = 'bad';
            if ($score >= $maxPoints) {
                $status = 'good';
            } elseif ($score > 0) {
                $status = 'ok';
            }
            $checks[] = [
                'group' => $group,
                'label' => $label,
                'status' => $status,
                'message' => $message,
                'score' => $score,
                'maxPoints' => $maxPoints,
            ];
        };

        // Title
        $title_has_kw = $this->rankbot_text_contains_keyword($title, $keyword);
        $add_check(
            'title',
            'Focus keyword in title',
            $title_has_kw ? 14 : 0,
            14,
            $title_has_kw ? 'Great — the keyword is present in the title.' : 'Add the focus keyword to the SEO title (or post title).'
        );

        $title_len = mb_strlen($this->rankbot_normalize_text($title));
        $title_len_score = 0;
        if ($title_len >= 40 && $title_len <= 60) $title_len_score = 6;
        elseif ($title_len >= 30 && $title_len <= 70) $title_len_score = 3;
        $add_check(
            'title',
            'Title length',
            $title_len_score,
            6,
            ($title_len_score === 6) ? 'Nice — title length looks optimal.' : 'Aim for ~40–60 characters for the best SERP fit.'
        );

        // Meta
        $desc_has_kw = $this->rankbot_text_contains_keyword($desc, $keyword);
        $add_check(
            'meta',
            'Focus keyword in meta description',
            $desc_has_kw ? 10 : 0,
            10,
            $desc_has_kw ? 'Good — keyword is present in the meta description.' : 'Include the focus keyword naturally in the meta description.'
        );

        $desc_len = mb_strlen($this->rankbot_normalize_text($desc));
        $desc_len_score = 0;
        if ($desc_len >= 120 && $desc_len <= 160) $desc_len_score = 5;
        elseif ($desc_len >= 80 && $desc_len <= 180) $desc_len_score = 2;
        $add_check(
            'meta',
            'Meta description length',
            $desc_len_score,
            5,
            ($desc_len_score === 5) ? 'Great — description length is within a good range.' : 'Aim for ~120–160 characters.'
        );

        // URL
        $add_check(
            'url',
            'Keyword in URL slug',
            $slug_contains_kw ? 5 : 0,
            5,
            $slug_contains_kw ? 'Good — the slug contains the keyword.' : 'Consider adding the keyword to the URL slug (if appropriate).'
        );

        $slug_len = mb_strlen($slug);
        $slug_len_score = 0;
        if ($slug_len > 0 && $slug_len <= 60) $slug_len_score = 5;
        elseif ($slug_len > 0 && $slug_len <= 80) $slug_len_score = 2;
        $add_check(
            'url',
            'Slug length',
            $slug_len_score,
            5,
            ($slug_len_score === 5) ? 'Nice — URL is short and readable.' : 'Keep the URL slug concise (ideally < 60 chars).'
        );

        // Headings
        $add_check(
            'headings',
            'Keyword in headings',
            $has_kw_in_headings ? 10 : 0,
            10,
            $has_kw_in_headings ? 'Good — keyword appears in at least one heading.' : 'Try adding the keyword to an H1/H2 heading where it fits naturally.'
        );

        // Content
        $len_score = 0;
        if ($word_count >= $min_words) $len_score = 10;
        elseif ($word_count >= $ok_words) $len_score = 5;
        $add_check(
            'content',
            'Content length',
            $len_score,
            10,
            ($len_score === 10) ? 'Great — content length looks solid.' : 'Add more useful content (target at least ' . $min_words . ' words).'
        );

        $add_check(
            'content',
            'Keyword appears early',
            $kw_early ? 10 : 0,
            10,
            $kw_early ? 'Good — keyword appears early in the content.' : 'Try to mention the keyword in the first ~80 words.'
        );

        $density_score = 0;
        if ($density >= 0.5 && $density <= 2.5) $density_score = 10;
        elseif ($density >= 0.2 && $density <= 4.0) $density_score = 5;
        $add_check(
            'content',
            'Keyword density',
            $density_score,
            10,
            ($density_score === 10) ? 'Nice — keyword density looks natural.' : 'Keep density roughly ~0.5%–2.5% (avoid stuffing).'
        );

        // Media
        $img_score = 0;
        if ($img_count === 0) {
            $img_score = 0;
        } elseif ($img_kw_alt) {
            $img_score = 5;
        } else {
            $img_score = 3;
        }
        $add_check(
            'media',
            'Images & alt text',
            $img_score,
            5,
            ($img_count === 0) ? 'No images found — consider adding at least one relevant image.' : ($img_kw_alt ? 'Great — at least one image alt contains the keyword.' : 'Add descriptive alt text (consider including the keyword on one relevant image).')
        );

        // Links
        $add_check(
            'links',
            'Internal links',
            $has_internal ? 5 : 0,
            5,
            $has_internal ? 'Good — internal links found.' : 'Add at least one internal link to a relevant page/product/category.'
        );

        $add_check(
            'links',
            'External links',
            $has_external ? 5 : 0,
            5,
            $has_external ? 'Good — external link found.' : 'Consider linking to a high-quality external source (only if it adds value).'
        );

        // Total
        $max_total = 0;
        $raw_total = 0;
        foreach ($checks as $c) {
            $max_total += (int) $c['maxPoints'];
            $raw_total += (int) $c['score'];
        }
        $score = ($max_total > 0) ? (int) round(($raw_total / $max_total) * 100) : 0;
        if ($score < 0) $score = 0;
        if ($score > 100) $score = 100;

        return [
            'score' => $score,
            'results' => $checks,
            'debug' => [
                'title_source' => (string) $seo['title_source'],
                'desc_source' => (string) $seo['desc_source'],
                'title_len' => (int) mb_strlen($this->rankbot_normalize_text($title)),
                'desc_len' => (int) mb_strlen($this->rankbot_normalize_text($desc)),
                'title_preview' => mb_substr($this->rankbot_normalize_text($title), 0, 60),
                'desc_preview' => mb_substr($this->rankbot_normalize_text($desc), 0, 80),
            ],
        ];
    }

    private function rankbot_get_seo_recalc_state(): array
    {
        $state = get_option(self::RANKBOT_SEO_RECALC_STATE_OPTION, []);
        return is_array($state) ? $state : [];
    }

    private function rankbot_set_seo_recalc_state(array $state): void
    {
        update_option(self::RANKBOT_SEO_RECALC_STATE_OPTION, $state, false);
    }

    private function rankbot_count_missing_seo_scores(): int
    {
        global $wpdb;
        $keyword_keys = [
            '_rankbot_focus_keyword',
            '_yoast_wpseo_focuskw',
            'rank_math_focus_keyword',
        ];

        $cache_key = 'rankbot_missing_seo_scores_count_' . md5(implode('|', $keyword_keys));
        $cached = wp_cache_get($cache_key, 'rankbotai-seo-optimizer');
        if (false !== $cached) {
            return is_numeric($cached) ? (int) $cached : 0;
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $count = $wpdb->get_var(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $wpdb->posts and $wpdb->postmeta are safe table names.
                    "SELECT COUNT(DISTINCT p.ID)
                 FROM {$wpdb->posts} p
                 INNER JOIN {$wpdb->postmeta} kw ON kw.post_id = p.ID AND kw.meta_key IN (%s, %s, %s) AND kw.meta_value <> ''
                 LEFT JOIN {$wpdb->postmeta} sc ON sc.post_id = p.ID AND sc.meta_key = '_rankbot_seo_score'
                 WHERE p.post_type IN ('post','page','product')
                 AND p.post_status IN ('publish','draft','pending','private')
                 AND (sc.meta_id IS NULL OR sc.meta_value = '' OR sc.meta_value = '0')",
                $keyword_keys[0],
                $keyword_keys[1],
                $keyword_keys[2]
            )
        );

        wp_cache_set($cache_key, $count, 'rankbotai-seo-optimizer', 60);
        return is_numeric($count) ? (int) $count : 0;
    }

    private function rankbot_fetch_next_missing_score_ids(int $last_id, int $limit): array
    {
        global $wpdb;
        $keyword_keys = [
            '_rankbot_focus_keyword',
            '_yoast_wpseo_focuskw',
            'rank_math_focus_keyword',
        ];

        $limit = max(1, min(500, $limit));

        $cache_key = 'rankbot_missing_seo_score_ids_' . md5(implode('|', $keyword_keys) . '|' . $last_id . '|' . $limit);
        $cached = wp_cache_get($cache_key, 'rankbotai-seo-optimizer');
        if (false !== $cached && is_array($cached)) {
            return array_values(array_map('absint', $cached));
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $rows = $wpdb->get_col(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $wpdb->posts and $wpdb->postmeta are safe table names.
                "SELECT DISTINCT p.ID
                 FROM {$wpdb->posts} p
                 INNER JOIN {$wpdb->postmeta} kw ON kw.post_id = p.ID AND kw.meta_key IN (%s, %s, %s) AND kw.meta_value <> ''
                 LEFT JOIN {$wpdb->postmeta} sc ON sc.post_id = p.ID AND sc.meta_key = '_rankbot_seo_score'
                 WHERE p.ID > %d
                 AND p.post_type IN ('post','page','product')
                 AND p.post_status IN ('publish','draft','pending','private')
                 AND (sc.meta_id IS NULL OR sc.meta_value = '' OR sc.meta_value = '0')
                 ORDER BY p.ID ASC
                 LIMIT %d",
                $keyword_keys[0],
                $keyword_keys[1],
                $keyword_keys[2],
                $last_id,
                $limit
            )
        );

        if (!is_array($rows)) {
            return [];
        }

        $rows = array_values(array_map('absint', $rows));
        wp_cache_set($cache_key, $rows, 'rankbotai-seo-optimizer', 60);
        return $rows;
    }

    private function rankbot_start_seo_recalc(bool $manual = false): array
    {
        $enabled = get_option(self::RANKBOT_SEO_RECALC_ENABLED_OPTION, 'yes');
        if ($enabled !== 'yes' && !$manual) {
            return $this->rankbot_get_seo_recalc_state();
        }

        $state = $this->rankbot_get_seo_recalc_state();
        if (!empty($state['running'])) {
            return $state;
        }

        $total = $this->rankbot_count_missing_seo_scores();
        $state = [
            'running' => ($total > 0),
            'started_at' => time(),
            'last_run' => null,
            'finished_at' => null,
            'processed' => 0,
            'total' => $total,
            'last_id' => 0,
            'last_error' => null,
            'manual' => $manual,
        ];
        $this->rankbot_set_seo_recalc_state($state);

        if ($total > 0) {
            if (!wp_next_scheduled('rankbot_seo_recalc_batch')) {
                wp_schedule_single_event(time() + 5, 'rankbot_seo_recalc_batch');
            }
        } else {
            $state['running'] = false;
            $state['finished_at'] = time();
            $this->rankbot_set_seo_recalc_state($state);
        }

        return $state;
    }

    public function maybe_autostart_seo_recalc(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        $enabled = get_option(self::RANKBOT_SEO_RECALC_ENABLED_OPTION, 'yes');
        if ($enabled !== 'yes') {
            return;
        }

        $state = $this->rankbot_get_seo_recalc_state();
        if (!empty($state['running'])) {
            return;
        }

        $last_auto = (int) get_option(self::RANKBOT_SEO_RECALC_LAST_AUTOSTART_OPTION, 0);
        // Don’t autostart more than once per day.
        if ($last_auto > 0 && (time() - $last_auto) < DAY_IN_SECONDS) {
            return;
        }

        $missing = $this->rankbot_count_missing_seo_scores();
        if ($missing < 1) {
            return;
        }

        update_option(self::RANKBOT_SEO_RECALC_LAST_AUTOSTART_OPTION, time(), false);
        $this->rankbot_start_seo_recalc(false);
    }

    public function handle_cron_seo_recalc_batch(): void
    {
        $state = $this->rankbot_get_seo_recalc_state();
        if (empty($state['running'])) {
            return;
        }

        $batch_size = 20;
        $last_id = isset($state['last_id']) ? (int) $state['last_id'] : 0;
        $ids = $this->rankbot_fetch_next_missing_score_ids($last_id, $batch_size);

        if (empty($ids)) {
            $state['running'] = false;
            $state['finished_at'] = time();
            $state['last_run'] = time();
            $this->rankbot_set_seo_recalc_state($state);
            return;
        }

        foreach ($ids as $id) {
            $kw = $this->rankbot_get_focus_keyword_for_post((int) $id);
            if ($kw === '') {
                $state['last_id'] = (int) $id;
                continue;
            }

            $analysis = $this->rankbot_build_seo_analysis((int) $id, $kw);
            $score = isset($analysis['score']) ? (int) $analysis['score'] : 0;
            if ($score < 0) $score = 0;
            if ($score > 100) $score = 100;

            update_post_meta((int) $id, '_rankbot_seo_score', (string) $score);

            $state['processed'] = ((int) ($state['processed'] ?? 0)) + 1;
            $state['last_id'] = (int) $id;
        }

        $state['last_run'] = time();
        $total = (int) ($state['total'] ?? 0);
        $processed = (int) ($state['processed'] ?? 0);
        if ($total > 0 && $processed >= $total) {
            $state['running'] = false;
            $state['finished_at'] = time();
            $this->rankbot_set_seo_recalc_state($state);
            return;
        }

        $this->rankbot_set_seo_recalc_state($state);
        wp_schedule_single_event(time() + 10, 'rankbot_seo_recalc_batch');
    }

    public function handle_ajax_seo_recalc_start(): void
    {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        $state = $this->rankbot_start_seo_recalc(true);
        wp_send_json_success(['state' => $state]);
    }

    public function handle_ajax_seo_recalc_status(): void
    {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        $state = $this->rankbot_get_seo_recalc_state();
        $processed = (int) ($state['processed'] ?? 0);
        $total = (int) ($state['total'] ?? 0);
        $pct = ($total > 0) ? (int) floor(($processed / $total) * 100) : 0;
        if ($pct < 0) $pct = 0;
        if ($pct > 100) $pct = 100;
        $state['percent'] = $pct;
        wp_send_json_success(['state' => $state]);
    }

    // --- Cost Estimation Handler ---
    public function handle_ajax_estimate_cost()
    {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        // 1) Collect IDs to estimate over
        $mode = isset($_POST['mode']) ? sanitize_key(wp_unslash($_POST['mode'])) : 'selected';
        $item_ids = [];

        if ($mode === 'selected') {
            if (isset($_POST['ids'])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via absint() below.
                $raw_ids = wp_unslash($_POST['ids']);
                $raw_ids = is_array($raw_ids) ? $raw_ids : [$raw_ids];
                foreach ($raw_ids as $raw_id) {
                    $id = absint($raw_id);
                    if ($id > 0) {
                        $item_ids[] = $id;
                    }
                }
            }
            $item_ids = array_values(array_unique($item_ids));
        } else {
             // Replicate filter logic from handle_ajax_bulk_start (simplified)
             // We need to query for total IDs
             $query_args = $this->rankbot_build_bulk_query_args_from_post();
             if (empty($query_args)) {
                 wp_send_json_error('Invalid request.');
             }
             $query_args['fields'] = 'ids';
             $query_args['posts_per_page'] = -1;
             $query_args['no_found_rows'] = true; // Optimization

             $q = new WP_Query($query_args);
             if (is_array($q->posts) && !empty($q->posts)) {
                 foreach ($q->posts as $pid) {
                     $pid = absint($pid);
                     if ($pid > 0) {
                         $item_ids[] = $pid;
                     }
                 }
             }
        }

        $item_count = count($item_ids);
        if ($item_count < 1) {
            wp_send_json_error('No items to process.');
        }

        // 2) Get Balance + per-action costs for the *selected model*
        $account = $this->api->get_account_status();
        $balance = isset($account['tokens']) ? (float) $account['tokens'] : 0.0;

        // Costs are returned by backend and may be keyed by action name.
        $costs = isset($account['costs']) && is_array($account['costs']) ? $account['costs'] : [];
        if (empty($costs)) {
            // Safe fallback defaults if API fails.
            $costs = [
                'research' => 1,
                'snippet' => 2,
                'post_optimize' => 5,
                'complex' => 10,
                // Legacy keys
                'simple' => 5,
                'optimize' => 5,
            ];
        }

        $requested_action = isset($_POST['bulk_action']) ? sanitize_key(wp_unslash($_POST['bulk_action'])) : 'auto';

        $get_action_cost = static function (array $costs, string $action): float {
            $action = sanitize_key($action);
            if (isset($costs[$action]) && is_numeric($costs[$action])) {
                return (float) $costs[$action];
            }
            // Legacy/fallback mappings.
            if ($action === 'post_optimize') {
                if (isset($costs['optimize']) && is_numeric($costs['optimize'])) return (float) $costs['optimize'];
                if (isset($costs['simple']) && is_numeric($costs['simple'])) return (float) $costs['simple'];
                return 5.0;
            }
            if ($action === 'complex') {
                if (isset($costs['complex']) && is_numeric($costs['complex'])) return (float) $costs['complex'];
                return 10.0;
            }
            if ($action === 'research') {
                if (isset($costs['research']) && is_numeric($costs['research'])) return (float) $costs['research'];
                return 1.0;
            }
            if ($action === 'snippet') {
                if (isset($costs['snippet']) && is_numeric($costs['snippet'])) return (float) $costs['snippet'];
                return 2.0;
            }
            return 5.0;
        };

        // 3) Resolve per-item action (auto depends on post_type), then sum costs.
        global $wpdb;
        $post_types_by_id = [];
        $chunk_size = 800;
        for ($i = 0; $i < count($item_ids); $i += $chunk_size) {
            $chunk = array_slice($item_ids, $i, $chunk_size);
            $chunk = array_values(array_filter(array_map('absint', $chunk)));
            if (empty($chunk)) continue;

            $posts = get_posts([
                'post__in' => $chunk,
                'post_type' => 'any',
                'posts_per_page' => count($chunk),
                'orderby' => 'post__in',
                'fields' => 'all',
                'no_found_rows' => true,
                'suppress_filters' => true,
            ]);
            if (is_array($posts)) {
                foreach ($posts as $p) {
                    if (!($p instanceof WP_Post)) {
                        continue;
                    }
                    $pid = absint($p->ID);
                    if ($pid < 1) {
                        continue;
                    }
                    $pt = sanitize_key((string) $p->post_type);
                    if ($pt !== '') {
                        $post_types_by_id[$pid] = $pt;
                    }
                }
            }
        }

        $action_counts = [];
        $total_cost = 0.0;
        foreach ($item_ids as $pid) {
            $pid = absint($pid);
            if ($pid < 1) continue;
            $pt = $post_types_by_id[$pid] ?? '';
            if ($pt === '') {
                $pt = sanitize_key((string) get_post_type($pid));
            }
            if ($pt === '') continue;

            $resolved_action = $this->rankbot_resolve_bulk_action($pt, $requested_action);
            if ($resolved_action === '') continue;

            $action_counts[$resolved_action] = isset($action_counts[$resolved_action]) ? ((int) $action_counts[$resolved_action] + 1) : 1;
            $total_cost += $get_action_cost($costs, $resolved_action);
        }

        $effective_count = 0;
        foreach ($action_counts as $cnt) {
            $effective_count += (int) $cnt;
        }

        if ($effective_count < 1) {
            wp_send_json_error('No supported items to process.');
        }

        if ($total_cost < 0) {
            $total_cost = 0.0;
        }

        // If all items resolve to the same action, expose per-item cost.
        $price_per_item = null;
        if (count($action_counts) === 1) {
            $only_action = (string) array_key_first($action_counts);
            $price_per_item = $get_action_cost($costs, $only_action);
        }

        $action_label = 'Auto Optimization';
        if ($requested_action === 'research') $action_label = 'Keyword Research';
        if ($requested_action === 'snippet') $action_label = 'Snippet Generation';
        if ($requested_action === 'complex') $action_label = 'Product Optimization (Complex)';
        if ($requested_action === 'post_optimize') $action_label = 'Post Optimization';
        if ($requested_action === 'auto' && count($action_counts) > 1) {
            $action_label = 'Auto Optimization (Mixed Types)';
        }

        // 4) Add selected model info for the UI.
        $selected_model_id = get_option('rankbot_selected_model', '');
        $selected_model_id = is_string($selected_model_id) ? trim($selected_model_id) : '';
        $selected_model_label = $selected_model_id;
        if ($selected_model_id !== '') {
            $models = $this->api->get_models();
            if (is_array($models)) {
                foreach ($models as $m) {
                    if (!is_array($m)) continue;
                    $mid = isset($m['id']) ? (string) $m['id'] : '';
                    if ($mid === $selected_model_id) {
                        $selected_model_label = isset($m['name']) && is_string($m['name']) && $m['name'] !== ''
                            ? $m['name']
                            : $selected_model_id;
                        break;
                    }
                }
            }
        }

        $shortfall = max(0, $total_cost - $balance);
        $can_afford = ($shortfall <= 0);

        // Fetch Plans if needed (optimistic fetching)
        $formatted_plans = [];
        if (!$can_afford) {
            $raw_plans = $this->api->get_plans();
            $base_url = rtrim($this->api->get_api_url(), '/');
            $auth_key = $this->api->get_key(); // Get site key for auto-login

            // 1. Add "Exact Shortfall" option
            if ($shortfall > 0) {
                 // We don't calculate price here to avoid rate mismatch. Backend handles it.
                 $formatted_plans[] = [
                    'id' => 'custom_shortfall',
                    'name' => 'Exact Amount',
                    'type' => 'packet',
                    'tokens' => $shortfall,
                    'tokens_formatted' => number_format($shortfall),
                    'price' => 'Calculated at checkout', 
                    'buy_url' => $base_url . '/dashboard/topup?custom_tokens=' . $shortfall . '&auth_key=' . esc_attr($auth_key),
                    'is_highlight' => true,
                    'is_custom' => true
                 ];
            }

            // 2. Add Standard Plans
            foreach ($raw_plans as $p) {
                // Ensure required fields exist
                if (!isset($p['price'])) $p['price'] = 0;
                if (!isset($p['tokens'])) $p['tokens'] = 0;
                
                $p['tokens_formatted'] = number_format((int)$p['tokens']);
                
                // Construct Buy URL
                $billing = ($p['type'] ?? '') === 'subscription' ? 'monthly' : 'one-time';
                $p['buy_url'] = $base_url . '/dashboard/topup?plan_id=' . ($p['id'] ?? 0) . '&billing=' . $billing . '&auth_key=' . esc_attr($auth_key);
                
                $formatted_plans[] = $p;
            }
        }

        wp_send_json_success([
            'count' => $effective_count,
            'action_label' => $action_label,
            'price_per_item' => $price_per_item,
            'total_cost' => $total_cost,
            'balance' => $balance,
            'can_afford' => $can_afford,
            'shortfall' => $shortfall,
            'plans' => $formatted_plans,
            'model_id' => $selected_model_id,
            'model_label' => $selected_model_label,
            'action_counts' => $action_counts,
        ]);
    }

    private function rankbot_build_bulk_query_args_from_post(): array
    {
        if (!isset($_POST['nonce'])) {
            return [];
        }
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'rankbot_optimize')) {
            wp_die(esc_html__('Security check failed.', 'rankbotai-seo-optimizer'), 403);
        }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized below.
        $post_type = isset($_POST['post_type']) ? (array) wp_unslash($_POST['post_type']) : [];
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized below.
        $post_status = isset($_POST['post_status']) ? (array) wp_unslash($_POST['post_status']) : [];

        $post_type = array_values(array_filter(array_map('sanitize_key', array_map('strval', $post_type))));
        $post_status = array_values(array_filter(array_map('sanitize_key', array_map('strval', $post_status))));
        
        $score_min = isset($_POST['score_min']) ? absint(wp_unslash($_POST['score_min'])) : 0;
        $score_max = isset($_POST['score_max']) ? absint(wp_unslash($_POST['score_max'])) : 100;

        $score_between = [
            'key' => '_rankbot_seo_score',
            'value' => [$score_min, $score_max],
            'compare' => 'BETWEEN',
            'type' => 'NUMERIC',
        ];
        $score_clause = ($score_min <= 0)
            ? [
                'relation' => 'OR',
                $score_between,
                ['key' => '_rankbot_seo_score', 'compare' => 'NOT EXISTS'],
            ]
            : $score_between;

        $meta_query = [
            'relation' => 'AND',
            $score_clause,
        ];

        $only_unprocessed = isset($_POST['only_unprocessed']) ? absint(wp_unslash($_POST['only_unprocessed'])) : 0;
        if ($only_unprocessed === 1) {
             $meta_query[] = [
                'relation' => 'OR',
                ['key' => '_rankbot_history', 'compare' => 'NOT EXISTS'],
                ['key' => '_rankbot_history', 'value' => '', 'compare' => '='],
            ];
        }

        $no_keyword = isset($_POST['no_keyword']) ? absint(wp_unslash($_POST['no_keyword'])) : 0;

        $query_args = [
            'post_type' => $post_type,
            'post_status' => $post_status,
            'meta_query' => $meta_query,
            'rankbot_no_keyword' => $no_keyword,
        ];
        
        $search = isset($_POST['search']) ? sanitize_text_field(wp_unslash($_POST['search'])) : '';
        if ($search !== '') {
            $query_args['s'] = $search;
        }

        return $query_args;
    }

    private function rankbot_bulk_queue_table(): string
    {
        global $wpdb;
        return $wpdb->prefix . 'rankbot_bulk_queue';
    }

    private function rankbot_db_table_exists(string $table): bool
    {
        global $wpdb;
        if ($table === '') {
            return false;
        }

        $prefix = (string) $wpdb->prefix;
        if ($prefix === '' || !str_starts_with($table, $prefix)) {
            return false;
        }

        $cache_key = 'rankbot_table_exists_' . md5($table);
        $cached = wp_cache_get($cache_key, 'rankbotai-seo-optimizer');
        if ($cached !== false) {
            return (bool) $cached;
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $exists = ($wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table)) === $table);
        wp_cache_set($cache_key, $exists, 'rankbotai-seo-optimizer', 300);

        return (bool) $exists;
    }

    private function rankbot_db_column_exists(string $table, string $column): bool
    {
        global $wpdb;
        if ($table === '' || $column === '') {
            return false;
        }

        $prefix = (string) $wpdb->prefix;
        if ($prefix === '' || !str_starts_with($table, $prefix)) {
            return false;
        }

        $cache_key = 'rankbot_column_exists_' . md5($table . '|' . $column);
        $cached = wp_cache_get($cache_key, 'rankbotai-seo-optimizer');
        if ($cached !== false) {
            return (bool) $cached;
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $exists = !empty($wpdb->get_var($wpdb->prepare('SHOW COLUMNS FROM %i LIKE %s', $table, $column)));
        wp_cache_set($cache_key, $exists, 'rankbotai-seo-optimizer', 300);

        return (bool) $exists;
    }

    private function rankbot_bulk_queue_available(): bool
    {
        global $wpdb;
        $table = $this->rankbot_bulk_queue_table();
        return $this->rankbot_db_table_exists($table);
    }

    private function rankbot_bulk_insert_rows(string $table, array $rows): int
    {
        global $wpdb;

        if ($table === '' || empty($rows)) {
            return 0;
        }

        $placeholders = [];
        $values = [];

        foreach ($rows as $row) {
            if (!is_array($row) || count($row) < 12) {
                continue;
            }

            $placeholders[] = "(%s,%s,%d,%s,%s,%s,%s,%d,NULLIF(%s, ''),%s,%s,%s)";
            $values[] = (string) $row[0];
            $values[] = (string) $row[1];
            $values[] = (int) $row[2];
            $values[] = (string) $row[3];
            $values[] = (string) $row[4];
            $values[] = (string) $row[5];
            $values[] = (string) $row[6];
            $values[] = (int) $row[7];
            $values[] = (string) $row[8];
            $values[] = (string) $row[9];
            $values[] = (string) $row[10];
            $values[] = (string) $row[11];
        }

        if (empty($placeholders)) {
            return 0;
        }

        $sql = "INSERT IGNORE INTO %i
            (bulk_id, site_key_hash, object_id, object_type, action, status, job_id, attempts, next_attempt_at, last_error, created_at, updated_at)
            VALUES " . implode(',', $placeholders);
        $values_with_table = array_merge([$table], $values);
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $result = $wpdb->query($wpdb->prepare($sql, ...$values_with_table));

        if ($result === false) {
            return 0;
        }

        return (int) $result;
    }

    private function rankbot_bulk_remaining_count(string $site_key_hash): int
    {
        if (!$this->rankbot_bulk_queue_available()) {
            return 0;
        }

        global $wpdb;
        $table = $this->rankbot_bulk_queue_table();
        $cache_key = 'rankbot_bulk_remaining_' . md5($site_key_hash);
        $cached = wp_cache_get($cache_key, 'rankbotai-seo-optimizer');
        if ($cached !== false) {
            return (int) $cached;
        }

        if ($site_key_hash !== '') {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $count = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status IN ('queued','retry_wait','dispatched')",
                    $table,
                    $site_key_hash
                )
            );
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $count = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM %i WHERE site_key_hash = %s AND status IN ('queued','retry_wait','dispatched')",
                    $table,
                    $site_key_hash
                )
            );
        }
        wp_cache_set($cache_key, (int) $count, 'rankbotai-seo-optimizer', 30);

        return (int) $count;
    }

    private function rankbot_bulk_dispatch_state_key(string $site_key_hash): string
    {
        $suffix = $site_key_hash !== '' ? $site_key_hash : 'default';
        return 'rankbot_bulk_dispatch_state_' . $suffix;
    }

    private function rankbot_bulk_last_tick_key(string $site_key_hash): string
    {
        $suffix = $site_key_hash !== '' ? $site_key_hash : 'default';
        return 'rankbot_bulk_last_tick_' . $suffix;
    }

    private function rankbot_bulk_get_last_tick(string $site_key_hash): int
    {
        $key = $this->rankbot_bulk_last_tick_key($site_key_hash);
        $ts = get_option($key, 0);
        return (int) $ts;
    }

    private function rankbot_bulk_set_last_tick(string $site_key_hash, int $timestamp): void
    {
        $key = $this->rankbot_bulk_last_tick_key($site_key_hash);
        update_option($key, (int) $timestamp, false);
    }

    private function rankbot_bulk_get_dispatch_state(string $site_key_hash): array
    {
        $key = $this->rankbot_bulk_dispatch_state_key($site_key_hash);
        $state = get_option($key, []);
        if (!is_array($state)) {
            $state = [];
        }

        $min = (int) apply_filters('rankbot_bulk_dispatch_min', 1);
        $max = (int) apply_filters('rankbot_bulk_dispatch_max', 10);
        $base = (int) apply_filters('rankbot_bulk_dispatch_base', 10);

        $state = array_merge([
            'limit' => $base,
            'min' => max(1, $min),
            'max' => max(1, $max),
            'success_streak' => 0,
            'error_streak' => 0,
            'updated_at' => current_time('mysql'),
        ], $state);

        $state['min'] = max(1, (int) $state['min']);
        $state['max'] = max($state['min'], (int) $state['max']);
        $state['limit'] = max($state['min'], min($state['max'], (int) $state['limit']));

        return $state;
    }

    private function rankbot_bulk_set_dispatch_state(string $site_key_hash, array $state): void
    {
        $key = $this->rankbot_bulk_dispatch_state_key($site_key_hash);
        $state['updated_at'] = current_time('mysql');
        update_option($key, $state, false);
    }

    private function rankbot_bulk_pause_state_key(string $site_key_hash): string
    {
        $suffix = $site_key_hash !== '' ? $site_key_hash : 'default';
        return 'rankbot_bulk_pause_state_' . $suffix;
    }

    private function rankbot_bulk_get_pause_state(string $site_key_hash): array
    {
        $key = $this->rankbot_bulk_pause_state_key($site_key_hash);
        $state = get_option($key, []);
        if (!is_array($state)) {
            $state = [];
        }

        $state = array_merge([
            'until' => 0,
            'reason' => '',
            'updated_at' => current_time('mysql'),
        ], $state);

        $state['until'] = (int) $state['until'];
        $state['reason'] = (string) $state['reason'];
        return $state;
    }

    private function rankbot_bulk_set_pause_state(string $site_key_hash, array $state): void
    {
        $key = $this->rankbot_bulk_pause_state_key($site_key_hash);
        $state['updated_at'] = current_time('mysql');
        update_option($key, $state, false);
    }

    private function rankbot_bulk_clear_pause_state(string $site_key_hash): void
    {
        delete_option($this->rankbot_bulk_pause_state_key($site_key_hash));
    }

    private function rankbot_bulk_adjust_dispatch_limit(string $site_key_hash, int $dispatched, int $errors): int
    {
        $state = $this->rankbot_bulk_get_dispatch_state($site_key_hash);

        if ($errors > 0) {
            $state['error_streak'] = (int) $state['error_streak'] + 1;
            $state['success_streak'] = 0;
            $state['limit'] = max($state['min'], (int) floor($state['limit'] / 2));
        } elseif ($dispatched > 0) {
            $state['success_streak'] = (int) $state['success_streak'] + 1;
            $state['error_streak'] = 0;
            if ($state['success_streak'] >= 2) {
                $state['limit'] = min($state['max'], (int) $state['limit'] + 1);
                $state['success_streak'] = 0;
            }
        }

        $this->rankbot_bulk_set_dispatch_state($site_key_hash, $state);
        return (int) $state['limit'];
    }

    private function rankbot_bulk_next_retry_delay(int $attempts): int
    {
        $attempts = max(1, $attempts);
        $base = (int) apply_filters('rankbot_bulk_retry_base_delay', 60);
        $step = (int) apply_filters('rankbot_bulk_retry_step_delay', 30);
        $max = (int) apply_filters('rankbot_bulk_retry_max_delay', 300);

        $delay = $base + ($attempts - 1) * $step;
        if ($delay > $max) {
            $delay = $max;
        }

        $jitter = wp_rand(0, 30);
        return $delay + $jitter;
    }

    private function rankbot_bulk_schedule_runner(int $delay = 10): void
    {
        $delay = max(1, $delay);
        $now = time();
        $next = wp_next_scheduled('rankbot_bulk_process_task');
        if ($next && $next < ($now - 5)) {
            wp_clear_scheduled_hook('rankbot_bulk_process_task');
            $next = false;
        }
        if ($next && $next <= ($now + $delay + 2)) {
            return;
        }
        if ($next) {
            wp_clear_scheduled_hook('rankbot_bulk_process_task');
        }
        wp_schedule_single_event($now + $delay, 'rankbot_bulk_process_task');
    }

    private function rankbot_bulk_get_active_bulk_ids(string $site_key_hash): array
    {
        if (!$this->rankbot_bulk_queue_available()) {
            return [];
        }
        global $wpdb;
        $table = $this->rankbot_bulk_queue_table();
        $cache_key = 'rankbot_bulk_active_ids_' . md5($site_key_hash);
        $cached = wp_cache_get($cache_key, 'rankbotai-seo-optimizer');
        if ($cached !== false && is_array($cached)) {
            return $cached;
        }

        if ($site_key_hash !== '') {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $rows = $wpdb->get_col(
                $wpdb->prepare(
                    "SELECT DISTINCT bulk_id FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status IN ('queued','retry_wait','dispatched')",
                    $table,
                    $site_key_hash
                )
            );
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $rows = $wpdb->get_col(
                $wpdb->prepare(
                    "SELECT DISTINCT bulk_id FROM %i WHERE site_key_hash = %s AND status IN ('queued','retry_wait','dispatched')",
                    $table,
                    $site_key_hash
                )
            );
        }
        if (!is_array($rows)) {
            return [];
        }
        $rows = array_values(array_unique(array_filter(array_map('strval', $rows))));
        wp_cache_set($cache_key, $rows, 'rankbotai-seo-optimizer', 30);
        return $rows;
    }

    private function rankbot_unschedule_job_poll(string $job_id): void
    {
        if ($job_id === '') return;
        // There can be multiple scheduled events; clear them all for this arg.
        wp_clear_scheduled_hook('rankbot_poll_job', [(string) $job_id]);
    }

    public function rankbot_filter_no_keyword_posts_join(string $join, $query): string
    {
        if (!($query instanceof WP_Query)) return $join;
        if ((int) $query->get('rankbot_no_keyword') !== 1) return $join;

        global $wpdb;
        $pm = $wpdb->postmeta;
        $p = $wpdb->posts;

        if (strpos($join, ' rb_ykw ') === false) {
            $join .= " LEFT JOIN {$pm} rb_ykw ON ({$p}.ID = rb_ykw.post_id AND rb_ykw.meta_key = '_yoast_wpseo_focuskw') ";
        }
        if (strpos($join, ' rb_rmkw ') === false) {
            $join .= " LEFT JOIN {$pm} rb_rmkw ON ({$p}.ID = rb_rmkw.post_id AND rb_rmkw.meta_key = 'rank_math_focus_keyword') ";
        }
        if (strpos($join, ' rb_rbkw ') === false) {
            $join .= " LEFT JOIN {$pm} rb_rbkw ON ({$p}.ID = rb_rbkw.post_id AND rb_rbkw.meta_key = '_rankbot_focus_keyword') ";
        }

        return $join;
    }

    public function rankbot_filter_no_keyword_posts_where(string $where, $query): string
    {
        if (!($query instanceof WP_Query)) return $where;
        if ((int) $query->get('rankbot_no_keyword') !== 1) return $where;

        $where .= " AND ( (rb_ykw.meta_id IS NULL OR rb_ykw.meta_value = '') AND (rb_rmkw.meta_id IS NULL OR rb_rmkw.meta_value = '') AND (rb_rbkw.meta_id IS NULL OR rb_rbkw.meta_value = '') ) ";
        return $where;
    }

    public function rankbot_filter_no_keyword_posts_distinct(string $distinct, $query): string
    {
        if (!($query instanceof WP_Query)) return $distinct;
        if ((int) $query->get('rankbot_no_keyword') !== 1) return $distinct;
        return 'DISTINCT';
    }

    private function schedule_job_poll(string $job_id): void
    {
        if ($job_id === '') return;

        if (!wp_next_scheduled('rankbot_poll_job', [$job_id])) {
            // Faster first poll improves Bulk UX (rows unlock sooner).
            wp_schedule_single_event(time() + 5, 'rankbot_poll_job', [$job_id]);
        }
    }

    private function reschedule_job_poll(string $job_id): void
    {
        $key = 'rankbot_job_attempts_' . $job_id;
        $attempts = (int) get_transient($key);
        $attempts++;
        set_transient($key, $attempts, DAY_IN_SECONDS);

        // Backoff up to 5 minutes.
        $delay = min(300, 10 + ($attempts * 10));
        wp_schedule_single_event(time() + $delay, 'rankbot_poll_job', [$job_id]);
    }

    public function handle_cron_poll_job(string $job_id): void
    {
        if ($job_id === '') return;

        global $wpdb;
        $table_name = $wpdb->prefix . 'rankbot_jobs';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM %i WHERE job_id = %s LIMIT 1",
                $table_name,
                $job_id
            ),
            ARRAY_A
        );
        if (!$row) {
            return;
        }

        $local_status = isset($row['status']) ? (string) $row['status'] : '';
        if ($local_status === 'completed' || $local_status === 'failed' || $local_status === 'cancelled' || $local_status === 'error') {
            return;
        }

        // Guardrail: if a job is polled for too long without resolution, mark it failed so bulk UI can finish.
        // We already do exponential-ish backoff up to 5 minutes, so 120 attempts is many hours.
        $attempts_key = 'rankbot_job_attempts_' . $job_id;
        $attempts = (int) get_transient($attempts_key);
        if ($attempts >= 120) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->update($table_name, ['status' => 'failed'], ['job_id' => $job_id]);
            return;
        }

        $result = $this->api->check_job((string) $job_id);
        // Note: API responses for failed jobs can include an `error` field.
        // Only treat `error` as a transport-level failure when no status is provided.
        if (isset($result['error']) && !isset($result['status'])) {
            $this->reschedule_job_poll((string) $job_id);
            return;
        }

        $remote_status = isset($result['status']) ? (string) $result['status'] : '';
        $remote_error = '';
        if (is_array($result) && isset($result['error'])) {
            $remote_error = is_string($result['error']) ? (string) $result['error'] : wp_json_encode($result['error']);
        }

        $has_duration_cols = false;
        try {
            $has_started = $this->rankbot_db_column_exists($table_name, 'started_at');
            $has_completed = $this->rankbot_db_column_exists($table_name, 'completed_at');
            $has_duration = $this->rankbot_db_column_exists($table_name, 'duration_sec');
            $has_duration_cols = !empty($has_started) && !empty($has_completed) && !empty($has_duration);
        } catch (\Exception $e) {
            $has_duration_cols = false;
        }

        // Keep local status in sync for better admin UI (queued -> processing).
        if ($remote_status === 'processing' && ($local_status === 'queued' || $local_status === 'pending')) {
            if ($has_duration_cols) {
                $now = current_time('mysql');
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->query(
                    $wpdb->prepare(
                        "UPDATE %i SET status = 'processing', started_at = COALESCE(started_at, %s) WHERE job_id = %s AND status IN ('queued','pending')",
                        $table_name,
                        $now,
                        $job_id
                    )
                );
            } else {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->update($table_name, ['status' => 'processing'], ['job_id' => $job_id]);
            }

            // Refresh local var for downstream logic
            $local_status = 'processing';
        }

        if ($remote_status === 'completed' && isset($result['result'])) {
            // Apply exactly once (cron and AJAX polling can race).
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $claimed = $wpdb->query(
                $wpdb->prepare(
                    "UPDATE %i SET status = 'completed' WHERE job_id = %s AND status IN ('queued','processing','pending')",
                    $table_name,
                    $job_id
                )
            );

            if ((int) $claimed < 1) {
                return;
            }

            if ($has_duration_cols) {
                $now = current_time('mysql');
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->query(
                    $wpdb->prepare(
                        "UPDATE %i SET completed_at = %s, duration_sec = TIMESTAMPDIFF(SECOND, COALESCE(started_at, created_at), %s) WHERE job_id = %s",
                        $table_name,
                        $now,
                        $now,
                        $job_id
                    )
                );
            }

            $object_id = isset($row['object_id']) ? absint($row['object_id']) : 0;
            $object_type = isset($row['object_type']) ? (string) $row['object_type'] : 'post';
            $action = isset($row['action']) ? (string) $row['action'] : '';

            // If a backup restore happened after this job was queued, do not apply the result.
            if ($object_type === 'post' && $object_id > 0) {
                $restored_at = (int) get_post_meta($object_id, '_rankbot_restored_at', true);
                if ($restored_at > 0) {
                    $created_at = isset($row['created_at']) ? (string) $row['created_at'] : '';
                    $created_ts = $created_at !== '' ? (int) strtotime($created_at) : 0;
                    if ($created_ts > 0 && $restored_at > $created_ts) {
                        return;
                    }
                }
            }

            if ($object_type === 'term') {
                $apply = $this->apply_term_optimization_result($object_id, $result['result']);
            } else {
                $apply = $this->apply_optimization_result($object_id, $result['result'], $action, (string) $job_id);
            }

            if (isset($apply['error'])) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->update($table_name, ['status' => 'failed'], ['job_id' => $job_id]);

                // Restore pre-run snapshot if available (avoid leaving content half-applied).
                $backup_ts = (int) get_transient('rankbot_job_backup_ts_' . $job_id);
                if ($backup_ts > 0 && $object_type === 'post' && $object_id > 0) {
                    $this->rankbot_restore_backup_snapshot($object_id, $backup_ts);
                }
                delete_transient('rankbot_job_backup_ts_' . $job_id);
            }

            // Bulk queue sync: if this job belongs to a bulk row, mark that row completed.
            if ($this->rankbot_bulk_queue_available()) {
                $queue_table = $this->rankbot_bulk_queue_table();
                $site_key_hash = $this->rankbot_site_key_hash();
                $now = current_time('mysql');

                if ($site_key_hash !== '') {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $q = $wpdb->get_row(
                        $wpdb->prepare(
                            "SELECT id FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status = 'dispatched' AND job_id = %s ORDER BY id DESC LIMIT 1",
                            $queue_table,
                            $site_key_hash,
                            $job_id
                        ),
                        ARRAY_A
                    );
                } else {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $q = $wpdb->get_row(
                        $wpdb->prepare(
                            "SELECT id FROM %i WHERE site_key_hash = %s AND status = 'dispatched' AND job_id = %s ORDER BY id DESC LIMIT 1",
                            $queue_table,
                            $site_key_hash,
                            $job_id
                        ),
                        ARRAY_A
                    );
                }

                if (is_array($q) && !empty($q['id'])) {
                    $queue_id = (int) $q['id'];
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->update(
                        $queue_table,
                        [
                            'status' => 'completed',
                            'updated_at' => $now,
                        ],
                        ['id' => $queue_id]
                    );

                    // Nudge the bulk runner so overall counters finish promptly.
                    $this->rankbot_bulk_schedule_runner(5);
                }
            }

            delete_transient('rankbot_job_backup_ts_' . $job_id);
            delete_transient('rankbot_job_attempts_' . $job_id);

            return;
        }

        if ($remote_status === 'failed') {
            // If the job failed, restore the pre-run snapshot (if any) to avoid leaving content half-broken.
            $backup_ts = (int) get_transient('rankbot_job_backup_ts_' . $job_id);
            $object_id = isset($row['object_id']) ? absint($row['object_id']) : 0;
            $object_type = isset($row['object_type']) ? (string) $row['object_type'] : 'post';
            if ($backup_ts > 0 && $object_type === 'post' && $object_id > 0) {
                $this->rankbot_restore_backup_snapshot($object_id, $backup_ts);
                delete_transient('rankbot_job_backup_ts_' . $job_id);
            }

            if ($has_duration_cols) {
                $now = current_time('mysql');
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->query(
                    $wpdb->prepare(
                        "UPDATE %i SET status = 'failed', completed_at = %s, duration_sec = TIMESTAMPDIFF(SECOND, COALESCE(started_at, created_at), %s) WHERE job_id = %s",
                        $table_name,
                        $now,
                        $now,
                        $job_id
                    )
                );
            } else {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->update($table_name, ['status' => 'failed'], ['job_id' => $job_id]);
            }

            // Bulk queue sync: requeue failed bulk items for retry (or cancel if max attempts reached).
            if ($this->rankbot_bulk_queue_available()) {
                $queue_table = $this->rankbot_bulk_queue_table();
                $site_key_hash = $this->rankbot_site_key_hash();
                $now = current_time('mysql');

                if ($site_key_hash !== '') {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $q = $wpdb->get_row(
                        $wpdb->prepare(
                            "SELECT id, attempts FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status = 'dispatched' AND job_id = %s ORDER BY id DESC LIMIT 1",
                            $queue_table,
                            $site_key_hash,
                            $job_id
                        ),
                        ARRAY_A
                    );
                } else {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $q = $wpdb->get_row(
                        $wpdb->prepare(
                            "SELECT id, attempts FROM %i WHERE site_key_hash = %s AND status = 'dispatched' AND job_id = %s ORDER BY id DESC LIMIT 1",
                            $queue_table,
                            $site_key_hash,
                            $job_id
                        ),
                        ARRAY_A
                    );
                }

                if (is_array($q) && !empty($q['id'])) {
                    $queue_id = (int) $q['id'];
                    $attempts = isset($q['attempts']) ? (int) $q['attempts'] : 0;
                    $attempts++;

                    // Default to a finite retry count to avoid infinite loops on hard failures.
                    $max_attempts = (int) apply_filters('rankbot_bulk_max_attempts', 3);
                    if ($max_attempts < 0) {
                        $max_attempts = 0;
                    }

                    // Empty provider response is usually transient, but should not loop forever.
                    if ($remote_error !== '' && stripos($remote_error, 'Empty content received from OpenAI') !== false) {
                        $max_attempts = ($max_attempts > 0) ? min($max_attempts, 2) : 2;
                    }

                    $last_error = 'job_failed';
                    if ($remote_error !== '') {
                        $safe = wp_strip_all_tags((string) $remote_error);
                        $safe = substr($safe, 0, 220);
                        $last_error = 'job_failed: ' . $safe;
                    }

                    if ($max_attempts > 0 && $attempts >= $max_attempts) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update(
                            $queue_table,
                            [
                                'status' => 'cancelled',
                                'attempts' => $attempts,
                                'next_attempt_at' => null,
                                'job_id' => '',
                                'last_error' => $last_error,
                                'updated_at' => $now,
                            ],
                            ['id' => $queue_id]
                        );
                    } else {
                        $delay = $this->rankbot_bulk_next_retry_delay($attempts);
                        $next_attempt_at = gmdate('Y-m-d H:i:s', time() + $delay);

                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update(
                            $queue_table,
                            [
                                'status' => 'retry_wait',
                                'attempts' => $attempts,
                                'next_attempt_at' => $next_attempt_at,
                                'job_id' => '',
                                'last_error' => $last_error,
                                'updated_at' => $now,
                            ],
                            ['id' => $queue_id]
                        );

                        $this->rankbot_bulk_schedule_runner(5);
                    }
                }
            }

            $this->rankbot_unschedule_job_poll((string) $job_id);
            delete_transient('rankbot_job_attempts_' . $job_id);
            return;
        }

        $this->reschedule_job_poll((string) $job_id);
    }

    public function enqueue_gutenberg_rankbot_sidebar()
    {
        if (!is_admin()) return;
        if (!function_exists('get_current_screen')) return;

        $screen = get_current_screen();
        if (!$screen) return;

        // Only in the post editor (block editor context).
        if (!isset($screen->base) || $screen->base !== 'post') return;
        if (!isset($screen->post_type) || !is_string($screen->post_type) || $screen->post_type === '') return;

        // Keep consistent with other editor controls.
        $allowed_types = ['post', 'page', 'product'];
        if (!in_array($screen->post_type, $allowed_types, true)) return;

        $post_id = 0;
        $post_id_raw = filter_input(INPUT_GET, 'post', FILTER_VALIDATE_INT);
        if ($post_id_raw) {
            $post_id = (int) $post_id_raw;
        }
        $initialScore = $post_id ? get_post_meta($post_id, '_rankbot_seo_score', true) : '';

        wp_enqueue_script(
            'rankbot-gutenberg-sidebar',
            RANKBOT_PLUGIN_URL . 'assets/js/rankbot-gutenberg-sidebar.js',
            ['wp-plugins', 'wp-edit-post', 'wp-element', 'wp-components', 'wp-data', 'wp-i18n'],
            defined('RANKBOT_VERSION') ? RANKBOT_VERSION : '1.0.0',
            true
        );

        wp_localize_script('rankbot-gutenberg-sidebar', 'RankBotGutenberg', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('rankbot_optimize'),
            'postId' => $post_id,
            'initialScore' => $initialScore,
            'postType' => $screen->post_type,
        ]);
    }

    public function handle_ajax_dismiss_job() {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        $job_id = isset($_POST['job_id']) ? sanitize_text_field(wp_unslash($_POST['job_id'])) : '';
        if ($job_id === '') {
            wp_send_json_error('Missing job_id');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'rankbot_jobs';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->delete($table_name, ['job_id' => $job_id]);
        wp_send_json_success();
    }

    public function handle_ajax_get_plans() {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        
        $plans = $this->api->get_plans();
        
        if (empty($plans)) {
            wp_send_json_error('Failed to fetch plans');
        }
        
        wp_send_json_success(['plans' => $plans]);
    }

    public function render_category_edit_fields($term) {
        $nonce = wp_create_nonce('rankbot_optimize');
        ?>
        <!-- Hidden container originally, moved by JS -->
        <tr class="form-field rankbot-original-row" style="display:none;">
            <th scope="row" valign="top"><label>RankBotAI Optimization</label></th>
            <td>
                <div id="rankbot-category-wrap" style="background:#f8fafc; border:1px solid #cbd5e1; padding:15px; border-radius:6px; margin-top:8px; max-width: 800px;">
                    <div style="display:flex; align-items:center; gap:12px; margin-bottom:10px;">
                        <button type="button" class="button button-primary rankbot-optimize-term-btn" data-action="category_full" style="display:flex; align-items:center; gap:6px; padding:6px 16px; font-size:14px; background:#4f46e5; border:none; height:auto;">
                            <span class="dashicons dashicons-superhero" style="color:white; font-size:18px; width:18px; height:18px;"></span> 
                            <span style="color:white; font-weight:600;">Run AI Optimization</span>
                        </button>
                        <span class="spinner" style="float:none; margin:0;"></span>
                    </div>
                    <p class="description" style="font-size:13px; color:#64748b; margin:0;">
                        Auto-generate description, SEO title, meta description, and update slug based on keyword.
                    </p>
                </div>
                <!-- Status/Log Area -->
                <div id="rankbot-status-log" style="margin-top:5px; color:#4f46e5; font-weight:500; font-size:13px; display:none;"></div>
                <?php
                wp_enqueue_script(
                    'rankbot-category-term',
                    RANKBOT_PLUGIN_URL . 'assets/js/rankbot-category-term.js',
                    array( 'jquery' ),
                    defined( 'RANKBOT_VERSION' ) ? RANKBOT_VERSION : '1.0.0',
                    true
                );
                wp_localize_script( 'rankbot-category-term', 'rankbotTermData', array(
                    'termId'   => (int) $term->term_id,
                    'taxonomy' => esc_js( (string) $term->taxonomy ),
                    'termLink' => esc_js( get_term_link( $term ) ),
                    'nonce'    => wp_create_nonce( 'rankbot_optimize' ),
                ) );
                ?>
            </td>
        </tr>
        <?php
    }

    public function handle_ajax_optimize_term() {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        
        $term_id = isset($_POST['term_id']) ? intval(wp_unslash($_POST['term_id'])) : 0;
        $taxonomy = isset($_POST['taxonomy']) ? sanitize_text_field(wp_unslash($_POST['taxonomy'])) : '';
        if (!$term_id || $taxonomy === '') {
            wp_send_json_error('Missing term_id or taxonomy');
        }
        
        $term = get_term($term_id, $taxonomy);
        if (!$term || is_wp_error($term)) wp_send_json_error('Term not found');
        
        // Context Gathering: Get products in this category
        $products = [];
        $args = [
            'post_type' => ($taxonomy === 'product_cat') ? 'product' : 'post',
            'posts_per_page' => 5, // Get 5 sample products
            'tax_query' => [
                [
                    'taxonomy' => $taxonomy,
                    'field' => 'term_id',
                    'terms' => $term_id,
                ]
            ],
            'orderby' => 'rand', // Random sample valid for context
            'post_status' => 'publish'
        ];
        
        $query = new WP_Query($args);
        while($query->have_posts()) {
            $query->the_post();
            // Get more robust details if possible
            $excerpt = mb_substr(wp_strip_all_tags(get_the_excerpt() ?: get_the_content()), 0, 150);
            $products[] = [
                'name' => get_the_title(),
                'excerpt' => $excerpt ?: 'No description'
            ];
        }
        wp_reset_postdata();
        
        // --- NEW: Shop/Site Context ---
        $site_name = get_bloginfo('name');
        $site_desc = get_bloginfo('description');
        
        $params = [
            'category_name' => $term->name,
            'current_description' => wp_strip_all_tags($term->description),
            'url' => isset($_POST['term_link']) ? esc_url_raw(wp_unslash($_POST['term_link'])) : '',
            'site_name' => $site_name,
            'site_tagline' => $site_desc,
            'products_sample' => $products,
            'products_count' => count($products),
            'taxonomy' => $taxonomy,
            'language' => get_option('rankbot_language', 'auto'),
            'type' => 'category_optimization' // Instruction for AI model
        ];
        
        // Fallback strategy in prompt logic (server-side mostly, but we pass data)
        // If products are empty, we rely on category_name and site context.
        
        // Call API
        $result = $this->api->generate('category_optimize', $params);
        
        if (isset($result['error'])) wp_send_json_error($result['error']);

        // Handle Async Job Response
        if (isset($result['status']) && $result['status'] === 'queued') {
            // Create Job Record for Persistence
            global $wpdb;
            $table_name = $wpdb->prefix . 'rankbot_jobs';
            $site_key_hash = $this->rankbot_site_key_hash();
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->insert(
                $table_name,
                [
                    'job_id' => $result['job_id'],
                    'site_key_hash' => $site_key_hash,
                    'object_id' => $term_id,
                    'object_type' => 'term', // Assuming 'term' for categories
                    'action' => 'category_optimize',
                    'status' => 'queued',
                    'created_at' => current_time('mysql')
                ]
            );

            $this->schedule_job_poll((string) $result['job_id']);

            wp_send_json_success([
                'status' => 'queued',
                'job_id' => $result['job_id'],
                'message' => 'Optimization started in background...'
            ]);
        }
        
        // Expected Data: description, seo_title, seo_description, keyword
        $data = $result['data'] ?? [];
        
        if (empty($data)) {
             wp_send_json_error('AI returned empty result.');
        }

        // Update Description
        if (!empty($data['description'])) {
            wp_update_term($term_id, $taxonomy, ['description' => $data['description']]);
        }
        
        // Update SEO Meta
        $seo_meta = [];
        if (!empty($data['seo_title'])) $seo_meta['title'] = $data['seo_title'];
        if (!empty($data['seo_description'])) $seo_meta['description'] = $data['seo_description'];
        if (!empty($data['keyword'])) $seo_meta['keyword'] = $data['keyword'];
        
        if (!empty($seo_meta)) {
            RankBot_SEO::update_term_meta($term_id, $seo_meta);
        }
        
        // Save History
        $history = get_term_meta($term_id, '_rankbot_history', true) ?: [];
        $history[] = [
            'date' => current_time('mysql'),
            'action' => 'Category Optimization',
            'cost' => $result['cost'] ?? 0
        ];
        update_term_meta($term_id, '_rankbot_history', $history);
        
        wp_send_json_success(['description' => $data['description'] ?? '']);
    }

    public function handle_ajax_optimize_post() {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $post_id = isset($_POST['post_id']) ? absint(wp_unslash($_POST['post_id'])) : 0;
        if (!$post_id) {
            wp_send_json_error('Missing post_id');
        }
        $post = get_post($post_id);
        if (!$post) wp_send_json_error('Post not found');

        // Single and Bulk must behave identically: reuse the same params builder.
        $params = $this->rankbot_build_params_for_bulk($post_id, $post, 'post_optimize');

        if (function_exists('rankbot_log')) {
            $kw = isset($params['focus_keyword']) ? (string) $params['focus_keyword'] : '';
            rankbot_log('AJAX optimize_post: built params', [
                'post_id' => (int) $post_id,
                'post_type' => (string) ($post->post_type ?? ''),
                'allow_content_update' => isset($params['allow_content_update']) ? (string) $params['allow_content_update'] : null,
                'skip_content' => isset($params['skip_content']) ? (int) $params['skip_content'] : null,
                'focus_keyword_len' => strlen($kw),
                'focus_keyword_preview' => function_exists('mb_substr') ? mb_substr($kw, 0, 120) : substr($kw, 0, 120),
                'title_len' => isset($params['title']) ? strlen((string) $params['title']) : 0,
                'content_len' => isset($params['content']) ? strlen((string) $params['content']) : 0,
                'images_count' => (isset($params['images']) && is_array($params['images'])) ? count($params['images']) : 0,
                'internal_links_count' => (isset($params['internal_links']) && is_array($params['internal_links'])) ? count($params['internal_links']) : 0,
                'request_id' => isset($params['request_id']) ? (string) $params['request_id'] : null,
            ]);
        }
        
        $dispatch = $this->rankbot_dispatch_generate_and_register_post_job($post_id, 'post_optimize', $params);

        if (function_exists('rankbot_log')) {
            rankbot_log('AJAX optimize_post: dispatch result', [
                'post_id' => (int) $post_id,
                'has_error' => isset($dispatch['error']),
                'mode' => isset($dispatch['mode']) ? (string) $dispatch['mode'] : null,
                'job_id' => isset($dispatch['job_id']) ? (string) $dispatch['job_id'] : null,
                'error' => isset($dispatch['error']) ? (string) $dispatch['error'] : null,
                'http_code' => isset($dispatch['http_code']) ? (int) $dispatch['http_code'] : null,
                'error_type' => isset($dispatch['error_type']) ? (string) $dispatch['error_type'] : null,
            ]);
        }

        if (isset($dispatch['error'])) {
            wp_send_json_error((string) $dispatch['error']);
        }

        if (isset($dispatch['mode']) && $dispatch['mode'] === 'async') {
            wp_send_json_success([
                'status' => 'queued',
                'job_id' => (string) ($dispatch['job_id'] ?? ''),
                'message' => 'Optimization started in background...'
            ]);
        }

        $result = isset($dispatch['result']) && is_array($dispatch['result']) ? $dispatch['result'] : [];
        if (empty($result)) {
            wp_send_json_error('Empty result');
        }

        $this->process_optimization_result($post_id, $result, 'post_optimize', '');

        wp_send_json_success(['message' => 'Optimization Complete']);
    }

    public function save_seo_score_meta($post_id) {
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!current_user_can('edit_post', $post_id)) return;

        // Verify core post edit nonce if present. We only store meta when the request is a valid post update.
        if (!isset($_POST['_wpnonce'])) return;
        $wpnonce = sanitize_text_field(wp_unslash($_POST['_wpnonce']));
        if (!wp_verify_nonce($wpnonce, 'update-post_' . $post_id)) return;
        
        if (isset($_POST['rankbot_seo_score'])) {
            $score = sanitize_text_field(wp_unslash($_POST['rankbot_seo_score']));
            update_post_meta($post_id, '_rankbot_seo_score', $score);
        }
    }

    public function handle_ajax_save_keyword() {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('edit_posts')) wp_send_json_error('Permission denied');

        $post_id = isset($_POST['post_id']) ? absint(wp_unslash($_POST['post_id'])) : 0;
        $keyword = isset($_POST['keyword']) ? sanitize_text_field(wp_unslash($_POST['keyword'])) : '';
        if (!$post_id) {
            wp_send_json_error('Missing post_id');
        }
        // Backup before changing SEO meta/keyword
        $this->create_backup($post_id, 'save_keyword');
        update_post_meta($post_id, '_rankbot_focus_keyword', $keyword);
        if (class_exists('RankBot_SEO')) {
            RankBot_SEO::update_meta($post_id, ['keyword' => $keyword]);
        }
        wp_send_json_success('Saved');
    }

    public function handle_ajax_generate_unique_keyword() {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        $post_id = isset($_POST['post_id']) ? absint(wp_unslash($_POST['post_id'])) : 0;
        if (!$post_id) {
            wp_send_json_error('Missing post_id');
        }
        $post = get_post($post_id);

        $post_type = $post ? (string) get_post_type($post) : '';
        $page_type = ($post_type === 'product') ? 'product' : 'post';

        $categories = [];
        $tags = [];
        if ($post_id) {
            $cat_tax = ($page_type === 'product') ? 'product_cat' : 'category';
            $tag_tax = ($page_type === 'product') ? 'product_tag' : 'post_tag';

            $cat_terms = wp_get_post_terms($post_id, $cat_tax, ['fields' => 'names']);
            if (!is_wp_error($cat_terms) && is_array($cat_terms)) {
                $categories = $cat_terms;
            }

            $tag_terms = wp_get_post_terms($post_id, $tag_tax, ['fields' => 'names']);
            if (!is_wp_error($tag_terms) && is_array($tag_terms)) {
                $tags = $tag_terms;
            }
        }

        
        // Get all used keywords (Yoast, RankMath, and RankBot)
        global $wpdb;

        $cache_key = 'rankbot_used_keywords_exclude_' . $post_id;
        $used_keywords = wp_cache_get($cache_key, 'rankbotai-seo-optimizer');
        if (false === $used_keywords) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $used_keywords = $wpdb->get_col(
                $wpdb->prepare(
                    "SELECT meta_value FROM {$wpdb->postmeta}
                     WHERE meta_key IN ('_rankbot_focus_keyword', '_yoast_wpseo_focuskw', 'rank_math_focus_keyword')
                     AND post_id != %d",
                    $post_id
                )
            );
            wp_cache_set($cache_key, is_array($used_keywords) ? $used_keywords : [], 'rankbotai-seo-optimizer', 300);
        }
        
        // Limit to 500 to avoid payload issues
        $exclude_list = array_slice($used_keywords, 0, 500);
        
        $title = $post ? (string) $post->post_title : '';
        $desc = $post ? mb_substr(wp_strip_all_tags((string) $post->post_content), 0, 1000) : '';

        $params = [
            // Backward compatible keys used by the API service
            'product_name' => $title,
            'description' => $desc,

            // Uniqueness constraint
            'exclude_keywords' => $exclude_list,

            // Extra context (safe to ignore server-side)
            'page_type' => $page_type,
            'post_type' => $post_type,
            'categories' => $categories,
            'tags' => $tags,

            // Let server auto-detect language by default
            'language' => 'auto',

            // Server builds prompts; plugin only sends context
        ];
        
        // Keyword research can be async. Do NOT busy-wait in PHP (avoids request timeouts);
        // the classic editor UI will poll via rankbot_check_job when queued.
        $result = $this->api->generate('keyword_research', $params);

        if (isset($result['status']) && $result['status'] === 'queued' && !empty($result['job_id'])) {
            wp_send_json_success([
                'status' => 'queued',
                'job_id' => (string) $result['job_id'],
                'message' => 'Keyword generation started...'
            ]);
        }
        
        if (isset($result['error'])) wp_send_json_error($result['error']);
        
        // Prefer server-selected best keyword
        $unique = '';
        if (isset($result['data']['keyword']) && is_string($result['data']['keyword'])) {
            $unique = trim($result['data']['keyword']);
        }

        // Fallback to first item from keywords list
        if (!$unique && isset($result['data']['keywords'])) {
            if (is_array($result['data']['keywords'])) {
                $unique = trim((string)($result['data']['keywords'][0] ?? ''));
            } else {
                $parts = array_map('trim', explode(',', (string)$result['data']['keywords']));
                $unique = trim((string)($parts[0] ?? ''));
            }
        }
        
        if ($unique) {
            // Backup before changing SEO meta/keyword
            $this->create_backup($post_id, 'generate_keyword');

            // Auto save it
            update_post_meta($post_id, '_rankbot_focus_keyword', $unique);
            if (class_exists('RankBot_SEO')) {
                RankBot_SEO::update_meta($post_id, ['keyword' => $unique]);
            }
            wp_send_json_success(['keyword' => $unique]);
        } else {
            // If all taken, fallback to first one but warn? Or just return first.
             $fallback = '';
             if ($fallback) {
                 $this->create_backup($post_id, 'generate_keyword');
                 update_post_meta($post_id, '_rankbot_focus_keyword', $fallback);
                 if (class_exists('RankBot_SEO')) {
                     RankBot_SEO::update_meta($post_id, ['keyword' => $fallback]);
                 }
             }
             wp_send_json_success(['keyword' => $fallback, 'warning' => 'Could not find completely unique keyword.']);
        }
    }

    public function handle_ajax_analyze_seo()
    {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Permission denied');
        }

        $post_id = isset($_POST['post_id']) ? absint(wp_unslash($_POST['post_id'])) : 0;
        if (!$post_id) {
            wp_send_json_error('Missing post_id');
        }

        $kw = $this->rankbot_get_focus_keyword_for_post($post_id);
        if ($kw === '') {
            wp_send_json_success([
                'score' => 0,
                'results' => [],
                'debug' => [
                    'title_source' => 'n/a',
                    'desc_source' => 'n/a',
                    'title_len' => 0,
                    'desc_len' => 0,
                    'title_preview' => '',
                    'desc_preview' => '',
                ],
            ]);
        }

        $analysis = $this->rankbot_build_seo_analysis($post_id, $kw);
        wp_send_json_success($analysis);
    }

    public function handle_ajax_get_backups()
    {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        $post_id = isset($_POST['post_id']) ? absint(wp_unslash($_POST['post_id'])) : 0;
        if (!$post_id) {
            wp_send_json_error('Missing post_id');
        }
        $backups = get_post_meta($post_id, '_rankbot_backups', true) ?: [];
        // Sort newest first
        usort($backups, function($a, $b) {
            return strtotime($b['date']) - strtotime($a['date']);
        });
        wp_send_json_success(['backups' => $backups]);
    }

    public function handle_ajax_restore_backup()
    {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        $post_id = isset($_POST['post_id']) ? absint(wp_unslash($_POST['post_id'])) : 0;
        $backup_timestamp = isset($_POST['timestamp']) ? sanitize_text_field(wp_unslash($_POST['timestamp'])) : '';
        if (!$post_id || $backup_timestamp === '') {
            wp_send_json_error('Missing parameters');
        }

        if (!current_user_can('edit_post', $post_id)) {
            wp_send_json_error('Permission denied');
        }

        // Cancel any pending background job polling for this post so restored content is not overwritten.
        global $wpdb;
        $table_name = $wpdb->prefix . 'rankbot_jobs';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $pending = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT job_id FROM %i WHERE object_id = %d AND object_type = 'post' AND status IN ('queued','processing')",
                $table_name,
                $post_id
            )
        );
        if (is_array($pending) && !empty($pending)) {
            foreach ($pending as $jid) {
                $jid = (string) $jid;
                if (function_exists('wp_clear_scheduled_hook')) {
                    wp_clear_scheduled_hook('rankbot_poll_job', [$jid]);
                }
                delete_transient('rankbot_job_attempts_' . $jid);
            }
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM %i WHERE object_id = %d AND object_type = 'post' AND status IN ('queued','processing')",
                    $table_name,
                    $post_id
                )
            );
        }
        
        $backups = get_post_meta($post_id, '_rankbot_backups', true) ?: [];
        $target = null;
        
        foreach($backups as $b) {
            if (isset($b['timestamp']) && (string) $b['timestamp'] === (string) $backup_timestamp) {
                $target = $b['data'];
                break;
            }
        }
        
        if (!$target) wp_send_json_error('Backup not found');
        
        // Restore Main Post Data
        $update_data = [
            'ID' => $post_id,
            'post_title' => isset($target['post_title']) ? (string) $target['post_title'] : '',
            'post_content' => isset($target['post_content']) ? (string) $target['post_content'] : '',
            'post_excerpt' => isset($target['post_excerpt']) ? (string) $target['post_excerpt'] : ''
        ];

        // Restore slug if present in backup
        if (isset($target['post_name'])) {
            $update_data['post_name'] = (string) $target['post_name'];
        }

        // wp_update_post expects slashed data for fields that may contain HTML.
        $updateRes = wp_update_post(wp_slash($update_data), true);
        if (is_wp_error($updateRes)) {
            wp_send_json_error('Restore failed: ' . $updateRes->get_error_message());
        }

        clean_post_cache($post_id);

        // Restore Meta (and clear known keys that are missing in the backup)
        $known_meta_keys = [
            // Yoast
            '_yoast_wpseo_title',
            '_yoast_wpseo_metadesc',
            '_yoast_wpseo_focuskw',

            // Rank Math
            'rank_math_title',
            'rank_math_description',
            'rank_math_focus_keyword',

            // RankBot internal
            '_rankbot_focus_keyword',
            '_rankbot_seo_score',
            '_rankbot_history'
        ];

        $backup_meta = [];
        if (isset($target['meta']) && is_array($target['meta'])) {
            $backup_meta = $target['meta'];
        }

        foreach ($known_meta_keys as $key) {
            if (array_key_exists($key, $backup_meta)) {
                update_post_meta($post_id, $key, $backup_meta[$key]);
            } else {
                delete_post_meta($post_id, $key);
            }
        }
        
        // Restore Alt Tags
        if (isset($target['image_alts']) && is_array($target['image_alts'])) {
             foreach($target['image_alts'] as $img_id => $alt) {
                  $img_id = absint($img_id);
                  if ($img_id < 1) continue;

                  $alt = is_string($alt) ? $alt : '';
                  $alt = sanitize_text_field($alt);
                  if ($alt === '') {
                      delete_post_meta($img_id, '_wp_attachment_image_alt');
                  } else {
                      update_post_meta($img_id, '_wp_attachment_image_alt', $alt);
                  }
             }
        }

        // Guard: if a job completes later, do not apply it over restored state.
        update_post_meta($post_id, '_rankbot_restored_at', time());

        wp_send_json_success(['message' => 'Restored successfully!']);
    }

    public function handle_ajax_check_job()
    {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        $job_id = isset($_POST['job_id']) ? sanitize_text_field(wp_unslash($_POST['job_id'])) : '';
        $post_id = isset($_POST['post_id']) ? absint(wp_unslash($_POST['post_id'])) : 0;
        $term_id = isset($_POST['term_id']) ? absint(wp_unslash($_POST['term_id'])) : 0;
        $action_type = isset($_POST['rankbot_action_type']) ? sanitize_text_field(wp_unslash($_POST['rankbot_action_type'])) : '';

        if (function_exists('rankbot_log')) {
            rankbot_log('AJAX check_job start', [
                'job_id' => (string) $job_id,
                'post_id' => (int) $post_id,
                'term_id' => (int) $term_id,
                'action_type' => (string) $action_type,
            ]);
        }

        if ($job_id === '') {
            wp_send_json_success([
                'status' => 'failed',
                'error' => 'Missing job_id',
            ]);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'rankbot_jobs';
        $site_key_hash = $this->rankbot_site_key_hash();
        $job_row = null;

        // Resolve local job mapping to enforce object_id + site_key_hash binding.
        if ($this->rankbot_db_table_exists($table_name)) {
            if ($site_key_hash !== '') {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $job_row = $wpdb->get_row(
                    $wpdb->prepare(
                        "SELECT * FROM %i WHERE job_id = %s AND (site_key_hash = %s OR site_key_hash = '') LIMIT 1",
                        $table_name,
                        $job_id,
                        $site_key_hash
                    ),
                    ARRAY_A
                );
            } else {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $job_row = $wpdb->get_row(
                    $wpdb->prepare(
                        "SELECT * FROM %i WHERE job_id = %s LIMIT 1",
                        $table_name,
                        $job_id
                    ),
                    ARRAY_A
                );
            }
        }

        if (is_array($job_row)) {
            $row_object_id = isset($job_row['object_id']) ? absint($job_row['object_id']) : 0;
            $row_object_type = isset($job_row['object_type']) ? (string) $job_row['object_type'] : 'post';

            if ($row_object_type === 'term') {
                if ($row_object_id > 0) {
                    $term_id = $row_object_id;
                    $post_id = 0;
                }
            } else {
                if ($row_object_id > 0) {
                    $post_id = $row_object_id;
                    $term_id = 0;
                }
            }

            if ($action_type === '' && !empty($job_row['action'])) {
                $action_type = (string) $job_row['action'];
            }
        } elseif ($post_id === 0 && $term_id === 0) {
            wp_send_json_success([
                'status' => 'failed',
                'error' => 'Job not found locally',
            ]);
        }

        $result = $this->api->check_job($job_id);

        if (function_exists('rankbot_log')) {
            rankbot_log('AJAX check_job api returned', [
                'job_id' => (string) $job_id,
                'has_error' => isset($result['error']),
                'status' => isset($result['status']) ? (string) $result['status'] : null,
                'keys' => is_array($result) ? array_values(array_slice(array_keys($result), 0, 30)) : [],
            ]);
        }

        if (isset($result['error'])) {
            wp_send_json_success([
                'status' => 'failed',
                'error' => (string) $result['error'],
            ]);
        }

        $remote_status = isset($result['status']) ? (string) $result['status'] : '';

        $has_duration_cols = false;
        try {
            if ($this->rankbot_db_table_exists($table_name)) {
                $has_started = $this->rankbot_db_column_exists($table_name, 'started_at');
                $has_completed = $this->rankbot_db_column_exists($table_name, 'completed_at');
                $has_duration = $this->rankbot_db_column_exists($table_name, 'duration_sec');
                $has_duration_cols = !empty($has_started) && !empty($has_completed) && !empty($has_duration);
            }
        } catch (\Exception $e) {
            $has_duration_cols = false;
        }

        // Keep local status in sync when AJAX polling detects processing.
        if ($remote_status === 'processing' && is_array($job_row)) {
            $local_status = isset($job_row['status']) ? (string) $job_row['status'] : '';
            if ($local_status === 'queued' || $local_status === 'pending') {
                if ($has_duration_cols) {
                    $now = current_time('mysql');
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->query(
                        $wpdb->prepare(
                            "UPDATE %i SET status = 'processing', started_at = COALESCE(started_at, %s) WHERE job_id = %s AND status IN ('queued','pending')",
                            $table_name,
                            $now,
                            $job_id
                        )
                    );
                } else {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->update($table_name, ['status' => 'processing'], ['job_id' => $job_id]);
                }
            }
        }

        if ($remote_status === 'completed' && isset($result['result'])) {
            // Apply exactly once (AJAX and WP-Cron can race).
            // Do NOT rely solely on custom table rows being present: some installs may not have the table,
            // or the job row might be missing, which would previously skip applying changes.
            $safe_job_id = preg_replace('/[^a-zA-Z0-9_\-]/', '', (string) $job_id);
            if ($safe_job_id === '') {
                $safe_job_id = sha1((string) $job_id);
            }

            if ($post_id > 0) {
                $claim_key = '_rankbot_applied_job_' . $safe_job_id;
                // add_post_meta is atomic for the same key+post; if it already exists, we already applied.
                if (!add_post_meta((int) $post_id, (string) $claim_key, (string) time(), true)) {
                    wp_send_json_success($result);
                }
            } elseif ($term_id > 0) {
                $claim_key = '_rankbot_applied_job_' . $safe_job_id;
                if (!add_term_meta((int) $term_id, (string) $claim_key, (string) time(), true)) {
                    wp_send_json_success($result);
                }
            }

            // MARK AS COMPLETED in local table when available (best-effort).
            $claimed = 1;
            if ($this->rankbot_db_table_exists($table_name)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $claimed = $wpdb->query(
                    $wpdb->prepare(
                        "UPDATE %i SET status = 'completed' WHERE job_id = %s AND status IN ('queued','processing','pending')",
                        $table_name,
                        $job_id
                    )
                );
            }

            if ($has_duration_cols) {
                $now = current_time('mysql');
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->query(
                    $wpdb->prepare(
                        "UPDATE %i SET completed_at = %s, duration_sec = TIMESTAMPDIFF(SECOND, COALESCE(started_at, created_at), %s) WHERE job_id = %s",
                        $table_name,
                        $now,
                        $now,
                        $job_id
                    )
                );
            }

            if (function_exists('rankbot_log')) {
                rankbot_log('AJAX check_job completed: db updated', [
                    'job_id' => (string) $job_id,
                    'post_id' => (int) $post_id,
                    'term_id' => (int) $term_id,
                    'db_last_error' => (string) $wpdb->last_error,
                ]);
            }

            if ($post_id) {
                wp_cache_delete('rankbot_latest_job_' . $post_id, 'rankbotai-seo-optimizer');
            }

            // If user restored a backup after this job was queued, do NOT apply results.
            if ($post_id > 0) {
                $restored_at = (int) get_post_meta($post_id, '_rankbot_restored_at', true);
                if ($restored_at > 0) {
                    // Fetch local job created_at for comparison.
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $created_at = $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT created_at FROM %i WHERE job_id = %s LIMIT 1",
                            $table_name,
                            $job_id
                        )
                    );
                    $created_ts = is_string($created_at) ? (int) strtotime($created_at) : 0;
                    if ($created_ts > 0 && $restored_at > $created_ts) {
                        if (function_exists('rankbot_log')) {
                            rankbot_log('AJAX check_job: skip apply due to restore guard', [
                                'job_id' => (string) $job_id,
                                'post_id' => (int) $post_id,
                                'restored_at' => (int) $restored_at,
                                'job_created_at' => (string) $created_at,
                            ]);
                        }
                        wp_send_json_success($result);
                    }
                }
            }

            if ($term_id > 0) {
                // Cleanup Term Meta (if any was used for persistence, currently using table for new flow)
                $this->process_term_optimization_result($term_id, $result['result']);
            } else {
                if (function_exists('rankbot_log')) {
                    $rr = $result['result'];
                    $data = (is_array($rr) && isset($rr['data']) && is_array($rr['data'])) ? $rr['data'] : [];
                    $kw = '';
                    if (isset($data['focus_keyword']) && is_string($data['focus_keyword'])) $kw = $data['focus_keyword'];
                    elseif (isset($data['keyword']) && is_string($data['keyword'])) $kw = $data['keyword'];

                    rankbot_log('AJAX check_job completed: result summary', [
                        'job_id' => (string) $job_id,
                        'post_id' => (int) $post_id,
                        'action' => (string) $action_type,
                        'result_keys' => is_array($rr) ? array_values(array_slice(array_keys($rr), 0, 40)) : [],
                        'data_keys' => is_array($data) ? array_values(array_slice(array_keys($data), 0, 80)) : [],
                        'keyword_len' => strlen((string) $kw),
                        'keyword_preview' => function_exists('mb_substr') ? mb_substr((string) $kw, 0, 120) : substr((string) $kw, 0, 120),
                        'has_seo_title' => isset($data['seo_title']) || isset($data['meta_title']),
                        'has_seo_description' => isset($data['seo_description']) || isset($data['meta_description']),
                        'has_content' => isset($data['content']) || isset($data['description']),
                        'content_len' => isset($data['content']) ? strlen((string) $data['content']) : (isset($data['description']) ? strlen((string) $data['description']) : 0),
                    ]);
                }
                $this->process_optimization_result($post_id, $result['result'], $action_type, (string) $job_id);
            }
            
            wp_send_json_success($result);
        } elseif ($remote_status === 'failed') {
             // MARK AS FAILED
             if ($has_duration_cols) {
                 $now = current_time('mysql');
                 // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                 $wpdb->query(
                     $wpdb->prepare(
                         "UPDATE %i SET status = 'failed', completed_at = %s, duration_sec = TIMESTAMPDIFF(SECOND, COALESCE(started_at, created_at), %s) WHERE job_id = %s",
                         $table_name,
                         $now,
                         $now,
                         $job_id
                     )
                 );
             } else {
                 // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                 $wpdb->update(
                    $table_name,
                    ['status' => 'failed'],
                    ['job_id' => $job_id]
                );
             }

             if (function_exists('rankbot_log')) {
                 rankbot_log('AJAX check_job failed: db updated', [
                     'job_id' => (string) $job_id,
                     'post_id' => (int) $post_id,
                     'term_id' => (int) $term_id,
                     'db_last_error' => (string) $wpdb->last_error,
                     'error' => isset($result['error']) ? (string) $result['error'] : null,
                 ], 'error');
             }

             if ($post_id) {
                 wp_cache_delete('rankbot_latest_job_' . $post_id, 'rankbotai-seo-optimizer');
             }
             
             wp_send_json_success($result);
        } else {
            // queued / processing
            wp_send_json_success($result);
        }
    }

    private function apply_term_optimization_result($term_id, $result): array
    {
        $term = get_term($term_id);
        if (!$term || is_wp_error($term)) return ['error' => 'Term not found'];

        if (function_exists('rankbot_log')) {
            rankbot_log('Apply term optimization start', [
                'term_id' => (int) $term_id,
                'taxonomy' => is_object($term) ? (string) $term->taxonomy : '',
                'result_keys' => is_array($result) ? array_values(array_slice(array_keys($result), 0, 40)) : [],
                'data_keys' => (is_array($result) && isset($result['data']) && is_array($result['data']))
                    ? array_values(array_slice(array_keys($result['data']), 0, 40))
                    : [],
            ]);
        }

        $data = $result['data'] ?? [];
        if (empty($data)) return ['error' => 'Empty result data'];

        // Update Description
        if (!empty($data['description'])) {
            $clean_desc = $this->rankbot_sanitize_generated_html((string) $data['description']);
            $termRes = wp_update_term($term_id, $term->taxonomy, ['description' => $clean_desc]);
            if (function_exists('rankbot_log')) {
                rankbot_log('Apply term optimization: wp_update_term(description)', [
                    'term_id' => (int) $term_id,
                    'ok' => !is_wp_error($termRes),
                    'error' => is_wp_error($termRes) ? $termRes->get_error_message() : null,
                ], is_wp_error($termRes) ? 'error' : 'info');
            }
        }

        // Update SEO Meta
        $seo_meta = [];
        if (!empty($data['seo_title'])) $seo_meta['title'] = $data['seo_title'];
        if (!empty($data['seo_description'])) $seo_meta['description'] = $data['seo_description'];
        if (!empty($data['keyword'])) $seo_meta['keyword'] = $data['keyword'];

        // Slug Update Logic
        // Option key stored in admin page is 'rankbot_update_slug'
        $allow_slug_update = get_option('rankbot_update_slug', 'no'); 
        
        if ($allow_slug_update === 'yes') {
             // 1. Try explicit slug if AI returned it (not currently prompted but future proof)
             $new_slug = '';
             if (!empty($data['slug'])) {
                 $new_slug = $data['slug'];
             } 
             // 2. Fallback to keyword if available
             elseif (!empty($data['keyword'])) {
                 $new_slug = $data['keyword'];
             }

             if ($new_slug) {
                 // Ensure transliteration if native slugify for DB does not handle cyrillic well
                 
                 // Manual translit map for Cyrillic (common standard)
                 $cyr = [
                     'а','б','в','г','д','е','ё','ж','з','и','й','к','л','м','н','о','п',
                     'р','с','т','у','ф','х','ц','ч','ш','щ','ъ','ы','ь','э','ю','я',
                     'А','Б','В','Г','Д','Е','Ё','Ж','З','И','Й','К','Л','М','Н','О','П',
                     'Р','С','Т','У','Ф','Х','Ц','Ч','Ш','Щ','Ъ','Ы','Ь','Э','Ю','Я'
                 ];
                 $lat = [
                     'a','b','v','g','d','e','io','zh','z','i','y','k','l','m','n','o','p',
                     'r','s','t','u','f','h','ts','ch','sh','sht','a','i','y','e','yu','ya',
                     'A','B','V','G','D','E','Io','Zh','Z','I','Y','K','L','M','N','O','P',
                     'R','S','T','U','F','H','Ts','Ch','Sh','Sht','A','I','Y','E','Yu','Ya'
                 ];
                 
                 // Pre-process for transliteration if Cyrillic characters exist
                 if (preg_match('/[А-Яа-яЁё]/u', $new_slug)) {
                     $new_slug = str_replace($cyr, $lat, $new_slug);
                 }

                 $slugRes = wp_update_term($term_id, $term->taxonomy, ['slug' => sanitize_title($new_slug)]);
                 if (function_exists('rankbot_log')) {
                     rankbot_log('Apply term optimization: wp_update_term(slug)', [
                         'term_id' => (int) $term_id,
                         'ok' => !is_wp_error($slugRes),
                         'error' => is_wp_error($slugRes) ? $slugRes->get_error_message() : null,
                     ], is_wp_error($slugRes) ? 'error' : 'info');
                 }
             }
        }

        if (!empty($seo_meta)) {
            RankBot_SEO::update_term_meta($term_id, $seo_meta);
        }

        // Save History
        $history = get_term_meta($term_id, '_rankbot_history', true) ?: [];
        $history[] = [
            'date' => current_time('mysql'),
            'action' => 'Category Optimization',
            'cost' => $result['cost'] ?? 0
        ];
        update_term_meta($term_id, '_rankbot_history', $history);

        return [
            'status' => 'completed',
            'description' => $data['description'] ?? '',
            'meta' => $seo_meta
        ];
    }

    private function process_term_optimization_result($term_id, $result)
    {
        $out = $this->apply_term_optimization_result($term_id, $result);
        if (isset($out['error'])) {
            wp_send_json_error($out['error']);
        }
        wp_send_json_success($out);
    }

    public function handle_ajax_optimize()
    {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        
        $post_id = isset($_POST['post_id']) ? intval(wp_unslash($_POST['post_id'])) : 0;
        $requested_action = isset($_POST['rankbot_action_type']) ? sanitize_text_field(wp_unslash($_POST['rankbot_action_type'])) : ''; // 'action' is reserved in wp_ajax
        if (!$post_id) {
            wp_send_json_error('Missing post_id');
        }
        
        $post = get_post($post_id);
        if (!$post) wp_send_json_error('Post not found');

        $resolved_action = $this->rankbot_resolve_action((string) $post->post_type, (string) $requested_action);
        if ($resolved_action === '') {
            wp_send_json_error('Unsupported action for this post type');
        }

        // Single and Bulk must behave identically: reuse the same params builder.
        $params = $this->rankbot_build_params_for_bulk($post_id, $post, (string) $resolved_action);

        $dispatch = $this->rankbot_dispatch_generate_and_register_post_job($post_id, (string) $resolved_action, $params);
        if (isset($dispatch['error'])) {
            wp_send_json_error((string) $dispatch['error']);
        }

        if (isset($dispatch['mode']) && $dispatch['mode'] === 'async') {
            wp_send_json_success([
                'status' => 'queued',
                'job_id' => (string) ($dispatch['job_id'] ?? ''),
                'message' => 'Processing in background...'
            ]);
            wp_die();
        }

        $result = isset($dispatch['result']) && is_array($dispatch['result']) ? $dispatch['result'] : [];
        $this->process_optimization_result($post_id, $result, (string) $resolved_action, '');
    }

    private function apply_optimization_result($post_id, $result, $action, string $job_id = ''): array
    {
        $post = get_post($post_id);

        if (!$post) {
            if (function_exists('rankbot_log')) {
                rankbot_log('Apply optimization: post not found', [
                    'post_id' => (int) $post_id,
                    'action' => is_string($action) ? $action : '',
                ], 'error');
            }
            return ['error' => 'Post not found'];
        }

        if ($job_id !== '') {
            $applied_jobs = get_post_meta($post_id, '_rankbot_applied_jobs', true);
            if (!is_array($applied_jobs)) {
                $applied_jobs = [];
            }
            if (in_array($job_id, $applied_jobs, true)) {
                if (function_exists('rankbot_log')) {
                    rankbot_log('Apply optimization skipped: job already applied', [
                        'post_id' => (int) $post_id,
                        'job_id' => (string) $job_id,
                        'action' => is_string($action) ? $action : '',
                    ]);
                }
                return [
                    'message' => 'Already applied',
                    'already_applied' => true,
                    'last_run' => current_time('mysql')
                ];
            }
        }

        if (function_exists('rankbot_log')) {
            rankbot_log('Apply optimization start', [
                'post_id' => (int) $post_id,
                'post_type' => (string) ($post->post_type ?? ''),
                'action' => is_string($action) ? $action : '',
                'result_keys' => is_array($result) ? array_values(array_slice(array_keys($result), 0, 40)) : [],
                'data_keys' => (is_array($result) && isset($result['data']) && is_array($result['data']))
                    ? array_values(array_slice(array_keys($result['data']), 0, 60))
                    : [],
            ]);
        }

        $original_content_html = $post && isset($post->post_content) ? (string) $post->post_content : '';

        $focus_keyword = '';

        // CREATE/RESOLVE BACKUP
        // Prefer the backup that was created at queue time for this job.
        $backup_ts = 0;
        if ($job_id !== '') {
            $backup_ts = (int) get_transient('rankbot_job_backup_ts_' . $job_id);
        }
        if ($backup_ts <= 0) {
            $backup_ts = $this->create_backup($post_id, is_string($action) ? $action : 'optimize');
        }

        // Apply SEO Meta
        $meta_data = [];
        // IMPORTANT: RankMath/Yoast focus keyword must be a SINGLE phrase.
        // Do not save comma-separated keyword lists into focus keyword field.
        $keywordCandidate = '';
        if (isset($result['data']['focus_keyword']) && is_string($result['data']['focus_keyword'])) {
            $keywordCandidate = (string) $result['data']['focus_keyword'];
        } elseif (isset($result['data']['keyword']) && is_string($result['data']['keyword'])) {
            $keywordCandidate = (string) $result['data']['keyword'];
        }
        $keywordCandidate = trim($keywordCandidate);
        if ($keywordCandidate !== '') {
            $focus_keyword = $keywordCandidate;
        }

        // If the API didn't return a keyword (or returned empty), keep using the stored focus keyword.
        if ($focus_keyword === '') {
            $focus_keyword = trim((string) $this->rankbot_get_focus_keyword_for_post((int) $post_id));
        }

        // Only update SEO plugin focus keyword when we actually have one.
        if ($focus_keyword !== '') {
            $meta_data['keyword'] = $focus_keyword;
        }
        
        // Handle various API response keys
        if(isset($result['data']['seo_title'])) $meta_data['title'] = $result['data']['seo_title'];
        if(isset($result['data']['seo_description'])) $meta_data['description'] = $result['data']['seo_description'];
        
        // Handle snippet direct result (legacy)
        if(isset($result['data']['title'])) $meta_data['title'] = $result['data']['title'];
        if(isset($result['data']['description'])) $meta_data['description'] = $result['data']['description'];

        // Handle strict JSON keys (current standard)
        if(isset($result['data']['meta_title'])) $meta_data['title'] = $result['data']['meta_title'];
        if(isset($result['data']['meta_description'])) $meta_data['description'] = $result['data']['meta_description'];

        // Normalize casing of direct keyphrase usage (quality)
        if (!empty($focus_keyword)) {
            $kw_title = $this->rankbot_mb_title_case($focus_keyword);
            $kw_sentence = $this->rankbot_mb_ucfirst($focus_keyword);

            if (isset($meta_data['title'])) {
                $meta_data['title'] = $this->rankbot_replace_phrase_ci((string) $meta_data['title'], $focus_keyword, $kw_title);
            }
            if (isset($meta_data['description'])) {
                $meta_data['description'] = $this->rankbot_replace_phrase_ci((string) $meta_data['description'], $focus_keyword, $kw_sentence);
            }
        }

        if (!empty($meta_data)) {
            RankBot_SEO::update_meta($post_id, $meta_data);
        }

        if (function_exists('rankbot_log')) {
            rankbot_log('Apply optimization: resolved keyword/meta', [
                'post_id' => (int) $post_id,
                'action' => is_string($action) ? $action : '',
                'keyword_len' => strlen((string) $focus_keyword),
                'keyword_preview' => function_exists('mb_substr') ? mb_substr((string) $focus_keyword, 0, 120) : substr((string) $focus_keyword, 0, 120),
                'meta_title_len' => isset($meta_data['title']) ? strlen((string) $meta_data['title']) : 0,
                'meta_desc_len' => isset($meta_data['description']) ? strlen((string) $meta_data['description']) : 0,
                'meta_keys' => array_values(array_keys($meta_data)),
                'result_data_keys' => (is_array($result) && isset($result['data']) && is_array($result['data']))
                    ? array_values(array_slice(array_keys($result['data']), 0, 60))
                    : [],
            ]);
        }

        // Apply Content Updates (for Complex Optimization)
        $post_update = ['ID' => $post_id];
        $needs_update = false;

        $allow_title_update = get_option('rankbot_update_product_title', 'no');
        $allow_slug_update = get_option('rankbot_update_slug', 'no');
        $allow_main_content_update = get_option('rankbot_update_main_content', 'yes');

        if (function_exists('rankbot_log')) {
            rankbot_log('Apply optimization options', [
                'post_id' => (int) $post_id,
                'allow_title_update' => (string) $allow_title_update,
                'allow_slug_update' => (string) $allow_slug_update,
                'allow_main_content_update' => (string) $allow_main_content_update,
            ]);
        }

        // Update Title if allowed and provided.
        // Products (complex) usually return product_name.
        // Posts (post_optimize) commonly return seo_title/meta_title (and may return title).
        if ($allow_title_update === 'yes') {
            $titleCandidate = '';
            if (isset($result['data']['product_name']) && is_string($result['data']['product_name'])) {
                $titleCandidate = (string) $result['data']['product_name'];
            } elseif (isset($result['data']['title']) && is_string($result['data']['title'])) {
                $titleCandidate = (string) $result['data']['title'];
            } elseif (isset($result['data']['post_title']) && is_string($result['data']['post_title'])) {
                $titleCandidate = (string) $result['data']['post_title'];
            } elseif (isset($result['data']['seo_title']) && is_string($result['data']['seo_title'])) {
                $titleCandidate = (string) $result['data']['seo_title'];
            } elseif (isset($result['data']['meta_title']) && is_string($result['data']['meta_title'])) {
                $titleCandidate = (string) $result['data']['meta_title'];
            }

            $titleCandidate = trim((string) $titleCandidate);
            if ($titleCandidate !== '') {
                if (!empty($focus_keyword)) {
                    $titleCandidate = $this->rankbot_replace_phrase_ci(
                        $titleCandidate,
                        $focus_keyword,
                        $this->rankbot_mb_title_case($focus_keyword)
                    );
                }
                $post_update['post_title'] = $titleCandidate;
                $needs_update = true;
            }
        }

        // Update Slug if allowed
        if ($allow_slug_update === 'yes') {
            $cyr = [
                 'а','б','в','г','д','е','ё','ж','з','и','й','к','л','м','н','о','п',
                 'р','с','т','у','ф','х','ц','ч','ш','щ','ъ','ы','ь','э','ю','я',
                 'А','Б','В','Г','Д','Е','Ё','Ж','З','И','Й','К','Л','М','Н','О','П',
                 'Р','С','Т','У','Ф','Х','Ц','Ч','Ш','Щ','Ъ','Ы','Ь','Э','Ю','Я'
            ];
            $lat = [
                 'a','b','v','g','d','e','io','zh','z','i','y','k','l','m','n','o','p',
                 'r','s','t','u','f','h','ts','ch','sh','sht','a','i','y','e','yu','ya',
                 'A','B','V','G','D','E','Io','Zh','Z','I','Y','K','L','M','N','O','P',
                 'R','S','T','U','F','H','Ts','Ch','Sh','Sht','A','I','Y','E','Yu','Ya'
            ];

            // Prioritize explicit slug from API (AI generated)
            if (isset($result['data']['slug']) && !empty($result['data']['slug'])) {
                $new_slug = $result['data']['slug'];
                // Translit if needed
                if (preg_match('/[А-Яа-яЁё]/u', $new_slug)) {
                     $new_slug = str_replace($cyr, $lat, $new_slug);
                }
                $post_update['post_name'] = sanitize_title($new_slug);
                $needs_update = true;
            } 
            // Fallback: if we updated the title, let's update slug to match the new title
            elseif ($needs_update && isset($post_update['post_title'])) {
                $new_slug = $post_update['post_title'];
                // Translit if needed
                if (preg_match('/[А-Яа-яЁё]/u', $new_slug)) {
                     $new_slug = str_replace($cyr, $lat, $new_slug);
                }
                $post_update['post_name'] = sanitize_title($new_slug);
            }
        }

        if ($allow_main_content_update === 'yes' && isset($result['data']['description'])) {
            $new_content = $this->rankbot_sanitize_generated_html((string) $result['data']['description']);
            // Safety: never overwrite existing content with an empty string.
            if (trim(wp_strip_all_tags((string) $new_content)) !== '') {
                $post_update['post_content'] = $new_content;
                $needs_update = true;
            }
        }
        // Handle Post Content (Articles)
        if ($allow_main_content_update === 'yes' && isset($result['data']['content'])) {
            $new_content = $this->rankbot_sanitize_generated_html((string) $result['data']['content']);
            if (trim(wp_strip_all_tags((string) $new_content)) !== '') {
                $post_update['post_content'] = $new_content;
                $needs_update = true;
            }
        }

        if (function_exists('rankbot_log')) {
            $newTitle = isset($post_update['post_title']) ? (string) $post_update['post_title'] : '';
            $newContent = isset($post_update['post_content']) ? (string) $post_update['post_content'] : '';
            rankbot_log('Apply optimization: computed changes summary', [
                'post_id' => (int) $post_id,
                'action' => is_string($action) ? $action : '',
                'allow_title_update' => (string) $allow_title_update,
                'allow_main_content_update' => (string) $allow_main_content_update,
                'will_update_title' => isset($post_update['post_title']),
                'will_update_content' => isset($post_update['post_content']),
                'new_title_len' => strlen($newTitle),
                'new_content_len' => strlen($newContent),
                'new_content_words' => $newContent !== '' ? (int) str_word_count(wp_strip_all_tags($newContent)) : 0,
            ]);
        }
        if ($allow_main_content_update === 'yes' && isset($result['data']['short_description'])) {
            $new_excerpt = $this->rankbot_sanitize_generated_html((string) $result['data']['short_description']);
            if (trim(wp_strip_all_tags((string) $new_excerpt)) !== '') {
                $post_update['post_excerpt'] = $new_excerpt;
                $needs_update = true;
            }
        }

        // Normalize keyword casing inside headings (H2-H6) when content is updated
        if (!empty($focus_keyword) && isset($post_update['post_content'])) {
            $post_update['post_content'] = $this->rankbot_normalize_keyword_in_headings(
                (string) $post_update['post_content'],
                $focus_keyword,
                $this->rankbot_mb_title_case($focus_keyword)
            );
        }

        // Product-specific rule (Complex only):
        // If the original product description had no images, remove any AI-invented <img> tags.
        // Then ensure the featured image is present in the description HTML.
        if (
            $post &&
            $post->post_type === 'product' &&
            is_string($action) &&
            $action === 'complex' &&
            isset($post_update['post_content'])
        ) {
            $orig_has_img = preg_match('/<img\b/i', $original_content_html) === 1;
            $new_html = (string) $post_update['post_content'];

            if (!$orig_has_img) {
                $new_html = preg_replace('/<img\b[^>]*>/i', '', $new_html);
            }

            $featured_id_for_insert = get_post_thumbnail_id($post_id);
            if ($featured_id_for_insert) {
                $new_html = $this->rankbot_ensure_featured_image_in_html(
                    $new_html,
                    $featured_id_for_insert,
                    $focus_keyword,
                    (string) ($post_update['post_title'] ?? ($post->post_title ?? ''))
                );
            }

            $post_update['post_content'] = $new_html;
            $needs_update = true;
        }

        // Final HTML safety pass to prevent broken layouts (unclosed tags) and reduce blank paragraph spam.
        if (isset($post_update['post_content'])) {
            $post_update['post_content'] = $this->rankbot_sanitize_generated_html((string) $post_update['post_content']);
        }
        if (isset($post_update['post_excerpt'])) {
            $post_update['post_excerpt'] = $this->rankbot_sanitize_generated_html((string) $post_update['post_excerpt']);
        }

        if (function_exists('rankbot_log')) {
            rankbot_log('Apply optimization: computed post_update', [
                'post_id' => (int) $post_id,
                'needs_update' => (bool) $needs_update,
                'computed_fields' => array_values(array_diff(array_keys($post_update), ['ID'])),
                'has_product_name' => isset($result['data']['product_name']),
                'has_description' => isset($result['data']['description']),
                'has_content' => isset($result['data']['content']),
                'has_short_description' => isset($result['data']['short_description']),
            ]);
        }

        if ($needs_update) {
            $updateRes = wp_update_post($post_update, true);
            if (function_exists('rankbot_log')) {
                rankbot_log('Apply optimization: wp_update_post(main)', [
                    'post_id' => (int) $post_id,
                    'needs_update' => (bool) $needs_update,
                    'updated_fields' => array_values(array_diff(array_keys($post_update), ['ID'])),
                    'ok' => !is_wp_error($updateRes),
                    'wp_error' => is_wp_error($updateRes) ? $updateRes->get_error_message() : null,
                    'result' => is_wp_error($updateRes) ? null : (int) $updateRes,
                ], is_wp_error($updateRes) ? 'error' : 'info');
            }
        }

        // Apply Image Alt Text
        // Requirements: all images must get alts (featured, gallery, and content images)
        $featured_id = get_post_thumbnail_id($post_id);
        $gallery_ids = [];

        if (class_exists('WooCommerce') && $post->post_type === 'product') {
            $product = wc_get_product($post_id);
            if ($product) {
                $gallery_ids = $product->get_gallery_image_ids();
            }
        }

        $content_html_for_images = '';
        if (isset($post_update['post_content'])) {
            $content_html_for_images = (string) $post_update['post_content'];
        } elseif ($post && isset($post->post_content)) {
            $content_html_for_images = (string) $post->post_content;
        }

        $inline_ids = $this->rankbot_extract_attachment_ids_from_html($content_html_for_images);
        $image_ids = array_values(array_unique(array_filter(array_merge($featured_id ? [$featured_id] : [], $gallery_ids, $inline_ids))));

        $alts_from_api = [];
        if (isset($result['data']['image_alts']) && is_array($result['data']['image_alts'])) {
            $alts_from_api = $result['data']['image_alts'];
        }

        // Build attachment-id => alt map from API (supports both list and map)
        $alt_map = [];
        $is_list = !empty($alts_from_api) && array_keys($alts_from_api) === range(0, count($alts_from_api) - 1);
        if (!$is_list) {
            foreach ($alts_from_api as $maybe_id => $maybe_alt) {
                $img_id = absint($maybe_id);
                if ($img_id > 0 && $maybe_alt !== '') {
                    $alt_map[$img_id] = (string) $maybe_alt;
                }
            }
        }

        $fallback_alt = '';
        if (!empty($focus_keyword)) {
            $fallback_alt = $this->rankbot_mb_ucfirst($focus_keyword);
        } elseif (isset($post_update['post_title'])) {
            $fallback_alt = (string) $post_update['post_title'];
        } elseif ($post && isset($post->post_title)) {
            $fallback_alt = (string) $post->post_title;
        }

        // Ensure attachment alt meta exists for every discovered attachment image
        // If API returns a list, only apply it when we can confidently match counts.
        $alt_list_targets = $image_ids;
        if ($is_list) {
            if (!empty($inline_ids) && count($alts_from_api) === count($inline_ids)) {
                $alt_list_targets = $inline_ids;
            } elseif (count($alts_from_api) === count($image_ids)) {
                $alt_list_targets = $image_ids;
            } else {
                $alt_list_targets = [];
            }
        }

        $targets = $is_list ? $alt_list_targets : $image_ids;

        // Ensure the backup contains pre-change alts for every image ID we are about to touch.
        // This guarantees restore can fully revert all attachment alt mutations done by this run.
        if ($backup_ts > 0 && !empty($targets)) {
            $backups = get_post_meta($post_id, '_rankbot_backups', true);
            if (is_array($backups) && !empty($backups)) {
                foreach ($backups as $idx => $b) {
                    if (!isset($b['timestamp']) || (int) $b['timestamp'] !== (int) $backup_ts) {
                        continue;
                    }
                    if (!isset($backups[$idx]['data']) || !is_array($backups[$idx]['data'])) {
                        $backups[$idx]['data'] = [];
                    }
                    if (!isset($backups[$idx]['data']['image_alts']) || !is_array($backups[$idx]['data']['image_alts'])) {
                        $backups[$idx]['data']['image_alts'] = [];
                    }

                    foreach ($targets as $img_id) {
                        $img_id = absint($img_id);
                        if ($img_id < 1) continue;
                        if (!array_key_exists($img_id, $backups[$idx]['data']['image_alts'])) {
                            $backups[$idx]['data']['image_alts'][$img_id] = get_post_meta($img_id, '_wp_attachment_image_alt', true);
                        }
                    }

                    update_post_meta($post_id, '_rankbot_backups', $backups);
                    break;
                }
            }
        }

        foreach ($targets as $index => $img_id) {
            $alt_value = '';

            if (isset($alt_map[$img_id])) {
                $alt_value = $alt_map[$img_id];
            } elseif ($is_list && isset($alts_from_api[$index])) {
                $alt_value = (string) $alts_from_api[$index];
            } else {
                $alt_value = $fallback_alt;
            }

            $alt_value = sanitize_text_field($alt_value);
            if ($alt_value !== '') {
                update_post_meta($img_id, '_wp_attachment_image_alt', $alt_value);
            }
        }

        // Ensure content HTML has alt attributes too (raw HTML does not auto-update from attachment meta)
        if (!empty($content_html_for_images)) {
            // Extend map with attachment meta values so HTML can be filled reliably
            foreach ($inline_ids as $inline_id) {
                if (!isset($alt_map[$inline_id])) {
                    $existing_alt = get_post_meta($inline_id, '_wp_attachment_image_alt', true);
                    if (is_string($existing_alt) && $existing_alt !== '') {
                        $alt_map[$inline_id] = $existing_alt;
                    }
                }
            }

            $new_content_html = $this->rankbot_ensure_img_alts_in_html($content_html_for_images, $alt_map, $fallback_alt);

            $new_content_html = $this->rankbot_sanitize_generated_html((string) $new_content_html);

            if ($new_content_html !== $content_html_for_images) {
                $altHtmlRes = wp_update_post([
                    'ID' => $post_id,
                    'post_content' => $new_content_html,
                ], true);

                if (function_exists('rankbot_log')) {
                    rankbot_log('Apply optimization: wp_update_post(alt html)', [
                        'post_id' => (int) $post_id,
                        'ok' => !is_wp_error($altHtmlRes),
                        'wp_error' => is_wp_error($altHtmlRes) ? $altHtmlRes->get_error_message() : null,
                    ], is_wp_error($altHtmlRes) ? 'error' : 'info');
                }
            }
        }

        // Update History
        $history = get_post_meta($post_id, '_rankbot_history', true) ?: [];
        $history[] = [
            'date' => current_time('mysql'),
            'action' => ucfirst($action),
            'type' => $post->post_type,
            'cost' => $result['cost'] ?? 0
        ];
        update_post_meta($post_id, '_rankbot_history', $history);

        // Recalculate SEO score after applying changes (meta/content/keyword).
        // Bulk UI renders score from `_rankbot_seo_score`, so keep it in sync.
        $kw_for_score = $focus_keyword;
        if ($kw_for_score === '') {
            $kw_for_score = $this->rankbot_get_focus_keyword_for_post((int) $post_id);
        }
        if (is_string($kw_for_score) && trim($kw_for_score) !== '') {
            $analysis = $this->rankbot_build_seo_analysis((int) $post_id, (string) $kw_for_score);
            $score = isset($analysis['score']) ? (int) $analysis['score'] : 0;
            if ($score < 0) $score = 0;
            if ($score > 100) $score = 100;
            update_post_meta((int) $post_id, '_rankbot_seo_score', (string) $score);
        }

        if ($job_id !== '') {
            $applied_jobs = get_post_meta($post_id, '_rankbot_applied_jobs', true);
            if (!is_array($applied_jobs)) {
                $applied_jobs = [];
            }
            $applied_jobs[] = $job_id;
            $applied_jobs = array_values(array_unique(array_filter(array_map('strval', $applied_jobs))));
            if (count($applied_jobs) > 50) {
                $applied_jobs = array_slice($applied_jobs, -50);
            }
            update_post_meta($post_id, '_rankbot_applied_jobs', $applied_jobs);
        }

        // Cleanup job-scoped backup pointer
        if ($job_id !== '') {
            delete_transient('rankbot_job_backup_ts_' . $job_id);
        }
        
        return [
            'message' => 'Optimization complete!',
            'meta' => $meta_data,
            'history_count' => count($history),
            'last_run' => current_time('mysql')
        ];
    }

    private function process_optimization_result($post_id, $result, $action, string $job_id = '')
    {
        $out = $this->apply_optimization_result($post_id, $result, $action, $job_id);
        if (isset($out['error'])) {
            wp_send_json_error($out['error']);
        }
        wp_send_json_success($out);
    }

    private function create_backup($post_id, $action = ''): int
    {
        $post = get_post($post_id);
        if (!$post) return 0;

        // Caller should pass explicit action; avoid inferring from request superglobals.
        if (!is_string($action)) {
            $action = '';
        }

        $backups = get_post_meta($post_id, '_rankbot_backups', true) ?: [];
        
        // Meta keys to backup
        $meta_keys = [
            // Yoast
            '_yoast_wpseo_title',
            '_yoast_wpseo_metadesc',
            '_yoast_wpseo_focuskw',

            // Rank Math
            'rank_math_title',
            'rank_math_description',
            'rank_math_focus_keyword',

            // RankBot internal
            '_rankbot_focus_keyword',
            '_rankbot_seo_score',
            // RankBot history (this is mutated by optimization)
            '_rankbot_history'
        ];
        
        $meta_backup = [];
        foreach($meta_keys as $key) {
             if (metadata_exists('post', $post_id, $key)) {
                 $meta_backup[$key] = get_post_meta($post_id, $key, true);
             }
        }
        
        // Image Alts Backup
        $image_alts = [];
        $featured_id = get_post_thumbnail_id($post_id);
        $gallery_ids = [];
        if (class_exists('WooCommerce') && $post->post_type === 'product') {
            $product = wc_get_product($post_id);
            if ($product) $gallery_ids = $product->get_gallery_image_ids();
        }
        $inline_ids = $this->rankbot_extract_attachment_ids_from_html((string) $post->post_content);
        $image_ids = array_values(array_unique(array_filter(array_merge($featured_id ? [$featured_id] : [], $gallery_ids, $inline_ids))));

        foreach($image_ids as $img_id) {
            $image_alts[$img_id] = get_post_meta($img_id, '_wp_attachment_image_alt', true);
        }

        $ts = time();
        $new_backup = [
            'timestamp' => $ts,
            'date' => current_time('mysql'),
            'action' => is_string($action) ? sanitize_key($action) : '',
            'post_type' => isset($post->post_type) ? (string) $post->post_type : '',
            'data' => [
                'post_title' => $post->post_title,
                'post_name' => $post->post_name,
                'post_content' => $post->post_content,
                'post_excerpt' => $post->post_excerpt,
                'meta' => $meta_backup,
                'image_alts' => $image_alts
            ]
        ];
        
        // Add to start, keep max 10
        array_unshift($backups, $new_backup);
        $backups = array_slice($backups, 0, 10);
        
        update_post_meta($post_id, '_rankbot_backups', $backups);

        return (int) $ts;
    }

    private function rankbot_mb_ucfirst($text)
    {
        $text = (string) $text;
        if ($text === '') return '';
        if (!function_exists('mb_substr') || !function_exists('mb_strtoupper')) {
            return ucfirst($text);
        }
        $first = mb_substr($text, 0, 1);
        $rest = mb_substr($text, 1);
        return mb_strtoupper($first) . $rest;
    }

    private function rankbot_mb_title_case($text)
    {
        $text = (string) $text;
        if ($text === '') return '';
        if (function_exists('mb_convert_case')) {
            return mb_convert_case($text, MB_CASE_TITLE, 'UTF-8');
        }
        return ucwords($text);
    }

    private function rankbot_sanitize_generated_html(string $html): string
    {
        $html = (string) $html;
        if ($html === '') return '';

        // 1) Best-effort fix for broken/unclosed tags that can break the whole page layout.
        if (function_exists('force_balance_tags')) {
            $html = force_balance_tags($html);
        }

        // 2) Collapse repeated empty paragraphs like <p>&nbsp;</p> spam.
        // Convert common NBSP forms to a canonical one for easier matching.
        $html = str_replace(['&#160;', "\xC2\xA0"], '&nbsp;', $html);

        // Remove excessive runs of empty paragraphs (keep at most 1).
        // Matches: <p>\s*&nbsp;\s*</p> or <p>\s*</p>
        $pattern = '/(?:\s*<p>\s*(?:&nbsp;)?\s*<\/p>\s*){2,}/i';
        $html = preg_replace($pattern, "\n<p>&nbsp;</p>\n", $html);

        // Trim leading/trailing empty paragraphs
        $html = preg_replace('/^(\s*<p>\s*(?:&nbsp;)?\s*<\/p>\s*)+/i', '', $html);
        $html = preg_replace('/(\s*<p>\s*(?:&nbsp;)?\s*<\/p>\s*)+$/i', '', $html);

        return (string) $html;
    }

    private function rankbot_replace_phrase_ci($text, $needle, $replacement)
    {
        $text = (string) $text;
        $needle = (string) $needle;
        $replacement = (string) $replacement;

        if ($text === '' || $needle === '' || $replacement === '') return $text;

        return preg_replace('/' . preg_quote($needle, '/') . '/iu', $replacement, $text);
    }

    private function rankbot_ensure_featured_image_in_html($html, $attachment_id, $focus_keyword, $fallback_title)
    {
        $html = (string) $html;
        $attachment_id = absint($attachment_id);
        if ($attachment_id <= 0) return $html;

        // Already present?
        if (preg_match('/\bwp-image-' . $attachment_id . '\b/i', $html) || preg_match('/\bdata-id=["\']' . $attachment_id . '["\']/i', $html)) {
            return $html;
        }

        $src = wp_get_attachment_url($attachment_id);
        if (!$src) return $html;

        if (stripos($html, (string) $src) !== false) {
            return $html;
        }

        $alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
        if (!is_string($alt) || $alt === '') {
            if (!empty($focus_keyword)) {
                $alt = $this->rankbot_mb_ucfirst((string) $focus_keyword);
            } else {
                $alt = (string) $fallback_title;
            }
        }

        $src_attr = esc_url_raw($src);
        $alt_attr = esc_attr($alt);

        // Constrain visual size: ~600px width (responsive down on mobile)
        $img = '<p><img class="wp-image-' . $attachment_id . '" src="' . $src_attr . '" alt="' . $alt_attr . '" width="600" style="max-width:600px;width:100%;height:auto;" /></p>';

        // Insert into the middle of the description (after the middle </p>)
        if (preg_match_all('/<\/p\s*>/i', $html, $matches, PREG_OFFSET_CAPTURE) && !empty($matches[0])) {
            $count = count($matches[0]);
            $midIndex = (int) floor($count / 2);
            $midPos = (int) $matches[0][$midIndex][1];
            $insertAt = $midPos + strlen($matches[0][$midIndex][0]);
            return substr($html, 0, $insertAt) . $img . substr($html, $insertAt);
        }

        // Fallback: prepend if we can't find paragraph boundaries
        return $img . $html;
    }

    public function handle_ajax_bulk_count_active()
    {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
        // Allow 'rankbot_optimize' nonce too since it is used by JS
        $check1 = check_ajax_referer('rankbot_bulk_action', 'nonce', false);
        $check2 = check_ajax_referer('rankbot_optimize', 'nonce', false);
        
        if (!$check1 && !$check2) {
             wp_send_json_error('Nonce verification failed');
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Forbidden');
        }

        $site_key_hash = $this->rankbot_site_key_hash();
        $remaining = $this->rankbot_bulk_remaining_count($site_key_hash);

        global $wpdb;
        $table = $wpdb->prefix . 'rankbot_jobs';

        $active_jobs = 0;
        if ($this->rankbot_db_table_exists($table)) {
            if ($site_key_hash !== '') {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $active_jobs = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status IN ('processing', 'pending', 'queued') AND created_at > DATE_SUB(NOW(), INTERVAL 2 HOUR)",
                        $table,
                        $site_key_hash
                    )
                );
            } else {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $active_jobs = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM %i WHERE site_key_hash = %s AND status IN ('processing', 'pending', 'queued') AND created_at > DATE_SUB(NOW(), INTERVAL 2 HOUR)",
                        $table,
                        $site_key_hash
                    )
                );
            }
        }

        $count = max($remaining, $active_jobs);

        // Get Balance (bypass transient if user requests refresh, or just rely on short transient)
        // Since this is polled frequently, we should respect a short transient to avoid spamming the remote API too much
        // But the user requested "updates immediately".
        // Let's rely on the short 30s transient we set in add_admin_bar_balance, 
        // OR explicitely call get_balance() but only if we haven't checked recently. 
        // Actually, let's keep it simple: fetch from transient first, if empty fetch fresh.
        $balance = get_transient('rankbot_admin_balance');
        if (false === $balance) {
             $balance = $this->api->get_balance();
             set_transient('rankbot_admin_balance', $balance, 30);
        }

        wp_send_json_success([
            'count' => (int) $count,
            'remaining' => (int) $remaining,
            'active_jobs' => (int) $active_jobs,
            'balance' => number_format_i18n((float) $balance),
        ]);
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
    }

    private function rankbot_normalize_keyword_in_headings($html, $keyword, $keyword_replacement)
    {
        $html = (string) $html;
        $keyword = (string) $keyword;
        $keyword_replacement = (string) $keyword_replacement;
        if ($html === '' || $keyword === '' || $keyword_replacement === '') return $html;

        return preg_replace_callback(
            '/(<h[2-6]\\b[^>]*>)(.*?)(<\\/h[2-6]>)/is',
            function ($m) use ($keyword, $keyword_replacement) {
                $inner = $m[2];
                $inner = $this->rankbot_replace_phrase_ci($inner, $keyword, $keyword_replacement);
                return $m[1] . $inner . $m[3];
            },
            $html
        );
    }

    private function rankbot_extract_attachment_ids_from_html($html)
    {
        $html = (string) $html;
        if ($html === '') return [];

        $ids = [];

        // Typical WP content images have class like wp-image-123
        if (preg_match_all('/\\bwp-image-(\\d+)\\b/i', $html, $m)) {
            foreach ($m[1] as $id) {
                $ids[] = absint($id);
            }
        }

        // Some builders use data-id="123" on img tags
        if (preg_match_all('/<img[^>]+data-id=["\'](\\d+)["\'][^>]*>/i', $html, $m2)) {
            foreach ($m2[1] as $id) {
                $ids[] = absint($id);
            }
        }

        $ids = array_values(array_unique(array_filter($ids)));
        return $ids;
    }

    private function rankbot_collect_images_for_alt_generation($post_id, $post)
    {
        $post_id = absint($post_id);
        if (!$post_id || !$post) return [];

        $images = [];

        $featured_id = get_post_thumbnail_id($post_id);
        if ($featured_id) {
            $payload = $this->rankbot_build_image_payload($featured_id, 'featured');
            if ($payload) $images[] = $payload;
        }

        if (class_exists('WooCommerce') && isset($post->post_type) && $post->post_type === 'product') {
            $product = wc_get_product($post_id);
            if ($product) {
                foreach ($product->get_gallery_image_ids() as $img_id) {
                    $payload = $this->rankbot_build_image_payload($img_id, 'gallery');
                    if ($payload) $images[] = $payload;
                }
            }
        }

        $inline_ids = $this->rankbot_extract_attachment_ids_from_html((string) ($post->post_content ?? ''));
        foreach ($inline_ids as $img_id) {
            $payload = $this->rankbot_build_image_payload($img_id, 'content');
            if ($payload) $images[] = $payload;
        }

        return array_values($images);
    }

    private function rankbot_build_image_payload($attachment_id, $role)
    {
        $attachment_id = absint($attachment_id);
        if (!$attachment_id) return null;

        $title = get_the_title($attachment_id);
        $caption = function_exists('wp_get_attachment_caption') ? wp_get_attachment_caption($attachment_id) : '';
        $alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
        $src = function_exists('wp_get_attachment_url') ? wp_get_attachment_url($attachment_id) : '';
        $file = function_exists('get_attached_file') ? get_attached_file($attachment_id) : '';

        $filename = '';
        if (is_string($file) && $file !== '') {
            $filename = basename($file);
        }

        return [
            'id' => $attachment_id,
            'role' => is_string($role) ? $role : '',
            'title' => is_string($title) ? $title : '',
            'caption' => is_string($caption) ? $caption : '',
            'filename' => $filename,
            'src' => is_string($src) ? $src : '',
            'alt' => is_string($alt) ? $alt : '',
        ];
    }

    private function rankbot_ensure_img_alts_in_html($html, $alt_map, $fallback_alt)
    {
        $html = (string) $html;
        if ($html === '') return $html;

        if (!is_array($alt_map)) {
            $alt_map = [];
        }
        $fallback_alt = sanitize_text_field((string) $fallback_alt);

        return preg_replace_callback(
            '/<img\\b[^>]*>/i',
            function ($m) use ($alt_map, $fallback_alt) {
                $tag = $m[0];

                // If alt exists and is non-empty, keep it
                if (preg_match('/\\balt\\s*=\\s*(["\'])(.*?)\\1/i', $tag, $altm)) {
                    $current = trim((string) $altm[2]);
                    if ($current !== '') {
                        return $tag;
                    }
                }

                $alt_value = $fallback_alt;

                // Find attachment ID via wp-image-123 class
                if (preg_match('/\\bwp-image-(\\d+)\\b/i', $tag, $idm)) {
                    $img_id = absint($idm[1]);
                    if ($img_id > 0 && isset($alt_map[$img_id])) {
                        $alt_value = sanitize_text_field((string) $alt_map[$img_id]);
                    }
                }

                if ($alt_value === '') {
                    return $tag;
                }

                // Replace empty alt="" / alt='' or insert alt
                if (preg_match('/\\balt\\s*=\\s*(["\'])(.*?)\\1/i', $tag)) {
                    $tag = preg_replace('/\\balt\\s*=\\s*(["\'])(.*?)\\1/i', 'alt="' . esc_attr($alt_value) . '"', $tag, 1);
                } else {
                    $tag = preg_replace('/<img\\b/i', '<img alt="' . esc_attr($alt_value) . '"', $tag, 1);
                }

                return $tag;
            },
            $html
        );
    }

    public function handle_incoming_webhook()
    {
        $page = (string) filter_input(INPUT_GET, 'page', FILTER_DEFAULT);
        $api_key = filter_input(INPUT_GET, 'api_key', FILTER_DEFAULT);
        $magic_token = filter_input(INPUT_GET, 'magic_token', FILTER_DEFAULT);

        if ($page === 'rankbotai-seo-optimizer' && null !== $api_key) {
             if (!current_user_can('manage_options')) return;
             
             // Use the API class setter to ensure consistency with option names
             $this->api->set_key(sanitize_text_field((string) $api_key));
             
             if (null !== $magic_token) {
                 update_option('rankbot_magic_token', sanitize_text_field((string) $magic_token));
             }
             
             wp_safe_redirect(remove_query_arg(['api_key', 'magic_token']));
             exit;
        }
    }

    private function get_active_tasks_remaining_count(): int
    {
        $site_key_hash = $this->rankbot_site_key_hash();
        $remaining = $this->rankbot_bulk_remaining_count($site_key_hash);

        if ($remaining > 0) {
            return $remaining;
        }
        return 0;
    }

    public function add_menu_page()
    {
        // Check for active tasks to show indicator
        $remaining = $this->get_active_tasks_remaining_count();
        $menu_title = (string) esc_html__('RankBotAI', 'rankbotai-seo-optimizer');
        if ($remaining > 0) {
            // Custom badge style to avoid WP's update-plugins circular mess
            $menu_title .= ' <span class="rb-menu-badge" style="
                background-color: #059669; 
                color: #ffffff; 
                display: inline-block; 
                padding: 2px 6px; 
                border-radius: 99px; 
                font-size: 10px; 
                font-weight: 600; 
                line-height: normal;
                vertical-align: middle;
                margin-left: 2px;
            ">' . $remaining . '</span>';
        }

        // Main Menu
        add_menu_page(
            (string) __('RankBotAI', 'rankbotai-seo-optimizer'),
            $menu_title,
            'manage_options',
            'rankbotai-seo-optimizer',
            [$this, 'render_page'],
            'dashicons-superhero',
            6
        );
        
        $key = $this->api->get_key();
        
        // Submenus - Only if connected
        add_submenu_page(
            'rankbotai-seo-optimizer',
            (string) __('Dashboard', 'rankbotai-seo-optimizer'),
            '<span class="dashicons dashicons-dashboard" style="font-size:16px; margin-right:4px;"></span> ' . esc_html__('Dashboard', 'rankbotai-seo-optimizer'),
            'manage_options',
            'rankbotai-seo-optimizer',
            [$this, 'render_page']
        );

        // Bulk processing is always registered to avoid "Cannot load rankbot-bulk" when URL is opened directly.
        add_submenu_page(
            'rankbotai-seo-optimizer',
            (string) __('Bulk Processing', 'rankbotai-seo-optimizer'),
            '<span class="dashicons dashicons-grid-view" style="font-size:16px; margin-right:4px;"></span> ' . esc_html__('Bulk Processing', 'rankbotai-seo-optimizer'),
            'manage_options',
            'rankbot-bulk',
            [$this, 'render_bulk_page']
        );

        if ($key) {
            add_submenu_page(
                'rankbotai-seo-optimizer',
                (string) __('History', 'rankbotai-seo-optimizer'),
                '<span class="dashicons dashicons-backup" style="font-size:16px; margin-right:4px;"></span> ' . esc_html__('History', 'rankbotai-seo-optimizer'),
                'manage_options',
                'rankbot-history',
                [$this, 'render_history_page']
            );

            add_submenu_page(
                'rankbotai-seo-optimizer',
                (string) __('Settings', 'rankbotai-seo-optimizer'),
                '<span class="dashicons dashicons-admin-settings" style="font-size:16px; margin-right:4px;"></span> ' . esc_html__('Settings', 'rankbotai-seo-optimizer'),
                'manage_options',
                'rankbot-settings',
                [$this, 'render_settings_page']
            );
        }

        // llms.txt Tools — always available (no key required)
        add_submenu_page(
            'rankbotai-seo-optimizer',
            (string) __('llms.txt', 'rankbotai-seo-optimizer'),
            '<span class="dashicons dashicons-media-text" style="font-size:16px; margin-right:4px;"></span> ' . esc_html__('llms.txt', 'rankbotai-seo-optimizer'),
            'manage_options',
            'rankbot-llms',
            [$this, 'render_llms_page']
        );
    }
    
    public function render_bulk_page() 
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Permission denied', 'rankbotai-seo-optimizer'));
        }

        $key = $this->api->get_key();
        if (!$key) {
            echo '<div class="wrap"><h1>' . esc_html__('RankBotAI — Bulk Processing', 'rankbotai-seo-optimizer') . '</h1>';
            echo '<div class="notice notice-warning"><p>' . esc_html__('RankBot is not connected. Please add your API key in RankBotAI → Settings.', 'rankbotai-seo-optimizer') . '</p></div>';
            echo '</div>';
            return;
        }

        // Filters (require nonce for processing user-controlled inputs).
        $bulk_nonce_ok = false;
        if (isset($_GET['rankbot_bulk_nonce'])) {
            if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['rankbot_bulk_nonce'])), 'rankbot_bulk_filters')) {
                wp_die(esc_html__('Security check failed.', 'rankbotai-seo-optimizer'), 403);
            }
            $bulk_nonce_ok = true;
        }

        $post_types_in = [];
        $post_statuses_in = [];
        $search = '';
        $score_min = 0;
        $score_max = 100;
        $paged = 1;
        $per_page = 20;
        $no_keyword = 0;
        $proc_state = 'all';

        if ($bulk_nonce_ok) {
            if (isset($_GET['post_type'])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via sanitize_key() below.
                $raw = wp_unslash($_GET['post_type']);
                $post_types_in = is_array($raw) ? $raw : [(string) $raw];
            }
            $post_types_in = array_values(array_unique(array_map('sanitize_key', array_filter(array_map('strval', $post_types_in)))));

            if (isset($_GET['post_status'])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via sanitize_key() below.
                $raw = wp_unslash($_GET['post_status']);
                $post_statuses_in = is_array($raw) ? $raw : [(string) $raw];
            }
            $post_statuses_in = array_values(array_unique(array_map('sanitize_key', array_filter(array_map('strval', $post_statuses_in)))));

            $search = isset($_GET['s']) ? sanitize_text_field((string) wp_unslash($_GET['s'])) : '';
            $score_min = isset($_GET['score_min']) ? absint(wp_unslash($_GET['score_min'])) : 0;
            $score_max = isset($_GET['score_max']) ? absint(wp_unslash($_GET['score_max'])) : 100;
            $paged = isset($_GET['paged']) ? max(1, absint(wp_unslash($_GET['paged']))) : 1;
            $per_page = isset($_GET['per_page']) ? absint(wp_unslash($_GET['per_page'])) : 20;
            $per_page = max(1, min(999, $per_page));

            $no_keyword = isset($_GET['no_keyword']) ? 1 : 0;

            // Processing State.
            $proc_state = isset($_GET['proc_state']) ? sanitize_key(wp_unslash($_GET['proc_state'])) : 'all';
            // Fallback for old URL param if still present.
            if ($proc_state === 'all' && isset($_GET['only_unprocessed'])) {
                $old_val = absint(wp_unslash($_GET['only_unprocessed']));
                if ($old_val === 1) {
                    $proc_state = 'unprocessed';
                }
            }
        }

        $public_types = get_post_types(['public' => true], 'objects');
        $supported_types = [];
        foreach ($public_types as $pt => $obj) {
            if ($this->rankbot_is_supported_post_type((string) $pt, $obj)) {
                $supported_types[(string) $pt] = $obj;
            }
        }

        // Only show post types that actually have items (keeps the filter list compact and relevant).
        // We intentionally keep this check lightweight (1 post per type).
        $allowed_statuses = ['publish', 'draft', 'pending', 'private', 'future'];
        $supported_types_nonempty = [];
        foreach ($supported_types as $pt => $obj) {
            $probe = new WP_Query([
                'post_type' => (string) $pt,
                'post_status' => $allowed_statuses,
                'posts_per_page' => 1,
                'fields' => 'ids',
                'no_found_rows' => true,
                'orderby' => 'ID',
                'order' => 'DESC',
            ]);
            if (!empty($probe->posts)) {
                $supported_types_nonempty[(string) $pt] = $obj;
            }
        }
        if (!empty($supported_types_nonempty)) {
            $supported_types = $supported_types_nonempty;
        }
        if (empty($supported_types)) {
            echo '<div class="wrap"><h1>' . esc_html__('RankBotAI — Bulk Processing', 'rankbotai-seo-optimizer') . '</h1><p>' . esc_html__('No supported post types found.', 'rankbotai-seo-optimizer') . '</p></div>';;
            return;
        }

        // Default: all supported types when nothing is selected.
        if (empty($post_types_in)) {
            $post_types_in = array_keys($supported_types);
        } else {
            $post_types_in = array_values(array_intersect($post_types_in, array_keys($supported_types)));
            if (empty($post_types_in)) {
                $post_types_in = array_keys($supported_types);
            }
        }

        if (empty($post_statuses_in)) {
            $post_statuses_in = ['publish'];
        } else {
            $post_statuses_in = array_values(array_intersect($post_statuses_in, $allowed_statuses));
            if (empty($post_statuses_in)) {
                $post_statuses_in = ['publish'];
            }
        }

        $score_min = max(0, min(100, $score_min));
        $score_max = max(0, min(100, $score_max));
        if ($score_max < $score_min) {
            $tmp = $score_min;
            $score_min = $score_max;
            $score_max = $tmp;
        }

        $score_between = [
            'key' => '_rankbot_seo_score',
            'value' => [$score_min, $score_max],
            'compare' => 'BETWEEN',
            'type' => 'NUMERIC',
        ];
        $score_clause = ($score_min <= 0)
            ? [
                'relation' => 'OR',
                $score_between,
                ['key' => '_rankbot_seo_score', 'compare' => 'NOT EXISTS'],
            ]
            : $score_between;

        $meta_query = [
            'relation' => 'AND',
            $score_clause,
        ];

        $extra_query_args = [];

        // Process State Logic
        if ($proc_state === 'unprocessed') {
            $meta_query[] = [
                'relation' => 'OR',
                ['key' => '_rankbot_history', 'compare' => 'NOT EXISTS'],
                ['key' => '_rankbot_history', 'value' => '', 'compare' => '='],
            ];
        } elseif ($proc_state === 'processed') {
             $meta_query[] = [
                'key' => '_rankbot_history',
                'compare' => 'EXISTS'
            ];
             $meta_query[] = [
                'key' => '_rankbot_history',
                'value' => '',
                'compare' => '!='
            ];
        } elseif ($proc_state === 'errors') {
            global $wpdb;
            $cache_key = 'rankbot_bulk_failed_object_ids';
            $failed_ids = wp_cache_get($cache_key, 'rankbotai-seo-optimizer');
            if (false === $failed_ids) {
                $failed_ids = [];
                wp_cache_set($cache_key, $failed_ids, 'rankbotai-seo-optimizer', 30);
            }
            
            if (empty($failed_ids)) {
                $failed_ids = [0]; // Force no results
            }
            $extra_query_args['post__in'] = array_map('intval', $failed_ids);
        }

        $query_args = array_merge($extra_query_args, [
            'post_type' => $post_types_in,
            'post_status' => $post_statuses_in,
            'posts_per_page' => $per_page,
            'paged' => $paged,
            'orderby' => 'modified',
            'order' => 'DESC',
            'meta_query' => $meta_query,
            'rankbot_no_keyword' => $no_keyword ? 1 : 0,
        ]);


        if ($search !== '') {
            $query_args['s'] = $search;
        }

        $q = new WP_Query($query_args);
        
        // Fetch Job Status for current page posts to block duplicates
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
        $active_jobs = [];
        $last_completed_durations = [];
        if (!empty($q->posts)) {
            $current_ids = array_map('absint', wp_list_pluck($q->posts, 'ID'));
            $current_ids = array_values(array_filter($current_ids));
            if (!empty($current_ids)) {
                global $wpdb;
                $site_key_hash = $this->rankbot_site_key_hash();

                // For the Time column in Bulk: show last completed duration per record.
                $last_completed_durations = [];
                try {
                    $jobs_table = $wpdb->prefix . 'rankbot_jobs';
                    if ($this->rankbot_db_table_exists($jobs_table)) {
                        $has_duration = $this->rankbot_db_column_exists($jobs_table, 'duration_sec');
                        $has_started = $this->rankbot_db_column_exists($jobs_table, 'started_at');
                        $has_completed = $this->rankbot_db_column_exists($jobs_table, 'completed_at');
                        $jobs_has_timing = !empty($has_duration) && !empty($has_started) && !empty($has_completed);
                        if ($jobs_has_timing) {
                            foreach ($current_ids as $oid) {
                                if ($site_key_hash !== '') {
                                    $r = $wpdb->get_row(
                                        $wpdb->prepare(
                                            "SELECT duration_sec FROM %i
                                                WHERE (site_key_hash = %s OR site_key_hash = '')
                                                  AND object_type = 'post'
                                                  AND status = 'completed'
                                                  AND created_at > DATE_SUB(NOW(), INTERVAL 30 DAY)
                                                  AND object_id = %d
                                                ORDER BY id DESC
                                                LIMIT 1",
                                            $jobs_table,
                                            $site_key_hash,
                                            (int) $oid
                                        ),
                                        ARRAY_A
                                    );
                                } else {
                                    $r = $wpdb->get_row(
                                        $wpdb->prepare(
                                            "SELECT duration_sec FROM %i
                                                WHERE site_key_hash = %s
                                                  AND object_type = 'post'
                                                  AND status = 'completed'
                                                  AND created_at > DATE_SUB(NOW(), INTERVAL 30 DAY)
                                                  AND object_id = %d
                                                ORDER BY id DESC
                                                LIMIT 1",
                                            $jobs_table,
                                            $site_key_hash,
                                            (int) $oid
                                        ),
                                        ARRAY_A
                                    );
                                }

                                if (is_array($r) && isset($r['duration_sec'])) {
                                    $dur = (int) $r['duration_sec'];
                                    if ($dur > 0) {
                                        $last_completed_durations[(int) $oid] = $dur;
                                    }
                                }
                            }
                        }
                    }
                } catch (\Exception $e) {
                    $last_completed_durations = [];
                }

                if ($this->rankbot_bulk_queue_available()) {
                    $queue_table = $this->rankbot_bulk_queue_table();
                    $rows = [];
                    foreach ($current_ids as $oid) {
                        if ($site_key_hash !== '') {
                            $row = $wpdb->get_row(
                                $wpdb->prepare(
                                    "SELECT id, object_id, status, job_id FROM %i
                                        WHERE (site_key_hash = %s OR site_key_hash = '')
                                          AND object_id = %d
                                          AND status IN ('queued','retry_wait','dispatched')
                                        ORDER BY id DESC
                                        LIMIT 1",
                                    $queue_table,
                                    $site_key_hash,
                                    (int) $oid
                                ),
                                ARRAY_A
                            );
                        } else {
                            $row = $wpdb->get_row(
                                $wpdb->prepare(
                                    "SELECT id, object_id, status, job_id FROM %i
                                        WHERE site_key_hash = %s
                                          AND object_id = %d
                                          AND status IN ('queued','retry_wait','dispatched')
                                        ORDER BY id DESC
                                        LIMIT 1",
                                    $queue_table,
                                    $site_key_hash,
                                    (int) $oid
                                ),
                                ARRAY_A
                            );
                        }
                        if (is_array($row)) {
                            $rows[] = $row;
                        }
                    }

                    $job_ids = [];
                    if (is_array($rows)) {
                        foreach ($rows as $row) {
                            $oid = isset($row['object_id']) ? absint($row['object_id']) : 0;
                            if ($oid < 1) {
                                continue;
                            }
                            if (isset($active_jobs[$oid])) {
                                continue;
                            }
                            $active_jobs[$oid] = $row;
                            if (!empty($row['job_id']) && (string) $row['status'] === 'dispatched') {
                                $job_ids[] = (string) $row['job_id'];
                            }
                        }
                    }

                    if (!empty($job_ids)) {
                        $jobs_table = $wpdb->prefix . 'rankbot_jobs';
                        if ($this->rankbot_db_table_exists($jobs_table)) {
                            $jobs_has_timing = false;
                            try {
                                $has_duration = $this->rankbot_db_column_exists($jobs_table, 'duration_sec');
                                $jobs_has_timing = !empty($has_duration);
                            } catch (\Exception $e) {
                                $jobs_has_timing = false;
                            }

                            $job_ids = array_values(array_unique($job_ids));
                            $job_map = [];
                            foreach ($job_ids as $jid) {
                                if ($jobs_has_timing) {
                                    $jr = $wpdb->get_row(
                                        $wpdb->prepare(
                                            "SELECT job_id, status, created_at, started_at, completed_at, duration_sec FROM %i WHERE job_id = %s",
                                            $jobs_table,
                                            (string) $jid
                                        ),
                                        ARRAY_A
                                    );
                                } else {
                                    $jr = $wpdb->get_row(
                                        $wpdb->prepare(
                                            "SELECT job_id, status, created_at FROM %i WHERE job_id = %s",
                                            $jobs_table,
                                            (string) $jid
                                        ),
                                        ARRAY_A
                                    );
                                }

                                if (is_array($jr) && !empty($jr['job_id'])) {
                                    $job_map[(string) $jr['job_id']] = $jr;
                                }
                            }

                            foreach ($active_jobs as $oid => $info) {
                                if (!is_array($info)) {
                                    continue;
                                }
                                if (($info['status'] ?? '') !== 'dispatched') {
                                    continue;
                                }
                                $jid = (string) ($info['job_id'] ?? '');
                                if ($jid !== '' && isset($job_map[$jid])) {
                                    $st = isset($job_map[$jid]['status']) ? (string) $job_map[$jid]['status'] : '';
                                    if (in_array($st, ['queued', 'processing', 'pending'], true)) {
                                        $active_jobs[$oid]['status'] = $st;
                                        if ($jobs_has_timing) {
                                            $active_jobs[$oid]['created_at'] = $job_map[$jid]['created_at'] ?? null;
                                            $active_jobs[$oid]['started_at'] = $job_map[$jid]['started_at'] ?? null;
                                            $active_jobs[$oid]['completed_at'] = $job_map[$jid]['completed_at'] ?? null;
                                            $active_jobs[$oid]['duration_sec'] = $job_map[$jid]['duration_sec'] ?? null;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                $missing = [];
                foreach ($current_ids as $oid) {
                    if (!isset($active_jobs[$oid])) {
                        $missing[] = $oid;
                    }
                }

                if (!empty($missing)) {
                    $jobs_table = $wpdb->prefix . 'rankbot_jobs';
                    if ($this->rankbot_db_table_exists($jobs_table)) {
                        $jobs_has_timing = false;
                        try {
                            $has_duration = $this->rankbot_db_column_exists($jobs_table, 'duration_sec');
                            $jobs_has_timing = !empty($has_duration);
                        } catch (\Exception $e) {
                            $jobs_has_timing = false;
                        }
                        foreach ($missing as $oid) {
                            if ($site_key_hash !== '') {
                                if ($jobs_has_timing) {
                                    $row = $wpdb->get_row(
                                        $wpdb->prepare(
                                            "SELECT object_id, status, job_id, created_at, started_at, completed_at, duration_sec FROM %i
                                                WHERE (site_key_hash = %s OR site_key_hash = '')
                                                  AND object_type = 'post'
                                                  AND status IN ('processing','pending','queued')
                                                  AND created_at > DATE_SUB(NOW(), INTERVAL 6 HOUR)
                                                  AND object_id = %d
                                                ORDER BY id DESC
                                                LIMIT 1",
                                            $jobs_table,
                                            $site_key_hash,
                                            (int) $oid
                                        ),
                                        ARRAY_A
                                    );
                                } else {
                                    $row = $wpdb->get_row(
                                        $wpdb->prepare(
                                            "SELECT object_id, status, job_id, created_at FROM %i
                                                WHERE (site_key_hash = %s OR site_key_hash = '')
                                                  AND object_type = 'post'
                                                  AND status IN ('processing','pending','queued')
                                                  AND created_at > DATE_SUB(NOW(), INTERVAL 6 HOUR)
                                                  AND object_id = %d
                                                ORDER BY id DESC
                                                LIMIT 1",
                                            $jobs_table,
                                            $site_key_hash,
                                            (int) $oid
                                        ),
                                        ARRAY_A
                                    );
                                }
                            } else {
                                if ($jobs_has_timing) {
                                    $row = $wpdb->get_row(
                                        $wpdb->prepare(
                                            "SELECT object_id, status, job_id, created_at, started_at, completed_at, duration_sec FROM %i
                                                WHERE site_key_hash = %s
                                                  AND object_type = 'post'
                                                  AND status IN ('processing','pending','queued')
                                                  AND created_at > DATE_SUB(NOW(), INTERVAL 6 HOUR)
                                                  AND object_id = %d
                                                ORDER BY id DESC
                                                LIMIT 1",
                                            $jobs_table,
                                            $site_key_hash,
                                            (int) $oid
                                        ),
                                        ARRAY_A
                                    );
                                } else {
                                    $row = $wpdb->get_row(
                                        $wpdb->prepare(
                                            "SELECT object_id, status, job_id, created_at FROM %i
                                                WHERE site_key_hash = %s
                                                  AND object_type = 'post'
                                                  AND status IN ('processing','pending','queued')
                                                  AND created_at > DATE_SUB(NOW(), INTERVAL 6 HOUR)
                                                  AND object_id = %d
                                                ORDER BY id DESC
                                                LIMIT 1",
                                            $jobs_table,
                                            $site_key_hash,
                                            (int) $oid
                                        ),
                                        ARRAY_A
                                    );
                                }
                            }

                            if (is_array($row) && !empty($row['object_id'])) {
                                $active_jobs[(int) $row['object_id']] = $row;
                            }
                        }
                    }
                }
            }
        }

        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
        $nonce = wp_create_nonce('rankbot_optimize');
        $page_url = admin_url('admin.php?page=rankbot-bulk');

        // Enqueue Assets
        // Optimization: Dequeue potential external blockers on this heavy page to avoid 'stats.wp.com' timeouts
        wp_dequeue_script('devicepx');
        wp_dequeue_script('grofiles-cards');
        wp_dequeue_script('wp-stats');
        wp_dequeue_script('woo-tracks');

        wp_enqueue_style('rankbot-dashicons', includes_url('css/dashicons.min.css'));
        wp_enqueue_style('rankbot-bulk-css', RANKBOT_PLUGIN_URL . 'assets/css/rankbot-admin-bulk.css', [], defined('RANKBOT_VERSION') ? RANKBOT_VERSION : '1.0.0');
        wp_enqueue_script('rankbot-bulk-js', RANKBOT_PLUGIN_URL . 'assets/js/rankbot-admin-bulk.js', ['jquery'], defined('RANKBOT_VERSION') ? RANKBOT_VERSION : '1.0.0', true);

        $only_unprocessed = ($proc_state === 'unprocessed') ? 1 : 0;
        
        wp_localize_script('rankbot-bulk-js', 'rankbotBulkData', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => $nonce,
            'filters' => [
               'post_type' => array_values($post_types_in), 
               'post_status' => array_values($post_statuses_in),
               'score_min' => (int)$score_min,
               'score_max' => (int)$score_max,
               'search' => (string)$search,
               'no_keyword' => (int)$no_keyword,
               'only_unprocessed' => isset($only_unprocessed) ? (int)$only_unprocessed : 0,
            ],
            'i18n' => [
               'selected' => __('selected', 'rankbotai-seo-optimizer'),
               'confirmOne' => __('Queue processing for this item?', 'rankbotai-seo-optimizer'),
               'confirmAll' => __('Queue processing for ALL items matching current filters?', 'rankbotai-seo-optimizer'),
               'confirmSelected' => __('Queue processing for selected items?', 'rankbotai-seo-optimizer'),
               'confirmStop' => __('Stop all running bulk tasks?', 'rankbotai-seo-optimizer'),
                    'confirmClearQueue' => __('Are you sure you want to clear the current bulk queue? Pending items will be cancelled.', 'rankbotai-seo-optimizer'),
                    'confirmClearAll' => __('This will delete ALL RankBot job records for this API key on this WordPress site (including completed/failed) and reset processed state. Continue?', 'rankbotai-seo-optimizer'),
               'noSelection' => __('No items selected.', 'rankbotai-seo-optimizer'),
                    /* translators: %s: bulk task id. */
               'taskStarted' => __('Bulk task started: %s', 'rankbotai-seo-optimizer'),
               'tasksStopped' => __('Bulk tasks stopped.', 'rankbotai-seo-optimizer'),
                    /* translators: {remaining}: remaining items, {completed}: finished items, {total}: total items, {queued}: queued items, {retry}: retry items, {dispatched}: in-flight items, {failed}: failed/cancelled items. */
                'statusActive' => __('Active queue: <strong>{remaining}</strong>. Done {completed}/{total}; queued {queued}; retry {retry}; in-flight {dispatched}; failed {failed}.', 'rankbotai-seo-optimizer'),
               'statusIdle' => __('Idle', 'rankbotai-seo-optimizer'),
               'error' => __('Error', 'rankbotai-seo-optimizer'),
               'networkError' => __('Network error', 'rankbotai-seo-optimizer'),
            ]
        ]);

        include RANKBOT_PLUGIN_DIR . 'views/bulk-page.php';
    }

    private function rankbot_is_supported_post_type(string $pt, $obj): bool
    {
        $pt = sanitize_key($pt);
        if ($pt === '' || $pt === 'attachment') return false;
        if (!is_object($obj)) return false;

        // Bulk Processing: only Posts and Products for now.
        // Pages are intentionally not supported.
        if ($pt === 'post') {
            // ok
        } elseif ($pt === 'product') {
            if (!class_exists('WooCommerce')) return false;
        } else {
            return false;
        }

        // Must be editable and have title.
        if (!post_type_supports($pt, 'title')) return false;

        // We rely primarily on post_content; require editor support to avoid weird types.
        if (!post_type_supports($pt, 'editor')) return false;

        return true;
    }

    private function rankbot_available_actions_for_post_type(string $pt): array
    {
        $pt = sanitize_key($pt);
        // These are API action types.
        // We expose them consistently in Bulk UI, and let `auto` resolve to the right full optimization.
        return ['research', 'snippet', 'auto'];
    }

    public function handle_ajax_bulk_start(): void
    {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        if (!$this->rankbot_bulk_queue_available()) {
            if (function_exists('rankbot_create_tables')) {
                rankbot_create_tables();
            }
            if (!$this->rankbot_bulk_queue_available()) {
                wp_send_json_error('Bulk queue table is not available');
            }
        }

        $mode = isset($_POST['mode']) ? sanitize_key((string) wp_unslash($_POST['mode'])) : '';
        $bulk_action = isset($_POST['bulk_action']) ? sanitize_key((string) wp_unslash($_POST['bulk_action'])) : 'auto';
        if (!in_array($mode, ['selected', 'all'], true)) {
            wp_send_json_error('Invalid mode');
        }

        $site_key_hash = $this->rankbot_site_key_hash();
        $this->rankbot_bulk_clear_pause_state($site_key_hash);
        $bulk_id = function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : uniqid('rb_bulk_', true);
        $now = current_time('mysql');

        $ids = [];
        if ($mode === 'selected') {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via absint() in loop.
            $raw = isset($_POST['ids']) ? (array) wp_unslash($_POST['ids']) : [];
            foreach ($raw as $id) {
                $pid = absint($id);
                if ($pid > 0) {
                    $ids[] = $pid;
                }
            }
            $ids = array_values(array_unique($ids));
            if (empty($ids)) {
                wp_send_json_error('No ids provided');
            }
        }

        // Validate action availability for Complex.
        if ($bulk_action === 'complex') {
            if ($mode === 'all') {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized below.
                $raw_types = isset($_POST['post_type']) ? (array) wp_unslash($_POST['post_type']) : [];
                $raw_types = array_values(array_unique(array_map('sanitize_key', array_filter(array_map('strval', $raw_types)))));
                if (!(count($raw_types) === 1 && $raw_types[0] === 'product')) {
                    wp_send_json_error('Complex is only available for products. Select only Product post type.');
                }
            } else {
                foreach ($ids as $pid) {
                    $pt = get_post_type($pid);
                    if ($pt !== 'product') {
                        wp_send_json_error('Complex is only available for products.');
                    }
                }
            }
        }

        global $wpdb;
        $table = $this->rankbot_bulk_queue_table();
        $inserted = 0;
        $chunk_size = 200;

        if ($mode === 'selected') {
            // Strict de-duplication: do not enqueue items that are already active in the bulk queue.
            $active_object_ids = [];
            if (!empty($ids)) {
                $in_placeholders = implode(',', array_fill(0, count($ids), '%d'));
                if ($site_key_hash !== '') {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $existing = $wpdb->get_col(
                        $wpdb->prepare(
                            "SELECT DISTINCT object_id FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status IN ('queued','retry_wait','dispatched') AND object_id IN ($in_placeholders)",
                            ...array_merge([$table, $site_key_hash], $ids)
                        )
                    );
                } else {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $existing = $wpdb->get_col(
                        $wpdb->prepare(
                            "SELECT DISTINCT object_id FROM %i WHERE site_key_hash = %s AND status IN ('queued','retry_wait','dispatched') AND object_id IN ($in_placeholders)",
                            ...array_merge([$table, $site_key_hash], $ids)
                        )
                    );
                }
                if (is_array($existing) && !empty($existing)) {
                    foreach ($existing as $eid) {
                        $eid = absint($eid);
                        if ($eid > 0) {
                            $active_object_ids[$eid] = true;
                        }
                    }
                }
            }

            $rows = [];
            foreach ($ids as $post_id) {
                $post_id = absint($post_id);
                if ($post_id < 1) {
                    continue;
                }
                if (isset($active_object_ids[$post_id])) {
                    // Already in-flight/queued/retrying.
                    continue;
                }
                $post_type = get_post_type($post_id);
                if (!$post_type) {
                    continue;
                }
                $action = $this->rankbot_resolve_bulk_action((string) $post_type, (string) $bulk_action);
                if ($action === '') {
                    continue;
                }
                $rows[] = [$bulk_id, $site_key_hash, $post_id, 'post', $action, 'queued', '', 0, '', '', $now, $now];
                if (count($rows) >= $chunk_size) {
                    $inserted += $this->rankbot_bulk_insert_rows($table, $rows);
                    $rows = [];
                }
            }
            if (!empty($rows)) {
                $inserted += $this->rankbot_bulk_insert_rows($table, $rows);
            }
        } else {
            $query_args = $this->rankbot_build_bulk_query_args_from_post();
            if (empty($query_args)) {
                wp_send_json_error(__('Invalid filters', 'rankbotai-seo-optimizer'));
            }
            $query_args['fields'] = 'ids';
            $query_args['posts_per_page'] = $chunk_size;
            $query_args['orderby'] = 'ID';
            $query_args['order'] = 'ASC';
            $page = 1;
            $keep_running = true;

            while ($keep_running) {
                $query_args['paged'] = $page;
                $q = new WP_Query($query_args);
                $page_ids = is_array($q->posts) ? $q->posts : [];
                if (empty($page_ids)) {
                    break;
                }

                // Strict de-duplication per chunk: do not enqueue items that are already active in the bulk queue.
                $active_object_ids = [];
                if (!empty($page_ids)) {
                    $in_placeholders = implode(',', array_fill(0, count($page_ids), '%d'));
                    $page_ids_int = array_map('absint', $page_ids);
                    if ($site_key_hash !== '') {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        $existing = $wpdb->get_col(
                            $wpdb->prepare(
                                "SELECT DISTINCT object_id FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status IN ('queued','retry_wait','dispatched') AND object_id IN ($in_placeholders)",
                                ...array_merge([$table, $site_key_hash], $page_ids_int)
                            )
                        );
                    } else {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        $existing = $wpdb->get_col(
                            $wpdb->prepare(
                                "SELECT DISTINCT object_id FROM %i WHERE site_key_hash = %s AND status IN ('queued','retry_wait','dispatched') AND object_id IN ($in_placeholders)",
                                ...array_merge([$table, $site_key_hash], $page_ids_int)
                            )
                        );
                    }
                    if (is_array($existing) && !empty($existing)) {
                        foreach ($existing as $eid) {
                            $eid = absint($eid);
                            if ($eid > 0) {
                                $active_object_ids[$eid] = true;
                            }
                        }
                    }
                }

                $rows = [];
                foreach ($page_ids as $post_id) {
                    $post_id = absint($post_id);
                    if ($post_id < 1) {
                        continue;
                    }
                    if (isset($active_object_ids[$post_id])) {
                        // Already in-flight/queued/retrying.
                        continue;
                    }
                    $post_type = get_post_type($post_id);
                    if (!$post_type) {
                        continue;
                    }
                    $action = $this->rankbot_resolve_bulk_action((string) $post_type, (string) $bulk_action);
                    if ($action === '') {
                        continue;
                    }
                    $rows[] = [$bulk_id, $site_key_hash, $post_id, 'post', $action, 'queued', '', 0, '', '', $now, $now];
                    if (count($rows) >= $chunk_size) {
                        $inserted += $this->rankbot_bulk_insert_rows($table, $rows);
                        $rows = [];
                    }
                }
                if (!empty($rows)) {
                    $inserted += $this->rankbot_bulk_insert_rows($table, $rows);
                }

                $page++;
                $keep_running = ($page <= (int) $q->max_num_pages);
            }
        }

        if ($inserted < 1) {
            wp_send_json_error('No items queued');
        }

        if (function_exists('rankbot_log')) {
            rankbot_log('Bulk queued items', [
                'bulk_id' => (string) $bulk_id,
                'queued' => (int) $inserted,
                'bulk_action' => (string) $bulk_action,
            ]);
        }

        $this->rankbot_bulk_schedule_runner(2);

        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
        wp_send_json_success([
            'bulk_id' => (string) $bulk_id,
            'queued' => (int) $inserted,
        ]);
    }

    public function handle_ajax_bulk_overall_status(): void
    {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $summary = [
            'active_tasks' => 0,
            'processed' => 0,
            'queued' => 0,
            'errors_count' => 0,
            'total' => 0,
            'completed' => 0,
            'completed_real' => 0,
            'failed_real' => 0,
            'finished_real' => 0,
            'remaining' => 0,
            'dispatched' => 0,
            'retry_wait' => 0,
            'active_jobs' => 0,
            'cancelled' => 0,
            'pause_reason' => '',
            'pause_until' => 0,
        ];

        $tasks = [];

        $site_key_hash = $this->rankbot_site_key_hash();

        if ($this->rankbot_bulk_queue_available()) {
            $bulk_ids = $this->rankbot_bulk_get_active_bulk_ids($site_key_hash);
            if (!empty($bulk_ids)) {
                global $wpdb;
                $table = $this->rankbot_bulk_queue_table();
                $counts = [
                    'queued' => 0,
                    'dispatched' => 0,
                    'retry_wait' => 0,
                    'completed' => 0,
                    'cancelled' => 0,
                ];
                foreach ($bulk_ids as $bulk_id) {
                    if ($site_key_hash !== '') {
                        $rows = $wpdb->get_results(
                            $wpdb->prepare(
                                "SELECT status, COUNT(*) AS c FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND bulk_id = %s GROUP BY status",
                                $table,
                                $site_key_hash,
                                (string) $bulk_id
                            ),
                            ARRAY_A
                        );
                    } else {
                        $rows = $wpdb->get_results(
                            $wpdb->prepare(
                                "SELECT status, COUNT(*) AS c FROM %i WHERE site_key_hash = %s AND bulk_id = %s GROUP BY status",
                                $table,
                                $site_key_hash,
                                (string) $bulk_id
                            ),
                            ARRAY_A
                        );
                    }

                    if (is_array($rows)) {
                        foreach ($rows as $r) {
                            $st = isset($r['status']) ? (string) $r['status'] : '';
                            $c = isset($r['c']) ? (int) $r['c'] : 0;
                            if (isset($counts[$st])) {
                                $counts[$st] += $c;
                            }
                        }
                    }
                }

                $summary['active_tasks'] = count($bulk_ids);
                $summary['queued'] = $counts['queued'];
                $summary['dispatched'] = $counts['dispatched'];
                $summary['retry_wait'] = $counts['retry_wait'];
                // For progress UI, treat cancelled as "finished" so totals remain stable and the bar can reach 100%.
                $summary['completed'] = $counts['completed'] + $counts['cancelled'];
                $summary['cancelled'] = $counts['cancelled'];
                $summary['completed_real'] = $counts['completed'];
                $summary['finished_real'] = $counts['completed'] + $counts['cancelled'];
                $summary['failed_real'] = $counts['cancelled'];
                $summary['errors_count'] = $counts['retry_wait'];
                // Include cancelled in total so the denominator doesn't shrink when items hit max-attempts.
                $summary['total'] = $counts['queued'] + $counts['dispatched'] + $counts['retry_wait'] + $counts['completed'] + $counts['cancelled'];
                $summary['remaining'] = $counts['queued'] + $counts['dispatched'] + $counts['retry_wait'];
                $summary['processed'] = $counts['completed'] + $counts['cancelled'];
            }

            $pause_state = $this->rankbot_bulk_get_pause_state($site_key_hash);
            if (!empty($pause_state)) {
                $summary['pause_reason'] = (string) ($pause_state['reason'] ?? '');
                $summary['pause_until'] = (int) ($pause_state['until'] ?? 0);
            }
        }

        // Also report active jobs (single + bulk) for UI fallback.
        try {
            global $wpdb;
            $table_name = $wpdb->prefix . 'rankbot_jobs';
            if ($site_key_hash !== '') {
                $summary['active_jobs'] = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status IN ('processing','pending','queued') AND created_at > DATE_SUB(NOW(), INTERVAL 6 HOUR)",
                        $table_name,
                        $site_key_hash
                    )
                );
            } else {
                $summary['active_jobs'] = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM %i WHERE site_key_hash = %s AND status IN ('processing','pending','queued') AND created_at > DATE_SUB(NOW(), INTERVAL 6 HOUR)",
                        $table_name,
                        $site_key_hash
                    )
                );
            }
        } catch (\Exception $e) {
            $summary['active_jobs'] = 0;
        }

        // Ensure dispatcher keeps running while there is remaining work.
        if (!empty($summary['remaining']) && (int) $summary['remaining'] > 0) {
            $now_ts = (int) current_time('timestamp');
            $last_tick = $this->rankbot_bulk_get_last_tick($site_key_hash);
            $next = wp_next_scheduled('rankbot_bulk_process_task');
            $stale = ($last_tick > 0 && ($now_ts - $last_tick) > 120);
            if ($next && $next < ($now_ts - 60)) {
                $stale = true;
            }

            if (!$next || $stale) {
                $this->rankbot_bulk_schedule_runner(5);
            }

            if ($stale && function_exists('spawn_cron')) {
                spawn_cron();
            }
        }

        wp_send_json_success([
            'summary' => $summary,
            'tasks' => $tasks,
        ]);
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
    }

    public function handle_ajax_bulk_items_status(): void
    {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized below.
        $raw = isset($_POST['ids']) ? (array) wp_unslash($_POST['ids']) : [];
        $ids = [];
        foreach ($raw as $v) {
            $id = absint($v);
            if ($id > 0) $ids[] = $id;
        }
        $ids = array_values(array_unique($ids));
        $ids = array_slice($ids, 0, 300);

        if (empty($ids)) {
            wp_send_json_success(['items' => []]);
        }

        $items = [];
        $site_key_hash = $this->rankbot_site_key_hash();

        if ($this->rankbot_bulk_queue_available()) {
            global $wpdb;
            $table = $this->rankbot_bulk_queue_table();
            $rows = [];
            foreach ($ids as $object_id) {
                if ($site_key_hash !== '') {
                    $row = $wpdb->get_row(
                        $wpdb->prepare(
                            "SELECT id, object_id, status, job_id FROM %i
                                WHERE (site_key_hash = %s OR site_key_hash = '')
                                  AND object_type = 'post'
                                  AND object_id = %d
                                  AND status IN ('queued','retry_wait','dispatched')
                                ORDER BY id DESC
                                LIMIT 1",
                            $table,
                            $site_key_hash,
                            (int) $object_id
                        ),
                        ARRAY_A
                    );
                } else {
                    $row = $wpdb->get_row(
                        $wpdb->prepare(
                            "SELECT id, object_id, status, job_id FROM %i
                                WHERE site_key_hash = %s
                                  AND object_type = 'post'
                                  AND object_id = %d
                                  AND status IN ('queued','retry_wait','dispatched')
                                ORDER BY id DESC
                                LIMIT 1",
                            $table,
                            $site_key_hash,
                            (int) $object_id
                        ),
                        ARRAY_A
                    );
                }
                if (is_array($row)) {
                    $rows[] = $row;
                }
            }

            $job_ids = [];
            if (is_array($rows)) {
                foreach ($rows as $r) {
                    $oid = isset($r['object_id']) ? absint($r['object_id']) : 0;
                    if ($oid < 1) {
                        continue;
                    }
                    if (isset($items[(string) $oid])) {
                        continue;
                    }
                    $status = isset($r['status']) ? (string) $r['status'] : '';
                    $job_id = isset($r['job_id']) ? (string) $r['job_id'] : '';
                    $items[(string) $oid] = [
                        'status' => $status,
                        'job_id' => $job_id,
                    ];
                    if ($status === 'dispatched' && $job_id !== '') {
                        $job_ids[] = $job_id;
                    }
                }
            }

            if (!empty($job_ids)) {
                $job_ids = array_values(array_unique($job_ids));
                $jobs_table = $wpdb->prefix . 'rankbot_jobs';
                $phj = implode(',', array_fill(0, count($job_ids), '%s'));
                $job_rows = $wpdb->get_results(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $wpdb->prepare(
                        "SELECT job_id, status FROM %i WHERE job_id IN ($phj)",
                        ...array_merge([$jobs_table], $job_ids)
                    ),
                    ARRAY_A
                );
                $job_map = [];
                if (is_array($job_rows)) {
                    foreach ($job_rows as $jr) {
                        $jid = isset($jr['job_id']) ? (string) $jr['job_id'] : '';
                        $st = isset($jr['status']) ? (string) $jr['status'] : '';
                        if ($jid !== '') {
                            $job_map[$jid] = $st;
                        }
                    }
                }

                foreach ($items as $oid => $info) {
                    if (!is_array($info)) {
                        continue;
                    }
                    if ($info['status'] !== 'dispatched') {
                        continue;
                    }
                    $jid = (string) ($info['job_id'] ?? '');
                    if ($jid !== '' && isset($job_map[$jid])) {
                        $st = (string) $job_map[$jid];
                        // Reflect actual job status, including completion, so the row can unlock immediately.
                        if (in_array($st, ['queued', 'processing', 'pending', 'completed', 'failed', 'cancelled', 'error'], true)) {
                            $items[$oid]['status'] = $st;
                        }
                    }
                }
            }
        }

        $missing = [];
        foreach ($ids as $oid) {
            if (!isset($items[(string) $oid])) {
                $missing[] = $oid;
            }
        }

        if (!empty($missing)) {
            global $wpdb;
            $table = $wpdb->prefix . 'rankbot_jobs';
            if ($this->rankbot_db_table_exists($table)) {
                foreach ($missing as $object_id) {
                    if ($site_key_hash !== '') {
                        $r = $wpdb->get_row(
                            $wpdb->prepare(
                                "SELECT t.object_id, t.status, t.job_id FROM %i t
                                    INNER JOIN (
                                        SELECT object_id, MAX(id) AS id
                                        FROM %i
                                        WHERE (site_key_hash = %s OR site_key_hash = '')
                                          AND object_type = 'post'
                                          AND status IN ('processing','pending','queued')
                                          AND created_at > DATE_SUB(NOW(), INTERVAL 6 HOUR)
                                          AND object_id = %d
                                        GROUP BY object_id
                                    ) x ON t.id = x.id",
                                $table,
                                $table,
                                $site_key_hash,
                                (int) $object_id
                            ),
                            ARRAY_A
                        );
                    } else {
                        $r = $wpdb->get_row(
                            $wpdb->prepare(
                                "SELECT t.object_id, t.status, t.job_id FROM %i t
                                    INNER JOIN (
                                        SELECT object_id, MAX(id) AS id
                                        FROM %i
                                        WHERE site_key_hash = %s
                                          AND object_type = 'post'
                                          AND status IN ('processing','pending','queued')
                                          AND created_at > DATE_SUB(NOW(), INTERVAL 6 HOUR)
                                          AND object_id = %d
                                        GROUP BY object_id
                                    ) x ON t.id = x.id",
                                $table,
                                $table,
                                $site_key_hash,
                                (int) $object_id
                            ),
                            ARRAY_A
                        );
                    }

                    if (is_array($r) && !empty($r['object_id'])) {
                        $oid = absint($r['object_id']);
                        if ($oid > 0) {
                            $items[(string) $oid] = [
                                'status' => isset($r['status']) ? (string) $r['status'] : '',
                                'job_id' => isset($r['job_id']) ? (string) $r['job_id'] : '',
                            ];
                        }
                    }
                }
            }
        }

        wp_send_json_success(['items' => $items]);
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
    }

    public function handle_ajax_clear_process(): void
    {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        global $wpdb;
        $site_key_hash = $this->rankbot_site_key_hash();
        $queue_cleared = 0;
        $job_ids = [];
        $object_ids = [];

        if ($this->rankbot_bulk_queue_available()) {
            $queue_table = $this->rankbot_bulk_queue_table();
            if ($site_key_hash !== '') {
                $object_ids = $wpdb->get_col($wpdb->prepare(
                    "SELECT DISTINCT object_id FROM %i WHERE (site_key_hash = %s OR site_key_hash = '')",
                    $queue_table,
                    $site_key_hash
                ));
                $job_ids = $wpdb->get_col($wpdb->prepare(
                    "SELECT DISTINCT job_id FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND job_id <> ''",
                    $queue_table,
                    $site_key_hash
                ));
                $queue_cleared = (int) $wpdb->query($wpdb->prepare(
                    "DELETE FROM %i WHERE (site_key_hash = %s OR site_key_hash = '')",
                    $queue_table,
                    $site_key_hash
                ));
            } else {
                $object_ids = $wpdb->get_col($wpdb->prepare(
                    "SELECT DISTINCT object_id FROM %i WHERE site_key_hash = %s",
                    $queue_table,
                    $site_key_hash
                ));
                $job_ids = $wpdb->get_col($wpdb->prepare(
                    "SELECT DISTINCT job_id FROM %i WHERE site_key_hash = %s AND job_id <> ''",
                    $queue_table,
                    $site_key_hash
                ));
                $queue_cleared = (int) $wpdb->query($wpdb->prepare(
                    "DELETE FROM %i WHERE site_key_hash = %s",
                    $queue_table,
                    $site_key_hash
                ));
            }
        }

        $jobs_table = $wpdb->prefix . 'rankbot_jobs';
        if (!empty($object_ids) && $this->rankbot_db_table_exists($jobs_table)) {
            $object_ids = array_values(array_unique(array_filter(array_map('absint', $object_ids))));
            if (!empty($object_ids)) {
                foreach ($object_ids as $object_id) {
                    if ($site_key_hash !== '') {
                        $wpdb->query(
                            $wpdb->prepare(
                                "UPDATE %i
                                    SET status = 'cancelled'
                                    WHERE (site_key_hash = %s OR site_key_hash = '')
                                      AND object_type = 'post'
                                      AND status IN ('queued','processing','pending')
                                      AND object_id = %d",
                                $jobs_table,
                                $site_key_hash,
                                (int) $object_id
                            )
                        );
                    } else {
                        $wpdb->query(
                            $wpdb->prepare(
                                "UPDATE %i
                                    SET status = 'cancelled'
                                    WHERE site_key_hash = %s
                                      AND object_type = 'post'
                                      AND status IN ('queued','processing','pending')
                                      AND object_id = %d",
                                $jobs_table,
                                $site_key_hash,
                                (int) $object_id
                            )
                        );
                    }
                }
            }
        }

        if (!empty($job_ids)) {
            $job_ids = array_values(array_unique(array_filter(array_map('strval', $job_ids))));
            foreach ($job_ids as $job_id) {
                $this->rankbot_unschedule_job_poll((string) $job_id);
            }
        }

        delete_option($this->rankbot_bulk_dispatch_state_key($site_key_hash));
        $this->rankbot_bulk_clear_pause_state($site_key_hash);

        // Clear scheduled cron hooks
        wp_clear_scheduled_hook('rankbot_bulk_process_task');

        wp_send_json_success([
            'cleared' => true,
            'queue_cleared' => (int) $queue_cleared,
            'message' => 'Queue cleared completely.',
        ]);
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    public function handle_ajax_clear_all_jobs(): void
    {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        global $wpdb;
        $site_key_hash = $this->rankbot_site_key_hash();

        $queue_deleted = 0;
        $jobs_deleted = 0;

        // Collect job_ids for unscheduling pollers before deletion.
        $job_ids = [];

        // 1) Clear bulk queue table (if present)
        if ($this->rankbot_bulk_queue_available()) {
            $queue_table = $this->rankbot_bulk_queue_table();
            if ($site_key_hash !== '') {
                $qid = $wpdb->get_col($wpdb->prepare(
                    "SELECT DISTINCT job_id FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND job_id <> ''",
                    $queue_table,
                    $site_key_hash
                ));
            } else {
                $qid = $wpdb->get_col($wpdb->prepare(
                    "SELECT DISTINCT job_id FROM %i WHERE site_key_hash = %s AND job_id <> ''",
                    $queue_table,
                    $site_key_hash
                ));
            }
            if (is_array($qid) && !empty($qid)) {
                $job_ids = array_merge($job_ids, $qid);
            }

            if ($site_key_hash !== '') {
                $queue_deleted = (int) $wpdb->query($wpdb->prepare(
                    "DELETE FROM %i WHERE (site_key_hash = %s OR site_key_hash = '')",
                    $queue_table,
                    $site_key_hash
                ));
            } else {
                $queue_deleted = (int) $wpdb->query($wpdb->prepare(
                    "DELETE FROM %i WHERE site_key_hash = %s",
                    $queue_table,
                    $site_key_hash
                ));
            }
        }

        // 2) Clear jobs table (local history of server jobs)
        $jobs_table = $wpdb->prefix . 'rankbot_jobs';
        if ($this->rankbot_db_table_exists($jobs_table)) {
            if ($site_key_hash !== '') {
                $jid = $wpdb->get_col($wpdb->prepare(
                    "SELECT DISTINCT job_id FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND job_id <> ''",
                    $jobs_table,
                    $site_key_hash
                ));
            } else {
                $jid = $wpdb->get_col($wpdb->prepare(
                    "SELECT DISTINCT job_id FROM %i WHERE site_key_hash = %s AND job_id <> ''",
                    $jobs_table,
                    $site_key_hash
                ));
            }
            if (is_array($jid) && !empty($jid)) {
                $job_ids = array_merge($job_ids, $jid);
            }

            // NOTE: Delete all job records for this API key hash.
            // This resets "processed" UI state and allows re-enqueueing without dedupe blocks.
            if ($site_key_hash !== '') {
                $jobs_deleted = (int) $wpdb->query($wpdb->prepare(
                    "DELETE FROM %i WHERE (site_key_hash = %s OR site_key_hash = '')",
                    $jobs_table,
                    $site_key_hash
                ));
            } else {
                $jobs_deleted = (int) $wpdb->query($wpdb->prepare(
                    "DELETE FROM %i WHERE site_key_hash = %s",
                    $jobs_table,
                    $site_key_hash
                ));
            }
        }

        // 3) Unschedule any pending per-job poll hooks
        $job_ids = array_values(array_unique(array_filter(array_map('strval', (array) $job_ids))));
        foreach ($job_ids as $job_id) {
            $this->rankbot_unschedule_job_poll((string) $job_id);
        }

        // 4) Reset bulk state/pause + cron runner
        delete_option($this->rankbot_bulk_dispatch_state_key($site_key_hash));
        $this->rankbot_bulk_clear_pause_state($site_key_hash);
        wp_clear_scheduled_hook('rankbot_bulk_process_task');

        wp_send_json_success([
            'cleared' => true,
            'queue_deleted' => (int) $queue_deleted,
            'jobs_deleted' => (int) $jobs_deleted,
            'message' => 'All RankBot jobs were cleared for this site key.',
        ]);
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    public function handle_ajax_clear_local_history(): void
    {
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        if ($this->rankbot_history_storage_mode() !== 'wordpress') {
            wp_send_json_error(__('History is loaded from server. Local delete is disabled.', 'rankbotai-seo-optimizer'));
        }

        $deleted_postmeta = (int) delete_metadata('post', 0, '_rankbot_history', '', true);
        $deleted_termmeta = (int) delete_metadata('term', 0, '_rankbot_history', '', true);

        wp_send_json_success([
            'cleared' => true,
            'deleted_postmeta' => (int) $deleted_postmeta,
            'deleted_termmeta' => (int) $deleted_termmeta,
        ]);
    }

    public function handle_ajax_bulk_stop_all(): void
    {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        if (!$this->rankbot_bulk_queue_available()) {
            wp_send_json_success(['stopped' => 0]);
        }

        global $wpdb;
        $site_key_hash = $this->rankbot_site_key_hash();
        $this->rankbot_bulk_clear_pause_state($site_key_hash);
        $queue_table = $this->rankbot_bulk_queue_table();
        $now = current_time('mysql');

        // Cancel queued and retry-wait items, but keep dispatched ones running.
        if ($site_key_hash !== '') {
            $stopped = (int) $wpdb->query(
                $wpdb->prepare(
                    "UPDATE %i
                        SET status = 'cancelled',
                            job_id = '',
                            next_attempt_at = NULL,
                            last_error = %s,
                            updated_at = %s
                        WHERE (site_key_hash = %s OR site_key_hash = '')
                          AND status IN ('queued','retry_wait')",
                    $queue_table,
                    'stopped_by_user',
                    $now,
                    $site_key_hash
                )
            );
        } else {
            $stopped = (int) $wpdb->query(
                $wpdb->prepare(
                    "UPDATE %i
                        SET status = 'cancelled',
                            job_id = '',
                            next_attempt_at = NULL,
                            last_error = %s,
                            updated_at = %s
                        WHERE site_key_hash = %s
                          AND status IN ('queued','retry_wait')",
                    $queue_table,
                    'stopped_by_user',
                    $now,
                    $site_key_hash
                )
            );
        }

        // Keep polling for dispatched jobs to complete.
        $in_flight = 0;
        if ($site_key_hash !== '') {
            $in_flight = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status = 'dispatched'",
                    $queue_table,
                    $site_key_hash
                )
            );
        } else {
            $in_flight = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM %i WHERE site_key_hash = %s AND status = 'dispatched'",
                    $queue_table,
                    $site_key_hash
                )
            );
        }
        if ($in_flight > 0) {
            $this->rankbot_bulk_schedule_runner(10);
        }

        wp_send_json_success([
            'stopped' => (int) $stopped,
            'remaining' => (int) $in_flight,
            'message' => 'Queue paused. Waiting for dispatched items to finish.',
        ]);
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    public function handle_ajax_bulk_status(): void
    {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        check_ajax_referer('rankbot_optimize', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $bulk_id = isset($_POST['task_id']) ? sanitize_text_field((string) wp_unslash($_POST['task_id'])) : '';
        if ($bulk_id === '') {
            wp_send_json_error('Missing task_id');
        }
        if (!$this->rankbot_bulk_queue_available()) {
            wp_send_json_error('Bulk queue table is not available');
        }

        global $wpdb;
        $site_key_hash = $this->rankbot_site_key_hash();
        $queue_table = $this->rankbot_bulk_queue_table();
        if ($site_key_hash !== '') {
            $rows = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT status, COUNT(*) AS c FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND bulk_id = %s GROUP BY status",
                    $queue_table,
                    $site_key_hash,
                    $bulk_id
                ),
                ARRAY_A
            );
        } else {
            $rows = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT status, COUNT(*) AS c FROM %i WHERE site_key_hash = %s AND bulk_id = %s GROUP BY status",
                    $queue_table,
                    $site_key_hash,
                    $bulk_id
                ),
                ARRAY_A
            );
        }

        $counts = [
            'queued' => 0,
            'dispatched' => 0,
            'retry_wait' => 0,
            'completed' => 0,
            'cancelled' => 0,
        ];
        if (is_array($rows)) {
            foreach ($rows as $r) {
                $st = isset($r['status']) ? (string) $r['status'] : '';
                $c = isset($r['c']) ? (int) $r['c'] : 0;
                if (isset($counts[$st])) {
                    $counts[$st] += $c;
                }
            }
        }

        $total = $counts['queued'] + $counts['dispatched'] + $counts['retry_wait'] + $counts['completed'];
        $remaining = $counts['queued'] + $counts['dispatched'] + $counts['retry_wait'];

        wp_send_json_success([
            'task' => [
                'bulk_id' => (string) $bulk_id,
                'queued' => (int) $counts['queued'],
                'dispatched' => (int) $counts['dispatched'],
                'retry_wait' => (int) $counts['retry_wait'],
                'completed' => (int) $counts['completed'],
                'cancelled' => (int) $counts['cancelled'],
                'total' => (int) $total,
                'remaining' => (int) $remaining,
            ],
        ]);
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    public function handle_cron_bulk_process_task(string $task_id = ''): void
    {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        if (!$this->rankbot_bulk_queue_available()) {
            return;
        }

        // phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Needed for long-running cron processing.
        @set_time_limit(0);
        if (function_exists('ignore_user_abort')) {
            ignore_user_abort(true);
        }

        global $wpdb;
        $queue_table = $this->rankbot_bulk_queue_table();
        $jobs_table = $wpdb->prefix . 'rankbot_jobs';
        $site_key_hash = $this->rankbot_site_key_hash();

        if ($this->rankbot_bulk_remaining_count($site_key_hash) < 1) {
            wp_clear_scheduled_hook('rankbot_bulk_process_task');
            return;
        }

        $lock_name = 'rankbot_bulk_runner_' . ($site_key_hash !== '' ? $site_key_hash : 'default');
        if (strlen($lock_name) > 60) {
            $lock_name = 'rankbot_bulk_runner_' . sha1($lock_name);
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $got_lock = $wpdb->get_var($wpdb->prepare('SELECT GET_LOCK(%s, 0)', $lock_name));
        if ((string) $got_lock !== '1') {
            return;
        }

        try {
            $bulk_ids = $this->rankbot_bulk_get_active_bulk_ids($site_key_hash);
            if (empty($bulk_ids)) {
                return;
            }

            $now = current_time('mysql');
            $now_ts = (int) current_time('timestamp');
            $dispatched_new = 0;
            $errors = 0;
            $start = microtime(true);
            $budget = (int) apply_filters('rankbot_bulk_dispatch_budget', 15);
            if ($budget < 3) {
                $budget = 3;
            }
            $time_exceeded = false;
            // Default to finite retries to prevent endless retry loops.
            $max_attempts = (int) apply_filters('rankbot_bulk_max_attempts', 3);
            if ($max_attempts < 0) {
                $max_attempts = 0;
            }

            $this->rankbot_bulk_set_last_tick($site_key_hash, $now_ts);
            // Schedule early to survive unexpected timeouts.
            $this->rankbot_bulk_schedule_runner(30);

            // Sync dispatched rows with job results (limited batch).
            if ($this->rankbot_db_table_exists($jobs_table)) {
                $stuck_seconds = (int) apply_filters('rankbot_bulk_stuck_seconds', 420);
                if ($stuck_seconds < 120) {
                    $stuck_seconds = 120;
                }

                $forced_poll_after = (int) apply_filters('rankbot_bulk_forced_poll_after_seconds', 60);
                if ($forced_poll_after < 20) {
                    $forced_poll_after = 20;
                }
                $forced_poll_limit = (int) apply_filters('rankbot_bulk_forced_poll_limit', 5);
                if ($forced_poll_limit < 0) {
                    $forced_poll_limit = 0;
                }
                $forced_polls = 0;

                if ($site_key_hash !== '') {
                    $rows = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT id, object_id, job_id, attempts FROM %i
                                WHERE (site_key_hash = %s OR site_key_hash = '')
                                  AND status = 'dispatched'
                                  AND job_id <> ''
                                ORDER BY id ASC
                                LIMIT 200",
                            $queue_table,
                            $site_key_hash
                        ),
                        ARRAY_A
                    );
                } else {
                    $rows = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT id, object_id, job_id, attempts FROM %i
                                WHERE site_key_hash = %s
                                  AND status = 'dispatched'
                                  AND job_id <> ''
                                ORDER BY id ASC
                                LIMIT 200",
                            $queue_table,
                            $site_key_hash
                        ),
                        ARRAY_A
                    );
                }

                $job_ids = [];
                if (is_array($rows)) {
                    foreach ($rows as $row) {
                        $jid = isset($row['job_id']) ? (string) $row['job_id'] : '';
                        if ($jid !== '') {
                            $job_ids[] = $jid;
                        }
                    }
                }
                $job_ids = array_values(array_unique(array_filter($job_ids)));

                $job_map = [];
                if (!empty($job_ids)) {
                    $jobs_has_started = false;
                    $jobs_has_started = $this->rankbot_db_column_exists($jobs_table, 'started_at');

                    foreach ($job_ids as $jid) {
                        if ($jobs_has_started) {
                            $jr = $wpdb->get_row(
                                $wpdb->prepare(
                                    "SELECT job_id, status, created_at, started_at FROM %i WHERE job_id = %s",
                                    $jobs_table,
                                    (string) $jid
                                ),
                                ARRAY_A
                            );
                        } else {
                            $jr = $wpdb->get_row(
                                $wpdb->prepare(
                                    "SELECT job_id, status, created_at FROM %i WHERE job_id = %s",
                                    $jobs_table,
                                    (string) $jid
                                ),
                                ARRAY_A
                            );
                        }
                        if (is_array($jr) && !empty($jr['job_id'])) {
                            $job_map[(string) $jr['job_id']] = $jr;
                        }
                    }
                }

                if (is_array($rows)) {
                    foreach ($rows as $row) {
                        $queue_id = isset($row['id']) ? (int) $row['id'] : 0;
                        $object_id = isset($row['object_id']) ? absint($row['object_id']) : 0;
                        $job_id = isset($row['job_id']) ? (string) $row['job_id'] : '';
                        $attempts = isset($row['attempts']) ? (int) $row['attempts'] : 0;
                        if ($queue_id < 1 || $job_id === '') {
                            continue;
                        }

                        $job_row = isset($job_map[$job_id]) && is_array($job_map[$job_id]) ? $job_map[$job_id] : null;
                        $job_status = is_array($job_row) && isset($job_row['status']) ? (string) $job_row['status'] : '';
                        if ($job_status === '') {
                            $attempts++;
                            $delay = $this->rankbot_bulk_next_retry_delay($attempts);
                            $next_attempt_at = gmdate('Y-m-d H:i:s', $now_ts + $delay);

                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->update(
                                $queue_table,
                                [
                                    'status' => 'retry_wait',
                                    'attempts' => $attempts,
                                    'next_attempt_at' => $next_attempt_at,
                                    'job_id' => '',
                                    'last_error' => 'job_missing',
                                    'updated_at' => $now,
                                ],
                                ['id' => $queue_id, 'status' => 'dispatched']
                            );
                            $errors++;
                            $this->rankbot_unschedule_job_poll($job_id);
                            continue;
                        }

                        // Detect stuck jobs and auto-retry with backup restore.
                        if (in_array($job_status, ['queued', 'processing', 'pending'], true)) {
                            $started_at = is_array($job_row) ? (string) ($job_row['started_at'] ?? '') : '';
                            $created_at = is_array($job_row) ? (string) ($job_row['created_at'] ?? '') : '';
                            $start_ts = $started_at !== '' ? (int) strtotime($started_at) : 0;
                            if ($start_ts < 1 && $created_at !== '') {
                                $start_ts = (int) strtotime($created_at);
                            }

                            // If local status stays active for a while, force a remote check to sync faster
                            // (WP-Cron may be delayed; this keeps Bulk responsive).
                            if (
                                $forced_poll_limit > 0 &&
                                $forced_polls < $forced_poll_limit &&
                                $start_ts > 0 &&
                                ($now_ts - $start_ts) >= $forced_poll_after
                            ) {
                                $forced_polls++;
                                $this->handle_cron_poll_job($job_id);
                            }

                            if ($start_ts > 0 && ($now_ts - $start_ts) > $stuck_seconds) {
                                // Restore pre-run snapshot if available.
                                $backup_ts = (int) get_transient('rankbot_job_backup_ts_' . $job_id);
                                if ($backup_ts > 0 && $object_id > 0) {
                                    $this->rankbot_restore_backup_snapshot($object_id, $backup_ts);
                                    delete_transient('rankbot_job_backup_ts_' . $job_id);
                                }

                                $attempts++;
                                $delay = $this->rankbot_bulk_next_retry_delay($attempts);
                                $next_attempt_at = gmdate('Y-m-d H:i:s', $now_ts + $delay);

                                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                                $wpdb->update(
                                    $queue_table,
                                    [
                                        'status' => 'retry_wait',
                                        'attempts' => $attempts,
                                        'next_attempt_at' => $next_attempt_at,
                                        'job_id' => '',
                                        'last_error' => 'stuck_timeout',
                                        'updated_at' => $now,
                                    ],
                                    ['id' => $queue_id, 'status' => 'dispatched']
                                );
                                $errors++;
                                $this->rankbot_unschedule_job_poll($job_id);
                                delete_transient('rankbot_job_attempts_' . $job_id);
                                continue;
                            }
                        }

                        if ($job_status === 'completed') {
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->update(
                                $queue_table,
                                [
                                    'status' => 'completed',
                                    'updated_at' => $now,
                                ],
                                ['id' => $queue_id, 'status' => 'dispatched']
                            );
                            continue;
                        }

                        if (in_array($job_status, ['failed', 'error', 'cancelled'], true)) {
                            // Restore pre-run snapshot if available.
                            $backup_ts = (int) get_transient('rankbot_job_backup_ts_' . $job_id);
                            if ($backup_ts > 0 && $object_id > 0) {
                                $this->rankbot_restore_backup_snapshot($object_id, $backup_ts);
                                delete_transient('rankbot_job_backup_ts_' . $job_id);
                            }

                            $attempts++;
                            $delay = $this->rankbot_bulk_next_retry_delay($attempts);
                            $next_attempt_at = gmdate('Y-m-d H:i:s', $now_ts + $delay);

                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->update(
                                $queue_table,
                                [
                                    'status' => 'retry_wait',
                                    'attempts' => $attempts,
                                    'next_attempt_at' => $next_attempt_at,
                                    'job_id' => '',
                                    'last_error' => 'job_' . $job_status,
                                    'updated_at' => $now,
                                ],
                                ['id' => $queue_id, 'status' => 'dispatched']
                            );
                            $errors++;
                            $this->rankbot_unschedule_job_poll($job_id);
                            continue;
                        }
                    }
                }
            }

            // Respect pause state (e.g., insufficient balance) before dispatching new work.
            $pause_state = $this->rankbot_bulk_get_pause_state($site_key_hash);
            $pause_until = isset($pause_state['until']) ? (int) $pause_state['until'] : 0;
            if ($pause_until > $now_ts) {
                $delay = max(10, min(300, $pause_until - $now_ts));
                $this->rankbot_bulk_schedule_runner($delay);
                return;
            }

            // Compute free slots based on adaptive limit.
            if ($site_key_hash !== '') {
                $in_flight = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status = 'dispatched'",
                        $queue_table,
                        $site_key_hash
                    )
                );
            } else {
                $in_flight = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM %i WHERE site_key_hash = %s AND status = 'dispatched'",
                        $queue_table,
                        $site_key_hash
                    )
                );
            }

            // Additional per-action concurrency limits.
            // Post content generation (post auto -> complex on posts) can be less reliable under high parallelism
            // (timeouts, provider rate limits, transient upstream issues). Keep it sequential by default
            // to match single-run reliability.
            $selected_model = get_option('rankbot_selected_model', '');
            $selected_model = is_string($selected_model) ? trim($selected_model) : '';
            $selected_model_lc = strtolower($selected_model);
            $is_oss_model = false;
            if ($selected_model_lc !== '') {
                $is_oss_model = str_starts_with($selected_model_lc, 'rankbot-ai-oss')
                    || (bool) preg_match('/(^|[^a-z0-9])oss([^a-z0-9]|$)/', $selected_model_lc);
            }
            $is_oss_model = (bool) apply_filters('rankbot_is_oss_model', $is_oss_model, $selected_model, $site_key_hash);

            $post_optimize_inflight_default = 1;
            $post_optimize_inflight_limit = (int) apply_filters(
                'rankbot_bulk_post_optimize_inflight_limit',
                $post_optimize_inflight_default,
                $site_key_hash,
                $selected_model,
                $is_oss_model
            );
            if ($post_optimize_inflight_limit < 1) {
                $post_optimize_inflight_limit = 1;
            }
            $post_optimize_inflight = 0;
            if ($post_optimize_inflight_limit > 0 && $in_flight > 0) {
                if ($site_key_hash !== '') {
                    $inflight_rows = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT object_id, action FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status = 'dispatched' ORDER BY id ASC LIMIT 200",
                            $queue_table,
                            $site_key_hash
                        ),
                        ARRAY_A
                    );
                } else {
                    $inflight_rows = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT object_id, action FROM %i WHERE site_key_hash = %s AND status = 'dispatched' ORDER BY id ASC LIMIT 200",
                            $queue_table,
                            $site_key_hash
                        ),
                        ARRAY_A
                    );
                }
                if (is_array($inflight_rows)) {
                    foreach ($inflight_rows as $r) {
                        $oid = isset($r['object_id']) ? absint($r['object_id']) : 0;
                        if ($oid < 1) {
                            continue;
                        }
                        $p = get_post($oid);
                        if (!$p) {
                            continue;
                        }
                        $bulk_action = isset($r['action']) ? (string) $r['action'] : '';
                        if ($bulk_action === '') {
                            continue;
                        }
                        $resolved = $this->rankbot_resolve_bulk_action((string) $p->post_type, $bulk_action);
                        $is_post_content_job = ($p->post_type === 'post') && in_array($resolved, ['post_optimize', 'complex'], true);
                        if ($is_post_content_job) {
                            $post_optimize_inflight++;
                            if ($post_optimize_inflight >= $post_optimize_inflight_limit) {
                                break;
                            }
                        }
                    }
                }
            }
            $limit_state = $this->rankbot_bulk_get_dispatch_state($site_key_hash);
            $limit = isset($limit_state['limit']) ? (int) $limit_state['limit'] : 1;
            $available = max(0, $limit - $in_flight);

            if ($available > 0) {
                if ($site_key_hash !== '') {
                    $rows = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT id, bulk_id, object_id, action, attempts FROM %i
                                WHERE (site_key_hash = %s OR site_key_hash = '')
                                  AND status IN ('queued','retry_wait')
                                  AND (status = 'queued' OR next_attempt_at IS NULL OR next_attempt_at <= %s)
                                ORDER BY (CASE WHEN status = 'queued' THEN 0 ELSE 1 END), COALESCE(next_attempt_at, updated_at) ASC, id ASC
                                LIMIT %d",
                            $queue_table,
                            $site_key_hash,
                            $now,
                            $available
                        ),
                        ARRAY_A
                    );
                } else {
                    $rows = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT id, bulk_id, object_id, action, attempts FROM %i
                                WHERE site_key_hash = %s
                                  AND status IN ('queued','retry_wait')
                                  AND (status = 'queued' OR next_attempt_at IS NULL OR next_attempt_at <= %s)
                                ORDER BY (CASE WHEN status = 'queued' THEN 0 ELSE 1 END), COALESCE(next_attempt_at, updated_at) ASC, id ASC
                                LIMIT %d",
                            $queue_table,
                            $site_key_hash,
                            $now,
                            $available
                        ),
                        ARRAY_A
                    );
                }

                if (is_array($rows)) {
                    foreach ($rows as $row) {
                        if ((microtime(true) - $start) >= $budget) {
                            $time_exceeded = true;
                            break;
                        }

                        $queue_id = isset($row['id']) ? (int) $row['id'] : 0;
                        $bulk_id = isset($row['bulk_id']) ? (string) $row['bulk_id'] : '';
                        $object_id = isset($row['object_id']) ? absint($row['object_id']) : 0;
                        $action = isset($row['action']) ? (string) $row['action'] : '';
                        $attempts = isset($row['attempts']) ? (int) $row['attempts'] : 0;

                        if ($queue_id < 1 || $object_id < 1 || $action === '') {
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->update(
                                $queue_table,
                                [
                                    'status' => 'cancelled',
                                    'last_error' => 'invalid_row',
                                    'updated_at' => $now,
                                ],
                                ['id' => $queue_id]
                            );
                            continue;
                        }

                        $post = get_post($object_id);
                        if (!$post) {
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->update(
                                $queue_table,
                                [
                                    'status' => 'cancelled',
                                    'last_error' => 'post_missing',
                                    'updated_at' => $now,
                                ],
                                ['id' => $queue_id]
                            );
                            continue;
                        }

                        // Respect per-action in-flight limits (keep post content generation sequential by default).
                        if ($post_optimize_inflight_limit > 0 && $post_optimize_inflight >= $post_optimize_inflight_limit) {
                            $resolved_action = $this->rankbot_resolve_bulk_action((string) $post->post_type, $action);
                            $is_post_content_job = ($post->post_type === 'post') && in_array($resolved_action, ['post_optimize', 'complex'], true);
                            if ($is_post_content_job) {
                                continue;
                            }
                        }

                        $request_id = '';
                        if ($bulk_id !== '') {
                            $request_id = 'rb_' . sha1($site_key_hash . '|' . $bulk_id . '|' . $action . '|' . $object_id);
                        }

                        $err = null;
                        $job_id = $this->rankbot_bulk_queue_for_post($object_id, $action, 0, $err, $request_id);
                        if ($job_id === 'sync') {
                            $this->rankbot_bulk_clear_pause_state($site_key_hash);
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->update(
                                $queue_table,
                                [
                                    'status' => 'completed',
                                    'job_id' => '',
                                    'updated_at' => $now,
                                ],
                                ['id' => $queue_id]
                            );
                            $dispatched_new++;
                            continue;
                        }

                        if ($job_id !== '') {
                            $this->rankbot_bulk_clear_pause_state($site_key_hash);
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->update(
                                $queue_table,
                                [
                                    'status' => 'dispatched',
                                    'job_id' => (string) $job_id,
                                    'updated_at' => $now,
                                ],
                                ['id' => $queue_id]
                            );

                            // Track in-flight post content jobs to enforce sequential processing.
                            $resolved_action = $this->rankbot_resolve_bulk_action((string) $post->post_type, $action);
                            $is_post_content_job = ($post->post_type === 'post') && in_array($resolved_action, ['post_optimize', 'complex'], true);
                            if ($is_post_content_job) {
                                $post_optimize_inflight++;
                            }
                            $dispatched_new++;
                            continue;
                        }

                        $err_type = is_array($err) ? (string) ($err['type'] ?? '') : '';
                        if ($err_type === '') {
                            $err_type = 'dispatch_failed';
                        }

                        if ($err_type === 'model_not_allowed') {
                            // Fatal: do not retry. The user must change plan/model.
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->update(
                                $queue_table,
                                [
                                    'status' => 'cancelled',
                                    'last_error' => $err_type,
                                    'updated_at' => $now,
                                ],
                                ['id' => $queue_id]
                            );
                            $errors++;
                            continue;
                        }

                        if (in_array($err_type, ['insufficient_balance', 'not_connected'], true)) {
                            $pause_for = (int) apply_filters('rankbot_bulk_pause_delay', 600, $err_type, $err);
                            if ($pause_for < 60) {
                                $pause_for = 60;
                            }
                            $pause_until = $now_ts + $pause_for;
                            $this->rankbot_bulk_set_pause_state($site_key_hash, [
                                'until' => $pause_until,
                                'reason' => $err_type,
                            ]);

                            $next_attempt_at = gmdate('Y-m-d H:i:s', $pause_until);
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->update(
                                $queue_table,
                                [
                                    'status' => 'retry_wait',
                                    'attempts' => $attempts,
                                    'next_attempt_at' => $next_attempt_at,
                                    'job_id' => '',
                                    'last_error' => $err_type,
                                    'updated_at' => $now,
                                ],
                                ['id' => $queue_id]
                            );
                            $errors++;
                            $time_exceeded = true;
                            break;
                        }

                        $attempts++;
                        if ($max_attempts > 0 && $attempts >= $max_attempts) {
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->update(
                                $queue_table,
                                [
                                    'status' => 'cancelled',
                                    'attempts' => $attempts,
                                    'next_attempt_at' => null,
                                    'job_id' => '',
                                    'last_error' => 'max_attempts',
                                    'updated_at' => $now,
                                ],
                                ['id' => $queue_id]
                            );
                            $errors++;
                            continue;
                        }

                        $delay = $this->rankbot_bulk_next_retry_delay($attempts);
                        $next_attempt_at = gmdate('Y-m-d H:i:s', $now_ts + $delay);

                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update(
                            $queue_table,
                            [
                                'status' => 'retry_wait',
                                'attempts' => $attempts,
                                'next_attempt_at' => $next_attempt_at,
                                'job_id' => '',
                                'last_error' => $err_type,
                                'updated_at' => $now,
                            ],
                            ['id' => $queue_id]
                        );
                        $errors++;
                    }
                }
            }

            if ($dispatched_new > 0 || $errors > 0) {
                $this->rankbot_bulk_adjust_dispatch_limit($site_key_hash, $dispatched_new, $errors);
            }

            // Schedule next tick if there is still work.
            if ($site_key_hash !== '') {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $remaining = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status IN ('queued','retry_wait','dispatched')",
                        $queue_table,
                        $site_key_hash
                    )
                );
            } else {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $remaining = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM %i WHERE site_key_hash = %s AND status IN ('queued','retry_wait','dispatched')",
                        $queue_table,
                        $site_key_hash
                    )
                );
            }

            if ($remaining > 0) {
                if ($site_key_hash !== '') {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $queued_count = (int) $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status = 'queued'",
                            $queue_table,
                            $site_key_hash
                        )
                    );
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $ready_retry = (int) $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM %i
                                WHERE (site_key_hash = %s OR site_key_hash = '')
                                  AND status = 'retry_wait'
                                  AND next_attempt_at IS NOT NULL
                                  AND next_attempt_at <= %s",
                            $queue_table,
                            $site_key_hash,
                            $now
                        )
                    );
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $next_retry = $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT MIN(next_attempt_at) FROM %i
                                WHERE (site_key_hash = %s OR site_key_hash = '')
                                  AND status = 'retry_wait'
                                  AND next_attempt_at IS NOT NULL",
                            $queue_table,
                            $site_key_hash
                        )
                    );
                } else {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $queued_count = (int) $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM %i WHERE site_key_hash = %s AND status = 'queued'",
                            $queue_table,
                            $site_key_hash
                        )
                    );
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $ready_retry = (int) $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM %i
                                WHERE site_key_hash = %s
                                  AND status = 'retry_wait'
                                  AND next_attempt_at IS NOT NULL
                                  AND next_attempt_at <= %s",
                            $queue_table,
                            $site_key_hash,
                            $now
                        )
                    );
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $next_retry = $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT MIN(next_attempt_at) FROM %i
                                WHERE site_key_hash = %s
                                  AND status = 'retry_wait'
                                  AND next_attempt_at IS NOT NULL",
                            $queue_table,
                            $site_key_hash
                        )
                    );
                }

                $delay = 20;
                if ($queued_count > 0 || $ready_retry > 0) {
                    $delay = 5;
                } elseif (is_string($next_retry) && $next_retry !== '') {
                    $next_ts = strtotime($next_retry);
                    if ($next_ts && $next_ts > $now_ts) {
                        $delay = max(5, (int) ($next_ts - $now_ts));
                        $delay = min(300, $delay);
                    }
                }

                if ($time_exceeded) {
                    $delay = 5;
                }

                $this->rankbot_bulk_schedule_runner($delay);
            }
        } finally {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->get_var($wpdb->prepare('SELECT RELEASE_LOCK(%s)', $lock_name));
        }
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    private function rankbot_bulk_queue_for_post(int $post_id, string $bulk_action, int $initiator_user_id = 0, ?array &$error = null, string $request_id = ''): string
    {
        $post = get_post($post_id);
        if (!$post) return '';

        $error = null;

        $site_key_hash = $this->rankbot_site_key_hash();

        // During WP-Cron, there is no logged-in user; use initiator for permissions.
        if ($initiator_user_id > 0) {
            if (!user_can($initiator_user_id, 'edit_post', $post_id)) {
                if (function_exists('rankbot_log')) {
                    rankbot_log('Bulk queue denied: user cannot edit post', [
                        'post_id' => (int) $post_id,
                        'user_id' => (int) $initiator_user_id,
                    ], 'warning');
                }
                return '';
            }
        }

        $post_type = (string) $post->post_type;
        $action = $this->rankbot_resolve_action($post_type, $bulk_action);
        if ($action === '') return '';

        $params = $this->rankbot_build_params_for_bulk($post_id, $post, $action);
        if (!is_array($params) || empty($params)) return '';
        if ($request_id !== '') {
            $params['request_id'] = (string) $request_id;
        }

        // Strict de-duplication under concurrency: only one request may enqueue/generate per (site_key_hash, post_id).
        // This prevents double-clicks / parallel cron runs from sending the same post multiple times to the API.
        $lock_name = 'rankbot_job_' . $site_key_hash . '_' . (string) $post_id;
        if (strlen($lock_name) > 60) {
            $lock_name = 'rankbot_job_' . sha1($lock_name);
        }

        $got_lock = wp_cache_add($lock_name, 1, 'rankbotai-seo-optimizer', 10);
        if (!$got_lock) {
            return '';
        }

        try {
            // Prevent duplicate queueing: if there is already an active job for this object, reuse it.
            $existing_job_id = (string) get_post_meta($post_id, '_rankbot_last_job_id', true);
            $existing_status = (string) get_post_meta($post_id, '_rankbot_last_job_status', true);
            $existing_site_hash = (string) get_post_meta($post_id, '_rankbot_last_job_site_key_hash', true);
            if (
                $existing_job_id !== ''
                && in_array($existing_status, ['queued', 'processing', 'pending'], true)
                && ($site_key_hash === '' || $existing_site_hash === '' || hash_equals($existing_site_hash, $site_key_hash))
            ) {
                $this->schedule_job_poll($existing_job_id);
                return $existing_job_id;
            }

            $bulk_timeout = (int) apply_filters('rankbot_bulk_generate_timeout', 25, $action, $params, $post_id);
            $dispatch = $this->rankbot_dispatch_generate_and_register_post_job($post_id, $action, $params, [
                'timeout' => $bulk_timeout,
            ]);

            if (isset($dispatch['error'])) {
                $msg = (string) $dispatch['error'];
                $http_code = (int) ($dispatch['http_code'] ?? 0);
                $err_type = (string) ($dispatch['error_type'] ?? '');
                $err_code = (string) ($dispatch['error_code'] ?? '');

                $resolved = 'dispatch_failed';
                if ($http_code === 402 || stripos($msg, 'insufficient balance') !== false) {
                    $resolved = 'insufficient_balance';
                } elseif (
                    $http_code === 403
                    && (stripos($msg, 'model is not available') !== false || stripos($msg, 'premium model') !== false)
                ) {
                    $resolved = 'model_not_allowed';
                } elseif ($http_code === 401 || $http_code === 403 || stripos($msg, 'not connected') !== false) {
                    $resolved = 'not_connected';
                } elseif ($err_type === 'wp_error' && (stripos($msg, 'cURL error 28') !== false || stripos($msg, 'timed out') !== false)) {
                    $resolved = 'timeout';
                }

                $error = [
                    'type' => $resolved,
                    'message' => $msg,
                    'http_code' => $http_code,
                    'error_type' => $err_type,
                    'error_code' => $err_code,
                ];

                return '';
            }

            if (isset($dispatch['mode']) && $dispatch['mode'] === 'async' && !empty($dispatch['job_id'])) {
                return (string) $dispatch['job_id'];
            }

            if (isset($dispatch['mode']) && $dispatch['mode'] === 'sync' && isset($dispatch['result']) && is_array($dispatch['result'])) {
                $payload = isset($dispatch['result']['result']) ? $dispatch['result']['result'] : $dispatch['result'];
                $this->apply_optimization_result($post_id, $payload, $action, '');
                return 'sync';
            }
        } finally {
            wp_cache_delete($lock_name, 'rankbotai-seo-optimizer');
        }

        return '';
    }

    private function rankbot_restore_backup_snapshot(int $post_id, int $backup_ts): bool
    {
        if ($post_id < 1 || $backup_ts < 1) return false;

        $backups = get_post_meta($post_id, '_rankbot_backups', true);
        if (!is_array($backups) || empty($backups)) return false;

        $target = null;
        foreach ($backups as $b) {
            if (isset($b['timestamp']) && (int) $b['timestamp'] === (int) $backup_ts && isset($b['data']) && is_array($b['data'])) {
                $target = $b['data'];
                break;
            }
        }
        if (!is_array($target)) return false;

        $update_data = [
            'ID' => $post_id,
            'post_title' => isset($target['post_title']) ? (string) $target['post_title'] : '',
            'post_content' => isset($target['post_content']) ? (string) $target['post_content'] : '',
            'post_excerpt' => isset($target['post_excerpt']) ? (string) $target['post_excerpt'] : '',
        ];
        if (isset($target['post_name'])) {
            $update_data['post_name'] = (string) $target['post_name'];
        }

        $updateRes = wp_update_post(wp_slash($update_data), true);
        if (is_wp_error($updateRes)) {
            if (function_exists('rankbot_log')) {
                rankbot_log('Auto-restore backup failed', [
                    'post_id' => (int) $post_id,
                    'backup_ts' => (int) $backup_ts,
                    'wp_error' => $updateRes->get_error_message(),
                ], 'error');
            }
            return false;
        }

        clean_post_cache($post_id);

        $known_meta_keys = [
            '_yoast_wpseo_title',
            '_yoast_wpseo_metadesc',
            '_yoast_wpseo_focuskw',
            'rank_math_title',
            'rank_math_description',
            'rank_math_focus_keyword',
            '_rankbot_focus_keyword',
            '_rankbot_seo_score',
            '_rankbot_history',
        ];

        $backup_meta = [];
        if (isset($target['meta']) && is_array($target['meta'])) {
            $backup_meta = $target['meta'];
        }

        foreach ($known_meta_keys as $key) {
            if (array_key_exists($key, $backup_meta)) {
                update_post_meta($post_id, $key, $backup_meta[$key]);
            } else {
                delete_post_meta($post_id, $key);
            }
        }

        if (isset($target['image_alts']) && is_array($target['image_alts'])) {
            foreach ($target['image_alts'] as $img_id => $alt) {
                $img_id = absint($img_id);
                if ($img_id < 1) continue;

                $alt = is_string($alt) ? $alt : '';
                $alt = sanitize_text_field($alt);
                if ($alt === '') {
                    delete_post_meta($img_id, '_wp_attachment_image_alt');
                } else {
                    update_post_meta($img_id, '_wp_attachment_image_alt', $alt);
                }
            }
        }

        // Prevent late job completion from overwriting restored state.
        update_post_meta($post_id, '_rankbot_restored_at', time());

        return true;
    }

    private function rankbot_resolve_action(string $post_type, string $requested): string
    {
        $requested = sanitize_key($requested);
        $post_type = sanitize_key($post_type);

        if ($requested === 'auto' || $requested === '') {
            // Bulk "auto": products use the heavy "complex" pipeline.
            // Posts use "post_optimize" (single-post optimization behavior).
            // Can be overridden via filter for advanced setups.
            $default = ($post_type === 'product') ? 'complex' : 'post_optimize';
            $filtered = apply_filters('rankbot_bulk_auto_action', $default, $post_type);
            $filtered = is_string($filtered) ? sanitize_key($filtered) : '';
            if ($filtered !== '') {
                return $filtered;
            }
            return $default;
        }

        if ($requested === 'research') {
            return 'research';
        }

        if ($requested === 'snippet') {
            return 'snippet';
        }

        if ($requested === 'complex') {
            // Back-compat: if older bulk rows were queued with action=complex for posts,
            // map them to post_optimize so bulk can proceed.
            if ($post_type === 'post') {
                return 'post_optimize';
            }
            return ($post_type === 'product') ? 'complex' : '';
        }

        if ($requested === 'post_optimize') {
            return 'post_optimize';
        }

        return '';
    }

    // Back-compat wrapper: older code paths still call this name.
    private function rankbot_resolve_bulk_action(string $post_type, string $requested): string
    {
        return $this->rankbot_resolve_action($post_type, $requested);
    }

    private function rankbot_dispatch_generate_and_register_post_job(int $post_id, string $action, array $params, array $options = []): array
    {
        $action = sanitize_key($action);
        if ($post_id < 1 || $action === '' || !is_array($params) || empty($params)) {
            return ['error' => 'Invalid dispatch parameters'];
        }

        $timeout = isset($options['timeout']) ? (int) $options['timeout'] : 0;
        if ($timeout > 0 && $timeout < 5) {
            $timeout = 5;
        }

        $timeout_filter = null;
        if ($timeout > 0) {
            $timeout_filter = static function ($value, $action_arg = null, $params_arg = null) use ($timeout) {
                return $timeout;
            };
            add_filter('rankbot_api_generate_timeout', $timeout_filter, 10, 3);
        }

        $result = $this->api->generate($action, $params);

        if ($timeout_filter) {
            remove_filter('rankbot_api_generate_timeout', $timeout_filter, 10);
        }

        if (isset($result['error'])) {
            // Preserve error shape (http_code/error_type/error_code may exist) for callers.
            return is_array($result) ? $result : ['error' => 'Unknown error'];
        }

        // Async job (queued OR reused processing/pending job)
        // API may return {status:'processing', reused:true} when a job already exists.
        if (isset($result['status']) && isset($result['job_id']) && in_array((string) $result['status'], ['queued', 'processing', 'pending'], true)) {
            $serverStatus = (string) $result['status'];

            global $wpdb;
            $table_name = $wpdb->prefix . 'rankbot_jobs';
            $site_key_hash = $this->rankbot_site_key_hash();

            // Create backup before any changes are applied.
            $backup_ts = $this->create_backup($post_id, $action);
            set_transient('rankbot_job_backup_ts_' . (string) $result['job_id'], (int) $backup_ts, DAY_IN_SECONDS);

            // Ensure we have a local job row. (job_id is UNIQUE, so INSERT IGNORE is safe.)
            $job_obj = (object) [
                'job_id' => (string) $result['job_id'],
                'status' => ($serverStatus === 'queued' ? 'queued' : $serverStatus),
                'created_at' => current_time('mysql'),
            ];
            update_post_meta($post_id, '_rankbot_last_job_id', (string) $result['job_id']);
            update_post_meta($post_id, '_rankbot_last_job_status', $job_obj->status);
            update_post_meta($post_id, '_rankbot_last_job_created_at', $job_obj->created_at);
            update_post_meta($post_id, '_rankbot_last_job_site_key_hash', $site_key_hash);

            wp_cache_set('rankbot_latest_job_' . (int) $post_id, $job_obj, 'rankbotai-seo-optimizer', 30);

            $this->schedule_job_poll((string) $result['job_id']);
            return [
                'mode' => 'async',
                'job_id' => (string) $result['job_id'],
                'server_status' => $serverStatus,
                'result' => $result,
            ];
        }

        return [
            'mode' => 'sync',
            'result' => is_array($result) ? $result : [],
        ];
    }

    private function rankbot_build_params_for_bulk(int $post_id, $post, string $action): array
    {
        $post = get_post($post_id);
        if (!$post) return [];

        $existing_focus_kw_source = '';
        $existing_focus_kw = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
        if ($existing_focus_kw) {
            $existing_focus_kw_source = 'yoast';
        }
        if (!$existing_focus_kw) {
            $existing_focus_kw = get_post_meta($post_id, 'rank_math_focus_keyword', true);
            if ($existing_focus_kw) {
                $existing_focus_kw_source = 'rankmath';
            }
        }
        if (!$existing_focus_kw) {
            $existing_focus_kw = get_post_meta($post_id, '_rankbot_focus_keyword', true);
            if ($existing_focus_kw) {
                $existing_focus_kw_source = 'rankbot';
            }
        }

        $allow_main_content_update = (get_option('rankbot_update_main_content', 'yes') === 'yes');

        if ($action === 'post_optimize') {
            $site_name = get_bloginfo('name');
            $params = [
                'post_id' => $post_id,
                'title' => (string) $post->post_title,
                'url' => get_permalink($post_id),
                'content' => (string) $post->post_content,
                'site_name' => $site_name,
                'post_type' => (string) $post->post_type,
                'focus_keyword' => (string) $existing_focus_kw,
                'language' => get_option('rankbot_language', 'auto'),
                'type' => 'post_optimization',
                'length_mode' => 'auto',
                'allow_content_update' => $allow_main_content_update ? 'yes' : 'no',
                'skip_content' => $allow_main_content_update ? 0 : 1,
            ];
            $params['images'] = $this->rankbot_collect_images_for_alt_generation($post_id, $post);

            // Provide safe internal links to avoid URL hallucinations.
            $cats = [];
            $terms = get_the_terms($post_id, 'category');
            if ($terms && !is_wp_error($terms)) {
                foreach ($terms as $term) $cats[] = $term->name;
            }
            $params['categories'] = implode(', ', $cats);

            $args = [
                'category__in' => wp_get_post_categories($post_id),
                'post__not_in' => [$post_id],
                'posts_per_page' => 6,
                'orderby' => 'rand',
                'post_status' => 'publish'
            ];
            $related_query = new WP_Query($args);
            $internal_links = [];
            while ($related_query->have_posts()) {
                $related_query->the_post();
                $context_src = (string) (get_the_excerpt() ?: get_the_content());
                $internal_links[] = [
                    'title' => get_the_title(),
                    'url' => get_permalink(),
                    'context' => mb_substr(wp_strip_all_tags($context_src), 0, 160) . '...'
                ];
            }
            wp_reset_postdata();
            $params['internal_links'] = $internal_links;

            // Safe external links: use Wikipedia search (avoid hallucinated exact URLs)
            $lang = (string) get_option('rankbot_language', 'auto');
            $wikiHost = 'https://en.wikipedia.org';
            if ($lang === 'ru') $wikiHost = 'https://ru.wikipedia.org';
            if ($lang === 'uk') $wikiHost = 'https://uk.wikipedia.org';
            $params['external_links'] = [
                [
                    'title' => 'Wikipedia: ' . (string) $post->post_title,
                    'url' => $wikiHost . '/wiki/Special:Search?search=' . rawurlencode((string) $post->post_title),
                ],
            ];

            if (function_exists('rankbot_log')) {
                $kw = (string) $existing_focus_kw;
                rankbot_log('Build params: post_optimize', [
                    'post_id' => (int) $post_id,
                    'post_type' => (string) ($post->post_type ?? ''),
                    'focus_keyword_source' => (string) $existing_focus_kw_source,
                    'focus_keyword_len' => strlen($kw),
                    'focus_keyword_preview' => function_exists('mb_substr') ? mb_substr($kw, 0, 120) : substr($kw, 0, 120),
                    'allow_content_update' => $allow_main_content_update ? 'yes' : 'no',
                    'content_len' => strlen((string) $post->post_content),
                    'internal_links_count' => is_array($internal_links) ? count($internal_links) : 0,
                ]);
            }

            return $params;
        }

        $params = [
            'post_id' => $post_id,
            'product_name' => (string) $post->post_title,
            'description' => ($post->post_type === 'product') ? (string) $post->post_content : wp_strip_all_tags((string) $post->post_content),
            'short_desc' => wp_strip_all_tags((string) $post->post_excerpt),
            'title' => (string) $post->post_title,
            'content' => (string) $post->post_content,
            'url' => get_permalink($post_id),
            'language' => get_option('rankbot_language', 'auto'),
            'post_type' => (string) $post->post_type,
            'focus_keyword' => (string) $existing_focus_kw,
            'allow_content_update' => $allow_main_content_update ? 'yes' : 'no',
            'skip_content' => $allow_main_content_update ? 0 : 1,
            'instructions' => [],
        ];
        $params['images'] = $this->rankbot_collect_images_for_alt_generation($post_id, $post);

        if (get_option('rankbot_update_slug', 'no') === 'yes') {
            $params['instructions']['generate_slug'] = true;
            $params['instructions']['slug_language'] = 'en';
        }

        if (class_exists('WooCommerce') && $post->post_type === 'product') {
            $product = wc_get_product($post_id);
            if ($product) {
                $cats = [];
                $terms = get_the_terms($post_id, 'product_cat');
                if ($terms && !is_wp_error($terms)) {
                    foreach ($terms as $term) $cats[] = $term->name;
                }
                $params['categories'] = implode(', ', $cats);

                $params['price'] = $product->get_price();
                $params['currency'] = get_woocommerce_currency();

                $attributes = [];
                foreach ($product->get_attributes() as $attr) {
                    if ($attr->is_taxonomy()) {
                        $attr_terms = wc_get_product_terms($post_id, $attr->get_name(), ['fields' => 'names']);
                        $attributes[$attr->get_name()] = implode(', ', $attr_terms);
                    } else {
                        $attributes[$attr->get_name()] = $attr->get_options();
                    }
                }
                $params['attributes'] = $attributes;

                $params['sku'] = $product->get_sku();

                $related_ids = wc_get_related_products($post_id, 4);
                $internal_links = [];
                foreach ($related_ids as $rel_id) {
                    $rel_post = get_post($rel_id);
                    $context_src = '';
                    if ($rel_post) {
                        $context_src = (string) ($rel_post->post_excerpt ?: $rel_post->post_content);
                    }
                    $internal_links[] = [
                        'title' => get_the_title($rel_id),
                        'url' => get_permalink($rel_id),
                        'context' => mb_substr(wp_strip_all_tags($context_src), 0, 150) . '...'
                    ];
                }
                $params['internal_links'] = $internal_links;
            }
        } else {
            $cats = [];
            $terms = get_the_terms($post_id, 'category');
            if ($terms && !is_wp_error($terms)) {
                foreach ($terms as $term) $cats[] = $term->name;
            }
            $params['categories'] = implode(', ', $cats);

            $args = [
                'category__in' => wp_get_post_categories($post_id),
                'post__not_in' => [$post_id],
                'posts_per_page' => 4,
                'orderby' => 'rand',
                'post_status' => 'publish'
            ];
            $related_query = new WP_Query($args);
            $internal_links = [];
            while ($related_query->have_posts()) {
                $related_query->the_post();
                $context_src = (string) (get_the_excerpt() ?: get_the_content());
                $internal_links[] = [
                    'title' => get_the_title(),
                    'url' => get_permalink(),
                    'context' => mb_substr(wp_strip_all_tags($context_src), 0, 160) . '...'
                ];
            }
            wp_reset_postdata();
            $params['internal_links'] = $internal_links;

            // Safe external links: use Wikipedia search (avoid hallucinated exact URLs)
            $lang = (string) get_option('rankbot_language', 'auto');
            $wikiHost = 'https://en.wikipedia.org';
            if ($lang === 'ru') $wikiHost = 'https://ru.wikipedia.org';
            if ($lang === 'uk') $wikiHost = 'https://uk.wikipedia.org';
            $params['external_links'] = [
                [
                    'title' => 'Wikipedia: ' . (string) $post->post_title,
                    'url' => $wikiHost . '/wiki/Special:Search?search=' . rawurlencode((string) $post->post_title),
                ],
            ];
        }

        return $params;
    }

    public function render_history_page()
    {
        $history_nonce = wp_create_nonce('rankbot_history_filters');

        $status_filter = 'all';
        $type_filter = 'all';
        $paged = isset($_GET['paged']) ? max(1, absint(wp_unslash($_GET['paged']))) : 1;
        $per_page = isset($_GET['per_page']) ? absint(wp_unslash($_GET['per_page'])) : 20;
        $allowed_per_page = [20, 50, 100, 200];
        if (!in_array($per_page, $allowed_per_page, true)) {
            $per_page = 20;
        }

        $history_nonce_ok = false;
        if (isset($_GET['rankbot_history_nonce'])) {
            if (!wp_verify_nonce(
                sanitize_text_field(wp_unslash($_GET['rankbot_history_nonce'])),
                'rankbot_history_filters'
            )) {
                wp_die(esc_html__('Security check failed.', 'rankbotai-seo-optimizer'), 403);
            }
            $history_nonce_ok = true;
        }

        if ($history_nonce_ok) {
            $status_filter = isset($_GET['status']) ? sanitize_key(wp_unslash($_GET['status'])) : 'all';
            $type_filter = isset($_GET['type']) ? sanitize_key(wp_unslash($_GET['type'])) : 'all';
            $pp = isset($_GET['per_page']) ? absint(wp_unslash($_GET['per_page'])) : 20;
            if (in_array($pp, $allowed_per_page, true)) {
                $per_page = $pp;
            }
        }

        $history = [];
        $pagination = [];
        $stats = [];
        $available_types = [];

        $data = $this->api->get_history($paged, $per_page, $status_filter, $type_filter);
        if (is_array($data) && isset($data['history'])) {
            $history = is_array($data['history']) ? $data['history'] : [];
            $pagination = is_array($data['pagination'] ?? null) ? $data['pagination'] : [];
            $stats = is_array($data['stats'] ?? null) ? $data['stats'] : [];
            $available_types = is_array($data['filters']['types'] ?? null) ? $data['filters']['types'] : [];
        } elseif (is_array($data)) {
            // Backward-compat: some servers might still return plain history list.
            $history = $data;
        }

        // Local timing enrichment disabled to avoid direct DB queries.
        $local_job_timing = [];

        // Build type options only from what actually exists in history (or from API-provided filters).
        // We never show all WP post types because it includes types that were never processed.
        if (empty($available_types) && !empty($history)) {
            $typesFromHistory = [];
            foreach ($history as $item) {
                $meta = [];
                if (isset($item['meta'])) {
                    $meta = is_array($item['meta']) ? $item['meta'] : json_decode((string)$item['meta'], true);
                }
                $params = $meta['params'] ?? [];

                $t = $params['post_type'] ?? ($params['taxonomy'] ?? ($item['resource_type'] ?? null));
                if (is_string($t) && $t !== '') {
                    $typesFromHistory[] = $t;
                }
            }
            $typesFromHistory = array_values(array_unique($typesFromHistory));
            sort($typesFromHistory);
            $available_types = $typesFromHistory;
        }
        
        $total_pages = isset($pagination['total_pages']) ? (int)$pagination['total_pages'] : 1;
        $total_items = isset($pagination['total_items']) ? (int)$pagination['total_items'] : count($history);

        $can_clear_local_history = ($this->rankbot_history_storage_mode() === 'wordpress');
        
        ?>
        <div class="wrap rankbot-wrap" style="max-width: 100%; margin-right: 20px;">
            <div class="rankbot-header-flex" style="margin-bottom: 20px;">
                <h1 class="wp-heading-inline" style="font-size: 24px; font-weight: 600;"><?php esc_html_e('Operation History', 'rankbotai-seo-optimizer'); ?></h1>
            </div>

            <!-- Stats Overview -->
            <?php if (!empty($stats)): ?>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 25px;">
                <div style="background: #fff; padding: 20px; border-radius: 8px; border: 1px solid #d1d5db; box-shadow: 0 1px 2px rgba(0,0,0,0.05);">
                    <div style="font-size: 13px; color: #6b7280; font-weight: 500; text-transform: uppercase;"><?php esc_html_e('Total Operations', 'rankbotai-seo-optimizer'); ?></div>
                    <div style="font-size: 24px; font-weight: 700; color: #111827; margin-top: 5px;">
                        <?php echo esc_html(number_format_i18n((int) ($stats['total_ops'] ?? 0))); ?>
                    </div>
                </div>
                <div style="background: #fff; padding: 20px; border-radius: 8px; border: 1px solid #d1d5db; box-shadow: 0 1px 2px rgba(0,0,0,0.05);">
                    <div style="font-size: 13px; color: #6b7280; font-weight: 500; text-transform: uppercase;"><?php esc_html_e('Success Rate', 'rankbotai-seo-optimizer'); ?></div>
                    <div style="font-size: 24px; font-weight: 700; color: #059669; margin-top: 5px;">
                        <?php echo esc_html(number_format_i18n((int) ($stats['success_rate'] ?? 0))); ?>%
                    </div>
                </div>
                <div style="background: #fff; padding: 20px; border-radius: 8px; border: 1px solid #d1d5db; box-shadow: 0 1px 2px rgba(0,0,0,0.05);">
                    <div style="font-size: 13px; color: #6b7280; font-weight: 500; text-transform: uppercase;"><?php esc_html_e('Total Tokens', 'rankbotai-seo-optimizer'); ?></div>
                    <div style="font-size: 24px; font-weight: 700; color: #4f46e5; margin-top: 5px;">
                        <?php echo esc_html(number_format_i18n((float) ($stats['total_spent'] ?? 0))); ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <!-- Filters -->
            <div class="tablenav top" style="display: flex; gap: 10px; align-items: center; background: #fff; padding: 15px; border: 1px solid #e5e7eb; border-radius: 8px 8px 0 0; margin-bottom: 0; border-bottom: none;">
                <form method="get" style="display: flex; gap: 10px; width: 100%;">
                    <input type="hidden" name="page" value="rankbot-history" />
                    <input type="hidden" name="rankbot_history_nonce" value="<?php echo esc_attr($history_nonce); ?>" />
                    
                    <select name="status" style="border-color: #d1d5db; color: #374151;">
                        <option value="all" <?php selected($status_filter, 'all'); ?>><?php esc_html_e('All Statuses', 'rankbotai-seo-optimizer'); ?></option>
                        <option value="completed" <?php selected($status_filter, 'completed'); ?>><?php esc_html_e('Completed', 'rankbotai-seo-optimizer'); ?></option>
                        <option value="processing" <?php selected($status_filter, 'processing'); ?>><?php esc_html_e('Processing', 'rankbotai-seo-optimizer'); ?></option>
                        <option value="failed" <?php selected($status_filter, 'failed'); ?>><?php esc_html_e('Failed', 'rankbotai-seo-optimizer'); ?></option>
                    </select>

                    <select name="type" style="border-color: #d1d5db; color: #374151;">
                        <option value="all" <?php selected($type_filter, 'all'); ?>><?php esc_html_e('All Types', 'rankbotai-seo-optimizer'); ?></option>
                        <?php 
                        foreach ($available_types as $type_slug) {
                            $label = ucfirst((string)$type_slug);

                            // Prefer a nice label from WP if possible
                            if (post_type_exists($type_slug)) {
                                $pt = get_post_type_object($type_slug);
                                if ($pt && isset($pt->label)) {
                                    $label = $pt->label;
                                }
                            } elseif (taxonomy_exists($type_slug)) {
                                $tx = get_taxonomy($type_slug);
                                if ($tx && isset($tx->label)) {
                                    $label = $tx->label;
                                }
                            }

                            echo '<option value="' . esc_attr($type_slug) . '" ' . selected($type_filter, $type_slug, false) . '>' . esc_html($label) . '</option>';
                        }
                        ?>
                    </select>

                    <select name="per_page" style="border-color: #d1d5db; color: #374151;">
                        <?php foreach ($allowed_per_page as $pp): ?>
                            <?php /* translators: %s: items per page. */ ?>
                            <option value="<?php echo esc_attr((string) $pp); ?>" <?php selected($per_page, $pp); ?>><?php echo esc_html(sprintf(__('%s / page', 'rankbotai-seo-optimizer'), (string) $pp)); ?></option>
                        <?php endforeach; ?>
                    </select>

                    <input type="submit" id="post-query-submit" class="button" value="<?php echo esc_attr__('Filter', 'rankbotai-seo-optimizer'); ?>" style="border-color: #d1d5db;">

                    <button
                        type="button"
                        id="rb-clear-local-history"
                        class="button"
                        style="border-color:#ef4444; color:#b91c1c; background:#fff;"
                        <?php echo $can_clear_local_history ? '' : 'disabled="disabled"'; ?>
                        title="<?php echo esc_attr($can_clear_local_history ? __('Delete all history stored in WordPress DB (irreversible).', 'rankbotai-seo-optimizer') : __('History is loaded from server. Local delete is disabled.', 'rankbotai-seo-optimizer')); ?>"
                    >
                        <?php esc_html_e('Clear Local History', 'rankbotai-seo-optimizer'); ?>
                    </button>
                    
                    <div style="margin-left: auto; color: #6b7280; display: flex; align-items: center;">
                        <?php /* translators: %s: total items count. */ ?>
                        <span class="displaying-num" style="font-size: 13px;"><?php echo esc_html(sprintf(__('%s items', 'rankbotai-seo-optimizer'), number_format_i18n((int) $total_items))); ?></span>
                    </div>
                </form>
            </div>
            <table class="rankbot-history-table">
                <colgroup>
                    <col style="width: 150px;"> <!-- Date -->
                    <col style="width: 140px;">  <!-- ID -->
                    <col style="width: 120px;"> <!-- Action (Type of Processing) -->
                    <col style="width: 80px;">  <!-- Post Type -->
                    <col style="width: 25%;">   <!-- Title -->
                    <col style="width: 100px;"> <!-- Status -->
                    <col style="width: 90px;">  <!-- Time -->
                    <col style="width: 100px;"> <!-- Model -->
                    <col style="width: 100px;"> <!-- Tokens -->
                </colgroup>
                <thead>
                    <tr>
                        <th><?php esc_html_e('Date', 'rankbotai-seo-optimizer'); ?></th>
                        <th><?php esc_html_e('Record ID', 'rankbotai-seo-optimizer'); ?></th>
                        <th><?php esc_html_e('Action', 'rankbotai-seo-optimizer'); ?></th>
                        <th><?php esc_html_e('Type', 'rankbotai-seo-optimizer'); ?></th>
                        <th><?php esc_html_e('Record / Title', 'rankbotai-seo-optimizer'); ?></th>
                        <th><?php esc_html_e('Status', 'rankbotai-seo-optimizer'); ?></th>
                        <th><?php esc_html_e('Time', 'rankbotai-seo-optimizer'); ?></th>
                        <th><?php esc_html_e('Model', 'rankbotai-seo-optimizer'); ?></th>
                        <th><?php esc_html_e('Tokens', 'rankbotai-seo-optimizer'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($history)): ?>
                        <tr><td colspan="9" style="text-align: center; color: #6b7280; padding: 40px;"><?php esc_html_e('No history found for current filters.', 'rankbotai-seo-optimizer'); ?></td></tr>
                    <?php else: ?>
                        <?php foreach ($history as $item): ?>
                            <?php 
                                // Parse Meta
                                $meta = [];
                                if (isset($item['meta'])) {
                                    $meta = is_array($item['meta']) ? $item['meta'] : json_decode((string)$item['meta'], true);
                                }
                                $params = $meta['params'] ?? [];

                                // Operation/Job ID (server-side job id)
                                // Server returns job_id from site_operations.job_id; fallback to operation_id or meta.id.
                                $opId = '';
                                if (isset($item['job_id']) && is_string($item['job_id']) && $item['job_id'] !== '') {
                                    $opId = (string) $item['job_id'];
                                } elseif (isset($item['operation_id']) && is_string($item['operation_id']) && $item['operation_id'] !== '') {
                                    $opId = (string) $item['operation_id'];
                                } elseif (isset($meta['id']) && is_string($meta['id']) && $meta['id'] !== '') {
                                    $opId = (string) $meta['id'];
                                }
                                if ($opId === '') {
                                    $opId = '-';
                                }
                                
                                // Record ID
                                // Fallback sequence: params['post_id'] -> params['term_id'] -> item['resource_id'] -> meta['object_id']
                                $objId = $params['post_id'] ?? ($params['term_id'] ?? ($item['resource_id'] ?? ($meta['object_id'] ?? '-')));
                                
                                // Title / Link
                                $title = $params['title']
                                    ?? ($params['product_name']
                                        ?? ($params['category_name']
                                            ?? ($params['name'] ?? ($item['action'] ?? 'Unknown'))));
                                $url = $params['url'] ?? ($params['permalink'] ?? null);
                                
                                // Post Type
                                $postType = $params['post_type'] ?? ($params['taxonomy'] ?? ($item['resource_type'] ?? '-'));
                                
                                // Action Label
                                // The API returns 'action' field like 'research', 'complex', 'rankbot_optimize'
                                // Also check event_type as fallback (e.g., 'job_started', 'job_completed')
                                $rawAction = $item['action'] ?? '';
                                $eventType = isset($item['event_type']) ? (string) $item['event_type'] : '';
                                
                                // Try to get clean action from event_type if action is empty
                                if ($rawAction === '' && $eventType !== '') {
                                    $rawAction = preg_replace('/^job_/', '', $eventType);
                                }
                                
                                $actionLabel = ucfirst(str_replace(['rankbot_', '_'], ['', ' '], $rawAction));
                                
                                // Custom mapping for cleaner UI
                                $actionMap = [
                                    'research' => __('Research', 'rankbotai-seo-optimizer'),
                                    'snippet' => __('Snippet', 'rankbotai-seo-optimizer'),
                                    'complex' => __('Full SEO', 'rankbotai-seo-optimizer'),
                                    'full seo' => __('Full SEO', 'rankbotai-seo-optimizer'),
                                    'optimize' => __('SEO Opt', 'rankbotai-seo-optimizer'),
                                    'post optimize' => __('Optimize', 'rankbotai-seo-optimizer'),
                                    'bulk' => __('Bulk Job', 'rankbotai-seo-optimizer'),
                                    'started' => __('Started', 'rankbotai-seo-optimizer'),
                                    'completed' => __('Completed', 'rankbotai-seo-optimizer'),
                                    'failed' => __('Failed', 'rankbotai-seo-optimizer')
                                ];
                                $actionLower = strtolower($actionLabel);
                                foreach ($actionMap as $key => $val) {
                                    if ($key === $rawAction || $key === $actionLower) {
                                        $actionLabel = $val;
                                        break;
                                    }
                                }

                                
                                // Tokens (RankBot Currency)
                                $costVal = isset($item['cost']) ? (float)$item['cost'] : 0;
                                $totalTokens = ($costVal > 0) ? number_format($costVal) : '-';

                                // Status color
                                $status = strtolower($item['status'] ?? ($item['event_type'] ?? ''));
                                $badgeClass = 'rb-badge-neutral';
                                if (strpos($status, 'complete') !== false) $badgeClass = 'rb-badge-success';
                                if (strpos($status, 'fail') !== false || strpos($status, 'rejected') !== false) $badgeClass = 'rb-badge-error';
                                if (strpos($status, 'process') !== false || strpos($status, 'queue') !== false) $badgeClass = 'rb-badge-warning';
                                
                                $statusLabel = $item['status'] ?? 'log';

                                // Duration (best-effort from local WP job table)
                                $durationLabel = '-';
                                $jobIdForDuration = isset($item['job_id']) && is_string($item['job_id']) ? (string) $item['job_id'] : '';
                                if ($jobIdForDuration !== '' && isset($local_job_timing[$jobIdForDuration]) && is_array($local_job_timing[$jobIdForDuration])) {
                                    $jr = $local_job_timing[$jobIdForDuration];
                                    $dur = isset($jr['duration_sec']) ? (int) $jr['duration_sec'] : 0;
                                    if ($dur > 0) {
                                        $durationLabel = $this->rankbot_format_duration($dur);
                                    } else {
                                        $start_at = isset($jr['started_at']) ? (string) $jr['started_at'] : '';
                                        $created_at = isset($jr['created_at']) ? (string) $jr['created_at'] : '';
                                        $start_ts = $start_at !== '' ? strtotime($start_at) : false;
                                        if (!$start_ts) {
                                            $start_ts = $created_at !== '' ? strtotime($created_at) : false;
                                        }
                                        if ($start_ts) {
                                            $elapsed = max(0, time() - (int) $start_ts);
                                            if ($elapsed > 0) {
                                                $durationLabel = $this->rankbot_format_duration($elapsed);
                                            }
                                        }
                                    }
                                }

                                // Server-provided elapsed_sec (preferred when available; excludes retries by design)
                                if ($durationLabel === '-') {
                                    $elapsedSec = null;
                                    if (isset($item['elapsed_sec']) && is_numeric($item['elapsed_sec'])) {
                                        $elapsedSec = (float) $item['elapsed_sec'];
                                    } elseif (isset($meta['elapsed_sec']) && is_numeric($meta['elapsed_sec'])) {
                                        $elapsedSec = (float) $meta['elapsed_sec'];
                                    }
                                    if ($elapsedSec !== null && $elapsedSec > 0) {
                                        $durationLabel = $this->rankbot_format_duration((int) round($elapsedSec));
                                    }
                                }

                                // Guard: hide malformed job rows that have no record id and no meaningful title.
                                $rawActionForGuard = isset($item['action']) ? (string) $item['action'] : '';
                                $titleForGuard = is_string($title) ? trim($title) : '';
                                if (($objId === '-' || $objId === '' || $objId === null)
                                    && ($titleForGuard === '' || $titleForGuard === $rawActionForGuard)
                                    && ($opId !== '-')
                                ) {
                                    continue;
                                }

                                // Error details tooltip (only for error-like statuses)
                                $details = $item['details'] ?? ($item['error_text'] ?? '');
                                if (!is_string($details)) {
                                    $details = '';
                                }
                                $statusTitle = '';
                                if ($badgeClass === 'rb-badge-error' && $details !== '') {
                                    $statusTitle = $details;
                                }
                            ?>
                            <tr>
                                <td style="color: #6b7280;">
                                    <?php
                                        $createdAt = isset($item['created_at']) ? (string) $item['created_at'] : '';
                                        $createdTs = $createdAt !== '' ? strtotime($createdAt) : false;
                                        $fmt = (string) get_option('date_format') . ' ' . (string) get_option('time_format');
                                        echo esc_html($createdTs ? wp_date($fmt, $createdTs) : '-');
                                    ?>
                                </td>
                                <td>
                                    <?php if ($objId !== '-' && $objId !== '' && $objId !== null): ?>
                                        <code style="background: #f3f4f6; padding: 2px 6px; border-radius: 4px; color: #111827; font-size: 11px;">#<?php echo esc_html((string) $objId); ?></code>
                                    <?php else: ?>
                                        <span style="color: #9ca3af;">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span style="font-weight: 600; color: #4f46e5; font-size: 12px; border: 1px solid #e0e7ff; background: #eef2ff; padding: 2px 6px; border-radius: 4px;">
                                        <?php echo esc_html($actionLabel); ?>
                                    </span>
                                </td>
                                <td style="text-transform: capitalize; color: #4b5563; font-size: 12px;">
                                    <?php echo esc_html($postType); ?>
                                </td>
                                <td title="<?php echo esc_attr($title); ?>">
                                    <?php if ($url): ?>
                                        <a href="<?php echo esc_url($url); ?>" target="_blank" style="font-weight: 500; text-decoration: none; color: #111827; display: inline-flex; align-items: center; max-width: 100%;">
                                            <span style="overflow: hidden; text-overflow: ellipsis; white-space: nowrap;"><?php echo esc_html($title); ?></span>
                                            <span class="dashicons dashicons-external" style="font-size: 12px; height: 12px; width: 12px; margin-left: 4px; flex-shrink: 0; color: #9ca3af;"></span>
                                        </a>
                                    <?php else: ?>
                                        <span style="font-weight: 500; color: #111827;"><?php echo esc_html($title); ?></span>
                                    <?php endif; ?>

                                    <?php if ($opId !== '-'):
                                        $shortJob = (strlen($opId) > 16) ? (substr($opId, 0, 16) . '…') : $opId;
                                    ?>
                                        <code title="<?php echo esc_attr($opId); ?>" style="margin-left: 8px; background: #fff; border: 1px dashed #d1d5db; padding: 1px 5px; border-radius: 4px; color: #6b7280; font-size: 11px;">job:<?php echo esc_html($shortJob); ?></code>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="rb-badge <?php echo esc_attr($badgeClass); ?>"<?php echo $statusTitle !== '' ? ' title="' . esc_attr($statusTitle) . '"' : ''; ?>>
                                        <?php echo esc_html($statusLabel); ?>
                                    </span>
                                </td>
                                <td style="color:#4b5563; font-size:12px;">
                                    <?php echo esc_html($durationLabel); ?>
                                </td>
                                <td>
                                    <span style="font-size: 12px; color: #4b5563; background: #f3f4f6; padding: 2px 6px; border-radius: 4px; border: 1px solid #e5e7eb;">
                                        <?php echo esc_html($item['model'] ?? '-'); ?>
                                    </span>
                                </td>
                                <td style="color: #111827; font-family: monospace; font-weight: 600;">
                                    <?php echo esc_html($totalTokens); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
                <div class="pagination-links">
                    <?php
                    echo wp_kses_post(paginate_links([
                        'base' => add_query_arg([
                            'paged' => '%#%',
                            'status' => $status_filter,
                            'type' => $type_filter,
                            'per_page' => $per_page,
                            'rankbot_history_nonce' => $history_nonce,
                        ]),
                        'format' => '',
                        'prev_text' => '&laquo;',
                        'next_text' => '&raquo;',
                        'total' => $total_pages,
                        'current' => $paged
                    ]));
                    ?>
                </div>
            <?php endif; ?>

            <!-- Clear Local History Modal -->
            <div id="rb-clear-history-modal" style="display:none; position:fixed; inset:0; background:rgba(17,24,39,.55); z-index:100000; align-items:center; justify-content:center;">
                <div style="background:#fff; width:min(560px, calc(100% - 40px)); border-radius:12px; border:1px solid #e5e7eb; box-shadow:0 10px 30px rgba(0,0,0,.2); overflow:hidden;">
                    <div style="padding:16px 18px; background:#fef2f2; border-bottom:1px solid #fee2e2;">
                        <div style="display:flex; align-items:center; gap:10px;">
                            <span class="dashicons dashicons-warning" style="color:#b91c1c;"></span>
                            <strong style="color:#7f1d1d;"><?php esc_html_e('Delete local history?', 'rankbotai-seo-optimizer'); ?></strong>
                        </div>
                    </div>
                    <div style="padding:18px; color:#374151;">
                        <p style="margin-top:0;"><?php esc_html_e('This will permanently delete all RankBot history stored in your WordPress database (post/term meta).', 'rankbotai-seo-optimizer'); ?></p>
                        <p style="margin-bottom:0; color:#6b7280;"><?php esc_html_e('It will not delete history on the RankBot server.', 'rankbotai-seo-optimizer'); ?></p>
                    </div>
                    <div style="display:flex; justify-content:flex-end; gap:10px; padding:14px 18px; background:#f9fafb; border-top:1px solid #e5e7eb;">
                        <button type="button" class="button" id="rb-clear-history-cancel"><?php esc_html_e('Cancel', 'rankbotai-seo-optimizer'); ?></button>
                        <button type="button" class="button button-primary" id="rb-clear-history-confirm" style="background:#dc2626; border-color:#dc2626;"><?php esc_html_e('Delete', 'rankbotai-seo-optimizer'); ?></button>
                    </div>
                </div>
            </div>
            <?php
            wp_enqueue_script(
                'rankbot-history',
                RANKBOT_PLUGIN_URL . 'assets/js/rankbot-history.js',
                array( 'jquery' ),
                defined( 'RANKBOT_VERSION' ) ? RANKBOT_VERSION : '1.0.0',
                true
            );
            wp_localize_script( 'rankbot-history', 'rankbotHistoryData', array(
                'canClear' => $can_clear_local_history ? true : false,
                'nonce'    => wp_create_nonce( 'rankbot_optimize' ),
                'i18n'     => array(
                    'failedToClear' => __( 'Failed to clear local history', 'rankbotai-seo-optimizer' ),
                ),
            ) );
            ?>
        </div>
        <?php
    }

    public function add_admin_bar_balance($admin_bar)
    {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        if (!current_user_can('manage_options')) return;
        
        $key = $this->api->get_key();
        if (!$key) return;

        $site_key_hash = $this->rankbot_site_key_hash();

        // Cache balance to avoid blocking HTTP call on every admin page load
        $balance = get_transient('rankbot_admin_balance');
        if (false === $balance) {
             $balance = $this->api->get_balance();
             set_transient('rankbot_admin_balance', $balance, 120);
        }

        // Fetch running jobs (best-effort; do not hide balance if table is missing)
        global $wpdb;
        $table = $wpdb->prefix . 'rankbot_jobs';
        $cache_group = 'rankbotai-seo-optimizer';
        $has_jobs_table = $this->rankbot_db_table_exists($table);

        $processingCount = 0;
        $queue_remaining = 0;
        if ($has_jobs_table) {
            $queue_remaining = $this->rankbot_bulk_remaining_count($site_key_hash);

            $processingCount = wp_cache_get('rankbot_adminbar_processing_count', $cache_group);
            if (false === $processingCount) {
                if ($site_key_hash !== '') {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $processingCount = $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status IN ('processing', 'pending', 'queued') AND created_at > DATE_SUB(NOW(), INTERVAL 2 HOUR)",
                            $table,
                            $site_key_hash
                        )
                    );
                } else {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $processingCount = $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM %i WHERE site_key_hash = %s AND status IN ('processing', 'pending', 'queued') AND created_at > DATE_SUB(NOW(), INTERVAL 2 HOUR)",
                            $table,
                            $site_key_hash
                        )
                    );
                }
                wp_cache_set('rankbot_adminbar_processing_count', $processingCount, $cache_group, 5);
            }
            $processingCount = absint($processingCount);
        }

        $displayCount = max((int) $processingCount, (int) $queue_remaining);
        
        $title = '<span class="ab-icon dashicons dashicons-superhero"></span> ' . esc_html(number_format_i18n((float) $balance)) . ' Tokens';
        if ($displayCount > 0) {
            $title .= ' <span style="background: #ef4444; color: white; border-radius: 99px; padding: 1px 6px; font-size: 10px; margin-left: 5px; vertical-align: middle;">' . esc_html((string) $displayCount) . ' Working...</span>';
        }

        $admin_bar->add_node([
            'id'    => 'rankbot-balance',
            'title' => $title,
            'href'  => admin_url('admin.php?page=rankbotai-seo-optimizer'),
            'meta'  => [
                'title' => 'RankBotAI'
            ]
        ]);
        
        // --- Child Item: Processing ---
        $active_jobs = [];
        if ($has_jobs_table) {
            $active_jobs = wp_cache_get('rankbot_adminbar_active_jobs', $cache_group);
            if (false === $active_jobs) {
                if ($site_key_hash !== '') {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $active_jobs = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT object_id, object_type, created_at FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status IN ('processing', 'pending', 'queued') AND created_at > DATE_SUB(NOW(), INTERVAL 2 HOUR) ORDER BY id DESC LIMIT 5",
                            $table,
                            $site_key_hash
                        )
                    );
                } else {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $active_jobs = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT object_id, object_type, created_at FROM %i WHERE site_key_hash = %s AND status IN ('processing', 'pending', 'queued') AND created_at > DATE_SUB(NOW(), INTERVAL 2 HOUR) ORDER BY id DESC LIMIT 5",
                            $table,
                            $site_key_hash
                        )
                    );
                }
                wp_cache_set('rankbot_adminbar_active_jobs', $active_jobs, $cache_group, 5);
            }
        }

        if (!empty($active_jobs)) {
            $admin_bar->add_node([
                'id'    => 'rankbot-active-header',
                'parent' => 'rankbot-balance',
                'title' => '<span style="color:#f87171; font-weight:bold; font-size:11px; text-transform:uppercase; display:block; padding: 5px 0;">Active Process (' . $displayCount . ')</span>',
                'group' => true,
            ]);

            foreach ($active_jobs as $row) {
                 $label = $this->get_object_label($row->object_id, $row->object_type);
                 $editLink = $this->get_edit_link($row->object_id, $row->object_type);

                 // Spinner
                 $spinner = '<span class="dashicons dashicons-update" style="font-size:14px; width:14px; height:14px; line-height:14px; animation: spin 2s linear infinite; color:#f87171; margin-right:5px;"></span>';
                 
                 $admin_bar->add_node([
                    'id'    => 'rankbot-active-' . $row->object_id,
                    'parent' => 'rankbot-balance',
                    'title' => '<div style="display:flex; align-items:center;">' . $spinner . '<span style="max-width:160px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap;">' . esc_html($label) . '</span></div>',
                    'href'  => $editLink
                ]);
            }
        }
        
        // --- Child Item: Latest Completed ---
        $completed = [];
        if ($has_jobs_table) {
            $completed = wp_cache_get('rankbot_adminbar_completed_jobs', $cache_group);
            if (false === $completed) {
                if ($site_key_hash !== '') {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $completed = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT object_id, object_type, action, created_at FROM %i WHERE (site_key_hash = %s OR site_key_hash = '') AND status = 'completed' ORDER BY id DESC LIMIT 10",
                            $table,
                            $site_key_hash
                        )
                    );
                } else {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $completed = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT object_id, object_type, action, created_at FROM %i WHERE site_key_hash = %s AND status = 'completed' ORDER BY id DESC LIMIT 10",
                            $table,
                            $site_key_hash
                        )
                    );
                }
                wp_cache_set('rankbot_adminbar_completed_jobs', $completed, $cache_group, 30);
            }
        }
        
        if (!empty($completed)) {
            $recently_completed_label = esc_html__('Recently Completed', 'rankbotai-seo-optimizer');
            $admin_bar->add_node([
                'id'    => 'rankbot-balance-history-label',
                'parent' => 'rankbot-balance',
                'title' => '<span style="color:#9ca3af; font-size:11px; text-transform:uppercase; letter-spacing:0.5px; display:block; padding: 5px 0; border-top:1px solid #ffffff33; margin-top:5px;">' . $recently_completed_label . '</span>',
                'group' => true
            ]);

            foreach ($completed as $row) {
                // Determine Title and Link
                $label = $this->get_object_label($row->object_id, $row->object_type);
                $editLink = $this->get_edit_link($row->object_id, $row->object_type);
                
                $timeAgo = human_time_diff(strtotime($row->created_at), current_time('timestamp'));
                // Shorten time ago
                $timeAgo = str_replace([' min', ' hour', ' day', ' week', ' month', ' year', 's'], ['m', 'h', 'd', 'w', 'mo', 'y', ''], $timeAgo);
                
                if($label) {
                    $admin_bar->add_node([
                        'id'    => 'rankbot-job-' . $row->object_id . '-' . wp_rand(100, 999),
                        'parent' => 'rankbot-balance',
                        'title' => '<div style="display:flex; justify-content:space-between; width:220px;"><span style="max-width:180px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap;">' . esc_html($label) . '</span> <span style="color:#9ca3af; font-size:10px;">' . $timeAgo . '</span></div>',
                        'href'  => $editLink
                    ]);
                }
            }
        }

        // View All Link
        $admin_bar->add_node([
            'id'    => 'rankbot-history-link',
            'parent' => 'rankbot-balance',
            'title' => esc_html__('View Full History →', 'rankbotai-seo-optimizer'),
            'href'  => admin_url('admin.php?page=rankbot-history'),
            'meta' => ['class' => 'rankbot-view-history']
        ]);
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    private function get_object_label($id, $type) {
        if ($type === 'post') {
            $post = get_post($id);
            /* translators: %d: post ID. */
            return $post ? mb_strimwidth($post->post_title, 0, 30, '...') : sprintf(esc_html__('Post #%d', 'rankbotai-seo-optimizer'), (int) $id);
        } elseif ($type === 'term') {
            $term = get_term($id);
            /* translators: %d: term ID. */
            return ($term && !is_wp_error($term)) ? mb_strimwidth($term->name, 0, 30, '...') : sprintf(esc_html__('Term #%d', 'rankbotai-seo-optimizer'), (int) $id);
        }
        /* translators: %d: item ID. */
        return sprintf(esc_html__('Item #%d', 'rankbotai-seo-optimizer'), (int) $id);
    }

    private function get_edit_link($id, $type) {
        if ($type === 'post') return get_edit_post_link($id);
        if ($type === 'term') {
            $term = get_term($id);
            return ($term && !is_wp_error($term)) ? get_edit_term_link($id, $term->taxonomy) : '#';
        }
        return '#';
    }

    public function render_settings_page() 
    {
        $models = $this->api->get_models();
        $current_model = get_option('rankbot_selected_model', 'gpt-3.5-turbo');
        $current_language = get_option('rankbot_language', 'auto');
        $show_seo_score = get_option('rankbot_show_seo_score', 'yes');

        $seo_recalc_enabled = get_option(self::RANKBOT_SEO_RECALC_ENABLED_OPTION, 'yes');
        $seo_recalc_state = $this->rankbot_get_seo_recalc_state();
        
        $update_slug = get_option('rankbot_update_slug', 'no');
        $update_title = get_option('rankbot_update_product_title', 'no');
        $update_main_content = get_option('rankbot_update_main_content', 'yes');

        // Fetch balance for display
        $balance = $this->api->get_balance();
        set_transient('rankbot_admin_balance', $balance, 300); // Cache for 5 mins

        $languages = [
            'auto' => __('Auto (Detect from content)', 'rankbotai-seo-optimizer'),
            'en' => __('English', 'rankbotai-seo-optimizer'),
            'ru' => __('Russian', 'rankbotai-seo-optimizer'),
            'uk' => __('Ukrainian', 'rankbotai-seo-optimizer'),
            'es' => __('Spanish', 'rankbotai-seo-optimizer'),
            'fr' => __('French', 'rankbotai-seo-optimizer'),
            'de' => __('German', 'rankbotai-seo-optimizer'),
            'it' => __('Italian', 'rankbotai-seo-optimizer'),
            'pt' => __('Portuguese', 'rankbotai-seo-optimizer'),
            'pl' => __('Polish', 'rankbotai-seo-optimizer'),
            'tr' => __('Turkish', 'rankbotai-seo-optimizer'),
        ];
        ?>
        <div class="wrap rankbot-settings-wrap">
            <h1 class="wp-heading-inline"><?php esc_html_e('RankBot Settings', 'rankbotai-seo-optimizer'); ?></h1>
             <?php if (null !== filter_input(INPUT_GET, 'success', FILTER_DEFAULT)): ?>
                <div class="notice notice-success is-dismissible" style="margin-left: 0;"><p><?php esc_html_e('Settings saved successfully.', 'rankbotai-seo-optimizer'); ?></p></div>
            <?php endif; ?>
            
            <div class="rankbot-header-row">
                <div class="rankbot-subtitle"><?php esc_html_e('Configuration & Preferences', 'rankbotai-seo-optimizer'); ?></div>
                <div class="rankbot-balance-badge">
                    <span class="dashicons dashicons-database"></span>
                    <span><?php echo esc_html(number_format_i18n((float) $balance)); ?> <?php esc_html_e('Tokens Available', 'rankbotai-seo-optimizer'); ?></span>
                    <a href="#" id="rankbot-manage-plan-btn" class="rankbot-topup-btn">+ <?php esc_html_e('Top Up', 'rankbotai-seo-optimizer'); ?></a>
                </div>
            </div>

            <!-- Plan Modal (same as Dashboard) -->
            <div id="rankbot-plan-modal" style="display:none; position:fixed; inset:0; background:rgba(0,0,0,0.5); z-index:100000; align-items:center; justify-content:center;">
                <div style="background:white; width:90%; max-width:900px; max-height:90vh; border-radius:12px; overflow:hidden; display:flex; flex-direction:column; box-shadow:0 25px 50px -12px rgba(0,0,0,0.25);">
                    <!-- Header -->
                    <div style="padding:16px 24px; border-bottom:1px solid #e5e7eb; display:flex; justify-content:space-between; align-items:center; background:#f9fafb;">
                        <h3 style="margin:0; font-size:18px; font-weight:600; color:#111827;"><?php esc_html_e('Manage Subscription & Top-up', 'rankbotai-seo-optimizer'); ?></h3>
                        <button type="button" onclick="document.getElementById('rankbot-plan-modal').style.display='none'" style="border:none; background:none; cursor:pointer; font-size:24px; color:#6b7280;">&times;</button>
                    </div>
                    
                    <!-- Content -->
                    <div style="display:flex; flex-direction:column; flex:1; overflow:hidden; background:#f8fafc;">
                        
                        <!-- Tabs -->
                        <div style="background:white; border-bottom:1px solid #e5e7eb; padding:0 24px;">
                            <button type="button" onclick="rbSwitchTab('subs')" id="rb-tab-subs" style="background:transparent; border:none; border-bottom:2px solid #4f46e5; color:#4f46e5; padding:16px 4px; margin-right:24px; font-weight:600; font-size:14px; cursor:pointer;"><?php echo esc_html__('Subscriptions', 'rankbotai-seo-optimizer'); ?></button>
                            <button type="button" onclick="rbSwitchTab('packets')" id="rb-tab-packets" style="background:transparent; border:none; border-bottom:2px solid transparent; color:#6b7280; padding:16px 4px; margin-right:24px; font-weight:500; font-size:14px; cursor:pointer;"><?php echo esc_html__('One-time Packets', 'rankbotai-seo-optimizer'); ?></button>
                            <button type="button" onclick="rbSwitchTab('free')" id="rb-tab-free" style="background:transparent; border:none; border-bottom:2px solid transparent; color:#6b7280; padding:16px 4px; font-weight:500; font-size:14px; cursor:pointer;"><?php echo esc_html__('Free Tier', 'rankbotai-seo-optimizer'); ?></button>
                        </div>

                        <div style="padding:24px; overflow-y:auto; flex:1;">
                            <div id="rankbot-plans-loader" style="text-align:center; padding:40px;">
                                <span class="spinner is-active" style="float:none; margin:0;"></span> <?php echo esc_html__('Loading plans...', 'rankbotai-seo-optimizer'); ?>
                            </div>
                            
                            <div id="rankbot-content-container" style="display:none;">
                                
                                <!-- Subscriptions View -->
                                <div id="rb-view-subs">
                                    <div style="display:flex; justify-content:flex-end; margin-bottom:20px;">
                                         <div style="background:#e5e7eb; padding:4px; border-radius:8px; display:inline-flex;">
                                            <button type="button" id="rb-bill-monthly" onclick="rbSetBilling('monthly')" style="border:none; background:white; padding:6px 16px; border-radius:6px; font-size:13px; font-weight:600; color:#111827; cursor:pointer; box-shadow:0 1px 2px rgba(0,0,0,0.1); transition:all 0.2s;"><?php echo esc_html__('Monthly', 'rankbotai-seo-optimizer'); ?></button>
                                            <button type="button" id="rb-bill-yearly" onclick="rbSetBilling('yearly')" style="border:none; background:transparent; padding:6px 16px; border-radius:6px; font-size:13px; font-weight:600; color:#4b5563; cursor:pointer; transition:all 0.2s;"><?php echo esc_html__('Yearly', 'rankbotai-seo-optimizer'); ?> <span style="font-size:10px; color:#059669; font-weight:800;">-20%</span></button>
                                         </div>
                                    </div>
                                    <div id="rankbot-plans-list-subs" style="display:grid; grid-template-columns: repeat(auto-fit, minmax(260px, 1fr)); gap:20px;"></div>
                                    <p id="rb-msg-no-subs" style="display:none; text-align:center; color:#6b7280; padding:20px;"><?php echo esc_html__('No subscription plans available.', 'rankbotai-seo-optimizer'); ?></p>
                                </div>

                                <!-- Packets View -->
                                <div id="rb-view-packets" style="display:none;">
                                    <div id="rankbot-plans-list-packets" style="display:grid; grid-template-columns: repeat(auto-fit, minmax(260px, 1fr)); gap:20px;"></div>
                                    <p id="rb-msg-no-packets" style="display:none; text-align:center; color:#6b7280; padding:20px;"><?php echo esc_html__('No token packets available.', 'rankbotai-seo-optimizer'); ?></p>
                                </div>

                                 <!-- Free View -->
                                <div id="rb-view-free" style="display:none;">
                                    <div id="rankbot-plans-list-free" style="display:grid; grid-template-columns: repeat(auto-fit, minmax(260px, 1fr)); gap:20px;"></div>
                                </div>

                            </div>
                        </div>
                    </div>
                    
                    <!-- Footer -->
                    <div style="padding:16px 24px; border-top:1px solid #e5e7eb; background:white; text-align:right;">
                        <button type="button" onclick="document.getElementById('rankbot-plan-modal').style.display='none'" class="button"><?php esc_html_e('Close', 'rankbotai-seo-optimizer'); ?></button>
                    </div>
                </div>
            </div>
            <?php
            wp_enqueue_script(
                'rankbot-settings',
                RANKBOT_PLUGIN_URL . 'assets/js/rankbot-settings.js',
                array( 'jquery' ),
                defined( 'RANKBOT_VERSION' ) ? RANKBOT_VERSION : '1.0.0',
                true
            );
            wp_localize_script( 'rankbot-settings', 'rankbotSettingsData', array(
                'nonce'   => wp_create_nonce( 'rankbot_optimize' ),
                'apiUrl'  => esc_url( rtrim( (string) $this->api->get_api_url(), '/' ) ),
                'authKey' => esc_js( $this->api->get_key() ),
                'i18n'    => array(
                    'items'             => __( 'items', 'rankbotai-seo-optimizer' ),
                    'failedToLoadPlans' => __( 'Failed to load plans. Please try again later.', 'rankbotai-seo-optimizer' ),
                    'networkError'      => __( 'Network error. Please check your connection.', 'rankbotai-seo-optimizer' ),
                    'running'           => __( 'Running', 'rankbotai-seo-optimizer' ),
                    'idle'              => __( 'Idle', 'rankbotai-seo-optimizer' ),
                ),
            ) );
            ?>
                        <form method="post" action="<?php echo esc_url(admin_url('admin.php')); ?>">
                <input type="hidden" name="rankbot_action" value="save_global_settings">
                <?php wp_nonce_field('rankbot_action'); ?>

                <div class="rankbot-grid">
                    <!-- LEFT COLUMN: SETTINGS -->
                    <div class="rankbot-main-col">
                        
                        <!-- AI Configuration -->
                        <div class="rankbot-card">
                            <div class="rankbot-card-header">
                                <h2 class="rankbot-card-title"><span class="dashicons dashicons-brain"></span> <?php esc_html_e('AI Model & Language', 'rankbotai-seo-optimizer'); ?></h2>
                            </div>
                            <div class="rankbot-card-body">
                                <div class="form-group">
                                    <label class="form-label" for="rankbot_model"><?php esc_html_e('AI Model Strategy', 'rankbotai-seo-optimizer'); ?></label>
                                    <select name="rankbot_model" id="rankbot_model" class="rankbot-select">
                                        <?php foreach ($models as $model): ?>
                                            <option value="<?php echo esc_attr($model['id']); ?>" <?php selected($current_model, $model['id']); ?>>
                                                <?php echo esc_html($model['name']); ?> (<?php /* translators: %s: token count. */ ?><?php echo esc_html(sprintf(__('%s tokens', 'rankbotai-seo-optimizer'), (string) $model['cost'])); ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <p class="form-description"><?php esc_html_e('Select the LLM model used for content generation. Higher tier models deliver better copy but consume more tokens.', 'rankbotai-seo-optimizer'); ?></p>
                                </div>

                                <div class="form-group">
                                    <label class="form-label" for="rankbot_language"><?php esc_html_e('Target Language', 'rankbotai-seo-optimizer'); ?></label>
                                    <select name="rankbot_language" id="rankbot_language" class="rankbot-select">
                                        <?php foreach ($languages as $code => $label): ?>
                                            <option value="<?php echo esc_attr($code); ?>" <?php selected($current_language, $code); ?>>
                                                <?php echo esc_html($label); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <p class="form-description"><?php esc_html_e('We can auto-detect the language from your content, or you can force a specific language output.', 'rankbotai-seo-optimizer'); ?></p>
                                </div>
                            </div>
                        </div>

                        <!-- Optimization Preferences -->
                        <div class="rankbot-card">
                            <div class="rankbot-card-header">
                                <h2 class="rankbot-card-title"><span class="dashicons dashicons-hammer"></span> <?php esc_html_e('Automation Controls', 'rankbotai-seo-optimizer'); ?></h2>
                            </div>
                            <div class="rankbot-card-body">
                                <div class="form-group">
                                    <div class="rankbot-toggle-row">
                                        <input type="checkbox" name="rankbot_update_product_title" id="rankbot_update_product_title" value="yes" <?php checked($update_title, 'yes'); ?> class="rankbot-toggle-input">
                                        <div>
                                            <label class="form-label" for="rankbot_update_product_title"><?php esc_html_e('Update Product/Post Title', 'rankbotai-seo-optimizer'); ?></label>
                                            <?php /* translators: %s: recommended setting label. */ ?>
                                            <p class="form-description"><?php echo wp_kses_post(sprintf(__('Allow RankBot to rewrite the main Title of the post/product. Recommended: %s (keep human titles).', 'rankbotai-seo-optimizer'), '<strong>' . esc_html__('No', 'rankbotai-seo-optimizer') . '</strong>')); ?></p>
                                        </div>
                                    </div>
                                </div>

                                <div class="form-group">
                                    <div class="rankbot-toggle-row">
                                        <input type="checkbox" name="rankbot_update_slug" id="rankbot_update_slug" value="yes" <?php checked($update_slug, 'yes'); ?> class="rankbot-toggle-input">
                                        <div>
                                            <label class="form-label" for="rankbot_update_slug"><?php esc_html_e('Update URL Slug (Permalink)', 'rankbotai-seo-optimizer'); ?></label>
                                            <p class="form-description"><?php echo wp_kses_post(__('Allow RankBot to rewrite the URL path based on the new title/keyword. <strong>Caution:</strong> Affects SEO if redirects aren\'t managed.', 'rankbotai-seo-optimizer')); ?></p>
                                        </div>
                                    </div>
                                </div>

                                <div class="form-group">
                                    <div class="rankbot-toggle-row">
                                        <input type="checkbox" name="rankbot_update_main_content" id="rankbot_update_main_content" value="yes" <?php checked($update_main_content, 'yes'); ?> class="rankbot-toggle-input">
                                        <div>
                                            <label class="form-label" for="rankbot_update_main_content"><?php esc_html_e('Update Product Description / Post Content', 'rankbotai-seo-optimizer'); ?></label>
                                            <p class="form-description"><?php esc_html_e('Allow RankBot to rewrite the main product description or the main post content. If disabled, RankBot will still do SEO meta/keyword/slug and other actions, but will not rewrite the main content to reduce token usage.', 'rankbotai-seo-optimizer'); ?></p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Interface Settings -->
                        <div class="rankbot-card">
                            <div class="rankbot-card-header">
                                <h2 class="rankbot-card-title"><span class="dashicons dashicons-layout"></span> <?php esc_html_e('Interface', 'rankbotai-seo-optimizer'); ?></h2>
                            </div>
                            <div class="rankbot-card-body">
                                <div class="form-group">
                                    <label class="form-label" for="rankbot_show_seo_score"><?php esc_html_e('SEO Score Panel', 'rankbotai-seo-optimizer'); ?></label>
                                    <select name="rankbot_show_seo_score" id="rankbot_show_seo_score" class="rankbot-select">
                                        <option value="yes" <?php selected($show_seo_score, 'yes'); ?>><?php esc_html_e('Show Panel', 'rankbotai-seo-optimizer'); ?></option>
                                        <option value="no" <?php selected($show_seo_score, 'no'); ?>><?php esc_html_e('Hide Panel', 'rankbotai-seo-optimizer'); ?></option>
                                    </select>
                                    <p class="form-description"><?php esc_html_e('Control the visibility of the SEO Score Analysis widget in the post editor.', 'rankbotai-seo-optimizer'); ?></p>
                                </div>
                            </div>
                        </div>

                        <!-- SEO Score Backfill -->
                        <div class="rankbot-card">
                            <div class="rankbot-card-header">
                                <h2 class="rankbot-card-title"><span class="dashicons dashicons-chart-line"></span> <?php esc_html_e('SEO Score Backfill', 'rankbotai-seo-optimizer'); ?></h2>
                            </div>
                            <div class="rankbot-card-body">
                                <div class="form-group">
                                    <div class="rankbot-toggle-row">
                                        <input type="checkbox" name="rankbot_seo_recalc_enabled" id="rankbot_seo_recalc_enabled" value="yes" <?php checked($seo_recalc_enabled, 'yes'); ?> class="rankbot-toggle-input">
                                        <div>
                                            <label class="form-label" for="rankbot_seo_recalc_enabled"><?php esc_html_e('Auto-recalculate missing SEO scores in background', 'rankbotai-seo-optimizer'); ?></label>
                                            <p class="form-description"><?php esc_html_e('Fixes legacy posts/products that show score 0 until opened. Runs in small WP‑Cron batches.', 'rankbotai-seo-optimizer'); ?></p>
                                        </div>
                                    </div>
                                </div>

                                <div class="form-group" style="margin-bottom: 12px;">
                                    <button type="button" class="button button-secondary" id="rankbot-seo-recalc-start" <?php echo (!empty($seo_recalc_state['running'])) ? 'disabled' : ''; ?>><?php esc_html_e('Run backfill now', 'rankbotai-seo-optimizer'); ?></button>
                                </div>

                                <div id="rankbot-seo-recalc-status" style="padding: 12px 14px; background:#f8fafc; border:1px solid #e2e8f0; border-radius:8px; color:#334155; font-size:13px;">
                                    <div style="display:flex; align-items:center; justify-content:space-between; gap:12px;">
                                        <div>
                                            <strong><?php esc_html_e('Status:', 'rankbotai-seo-optimizer'); ?></strong>
                                            <span id="rb-seo-recalc-state-text"><?php echo esc_html(!empty($seo_recalc_state['running']) ? __('Running', 'rankbotai-seo-optimizer') : __('Idle', 'rankbotai-seo-optimizer')); ?></span>
                                        </div>
                                        <div id="rb-seo-recalc-spinner" style="display:<?php echo !empty($seo_recalc_state['running']) ? 'inline-flex' : 'none'; ?>; align-items:center; gap:8px; color:#059669; font-weight:600;">
                                            <span class="spinner is-active" style="float:none; margin:0;"></span>
                                            <?php esc_html_e('Working', 'rankbotai-seo-optimizer'); ?>
                                        </div>
                                    </div>
                                    <div style="margin-top:8px; color:#64748b;">
                                        <span id="rb-seo-recalc-progress"><?php echo esc_html((string) ((int)($seo_recalc_state['processed'] ?? 0) . ' / ' . (int)($seo_recalc_state['total'] ?? 0))); ?></span>
                                    </div>
                                </div>
                            </div>
                        </div>

                         <div class="rankbot-save-bar">
                            <div style="font-size:13px; color:#6b7280;">
                                <span class="dashicons dashicons-yes" style="color: #10b981; font-size: 16px; vertical-align: text-bottom;"></span>
                                <?php esc_html_e('Auto-saved to database on submit', 'rankbotai-seo-optimizer'); ?>
                            </div>
                            <?php submit_button(__('Save Changes', 'rankbotai-seo-optimizer'), 'primary large', 'submit', false); ?>
                        </div>

                    </div>

                    <!-- RIGHT COLUMN: PROMO -->
                    <div class="rankbot-sidebar-col">
                        
                        <!-- Premium Card -->
                        <div class="rankbot-card promo-card">
                            <div class="rankbot-card-header">
                                <h2 class="rankbot-card-title">🚀 <?php esc_html_e('AI SEO AutoOptimize Pro', 'rankbotai-seo-optimizer'); ?></h2>
                            </div>
                            <div class="rankbot-card-body">
                                <p style="margin-top:0; color:#cbd5e1; line-height: 1.6;"><?php echo wp_kses_post(__('For stores with <strong>10,000+ products</strong>, we recommend our specialized plugin.', 'rankbotai-seo-optimizer')); ?></p>
                                <ul class="promo-list">
                                    <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('One-time payment (Lifetime)', 'rankbotai-seo-optimizer'); ?></li>
                                    <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('Pay only for API usage', 'rankbotai-seo-optimizer'); ?></li>
                                    <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('More cost-effective at scale', 'rankbotai-seo-optimizer'); ?></li>
                                </ul>
                                <a href="https://aiseo.buyreadysite.com/" target="_blank" class="promo-btn"><?php esc_html_e('View Plugin Details', 'rankbotai-seo-optimizer'); ?></a>
                            </div>
                        </div>

                        <!-- Other Products -->
                        <div class="rankbot-card">
                            <div class="rankbot-card-header">
                                <h2 class="rankbot-card-title"><?php esc_html_e('More from BuyReadySite', 'rankbotai-seo-optimizer'); ?></h2>
                            </div>
                            <div class="rankbot-card-body">
                                <div class="other-product-item">
                                    <div class="other-product-icon"><span class="dashicons dashicons-edit"></span></div>
                                    <div class="other-product-info">
                                        <h4>AI Content Wizard</h4>
                                        <p><?php esc_html_e('Generate SEO optimized articles in bulk.', 'rankbotai-seo-optimizer'); ?></p>
                                        <a href="https://aiwizard.buyreadysite.com/" target="_blank" class="other-product-link"><?php esc_html_e('Learn more →', 'rankbotai-seo-optimizer'); ?></a>
                                    </div>
                                </div>

                                <div class="other-product-item">
                                    <div class="other-product-icon"><span class="dashicons dashicons-megaphone"></span></div>
                                    <div class="other-product-info">
                                        <h4>Promopilot</h4>
                                        <p><?php esc_html_e('Automated social media marketing tools.', 'rankbotai-seo-optimizer'); ?></p>
                                        <a href="https://promopilot.link/" target="_blank" class="other-product-link"><?php esc_html_e('Learn more →', 'rankbotai-seo-optimizer'); ?></a>
                                    </div>
                                </div>
                                
                                <div style="margin-top: 20px; padding-top: 15px; border-top: 1px solid #f3f4f6; text-align: center;">
                                      <a href="https://buyreadysite.com" target="_blank" style="text-decoration: none; font-weight: 600; color: #4b5563; font-size: 13px;"><?php esc_html_e('Visit Official Website', 'rankbotai-seo-optimizer'); ?></a>
                                </div>
                            </div>
                        </div>
                        
                         <div class="rankbot-card" style="margin-bottom: 24px;">
                            <div class="rankbot-card-body" style="text-align: center;">
                                <p style="font-size: 13px; color: #6b7280; margin-bottom: 5px;"><?php esc_html_e('Need Help?', 'rankbotai-seo-optimizer'); ?></p>
                                <a href="https://buyreadysite.com/support" target="_blank" style="font-weight: 600; text-decoration: none; color: #2563eb;"><?php esc_html_e('Contact Support', 'rankbotai-seo-optimizer'); ?></a>
                            </div>
                        </div>

                    </div>
                </div>

            </form>
        </div>
        <?php
    }

    public function add_sidebar_metabox()
    {
        $screens = ['post', 'page', 'product'];
        foreach ($screens as $screen) {
            add_meta_box(
                'rankbot_stats_box',
                'RankBotAI Controls',
                [$this, 'render_stats_metabox'],
                $screen,
                'side',
                'high'
            );
        }
    }

    public function render_stats_metabox($post)
    {
        $key = $this->api->get_key();
        $balance = $key ? $this->api->get_balance() : 0;
        
        $history = get_post_meta($post->ID, '_rankbot_history', true) ?: [];
        $total_optimized = count($history);
        $last_optimized = !empty($history) ? end($history)['date'] : 'Never';
        $show_seo = get_option('rankbot_show_seo_score', 'yes') === 'yes';
        $seo_score = get_post_meta($post->ID, '_rankbot_seo_score', true);

        $score_int = is_numeric($seo_score) ? (int) $seo_score : 0;
        if ($score_int < 0) $score_int = 0;
        if ($score_int > 100) $score_int = 100;

        $recent_history = [];
        if (!empty($history) && is_array($history)) {
            $recent_history = array_slice(array_reverse($history), 0, 6);
        }

        $pretty_action = function ($raw) {
            $raw = is_string($raw) ? trim($raw) : '';
            if ($raw === '') return 'Action';
            $k = strtolower($raw);

            if ($k === 'research') return 'Keyword Research';
            if ($k === 'snippet') return 'SEO Snippet';
            if ($k === 'complex') return 'Auto-Optimize SEO';

            if ($k === 'post_optimize' || $k === 'optimize_post') return 'Auto-Optimize Article';
            if ($k === 'post optimization' || $k === 'post_optimization') return 'Post Optimization';

            // Keep legacy/custom labels (e.g. "Category Optimization")
            return $raw;
        };

        $total_cost = 0;
        foreach($history as $h) {
             if(isset($h['cost'])) $total_cost += (int)$h['cost'];
        }

        // Feature Costs (Approx)
        $costResearch = 1;
        $costSnippet = 2;
        $costComplex = 5;

        ?>
        <div class="rankbot-sidebar-panel">
            <?php if (!$key): ?>
                <div class="rankbot-connect-msg" style="padding: 10px; background: #f0f0f1; text-align: center;">
                    <span class="dashicons dashicons-lock"></span> Connect RankBot to use AI
                </div>
            <?php else: ?>
                
                <!-- Hidden inputs for JS -->
                <input type="hidden" id="rankbot_sidebar_post_id" value="<?php echo esc_attr((string) $post->ID); ?>">
                
                <?php if ($show_seo):
                    $scoreClass = '';
                    if($score_int < 50) $scoreClass = 'rankbot-score-bad';
                    elseif($score_int < 80) $scoreClass = 'rankbot-score-ok';
                    else $scoreClass = 'rankbot-score-good';
                ?>
                    <div id="rb-sidebar-score-val" class="rankbot-score-ring <?php echo esc_attr($scoreClass); ?>" style="--rb-score: <?php echo (int) $score_int; ?>;">
                        <div class="rb-score-text">
                            <div class="rb-score-num"><?php echo (int)$score_int; ?></div>
                            <div class="rb-score-sub">/100</div>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- SEO Checklist -->
                <?php
                $rb_post_title   = get_the_title($post);
                $rb_meta_title   = get_post_meta($post->ID, '_rankbot_title', true)
                                 ?: get_post_meta($post->ID, '_yoast_wpseo_title', true)
                                 ?: get_post_meta($post->ID, 'rank_math_title', true)
                                 ?: '';
                $rb_effective_title = $rb_meta_title ?: $rb_post_title;

                $rb_meta_desc    = get_post_meta($post->ID, '_rankbot_description', true)
                                 ?: get_post_meta($post->ID, '_yoast_wpseo_metadesc', true)
                                 ?: get_post_meta($post->ID, 'rank_math_description', true)
                                 ?: '';

                $rb_focus_kw     = $this->rankbot_get_focus_keyword_for_post($post->ID);

                $rb_content_raw  = $post->post_content;
                $rb_content_text = wp_strip_all_tags(do_shortcode($rb_content_raw));
                $rb_word_count   = str_word_count($rb_content_text);
                $rb_title_len    = mb_strlen($rb_effective_title);
                $rb_desc_len     = mb_strlen($rb_meta_desc);

                // Check images for alt text
                preg_match_all('/<img[^>]+>/i', $rb_content_raw, $rb_img_matches);
                $rb_img_total    = count($rb_img_matches[0]);
                $rb_img_no_alt   = 0;
                foreach ($rb_img_matches[0] as $rb_img_tag) {
                    if (!preg_match('/alt\s*=\s*["\'][^"\']+["\']/i', $rb_img_tag)) {
                        $rb_img_no_alt++;
                    }
                }

                // Check for internal/external links
                preg_match_all('/<a[^>]+href\s*=\s*["\']([^"\']+)["\'][^>]*>/i', $rb_content_raw, $rb_link_matches);
                $rb_site_host     = (string) wp_parse_url(home_url(), PHP_URL_HOST);
                $rb_internal      = 0;
                $rb_external      = 0;
                foreach (($rb_link_matches[1] ?? []) as $rb_href) {
                    $rb_link_host = (string) wp_parse_url($rb_href, PHP_URL_HOST);
                    if ($rb_link_host && $rb_link_host !== $rb_site_host) {
                        $rb_external++;
                    } elseif ($rb_href && $rb_href[0] !== '#') {
                        $rb_internal++;
                    }
                }

                // Build checklist items: [label, pass, hint]
                $rb_checklist = [];

                // 1. Focus keyword
                $rb_checklist[] = [
                    __('Focus keyword set', 'rankbotai-seo-optimizer'),
                    !empty($rb_focus_kw),
                    empty($rb_focus_kw) ? __('Add a focus keyword for SEO targeting', 'rankbotai-seo-optimizer') : '',
                ];

                // 2. Meta description
                $rb_checklist[] = [
                    __('Meta description', 'rankbotai-seo-optimizer'),
                    !empty($rb_meta_desc),
                    empty($rb_meta_desc) ? __('Write a compelling meta description', 'rankbotai-seo-optimizer') : '',
                ];

                // 3. Title length
                $rb_title_ok = ($rb_title_len >= 30 && $rb_title_len <= 65);
                $rb_checklist[] = [
                    /* translators: %d: number of characters */
                    sprintf(__('Title length (%d chars)', 'rankbotai-seo-optimizer'), $rb_title_len),
                    $rb_title_ok,
                    $rb_title_len < 30 ? __('Title is too short (aim for 30-65)', 'rankbotai-seo-optimizer') : ($rb_title_len > 65 ? __('Title is too long (aim for 30-65)', 'rankbotai-seo-optimizer') : ''),
                ];

                // 4. Description length
                if (!empty($rb_meta_desc)) {
                    $rb_desc_ok = ($rb_desc_len >= 120 && $rb_desc_len <= 160);
                    $rb_checklist[] = [
                        /* translators: %d: number of characters */
                        sprintf(__('Description length (%d chars)', 'rankbotai-seo-optimizer'), $rb_desc_len),
                        $rb_desc_ok,
                        $rb_desc_len < 120 ? __('Too short (aim for 120-160)', 'rankbotai-seo-optimizer') : ($rb_desc_len > 160 ? __('Too long (aim for 120-160)', 'rankbotai-seo-optimizer') : ''),
                    ];
                }

                // 5. Content length
                $rb_words_ok = $rb_word_count >= 300;
                $rb_checklist[] = [
                    /* translators: %d: word count */
                    sprintf(__('Content length (%d words)', 'rankbotai-seo-optimizer'), $rb_word_count),
                    $rb_words_ok,
                    $rb_word_count < 300 ? __('Aim for at least 300 words', 'rankbotai-seo-optimizer') : '',
                ];

                // 6. Images have alt text
                if ($rb_img_total > 0) {
                    $rb_checklist[] = [
                        /* translators: %1$d: total images, %2$d: images without alt */
                        sprintf(__('Image alt text (%1$d images, %2$d missing)', 'rankbotai-seo-optimizer'), $rb_img_total, $rb_img_no_alt),
                        $rb_img_no_alt === 0,
                        $rb_img_no_alt > 0 ? __('Add alt text to all images', 'rankbotai-seo-optimizer') : '',
                    ];
                }

                // 7. Internal links
                $rb_checklist[] = [
                    /* translators: %d: number of links */
                    sprintf(__('Internal links (%d)', 'rankbotai-seo-optimizer'), $rb_internal),
                    $rb_internal >= 1,
                    $rb_internal < 1 ? __('Add at least one internal link', 'rankbotai-seo-optimizer') : '',
                ];

                // Count passed
                $rb_passed = 0;
                foreach ($rb_checklist as $rb_item) {
                    if ($rb_item[1]) $rb_passed++;
                }
                $rb_total_checks = count($rb_checklist);
                ?>

                <div class="rankbot-sidebar-divider"></div>
                <div class="rankbot-recent-title" style="display:flex; justify-content:space-between; align-items:center;">
                    <?php echo esc_html__('SEO Checklist', 'rankbotai-seo-optimizer'); ?>
                    <span style="font-size:11px; font-weight:600; color:<?php echo $rb_passed === $rb_total_checks ? '#10b981' : '#f59e0b'; ?>;">
                        <?php echo esc_html($rb_passed . '/' . $rb_total_checks); ?>
                    </span>
                </div>

                <!-- Checklist progress bar -->
                <div style="height:4px; background:#f3f4f6; border-radius:2px; margin-bottom:8px; overflow:hidden;">
                    <div style="height:100%; width:<?php echo esc_attr((string) round(($rb_passed / max($rb_total_checks, 1)) * 100)); ?>%; background:<?php echo $rb_passed === $rb_total_checks ? '#10b981' : '#f59e0b'; ?>; border-radius:2px; transition:width 0.3s;"></div>
                </div>

                <?php foreach ($rb_checklist as $rb_check_item): ?>
                    <div style="display:flex; align-items:flex-start; gap:6px; margin-bottom:5px; font-size:11px; line-height:1.4;">
                        <?php if ($rb_check_item[1]): ?>
                            <span style="color:#10b981; flex-shrink:0; font-size:14px; line-height:1;" title="<?php echo esc_attr(__('Pass', 'rankbotai-seo-optimizer')); ?>">&#10003;</span>
                        <?php else: ?>
                            <span style="color:#f59e0b; flex-shrink:0; font-size:14px; line-height:1;" title="<?php echo esc_attr(__('Needs improvement', 'rankbotai-seo-optimizer')); ?>">&#9888;</span>
                        <?php endif; ?>
                        <div>
                            <span style="color:<?php echo $rb_check_item[1] ? '#374151' : '#92400e'; ?>; font-weight:<?php echo $rb_check_item[1] ? '400' : '500'; ?>;"><?php echo esc_html($rb_check_item[0]); ?></span>
                            <?php if (!empty($rb_check_item[2])): ?>
                                <div style="color:#9ca3af; font-size:10px; margin-top:1px;"><?php echo esc_html($rb_check_item[2]); ?></div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>

                <?php if (!empty($recent_history)): ?>
                    <div class="rankbot-recent-title">Recent processing</div>
                    <?php foreach ($recent_history as $entry):
                        $action_raw = isset($entry['action']) ? (string)$entry['action'] : '';
                        $action = $pretty_action($action_raw);
                        $date_raw = isset($entry['date']) ? (string)$entry['date'] : '';
                        $date_ts = $date_raw ? strtotime($date_raw) : false;
                        $date_fmt = $date_ts ? date_i18n('d.m.Y H:i', $date_ts) : '';
                        $cost = isset($entry['cost']) ? (int)$entry['cost'] : 0;
                    ?>
                        <div class="rankbot-recent-item">
                            <span class="rankbot-recent-action" title="<?php echo esc_attr($action_raw ?: $action); ?>"><?php echo esc_html($action); ?><?php if ($cost > 0): ?> <span style="font-weight:600; color:#646970;">· <?php echo (int)$cost; ?></span><?php endif; ?></span>
                            <span class="rankbot-recent-date"><?php echo esc_html($date_fmt); ?></span>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="rankbot-recent-title">Recent processing</div>
                    <div class="rankbot-recent-item">
                        <span class="rankbot-recent-action" style="font-weight:500; color:#646970;">No actions yet</span>
                        <span class="rankbot-recent-date"></span>
                    </div>
                <?php endif; ?>

                <div id="rankbot-sidebar-log"></div>

                <!-- Restore Modal (Hidden by default) -->
                <div id="rb-restore-modal" style="display:none; position:fixed; z-index:99999; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.5); align-items:center; justify-content:center;">
                    <div style="background:#fff; padding:20px; width:400px; max-width:90%; border-radius:6px; box-shadow:0 4px 15px rgba(0,0,0,0.2);">
                        <h3 style="margin-top:0;">Restore Backup</h3>
                        <div id="rb-restore-list" style="max-height:300px; overflow-y:auto; margin-bottom:15px; border:1px solid #ddd;"></div>
                        <button class="button" onclick="jQuery('#rb-restore-modal').hide()">Close</button>
                    </div>
                </div>

                <div class="rankbot-sidebar-divider"></div>

                <!-- Mini Stats -->
                <div class="rankbot-stat-item">
                    <span class="rankbot-stat-label">Balance</span>
                    <span class="rankbot-stat-value" style="color: #4f46e5;"><?php echo esc_html(number_format_i18n((float) $balance)); ?></span>
                </div>
                <!-- Usage Count -->
                <div class="rankbot-stat-item">
                    <span class="rankbot-stat-label">Actions</span>
                    <span class="rankbot-stat-value"><?php echo esc_html(number_format_i18n((int) $total_optimized)); ?></span>
                </div>

                <?php if (get_post_meta($post->ID, '_rankbot_backups', true)): ?>
                    <div class="rankbot-stat-item" style="margin-top: 10px;">
                        <span class="rankbot-stat-label">Restore</span>
                        <span class="rankbot-stat-value">
                            <a href="#" id="rb-side-restore" class="rankbot-restore-btn" onclick="return false;">
                                <span class="dashicons dashicons-undo"></span>
                                Backup &amp; Restore
                            </a>
                        </span>
                    </div>
                <?php endif; ?>
                
                <div style="margin-top: 10px; text-align: right;">
                     <a href="<?php echo esc_url(admin_url('admin.php?page=rankbot-history')); ?>" target="_blank" style="text-decoration: none; font-size: 11px; color: #2271b1;">View History &rarr;</a>
                </div>
                <?php
                wp_enqueue_script(
                    'rankbot-sidebar',
                    RANKBOT_PLUGIN_URL . 'assets/js/rankbot-sidebar.js',
                    array( 'jquery' ),
                    defined( 'RANKBOT_VERSION' ) ? RANKBOT_VERSION : '1.0.0',
                    true
                );
                wp_localize_script( 'rankbot-sidebar', 'rankbotSidebarData', array(
                    'postId' => (int) $post->ID,
                    'nonce'  => wp_create_nonce( 'rankbot_optimize' ),
                ) );
                ?>
            <?php endif; ?>
        </div>
        <?php
    }

    public function render_post_optimizer_metabox($post) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'rankbot_jobs';
        
        // Find ANY relevant job for this post (queued OR completed/failed pending notice)
        $cache_key = 'rankbot_latest_job_' . (int) $post->ID;
        $existing_job = wp_cache_get($cache_key, 'rankbotai-seo-optimizer');
        if (false === $existing_job) {
            $job_id = (string) get_post_meta($post->ID, '_rankbot_last_job_id', true);
            $status = (string) get_post_meta($post->ID, '_rankbot_last_job_status', true);
            $created_at = (string) get_post_meta($post->ID, '_rankbot_last_job_created_at', true);
            if ($job_id !== '') {
                $existing_job = (object) [
                    'job_id' => $job_id,
                    'status' => $status,
                    'created_at' => $created_at,
                ];
            } else {
                $existing_job = null;
            }
            wp_cache_set($cache_key, $existing_job, 'rankbotai-seo-optimizer', 30);
        }
        
        $pending_job_id = $existing_job && in_array($existing_job->status, ['queued', 'processing']) ? $existing_job->job_id : '';
        $completed_job = $existing_job && $existing_job->status === 'completed' ? $existing_job : null;
        $failed_job = $existing_job && $existing_job->status === 'failed' ? $existing_job : null;

        $nonce = wp_create_nonce('rankbot_optimize');
        ?>
        <div id="rankbot-post-optimizer-wrap" style="padding: 10px 0;">
            
            <!-- RESULT NOTIFICATIONS -->
            <?php if ($completed_job): ?>
                <div class="notice notice-success inline rankbot-notice" style="margin: 0 0 15px 0; padding:10px; border-left-color: #46b450;">
                    <p style="margin:0;">
                        <strong>Optimization Completed!</strong> <br>
                        Content was updated at <?php echo esc_html(gmdate('H:i', strtotime($existing_job->created_at . ' +1 minute'))); ?>.
                    </p>
                    <button type="button" class="button button-small rankbot-dismiss-job" data-id="<?php echo esc_attr($completed_job->job_id); ?>" style="margin-top:8px;">Dismiss</button>
                </div>
            <?php endif; ?>

            <?php if ($failed_job): ?>
                <div class="notice notice-error inline rankbot-notice" style="margin: 0 0 15px 0; padding:10px; border-left-color: #dc3232;">
                    <p style="margin:0;">
                        <strong>Optimization Failed</strong> <br>
                        Please try again or check logs.
                    </p>
                    <button type="button" class="button button-small rankbot-dismiss-job" data-id="<?php echo esc_attr($failed_job->job_id); ?>" style="margin-top:8px;">Dismiss</button>
                </div>
            <?php endif; ?>

            <p class="description" style="font-size: 14px; margin-bottom: 15px;">
                AI-powered Article Writer & Optimizer. 
                <br>• <strong>Short texts</strong> (< 300 words) will be expanded into full articles.
                <br>• <strong>Long reads</strong> will be optimized for SEO structure and readability.
                <br>• <strong>Media Safe:</strong> All images, iframes, and shortcodes are preserved.
            </p>
            
            <div style="display: flex; align-items: center; gap: 15px;">
                <button type="button" class="button button-primary button-large rankbot-optimize-post-btn" style="display: flex; align-items: center; gap: 8px; padding: 5px 20px; height: auto;">
                     <span class="dashicons dashicons-welcome-write-blog" style="font-size: 20px; width: 20px; height: 20px;"></span> 
                     <span style="font-weight: 600; font-size: 14px;">Run Article Optimizer</span>
                </button>
                <div class="spinner" style="float: none; margin: 0;"></div>
                <span class="rankbot-status-text" style="font-weight: 600; color: #646970; display: none;">Initializing...</span>
            </div>
            
            <div id="rankbot-post-log" style="margin-top: 15px; padding: 12px; background: #f0f7ff; border-left: 4px solid #2271b1; color: #2271b1; display: none;"></div>
            <?php
            wp_enqueue_script(
                'rankbot-post-optimizer',
                RANKBOT_PLUGIN_URL . 'assets/js/rankbot-post-optimizer.js',
                array( 'jquery' ),
                defined( 'RANKBOT_VERSION' ) ? RANKBOT_VERSION : '1.0.0',
                true
            );
            wp_localize_script( 'rankbot-post-optimizer', 'rankbotPostData', array(
                'pendingJobId' => esc_js( $pending_job_id ),
                'nonce'        => esc_js( $nonce ),
                'postId'       => (int) $post->ID,
            ) );
            ?>
        </div>
        <?php
    }

    public function render_product_buttons_after_title($post)
    {
        // Only for supported post types
        if (!in_array($post->post_type, ['post', 'page', 'product'])) {
            return;
        }
        
        $key = $this->api->get_key();
        $isConnected = !empty($key);
        $disabledClass = $isConnected ? '' : 'rankbot-disabled';
        
        // Fetch Costs from Server
        $status = $isConnected ? $this->api->get_account_status() : [];
        $costs = $status['costs'] ?? [];
        $costResearch = $costs['research'] ?? 5;
        $costSnippet = $costs['snippet'] ?? 5;
        $costComplex = $costs['complex'] ?? 10;
        
        $nonce = wp_create_nonce('rankbot_optimize');
        $hasBackups = !empty(get_post_meta($post->ID, '_rankbot_backups', true));
        $focusKeyword = get_post_meta($post->ID, '_rankbot_focus_keyword', true);
        
        // --- PRE-FETCH SEO PLUGIN DATA FOR JS ---
        $yoast_kw = get_post_meta($post->ID, '_yoast_wpseo_focuskw', true);
        $rank_kw = get_post_meta($post->ID, 'rank_math_focus_keyword', true);
        $server_side_kw = $focusKeyword;
        if($yoast_kw) $server_side_kw = $yoast_kw;
        elseif($rank_kw) $server_side_kw = $rank_kw;

        if($server_side_kw && strpos($server_side_kw, ',') !== false) {
             $parts = explode(',', $server_side_kw);
             $server_side_kw = trim($parts[0]);
        }
        ?>
        <?php
        wp_enqueue_script(
            'rankbot-toolbar',
            RANKBOT_PLUGIN_URL . 'assets/js/rankbot-toolbar.js',
            array( 'jquery' ),
            defined( 'RANKBOT_VERSION' ) ? RANKBOT_VERSION : '1.0.0',
            true
        );
        wp_localize_script( 'rankbot-toolbar', 'rankbotToolbarData', array(
            'nonce'         => esc_js( $nonce ),
            'postId'        => (int) $post->ID,
            'hasBackups'    => $hasBackups,
            'adminTopUpUrl' => esc_url( admin_url( 'admin.php?page=rankbotai-seo-optimizer' ) ),
        ) );
        $rb_server_data_js = sprintf(
            'window.rbServerData = %s; window.rankbotSingleCosts = %s;',
            wp_json_encode( array(
                'focusKeyword' => $server_side_kw,
                'source'       => $yoast_kw ? 'Yoast' : ( $rank_kw ? 'RankMath' : 'Manual' ),
            ) ),
            wp_json_encode( array(
                'research' => (float) $costResearch,
                'snippet'  => (float) $costSnippet,
                'complex'  => (float) $costComplex,
            ) )
        );
        wp_add_inline_script( 'rankbot-toolbar', $rb_server_data_js, 'before' );
        ?>
        <div class="rankbot-wrapper">
            <div class="rankbot-toolbar">
                <div class="rankbot-toolbar-brand">
                    <span class="dashicons dashicons-superhero" style="color: #6366f1; font-size: 20px;"></span>
                    <div class="rankbot-toolbar-title">RankBotAI</div>
                </div>

                <div class="rankbot-actions">
                        <div class="rankbot-action-btn <?php echo esc_attr($disabledClass); ?>" onclick="rankbotAction('research')">
                        <span class="dashicons dashicons-search"></span> Keyword Research
                            <?php if($isConnected): ?><div class="rankbot-badge-cost"><span class="dashicons dashicons-database" style="font-size:10px; width:10px; height:10px; line-height:10px;"></span> <?php echo esc_html(number_format_i18n((float) $costResearch)); ?></div><?php endif; ?>
                    </div>
                    
                        <div class="rankbot-action-btn <?php echo esc_attr($disabledClass); ?>" onclick="rankbotAction('snippet')">
                        <span class="dashicons dashicons-text"></span> Generate Snippet
                            <?php if($isConnected): ?><div class="rankbot-badge-cost"><span class="dashicons dashicons-database" style="font-size:10px; width:10px; height:10px; line-height:10px;"></span> <?php echo esc_html(number_format_i18n((float) $costSnippet)); ?></div><?php endif; ?>
                    </div>

                    <div style="flex-grow:1;"></div>

                    <div class="rankbot-action-btn primary <?php echo esc_attr($disabledClass); ?> rb-pulse" onclick="rankbotAction('complex')">
                        <span class="dashicons dashicons-performance"></span> Auto-Optimize SEO
                        <?php if($isConnected): ?><div class="rankbot-badge-cost"><span class="dashicons dashicons-database" style="font-size:10px; width:10px; height:10px; line-height:10px;"></span> <?php echo esc_html(number_format_i18n((float) $costComplex)); ?></div><?php endif; ?>
                        <?php if (!$isConnected): ?>
                            <div class="rankbot-tooltip">Connect plugin to use</div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <?php if(!$isConnected): ?>
                    <div style="font-size: 12px; color: #ef4444; font-weight: 500; display:flex; align-items:center;">
                        <span class="dashicons dashicons-lock" style="font-size:14px; margin-right:4px;"></span> Locked
                    </div>
                <?php endif; ?>
            </div>

            <!-- SEO Client Analysis -->
            <?php if($isConnected): ?>
                <input type="hidden" name="rankbot_seo_score" id="rankbot_seo_score_input" value="">
                <!-- Hidden KW input for JS logic -->
                <input type="hidden" id="rankbot_focus_keyword" value="<?php echo esc_attr($focusKeyword); ?>">

                <?php 
                $show_seo = get_option('rankbot_show_seo_score', 'yes') === 'yes';
                if ($show_seo): 
                ?>
                <div class="rankbot-sub-panel">
                    <div id="rankbot-seo-results">
                         <div class="rb-header-flex">
                            <div>
                                 <h3 style="margin:0; font-size:16px;">SEO Optimization Score</h3>
                                 <div style="font-size:12px; color:#64748b; margin-top:4px;">
                                    Focus Keyword: <strong id="rb-current-kw" style="color:#000;">--</strong>
                                    <span id="rb-kw-source" style="margin-left:5px; font-size:10px; background:#e0e7ff; color:#4338ca; padding:2px 6px; border-radius:10px; display:none;">Auto</span>
                                 </div>
                            </div>
                            <div style="text-align:right;">
                                <div id="rb-score-val" style="font-size:24px; font-weight:800; color:#334155;">0/100</div>
                            </div>
                         </div>
                         
                         <div class="rb-progress-track">
                              <div class="rb-progress-bar" id="rb-progress-bar"></div>
                         </div>
                        
                        <!-- Details Toggle -->
                        <div style="text-align:center; margin-bottom:15px;">
                             <button type="button" id="rb-toggle-details" onclick="toggleSeoDetails()" class="rb-details-btn">
                                 Show Details ▾
                             </button>
                        </div>

                        <div id="rb-checklist" style="display:none;">
                             <div id="rb-debug-info" style="font-size:10px; color:#94a3b8; padding:10px; border-bottom:1px solid #f1f5f9; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; display:none;"></div>
                             <div style="text-align:center; padding:20px; color:#94a3b8;">
                                <span class="dashicons dashicons-update" style="font-size:24px; animation:spin 2s infinite linear;"></span><br>
                                Initializing Analysis...
                             </div>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>

        <!-- Restore Modal -->
        <div id="rankbot-restore-modal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.5); z-index:99999; align-items:center; justify-content:center;">
            <div style="background:white; padding:24px; border-radius:12px; max-width:500px; width:100%; box-shadow:0 20px 25px -5px rgba(0,0,0,0.1);">
                <h3 style="margin-top:0;">Restore Backup</h3>
                <p>Select a previous version to restore. This will revert Title, Content, Excerpt, and SEO Meta.</p>
                <div id="rankbot-backup-list" style="max-height:300px; overflow-y:auto; margin-bottom:16px; border:1px solid #e5e7eb; border-radius:6px;">
                    <div style="padding:16px; text-align:center;">Loading backups...</div>
                </div>
                <div style="display:flex; justify-content:flex-end; gap:8px;">
                    <button type="button" class="rankbot-btn btn-text" onclick="document.getElementById('rankbot-restore-modal').style.display='none'">Cancel</button>
                </div>
            </div>
        </div>
        <?php
    }
    
    // Legacy metabox removed in favor of after_title hook
    public function render_page()
    {
        $key = $this->api->get_key();
        $magicToken = get_option('rankbot_magic_token');
        
        // Cache account status to improve page load speed
        $accountStatus = get_transient('rankbot_account_status_full');
        if (false === $accountStatus || !is_array($accountStatus)) {
             $accountStatus = $key ? $this->api->get_account_status() : [];
             // Cache for 60 seconds. Updates frequently enough but prevents sync block on every page refresh
             if ($key && !empty($accountStatus)) {
                 set_transient('rankbot_account_status_full', $accountStatus, 60);
             }
        }

        $balance = $accountStatus['tokens'] ?? 0;
        $plan = $accountStatus['plan']['name'] ?? 'Standard'; // Example structure

        $connectionParams = null;
        $error = null;
        $success = null;
        
        // Filter Logic
        $period_raw = filter_input(INPUT_GET, 'period', FILTER_DEFAULT);
        $period = $period_raw ? sanitize_text_field((string) $period_raw) : '30';
        $cutoff = null;
        if ($period === '7') {
            $cutoff = strtotime('-7 days');
        } elseif ($period === '30') {
            $cutoff = strtotime('-30 days');
        }
        
        $stats = [
            'period' => [
                'optimizations' => 0,
                'tokens_spent' => 0,
                'by_type' => []
            ],
            'total' => [
                'optimizations' => 0,
                'tokens_spent' => 0,
            ],
            'daily_spent' => []
        ];

        $error_raw = filter_input(INPUT_GET, 'error', FILTER_DEFAULT);
        if (null !== $error_raw) {
            $error = sanitize_text_field((string) $error_raw);
        }

        $success_raw = filter_input(INPUT_GET, 'success', FILTER_DEFAULT);
        if (null !== $success_raw) {
            $success = sanitize_text_field((string) $success_raw);
        }

        if ($key) {
            // Verify connection (cached to avoid slow HTTP on every page load)
            $verify = get_transient('rankbot_verify_connection');
            if (false === $verify) {
                $verify = $this->api->verify_connection();
                if (!isset($verify['error'])) {
                    set_transient('rankbot_verify_connection', $verify, 120);
                }
            }
            if (isset($verify['error'])) {
                $error = "Verification failed: " . $verify['error'];
            } else {
                $connectionParams = $verify['data'];
                if (isset($connectionParams['balance']['tokens'])) {
                    $balance = $connectionParams['balance']['tokens'];
                    // Update header cache to match dashboard
                    set_transient('rankbot_admin_balance', $balance, 300);
                }
                if (isset($connectionParams['plan']['name'])) {
                    $plan = $connectionParams['plan']['name'];
                } else {
                    $plan = 'Standard';
                }
            }

            // Fetch History for Stats (cached to avoid slow HTTP on every page load)
            $historyResponse = get_transient('rankbot_dashboard_history');
            if (false === $historyResponse) {
                $historyResponse = $this->api->get_history(1, 300);
                if (!empty($historyResponse)) {
                    set_transient('rankbot_dashboard_history', $historyResponse, 120);
                }
            }
            
            // Handle new API response structure
            $historyItems = [];
            $apiStats = [];
            
            if (isset($historyResponse['history'])) {
                $historyItems = $historyResponse['history'];
                $apiStats = $historyResponse['stats'] ?? [];
            } else {
                 $historyItems = is_array($historyResponse) ? $historyResponse : [];
            }

            // If API provided global totals, use them for the "Total" counters even when
            // the history list is empty (e.g., new installs, filtered views, legacy fallback).
            if (!empty($apiStats) && is_array($apiStats)) {
                $stats['total']['optimizations'] = (int)($apiStats['total_ops'] ?? 0);
                $stats['total']['tokens_spent'] = (float)($apiStats['total_spent'] ?? 0);
            }

            if (!empty($historyItems) && is_array($historyItems)) {
                $daily = [];

                foreach ($historyItems as $item) {
                    $cost = isset($item['cost']) ? floatval($item['cost']) : 0;
                    $type = $item['resource_type'] ?? ($item['meta']['params']['post_type'] ?? ($item['meta']['post_type'] ?? 'unknown'));
                    $createdAt = $item['created_at']; 
                    $ts = strtotime($createdAt);
                    $dateStr = $ts ? gmdate('Y-m-d', $ts) : '';

                    // Fallback calculate totals if API didn't provide them (or for consistency verification)
                    if (empty($apiStats)) {
                        $stats['total']['optimizations']++;
                        $stats['total']['tokens_spent'] += $cost;
                    }

                    // Period Filter
                    if ($period === 'all' || ($cutoff && $ts >= $cutoff)) {
                        $stats['period']['optimizations']++;
                        $stats['period']['tokens_spent'] += $cost;

                        if (!isset($stats['period']['by_type'][$type])) $stats['period']['by_type'][$type] = 0;
                        $stats['period']['by_type'][$type]++;
                        
                        // Daily for Graph (only for period)
                        if (!isset($daily[$dateStr])) $daily[$dateStr] = 0;
                        $daily[$dateStr] += $cost;
                    }
                }
                
                ksort($daily);
                $stats['daily_spent'] = $daily;
                
                if (!empty($stats['period']['by_type'])) {
                    arsort($stats['period']['by_type']);
                }
            }
        }

        // Admin dashboard chart data (enqueue local script; no CDN)
        $rankbot_chart_labels = [];
        $rankbot_chart_data = [];
        if (!empty($stats['daily_spent']) && is_array($stats['daily_spent'])) {
            $sliced = array_slice($stats['daily_spent'], -14, 14, true);
            foreach ($sliced as $d => $v) {
                $tsDay = strtotime((string)$d);
                $rankbot_chart_labels[] = $tsDay ? gmdate('M j', $tsDay) : (string)$d;
                $rankbot_chart_data[] = (float)$v;
            }
        } else {
            $rankbot_chart_labels = ['No Data'];
            $rankbot_chart_data = [0];
        }

        wp_enqueue_script(
            'rankbot-chartjs',
            RANKBOT_PLUGIN_URL . 'assets/js/chart.min.js',
            [],
            '4.4.1',
            true
        );

        wp_enqueue_script(
            'rankbot-admin-dashboard',
            RANKBOT_PLUGIN_URL . 'assets/js/rankbot-admin-dashboard.js',
            ['rankbot-chartjs'],
            defined('RANKBOT_VERSION') ? RANKBOT_VERSION : '1.0.0',
            true
        );

        wp_localize_script('rankbot-admin-dashboard', 'RankBotAdminDashboard', [
            'chartLabels' => array_values($rankbot_chart_labels),
            'chartData' => array_values($rankbot_chart_data),
        ]);

        include RANKBOT_PLUGIN_DIR . 'views/admin-page.php';
    }

    public function handle_actions()
    {
        if (!isset($_POST['rankbot_action']) || !current_user_can('manage_options')) {
            return;
        }

        check_admin_referer('rankbot_action');

        $action = sanitize_text_field(wp_unslash($_POST['rankbot_action']));

        if ($action === 'connect') {
            $apiUrl = isset($_POST['api_url']) ? sanitize_text_field(wp_unslash($_POST['api_url'])) : '';
            update_option('rankbot_api_url', $apiUrl);

            $result = $this->api->register_site();

            if (isset($result['error'])) {
                wp_safe_redirect(add_query_arg([
                    'page' => 'rankbotai-seo-optimizer',
                    'error' => rawurlencode((string) $result['error']),
                ], admin_url('admin.php')));
            } else {
                wp_safe_redirect(add_query_arg([
                    'page' => 'rankbotai-seo-optimizer',
                    'success' => 'connected',
                ], admin_url('admin.php')));
            }
            exit;
        }

        if ($action === 'save_settings') {
            if (isset($_POST['api_url'])) {
                 update_option('rankbot_api_url', sanitize_text_field(wp_unslash($_POST['api_url'])));
            }
            if (!empty($_POST['api_key'])) {
                 $this->api->set_key(sanitize_text_field(wp_unslash($_POST['api_key'])));
            }
            wp_safe_redirect(add_query_arg([
                'page' => 'rankbotai-seo-optimizer',
                'success' => 'settings_saved',
            ], admin_url('admin.php')));
            exit;
        }
        
        if ($action === 'save_global_settings') {
             if (isset($_POST['rankbot_model'])) {
                 update_option('rankbot_selected_model', sanitize_text_field(wp_unslash($_POST['rankbot_model'])));
             }
             if (isset($_POST['rankbot_language'])) {
                 update_option('rankbot_language', sanitize_text_field(wp_unslash($_POST['rankbot_language'])));
             }
             if (isset($_POST['rankbot_show_seo_score'])) {
                 update_option('rankbot_show_seo_score', sanitize_text_field(wp_unslash($_POST['rankbot_show_seo_score'])));
             }

             // New Settings
             update_option('rankbot_update_slug', isset($_POST['rankbot_update_slug']) ? 'yes' : 'no');
             update_option('rankbot_update_product_title', isset($_POST['rankbot_update_product_title']) ? 'yes' : 'no');
             update_option('rankbot_update_main_content', isset($_POST['rankbot_update_main_content']) ? 'yes' : 'no');

             // SEO score backfill
             update_option(self::RANKBOT_SEO_RECALC_ENABLED_OPTION, isset($_POST['rankbot_seo_recalc_enabled']) ? 'yes' : 'no');

             wp_safe_redirect(add_query_arg([
                 'page' => 'rankbot-settings',
                 'success' => 'saved',
             ], admin_url('admin.php')));
             exit;
        }

        if ($action === 'disconnect') {
            $this->api->reset_key();
            wp_safe_redirect(add_query_arg([
                'page' => 'rankbotai-seo-optimizer',
                'success' => 'disconnected',
            ], admin_url('admin.php')));
            exit;
        }
    }

    /* =========================================================
     * WordPress Dashboard Widget — SEO Health
     * ========================================================= */

    /**
     * Register the Dashboard widget.
     */
    public function register_dashboard_widget(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        wp_add_dashboard_widget(
            'rankbot_seo_health_widget',
            '<span class="dashicons dashicons-superhero" style="color:#4f46e5; margin-right:4px;"></span> RankBotAI — ' . esc_html__('SEO Health', 'rankbotai-seo-optimizer'),
            [$this, 'render_dashboard_widget']
        );
    }

    /**
     * Render the content of the Dashboard widget.
     */
    public function render_dashboard_widget(): void
    {
        $key = $this->api->get_key();

        // Count posts/pages
        $total_posts = (int) wp_count_posts('post')->publish;
        $total_pages = (int) wp_count_posts('page')->publish;
        $total_content = $total_posts + $total_pages;

        // Count posts with SEO score
        global $wpdb;
        $scored = (int) $wpdb->get_var(
            "SELECT COUNT(DISTINCT p.ID)
             FROM {$wpdb->posts} p
             INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_rankbot_seo_score'
             WHERE p.post_status = 'publish'
               AND p.post_type IN ('post', 'page')"
        );

        // Average SEO score
        $avg_score = 0;
        if ($scored > 0) {
            $avg_score = (float) $wpdb->get_var(
                "SELECT AVG(CAST(pm.meta_value AS UNSIGNED))
                 FROM {$wpdb->posts} p
                 INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_rankbot_seo_score'
                 WHERE p.post_status = 'publish'
                   AND p.post_type IN ('post', 'page')
                   AND pm.meta_value > 0"
            );
            $avg_score = round($avg_score);
        }

        // Count posts missing meta description
        $missing_desc = (int) $wpdb->get_var(
            "SELECT COUNT(p.ID)
             FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_rankbot_description'
             LEFT JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id AND pm2.meta_key = '_yoast_wpseo_metadesc'
             LEFT JOIN {$wpdb->postmeta} pm3 ON p.ID = pm3.post_id AND pm3.meta_key = 'rank_math_description'
             WHERE p.post_status = 'publish'
               AND p.post_type IN ('post', 'page')
               AND (pm.meta_value IS NULL OR pm.meta_value = '')
               AND (pm2.meta_value IS NULL OR pm2.meta_value = '')
               AND (pm3.meta_value IS NULL OR pm3.meta_value = '')"
        );

        // Count posts missing focus keyword
        $missing_kw = (int) $wpdb->get_var(
            "SELECT COUNT(p.ID)
             FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_rankbot_focus_keyword'
             LEFT JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id AND pm2.meta_key = '_yoast_wpseo_focuskw'
             LEFT JOIN {$wpdb->postmeta} pm3 ON p.ID = pm3.post_id AND pm3.meta_key = 'rank_math_focus_keyword'
             WHERE p.post_status = 'publish'
               AND p.post_type IN ('post', 'page')
               AND (pm.meta_value IS NULL OR pm.meta_value = '')
               AND (pm2.meta_value IS NULL OR pm2.meta_value = '')
               AND (pm3.meta_value IS NULL OR pm3.meta_value = '')"
        );

        // Score color
        $score_color = '#9ca3af'; // gray default
        if ($avg_score >= 70) {
            $score_color = '#10b981';
        } elseif ($avg_score >= 40) {
            $score_color = '#f59e0b';
        } elseif ($avg_score > 0) {
            $score_color = '#ef4444';
        }

        // Calculate coverage percentage
        $coverage = $total_content > 0 ? round(($scored / $total_content) * 100) : 0;
        ?>
        <style>
            .rb-widget-grid { display:grid; grid-template-columns: 1fr 1fr; gap:12px; margin-bottom:16px; }
            .rb-widget-stat { background:#f9fafb; border-radius:8px; padding:12px; text-align:center; border:1px solid #f3f4f6; }
            .rb-widget-stat-value { font-size:24px; font-weight:700; color:#111827; }
            .rb-widget-stat-label { font-size:11px; color:#6b7280; text-transform:uppercase; letter-spacing:0.05em; margin-top:2px; }
            .rb-widget-score-wrap { text-align:center; margin-bottom:16px; }
            .rb-widget-score-circle {
                width:72px; height:72px; border-radius:50%; margin:0 auto 8px auto;
                display:flex; align-items:center; justify-content:center;
                font-size:22px; font-weight:800;
                background: conic-gradient(<?php echo esc_attr($score_color); ?> <?php echo esc_attr((string)$avg_score); ?>%, #f3f4f6 0%);
                position:relative;
            }
            .rb-widget-score-circle::before {
                content:''; position:absolute; inset:6px; background:#fff; border-radius:50%;
            }
            .rb-widget-score-circle span { position:relative; z-index:1; color:<?php echo esc_attr($score_color); ?>; }
            .rb-widget-issue { display:flex; justify-content:space-between; padding:8px 0; border-bottom:1px solid #f3f4f6; font-size:13px; }
            .rb-widget-issue:last-child { border-bottom:none; }
            .rb-widget-issue-count { font-weight:600; }
            .rb-widget-issue-count.warning { color:#f59e0b; }
            .rb-widget-issue-count.danger { color:#ef4444; }
            .rb-widget-issue-count.success { color:#10b981; }
        </style>

        <?php if (!$key): ?>
            <div style="text-align:center; padding:16px 0;">
                <span class="dashicons dashicons-superhero" style="font-size:32px; color:#cbd5e1; width:32px; height:32px;"></span>
                <p style="color:#6b7280; margin:8px 0 16px 0;"><?php echo esc_html__('Connect RankBotAI to see your SEO health.', 'rankbotai-seo-optimizer'); ?></p>
                <a href="<?php echo esc_url(admin_url('admin.php?page=rankbotai-seo-optimizer')); ?>" class="button button-primary"><?php echo esc_html__('Connect Now', 'rankbotai-seo-optimizer'); ?></a>
            </div>
        <?php else: ?>
            <!-- Score Ring -->
            <?php if ($scored > 0): ?>
            <div class="rb-widget-score-wrap">
                <div class="rb-widget-score-circle"><span><?php echo esc_html((string)$avg_score); ?></span></div>
                <div style="font-size:12px; color:#6b7280;"><?php echo esc_html__('Average SEO Score', 'rankbotai-seo-optimizer'); ?></div>
            </div>
            <?php endif; ?>

            <!-- Stats Grid -->
            <div class="rb-widget-grid">
                <div class="rb-widget-stat">
                    <div class="rb-widget-stat-value"><?php echo esc_html((string)$total_content); ?></div>
                    <div class="rb-widget-stat-label"><?php echo esc_html__('Published', 'rankbotai-seo-optimizer'); ?></div>
                </div>
                <div class="rb-widget-stat">
                    <div class="rb-widget-stat-value"><?php echo esc_html((string)$coverage); ?>%</div>
                    <div class="rb-widget-stat-label"><?php echo esc_html__('Scored', 'rankbotai-seo-optimizer'); ?></div>
                </div>
            </div>

            <!-- Issues -->
            <div style="margin-bottom:12px;">
                <div class="rb-widget-issue">
                    <span style="color:#6b7280;"><?php echo esc_html__('Missing meta description', 'rankbotai-seo-optimizer'); ?></span>
                    <span class="rb-widget-issue-count <?php echo $missing_desc > 0 ? 'warning' : 'success'; ?>">
                        <?php echo esc_html((string)$missing_desc); ?>
                    </span>
                </div>
                <div class="rb-widget-issue">
                    <span style="color:#6b7280;"><?php echo esc_html__('Missing focus keyword', 'rankbotai-seo-optimizer'); ?></span>
                    <span class="rb-widget-issue-count <?php echo $missing_kw > 0 ? 'warning' : 'success'; ?>">
                        <?php echo esc_html((string)$missing_kw); ?>
                    </span>
                </div>
                <div class="rb-widget-issue">
                    <span style="color:#6b7280;"><?php echo esc_html__('Not analyzed yet', 'rankbotai-seo-optimizer'); ?></span>
                    <span class="rb-widget-issue-count <?php echo ($total_content - $scored) > 0 ? 'warning' : 'success'; ?>">
                        <?php echo esc_html((string)($total_content - $scored)); ?>
                    </span>
                </div>
            </div>

            <!-- Actions -->
            <div style="display:flex; gap:8px;">
                <a href="<?php echo esc_url(admin_url('admin.php?page=rankbotai-seo-optimizer')); ?>" class="button" style="flex:1; text-align:center;"><?php echo esc_html__('Dashboard', 'rankbotai-seo-optimizer'); ?></a>
                <a href="<?php echo esc_url(admin_url('admin.php?page=rankbot-bulk')); ?>" class="button button-primary" style="flex:1; text-align:center;"><?php echo esc_html__('Bulk Optimize', 'rankbotai-seo-optimizer'); ?></a>
            </div>
        <?php endif; ?>
        <?php
    }

    /* =========================================================
     * llms.txt Generator
     * ========================================================= */

    /**
     * Render the llms.txt generator page.
     */
    public function render_llms_page(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Permission denied', 'rankbotai-seo-optimizer'));
        }

        // Handle save action
        if (isset($_POST['rankbot_llms_action']) && $_POST['rankbot_llms_action'] === 'save') {
            check_admin_referer('rankbot_llms_save', '_wpnonce');

            $content = isset($_POST['llms_content']) ? sanitize_textarea_field(wp_unslash($_POST['llms_content'])) : '';
            update_option('rankbot_llms_txt_content', $content, false);

            // Write the file
            $write_result = $this->write_llms_txt($content);

            wp_safe_redirect(add_query_arg([
                'page'    => 'rankbot-llms',
                'success' => $write_result ? 'saved' : 'saved_no_file',
            ], admin_url('admin.php')));
            exit;
        }

        // Handle delete action
        if (isset($_POST['rankbot_llms_action']) && $_POST['rankbot_llms_action'] === 'delete') {
            check_admin_referer('rankbot_llms_delete', '_wpnonce_delete');
            
            $this->delete_llms_txt();
            delete_option('rankbot_llms_txt_content');

            wp_safe_redirect(add_query_arg([
                'page'    => 'rankbot-llms',
                'success' => 'deleted',
            ], admin_url('admin.php')));
            exit;
        }

        $saved_content = get_option('rankbot_llms_txt_content', '');
        $file_exists   = $this->llms_txt_file_exists();
        $file_url      = home_url('/llms.txt');

        // Force regenerate if requested or empty
        $regenerate = isset($_GET['regenerate']) && $_GET['regenerate'] === '1';
        if (empty($saved_content) || $regenerate) {
            $saved_content = $this->generate_llms_txt_content();
        }

        $success = isset($_GET['success']) ? sanitize_key($_GET['success']) : '';

        $ver = defined('RANKBOT_VERSION') ? RANKBOT_VERSION : '1.0.0';
        wp_enqueue_script(
            'rankbot-llms-js',
            RANKBOT_PLUGIN_URL . 'assets/js/rankbot-llms.js',
            [],
            $ver,
            true
        );

        include RANKBOT_PLUGIN_DIR . 'views/llms-page.php';
    }

    /**
     * Auto-generate llms.txt content from site info.
     */
    private function generate_llms_txt_content(): string
    {
        $site_name   = get_bloginfo('name');
        $description = get_bloginfo('description');
        $home        = home_url('/');

        $lines = [];
        $lines[] = '# ' . $site_name;
        $lines[] = '';
        if ($description) {
            $lines[] = '> ' . $description;
            $lines[] = '';
        }

        // About section
        $lines[] = '## About';
        $lines[] = '';
        $lines[] = $site_name . ' is a website available at ' . $home;
        $lines[] = '';

        // Main pages
        $lines[] = '## Main Pages';
        $lines[] = '';

        // Home
        $lines[] = '- [Home](' . $home . ')';

        // Get published pages
        $pages = get_pages([
            'sort_column' => 'menu_order',
            'sort_order'  => 'ASC',
            'number'      => 20,
            'post_status' => 'publish',
        ]);

        if ($pages) {
            foreach ($pages as $page) {
                $title = get_the_title($page);
                $url   = get_permalink($page);
                if ($title && $url) {
                    $lines[] = '- [' . $title . '](' . $url . ')';
                }
            }
        }

        $lines[] = '';

        // Recent posts
        $recent = get_posts([
            'numberposts'  => 10,
            'post_status'  => 'publish',
            'post_type'    => 'post',
            'orderby'      => 'date',
            'order'        => 'DESC',
        ]);

        if ($recent) {
            $lines[] = '## Recent Content';
            $lines[] = '';
            foreach ($recent as $post) {
                $title = get_the_title($post);
                $url   = get_permalink($post);
                if ($title && $url) {
                    $lines[] = '- [' . $title . '](' . $url . ')';
                }
            }
            $lines[] = '';
        }

        // Categories
        $cats = get_categories(['hide_empty' => true, 'number' => 15]);
        if ($cats && !is_wp_error($cats)) {
            $lines[] = '## Topics';
            $lines[] = '';
            foreach ($cats as $cat) {
                $lines[] = '- [' . $cat->name . '](' . get_category_link($cat) . '): ' . ($cat->description ?: $cat->name . ' articles');
            }
            $lines[] = '';
        }

        return implode("\n", $lines);
    }

    /**
     * Write llms.txt to site root.
     */
    private function write_llms_txt(string $content): bool
    {
        // Try ABSPATH first (standard WordPress)
        $path = ABSPATH . 'llms.txt';

        // Use WP_Filesystem
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }

        if ($wp_filesystem) {
            return $wp_filesystem->put_contents($path, $content, FS_CHMOD_FILE);
        }

        return false;
    }

    /**
     * Delete llms.txt from site root.
     */
    private function delete_llms_txt(): bool
    {
        $path = ABSPATH . 'llms.txt';

        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }

        if ($wp_filesystem && $wp_filesystem->exists($path)) {
            return $wp_filesystem->delete($path);
        }

        return false;
    }

    /**
     * Check if llms.txt exists in site root.
     */
    private function llms_txt_file_exists(): bool
    {
        return file_exists(ABSPATH . 'llms.txt');
    }
}
