/**
 * RankBotAI – Post Toolbar & SEO Engine
 *
 * Used in the classic editor toolbar (render_product_buttons_after_title).
 * Localized object: rankbotToolbarData
 *   nonce       (string)
 *   postId      (int)
 *   hasBackups  (bool)
 *   adminTopUpUrl (string)
 *
 * Also reads window.rbServerData and window.rankbotSingleCosts
 * set via wp_add_inline_script() in the PHP render method.
 */
(function ($) {
    'use strict';

    var nonce   = rankbotToolbarData.nonce;
    var postId  = parseInt(rankbotToolbarData.postId, 10);
    var topUpUrl = rankbotToolbarData.adminTopUpUrl;

    function escapeHtml(str) {
        var div = document.createElement('div');
        div.appendChild(document.createTextNode(str));
        return div.innerHTML;
    }

    /* ------------------------------------------------------------------ */
    /*  Focus Keyword helpers                                              */
    /* ------------------------------------------------------------------ */
    window.saveFocusKeyword = function (e) {
        var kw = document.getElementById('rankbot_focus_keyword').value;
        if (!kw) return alert('Enter a keyword first');

        var btn = e ? e.currentTarget : this;
        btn.innerHTML = 'Saving...';

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'rankbot_save_keyword',
                nonce: nonce,
                post_id: postId,
                keyword: kw
            },
            success: function () {
                btn.innerHTML = 'Saved';
                setTimeout(function () { btn.innerHTML = 'Save'; }, 2000);
            }
        });
    };

    window.generateUniqueKeyword = function (e) {
        var btn = e ? e.currentTarget : this;
        var original = btn.innerHTML;
        btn.innerHTML = 'Generating...';

        function pollKeywordJob(jobId) {
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'rankbot_check_job',
                    nonce: nonce,
                    post_id: postId,
                    job_id: jobId,
                    rankbot_action_type: 'keyword_research'
                },
                success: function (r) {
                    if (r && r.success && r.data && typeof r.data === 'object') {
                        if (r.data.status === 'queued' || r.data.status === 'processing') {
                            setTimeout(function () { pollKeywordJob(jobId); }, 2500);
                            return;
                        }
                        if (r.data.status === 'failed') {
                            btn.innerHTML = original;
                            alert('Error: ' + (r.data.error || 'Keyword generation failed'));
                            return;
                        }
                        if (r.data.status === 'completed' && r.data.result && r.data.result.data) {
                            var k = '';
                            if (typeof r.data.result.data.keyword === 'string') k = r.data.result.data.keyword;
                            if (!k && Array.isArray(r.data.result.data.keywords) && r.data.result.data.keywords.length) k = String(r.data.result.data.keywords[0] || '');
                            k = (k || '').trim();

                            btn.innerHTML = original;
                            if (k) {
                                document.getElementById('rankbot_focus_keyword').value = k;
                                saveFocusKeyword();
                            } else {
                                alert('Error: Empty keyword received');
                            }
                            return;
                        }
                        setTimeout(function () { pollKeywordJob(jobId); }, 2500);
                    } else if (r && r.success === false) {
                        btn.innerHTML = original;
                        alert('Error: ' + (r.data || 'Keyword generation failed'));
                    } else {
                        setTimeout(function () { pollKeywordJob(jobId); }, 2500);
                    }
                },
                error: function () {
                    setTimeout(function () { pollKeywordJob(jobId); }, 3500);
                }
            });
        }

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'rankbot_generate_unique_keyword',
                nonce: nonce,
                post_id: postId
            },
            success: function (res) {
                btn.innerHTML = original;
                if (res.success) {
                    if (res.data && typeof res.data === 'object' && res.data.status === 'queued' && res.data.job_id) {
                        pollKeywordJob(res.data.job_id);
                        return;
                    }
                    document.getElementById('rankbot_focus_keyword').value = res.data.keyword;
                    if (res.data.warning) alert(res.data.warning);
                    saveFocusKeyword();
                } else {
                    alert('Error: ' + res.data);
                }
            },
            error: function () {
                btn.innerHTML = original;
                alert('Network Error');
            }
        });
    };

    /* ------------------------------------------------------------------ */
    /*  Client-Side SEO Engine                                             */
    /* ------------------------------------------------------------------ */
    function getEditorContent() {
        var content = '';
        if (typeof tinymce !== 'undefined' && tinymce.activeEditor && !tinymce.activeEditor.isHidden()) {
            content = tinymce.activeEditor.getContent();
        } else {
            var el = document.getElementById('content');
            if (el) content = el.value;
        }
        if (!content && typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
            content = wp.data.select('core/editor').getEditedPostContent();
        }
        return content || '';
    }

    function getEditorTitle() {
        var title = '';
        var yoastT = document.querySelector('input[name="yoast_wpseo_title"]');
        if (yoastT && yoastT.value) title = yoastT.value;

        if (!title) {
            var rankT = document.querySelector('input[name="rank_math_title"]');
            if (rankT && rankT.value) title = rankT.value;
        }
        if (!title) {
            var rmPreview = document.querySelector('.rank-math-snippet-title');
            if (rmPreview) title = rmPreview.innerText;
            if (!title) {
                var rmGuten = document.querySelector('.rank-math-google-preview-title');
                if (rmGuten) title = rmGuten.innerText;
            }
        }
        if (!title && typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
            var meta = wp.data.select('core/editor').getEditedPostAttribute('meta');
            if (meta) {
                if (meta['_yoast_wpseo_title']) title = meta['_yoast_wpseo_title'];
                else if (meta['rank_math_title']) title = meta['rank_math_title'];
            }
            if (!title) {
                title = wp.data.select('core/editor').getEditedPostAttribute('title');
            }
        }
        if (!title && document.getElementById('title')) {
            title = document.getElementById('title').value;
        }
        return title || '';
    }

    function getEditorDescription() {
        var desc = '';
        var yoast = document.querySelector('textarea[name="yoast_wpseo_metadesc"]');
        if (yoast && yoast.value) desc = yoast.value;

        if (!desc) {
            var rankD = document.querySelector('textarea[name="rank_math_description"]');
            if (rankD && rankD.value) desc = rankD.value;
        }
        if (!desc) {
            var rmPreview = document.querySelector('.rank-math-snippet-description');
            if (rmPreview) desc = rmPreview.innerText;
        }
        if (!desc && typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
            var meta = wp.data.select('core/editor').getEditedPostAttribute('meta');
            if (meta) {
                if (meta['_yoast_wpseo_metadesc']) desc = meta['_yoast_wpseo_metadesc'];
                else if (meta['rank_math_description']) desc = meta['rank_math_description'];
            }
        }
        if (!desc) {
            var excerpt = document.getElementById('excerpt');
            if (excerpt) desc = excerpt.value;
        }
        return desc || '';
    }

    window.rbToggleAccordion = function (header) {
        header.parentElement.classList.toggle('open');
        var icon = header.querySelector('.dashicons-arrow-right, .dashicons-arrow-down');
        if (icon) icon.className = header.parentElement.classList.contains('open') ? 'dashicons dashicons-arrow-down' : 'dashicons dashicons-arrow-right';
    };

    function fetchExternalKeyword() {
        var kw = '';
        var source = '';

        if (window.rbServerData && window.rbServerData.focusKeyword) {
            kw = window.rbServerData.focusKeyword;
            source = window.rbServerData.source + ' (Saved)';
        }

        if (typeof wp !== 'undefined' && wp.data && wp.data.select) {
            try {
                var editor = wp.data.select('core/editor');
                if (editor) {
                    var meta = editor.getEditedPostAttribute('meta');
                    if (meta) {
                        if (meta['_yoast_wpseo_focuskw']) {
                            kw = meta['_yoast_wpseo_focuskw'];
                            source = 'Yoast (Gutenberg)';
                        } else if (meta['rank_math_focus_keyword']) {
                            kw = meta['rank_math_focus_keyword'];
                            source = 'RankMath (Gutenberg)';
                        }
                    }
                }
            } catch (e) { /* no-op */ }
        }

        if (!kw) {
            var yoast = document.querySelector('input[name="yoast_wpseo_focuskw"]') || document.getElementById('yoast_wpseo_focuskw');
            if (yoast && yoast.value) { kw = yoast.value; source = 'Yoast'; }
            if (!kw) {
                var rm = document.querySelector('input[name="rank_math_focus_keyword"]');
                if (rm && rm.value) { kw = rm.value; source = 'RankMath'; }
            }
        }
        if (!kw) {
            var stored = document.getElementById('rankbot_focus_keyword');
            if (stored && stored.value) { kw = stored.value; source = 'Manual'; }
        }
        if (kw && kw.indexOf(',') !== -1) {
            kw = kw.split(',')[0].trim();
        }

        if (kw) {
            var hInput = document.getElementById('rankbot_focus_keyword');
            if (hInput) hInput.value = kw;
            var d = document.getElementById('rb-current-kw');
            if (d) d.innerText = kw;
            var b = document.getElementById('rb-kw-source');
            if (b) {
                b.innerText = source;
                b.style.display = 'inline-block';
            }
        } else {
            var d2 = document.getElementById('rb-current-kw');
            if (d2) d2.innerText = 'Not Set';
        }
        return kw;
    }

    window.toggleSeoDetails = function () {
        var list = document.getElementById('rb-checklist');
        var btn = document.getElementById('rb-toggle-details');
        if (list.style.display === 'none') {
            list.style.display = 'block';
            btn.innerHTML = 'Hide Details ▴';
        } else {
            list.style.display = 'none';
            btn.innerHTML = 'Show Details ▾';
        }
    };

    function runLiveAnalysis() {
        var kw = fetchExternalKeyword();
        var scorePanel = document.getElementById('rankbot-seo-results');

        if (scorePanel && kw && document.getElementById('rb-current-kw')) {
            document.getElementById('rb-current-kw').innerText = kw;
        }

        if (!kw) {
            if (document.getElementById('rb-checklist')) {
                document.getElementById('rb-checklist').innerHTML = '<div style="padding:20px; text-align:center; color:#94a3b8;">Set a Focus Keyword to begin analysis.</div>';
            }
            return;
        }

        var listEl = document.getElementById('rb-checklist');
        var debugEl = document.getElementById('rb-debug-info');
        if (debugEl) {
            debugEl.style.display = 'block';
            debugEl.innerText = 'Analyzing saved post/meta | KW[' + kw + ']';
        }

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'rankbot_analyze_seo',
                nonce: nonce,
                post_id: postId
            },
            success: function (response) {
                if (response.success) {
                    renderSeoResults(response.data);
                } else {
                    if (listEl) listEl.innerHTML = '<div style="color:red; padding:10px;">Analysis Failed: ' + (response.data || 'Unknown') + '</div>';
                }
            }
        });
    }

    function renderSeoResults(data) {
        var totalScore = data.score;
        var results = data.results;

        var GROUPS = {
            title: { label: 'Title Tag', icon: 'dashicons-heading' },
            meta: { label: 'Meta Description', icon: 'dashicons-info-outline' },
            url: { label: 'URL Structure', icon: 'dashicons-admin-links' },
            headings: { label: 'Headings (H1-H6)', icon: 'dashicons-editor-justify' },
            media: { label: 'Images & Alt Text', icon: 'dashicons-format-image' },
            links: { label: 'Links Strategy', icon: 'dashicons-admin-site' },
            content: { label: 'Content Analysis', icon: 'dashicons-welcome-write-blog' },
            additional: { label: 'Additional', icon: 'dashicons-plus-alt' },
            basic: { label: 'General', icon: 'dashicons-admin-generic' }
        };

        var scoreEl = document.getElementById('rb-score-val');
        var barEl = document.getElementById('rb-progress-bar');
        var sideEl = document.getElementById('rb-sidebar-score-val');
        var hInput = document.getElementById('rankbot_seo_score_input');

        if (scoreEl) {
            var totalChecks = results.length;
            var passedChecks = results.filter(function (r) { return r.status === 'good'; }).length;
            scoreEl.innerHTML = '<div style="font-size:32px; font-weight:800; line-height:1;">' + totalScore + '/100</div>' +
                '<div style="font-size:12px; color:#64748b; margin-top:4px;">Passed: ' + passedChecks + '/' + totalChecks + '</div>';
        }
        if (hInput) hInput.value = totalScore;

        var colorClass = 'rankbot-score-bad';
        if (totalScore >= 80) colorClass = 'rankbot-score-good';
        else if (totalScore >= 50) colorClass = 'rankbot-score-ok';

        if (sideEl) {
            sideEl.classList.remove('rankbot-score-bad', 'rankbot-score-ok', 'rankbot-score-good');
            sideEl.classList.add(colorClass);
            if (!sideEl.classList.contains('rankbot-score-ring')) {
                sideEl.classList.add('rankbot-score-ring');
            }
            sideEl.style.setProperty('--rb-score', String(totalScore));
            var numEl = sideEl.querySelector('.rb-score-num');
            if (!numEl) {
                sideEl.innerHTML = '<div class="rb-score-text"><div class="rb-score-num">0</div><div class="rb-score-sub">/100</div></div>';
                numEl = sideEl.querySelector('.rb-score-num');
            }
            if (numEl) numEl.textContent = String(totalScore);
        }

        if (barEl) {
            barEl.style.width = totalScore + '%';
            barEl.className = 'rb-progress-bar ' + colorClass;
        }

        var debugEl = document.getElementById('rb-debug-info');
        if (debugEl && data.debug) {
            var tSrc = data.debug.title_source || 'n/a';
            var dSrc = data.debug.desc_source || 'n/a';
            var tLen = data.debug.title_len != null ? data.debug.title_len : 'n/a';
            var dLen = data.debug.desc_len != null ? data.debug.desc_len : 'n/a';
            var tPrev = data.debug.title_preview || '';
            var dPrev = data.debug.desc_preview || '';
            debugEl.style.display = 'block';
            debugEl.innerText = 'Analyzed: Title(' + tLen + ') from ' + tSrc + ': ' + tPrev + ' | Desc(' + dLen + ') from ' + dSrc + ': ' + dPrev;
        }

        var listEl = document.getElementById('rb-checklist');
        if (!listEl) return;

        var html = '';
        var grouped = {};
        results.forEach(function (r) {
            if (!grouped[r.group]) grouped[r.group] = [];
            grouped[r.group].push(r);
        });

        var groupOrder = ['title', 'meta', 'url', 'headings', 'content', 'media', 'links', 'additional', 'basic'];

        groupOrder.forEach(function (grpKey) {
            if (!grouped[grpKey]) return;
            var gInfo = GROUPS[grpKey] || { label: 'Other', icon: 'dashicons-star-filled' };
            html += '<div style="margin-bottom: 8px;"><div style="text-transform:uppercase; font-size:11px; font-weight:700; letter-spacing:0.5px; color:#94a3b8; margin: 12px 0 6px;">' + gInfo.label + '</div>';

            grouped[grpKey].forEach(function (r) {
                var icon = 'dashicons-warning';
                var labelColor = '#ef4444';

                if (r.status === 'ok') {
                    icon = 'dashicons-minus';
                    labelColor = '#f59e0b';
                } else if (r.status === 'good') {
                    icon = 'dashicons-yes-alt';
                    labelColor = '#22c55e';
                }

                var scoreBadge = '<span style="font-size:11px; padding:2px 8px; border-radius:12px; background:#f8fafc; border:1px solid #e2e8f0; color:#64748b; font-weight:500;">' + parseInt(r.score) + '/' + parseInt(r.maxPoints) + '</span>';
                var msgColor = (r.status === 'good') ? '#22c55e' : '#475569';
                var recBlock = '<div class="rb-accordion-content"><div style="color:' + msgColor + '; font-size:13px; line-height:1.4;">' + escapeHtml(r.message) + '</div></div>';

                html += '<div class="rb-accordion-item">' +
                    '<div class="rb-accordion-header" onclick="rbToggleAccordion(this)" style="border-left: 3px solid ' + labelColor + ';">' +
                    '<div style="display:flex; align-items:center; gap:10px; flex:1;">' +
                    '<span class="dashicons ' + icon + '" style="color:' + labelColor + '; font-size:18px;"></span>' +
                    '<span style="font-weight:500; color:#334155;">' + escapeHtml(r.label) + '</span></div>' +
                    scoreBadge + '</div>' + recBlock + '</div>';
            });

            html += '</div>';
        });

        listEl.innerHTML = html;
    }

    document.addEventListener('DOMContentLoaded', function () {
        setTimeout(runLiveAnalysis, 2500);
    });

    window.analyzeSEO = function () {
        runLiveAnalysis();
    };

    /* ------------------------------------------------------------------ */
    /*  Restore Modal (Toolbar)                                            */
    /* ------------------------------------------------------------------ */
    window.openRestoreModal = function () {
        var modal = document.getElementById('rankbot-restore-modal');
        var list = document.getElementById('rankbot-backup-list');
        modal.style.display = 'flex';

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'rankbot_get_backups',
                nonce: nonce,
                post_id: postId
            },
            success: function (response) {
                if (response.success && response.data.backups.length > 0) {
                    var html = '<table style="width:100%; text-align:left; border-collapse:collapse;">';
                    response.data.backups.forEach(function (b) {
                        var typeLabel = escapeHtml(b.action ? String(b.action) : 'legacy');
                        var safeDate = escapeHtml(b.date || '');
                        var safeTs = escapeHtml(String(b.timestamp || ''));
                        html += '<tr style="border-bottom:1px solid #f3f4f6;"><td style="padding:10px;">' + safeDate + '<div style="font-size:11px; color:#64748b; margin-top:2px;">' + typeLabel + '</div></td>' +
                            '<td style="padding:10px; text-align:right;"><button onclick="restoreBackup(\'' + safeTs + '\')" class="rankbot-btn btn-primary" style="padding:4px 12px; font-size:12px;">Restore</button></td></tr>';
                    });
                    html += '</table>';
                    list.innerHTML = html;
                } else {
                    list.innerHTML = '<div style="padding:16px; text-align:center;">No backups found.</div>';
                }
            }
        });
    };

    window.restoreBackup = function (timestamp) {
        if (!confirm('Are you sure? Current changes will be overwritten.')) return;

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'rankbot_restore_backup',
                nonce: nonce,
                post_id: postId,
                timestamp: timestamp
            },
            success: function (response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert('Error: ' + response.data);
                }
            }
        });
    };

    /* ------------------------------------------------------------------ */
    /*  Modern Modal Logic                                                 */
    /* ------------------------------------------------------------------ */
    var RB_Titles = {
        'research': 'Keyword Research',
        'snippet': 'Generate Snippet',
        'complex': 'Auto-Optimize SEO'
    };
    var RB_CurrentAction = null;
    var RB_EstimateCache = {};

    function rbFormatTokens(val) {
        var num = Number(val);
        if (!isFinite(num)) return '--';
        if (Math.abs(num - Math.round(num)) < 0.000001) return String(Math.round(num));
        return String(num.toFixed(2)).replace(/\.00$/, '');
    }

    function rbFetchEstimateCost(action, onDone) {
        if (typeof onDone !== 'function') return;

        var k = String(action || '');
        if (k && RB_EstimateCache[k]) {
            onDone(null, RB_EstimateCache[k]);
            return;
        }

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'rankbot_estimate_cost',
                nonce: nonce,
                mode: 'selected',
                ids: [postId],
                bulk_action: action
            },
            success: function (res) {
                if (res && res.success && res.data) {
                    if (k) RB_EstimateCache[k] = res.data;
                    onDone(null, res.data);
                } else {
                    onDone(new Error('estimate_failed'), null);
                }
            },
            error: function () {
                onDone(new Error('network'), null);
            }
        });
    }

    window.RankBotModal = {
        init: function () {
            if (document.getElementById('rankbot-process-modal')) return;
            var html = '<div id="rankbot-process-modal" class="rankbot-modal-overlay" style="display:none;">' +
                '<div class="rankbot-modal-card">' +
                '<button type="button" class="rankbot-modal-close" onclick="RankBotModal.close()">&times;</button>' +

                '<!-- STEP 1: CONFIRM -->' +
                '<div class="rankbot-modal-content" id="rankbot-modal-step-confirm">' +
                '<div class="rankbot-modal-icon-wrap"><span class="dashicons dashicons-superhero rankbot-modal-icon"></span></div>' +
                '<h3 class="rankbot-modal-title" id="rankbot-modal-main-title">Start Optimization?</h3>' +
                '<p class="rankbot-modal-desc">RankBot will analyze your content using GPT-4o and apply improvements automatically.</p>' +
                '<div class="rankbot-modal-cost-row"><span style="color:#64748b; font-size:13px; font-weight:500;">Estimated Cost:</span>' +
                '<div class="rankbot-cost-badge-large"><span class="dashicons dashicons-database"></span> <span id="rankbot-cost-display">--</span> Tokens</div></div>' +
                '<div class="rankbot-modal-actions"><button class="rankbot-btn-sec" onclick="RankBotModal.close()">Cancel</button>' +
                '<button class="rankbot-btn-pri" id="rankbot-modal-run-btn" onclick="RankBotModal.start()"><span class="dashicons dashicons-controls-play" style="font-size:16px;"></span> Run Now</button></div>' +
                '<div id="rankbot-modal-inline-error" style="display:none; margin-top:14px; background:#fff7ed; border:1px solid #fed7aa; color:#9a3412; padding:10px 12px; border-radius:12px; text-align:left; font-size:13px; line-height:1.4;"></div>' +
                '</div>' +

                '<!-- STEP 2: PROCESSING -->' +
                '<div class="rankbot-modal-content" id="rankbot-modal-step-process" style="display:none;">' +
                '<div class="rb-sci-spinner-wrap"><div class="rb-sci-spinner"><div class="rb-sci-ring rb-sci-ring-1"></div><div class="rb-sci-ring rb-sci-ring-2"></div><div class="rb-sci-ring rb-sci-ring-3"></div><div class="rb-sci-core"></div></div></div>' +
                '<h3 class="rankbot-modal-title">AI Processing Core</h3>' +
                '<p class="rankbot-modal-desc">Neural network analyzing content structure...<br><span style="font-size:12px; opacity:0.8;">You can close window, task runs in background.</span></p>' +
                '<div class="rankbot-modal-actions"><button class="rankbot-btn-sec" onclick="RankBotModal.close()">Close & Check Later</button></div>' +
                '</div>' +

                '<!-- STEP 3: SUCCESS -->' +
                '<div class="rankbot-modal-content" id="rankbot-modal-step-success" style="display:none;">' +
                '<div class="rankbot-success-anim"><svg viewBox="0 0 24 24" fill="none" class="rankbot-checkmark"><path d="M5 13L9 17L19 7" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"/></svg></div>' +
                '<h3 class="rankbot-modal-title">Optimization Done!</h3>' +
                '<p class="rankbot-modal-desc" id="rankbot-success-msg">Content successfully enhanced.</p>' +
                '<div class="rankbot-modal-actions-single"><button class="rankbot-btn-success" onclick="location.reload()">Refresh Page</button></div>' +
                '</div>' +

                '<!-- STEP 4: ERROR -->' +
                '<div class="rankbot-modal-content" id="rankbot-modal-step-error" style="display:none;">' +
                '<div class="rankbot-modal-icon-wrap" style="background:#fef2f2; color:#ef4444;"><span class="dashicons dashicons-warning rankbot-modal-icon"></span></div>' +
                '<h3 class="rankbot-modal-title" id="rankbot-error-title">Something went wrong</h3>' +
                '<p class="rankbot-modal-desc" id="rankbot-error-msg">Please try again.</p>' +
                '<div class="rankbot-modal-actions-single" id="rankbot-error-actions">' +
                '<button class="rankbot-btn-sec" onclick="RankBotModal.close()">Close</button>' +
                '<a class="rankbot-btn-pri" id="rankbot-topup-link" href="' + topUpUrl + '" style="display:none; text-decoration:none; justify-content:center; align-items:center; gap:8px;">Top Up Balance</a>' +
                '</div></div>' +

                '</div></div>';
            document.body.insertAdjacentHTML('beforeend', html);
        },
        open: function (action) {
            this.init();
            RB_CurrentAction = action;
            var title = RB_Titles[action] || 'Optimization';
            document.getElementById('rankbot-modal-main-title').innerText = title;

            var costEl = document.getElementById('rankbot-cost-display');
            var instantCost = null;
            try {
                if (window.rankbotSingleCosts && typeof window.rankbotSingleCosts[action] !== 'undefined') {
                    instantCost = window.rankbotSingleCosts[action];
                }
            } catch (e) { }
            if (costEl) {
                costEl.innerText = (instantCost !== null) ? rbFormatTokens(instantCost) : '...';
            }

            document.getElementById('rankbot-modal-step-confirm').style.display = 'block';
            document.getElementById('rankbot-modal-step-process').style.display = 'none';
            document.getElementById('rankbot-modal-step-success').style.display = 'none';
            document.getElementById('rankbot-modal-step-error').style.display = 'none';

            try {
                var errEl = document.getElementById('rankbot-modal-inline-error');
                if (errEl) { errEl.style.display = 'none'; errEl.innerText = ''; }
                var runBtn = document.getElementById('rankbot-modal-run-btn');
                if (runBtn) { runBtn.disabled = false; runBtn.style.opacity = ''; runBtn.style.cursor = ''; }
            } catch (e) { }

            document.getElementById('rankbot-process-modal').style.display = 'flex';

            rbFetchEstimateCost(action, function (err, data) {
                var costEl2 = document.getElementById('rankbot-cost-display');
                if (!costEl2) return;
                if (err || !data) {
                    costEl2.innerText = '--';
                    return;
                }
                costEl2.innerText = rbFormatTokens(data.total_cost);

                try {
                    if (!window.RB_LastEstimateByAction) window.RB_LastEstimateByAction = {};
                    window.RB_LastEstimateByAction[action] = data;
                } catch (e) { }

                try {
                    var bal = Number(data.balance);
                    var cost = Number(data.total_cost);
                    var canAfford = (Number.isFinite(bal) && Number.isFinite(cost)) ? (bal >= cost) : true;
                    if (!canAfford) {
                        var shortfall = Math.max(0, cost - bal);
                        var runBtn2 = document.getElementById('rankbot-modal-run-btn');
                        if (runBtn2) {
                            runBtn2.disabled = true;
                            runBtn2.style.opacity = '0.6';
                            runBtn2.style.cursor = 'not-allowed';
                        }
                        var errEl2 = document.getElementById('rankbot-modal-inline-error');
                        if (errEl2) {
                            errEl2.innerHTML = 'Insufficient balance. Missing <strong>' + rbFormatTokens(shortfall) + '</strong> tokens. Please top up to continue.';
                            errEl2.style.display = 'block';
                        }
                    }
                } catch (e) { }

                try {
                    var descEls = document.querySelectorAll('#rankbot-modal-step-confirm .rankbot-modal-desc');
                    if (descEls && descEls.length && data.model_label) {
                        descEls[0].innerHTML = 'RankBot will analyze your content using <strong>' + String(data.model_label) + '</strong> and apply improvements automatically.';
                    }
                } catch (e) { }
            });
        },
        close: function () {
            var el = document.getElementById('rankbot-process-modal');
            if (el) el.style.display = 'none';
        },
        start: function () {
            try {
                var last = (window.RB_LastEstimateByAction && window.RB_LastEstimateByAction[RB_CurrentAction]) ? window.RB_LastEstimateByAction[RB_CurrentAction] : null;
                if (last) {
                    var bal = Number(last.balance);
                    var cost = Number(last.total_cost);
                    if (Number.isFinite(bal) && Number.isFinite(cost) && bal < cost) {
                        this.showError('Insufficient balance. Please top up and try again.', { topup: true, title: 'Insufficient Balance' });
                        return;
                    }
                }
            } catch (e) { }

            document.getElementById('rankbot-modal-step-confirm').style.display = 'none';
            document.getElementById('rankbot-modal-step-process').style.display = 'block';
            rankbotRunAjax(RB_CurrentAction);
        },
        showError: function (msg, opts) {
            opts = opts || {};
            var el = document.getElementById('rankbot-process-modal');
            if (el && el.style.display !== 'none') {
                document.getElementById('rankbot-modal-step-confirm').style.display = 'none';
                document.getElementById('rankbot-modal-step-process').style.display = 'none';
                document.getElementById('rankbot-modal-step-success').style.display = 'none';
                document.getElementById('rankbot-modal-step-error').style.display = 'block';
                var t = document.getElementById('rankbot-error-title');
                var m = document.getElementById('rankbot-error-msg');
                if (t) t.innerText = opts.title ? String(opts.title) : 'Something went wrong';
                if (m) m.innerText = msg ? String(msg) : 'Please try again.';
                var topUp = document.getElementById('rankbot-topup-link');
                if (topUp) topUp.style.display = opts.topup ? 'inline-flex' : 'none';
            }
        },
        showSuccess: function (msg) {
            var el = document.getElementById('rankbot-process-modal');
            if (el && el.style.display !== 'none') {
                document.getElementById('rankbot-modal-step-process').style.display = 'none';
                document.getElementById('rankbot-modal-step-success').style.display = 'block';
                if (msg) document.getElementById('rankbot-success-msg').innerText = msg;
            } else {
                location.reload();
            }
        }
    };

    window.rankbotAction = function (action) {
        var btn = event.currentTarget;
        if (btn && btn.classList.contains('rankbot-disabled')) return;
        RankBotModal.open(action);
    };

    function rankbotRunAjax(action) {
        var rankbotJobCheckErrors = 0;

        function pollJob(jobId) {
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'rankbot_check_job',
                    nonce: nonce,
                    post_id: postId,
                    job_id: jobId,
                    rankbot_action_type: action
                },
                success: function (response) {
                    if (response && response.success && response.data && typeof response.data === 'object') {
                        rankbotJobCheckErrors = 0;
                        if (response.data.status === 'queued' || response.data.status === 'processing') {
                            setTimeout(function () { pollJob(jobId); }, 3000);
                        } else if (response.data.status === 'completed' || (response.data.message && !response.data.job_id)) {
                            RankBotModal.showSuccess(response.data.message);
                        } else {
                            RankBotModal.showError((response.data.error || 'Unknown error'), { title: 'Failed' });
                        }
                    } else {
                        if (response && response.success === false) {
                            RankBotModal.showError((response.data || 'Unknown error'), { title: 'Failed' });
                            return;
                        }
                        rankbotJobCheckErrors++;
                        if (rankbotJobCheckErrors <= 10) {
                            setTimeout(function () { pollJob(jobId); }, 3000);
                        } else {
                            RankBotModal.showError('Invalid response while checking job status.', { title: 'Error' });
                        }
                    }
                },
                error: function () {
                    rankbotJobCheckErrors++;
                    if (rankbotJobCheckErrors <= 10) {
                        setTimeout(function () { pollJob(jobId); }, 5000);
                    } else {
                        RankBotModal.showError('Network issues while checking job status.', { title: 'Network Error' });
                    }
                }
            });
        }

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'rankbot_optimize',
                nonce: nonce,
                post_id: postId,
                rankbot_action_type: action
            },
            success: function (response) {
                if (response && response.success && response.data && typeof response.data === 'object') {
                    if (response.data.status === 'queued' && response.data.job_id) {
                        pollJob(response.data.job_id);
                    } else {
                        RankBotModal.showSuccess(response.data.message);
                    }
                } else {
                    var msg = (response && response.data) ? response.data : 'Unknown error';
                    RankBotModal.showError(msg, { title: 'Error', topup: (String(msg).toLowerCase().indexOf('insufficient') !== -1) });
                }
            },
            error: function (xhr, status, error) {
                RankBotModal.showError('Network error: ' + error, { title: 'Network Error' });
            }
        });
    }

})(jQuery);
