/**
 * RankBotAI Toast Notification System
 * Provides beautiful, modern toast notifications.
 *
 * Usage:
 *   RankBotToast.success('Title saved!');
 *   RankBotToast.error('Failed to connect.');
 *   RankBotToast.warning('Low balance.');
 *   RankBotToast.info('Processing started.');
 *   RankBotToast.show({ type: 'success', message: '...', duration: 5000 });
 */
(function () {
    'use strict';

    var CONTAINER_ID = 'rb-toast-container';
    var DEFAULT_DURATION = 4000;

    var ICONS = {
        success: '<svg width="20" height="20" viewBox="0 0 20 20" fill="none"><circle cx="10" cy="10" r="10" fill="#10b981"/><path d="M6 10.5l2.5 2.5 5.5-5.5" stroke="#fff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        error:   '<svg width="20" height="20" viewBox="0 0 20 20" fill="none"><circle cx="10" cy="10" r="10" fill="#ef4444"/><path d="M7 7l6 6M13 7l-6 6" stroke="#fff" stroke-width="2" stroke-linecap="round"/></svg>',
        warning: '<svg width="20" height="20" viewBox="0 0 20 20" fill="none"><circle cx="10" cy="10" r="10" fill="#f59e0b"/><path d="M10 6v5M10 13.5v.5" stroke="#fff" stroke-width="2" stroke-linecap="round"/></svg>',
        info:    '<svg width="20" height="20" viewBox="0 0 20 20" fill="none"><circle cx="10" cy="10" r="10" fill="#3b82f6"/><path d="M10 9v5M10 6.5v.5" stroke="#fff" stroke-width="2" stroke-linecap="round"/></svg>'
    };

    function getContainer() {
        var el = document.getElementById(CONTAINER_ID);
        if (!el) {
            el = document.createElement('div');
            el.id = CONTAINER_ID;
            document.body.appendChild(el);
        }
        return el;
    }

    function escapeHtml(str) {
        var div = document.createElement('div');
        div.appendChild(document.createTextNode(str));
        return div.innerHTML;
    }

    function show(opts) {
        if (typeof opts === 'string') {
            opts = { message: opts };
        }
        var type = opts.type || 'info';
        var message = opts.message || '';
        var duration = opts.duration !== undefined ? opts.duration : DEFAULT_DURATION;

        var container = getContainer();

        var toast = document.createElement('div');
        toast.className = 'rb-toast rb-toast-' + type;
        toast.innerHTML =
            '<div class="rb-toast-icon">' + (ICONS[type] || ICONS.info) + '</div>' +
            '<div class="rb-toast-message">' + escapeHtml(message) + '</div>' +
            '<button class="rb-toast-close" type="button" aria-label="Close">&times;</button>';

        // Close on click
        toast.querySelector('.rb-toast-close').addEventListener('click', function () {
            dismiss(toast);
        });

        container.appendChild(toast);

        // Trigger enter animation
        requestAnimationFrame(function () {
            toast.classList.add('rb-toast-visible');
        });

        // Auto dismiss
        if (duration > 0) {
            setTimeout(function () {
                dismiss(toast);
            }, duration);
        }

        return toast;
    }

    function dismiss(toast) {
        if (!toast || toast.classList.contains('rb-toast-leaving')) {
            return;
        }
        toast.classList.add('rb-toast-leaving');
        toast.classList.remove('rb-toast-visible');
        setTimeout(function () {
            if (toast.parentNode) {
                toast.parentNode.removeChild(toast);
            }
        }, 300);
    }

    // Public API
    window.RankBotToast = {
        show: show,
        success: function (msg, duration) { return show({ type: 'success', message: msg, duration: duration }); },
        error:   function (msg, duration) { return show({ type: 'error', message: msg, duration: duration }); },
        warning: function (msg, duration) { return show({ type: 'warning', message: msg, duration: duration }); },
        info:    function (msg, duration) { return show({ type: 'info', message: msg, duration: duration }); },
        dismiss: dismiss
    };

    // Auto-convert WordPress admin notices to toasts on admin pages
    document.addEventListener('DOMContentLoaded', function () {
        var notices = document.querySelectorAll('.rankbot-wrap .notice, .rb-bulk-wrap .notice');
        notices.forEach(function (notice) {
            var text = (notice.textContent || '').trim();
            if (!text) return;

            var type = 'info';
            if (notice.classList.contains('notice-success') || notice.classList.contains('updated')) {
                type = 'success';
            } else if (notice.classList.contains('notice-error')) {
                type = 'error';
            } else if (notice.classList.contains('notice-warning')) {
                type = 'warning';
            }

            show({ type: type, message: text, duration: 6000 });
            notice.style.display = 'none';
        });
    });
})();
