/**
 * RankBotAI – Post Optimizer Metabox
 *
 * Used in the post optimizer metabox (render_post_optimizer_metabox).
 * Localized object: rankbotPostData
 *   - pendingJobId (string) Job ID to auto-resume (empty if none)
 *   - nonce        (string) Security nonce
 *   - postId       (int)    Current post ID
 *
 * @package RankBotAI
 */

/* global jQuery, ajaxurl, rankbotPostData */

jQuery(document).ready(function ($) {
    'use strict';

    var pendingJobId = rankbotPostData.pendingJobId;
    var nonce        = rankbotPostData.nonce;
    var postId       = rankbotPostData.postId;

    var wrap       = $('#rankbot-post-optimizer-wrap');
    var spinner    = wrap.find('.spinner');
    var log        = $('#rankbot-post-log');
    var btn        = wrap.find('.rankbot-optimize-post-btn');
    var statusText = wrap.find('.rankbot-status-text');

    // ---------- Dismiss Logic ----------

    $('.rankbot-dismiss-job').on('click', function () {
        var jobId  = $(this).data('id');
        var notice = $(this).closest('.rankbot-notice');
        notice.css('opacity', 0.5);

        $.post(ajaxurl, {
            action: 'rankbot_dismiss_job',
            job_id: jobId,
            nonce:  nonce
        }, function () {
            notice.slideUp();
        });
    });

    // ---------- Auto Resume Logic ----------

    if (pendingJobId) {
        setLoading(true);
        log.text('Resuming background task (' + pendingJobId + ')...').show();
        pollJob(pendingJobId);
    }

    // ---------- Optimize Button (Delegation for Gutenberg) ----------

    $(document).on('click', '.rankbot-optimize-post-btn', function (e) {
        e.preventDefault();

        if (!confirm('This will rewrite the article content using AI. Existing media/images will be preserved. Continue?')) {
            return;
        }

        setLoading(true);
        log.text('Starting...').show();

        // Step 1: Start Job
        $.post(ajaxurl, {
            action:  'rankbot_optimize_post',
            post_id: postId,
            nonce:   nonce
        }, function (res) {
            if (res && res.success && res.data && typeof res.data === 'object') {
                if (res.data.status === 'queued' && res.data.job_id) {
                    log.text('Job Queued (' + res.data.job_id + '). Waiting for AI...');
                    pollJob(res.data.job_id);
                } else {
                    handleSuccess(res.data);
                }
            } else {
                handleError(res.data);
            }
        }).fail(function () {
            handleError('Network error');
        });
    });

    // ---------- Helpers ----------

    function setLoading(isLoading) {
        if (isLoading) {
            spinner.addClass('is-active');
            statusText.show().text('Processing...');
            btn.prop('disabled', true);
        } else {
            spinner.removeClass('is-active');
            statusText.hide();
            btn.prop('disabled', false);
        }
    }

    function handleError(msg) {
        setLoading(false);
        var errSpan = $('<span style="color:red"></span>').text('Error: ' + (msg || 'Unknown'));
        log.empty().append(errSpan);
    }

    function handleSuccess() {
        setLoading(false);
        log.html('<span style="color:green; font-weight:bold;">Success! Page reloading...</span>');
        setTimeout(function () { location.reload(); }, 1500);
    }

    var pollAttempts = 0;
    var maxPollAttempts = 120; // ~10 min at 5s intervals

    function pollJob(jobId) {
        pollAttempts++;
        if (pollAttempts > maxPollAttempts) {
            handleError('Job timed out after ' + maxPollAttempts + ' attempts. Check history for results.');
            return;
        }
        setTimeout(function () {
            $.post(ajaxurl, {
                action:  'rankbot_check_job',
                job_id:  jobId,
                post_id: postId,
                nonce:   nonce
            }, function (res) {
                if (res && res.success && res.data && typeof res.data === 'object') {
                    var data   = res.data;
                    var status = data && typeof data === 'object' ? data.status : undefined;

                    if (!status) {
                        // Invalid payload; retry.
                        pollJob(jobId);
                        return;
                    }

                    if (status === 'completed') {
                        var finalData = data.result ? data.result : data;
                        handleSuccess(finalData);
                    } else if (status === 'failed') {
                        handleError(data.error);
                    } else {
                        // Still processing
                        log.text('AI is writing... ' + (data.message || ''));
                        pollJob(jobId);
                    }
                } else {
                    // Could be network glitch, retry
                    pollJob(jobId);
                }
            }).fail(function () {
                // Network fail, retry
                pollJob(jobId);
            });
        }, 5000); // 5 s interval for resume
    }
});
