/**
 * RankBotAI llms.txt Generator
 * Handles editor interactions, preview, character/line counts.
 */
(function () {
    'use strict';

    document.addEventListener('DOMContentLoaded', function () {
        var editor = document.getElementById('rb-llms-editor');
        var charsEl = document.getElementById('rb-llms-chars');
        var linesEl = document.getElementById('rb-llms-lines');
        var previewBtn = document.getElementById('rb-llms-preview');
        var previewModal = document.getElementById('rb-llms-preview-modal');
        var previewClose = document.getElementById('rb-llms-preview-close');
        var previewContent = document.getElementById('rb-llms-preview-content');
        var regenerateBtn = document.getElementById('rb-llms-regenerate');
        var hiddenInput = document.getElementById('rb-llms-content-hidden');
        var saveForm = document.getElementById('rb-llms-form');

        if (!editor) return;

        // Update character/line count
        function updateCounts() {
            var val = editor.value || '';
            if (charsEl) charsEl.textContent = val.length;
            if (linesEl) linesEl.textContent = val.split('\n').length;
        }

        editor.addEventListener('input', updateCounts);
        updateCounts();

        // Sync textarea content into hidden input on form submit
        if (saveForm && hiddenInput) {
            saveForm.addEventListener('submit', function () {
                hiddenInput.value = editor.value;
            });
        }

        // Tab key support in textarea
        editor.addEventListener('keydown', function (e) {
            if (e.key === 'Tab') {
                e.preventDefault();
                var start = this.selectionStart;
                var end = this.selectionEnd;
                this.value = this.value.substring(0, start) + '  ' + this.value.substring(end);
                this.selectionStart = this.selectionEnd = start + 2;
            }
        });

        // Preview
        if (previewBtn && previewModal) {
            previewBtn.addEventListener('click', function () {
                var content = editor.value || '';
                // Simple markdown-like rendering for preview
                var html = escapeHtml(content)
                    .replace(/^# (.+)$/gm, '<h1 style="font-size:22px; font-weight:700; color:#111827; margin:0 0 8px 0; font-family:-apple-system,sans-serif;">$1</h1>')
                    .replace(/^## (.+)$/gm, '<h2 style="font-size:17px; font-weight:600; color:#374151; margin:18px 0 8px 0; font-family:-apple-system,sans-serif;">$1</h2>')
                    .replace(/^> (.+)$/gm, '<blockquote style="border-left:3px solid #6366f1; padding:8px 16px; margin:8px 0; color:#6b7280; background:#f8fafc; border-radius:0 6px 6px 0; font-family:-apple-system,sans-serif;">$1</blockquote>')
                    .replace(/^- \[(.+?)\]\((.+?)\)(?:: (.+))?$/gm, function (m, title, url, desc) {
                        var line = '<div style="padding:4px 0 4px 16px; font-family:-apple-system,sans-serif;">&bull; <a href="' + url + '" style="color:#3b82f6; text-decoration:none; font-weight:500;">' + title + '</a>';
                        if (desc) {
                            line += '<span style="color:#9ca3af;"> — ' + desc + '</span>';
                        }
                        return line + '</div>';
                    })
                    .replace(/\n/g, '<br>');

                if (previewContent) previewContent.innerHTML = html;
                previewModal.style.display = 'flex';
            });
        }

        if (previewClose && previewModal) {
            previewClose.addEventListener('click', function () {
                previewModal.style.display = 'none';
            });
            previewModal.addEventListener('click', function (e) {
                if (e.target === previewModal) {
                    previewModal.style.display = 'none';
                }
            });
        }

        // Regenerate confirmation
        if (regenerateBtn) {
            regenerateBtn.addEventListener('click', function () {
                if (editor.value.trim() && !confirm('This will replace the current content with auto-generated data. Continue?')) {
                    return;
                }
                // Reload page with regenerate flag
                window.location.href = window.location.pathname + '?page=rankbot-llms&regenerate=1';
            });
        }
    });

    function escapeHtml(str) {
        var div = document.createElement('div');
        div.appendChild(document.createTextNode(str));
        return div.innerHTML;
    }
})();
