(function (wp) {
	// Relaxed check to allow partial loading if possible
	if (!wp || !wp.element || !wp.plugins) return;
	if (typeof RankBotGutenberg === 'undefined') return;

	var getComp = function(obj, key) {
		return (obj && typeof obj[key] !== 'undefined') ? obj[key] : undefined;
	};

	var el = wp.element.createElement;
	var useState = wp.element.useState;
	var useEffect = wp.element.useEffect;
	var Fragment = wp.element.Fragment || 'div'; // Fallback

	var registerPlugin = wp.plugins.registerPlugin;
	
	// Safe component retrieval — prefer wp.editor (WP 6.6+) with wp.editPost fallback
	var editorNs = wp.editor || wp.editPost || {};
	var PluginSidebar = getComp(editorNs, 'PluginSidebar') || getComp(wp.editPost, 'PluginSidebar');
	var PluginToolbarButton = getComp(editorNs, 'PluginToolbarButton') || getComp(wp.editPost, 'PluginToolbarButton');
	var PluginSidebarMoreMenuItem = getComp(editorNs, 'PluginSidebarMoreMenuItem') || getComp(wp.editPost, 'PluginSidebarMoreMenuItem');
	var Dashicon = getComp(wp.components, 'Dashicon');
	var Spinner = getComp(wp.components, 'Spinner');
	var Notice = getComp(wp.components, 'Notice');
	var Modal = getComp(wp.components, 'Modal');
	var Button = getComp(wp.components, 'Button');

	if (!PluginSidebar) {
		return;
	}

	var SIDEBAR_NAME = 'rankbot-seo-sidebar';
	var SIDEBAR_KEY = 'plugin-sidebar__' + SIDEBAR_NAME;

	function getPostId() {
		try {
			return wp.data.select('core/editor').getCurrentPostId();
		} catch (e) {
			return RankBotGutenberg.postId || 0;
		}
	}

	function isSidebarOpen() {
		try {
			return wp.data.select('core/edit-post').getActiveGeneralSidebarName() === SIDEBAR_KEY;
		} catch (e) {
			return false;
		}
	}

	function toggleSidebar() {
		try {
			var dsp = wp.data.dispatch('core/edit-post');
			if (isSidebarOpen()) {
				dsp.closeGeneralSidebar();
			} else {
				dsp.openGeneralSidebar(SIDEBAR_KEY);
			}
		} catch (e) {
			// no-op
		}
	}

	function postForm(data) {
		var body = new URLSearchParams();
		Object.keys(data).forEach(function (k) {
			if (typeof data[k] !== 'undefined' && data[k] !== null) body.append(k, String(data[k]));
		});

		return window.fetch(RankBotGutenberg.ajaxUrl, {
			method: 'POST',
			headers: {
				'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'
			},
			credentials: 'same-origin',
			body: body.toString()
		}).then(function (r) {
			return r.text().then(function (t) {
				try {
					return JSON.parse(t);
				} catch (e) {
					return { success: false, data: { message: 'Invalid JSON response', raw: String(t || '').slice(0, 200) } };
				}
			});
		});
	}

	function scoreColor(score) {
		if (score >= 80) return '#008a20';
		if (score >= 50) return '#dba617';
		return '#d63638';
	}

	function ScoreRing(props) {
		var score = props.score || 0;
		var color = scoreColor(score);
		var ringStyle = {
			width: '78px',
			height: '78px',
			borderRadius: '999px',
			position: 'relative',
			display: 'flex',
			alignItems: 'center',
			justifyContent: 'center',
			margin: '6px auto 12px auto',
			boxShadow: '0 1px 0 rgba(0,0,0,0.04), 0 6px 16px rgba(15, 23, 42, 0.08)',
			background: 'conic-gradient(' + color + ' ' + score + '%, #dcdcde 0)'
		};
		var innerStyle = {
			position: 'absolute',
			inset: '7px',
			background: '#ffffff',
			borderRadius: '999px',
			boxShadow: 'inset 0 0 0 1px #dcdcde'
		};
		var textWrap = {
			position: 'relative',
			display: 'flex',
			flexDirection: 'column',
			alignItems: 'center',
			justifyContent: 'center',
			lineHeight: '1'
		};
		var numStyle = {
			fontSize: '18px',
			fontWeight: '900',
			letterSpacing: '-0.3px',
			color: color
		};
		var subStyle = {
			marginTop: '2px',
			fontSize: '10px',
			fontWeight: '700',
			color: '#646970'
		};

		return el('div', { style: ringStyle }, [
			el('div', { style: innerStyle, key: 'inner' }),
			el('div', { style: textWrap, key: 'text' }, [
				el('div', { style: numStyle, key: 'num' }, String(score)),
				el('div', { style: subStyle, key: 'sub' }, '/100')
			])
		]);
	}

	function StatusIcon(status) {
		var icon = 'warning';
		var color = '#d63638';
		if (status === 'good') {
			icon = 'yes-alt';
			color = '#008a20';
		} else if (status === 'ok') {
			icon = 'minus';
			color = '#dba617';
		}
		if (Dashicon) {
			return el(Dashicon, { icon: icon, style: { color: color } });
		}
		// Fallback
		var symbol = (status === 'good') ? '✓' : (status === 'ok' ? '•' : '!');
		return el('span', { style: { color: color, fontWeight: 'bold' } }, symbol);
	}

	function RankBotSidebar() {
		var initialScore = parseInt(RankBotGutenberg.initialScore || '0', 10);
		if (isNaN(initialScore)) initialScore = 0;

		var _a = useState(initialScore), score = _a[0], setScore = _a[1];
		var _b = useState([]), results = _b[0], setResults = _b[1];
		var _c = useState(false), loading = _c[0], setLoading = _c[1];
		var _d = useState(''), error = _d[0], setError = _d[1];
		var _e = useState(''), lastUpdated = _e[0], setLastUpdated = _e[1];
		
		// Tool State
		var _t = useState(false), busy = _t[0], setBusy = _t[1];
		var _s = useState(''), toolStatus = _s[0], setToolStatus = _s[1];

		// Modal (estimate + confirmation)
		var _mo = useState(false), modalOpen = _mo[0], setModalOpen = _mo[1];
		var _mp = useState(null), modalPayload = _mp[0], setModalPayload = _mp[1];
		var _me = useState(null), modalEstimate = _me[0], setModalEstimate = _me[1];
		var _ml = useState(false), modalLoading = _ml[0], setModalLoading = _ml[1];
		var _merr = useState(''), modalError = _merr[0], setModalError = _merr[1];

		function closeModal() {
			setModalOpen(false);
			setModalPayload(null);
			setModalEstimate(null);
			setModalLoading(false);
			setModalError('');
		}

		function openModal(actionKey) {
			// If Modal component isn't available, fall back to immediate action.
			if (!Modal) {
				runTool(actionKey, false, false);
				return;
			}

			setModalPayload({ actionKey: actionKey || '' });
			setModalOpen(true);
			setModalEstimate(null);
			setModalError('');
			setModalLoading(true);

			var postId = getPostId();
			var estimateAction = (actionKey || 'auto');

			postForm({
				action: 'rankbot_estimate_cost',
				nonce: RankBotGutenberg.nonce,
				mode: 'selected',
				ids: postId,
				bulk_action: estimateAction
			}).then(function(res) {
				if (res && res.success && res.data && typeof res.data === 'object') {
					setModalEstimate(res.data);
				} else {
					setModalError((res && res.data && res.data.message) ? String(res.data.message) : 'Failed to estimate cost');
				}
			}).catch(function() {
				setModalError('Network error');
			}).finally(function() {
				setModalLoading(false);
			});
		}

		function runTool(type) {
			
			setBusy(true);
			setToolStatus('Starting...');
			
			var payload = {
				action: 'rankbot_optimize',
				nonce: RankBotGutenberg.nonce,
				post_id: getPostId()
			};
			payload.rankbot_action_type = type;

			postForm(payload).then(function(res) {
				if (res && res.success && res.data && typeof res.data === 'object') {
					if (res.data.status === 'queued' && res.data.job_id) {
						setToolStatus('Queued. AI is processing...');
						pollTool(res.data.job_id);
					} else {
						finishTool();
					}
				} else {
					setToolStatus('Error: ' + ((res.data && res.data.message) || 'Unknown'));
					setBusy(false);
				}
			}).catch(function() {
				setToolStatus('Network Error');
				setBusy(false);
			});
		}

		function pollTool(jobId) {
			setTimeout(function() {
				postForm({
					action: 'rankbot_check_job',
					job_id: jobId,
					post_id: getPostId(),
					nonce: RankBotGutenberg.nonce
				}).then(function(r) {
					if (r && r.success && r.data && typeof r.data === 'object') {
						if (r.data.status === 'completed') {
							finishTool();
						} else if (r.data.status === 'failed') {
							setToolStatus('Failed');
							setBusy(false);
						} else {
							setToolStatus('AI Writing... ' + (r.data.message || ''));
							pollTool(jobId);
						}
					} else {
						pollTool(jobId); // Retry
					}
				}).catch(function() { pollTool(jobId); });
			}, 3000);
		}

		function finishTool() {
			setToolStatus('Done! Reloading...');
			setTimeout(function() { window.location.reload(); }, 1500);
		}

		function refresh() {
			var postId = getPostId();
			if (!postId) {
				setError('No post ID');
				return;
			}
			setLoading(true);
			setError('');
			postForm({
				action: 'rankbot_analyze_seo',
				nonce: RankBotGutenberg.nonce,
				post_id: postId
			})
				.then(function (res) {
					if (!res || !res.success) {
						setError((res && res.data) ? String(res.data) : 'Analysis failed');
						return;
					}
					setScore(parseInt(res.data.score || 0, 10) || 0);
					setResults(Array.isArray(res.data.results) ? res.data.results : []);
					setLastUpdated(new Date().toLocaleString());
				})
				.catch(function () {
					setError('Network error');
				})
				.finally(function () {
					setLoading(false);
				});
		}

		useEffect(function () {
			refresh();
			// eslint-disable-next-line react-hooks/exhaustive-deps
		}, []);

		var headerStyle = {
			padding: '12px 12px 8px 12px',
			borderBottom: '1px solid #e2e8f0'
		};
		var metaStyle = {
			fontSize: '12px',
			color: '#646970',
			marginTop: '6px'
		};
		var btnStyle = {
			display: 'inline-flex',
			alignItems: 'center',
			gap: '6px',
			padding: '6px 10px',
			borderRadius: '999px',
			border: '1px solid #c7d2fe',
			background: '#eef2ff',
			color: '#4338ca',
			fontSize: '12px',
			fontWeight: '700',
			cursor: 'pointer'
		};

		var grouped = {};
		results.forEach(function (r) {
			var g = r.group || 'basic';
			if (!grouped[g]) grouped[g] = [];
			grouped[g].push(r);
		});

		var groupLabels = {
			title: 'Title Tag',
			meta: 'Meta Description',
			url: 'URL Structure',
			headings: 'Headings (H1-H6)',
			content: 'Content Analysis',
			media: 'Images & Alt Text',
			links: 'Links Strategy',
			additional: 'Additional',
			basic: 'General'
		};

		var order = ['title', 'meta', 'url', 'headings', 'content', 'media', 'links', 'additional', 'basic'];

		// --- Tools Section (Buttons) ---
		var toolsSection = el('div', { key: 'tools', style: { padding: '15px', background: '#f8fafc', borderBottom: '1px solid #e2e8f0' } }, [
			el('div', { key: 'th', style: { fontSize: '11px', fontWeight: '800', color: '#64748b', marginBottom: '10px', textTransform: 'uppercase', letterSpacing:'0.5px' } }, 'Quick Actions'),
			
			// Grid for standard buttons
			el('div', { key: 'grid', style: { display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '8px', marginBottom: '12px' } }, [
				el('button', { 
					key: 'res', type: 'button', disabled: busy, onClick: function(){ openModal('research'); },
					style: { padding: '8px', borderRadius: '6px', border: '1px solid #cbd5e1', background: '#fff', cursor: 'pointer', fontSize: '12px', fontWeight: '500', color: '#334155' } 
				}, 'Research KW'),
				el('button', { 
					key: 'snip', type: 'button', disabled: busy, onClick: function(){ openModal('snippet'); },
					style: { padding: '8px', borderRadius: '6px', border: '1px solid #cbd5e1', background: '#fff', cursor: 'pointer', fontSize: '12px', fontWeight: '500', color: '#334155' } 
				}, 'Gen Snippet'),
			]),
			
			// Auto Optimize Button
			el('button', { 
				key: 'opt', type: 'button', disabled: busy, onClick: function(){ openModal('complex'); },
				style: { 
					width: '100%', padding: '10px', borderRadius: '6px', border: 'none', 
					background: 'linear-gradient(135deg, #3b82f6 0%, #2563eb 100%)', color: '#fff', 
					cursor: 'pointer', fontWeight: '600', fontSize: '13px', marginBottom: '8px', display: 'flex', alignItems: 'center', justifyContent: 'center', gap: '6px',
					boxShadow: '0 4px 6px -1px rgba(37, 99, 235, 0.2)'
				} 
			}, [
				Dashicon ? el(Dashicon, { icon: 'performance', key: 'i' }) : null,
				'Auto-Optimize SEO'
			]),

			toolStatus ? el('div', { key: 'stat', style: { marginTop: '10px', padding: '8px', background: '#f1f5f9', borderRadius: '4px', fontSize: '11px', color: '#0f172a', fontWeight: '600', border:'1px solid #e2e8f0' } }, toolStatus) : null
		]);

		var modalNode = null;
		if (Modal && modalOpen) {
			var est = modalEstimate || {};
			var canAfford = (typeof est.can_afford === 'boolean') ? est.can_afford : true;
			var shortfall = (typeof est.shortfall !== 'undefined') ? est.shortfall : 0;
			var buyUrl = '';
			if (est && est.plans && Array.isArray(est.plans) && est.plans.length) {
				// Prefer the highlighted plan (custom shortfall) if present
				for (var i = 0; i < est.plans.length; i++) {
					if (est.plans[i] && est.plans[i].buy_url && est.plans[i].is_highlight) {
						buyUrl = String(est.plans[i].buy_url);
						break;
					}
				}
				if (!buyUrl && est.plans[0] && est.plans[0].buy_url) {
					buyUrl = String(est.plans[0].buy_url);
				}
			}

			modalNode = el(Modal, {
				title: 'RankBot Optimization',
				onRequestClose: closeModal
			}, [
				modalError ? el(Notice, { key: 'merr', status: 'error', isDismissible: false }, String(modalError)) : null,
				modalLoading ? el('div', { key: 'ml', style: { padding: '8px 0' } }, Spinner ? el(Spinner, {}) : 'Loading…') : null,
				(!modalLoading && modalEstimate) ? el('div', { key: 'est', style: { padding: '6px 0 10px 0', fontSize: '13px', color: '#1d2327' } }, [
					el('div', { key: 'act', style: { marginBottom: '6px' } }, 'Action: ' + (est.action_label || 'Optimization')),
					el('div', { key: 'mdl', style: { marginBottom: '6px' } }, 'Model: ' + (est.model_label || '')),
					el('div', { key: 'cst', style: { marginBottom: '6px' } }, 'Estimated cost: ' + String(est.total_cost || 0) + ' tokens'),
					el('div', { key: 'bal', style: { marginBottom: '6px' } }, 'Balance: ' + String(est.balance || 0) + ' tokens'),
					(!canAfford) ? el(Notice, { key: 'ins', status: 'warning', isDismissible: false }, 'Insufficient balance. Missing ' + String(shortfall || 0) + ' tokens.') : null,
					(!canAfford && buyUrl) ? el('a', { key: 'buy', href: buyUrl, target: '_blank', rel: 'noopener noreferrer' }, 'Top up balance') : null
				]) : null,
				el('div', { key: 'btns', style: { display: 'flex', gap: '8px', justifyContent: 'flex-end', paddingTop: '10px' } }, [
					Button ? el(Button, { key: 'cancel', isSecondary: true, onClick: closeModal }, 'Cancel') : el('button', { key: 'cancel2', type: 'button', onClick: closeModal }, 'Cancel'),
					Button ? el(Button, {
						key: 'run',
						isPrimary: true,
						disabled: !!busy || !!modalLoading || (!!modalEstimate && modalEstimate.can_afford === false),
						onClick: function() {
							// Hard block when insufficient balance.
							if (modalEstimate && modalEstimate.can_afford === false) return;
							var payload = modalPayload || { actionKey: '' };
							closeModal();
							runTool(payload.actionKey || null);
						}
					}, 'Run Now') : el('button', { key: 'run2', type: 'button', disabled: !!busy || !!modalLoading, onClick: function(){ var payload = modalPayload || { actionKey: '' }; closeModal(); runTool(payload.actionKey || null); } }, 'Run Now')
				])
			]);
		}

		return el('div', {}, [
			el('div', { style: headerStyle, key: 'hdr' }, [
				el('div', { key: 'title', style: { fontSize: '13px', fontWeight: 800, color: '#1d2327' } }, 'RankBot Analysis'),
				el(ScoreRing, { key: 'ring', score: score }),
				el('div', { key: 'meta', style: metaStyle }, lastUpdated ? ('Updated: ' + lastUpdated) : 'Updating…'),
				el('div', { key: 'actions', style: { display: 'flex', justifyContent: 'center', marginTop: '10px' } }, [
					el('button', { key: 'refresh', type: 'button', onClick: refresh, style: btnStyle, disabled: loading }, [
						Dashicon ? el(Dashicon, { icon: 'update', key: 'ic' }) : null,
						loading ? 'Refreshing…' : 'Refresh'
					])
				])
			]),

			toolsSection, // Inserted Tools Block

			error ? el(Notice, { key: 'err', status: 'error', isDismissible: false }, String(error)) : null,
			loading ? el('div', { key: 'sp', style: { padding: '10px 12px' } }, el(Spinner, {})) : null,

			el('div', { key: 'body', style: { padding: '10px 12px' } }, order.map(function (g) {
				if (!grouped[g] || !grouped[g].length) return null;
				return el('div', { key: 'g-' + g, style: { marginBottom: '12px' } }, [
					el('div', { key: 'gh', style: { textTransform: 'uppercase', fontSize: '11px', fontWeight: 800, letterSpacing: '0.5px', color: '#94a3b8', margin: '10px 0 6px 0' } }, groupLabels[g] || 'Other'),
					grouped[g].map(function (r, idx) {
						var rowStyle = {
							border: '1px solid #e2e8f0',
							borderRadius: '8px',
							background: '#fff',
							padding: '10px 10px',
							marginBottom: '8px'
						};
						return el('details', { key: g + '-' + idx, style: rowStyle }, [
							el('summary', { key: 's', style: { cursor: 'pointer', display: 'flex', alignItems: 'center', justifyContent: 'space-between', gap: '10px', listStyle: 'none' } }, [
								el('div', { key: 'l', style: { display: 'flex', alignItems: 'center', gap: '8px', minWidth: 0 } }, [
									StatusIcon(r.status),
									el('span', { key: 't', style: { fontSize: '13px', fontWeight: 600, color: '#334155', overflow: 'hidden', textOverflow: 'ellipsis', whiteSpace: 'nowrap' } }, r.label || 'Check')
								]),
								el('span', { key: 'pts', style: { fontSize: '11px', padding: '2px 8px', borderRadius: '999px', background: '#f8fafc', border: '1px solid #e2e8f0', color: '#64748b', fontWeight: 600, whiteSpace: 'nowrap' } }, (r.score || 0) + '/' + (r.maxPoints || 0))
							]),
							el('div', { key: 'm', style: { marginTop: '8px', fontSize: '12px', color: '#475569', lineHeight: 1.4 } }, r.message || '')
						]);
					})
				]);
			})),
			modalNode
		]);
	}

	registerPlugin('rankbot-gutenberg-seo-sidebar', {
		icon: 'chart-bar',
		render: function () {
			var children = [];
			
			if (PluginToolbarButton) {
				children.push(el(PluginToolbarButton, {
					key: 'btn',
					icon: 'chart-bar',
					label: 'RankBot Analysis',
					onClick: toggleSidebar,
					isPressed: isSidebarOpen()
				}));
			}

			// Fallback entry in the editor "More" menu (3 dots) when toolbar slot is not available.
			if (PluginSidebarMoreMenuItem) {
				children.push(el(PluginSidebarMoreMenuItem, {
					key: 'menu',
					target: SIDEBAR_NAME
				}, 'RankBot Analysis'));
			}

			children.push(el(PluginSidebar, {
				key: 'sb',
				name: SIDEBAR_NAME,
				title: 'RankBot Analysis',
				icon: 'chart-bar'
			}, el(RankBotSidebar, {})));

			return el(Fragment, {}, children);
		}
	});
})(window.wp);
