/**
 * RankBotAI – Category/Term Optimization
 *
 * Used on term.php / edit-tags.php screens.
 * Localized object: rankbotTermData
 *   - termId    (int)    Term ID being edited
 *   - taxonomy  (string) Taxonomy slug
 *   - termLink  (string) Permalink of the term
 *   - nonce     (string) Security nonce
 *
 * @package RankBotAI
 */

/* global jQuery, ajaxurl, rankbotTermData */

(function ($) {
    'use strict';

    $(document).ready(function () {
        // Move UI under the Name field
        // .term-name-wrap is standard in WP 4.x–6.x
        var target = $('.term-name-wrap td');
        if (target.length) {
            target.append($('#rankbot-category-wrap'));
            target.append($('#rankbot-status-log'));
        } else {
            // Fallback: show the original row
            $('.rankbot-original-row').show();
        }

        $('.rankbot-optimize-term-btn').on('click', function (e) {
            e.preventDefault();

            var btn       = $(this);
            var wrap      = $('#rankbot-category-wrap');
            var spinner   = wrap.find('.spinner');
            var log       = $('#rankbot-status-log');
            var termId    = rankbotTermData.termId;
            var taxonomy  = rankbotTermData.taxonomy;
            var termLink  = rankbotTermData.termLink;
            var nonce     = rankbotTermData.nonce;

            var originalText = btn.html();

            if (confirm('This will update the description, SEO fields, and Slug (if enabled). Continue?')) {
                setLoading(true);
                log.text('Starting optimization...').show();

                $.post(ajaxurl, {
                    action:    'rankbot_optimize_term',
                    term_id:   termId,
                    taxonomy:  taxonomy,
                    term_link: termLink,
                    nonce:     nonce
                }, function (res) {
                    if (res && res.success && res.data && typeof res.data === 'object') {
                        // Async Job Started
                        if (res.data.status === 'queued' && res.data.job_id) {
                            log.text('Job Queued (' + res.data.job_id + '). Waiting for AI...');
                            pollJob(res.data.job_id);
                        } else {
                            // Sync Response
                            handleSuccess(res.data);
                        }
                    } else {
                        setLoading(false);
                        log.hide();
                        alert('Error: ' + (res.data || 'Unknown error'));
                    }
                }).fail(function () {
                    setLoading(false);
                    log.text('Connection failed.').css('color', 'red');
                    alert('Connection error. Please try again.');
                });
            }

            function setLoading(isLoading) {
                if (isLoading) {
                    spinner.addClass('is-active');
                    btn.prop('disabled', true);
                    btn.html('<span class="dashicons dashicons-update" style="color:white; animation:spin 2s infinite linear;"></span> <span style="color:white;">Thinking...</span>');
                } else {
                    spinner.removeClass('is-active');
                    btn.prop('disabled', false);
                    btn.html(originalText);
                }
            }

            var rankbotPollAttempts = 0;
            var rankbotPollErrors   = 0;
            var rankbotMaxPollAttempts = 180; // ~7.5 min at 2.5 s

            function pollJob(jobId) {
                rankbotPollAttempts++;
                if (rankbotPollAttempts > rankbotMaxPollAttempts) {
                    // Don't mark failed locally: job may still complete server-side.
                    setLoading(false);
                    log.text('Still processing on server. Please refresh and check later.').css('color', '#b45309');
                    return;
                }

                setTimeout(function () {
                    $.post(ajaxurl, {
                        action:  'rankbot_check_job',
                        job_id:  jobId,
                        term_id: termId,
                        nonce:   nonce
                    }, function (res) {
                        if (res && res.success && res.data && typeof res.data === 'object') {
                            rankbotPollErrors = 0;
                            var data   = res.data;
                            var status = data && typeof data === 'object' ? data.status : undefined;

                            if (!status) {
                                // Invalid payload; treat as transient and retry with backoff.
                                rankbotPollErrors++;
                                var backoffInvalid = Math.min(20000, 2500 * (1 + rankbotPollErrors));
                                log.text('Temporary response issue. Retrying...').css('color', '#b45309');
                                setTimeout(function () { pollJob(jobId); }, backoffInvalid);
                                return;
                            }

                            if (status === 'completed') {
                                log.text('Job Completed! Reloading...');
                                var resultData = data.result ? data.result : data;
                                handleSuccess(resultData);
                            } else if (status === 'failed') {
                                setLoading(false);
                                log.text('Job Failed: ' + (data.error || 'Unknown'));
                                alert('Job failed: ' + (data.error || 'Unknown error'));
                            } else {
                                // Still processing
                                log.text('AI is thinking...');
                                pollJob(jobId);
                            }
                        } else {
                            // Transient error in check request; keep trying.
                            rankbotPollErrors++;
                            var backoff = Math.min(20000, 2500 * (1 + rankbotPollErrors));
                            log.text('Temporary connection issue. Retrying...').css('color', '#b45309');
                            setTimeout(function () { pollJob(jobId); }, backoff);
                        }
                    }).fail(function () {
                        // Network hiccup; keep trying with backoff.
                        rankbotPollErrors++;
                        var backoff = Math.min(20000, 2500 * (1 + rankbotPollErrors));
                        log.text('Network issue. Retrying...').css('color', '#b45309');
                        setTimeout(function () { pollJob(jobId); }, backoff);
                    });
                }, 2500); // 2.5 s poll
            }

            function handleSuccess() {
                setLoading(false);
                log.text('Success! Reloading page...');

                // Force reload to refresh WP admin UI
                window.location.reload(true);
            }
        });
    });
})(jQuery);
