(function($) {
    'use strict';

    const RbBulk = {
        isProcessingLocked: false,
        rowStatusXhr: null,
        lastBulkSnapshot: { total: 0, completed: 0 },

        init: function() {
            this.cacheDOM();
            this.bindEvents();
            this.initUI();
            this.initSlider();
            this.initPerPageSync();
            this.startPoller();
        },

        startPoller: function() {
            // Initial poll
            this.pollOverall();
            // Also poll for sidebar menu update (global)
            this.pollSideMenu();
        },

        pollSideMenu: function() {
            // Simple separate poller for menu bubble updates (can be longer interval)
            const self = this;
            if (window.rankbotMenuTimer) clearTimeout(window.rankbotMenuTimer);

            if (!window.rankbotBulkData || !window.rankbotBulkData.ajaxurl) return;

            $.post(window.rankbotBulkData.ajaxurl, {
                action: 'rankbot_bulk_count_active',
                nonce: window.rankbotBulkData.nonce
            })
            .done(function(res) {
                if (res.success) {
                    const count = parseInt(res.data.count || 0, 10);
                    const balance = res.data.balance || null;
                    self.updateMenuBubble(count);
                    self.updateTopBar(count, balance);
                }
            })
            .always(function() {
                // Poll every 10 sec for menu updates (even if request fails)
                window.rankbotMenuTimer = setTimeout(() => self.pollSideMenu(), 10000);
            });
        },

        updateTopBar: function(count, balanceStr) {
            // Update #wp-admin-bar-rankbot-balance >> .ab-item span (Working...)
            const $topNode = $('#wp-admin-bar-rankbot-balance');
            if (!$topNode.length) return;
            
            const $link = $topNode.children('a.ab-item');
            if (!$link.length) return;

            // Update Balance Text if provided
            if (balanceStr) {
                // Find the text node " XXX Tokens"
                // It sits usually between the dashicon span and the badge span (if any)
                $link.contents().filter(function() {
                    return this.nodeType === 3; // Text Node
                }).each(function() {
                    // Check if it looks like the balance text
                    if (this.nodeValue.indexOf('Tokens') !== -1) {
                        this.nodeValue = ' ' + balanceStr + ' Tokens ';
                    }
                });
            }
            
            // Try to find existing badge for "Working..."
            let $badge = $link.find('span#rb-top-working-badge');
            // If ID not found (older render), try by content/style
            if (!$badge.length) {
                $badge = $link.find('span').filter(function() {
                    return $(this).text().indexOf('Working...') !== -1 || $(this).css('background-color') === 'rgb(239, 68, 68)' || $(this).css('background-color') === '#ef4444';
                });
            }

            if (count > 0) {
                const label = count + ' Working...';
                if ($badge.length) {
                    $badge.text(label);
                    $badge.show();
                    // Ensure ID is set for future easier lookup
                    if (!$badge.attr('id')) $badge.attr('id', 'rb-top-working-badge');
                } else {
                    // Create it
                   const html = ' <span id="rb-top-working-badge" style="background: #ef4444; color: white; border-radius: 99px; padding: 1px 6px; font-size: 10px; margin-left: 5px; vertical-align: middle;">' + label + '</span>';
                   $link.append(html);
                }
            } else {
                if ($badge.length) $badge.hide();
            }
        },

        updateMenuBubble: function(count) {
            // Find the menu item
            const $menuItem = $('#toplevel_page_rankbot-ai');
            if (!$menuItem.length) return;

            // Server renders a custom badge `.rb-menu-badge`.
            const $name = $menuItem.find('.wp-menu-name');
            let $rbBadge = $menuItem.find('.rb-menu-badge');
            let $wpBubble = $menuItem.find('.update-plugins');

            if (count > 0) {
                if ($rbBadge.length) {
                    $rbBadge.text(count);
                } else if ($name.length) {
                    const html = ' <span class="rb-menu-badge" style="' +
                        'background-color:#059669;color:#ffffff;display:inline-block;padding:2px 6px;' +
                        'border-radius:99px;font-size:10px;font-weight:600;line-height:normal;vertical-align:middle;margin-left:2px;' +
                    '">' + count + '</span>';
                    $name.append(html);
                }
                if ($wpBubble.length) $wpBubble.remove();
            } else {
                if ($rbBadge.length) $rbBadge.remove();
                if ($wpBubble.length) $wpBubble.remove();
            }
        },

        cacheDOM: function() {
            this.$spinner = $('#rb-bulk-spinner');
            this.$statusBox = $('#rb-bulk-status');
            this.$selectAll = $('#rb-select-all');
            this.$selectedBar = $('#rb-selected-bar');
            this.$selectedCount = $('#rb-selected-count');
            
            this.$progressContainer = $('#rb-bulk-progress'); // Added container
            this.$progressBar = $('#rb-bulk-progress-bar');
            this.$progressText = $('#rb-bulk-progress-text');
            this.$stopBtn = $('#rb-bulk-stop');
            this.$items = $('input.rb-item');
            
            // Slider elements
            this.$scoreMin = $('#rb-score-min');
            this.$scoreMax = $('#rb-score-max');
            this.$scoreTrack = $('#rb-slider-highlight');
            this.$scoreDisplay = $('#rb-score-display');

            // Per-page inputs (top is the one submitted)
            this.$perPageTop = $('#rb-per-page-top');
            this.$perPageBottom = $('#rb-per-page-bottom');
        },

        initPerPageSync: function() {
            const self = this;
            if (!this.$perPageTop || !this.$perPageTop.length) return;

            function normalize(val) {
                const n = parseInt(val, 10);
                if (isNaN(n)) return 20;
                return Math.max(1, Math.min(999, n));
            }

            function setBoth(val) {
                const v = normalize(val);
                self.$perPageTop.val(v);
                if (self.$perPageBottom && self.$perPageBottom.length) {
                    self.$perPageBottom.val(v);
                }
            }

            // Keep in sync both ways
            this.$perPageTop.on('input change', function() {
                setBoth($(this).val());
            });
            if (this.$perPageBottom && this.$perPageBottom.length) {
                this.$perPageBottom.on('input change', function() {
                    setBoth($(this).val());
                });
            }

            // Submit on Enter (either input)
            const submitOnEnter = function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    const $form = $(this).closest('form');
                    if ($form.length) $form.trigger('submit');
                }
            };
            this.$perPageTop.on('keydown', submitOnEnter);
            if (this.$perPageBottom && this.$perPageBottom.length) {
                this.$perPageBottom.on('keydown', submitOnEnter);
            }

            // Ensure both start identical
            setBoth(this.$perPageTop.val());
        },

        initSlider: function() {
            if (!this.$scoreMin.length) return;
            
            const self = this;
            
            function updateSlider() {
                let min = parseInt(self.$scoreMin.val());
                let max = parseInt(self.$scoreMax.val());

                // Swift logic if they cross
                if (min > max) {
                    let tmp = min;
                    min = max;
                    max = tmp;
                }

                // Visual update
                const percentMin = (min / 100) * 100;
                const percentMax = (max / 100) * 100;

                self.$scoreTrack.css({
                    'left': percentMin + '%',
                    'width': (percentMax - percentMin) + '%'
                });

                self.$scoreDisplay.text(`(${min} - ${max})`);
            }

            this.$scoreMin.on('input', updateSlider);
            this.$scoreMax.on('input', updateSlider);
            updateSlider(); // Init
        },

        bindEvents: function() {
            const self = this;

            // Select All
            if (this.$selectAll.length) {
                this.$selectAll.on('change', function() {
                    const checked = $(this).is(':checked');
                    $('input.rb-item').prop('checked', checked);
                    self.refreshSelectedBar();
                });
            }

            // Individual Items
            $(document).on('change', 'input.rb-item', function() {
                self.refreshSelectedBar();
                // Update select all checkbox state
                if (!$(this).is(':checked')) {
                    self.$selectAll.prop('checked', false);
                }
            });

            // Bulk Action Buttons (Global)
            $('[data-bulk-action]').on('click', function(e) {
                e.preventDefault();
                const mode = $(this).data('bulk-mode');
                const action = $(this).data('bulk-action');
                
                if (mode === 'selected') {
                    self.handleSelectedAction(action);
                } else {
                    self.handleAllAction(action);
                }
            });

            // Selected Bar Actions
            $('[data-selected-action]').on('click', function(e) {
                e.preventDefault();
                const action = $(this).data('selected-action');
                self.handleSelectedAction(action);
            });

            // Single Item Actions
            $('.rb-run-one').on('click', function(e) {
                e.preventDefault();
                const id = $(this).data('id');
                const action = $(this).data('action');
                
                if (confirm(window.rankbotBulkData.i18n.confirmOne)) {
                    self.startTask('selected', [id], action);
                }
            });

            // Stop Button
            if (this.$stopBtn.length) {
                this.$stopBtn.on('click', function(e) {
                    e.preventDefault();
                    if (confirm(window.rankbotBulkData.i18n.confirmStop)) {
                        self.stopAll();
                    }
                });
            }

            // Clear Process Button
            $('#rb-btn-clear-process').on('click', function(e) {
                e.preventDefault();
                const msg = (window.rankbotBulkData && window.rankbotBulkData.i18n && window.rankbotBulkData.i18n.confirmClearQueue)
                    ? window.rankbotBulkData.i18n.confirmClearQueue
                    : 'Are you sure you want to force clear the processing queue? This will cancel pending jobs and reset the UI.';
                if (confirm(msg)) {
                    self.clearProcess();
                }
            });

            // Clear All Jobs Button
            $('#rb-btn-clear-all-jobs').on('click', function(e) {
                e.preventDefault();
                const msg = (window.rankbotBulkData && window.rankbotBulkData.i18n && window.rankbotBulkData.i18n.confirmClearAll)
                    ? window.rankbotBulkData.i18n.confirmClearAll
                    : 'This will delete ALL RankBot job history for this API key on this WordPress site (including completed/failed). Continue?';
                if (confirm(msg)) {
                    self.clearAllJobs();
                }
            });
        },

        initUI: function() {
            this.refreshSelectedBar();
            
            // Handle Sticky Bar Positioning relative to content
            const self = this;
            $(window).on('resize', function() {
                if (self.$selectedBar.hasClass('is-visible')) {
                    self.positionStickyBar();
                }
            });
        },

        positionStickyBar: function() {
            const $wrapper = $('.rb-bulk-wrap');
            if (!$wrapper.length || !this.$selectedBar.length) return;

            const wrapRect = $wrapper[0].getBoundingClientRect();
            // Calculate center of wrapper relative to viewport
            const centerX = wrapRect.left + (wrapRect.width / 2);
            
            this.$selectedBar.css('left', centerX + 'px');
        },

        refreshSelectedBar: function() {
            const ids = this.getSelectedIds();
            const count = ids.length;
            
            this.$selectedCount.text(count + ' ' + window.rankbotBulkData.i18n.selected);
            
            if (count > 0) {
                // Remove display:none if set, then add class for animation
                this.$selectedBar.css('display', 'flex'); 
                
                // Position before showing
                this.positionStickyBar();

                // Small timeout to allow display change to register before transform transition
                const self = this;
                requestAnimationFrame(function() {
                    self.$selectedBar.addClass('is-visible');
                });
            } else {
                this.$selectedBar.removeClass('is-visible');
                // Wait for transition to finish before hiding
                setTimeout(() => {
                    if (!this.$selectedBar.hasClass('is-visible')) {
                         this.$selectedBar.css('display', 'none');
                    }
                }, 300);
            }
        },

        getSelectedIds: function() {
            const ids = [];
            $('input.rb-item:checked').each(function() {
                ids.push($(this).val());
            });
            return ids;
        },

        setBusy: function(isBusy) {
            if (isBusy) {
                this.$spinner.addClass('is-active');
            } else {
                this.$spinner.removeClass('is-active');
            }
            
            $('button').prop('disabled', isBusy);
            // Re-enable stop button if we are busy just because tasks are running
            if (isBusy && this.$stopBtn.is(':visible')) {
                this.$stopBtn.prop('disabled', false);
            }
        },

        setProcessingLock: function(isLocked) {
            // Lock/unlock UI controls while background jobs are active.
            // This is separate from setBusy() because setBusy() is for AJAX-in-flight.
            const locked = !!isLocked;

            this.isProcessingLocked = locked;

            // Only lock bulk-level controls. Per-row controls are handled by per-row status.
            $('[data-bulk-action], [data-selected-action]').prop('disabled', locked).attr('aria-disabled', locked ? 'true' : 'false');
            $('#rb-select-all').prop('disabled', locked);

            // Optional: dim sticky bar actions while locked
            if (locked) {
                $('#rb-selected-bar').addClass('rb-locked');
            } else {
                $('#rb-selected-bar').removeClass('rb-locked');
            }
        },

        pollVisibleItemsStatus: function() {
            const self = this;
            if (self.rowStatusXhr) return;
            if (!window.rankbotBulkData || !window.rankbotBulkData.ajaxurl) return;

            const ids = [];
            $('tr[data-object-id]').each(function() {
                const id = parseInt($(this).data('object-id'), 10);
                if (!isNaN(id) && id > 0) ids.push(id);
            });
            if (!ids.length) return;

            self.rowStatusXhr = $.post(window.rankbotBulkData.ajaxurl, {
                action: 'rankbot_bulk_items_status',
                nonce: window.rankbotBulkData.nonce,
                ids: ids
            })
            .done(function(res) {
                if (!res || !res.success) return;
                const items = (res.data && res.data.items) ? res.data.items : {};
                self.applyRowJobStatuses(items);
            })
            .always(function() {
                self.rowStatusXhr = null;
            });
        },

        applyRowJobStatuses: function(items) {
            const self = this;
            const activeStatuses = { queued: true, pending: true, processing: true, dispatched: true, retry_wait: true };
            const labelMap = {
                queued: 'Queued',
                pending: 'Queued',
                processing: 'Processing',
                retry_wait: 'Retrying',
                dispatched: 'Waiting'
            };

            $('tr[data-object-id]').each(function() {
                const $row = $(this);
                const id = String($row.data('object-id'));
                const info = items && Object.prototype.hasOwnProperty.call(items, id) ? items[id] : null;
                const st = info && info.status ? String(info.status).toLowerCase() : '';
                const isActive = !!activeStatuses[st];

                const $pill = $row.find('.rb-job-status-pill');
                const $busy = $row.find('.rb-item-busy');
                const $checkbox = $row.find('input.rb-item');
                const $processingLabel = $row.find('.rb-row-processing-label');
                const $buttonsWrap = $row.find('.rb-row-actions-buttons');
                const $buttons = $row.find('.rb-run-one');

                if (isActive) {
                    $row.addClass('rb-row-active-job').attr('data-job-status', st);
                    if ($pill.length) {
                        const label = labelMap[st] || (st.charAt(0).toUpperCase() + st.slice(1));
                        $pill.text(label).show();
                    }
                    if ($busy.length) $busy.show();
                    if ($checkbox.length) $checkbox.prop('disabled', true);

                    if ($processingLabel.length) {
                        let lbl = 'Processing...';
                        if (st === 'queued' || st === 'pending') {
                            lbl = 'Queued...';
                        } else if (st === 'retry_wait') {
                            lbl = 'Retrying...';
                        } else if (st === 'dispatched') {
                            lbl = 'Waiting...';
                        }
                        $processingLabel.text(lbl).show();
                    }
                    if ($buttonsWrap.length) $buttonsWrap.hide();
                    if ($buttons.length) $buttons.prop('disabled', true).attr('aria-disabled', 'true');
                } else {
                    $row.removeClass('rb-row-active-job').attr('data-job-status', '');
                    if ($pill.length) $pill.hide().text('');
                    if ($busy.length) $busy.hide();
                    if ($checkbox.length) $checkbox.prop('disabled', false);

                    if ($processingLabel.length) $processingLabel.hide();
                    if ($buttonsWrap.length) $buttonsWrap.css('display', 'flex');
                    if ($buttons.length) {
                        $buttons.prop('disabled', false).attr('aria-disabled', 'false');
                    }
                }
            });
        },

        showStatus: function(html, type) {
            this.$statusBox.show()
                .removeClass('notice-info notice-error notice-success notice-warning')
                .addClass('notice-' + (type || 'info'))
                .html(html);
        },

        handleSelectedAction: function(action) {
            const ids = this.getSelectedIds();
            if (!ids.length) {
                this.showStatus(window.rankbotBulkData.i18n.noSelection, 'warning');
                return;
            }
            // Removed confirm()
            this.startTask('selected', ids, action);
        },

        handleAllAction: function(action) {
            // Removed confirm()
            this.startTask('all', [], action);
        },

        startTask: function(mode, ids, bulkAction) {
            const self = this;
            
            // Show Modal Immediately in 'loading' state
            const pseudoData = {
                count: (mode === 'selected' ? ids.length : 'All Filtered Items'),
                action_label: 'Calculating...', 
                loading: true
            };
            self.showCostModal(pseudoData, null); // No confirm callback yet

            // Estimate Cost
            self.setBusy(true);
            self.showStatus('Calculating cost...', 'info');

            const data = {
                action: 'rankbot_estimate_cost',
                nonce: window.rankbotBulkData.nonce,
                mode: mode,
                bulk_action: bulkAction || 'auto',
                // Pass same filters as before
                post_type: window.rankbotBulkData.filters.post_type,
                post_status: window.rankbotBulkData.filters.post_status,
                score_min: window.rankbotBulkData.filters.score_min,
                score_max: window.rankbotBulkData.filters.score_max,
                search: window.rankbotBulkData.filters.search,
                no_keyword: window.rankbotBulkData.filters.no_keyword,
                only_unprocessed: window.rankbotBulkData.filters.only_unprocessed,
                ids: ids
            };
            
            $.post(window.rankbotBulkData.ajaxurl, data)
                .done(function(res) {
                    self.setBusy(false);
                    if (res.success) {
                        // Update Modal with real data
                        self.showCostModal(res.data, function() {
                            // On Confirm
                            self.executeTask(mode, ids, bulkAction);
                        });
                    } else {
                        $('#rb-cost-modal').remove(); // Close loading modal
                        alert(res.data || 'Failed to estimate cost.');
                    }
                })
                .fail(function() {
                     self.setBusy(false);
                     $('#rb-cost-modal').remove(); // Close loading modal
                     alert('Network error calculating cost.');
                });
        },

        showCostModal: function(data, onConfirm) {
            // Remove existing modal if any
            $('#rb-cost-modal').remove();

            const isLoading = data.loading || false;
            const count = data.count || 0;
            const costNum = (typeof data.total_cost !== 'undefined') ? Number(data.total_cost) : NaN;
            const balanceNum = (typeof data.balance !== 'undefined') ? Number(data.balance) : NaN;
            const cost = Number.isFinite(costNum) ? new Intl.NumberFormat().format(costNum) : '--';
            const balance = Number.isFinite(balanceNum) ? new Intl.NumberFormat().format(balanceNum) : '--';
            const remaining = (Number.isFinite(balanceNum) && Number.isFinite(costNum))
                ? new Intl.NumberFormat().format(balanceNum - costNum)
                : '--';
            const canAfford = data.can_afford || false;
            const shortfallNum = (typeof data.shortfall !== 'undefined') ? Number(data.shortfall) : 0;
            const shortfall = Number.isFinite(shortfallNum) ? new Intl.NumberFormat().format(shortfallNum) : 0;
            const plans = data.plans || [];
            const actionLabel = data.action_label || 'Optimization';
            const modelLabel = data.model_label || data.model_id || '';
            const actionCounts = data.action_counts || null;

            let breakdownHtml = '';
            if (actionCounts && typeof actionCounts === 'object') {
                const keys = Object.keys(actionCounts);
                if (keys.length > 1) {
                    const parts = keys.map(k => `${k}: ${actionCounts[k]}`);
                    breakdownHtml = `<div style="font-size:12px; color:#6b7280; margin-top:6px;">Breakdown: ${parts.join(' · ')}</div>`;
                }
            }

            let contentHtml = '';

            if (isLoading) {
                contentHtml = `
                    <div style="padding:40px; text-align:center;">
                         <div class="rb-spinner" style="border: 4px solid #f3f3f3; border-top: 4px solid #3b82f6; border-radius: 50%; width: 40px; height: 40px; animation: spin 1s linear infinite; margin:0 auto 20px;"></div>
                         <div style="font-size:16px; color:#374151; font-weight:500;">Calculating cost...</div>
                         <div style="font-size:13px; color:#6b7280; margin-top:8px;">Please wait while we analyze selected items.</div>
                         <style>@keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }</style>
                    </div>
                `;
            } else {
                contentHtml = `
                    <div style="padding:24px; border-bottom:1px solid #e5e7eb;">
                        <h3 style="margin:0; font-size:18px; color:#111827;">${canAfford ? 'Confirm Bulk Processing' : 'Insufficient Balance'}</h3>
                         <div style="font-size:13px; color:#6b7280; margin-top:4px;">Task: <strong style="color:#374151;">${actionLabel}</strong></div>
                         ${modelLabel ? `<div style="font-size:13px; color:#6b7280; margin-top:4px;">Model: <strong style="color:#374151;">${modelLabel}</strong></div>` : ''}
                         ${breakdownHtml}
                    </div>
                    <div style="padding:24px;">
                        <div style="display:grid; grid-template-columns:1fr 1fr; gap:16px; margin-bottom:24px;">
                            <div style="background:#f3f4f6; padding:16px; border-radius:8px;">
                                <div style="font-size:12px; color:#6b7280; text-transform:uppercase; letter-spacing:0.05em; margin-bottom:4px;">Items</div>
                                <div style="font-size:24px; font-weight:700; color:#111827;">${count}</div>
                            </div>
                             <div style="background:#f3f4f6; padding:16px; border-radius:8px;">
                                <div style="font-size:12px; color:#6b7280; text-transform:uppercase; letter-spacing:0.05em; margin-bottom:4px;">Est. Cost</div>
                                <div style="font-size:24px; font-weight:700; color:${canAfford ? '#4f46e5' : '#ef4444'};">${cost} <span style="font-size:14px; color:#6b7280;">tokens</span></div>
                            </div>
                        </div>

                         <div style="display:flex; justify-content:space-between; align-items:center; padding-bottom:16px; border-bottom:1px solid #e5e7eb; margin-bottom:16px;">
                            <span style="color:#6b7280;">Current Balance</span>
                                     <span style="font-weight:600; font-size:16px;">${balance} tokens</span>
                        </div>

                        ${!canAfford ? `
                             <div style="background:#fef2f2; border-left:4px solid #ef4444; padding:12px; margin-bottom:20px;">
                                <p style="margin:0; color:#b91c1c; font-size:14px;">
                                    <strong>Missing ${shortfall} tokens.</strong> Please top up your balance to continue.
                                </p>
                             </div>
                             
                             <div style="margin-bottom:10px; font-weight:600; font-size:14px; display:flex; justify-content:space-between; align-items:center;">
                                <span>Quick Top-Up (Tokens Only)</span>
                                <span style="font-size:12px; color:#6b7280; font-weight:400;">Secure payment</span>
                             </div>
                             <!-- Packages Grid: One Row -->
                             <div style="display:flex; flex-wrap:wrap; gap:12px; padding:2px;">
                                ${plans
                                    .filter(p => !p.type || p.type === 'packet') // Packets + Custom
                                    .filter(p => p.is_custom || Number(p.price) > 0) // Filter out Free, keep custom
                                    .map(p => {
                                    const isHighlight = p.is_highlight || false;
                                    const isCustom = p.is_custom || false;
                                    const priceDisplay = isCustom ? p.price : '$' + p.price;
                                    const tokenVal = parseInt(String(p.tokens).replace(/,/g, ''));
                                    const estItems = isNaN(tokenVal) ? 0 : Math.floor(tokenVal / 3);
                                    
                                    // Flex Basis for responsiveness: min-width 180px, but grow to fill
                                    return `
                                    <a href="${p.buy_url}" target="_blank" style="flex:1 0 180px; display:flex; flex-direction:column; justify-content:space-between; padding:16px; border:1px solid ${isHighlight ? '#4f46e5' : '#e5e7eb'}; background:${isHighlight ? '#eef2ff' : '#ffffff'}; border-radius:8px; text-decoration:none; text-align:center; transition:all 0.2s; box-shadow:0 1px 2px rgba(0,0,0,0.05);" onmouseover="this.style.borderColor='#4f46e5'; this.style.transform='translateY(-1px)';" onmouseout="this.style.borderColor='${isHighlight ? '#4f46e5' : '#e5e7eb'}'; this.style.transform='translateY(0)';">
                                        <div style="font-weight:700; color:#111827; font-size:15px; margin-bottom:4px;">${p.name}</div>
                                        <div style="font-size:13px; color:#6b7280; margin-bottom:4px;">${p.tokens_formatted} tokens</div>
                                        ${estItems > 0 ? `<div style="font-size:11px; color:#10b981; margin-bottom:8px; font-weight:500;">~${estItems.toLocaleString()} items</div>` : ''}
                                        <div style="margin-top:auto; font-weight:600; color:#ffffff; background:${isHighlight ? '#4338ca' : '#4f46e5'}; padding:6px 12px; border-radius:6px; font-size:13px;">
                                            ${isCustom ? 'Pay Now &rarr;' : 'Buy for ' + priceDisplay}
                                        </div>
                                    </a>
                                    `;
                                }).join('')}
                                
                                <a href="admin.php?page=rankbot-ai" target="_blank" style="flex:1 0 100%; text-align:center; padding:12px; border:1px dashed #d1d5db; border-radius:8px; color:#6b7280; text-decoration:none; font-size:13px; transition:all 0.2s;" onmouseover="this.style.borderColor='#9ca3af'; this.style.color='#374151';" onmouseout="this.style.borderColor='#d1d5db'; this.style.color='#6b7280';">
                                    View Monthly Subscriptions & All Plans &rarr;
                                </a>
                             </div>
                        ` : `
                            <p style="margin:0; color:#6b7280; font-size:14px;">
                                Balance after processing: <strong>${remaining} tokens</strong>
                            </p>
                        `}
                    </div>
                    <div style="padding:24px; background:#f9fafb; display:flex; justify-content:flex-end; gap:12px;">
                        <button type="button" id="rb-modal-cancel" class="rb-btn rb-btn-secondary">Cancel</button>
                        ${canAfford ? 
                        `<button type="button" id="rb-modal-confirm" class="rb-btn rb-btn-primary">Start Processing</button>` : 
                        `<button type="button" disabled class="rb-btn rb-btn-primary" style="opacity:0.5; cursor:not-allowed;">Insufficient Balance</button>`
                        }
                    </div>
                `;
            }

            let html = `
            <div id="rb-cost-modal" style="position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.5); z-index:100000; display:flex; justify-content:center; align-items:center;">
                <div style="background:white; width: 90%; max-width: 900px; border-radius:12px; box-shadow:0 20px 25px -5px rgba(0,0,0,0.1); overflow:hidden; font-family:-apple-system, sans-serif;">
                    ${contentHtml}
                </div>
            </div>
            `;

            $('body').append(html);

            $('#rb-modal-cancel').on('click', function() {
                $('#rb-cost-modal').remove();
            });

            if (onConfirm) {
                $('#rb-modal-confirm').on('click', function() {
                    $('#rb-cost-modal').remove();
                    onConfirm();
                });
            }
        },

        executeTask: function(mode, ids, bulkAction) {
            const self = this;
            this.setBusy(true);

            const data = {
                action: 'rankbot_bulk_start',
                nonce: window.rankbotBulkData.nonce,
                mode: mode,
                bulk_action: bulkAction || 'auto',
                post_type: window.rankbotBulkData.filters.post_type,
                post_status: window.rankbotBulkData.filters.post_status,
                score_min: window.rankbotBulkData.filters.score_min,
                score_max: window.rankbotBulkData.filters.score_max,
                search: window.rankbotBulkData.filters.search,
                no_keyword: window.rankbotBulkData.filters.no_keyword,
                only_unprocessed: window.rankbotBulkData.filters.only_unprocessed,
                ids: ids
            };

            $.post(window.rankbotBulkData.ajaxurl, data)
                .done(function(res) {
                    if (res.success) {
                        const taskId = res.data.task_id || '';
                        self.showStatus(window.rankbotBulkData.i18n.taskStarted.replace('%s', taskId), 'info');
                        self.pollOverall();
                    } else {
                        self.showStatus(window.rankbotBulkData.i18n.error + ': ' + (res.data || 'Unknown'), 'error');
                        self.setBusy(false);
                    }
                })
                .fail(function(xhr) {
                    self.showStatus(window.rankbotBulkData.i18n.error + ': ' + xhr.statusText, 'error');
                    self.setBusy(false);
                });
        },

        stopAll: function() {
            const self = this;
            this.setBusy(true);

            $.post(window.rankbotBulkData.ajaxurl, {
                action: 'rankbot_bulk_stop_all',
                nonce: window.rankbotBulkData.nonce
            })
            .done(function(res) {
                 if (res.success) {
                    const msg = (res.data && res.data.message) ? res.data.message : window.rankbotBulkData.i18n.tasksStopped;
                    self.showStatus(msg, 'success');
                    self.pollOverall();
                 } else {
                    self.showStatus(window.rankbotBulkData.i18n.error + ': ' + (res.data || 'Unknown'), 'error');
                    self.setBusy(false);
                 }
            })
            .fail(function() {
                self.showStatus(window.rankbotBulkData.i18n.networkError, 'error');
                self.setBusy(false);
            });
        },

        clearProcess: function() {
            const self = this;
            this.setBusy(true);

            $.post(window.rankbotBulkData.ajaxurl, {
                action: 'rankbot_clear_process',
                nonce: window.rankbotBulkData.nonce
            })
            .done(function(res) {
                 if (res.success) {
                    const msg = (res.data && res.data.message) ? res.data.message : 'Queue cleared completely. Page will reload...';
                    self.showStatus(msg, 'success');
                    // Reload to reflect cleared state (re-enable buttons)
                    setTimeout(() => window.location.reload(), 1500);
                 } else {
                    self.showStatus(window.rankbotBulkData.i18n.error + ': ' + (res.data || 'Unknown'), 'error');
                    self.setBusy(false);
                 }
            })
            .fail(function() {
                self.showStatus(window.rankbotBulkData.i18n.networkError, 'error');
                self.setBusy(false);
            });
        },

        clearAllJobs: function() {
            const self = this;
            this.setBusy(true);

            $.post(window.rankbotBulkData.ajaxurl, {
                action: 'rankbot_clear_all_jobs',
                nonce: window.rankbotBulkData.nonce
            })
            .done(function(res) {
                if (res.success) {
                    const msg = (res.data && res.data.message) ? res.data.message : 'All jobs cleared. Page will reload...';
                    self.showStatus(msg, 'success');
                    setTimeout(() => window.location.reload(), 1500);
                } else {
                    self.showStatus(window.rankbotBulkData.i18n.error + ': ' + (res.data || 'Unknown'), 'error');
                    self.setBusy(false);
                }
            })
            .fail(function() {
                self.showStatus(window.rankbotBulkData.i18n.networkError, 'error');
                self.setBusy(false);
            });
        },

        pollOverall: function() {
            const self = this;
            if (this.pollTimer) clearTimeout(this.pollTimer);

            $.post(window.rankbotBulkData.ajaxurl, {
                action: 'rankbot_bulk_overall_status',
                nonce: window.rankbotBulkData.nonce
            })
            .done(function(res) {
                if (!res.success) {
                    self.showStatus('Status poll error: ' + (res.data || ''), 'warning');
                    self.setBusy(false);
                    return;
                }

                const s = res.data.summary || {};
                const total = Number(s.total || 0);
                const completed = Number(s.completed || 0);
                const queued = Number(s.queued || 0);
                const retryWait = Number(s.retry_wait || 0);
                const dispatched = Number(s.dispatched || 0);
                const failed = Number(s.cancelled || 0);
                const activeJobs = Number(s.active_jobs || 0);
                const pauseReason = String(s.pause_reason || '');
                const pauseUntil = Number(s.pause_until || 0);
                const nowTs = Math.floor(Date.now() / 1000);
                const isPaused = pauseReason !== '' && pauseUntil > nowTs;
                const remaining = Number(
                    (typeof s.remaining !== 'undefined')
                        ? s.remaining
                        : Math.max(0, total - completed)
                );
                const displayCount = Math.max(remaining, activeJobs);

                // "Bulk exists" can stay true even when everything is already completed (total>0).
                // For UX we consider bulk "running" only while there is remaining work.
                const hasBulkWork = (remaining > 0 || queued > 0 || retryWait > 0 || dispatched > 0);

                // Keep menu + top bar in sync even if the side poller fails.
                self.updateMenuBubble(displayCount);
                self.updateTopBar(displayCount, null);

                // Update Progress Bar based on completed items.
                const pct = (total > 0) ? Math.min(100, Math.round((completed / total) * 100)) : 0;
                self.$progressBar.css('width', pct + '%');

                // Remember last meaningful totals so we can still detect "finished" even if server drops stats to zeros.
                if (total > 0) {
                    self.lastBulkSnapshot.total = total;
                    self.lastBulkSnapshot.completed = completed;
                }
                
                // Keep tracking while bulk has remaining work or any jobs are in-flight.
                const isStillRunning = hasBulkWork || activeJobs > 0;

                if (isStillRunning) {
                     self.setProcessingLock(hasBulkWork);

                     // Live-update per-row statuses on the current page.
                     self.pollVisibleItemsStatus();

                    const totalForStatus = total > 0 ? total : displayCount;
                    const remainingForStatus = total > 0 ? remaining : displayCount;
                    const completedForStatus = total > 0 ? completed : 0;
                    const queuedForStatus = total > 0 ? queued : 0;
                    const retryForStatus = total > 0 ? retryWait : 0;
                    const dispatchedForStatus = total > 0 ? dispatched : 0;

                    let statusText = window.rankbotBulkData.i18n.statusActive
                        .replace('{remaining}', remainingForStatus)
                        .replace('{completed}', completedForStatus)
                        .replace('{total}', totalForStatus)
                        .replace('{queued}', queuedForStatus)
                        .replace('{retry}', retryForStatus)
                        .replace('{dispatched}', dispatchedForStatus)
                        .replace('{failed}', (total > 0 ? failed : 0));

                     if (isPaused) {
                         const pauseLeft = Math.max(0, pauseUntil - nowTs);
                         const pauseLabelMap = {
                             insufficient_balance: 'insufficient balance',
                             not_connected: 'not connected'
                         };
                         const pauseLabel = pauseLabelMap[pauseReason] || pauseReason.replace(/_/g, ' ');
                         statusText += ` <span style="opacity:0.85; font-size:11px;">(Paused: ${pauseLabel}; ${pauseLeft}s)</span>`;
                     } else if (hasBulkWork && dispatched > 0 && completed < total) {
                         statusText += ` <span style="opacity:0.7; font-size:11px;">(Waiting for results...)</span>`;
                     } else if (hasBulkWork && (queued > 0 || retryWait > 0)) {
                         statusText += ` <span style="opacity:0.7; font-size:11px;">(Dispatching...)</span>`;
                     }

                     self.$progressText.html(statusText);
                     
                     self.$progressContainer.slideDown(); 
                     if (hasBulkWork) {
                         self.$stopBtn.show();
                     } else {
                         self.$stopBtn.hide();
                     }
                     
                     // Keep Polling
                     self.pollTimer = setTimeout(function() { self.pollOverall(); }, 2500);
                } else {
                     self.setProcessingLock(false);
                     self.$progressText.text(window.rankbotBulkData.i18n.statusIdle);
                     self.$stopBtn.hide();
                     
                     // Update menu bubble to 0 immediately when done
                     self.updateMenuBubble(0);
                     self.updateTopBar(0, null);

                     // One more refresh to clear queued/processing markers.
                     self.pollVisibleItemsStatus();
                     
                            const finishedTotal = total > 0 ? total : (self.lastBulkSnapshot.total || 0);
                            const finishedCompleted = total > 0 ? completed : (self.lastBulkSnapshot.completed || 0);
                            if (self.$progressContainer.is(':visible') && finishedTotal > 0 && finishedCompleted >= finishedTotal) {
                        // Finished
                        setTimeout(() => {
                           self.$progressContainer.slideUp();
                           // Reload page to show results after small delay
                           window.location.reload(); 
                        }, 2000);
                     }
                     self.setBusy(false);
                }
            })
            .fail(function() {
                // Silently fail on network error or just stop polling?
                // self.setBusy(false); 
            });
        }
    };

    $(document).ready(function() {
        if ($('.rb-bulk-wrap').length) {
            RbBulk.init();
        }
    });

})(jQuery);
