<?php
/**
 * Plugin Name: Random API Generator
 * Description: "Draw me an API!" Gutenberg block and shortcode that fetch a random API from freepublicapis.com, with copy URL and sample code snippets.
 * Version: 1.0.0
 * Author: yogesha
 * License: GPLv2 or later
 * Text Domain: random-api-generator
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

declare(strict_types=1);

if (! defined('ABSPATH')) {
	exit;
}

define('RANDAPIGEN_PLUGIN_FILE', __FILE__);
define('RANDAPIGEN_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('RANDAPIGEN_PLUGIN_URL', plugin_dir_url(__FILE__));
define('RANDAPIGEN_PLUGIN_VERSION', '1.0.0');

// Includes
require_once RANDAPIGEN_PLUGIN_DIR . 'includes/class-randapigen-rest.php';
require_once RANDAPIGEN_PLUGIN_DIR . 'includes/class-randapigen-admin.php';

// Hooks
add_action('init', static function (): void {
	// Register shortcode
	add_shortcode('randapigen_api_generator', 'randapigen_render_widget');
});

add_action('enqueue_block_editor_assets', static function (): void {
	wp_enqueue_script(
		'randapigen-block',
		RANDAPIGEN_PLUGIN_URL . 'assets/js/block.js',
		['wp-blocks', 'wp-element', 'wp-i18n', 'wp-api-fetch'],
		RANDAPIGEN_PLUGIN_VERSION,
		true
	);

	wp_localize_script('randapigen-block', 'RANDAPIGEN_DATA', [
		'restUrl' => esc_url_raw(rest_url('randapigen-api-generator/v1/draw')),
		'nonce' => wp_create_nonce('wp_rest'),
	]);
});

add_action('wp_enqueue_scripts', static function (): void {
	wp_enqueue_style(
		'randapigen-style',
		RANDAPIGEN_PLUGIN_URL . 'assets/css/style.css',
		[],
		RANDAPIGEN_PLUGIN_VERSION
	);

	wp_enqueue_script(
		'randapigen-frontend',
		RANDAPIGEN_PLUGIN_URL . 'assets/js/frontend.js',
		['wp-api-fetch'],
		RANDAPIGEN_PLUGIN_VERSION,
		true
	);

	wp_localize_script('randapigen-frontend', 'RANDAPIGEN_DATA', [
		'restUrl' => esc_url_raw(rest_url('randapigen-api-generator/v1/draw')),
		'nonce' => wp_create_nonce('wp_rest'),
	]);
});

/**
 * Render the frontend widget (used by shortcode and as a fallback container for block).
 */
function randapigen_render_widget(): string
{

	ob_start();
	?>
	<div class="randapigen-widget" data-rest-url="<?php echo esc_attr(rest_url('randapigen-api-generator/v1/draw')); ?>">
		<button type="button" class="randapigen-button" data-action="draw">
			<?php echo esc_html__('Draw me an API!', 'random-api-generator'); ?>
		</button>
		<div class="randapigen-result" hidden>
			<div class="randapigen-meta">
				<div class="randapigen-name"></div>
				<div class="randapigen-category"></div>
				<div class="randapigen-summary"></div>
			</div>
			<div class="randapigen-actions">
				<input class="randapigen-endpoint" type="text" readonly value="" />
				<button type="button" class="randapigen-copy-url" data-action="copy">Copy URL</button>
			</div>
			<details class="randapigen-snippets">
				<summary><?php echo esc_html__('Try it in your site', 'random-api-generator'); ?></summary>
				<div class="randapigen-code">
					<p><strong>JavaScript</strong></p>
					<pre class="randapigen-code-js"><code></code></pre>
					<p><strong>PHP (WordPress)</strong></p>
					<pre class="randapigen-code-php"><code></code></pre>
				</div>
			</details>
		</div>
	</div>
	<?php
	return (string) ob_get_clean();
}

// Register a basic server-rendered block that outputs the same markup
add_action('init', static function (): void {
	register_block_type('randapigen-api-generator/draw-api', [
		'render_callback' => static function (): string {
			return randapigen_render_widget();
		},
	]);
});

// Bootstrap REST routes
add_action('rest_api_init', [\RANDAPIGEN\Rest\Random_Api_Controller::class, 'register_routes']);

// Admin settings
\RANDAPIGEN\Admin\Settings::bootstrap();

// i18n - WordPress.org automatically loads translations

// Settings link on Plugins page
add_filter('plugin_action_links_' . plugin_basename(__FILE__), static function (array $links): array {
	$settings_url = admin_url('options-general.php?page=randapigen-settings');
	array_unshift($links, '<a href="' . esc_url($settings_url) . '">' . esc_html__('Settings', 'random-api-generator') . '</a>');
	return $links;
});


