<?php
declare(strict_types=1);

namespace RANDAPIGEN\Rest;

use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

if (! defined('ABSPATH')) {
	exit;
}

class Random_Api_Controller
{
	public static function get_all_items(): array
	{
		$items = apply_filters('randapigen_data_source', static::get_fallback_items());
		return is_array($items) ? $items : [];
	}

	public static function register_routes(): void
	{
		register_rest_route(
			'randapigen-api-generator/v1',
			'/draw',
			[
				'methods' => 'GET',
				'callback' => [static::class, 'handle_draw'],
				'permission_callback' => '__return_true', // This is intentionally a public endpoint for retrieving random API data
			]
		);
	}

	/**
	 * Returns a random API entry.
	 * Data source: curated fallback array, filterable via 'randapigen_data_source'.
	 */
	public static function handle_draw(WP_REST_Request $request)
	{
		// Attempt to get data from filter so site owners can supply their own source
		$items = apply_filters('randapigen_data_source', static::get_fallback_items());
		if (! is_array($items) || empty($items)) {
			return new WP_Error('randapigen_empty', 'No API items available');
		}

		// Filter by allowed categories if configured
		$allowed = get_option('randapigen_allowed_categories', []);
		if (is_array($allowed) && ! empty($allowed)) {
			$items = array_values(array_filter($items, static function($it) use ($allowed){
				$cat = isset($it['category']) ? (string) $it['category'] : '';
				return in_array($cat, $allowed, true);
			}));
		}
		if (empty($items)) {
			return new WP_Error('randapigen_no_match', 'No API items match the selected categories');
		}

		$index = wp_rand(0, (int) (count($items) - 1));
		$item = $items[$index];

		$item = static::sanitize_item($item);
		$response = new WP_REST_Response($item, 200);
		$response->set_headers([
			'Cache-Control' => 'no-store',
		]);
		return $response;
	}

	private static function sanitize_item(array $item): array
	{
		return [
			'name' => isset($item['name']) ? sanitize_text_field((string) $item['name']) : '',
			'category' => isset($item['category']) ? sanitize_text_field((string) $item['category']) : '',
			'summary' => isset($item['summary']) ? wp_kses_post((string) $item['summary']) : '',
			'endpoint' => isset($item['endpoint']) ? esc_url_raw((string) $item['endpoint']) : '',
			'homepage' => isset($item['homepage']) ? esc_url_raw((string) $item['homepage']) : '',
		];
	}

	/**
	 * Fallback curated items derived from freepublicapis.com weekly selection.
	 * Source: https://www.freepublicapis.com/
	 */
	private static function get_fallback_items(): array
	{
		return [
			[
				'name' => 'The Cat API',
				'category' => 'Animals',
				'summary' => 'Get random images of cats.',
				'endpoint' => 'https://api.thecatapi.com/v1/images/search',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'HackerNews API',
				'category' => 'Public Data',
				'summary' => 'Public Hacker News data via Firebase.',
				'endpoint' => 'https://hacker-news.firebaseio.com/v0/topstories.json',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Exchange Rate API',
				'category' => 'Finance',
				'summary' => 'Open access currency conversion and exchange rate data.',
				'endpoint' => 'https://open.er-api.com/v6/latest/USD',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Free Dictionary API',
				'category' => 'Language',
				'summary' => 'Definitions, pronunciations, synonyms from Wiktionary.',
				'endpoint' => 'https://api.dictionaryapi.dev/api/v2/entries/en/example',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Transport API (CH)',
				'category' => 'Transportation',
				'summary' => 'Swiss public transport locations, connections, stationboard.',
				'endpoint' => 'https://transport.opendata.ch/v1/locations?query=zurich',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Open-Meteo Weather API',
				'category' => 'Weather',
				'summary' => 'Free weather forecast and historical data.',
				'endpoint' => 'https://api.open-meteo.com/v1/forecast?latitude=47.37&longitude=8.54&hourly=temperature_2m',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'PokéAPI',
				'category' => 'Gaming',
				'summary' => 'All the Pokémon data in a free, RESTful API.',
				'endpoint' => 'https://pokeapi.co/api/v2/pokemon/pikachu',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Open Trivia DB',
				'category' => 'Entertainment',
				'summary' => 'Trivia questions for games and apps.',
				'endpoint' => 'https://opentdb.com/api.php?amount=5&type=multiple',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'REST Countries',
				'category' => 'Geodata',
				'summary' => 'Information about countries via REST endpoints.',
				'endpoint' => 'https://restcountries.com/v3.1/name/switzerland',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'TheMealDB',
				'category' => 'Food & Drink',
				'summary' => 'Open recipe database.',
				'endpoint' => 'https://www.themealdb.com/api/json/v1/1/random.php',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'SpaceX API',
				'category' => 'Science',
				'summary' => 'SpaceX launch and rocket data.',
				'endpoint' => 'https://api.spacexdata.com/v4/launches/latest',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Advice Slip JSON API',
				'category' => 'Work',
				'summary' => 'Random advice generator.',
				'endpoint' => 'https://api.adviceslip.com/advice',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Agify.io',
				'category' => 'Public Data',
				'summary' => 'Predict age from a name.',
				'endpoint' => 'https://api.agify.io/?name=michael',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Genderize.io',
				'category' => 'Public Data',
				'summary' => 'Predict gender from a name.',
				'endpoint' => 'https://api.genderize.io/?name=ashley',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Nationalize.io',
				'category' => 'Public Data',
				'summary' => 'Predict nationality from a name.',
				'endpoint' => 'https://api.nationalize.io/?name=lucas',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'CoinDesk Bitcoin Price Index',
				'category' => 'Finance',
				'summary' => 'Current Bitcoin price index in multiple currencies.',
				'endpoint' => 'https://api.coindesk.com/v1/bpi/currentprice.json',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Bored API',
				'category' => 'Entertainment',
				'summary' => 'Find activities to do when you are bored.',
				'endpoint' => 'https://www.boredapi.com/api/activity',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Sunrise-Sunset',
				'category' => 'Nature',
				'summary' => 'Sunrise and sunset times for a given latitude/longitude.',
				'endpoint' => 'https://api.sunrise-sunset.org/json?lat=36.7201600&lng=-4.4203400&formatted=0',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'OpenAQ',
				'category' => 'Environment',
				'summary' => 'Open air quality data.',
				'endpoint' => 'https://api.openaq.org/v2/latest?limit=5&country=CH',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'OpenFDA Drug Label',
				'category' => 'Health',
				'summary' => 'Public FDA data for drugs (example label endpoint).',
				'endpoint' => 'https://api.fda.gov/drug/label.json?search=active_ingredient:ibuprofen&limit=1',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Art Institute of Chicago API',
				'category' => 'Art',
				'summary' => 'Public art data from the Art Institute of Chicago.',
				'endpoint' => 'https://api.artic.edu/api/v1/artworks?limit=1',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'GitHub Repos API',
				'category' => 'Development',
				'summary' => 'GitHub public repository information.',
				'endpoint' => 'https://api.github.com/repos/facebook/react',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'UK Carbon Intensity',
				'category' => 'Environment',
				'summary' => 'Carbon intensity of the UK electricity system.',
				'endpoint' => 'https://api.carbonintensity.org.uk/intensity',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Frankfurter Exchange Rates',
				'category' => 'Finance',
				'summary' => 'Foreign exchange rates published by the European Central Bank.',
				'endpoint' => 'https://api.frankfurter.app/latest',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'TheCocktailDB',
				'category' => 'Food & Drink',
				'summary' => 'Open cocktail database.',
				'endpoint' => 'https://www.thecocktaildb.com/api/json/v1/1/random.php',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'OpenDota',
				'category' => 'Gaming',
				'summary' => 'Open Dota 2 data.',
				'endpoint' => 'https://api.opendota.com/api/heroStats',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'disease.sh',
				'category' => 'Health',
				'summary' => 'Open disease data including COVID-19.',
				'endpoint' => 'https://disease.sh/v3/covid-19/all',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'iTunes Search API',
				'category' => 'Music',
				'summary' => 'Search the iTunes Store and Apple Books Store.',
				'endpoint' => 'https://itunes.apple.com/search?term=radiohead&entity=album',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'GBIF Species API',
				'category' => 'Nature',
				'summary' => 'Global Biodiversity Information Facility species search.',
				'endpoint' => 'https://api.gbif.org/v1/species?name=Puma',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Bible API',
				'category' => 'Spiritual',
				'summary' => 'Free Bible verses API.',
				'endpoint' => 'https://bible-api.com/john%203:16',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'balldontlie',
				'category' => 'Sport',
				'summary' => 'Free NBA data API.',
				'endpoint' => 'https://www.balldontlie.io/api/v1/players?search=James',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Swiss GeoAdmin Layers',
				'category' => 'Switzerland',
				'summary' => 'Swiss geospatial layers catalog.',
				'endpoint' => 'https://api3.geo.admin.ch/rest/services/api/MapServer/layers?lang=en',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'Teleport Urban Areas',
				'category' => 'Travel',
				'summary' => 'List of urban areas around the world.',
				'endpoint' => 'https://api.teleport.org/api/urban_areas/',
				'homepage' => 'https://www.freepublicapis.com/',
			],
			[
				'name' => 'LibreTranslate',
				'category' => 'Language',
				'summary' => 'Open-source machine translation API (public demo).',
				'endpoint' => 'https://libretranslate.de/languages',
				'homepage' => 'https://www.freepublicapis.com/',
			],
		];
	}
}


