<?php
declare(strict_types=1);

namespace RANDAPIGEN\Admin;

if (! defined('ABSPATH')) {
	exit;
}

class Settings
{
	public const OPTION_KEY = 'randapigen_allowed_categories';

	public static function bootstrap(): void
	{
		add_action('admin_menu', [static::class, 'add_menu']);
		add_action('admin_init', [static::class, 'register_settings']);
	}

	public static function add_menu(): void
	{
		add_options_page(
			__('Random API Generator', 'random-api-generator'),
			__('Random API Generator', 'random-api-generator'),
			'manage_options',
			'randapigen-settings',
			[static::class, 'render_page']
		);
	}

	public static function register_settings(): void
	{
		register_setting('randapigen_settings_group', self::OPTION_KEY, [
			'type' => 'array',
			'default' => [],
			'sanitize_callback' => [static::class, 'sanitize_categories'],
		]);

		add_settings_section(
			'randapigen_main',
			__('Filtering', 'random-api-generator'),
			'__return_false',
			'randapigen-settings'
		);

		add_settings_field(
			'randapigen_categories',
			__('Allowed categories', 'random-api-generator'),
			[static::class, 'render_categories_field'],
			'randapigen-settings',
			'randapigen_main'
		);
	}

	public static function sanitize_categories($value): array
	{
		if (! is_array($value)) {
			return [];
		}
		$clean = [];
		foreach ($value as $cat) {
			$cat = sanitize_text_field((string) $cat);
			if ($cat !== '') {
				$clean[] = $cat;
			}
		}
		return array_values(array_unique($clean));
	}

	private static function get_all_categories(): array
	{
		if (class_exists('\\RANDAPIGEN\\Rest\\Random_Api_Controller')) {
			$items = \RANDAPIGEN\Rest\Random_Api_Controller::get_all_items();
		} else {
			$items = apply_filters('randapigen_data_source', []);
		}
		$categories = [];
		if (is_array($items) && ! empty($items)) {
			foreach ($items as $it) {
				if (isset($it['category'])) {
					$categories[] = (string) $it['category'];
				}
			}
		}
		$categories = array_values(array_unique(array_filter(array_map('sanitize_text_field', $categories))));
		sort($categories, SORT_NATURAL | SORT_FLAG_CASE);
		return $categories;
	}

	public static function render_categories_field(): void
	{
		$selected = get_option(self::OPTION_KEY, []);
		if (! is_array($selected)) { $selected = []; }
		$all = static::get_all_categories();
		if (empty($all)) {
			echo '<p>' . esc_html__('No categories available from data source.', 'random-api-generator') . '</p>';
			return;
		}
		echo '<fieldset>';
		foreach ($all as $cat) {
			$id = 'randapigen_cat_' . md5($cat);
			$checked = in_array($cat, $selected, true) ? ' checked' : '';
			echo '<label for="' . esc_attr($id) . '" style="display:block;margin:.25rem 0;">';
			echo '<input type="checkbox" name="' . esc_attr(self::OPTION_KEY) . '[]" id="' . esc_attr($id) . '" value="' . esc_attr($cat) . '"' . esc_attr($checked) . ' /> ' . esc_html($cat);
			echo '</label>';
		}
		echo '<p class="description">' . esc_html__('When set, random draw will be limited to these categories.', 'random-api-generator') . '</p>';
		echo '</fieldset>';
	}

	public static function render_page(): void
	{
		?>
		<div class="wrap">
			<h1><?php echo esc_html__('Random API Generator', 'random-api-generator'); ?></h1>
			<form method="post" action="options.php">
				<?php
				settings_fields('randapigen_settings_group');
				do_settings_sections('randapigen-settings');
				submit_button();
				?>
			</form>
		</div>
		<?php
	}
}


