(function(wp) {
    const { registerBlockType } = wp.blocks;
    const { TextControl, PanelBody, RangeControl } = wp.components;
    const { InspectorControls } = wp.blockEditor;
    const { __ } = wp.i18n;
    const { useState, useEffect } = wp.element;

    registerBlockType('qumo/form-embed', {
        title: __('Qumo Form', 'qumo-ai-plugin'),
        icon: 'feedback', // WordPress dashicon
        category: 'embed',
        description: __('Embed your Qumo form directly into any page or post. Create your form at qumo.ai and paste the form URL here.', 'qumo-ai-plugin'),
        keywords: ['form', 'qumo', 'embed', 'survey'],
        
        attributes: {
            formUrl: {
                type: 'string',
                default: '',
            },
            height: {
                type: 'number',
                default: 800,
            },
            borderRadius: {
                type: 'number',
                default: 0,
            },
        },
        
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { formUrl, height, borderRadius } = attributes;
            const [isValidUrl, setIsValidUrl] = useState(true);
            
            // Validate URL format when it changes
            useEffect(() => {
                if (formUrl) {
                    const urlPattern = /^https:\/\/qumo\.ai\/s\//;
                    setIsValidUrl(urlPattern.test(formUrl));
                } else {
                    setIsValidUrl(true); // Empty is considered valid until filled
                }
            }, [formUrl]);
            
            return [
                // Inspector Controls (sidebar)
                wp.element.createElement(
                    InspectorControls,
                    { key: 'inspector' },
                    wp.element.createElement(
                        PanelBody,
                        {
                            title: __('Form Settings', 'qumo-ai-plugin'),
                            initialOpen: true,
                        },
                        wp.element.createElement(
                            TextControl,
                            {
                                label: __('Form URL', 'qumo-ai-plugin'),
                                value: formUrl,
                                onChange: function(value) {
                                    setAttributes({ formUrl: value });
                                },
                                help: isValidUrl 
                                    ? __('Enter the URL of your Qumo form (format: https://qumo.ai/s/your-form-link)', 'qumo-ai-plugin')
                                    : __('Invalid URL format! Please use: https://qumo.ai/s/your-form-link', 'qumo-ai-plugin'),
                                className: '' // removed error style
                            }
                        ),
                        wp.element.createElement(
                            'div',
                            { className: 'qumo-section-label' },
                            __('Appearance', 'qumo-ai-plugin')
                        ),
                        wp.element.createElement(
                            RangeControl,
                            {
                                label: __('Form Height (px)', 'qumo-ai-plugin'),
                                value: height,
                                onChange: function(value) {
                                    setAttributes({ height: value });
                                },
                                min: 300,
                                max: 1000,
                                step: 50,
                            }
                        ),
                        wp.element.createElement(
                            RangeControl,
                            {
                                label: __('Corner Rounding (px)', 'qumo-ai-plugin'),
                                value: borderRadius,
                                onChange: function(value) {
                                    setAttributes({ borderRadius: value });
                                },
                                min: 0,
                                max: 30,
                                step: 1,
                            }
                        )
                    )
                ),
                
                // Block UI
                wp.element.createElement(
                    'div',
                    { 
                        className: props.className,
                        style: {
                            padding: '30px',
                            border: '1px solid #e0e0e0',
                            backgroundColor: '#f9f9f9',
                            borderRadius: '12px',
                            boxShadow: '0 4px 15px rgba(0,0,0,0.05)'
                        }
                    },
                    wp.element.createElement(
                        'div',
                        { className: 'qumo-editor-header' },
                        wp.element.createElement(
                            'img',
                            { 
                                src: qumoPluginData.pluginUrl + '/logo-black.svg',
                                alt: 'Qumo Logo',
                                className: 'qumo-logo'
                            }
                        ),
                        wp.element.createElement(
                            'strong',
                            {},
                            __('Qumo Form Embed', 'qumo-ai-plugin')
                        )
                    ),
                    wp.element.createElement(
                        TextControl,
                        {
                            label: __('Form URL', 'qumo-ai-plugin'),
                            labelPosition: 'top',
                            placeholder: __('Enter your Qumo form URL here...', 'qumo-ai-plugin'),
                            value: formUrl,
                            onChange: function(value) {
                                setAttributes({ formUrl: value });
                            },
                        }
                    ),
                    !isValidUrl && wp.element.createElement(
                        'p',
                        { 
                            style: {
                                display: 'block',
                                margin: '16px 0',
                                padding: '12px 16px',
                                backgroundColor: '#fff1f1',
                                borderLeft: '4px solid #dc3232',
                                borderRadius: '4px',
                                color: '#1e1e1e',
                                fontSize: '14px',
                                lineHeight: '1.5'
                            }
                        },
                        __('❌ Invalid URL format! Please use: https://qumo.ai/s/your-form-link', 'qumo-ai-plugin')
                    ),
                    
                    wp.element.createElement(
                        'p',
                        { className: 'qumo-helper-text' },
                        __('👉 Open the block settings sidebar for more customization options.', 'qumo-ai-plugin')
                    ),
                    
                    formUrl && isValidUrl ? 
                        wp.element.createElement(
                            'div',
                            { 
                                className: 'qumo-form-preview',
                                style: {
                                    marginTop: '16px',
                                    padding: '0px 20px 20px 20px',
                                    backgroundColor: '#fff',
                                    border: '1px solid #eaeaea',
                                    borderRadius: '12px',
                                    boxShadow: '0 2px 10px rgba(0,0,0,0.03)'
                                }
                            },
                            wp.element.createElement(
                                'p',
                                { className: 'qumo-form-preview-label' },
                                __('Form Preview', 'qumo-ai-plugin')
                            ),
                            wp.element.createElement(
                                'iframe',
                                {
                                    src: formUrl,
                                    width: '100%',
                                    height: Math.min(height, 500) + 'px', // Cap preview height for better editor experience
                                    frameBorder: '0',
                                    style: { 
                                        border: 'none', 
                                        maxWidth: '100%',
                                        borderRadius: borderRadius + 'px'
                                    },
                                    title: __('Qumo Form Preview', 'qumo-ai-plugin')
                                }
                            )
                        ) :
                        wp.element.createElement(
                            'div',
                            { className: 'qumo-placeholder-container' },
                            wp.element.createElement(
                                'img',
                                { 
                                    src: qumoPluginData.pluginUrl + '/logo-black.svg',
                                    alt: 'Qumo Logo',
                                    className: 'qumo-placeholder-logo'
                                }
                            ),
                            wp.element.createElement(
                                'p',
                                { className: 'qumo-placeholder-text' },
                                __('Enter a valid Qumo form URL above to see a preview.', 'qumo-ai-plugin')
                            ),
                            wp.element.createElement(
                                'p',
                                { className: 'qumo-placeholder-subtext' },
                                __('This block embeds your Qumo form directly into your page content. First create your form at qumo.ai, then paste your form URL here.', 'qumo-ai-plugin')
                            )
                        )
                )
            ];
        },
        
        save: function() {
            // Dynamic block, rendered server-side
            return null;
        }
    });
})(window.wp);
