(function(wp) {
    const { registerBlockType } = wp.blocks;
    const { TextControl, PanelBody, RangeControl, ColorPicker, ToggleControl, RadioControl, __experimentalRadioGroup: RadioGroup, __experimentalRadio: Radio } = wp.components;
    const { InspectorControls } = wp.blockEditor;
    const { __ } = wp.i18n;
    const { useState, useEffect } = wp.element;

    registerBlockType('qumo/chat-icon', {
        title: __('Qumo Chat Icon', 'qumo-ai-plugin'),
        icon: 'format-chat', // WordPress dashicon
        category: 'embed',
        description: __('Add a floating chat button that opens your Qumo form in a popup window. Perfect for contact forms, surveys, or feedback collection.', 'qumo-ai-plugin'),
        keywords: ['chat', 'qumo', 'icon', 'popup', 'form'],
        
        attributes: {
            formUrl: {
                type: 'string',
                default: '',
            },
            bubbleColor: {
                type: 'string',
                default: '#2271b1',
            },
            iconColor: {
                type: 'string',
                default: '#ffffff',
            },
            position: {
                type: 'string',
                default: 'right',
            },
            popupHeight: {
                type: 'number',
                default: 800,
            },
            popupWidth: {
                type: 'number',
                default: 400,
            },
            bubbleText: {
                type: 'string',
                default: '',
            },
            showOnMobile: {
                type: 'boolean',
                default: true,
            },
            iconType: {
                type: 'string',
                default: 'chat',
            },
            popupBorderRadius: {
                type: 'number',
                default: 10,
            }
        },
        
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { formUrl, bubbleColor, iconColor, position, popupHeight, popupWidth, bubbleText, showOnMobile, iconType, popupBorderRadius } = attributes;
            const [isValidUrl, setIsValidUrl] = useState(true);
            
            // Validate URL format when it changes
            useEffect(() => {
                if (formUrl) {
                    const urlPattern = /^https:\/\/qumo\.ai\/s\//;
                    setIsValidUrl(urlPattern.test(formUrl));
                } else {
                    setIsValidUrl(true); // Empty is considered valid until filled
                }
            }, [formUrl]);
            
            // Position options
            const positionOptions = [
                { label: __('Bottom Right', 'qumo-ai-plugin'), value: 'right' },
                { label: __('Bottom Left', 'qumo-ai-plugin'), value: 'left' },
            ];

            // Icon options with matching SVG icons
            const iconOptions = [
                { value: 'chat', label: __('Chat', 'qumo-ai-plugin'), icon: `<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M7.9 20A9 9 0 1 0 4 16.1L2 22Z"/></svg>` },
                { value: 'help', label: __('Help', 'qumo-ai-plugin'), icon: `<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><path d="M9.09 9a3 3 0 0 1 5.83 1c0 2-3 3-3 3"/><path d="M12 17h.01"/></svg>` },
                { value: 'support', label: __('Support', 'qumo-ai-plugin'), icon: `<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><path d="m4.93 4.93 4.24 4.24"/><path d="m14.83 9.17 4.24-4.24"/><path d="m14.83 14.83 4.24 4.24"/><path d="m9.17 14.83-4.24 4.24"/><circle cx="12" cy="12" r="4"/></svg>` },
                { value: 'message', label: __('Message', 'qumo-ai-plugin'), icon: `<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect width="20" height="16" x="2" y="4" rx="2"/><path d="m22 7-8.97 5.7a1.94 1.94 0 0 1-2.06 0L2 7"/></svg>` }
            ];

            return [
                // Inspector Controls (sidebar)
                wp.element.createElement(
                    InspectorControls,
                    { key: 'inspector' },
                    wp.element.createElement(
                        PanelBody,
                        {
                            title: __('Form Settings', 'qumo-ai-plugin'),
                            initialOpen: true,
                        },
                        wp.element.createElement(
                            TextControl,
                            {
                                label: __('Form URL', 'qumo-ai-plugin'),
                                value: formUrl,
                                onChange: function(value) {
                                    setAttributes({ formUrl: value });
                                },
                                help: isValidUrl 
                                    ? __('Enter the URL of your Qumo form (format: https://qumo.ai/s/your-form-link)', 'qumo-ai-plugin')
                                    : __('Invalid URL format! Please use: https://qumo.ai/s/your-form-link', 'qumo-ai-plugin'),
                                className: isValidUrl ? '' : 'qumo-invalid-url',
                            }
                        )
                    ),
                    wp.element.createElement(
                        PanelBody,
                        {
                            title: __('Chat Icon Settings', 'qumo-ai-plugin'),
                            initialOpen: true,
                        },
                        wp.element.createElement(
                            'div',
                            { className: 'qumo-section-label' },
                            __('Appearance', 'qumo-ai-plugin')
                        ),
                        wp.element.createElement(
                            'p',
                            { className: 'qumo-color-picker-label' },
                            __('Icon Background Color', 'qumo-ai-plugin')
                        ),
                        wp.element.createElement(
                            ColorPicker,
                            {
                                color: bubbleColor,
                                onChangeComplete: function(color) {
                                    setAttributes({ bubbleColor: color.hex });
                                },
                                disableAlpha: true
                            }
                        ),
                        wp.element.createElement(
                            'p',
                            { className: 'qumo-color-picker-label' },
                            __('Icon Symbol Color', 'qumo-ai-plugin')
                        ),
                        wp.element.createElement(
                            ColorPicker,
                            {
                                color: iconColor,
                                onChangeComplete: function(color) {
                                    setAttributes({ iconColor: color.hex });
                                },
                                disableAlpha: true
                            }
                        ),
                        wp.element.createElement(
                            'div',
                            { className: 'qumo-icon-type-selector' },
                            wp.element.createElement(
                                'label',
                                { style: { display: 'block', marginBottom: '8px' } },
                                __('Icon Style', 'qumo-ai-plugin')
                            ),
                            wp.element.createElement(
                                'div',
                                { className: 'qumo-icon-options-grid' },
                                iconOptions.map(option => 
                                    wp.element.createElement(
                                        'div',
                                        { 
                                            key: option.value,
                                            className: `qumo-icon-option ${iconType === option.value ? 'selected' : ''}`,
                                            onClick: () => setAttributes({ iconType: option.value })
                                        },
                                        [
                                            wp.element.createElement(
                                                'span',
                                                { 
                                                    className: 'qumo-icon-svg',
                                                    dangerouslySetInnerHTML: { __html: option.icon }
                                                }
                                            ),
                                            wp.element.createElement(
                                                'span',
                                                { className: 'qumo-icon-option-label' }, 
                                                option.label
                                            )
                                        ]
                                    )
                                )
                            )
                        ),
                        wp.element.createElement(
                            'div',
                            { className: 'qumo-section-label' },
                            __('Positioning & Text', 'qumo-ai-plugin')
                        ),
                        wp.element.createElement(
                            TextControl,
                            {
                                label: __('Icon Text (Optional)', 'qumo-ai-plugin'),
                                value: bubbleText,
                                onChange: function(value) {
                                    setAttributes({ bubbleText: value });
                                },
                                help: __('Short text to show next to the chat icon (e.g. "Need help?"). Leave empty for icon only.', 'qumo-ai-plugin')
                            }
                        ),
                        wp.element.createElement(
                            'div',
                            { className: 'qumo-position-selector' },
                            wp.element.createElement(
                                'label',
                                { style: { marginBottom: '8px', display: 'block' } },
                                __('Screen Position', 'qumo-ai-plugin')
                            ),
                            positionOptions.map(option => 
                                wp.element.createElement(
                                    'div',
                                    { 
                                        key: option.value,
                                        className: `qumo-position-option ${position === option.value ? 'selected' : ''}`,
                                        onClick: () => setAttributes({ position: option.value })
                                    },
                                    option.label
                                )
                            )
                        ),
                        wp.element.createElement(
                            ToggleControl,
                            {
                                label: __('Show on Mobile Devices', 'qumo-ai-plugin'),
                                checked: showOnMobile,
                                onChange: function(value) {
                                    setAttributes({ showOnMobile: value });
                                },
                                help: showOnMobile 
                                    ? __('Chat icon will be visible on mobile devices', 'qumo-ai-plugin')
                                    : __('Chat icon will be hidden on mobile devices', 'qumo-ai-plugin')
                            }
                        )
                    ),
                    wp.element.createElement(
                        PanelBody,
                        {
                            title: __('Chat Window Settings', 'qumo-ai-plugin'),
                            initialOpen: true,
                        },
                        wp.element.createElement(
                            RangeControl,
                            {
                                label: __('Window Width (px)', 'qumo-ai-plugin'),
                                value: popupWidth,
                                onChange: function(value) {
                                    setAttributes({ popupWidth: value });
                                },
                                min: 280,
                                max: 1000,
                                step: 10,
                            }
                        ),
                        wp.element.createElement(
                            RangeControl,
                            {
                                label: __('Window Height (px)', 'qumo-ai-plugin'),
                                value: popupHeight,
                                onChange: function(value) {
                                    setAttributes({ popupHeight: value });
                                },
                                min: 300,
                                max: 1000,
                                step: 10,
                            }
                        ),
                        wp.element.createElement(
                            RangeControl,
                            {
                                label: __('Corner Rounding (px)', 'qumo-ai-plugin'),
                                value: popupBorderRadius,
                                onChange: function(value) {
                                    setAttributes({ popupBorderRadius: value });
                                },
                                min: 0,
                                max: 30,
                                step: 1,
                            }
                        )
                    )
                ),
                
                // Block UI
                wp.element.createElement(
                    'div',
                    { 
                        className: props.className,
                        style: {
                            padding: '30px',
                            border: '1px solid #e0e0e0',
                            backgroundColor: '#f9f9f9',
                            borderRadius: '12px',
                            boxShadow: '0 4px 15px rgba(0,0,0,0.05)'
                        }
                    },
                    wp.element.createElement(
                        'div',
                        { className: 'qumo-chat-icon-editor' },
                        wp.element.createElement(
                            'div',
                            { className: 'qumo-editor-header' },
                            wp.element.createElement(
                                'img',
                                { 
                                    src: qumoPluginData.pluginUrl + '/logo-black.svg',
                                    alt: 'Qumo Logo',
                                    className: 'qumo-logo'
                                }
                            ),
                            wp.element.createElement(
                                'strong',
                                {},
                                __('Qumo Chat Icon', 'qumo-ai-plugin')
                            )
                        ),
                        
                        // Form URL input section
                        wp.element.createElement(
                            'div', 
                            { style: { marginBottom: '20px' } },
                            wp.element.createElement(
                                TextControl,
                                {
                                    label: __('Form URL', 'qumo-ai-plugin'),
                                    labelPosition: 'top',
                                    placeholder: __('Enter your Qumo form URL here...', 'qumo-ai-plugin'),
                                    value: formUrl,
                                    onChange: function(value) {
                                        setAttributes({ formUrl: value });
                                    },
                                }
                            ),
                            !isValidUrl && wp.element.createElement(
                                'p',
                                { 
                                    style: {
                                        display: 'block',
                                        margin: '16px 0',
                                        padding: '12px 16px',
                                        backgroundColor: '#fff1f1',
                                        borderLeft: '4px solid #dc3232',
                                        borderRadius: '4px',
                                        color: '#1e1e1e',
                                        fontSize: '14px',
                                        lineHeight: '1.5'
                                    }
                                },
                                __('❌ Invalid URL format! Please use: https://qumo.ai/s/your-form-link', 'qumo-ai-plugin')
                            )
                        ),
                        
                        wp.element.createElement(
                            'div',
                            { className: 'qumo-helper-container' },
                            wp.element.createElement(
                                'p',
                                { className: 'qumo-helper-text' },
                                __('👉 Open the block settings sidebar for more customization options.', 'qumo-ai-plugin')
                            ),
                            wp.element.createElement(
                                'p',
                                { className: 'qumo-helper-text' },
                                __('💡 This block adds a chat button to your site. When clicked, it opens your Qumo form in a popup window. First create your form at qumo.ai, then paste your form URL here.', 'qumo-ai-plugin')
                            )
                        )
                    )
                )
            ];
        },
        
        save: function() {
            // Dynamic block, rendered server-side
            return null;
        }
    });
})(window.wp);
