<?php
/**
 * Qumo Blocks Class
 * Handles registration and rendering of custom blocks
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Qumo_Blocks {
    /**
     * Constructor
     */
    public function __construct() {
        add_action('init', array($this, 'register_blocks'));
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_editor_assets'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
    }

    /**
     * Register custom blocks
     */
    public function register_blocks() {
        // Register the form embed block
        register_block_type('qumo/form-embed', array(
            'editor_script' => 'qumo-form-embed-editor',
            'render_callback' => array($this, 'render_form_embed_block'),
            'attributes' => array(
                'formUrl' => array(
                    'type' => 'string',
                    'default' => '',
                ),
                'height' => array(
                    'type' => 'number',
                    'default' => 800,
                ),
                'borderRadius' => array(
                    'type' => 'number',
                    'default' => 0,
                ),
            ),
        ));

        // Register the chat icon block
        register_block_type('qumo/chat-icon', array(
            'editor_script' => 'qumo-chat-icon-editor',
            'render_callback' => array($this, 'render_chat_icon_block'),
            'attributes' => array(
                'formUrl' => array(
                    'type' => 'string',
                    'default' => '',
                ),
                'bubbleColor' => array(
                    'type' => 'string',
                    'default' => '#2271b1',
                ),
                'iconColor' => array(
                    'type' => 'string',
                    'default' => '#ffffff',
                ),
                'position' => array(
                    'type' => 'string',
                    'default' => 'right',
                ),
                'popupHeight' => array(
                    'type' => 'number',
                    'default' => 800,
                ),
                'popupWidth' => array(
                    'type' => 'number',
                    'default' => 400,
                ),
                'bubbleText' => array(
                    'type' => 'string',
                    'default' => '',
                ),
                'showOnMobile' => array(
                    'type' => 'boolean',
                    'default' => true,
                ),
                'iconType' => array(
                    'type' => 'string',
                    'default' => 'chat',
                ),
                'popupBorderRadius' => array(
                    'type' => 'number',
                    'default' => 10,
                ),
            ),
        ));
    }

    /**
     * Enqueue editor assets
     */
    public function enqueue_editor_assets() {
        // Enqueue form embed block editor JS
        wp_enqueue_script(
            'qumo-form-embed-editor',
            QUMO_AI_FORMS_URL . 'blocks/js/form-embed.js',
            array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components'),
            QUMO_AI_FORMS_VERSION,
            true
        );

        // Localize script to pass plugin URL
        wp_localize_script(
            'qumo-form-embed-editor',
            'qumoPluginData',
            array('pluginUrl' => QUMO_AI_FORMS_URL)
        );

        // Enqueue chat icon block editor JS
        wp_enqueue_script(
            'qumo-chat-icon-editor',
            QUMO_AI_FORMS_URL . 'blocks/js/chat-icon.js',
            array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components'),
            QUMO_AI_FORMS_VERSION,
            true
        );

        // Localize script to pass plugin URL
        wp_localize_script(
            'qumo-chat-icon-editor',
            'qumoPluginData',
            array('pluginUrl' => QUMO_AI_FORMS_URL)
        );

        // Enqueue editor CSS
        wp_enqueue_style(
            'qumo-form-embed-editor-style',
            QUMO_AI_FORMS_URL . 'blocks/css/form-embed.css',
            array(),
            QUMO_AI_FORMS_VERSION
        );

        // Enqueue chat icon editor CSS
        wp_enqueue_style(
            'qumo-chat-icon-editor-style',
            QUMO_AI_FORMS_URL . 'blocks/css/chat-icon.css',
            array(),
            QUMO_AI_FORMS_VERSION
        );

        // Enqueue common UI CSS
        wp_enqueue_style(
            'qumo-common-ui',
            QUMO_AI_FORMS_URL . 'blocks/css/common-ui.css',
            array(),
            QUMO_AI_FORMS_VERSION
        );

        // Enqueue editor extensions CSS
        wp_enqueue_style(
            'qumo-editor-extensions',
            QUMO_AI_FORMS_URL . 'blocks/css/editor-extensions.css',
            array('qumo-common-ui'),  // Make it depend on common UI
            QUMO_AI_FORMS_VERSION
        );
    }

    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets() {
        // Enqueue frontend CSS
        wp_enqueue_style(
            'qumo-form-embed-style',
            QUMO_AI_FORMS_URL . 'blocks/css/form-embed.css',
            array(),
            QUMO_AI_FORMS_VERSION
        );

        // Enqueue chat icon frontend CSS
        wp_enqueue_style(
            'qumo-chat-icon-style',
            QUMO_AI_FORMS_URL . 'blocks/css/chat-icon.css',
            array(),
            QUMO_AI_FORMS_VERSION
        );

        // Enqueue chat icon frontend JS
        wp_enqueue_script(
            'qumo-chat-icon-frontend',
            QUMO_AI_FORMS_URL . 'blocks/js/chat-icon-frontend.js',
            array(),
            QUMO_AI_FORMS_VERSION,
            true
        );

        // Add common UI to frontend too for shared styles
        wp_enqueue_style(
            'qumo-common-ui',
            QUMO_AI_FORMS_URL . 'blocks/css/common-ui.css',
            array(),
            QUMO_AI_FORMS_VERSION
        );
    }

    /**
     * Render the form embed block
     *
     * @param array $attributes Block attributes
     * @return string Block HTML
     */
    public function render_form_embed_block($attributes) {
        $form_url = !empty($attributes['formUrl']) ? esc_url($attributes['formUrl']) : '';
        $height = !empty($attributes['height']) ? intval($attributes['height']) : 800;
        $border_radius = isset($attributes['borderRadius']) ? intval($attributes['borderRadius']) : 0;
        
        // Validate URL format
        if (empty($form_url) || !preg_match('/^https:\/\/qumo\.ai\/s\//', $form_url)) {
            return '<p>' . esc_html__('Please enter a valid Qumo form URL (format: https://qumo.ai/s/your-form-link).', 'qumo-ai-forms') . '</p>';
        }

        ob_start();
        ?>
        <div class="qumo-form-embed-container loading">
            <iframe 
                src="<?php echo esc_url($form_url); ?>" 
                width="100%" 
                height="<?php echo esc_attr($height); ?>"
                frameborder="0" 
                allowtransparency="true"
                allow="microphone; clipboard-read; clipboard-write"
                style="border: none; max-width: 100%; border-radius: <?php echo esc_attr($border_radius); ?>px;"
                title="<?php echo esc_attr__('Qumo Form', 'qumo-ai-forms'); ?>"
                onload="this.parentElement.classList.remove('loading')"
            ></iframe>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Render the chat icon block
     *
     * @param array $attributes Block attributes
     * @return string Block HTML
     */
    public function render_chat_icon_block($attributes) {
        $form_url = !empty($attributes['formUrl']) ? esc_url($attributes['formUrl']) : '';
        $bubble_color = !empty($attributes['bubbleColor']) ? esc_attr($attributes['bubbleColor']) : '#2271b1';
        $icon_color = !empty($attributes['iconColor']) ? esc_attr($attributes['iconColor']) : '#ffffff';
        $position = !empty($attributes['position']) ? esc_attr($attributes['position']) : 'right';
        $popup_height = !empty($attributes['popupHeight']) ? intval($attributes['popupHeight']) : 800;
        $popup_width = !empty($attributes['popupWidth']) ? intval($attributes['popupWidth']) : 400;
        $bubble_text = !empty($attributes['bubbleText']) ? esc_html($attributes['bubbleText']) : '';
        $show_on_mobile = isset($attributes['showOnMobile']) ? (bool)$attributes['showOnMobile'] : true;
        $icon_type = !empty($attributes['iconType']) ? esc_attr($attributes['iconType']) : 'chat';
        $popup_border_radius = isset($attributes['popupBorderRadius']) ? intval($attributes['popupBorderRadius']) : 10;
        
        // Set fixed bubble size
        $bubble_size = 50; // This matches the CSS default size

        // Validate URL format
        if (empty($form_url) || !preg_match('/^https:\/\/qumo\.ai\/s\//', $form_url)) {
            return '<p>' . esc_html__('Please enter a valid Qumo form URL (format: https://qumo.ai/s/your-form-link).', 'qumo-ai-forms') . '</p>';
        }

        // Generate a unique ID for this chat icon instance
        $unique_id = 'qumo-chat-' . uniqid();

        // Get icon SVG based on selected type
        $icon_svg = $this->get_chat_icon_svg($icon_type, $icon_color);

        ob_start();
        ?>
        <div class="qumo-chat-icon position-<?php echo esc_attr($position); ?> <?php echo $show_on_mobile ? '' : esc_attr('qumo-chat-icon-mobile-hidden'); ?>">
            <?php if (!empty($bubble_text)): ?>
            <div class="qumo-chat-icon-text" data-text-id="<?php echo esc_attr($unique_id); ?>-text">
                <?php echo esc_html($bubble_text); ?>
                <span class="qumo-chat-icon-text-close" role="button" aria-label="<?php esc_attr_e('Close message', 'qumo-ai-forms'); ?>">×</span>
            </div>
            <?php endif; ?>
            <div 
                class="qumo-chat-icon-button" 
                style="background-color: <?php echo esc_attr($bubble_color); ?>; width: <?php echo esc_attr($bubble_size); ?>px; height: <?php echo esc_attr($bubble_size); ?>px;"
                data-popup-id="<?php echo esc_attr($unique_id); ?>"
                aria-label="<?php echo esc_attr__('Open chat form', 'qumo-ai-forms'); ?>"
            >
                <?php echo wp_kses($icon_svg, array(
                    'svg' => array(
                        'xmlns' => true,
                        'viewBox' => true,
                        'fill' => true,
                        'width' => true,
                        'height' => true,
                        'preserveAspectRatio' => true,
                        'stroke' => true,
                        'stroke-width' => true,
                        'stroke-linecap' => true,
                        'stroke-linejoin' => true,
                        'style' => true
                    ),
                    'path' => array(
                        'd' => true,
                        'stroke' => true,
                        'style' => true
                    ),
                    'circle' => array(
                        'cx' => true,
                        'cy' => true,
                        'r' => true,
                        'style' => true
                    ),
                    'rect' => array(
                        'width' => true,
                        'height' => true,
                        'x' => true,
                        'y' => true,
                        'rx' => true,
                        'style' => true
                    )
                )); ?>
            </div>
        </div>

        <div id="<?php echo esc_attr($unique_id); ?>" class="qumo-chat-popup position-<?php echo esc_attr($position); ?>" style="width: <?php echo esc_attr($popup_width); ?>px; height: <?php echo esc_attr($popup_height); ?>px; border-radius: <?php echo esc_attr($popup_border_radius); ?>px;">
            <div class="qumo-chat-popup-content">
                <button class="qumo-chat-popup-overlay-close" aria-label="<?php echo esc_attr__('Close chat form', 'qumo-ai-forms'); ?>">×</button>
                <iframe 
                    data-src="<?php echo esc_url($form_url); ?>" 
                    frameborder="0" 
                    allowtransparency="true"
                    allow="microphone; clipboard-read; clipboard-write"
                    title="<?php echo esc_attr__('Qumo Chat Form', 'qumo-ai-forms'); ?>"
                ></iframe>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Get SVG icon for chat icon based on type
     * 
     * @param string $icon_type The type of icon
     * @param string $color Icon color
     * @return string SVG markup
     */
    private function get_chat_icon_svg($icon_type, $color) {
        $color = esc_attr($color);
        switch ($icon_type) {
            case 'help':
                return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="none" stroke="' . $color . '" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 24px; height: 24px;"><circle cx="12" cy="12" r="10"/><path d="M9.09 9a3 3 0 0 1 5.83 1c0 2-3 3-3 3"/><path d="M12 17h.01"/></svg>';
            case 'support':
                return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="none" stroke="' . $color . '" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 24px; height: 24px;"><circle cx="12" cy="12" r="10"/><path d="m4.93 4.93 4.24 4.24"/><path d="m14.83 9.17 4.24-4.24"/><path d="m14.83 14.83 4.24 4.24"/><path d="m9.17 14.83-4.24 4.24"/><circle cx="12" cy="12" r="4"/></svg>';
            case 'message':
                return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="none" stroke="' . $color . '" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 24px; height: 24px;"><rect width="20" height="16" x="2" y="4" rx="2"/><path d="m22 7-8.97 5.7a1.94 1.94 0 0 1-2.06 0L2 7"/></svg>';
            case 'chat':
            default:
                return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="none" stroke="' . $color . '" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 24px; height: 24px;"><path d="M7.9 20A9 9 0 1 0 4 16.1L2 22Z"/></svg>';
        }
    }
}

// Initialize the blocks class
new Qumo_Blocks();
