<?php
namespace QuietAdmin\Features;

if ( ! defined( 'ABSPATH' ) ) { exit; }

class Dashboard {
    private $option_key = 'quietadmin_dashboard';
    private $opts = null;

    public function id(){ return 'dashboard'; }
    public function title(){ return __( 'Dashboard Cleanup', 'quiet-admin' ); }

    public function defaults(){
        return [ 'hidden' => [], 'zen' => false ];
    }

    private function get_options() : array {
        if ( isset( $this->opts ) ) return $this->opts;
        $stored = get_option( $this->option_key, [] );
        $this->opts = wp_parse_args( is_array( $stored ) ? $stored : [], $this->defaults() );
        return $this->opts;
    }

    private function update_options( array $opts ) : void {
        $this->opts = $opts;
        update_option( $this->option_key, $this->opts );
    }

    public function boot() : void {
        add_action( 'wp_dashboard_setup', [ $this, 'cleanup_widgets' ], 9999 );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_dashboard_css' ] );
    }

    public function enqueue_dashboard_css( $hook ) : void {
        if ( 'index.php' !== $hook ) { return; }
        $o = $this->get_options();
        if ( empty( $o['zen'] ) ) { return; }

        wp_register_style( 'quietadmin-admin', false, [], QUIETADMIN_VERSION );
        wp_enqueue_style( 'quietadmin-admin' );
        wp_add_inline_style( 'quietadmin-admin', '#wpfooter,#screen-meta-links{display:none!important}' );
    }

    public function cleanup_widgets() : void {
        $o      = $this->get_options();
        $hidden = (array) ( $o['hidden'] ?? [] );

        if ( in_array( 'welcome_panel', $hidden, true ) ) {
            remove_action( 'welcome_panel', 'wp_welcome_panel' );
        }
        if ( empty( $hidden ) ) return;

        foreach ( [ 'normal', 'side', 'column3', 'column4' ] as $ctx ) {
            foreach ( $hidden as $id ) {
                remove_meta_box( $id, 'dashboard', $ctx );
            }
        }
    }

    public function render_settings() : void {
        $o       = $this->get_options();
        $hidden  = (array) ( $o['hidden'] ?? [] );
        $widgets = $this->get_dashboard_widgets_list();
        ?>
        <div class="qa-field" style="margin-bottom:12px">
            <label>
                <input type="checkbox" name="quietadmin_dashboard[zen]" value="1" <?php checked( ! empty( $o['zen'] ) ); ?> />
                <?php echo esc_html__( 'Enable Zen Mode on the Dashboard (hide footer & “Screen Options”).', 'quiet-admin' ); ?>
            </label>
        </div>
        <div class="qa-field">
            <label style="font-weight:600; display:block; margin-bottom:6px;">
                <?php echo esc_html__( 'Hide these Dashboard widgets', 'quiet-admin' ); ?>
            </label>
            <?php if ( empty( $widgets ) ) : ?>
                <em><?php echo esc_html__( 'No widgets detected. Tip: visit the Dashboard once so WordPress registers all core/plugin widgets.', 'quiet-admin' ); ?></em>
            <?php else : ?>
                <fieldset style="max-height:320px; overflow:auto; border:1px solid #ccd0d4; border-radius:6px; padding:8px; background:#fff;">
                    <?php foreach ( $widgets as $id => $title ) : ?>
                        <label style="display:block; margin:4px 0;">
                            <input type="checkbox" name="quietadmin_dashboard[hidden][]" value="<?php echo esc_attr( $id ); ?>"
                                <?php checked( in_array( $id, $hidden, true ) ); ?> />
                            <?php echo esc_html( $title ); ?>
                            <code style="opacity:.7"><?php echo esc_html( $id ); ?></code>
                        </label>
                    <?php endforeach; ?>
                </fieldset>
            <?php endif; ?>
        </div>
        <?php
    }

    public function save_from_request( array $src ) : void {
        if ( empty( $src['quietadmin_dashboard'] ) || ! is_array( $src['quietadmin_dashboard'] ) ) return;

        $in   = $src['quietadmin_dashboard'];
        $next = $this->get_options();

        $next['zen'] = ! empty( $in['zen'] );

        $hidden = [];
        if ( isset( $in['hidden'] ) && is_array( $in['hidden'] ) ) {
            foreach ( $in['hidden'] as $id ) {
                $id = sanitize_key( $id );
                if ( $id ) { $hidden[] = $id; }
            }
        }
        $next['hidden'] = array_values( array_unique( $hidden ) );

        $this->update_options( $next );
    }

    private function get_dashboard_widgets_list() : array {
        if ( ! function_exists( 'wp_dashboard_setup' ) ) {
            require_once ABSPATH . 'wp-admin/includes/dashboard.php';
        }

        if ( function_exists( 'set_current_screen' ) ) {
            try { set_current_screen( 'dashboard' ); } catch ( \Throwable $e ) {}
        }
        if ( function_exists( 'wp_dashboard_setup' ) ) {
            wp_dashboard_setup();
        }

        global $wp_meta_boxes;
        $out = [];
        if ( empty( $wp_meta_boxes['dashboard'] ) ) { return $out; }

        foreach ( $wp_meta_boxes['dashboard'] as $ctx => $by_pri ) {
            foreach ( (array) $by_pri as $boxes ) {
                foreach ( (array) $boxes as $id => $box ) {
                    if ( empty( $id ) ) { continue; }
                    $title      = isset( $box['title'] ) ? wp_strip_all_tags( $box['title'] ) : $id;
                    $out[ $id ] = $title;
                }
            }
        }
        ksort( $out, SORT_NATURAL | SORT_FLAG_CASE );
        return $out;
    }

}
