<?php
namespace QuietAdmin;

if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Settings orchestrator.
 * Discovers feature classes and delegates rendering/saving.
 */
class Settings {
    private static $instance = null;

    /** @var array<int,object> */
    private $features = [];

    public static function instance() {
        return self::$instance ?: self::$instance = new self();
    }

    private function __construct() {
        // Load feature classes.
        $this->maybe_require( 'features/class-qa-welcome.php' );
        $this->maybe_require( 'features/class-qa-notices.php' );
        $this->maybe_require( 'features/class-qa-comments.php' );
        $this->maybe_require( 'features/class-qa-dashboard.php' );

        // Register discovered features.
        $this->register_if_exists( '\\QuietAdmin\\Features\\Welcome' );
        $this->register_if_exists( '\\QuietAdmin\\Features\\Notices' );
        $this->register_if_exists( '\\QuietAdmin\\Features\\Comments' );
        $this->register_if_exists( '\\QuietAdmin\\Features\\Dashboard' );
        
        foreach ( $this->features as $f ) {
            if ( method_exists( $f, 'boot' ) ) {
                $f->boot();
            }
        }
        add_action( 'admin_menu', [ $this, 'register_menu' ] );
        add_action( 'admin_post_quietadmin_save_settings', [ $this, 'handle_save' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_settings_assets' ] );

    }

    public function enqueue_settings_assets( $hook ) {
        if ( 'settings_page_quiet-admin' !== $hook ) { return; }

        wp_register_style( 'quietadmin-admin', false, [], QUIETADMIN_VERSION );
        wp_enqueue_style( 'quietadmin-admin' );
        wp_add_inline_style( 'quietadmin-admin', '
            .qa-settings .qa-tabs{ display:grid; grid-template-columns:220px 1fr; gap:16px; margin-top:12px }
            .qa-settings .qa-tabs__nav{ background:#fff; border:1px solid #dce0e4; border-radius:10px; padding:8px; display:flex; flex-direction:column; gap:6px }
            .qa-settings .qa-tab{ display:block; text-decoration:none; background:transparent; border:0; padding:10px 12px; border-radius:6px; cursor:pointer; font-size:14px; color:#1d2327 }
            .qa-settings .qa-tab.is-active{ background:#f3f4f6; font-weight:600 }
            .qa-settings .qa-card{ background:#fff; border:1px solid #dce0e4; border-radius:12px; padding:16px; box-shadow:0 1px 2px rgba(0,0,0,.03) }
            .qa-settings .qa-grid{ display:grid; grid-template-columns:repeat(2,minmax(0,1fr)); gap:8px 16px }
            @media (max-width:782px){ .qa-settings .qa-tabs{ grid-template-columns:1fr } }
        ' );
    }

    private function maybe_require( $rel ) {
        $path = trailingslashit( QUIETADMIN_DIR_PATH ) . 'includes/' . ltrim( $rel, '/' );
        if ( file_exists( $path ) ) { require_once $path; }
    }

    private function register_if_exists( $fqcn ) {
        if ( class_exists( $fqcn ) ) {
            $this->features[] = new $fqcn();
        }
    }

    /* ---------- Menus ---------- */

    public function register_menu() {
        add_options_page(
            __( 'Quiet Admin', 'quiet-admin' ),
            __( 'Quiet Admin', 'quiet-admin' ),
            'manage_options',
            'quiet-admin',
            [ $this, 'render_page' ]
        );
    }

    public function register_network_menu() {
        add_menu_page(
            __( 'Quiet Admin (Network)', 'quiet-admin' ),
            __( 'Quiet Admin', 'quiet-admin' ),
            'manage_network_options',
            'quiet-admin-network',
            [ $this, 'render_network_page' ]
        );
    }

    /* ---------- Saving ---------- */

    public function handle_save() {
        if ( ! current_user_can( 'manage_options' ) ) { wp_die( 'forbidden' ); }
        check_admin_referer( 'quietadmin_save_settings', 'quietadmin_save_nonce' );
        $posted_tab = isset( $_POST['quietadmin_active_tab'] )
            ? sanitize_key( wp_unslash( $_POST['quietadmin_active_tab'] ) )
            : $this->active_tab();
        $ids = [];
        foreach ( $this->features as $f ) {
            if ( method_exists( $f, 'id' ) ) { $ids[] = $f->id(); }
        }

        $src = [];
        foreach ( $ids as $id ) {
            $key = 'quietadmin_' . $id;

            if ( isset( $_POST[ $key ] ) && is_array( $_POST[ $key ] ) ) {
                $src[ $key ] = map_deep( wp_unslash( $_POST[ $key ] ), 'sanitize_text_field' );
            }
        }

        foreach ( $this->features as $f ) {
            if ( method_exists( $f, 'save_from_request' ) ) {
                $f->save_from_request( $src );
            }
        }

        add_settings_error( 'quiet-admin', 'qa_saved', __( 'Settings saved.', 'quiet-admin' ), 'updated' );
        set_transient( 'settings_errors', get_settings_errors(), 30 );

        $posted_tab = isset( $_POST['quietadmin_active_tab'] ) ? sanitize_key( wp_unslash( $_POST['quietadmin_active_tab'] ) ) : $this->active_tab();
        if ( ! in_array( $posted_tab, $ids, true ) ) { $posted_tab = $this->active_tab(); }

        $url = add_query_arg( [ 'page' => 'quiet-admin', 'tab' => $posted_tab ], admin_url( 'options-general.php' ) );
        wp_safe_redirect( $url );
        exit;
    }

    private function active_tab() : string {
        $default = '';
        foreach ( $this->features as $f ) { if ( method_exists( $f, 'id' ) ) { $default = $f->id(); break; } }

        $tab = filter_input( INPUT_GET, 'tab', FILTER_SANITIZE_STRING );
        $tab = $tab ? sanitize_key( $tab ) : $default;

        $ids = [];
        foreach ( $this->features as $f ) { if ( method_exists( $f, 'id' ) ) { $ids[] = $f->id(); } }
        if ( ! in_array( $tab, $ids, true ) ) { $tab = $default; }

        return $tab;
    }

    public function render_page() {
        if ( ! current_user_can( 'manage_options' ) ) { return; }

        $base_url   = menu_page_url( 'quiet-admin', false );
        if ( ! $base_url ) { $base_url = admin_url( 'options-general.php?page=quiet-admin' ); }
        $active_tab = $this->active_tab();

        $active_feature = null;
        foreach ( $this->features as $f ) {
            if ( method_exists( $f, 'id' ) && $f->id() === $active_tab ) { $active_feature = $f; break; }
        }
        ?>
        <div class="wrap qa-settings">
            <h1>Quiet Admin</h1>

            <?php settings_errors( 'quiet-admin' ); ?>

            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'quietadmin_save_settings', 'quietadmin_save_nonce' ); ?>
                <input type="hidden" name="action" value="quietadmin_save_settings"/>
                <input type="hidden" name="quietadmin_active_tab" value="<?php echo esc_attr( $active_tab ); ?>"/>

                <div class="qa-tabs">
                    <nav class="qa-tabs__nav" aria-label="<?php echo esc_attr__( 'Quiet Admin sections', 'quiet-admin' ); ?>">
                        <?php foreach ( $this->features as $f ) :
                            if ( ! method_exists( $f, 'id' ) || ! method_exists( $f, 'title' ) ) { continue; }
                            $id   = $f->id();
                            $href = add_query_arg( 'tab', $id, $base_url ); ?>
                            <a href="<?php echo esc_url( $href ); ?>" class="qa-tab <?php echo $active_tab === $id ? 'is-active' : ''; ?>">
                                <?php echo esc_html( $f->title() ); ?>
                            </a>
                        <?php endforeach; ?>
                    </nav>

                    <section class="qa-tabpanel is-active" data-tabpanel="<?php echo esc_attr( $active_tab ); ?>">
                        <div class="qa-card">
                            <?php
                            if ( $active_feature && method_exists( $active_feature, 'render_settings' ) ) {
                                $active_feature->render_settings();
                            } else {
                                echo '<p>'.esc_html__( 'Nothing to show.', 'quiet-admin' ).'</p>';
                            }
                            ?>
                        </div>
                    </section>
                </div>

                <?php submit_button( __( 'Save Changes', 'quiet-admin' ) ); ?>
            </form>
        </div>
        <?php
    }

    public function render_network_page() {
        if ( ! current_user_can( 'manage_network_options' ) ) { return; }
        echo '<div class="wrap"><h1>'.esc_html__('Quiet Admin (Network Defaults)','quiet-admin').'</h1>';
        echo '<p class="description">'.esc_html__('Network defaults can be added per feature in future iterations.','quiet-admin').'</p></div>';
    }

    public function handle_network_save() {
        if ( ! current_user_can( 'manage_network_options' ) ) { wp_die('forbidden'); }
        wp_safe_redirect( network_admin_url( 'admin.php?page=quiet-admin-network&updated=1' ) );
        exit;
    }
}
