<?php
/**
 * Plugin Name: QuickShop AI for ChatGPT
 * Description: Sell your WooCommerce products in ChatGPT
 * Version: 1.0.3
 * Author: webmaster85
 * Text Domain: quickshop-ai-for-chatgpt
 * Requires at least: 6.2
 * Requires PHP: 8.0
 * Requires Plugins: woocommerce
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined('ABSPATH') ) exit;

define('ICGPT_VERSION', '1.0.0');
define('ICGPT_DIR', plugin_dir_path(__FILE__));
define('ICGPT_URL', plugin_dir_url(__FILE__));
define('ICGPT_LOG_TABLE', 'icgpt_logs');

final class ICGPT_Plugin {
    public function __construct() {
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
        add_action('plugins_loaded', [$this, 'init']);
    }
    public function activate() {      
	    if ( ! class_exists( 'WooCommerce' ) ) {
        deactivate_plugins( plugin_basename( __FILE__ ) );      
        wp_die(
            esc_html__( 'WooCommerce is required for QuickShop AI for ChatGPT.', 'quickshop-ai-for-chatgpt' ),
            esc_html__( 'Plugin dependency missing', 'quickshop-ai-for-chatgpt' ),
            [ 'response' => 500 ]
        );
		}	  
        add_option('icgpt_enable', 0);
        add_option('icgpt_merchant_id', '');
        add_option('icgpt_webhook_secret', wp_generate_password(32, false));
        add_option('icgpt_default_enable_checkout', 0);
        add_option('icgpt_test_mode', 1);    
        global $wpdb;
        $table = $wpdb->prefix . ICGPT_LOG_TABLE;
        $charset = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE IF NOT EXISTS `$table` (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            created_at DATETIME NOT NULL,
            level VARCHAR(20) NOT NULL,
            message TEXT NOT NULL,
            context LONGTEXT NULL,
            PRIMARY KEY (id),
            KEY level_idx (level),
            KEY created_at_idx (created_at)
        ) $charset;";
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }
    public function deactivate() {     
    }
    public function init() {   
        require_once ICGPT_DIR . 'includes/class-icgpt-logger.php';
        require_once ICGPT_DIR . 'includes/class-icgpt-admin-assets.php';
        require_once ICGPT_DIR . 'includes/class-icgpt-settings.php';
        require_once ICGPT_DIR . 'includes/class-icgpt-product-flags.php';
        require_once ICGPT_DIR . 'includes/class-icgpt-validator.php';
        require_once ICGPT_DIR . 'includes/class-icgpt-feed.php';
        require_once ICGPT_DIR . 'includes/class-icgpt-webhooks.php';
        require_once ICGPT_DIR . 'includes/class-icgpt-onboarding.php';
        new ICGPT_Admin_Assets();
        new ICGPT_Settings();
        new ICGPT_Product_Flags();
        new ICGPT_Feed();
        new ICGPT_Webhooks();
        new ICGPT_Onboarding();  
        add_action('admin_post_icgpt_rotate_secret', [$this, 'rotate_secret']);
        add_action('admin_post_icgpt_send_test_order', [$this, 'send_test_order']);
    }

  public function rotate_secret() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {        
            wp_die(
                esc_html__( 'Forbidden', 'quickshop-ai-for-chatgpt' ),
                esc_html__( 'Error', 'quickshop-ai-for-chatgpt' ),
                [ 'response' => 403 ]
            );
        }
        check_admin_referer( 'icgpt_rotate_secret' );
        update_option( 'icgpt_webhook_secret', wp_generate_password( 32, false ) );
        ICGPT_Logger::info( 'Webhook secret rotated by admin.' );
        wp_safe_redirect( wp_get_referer() ?: admin_url( 'admin.php?page=icgpt-settings' ) );
        exit;
    }

    public function send_test_order() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
        
            wp_die(
                esc_html__( 'Forbidden', 'quickshop-ai-for-chatgpt' ),
                esc_html__( 'Error', 'quickshop-ai-for-chatgpt' ),
                [ 'response' => 403 ]
            );
        }
        check_admin_referer( 'icgpt_send_test' );   
        $payload = [
            'id'             => 'test_evt_' . time(),
            'transaction_id' => 'txn_' . wp_rand( 10000, 99999 ),
            'line_items'     => [      
            ],
            'shipping_total' => 0,
            'tax_total'      => 0,
            'customer'       => [
                'first_name' => 'Chat',
                'last_name'  => 'GPT',
                'email'      => 'buyer@example.com',
                'phone'      => '',
                'billing'    => [
                    'address_1' => '123 Test St',
                    'address_2' => '',
                    'city'      => 'Testville',
                    'state'     => 'TS',
                    'postcode'  => '00000',
                    'country'   => 'US',
                ],
                'shipping'   => [
                    'address_1' => '123 Test St',
                    'address_2' => '',
                    'city'      => 'Testville',
                    'state'     => 'TS',
                    'postcode'  => '00000',
                    'country'   => 'US',
                ],
            ],
            'test_mode'      => true, 
        ];
        $args = [ 'status' => 'publish', 'limit' => 1, 'return' => 'ids' ];
        $ids  = wc_get_products( $args );
        if ( ! empty( $ids ) ) {
            $payload['line_items'][] = [ 'product_id' => (int) $ids[0], 'quantity' => 1 ];
        }

        $body = wp_json_encode( $payload );
        $sig  = base64_encode( hash_hmac( 'sha256', $body, get_option( 'icgpt_webhook_secret' ), true ) );

        $resp = wp_remote_post(
            rest_url( 'icgpt/v1/webhook/order' ),
            [
                'headers' => [
                    'x-icgpt-signature' => $sig,
					 'x-icgpt-local-test' => '1', 
                    'Content-Type'      => 'application/json',
                ],
                'body'    => $body,
                'timeout' => 20,
            ]
        );

        if ( is_wp_error( $resp ) ) {
            ICGPT_Logger::error( 'Test webhook failed', [ 'error' => $resp->get_error_message() ] );
        } else {
            ICGPT_Logger::info( 'Test webhook sent', [ 'response' => wp_remote_retrieve_body( $resp ) ] );
        }

        wp_safe_redirect( wp_get_referer() ?: admin_url( 'admin.php?page=icgpt-settings' ) );
        exit;
    }
}
new ICGPT_Plugin();