<?php
if ( ! defined('ABSPATH') ) exit;

class ICGPT_Webhooks {
    public function __construct() {		
		add_action( 'rest_api_init', function () {
			register_rest_route(
				'icgpt/v1',
				'/webhook/order',
				[
					'methods'  => 'POST',
					'callback' => [ $this, 'handle_order' ],			
					'permission_callback' => function ( WP_REST_Request $request ) {
						$enabled   = (bool) get_option( 'icgpt_enable' );
						$test_mode = (bool) get_option( 'icgpt_test_mode' );
						$secret    = (string) get_option( 'icgpt_webhook_secret', '' );				
						$is_local_test = $test_mode && '1' === $request->get_header( 'x-icgpt-local-test' );
						if ( $is_local_test ) {
							return true;
						}					
						if ( ! $enabled || '' === $secret ) {
							return false;
						}
						$sig_raw = $request->get_header( 'x-icgpt-signature' );
						return is_string( $sig_raw ) && '' !== $sig_raw;
					},
				]
			);
		} );		
    }
		private function verify_signature( WP_REST_Request $req ) {
			$secret = get_option( 'icgpt_webhook_secret', '' );
			$sig_raw = $req->get_header( 'x-icgpt-signature' );
			$sig     = is_string( $sig_raw ) ? sanitize_text_field( wp_unslash( $sig_raw ) ) : '';	
			$body_raw = $req->get_body();
			$body     = is_string( $body_raw ) ? $body_raw : '';

			if ( ! $secret || ! $sig ) {
				return false;
			}

			$calc = base64_encode( hash_hmac( 'sha256', $body, $secret, true ) );
			return hash_equals( $calc, $sig );
		}	
	private function clean_text($v) {
    return mb_substr( sanitize_text_field( (string) $v ), 0, 190 );
	}

  public function handle_order( WP_REST_Request $req ) {
    $body_raw = $req->get_body();
    $body     = is_string( $body_raw ) ? wp_unslash( $body_raw ) : '';

    if ( ! $this->verify_signature( $req ) ) {     
        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        ICGPT_Logger::warning( 'Webhook bad signature', [ 'ip' => $ip ] );
        return new WP_REST_Response( [ 'ok' => false, 'error' => 'bad_signature' ], 401 );
    }

    $payload = json_decode( $body, true );
    if ( json_last_error() !== JSON_ERROR_NONE ) {
        ICGPT_Logger::warning( 'Webhook bad json', [ 'err' => json_last_error_msg() ] );
        return new WP_REST_Response( [ 'ok' => false, 'error' => 'bad_json' ], 400 );
    }

    if ( empty( $payload['id'] ) || empty( $payload['line_items'] ) || ! is_array( $payload['line_items'] ) ) {
        ICGPT_Logger::warning( 'Webhook bad payload' );
        return new WP_REST_Response( [ 'ok' => false, 'error' => 'bad_payload' ], 400 );
    }

    try {
        $order = wc_create_order();

        foreach ( $payload['line_items'] as $li ) {
            $product_id = absint( $li['product_id'] ?? 0 );
            if ( ! $product_id ) {
                continue;
            }
            $qty     = max( 1, (int) ( $li['quantity'] ?? 1 ) );
            $product = wc_get_product( $product_id );
            if ( $product ) {
                $order->add_product( $product, $qty );
            }
        }

        if ( ! empty( $payload['shipping_total'] ) ) {
            $order->set_shipping_total( (float) $payload['shipping_total'] );
        }
        if ( ! empty( $payload['tax_total'] ) ) {
            $order->set_cart_tax( (float) $payload['tax_total'] );
        }

        if ( ! empty( $payload['customer'] ) && is_array( $payload['customer'] ) ) {
            $c = $payload['customer'];

            $order->set_billing_first_name( $this->clean_text( $c['first_name'] ?? '' ) );
            $order->set_billing_last_name(  $this->clean_text( $c['last_name'] ?? '' ) );
            $order->set_billing_email(      sanitize_email( $c['email'] ?? '' ) );
            $order->set_billing_phone(      $this->clean_text( $c['phone'] ?? '' ) );

            $bill = is_array( $c['billing']  ?? null ) ? $c['billing']  : [];
            $ship = is_array( $c['shipping'] ?? null ) ? $c['shipping'] : [];

            foreach ( [ 'address_1','address_2','city','state','postcode','country' ] as $f ) {
                $bf = $this->clean_text( $bill[ $f ] ?? '' );
                $sf = $this->clean_text( $ship[ $f ] ?? ( $bill[ $f ] ?? '' ) );
                $order->{"set_billing_$f"}( $bf );
                $order->{"set_shipping_$f"}( $sf );
            }
        }

        $order->calculate_totals();
        $raw_txn = ! empty( $payload['transaction_id'] ) ? (string) $payload['transaction_id'] : ( 'chatgpt_' . sanitize_key( (string) $payload['id'] ) );
        $txn_id  = sanitize_text_field( wp_unslash( $raw_txn ) );
        $order->payment_complete( $txn_id );
        $order->set_payment_method_title( 'ChatGPT Instant Checkout' );
        $order->add_order_note( 'Order created via ChatGPT Instant Checkout (AI).' );

        wc_reduce_stock_levels( $order->get_id() );
        $order->save();

        ICGPT_Logger::info( 'Webhook order created', [ 'order_id' => $order->get_id(), 'payload_id' => sanitize_text_field( (string) $payload['id'] ) ] );
        return new WP_REST_Response( [ 'ok' => true, 'order_id' => $order->get_id() ], 200 );

    } catch ( \Throwable $e ) {
        ICGPT_Logger::error( 'Webhook exception', [ 'error' => $e->getMessage() ] );
        return new WP_REST_Response( [ 'ok' => false, 'error' => 'server_error' ], 500 );
    }
}  
}