<?php
if ( ! defined('ABSPATH') ) exit;

class ICGPT_Feed {
    public function __construct() {
		add_action( 'rest_api_init', function () {
			register_rest_route(
				'icgpt/v1',
				'/feed',
				[
					'methods'             => 'GET',
					'callback'            => [ $this, 'feed' ],			
					'permission_callback' => function () {
						return (bool) get_option( 'icgpt_enable' );
					},
				]
			);
		} );
        add_action('save_post_product', [$this, 'log_feed_refresh']);
        add_action('woocommerce_product_set_stock', [$this, 'log_feed_refresh']);
    }

    public function log_feed_refresh() {
        ICGPT_Logger::info('Feed data changed (product/stock updated).');
    }

    private function product_to_record(WC_Product $p) {
        $id     = $p->get_id();
        $enable_search   = (bool) get_post_meta($id, ICGPT_Product_Flags::META_ENABLE_SEARCH, true);
        $enable_checkout = (bool) get_post_meta($id, ICGPT_Product_Flags::META_ENABLE_CHECKOUT, true);
        if ($enable_checkout && ! $enable_search) $enable_checkout = false;

        $img  = wp_get_attachment_image_src($p->get_image_id(), 'full');
        $price= wc_get_price_including_tax($p);
        $currency = get_woocommerce_currency();
        $stock= $p->get_stock_status() === 'instock' ? 'in_stock' : 'out_of_stock';

        $desc = $p->get_description() ?: $p->get_short_description();

        return [
            'enable_search'   => ICGPT_Validator::bool_string($enable_search),
            'enable_checkout' => ICGPT_Validator::bool_string($enable_checkout),

            'id'          => (string)$id,
            'title'       => ICGPT_Validator::str($p->get_name(), 300),
            'description' => ICGPT_Validator::str($desc, 3000),
            'link'        => get_permalink($id),

            'image_link'   => $img ? $img[0] : '',
            'price'        => ICGPT_Validator::price_format($price, $currency),
            'availability' => ICGPT_Validator::availability($stock),
            'inventory_quantity'=> max(0, (int) $p->get_stock_quantity()),
            'item_group_id'=> $p->is_type('variation') ? (string)$p->get_parent_id() : (string)$id,
        ];
    }

    public function feed(WP_REST_Request $req) {
        if ( ! get_option('icgpt_enable') ) {
            return new WP_REST_Response(['enabled'=>false,'message'=>'Integration disabled'], 200);
        }
        $currency = get_woocommerce_currency();
        if ( ! ICGPT_Validator::currency_code_is_valid($currency) ) {
            return new WP_REST_Response(['enabled'=>true,'error'=>'invalid_currency'], 500);
        }

        $args = ['status'=>'publish','limit'=>500,'return'=>'objects'];
        $products = wc_get_products($args);
        $records = array_map([$this, 'product_to_record'], $products);

        $payload = [
            'merchant_id' => get_option('icgpt_merchant_id', ''),
            'generated_at'=> gmdate('c'),
            'currency'    => $currency,
            'records'     => $records
        ];

        ICGPT_Logger::info('Feed served', ['count'=>count($records)]);
        return new WP_REST_Response($payload, 200);
    }
}