<?php
/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @package    Quick_Chat_Buttons
 * @subpackage Quick_Chat_Buttons/includes
 * @author     Klaxon.app <contact@klaxon.app>
 * @license    GPL2
 * @link       https://klaxon.app
 * @since      1.0.0
 */
defined('ABSPATH') or die('Direct Access is not allowed');
class Quick_Chat_Buttons
{

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @var    Quick_Chat_Buttons_Loader    $loader    Maintains and registers all hooks for the plugin.
     * @since  1.0.0
     * @access protected
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @var    string    $pluginName    The string used to uniquely identify this plugin.
     * @since  1.0.0
     * @access protected
     */
    protected $pluginName;

    /**
     * The current version of the plugin.
     *
     * @var    string    $version    The current version of the plugin.
     * @since  1.0.0
     * @access protected
     */
    protected $version;


    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     */
    public function __construct()
    {
        if (defined('QUICK_CHAT_BUTTONS_VERSION')) {
            $this->version = QUICK_CHAT_BUTTONS_VERSION;
        } else {
            $this->version = '1.0.0';
        }

        $this->pluginName = 'quick-chat-buttons';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();

    }//end __construct()


    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Quick_Chat_Buttons_Loader. Orchestrates the hooks of the plugin.
     * - Quick_Chat_Buttons_i18n. Defines internationalization functionality.
     * - Quick_Chat_Buttons_Admin. Defines all hooks for the admin area.
     * - Quick_Chat_Buttons_Public. Defines all hooks for the public side of the site.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since  1.0.0
     * @access private
     */
    private function load_dependencies()
    {
        /*
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */

        include_once plugin_dir_path(dirname(__FILE__)).'includes/class-quick-chat-buttons-loader.php';

        /*
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */

        include_once plugin_dir_path(dirname(__FILE__)).'includes/class-quick-chat-buttons-i18n.php';

        /*
         * The class responsible for defining all actions that occur in the admin area.
         */

        include_once plugin_dir_path(dirname(__FILE__)).'admin/class-quick-chat-buttons-admin.php';

        /*
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */

        include_once plugin_dir_path(dirname(__FILE__)).'public/class-quick-chat-buttons-public.php';

        $this->loader = new Quick_Chat_Buttons_Loader();

    }//end load_dependencies()


    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Quick_Chat_Buttons_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since  1.0.0
     * @access private
     */
    private function set_locale()
    {

        $pluginI18n = new Quick_Chat_Buttons_I18n();

        $this->loader->add_action('plugins_loaded', $pluginI18n, 'load_plugin_textdomain');

    }//end set_locale()


    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since  1.0.0
     * @access private
     */
    private function define_admin_hooks()
    {

        $pluginAdmin = new Quick_Chat_Buttons_Admin($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('admin_enqueue_scripts', $pluginAdmin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $pluginAdmin, 'enqueue_scripts');
        $this->loader->add_action('admin_menu', $pluginAdmin, 'admin_menu');
        $this->loader->add_action('save_qc_buttons_setting', $pluginAdmin, 'admin_menu');
        $this->loader->add_filter('plugin_action_links_'.QCB_PLUGIN_BASE, $pluginAdmin, 'setting_and_upgrade_link');
        $this->loader->add_action('wp_ajax_save_qc_buttons_setting', $pluginAdmin, "save_qc_buttons_setting");
        $this->loader->add_action('wp_ajax_get_qcb_settings', $pluginAdmin, "get_qcb_settings");
        $this->loader->add_action('wp_ajax_remove_qcb_widget', $pluginAdmin, "remove_qcb_widget");
        $this->loader->add_action('wp_ajax_save_qcb_widget_title', $pluginAdmin, "save_qcb_widget_title");
        $this->loader->add_action('wp_ajax_qcb_change_widget_status', $pluginAdmin, "qcb_change_widget_status");
        $this->loader->add_action('in_admin_header', $pluginAdmin, "in_admin_header");

    }//end define_admin_hooks()


    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since  1.0.0
     * @access private
     */
    private function define_public_hooks()
    {

        $pluginPublic = new Quick_Chat_Buttons_Public($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('wp_enqueue_scripts', $pluginPublic, 'enqueue_styles');
        $this->loader->add_action('wp_enqueue_scripts', $pluginPublic, 'enqueue_scripts');

    }//end define_public_hooks()


    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since 1.0.0
     */
    public function run()
    {
        $this->loader->run();

    }//end run()


    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since  1.0.0
     * @return string    The name of the plugin.
     */
    public function get_plugin_name()
    {
        return $this->pluginName;

    }//end get_plugin_name()


    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since  1.0.0
     * @return Quick_Chat_Buttons_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader()
    {
        return $this->loader;

    }//end get_loader()


    /**
     * Retrieve the version number of the plugin.
     *
     * @since  1.0.0
     * @return string    The version number of the plugin.
     */
    public function get_version()
    {
        return $this->version;

    }//end get_version()


    /**
     * Retrieve the version number of the plugin.
     *
     * @since  1.0.0
     * @return string    The version number of the plugin.
     */
    public static function get_buttons()
    {
        return [
            "whatsapp" => [
                'label' => 'Phone Number',
                'title' => 'WhatsApp',
                'example' => '+7485632541',
                'bg_color' => '#4dc247',
                'icon_color' => '#ffffff',
                'icon' => '<svg viewBox="0 0 56.693 56.693" xml:space="preserve" xmlns="http://www.w3.org/2000/svg"><path class="st0" d="m46.38 10.714c-4.6512-4.6565-10.836-7.222-17.427-7.2247-13.578 0-24.63 11.051-24.635 24.633-0.0019 4.342 1.1325 8.58 3.2884 12.316l-3.495 12.766 13.06-3.4257c3.5982 1.9626 7.6495 2.9971 11.773 2.9985h0.01 2e-4c13.577 0 24.629-11.052 24.635-24.635 0.0024-6.5826-2.5577-12.772-7.2088-17.428zm-17.426 37.902h-0.0083c-3.674-0.0014-7.2777-0.9886-10.422-2.8541l-0.7476-0.4437-7.7497 2.0328 2.0686-7.5558-0.4869-0.7748c-2.0496-3.26-3.1321-7.028-3.1305-10.897 0.0044-11.289 9.19-20.474 20.484-20.474 5.469 0.0017 10.61 2.1344 14.476 6.0047 3.8658 3.8703 5.9936 9.0148 5.9914 14.486-0.0046 11.29-9.1899 20.476-20.476 20.476z"/><path class="st0" d="m40.185 33.281c-0.6155-0.3081-3.6419-1.797-4.2061-2.0026-0.5642-0.2054-0.9746-0.3081-1.3849 0.3081-0.4103 0.6161-1.59 2.0027-1.9491 2.4136-0.359 0.4106-0.7182 0.4623-1.3336 0.1539-0.6155-0.3081-2.5989-0.958-4.95-3.0551-1.83-1.6323-3.0653-3.6479-3.4245-4.2643-0.359-0.6161-0.0382-0.9492 0.27-1.2562 0.2769-0.2759 0.6156-0.7189 0.9234-1.0784 0.3077-0.3593 0.4103-0.6163 0.6155-1.0268 0.2052-0.4109 0.1027-0.7704-0.0513-1.0784-0.1539-0.3081-1.3849-3.3379-1.8978-4.5706-0.4998-1.2001-1.0072-1.0375-1.3851-1.0566-0.3585-0.0179-0.7694-0.0216-1.1797-0.0216s-1.0773 0.1541-1.6414 0.7702c-0.5642 0.6163-2.1545 2.1056-2.1545 5.1351 0 3.0299 2.2057 5.9569 2.5135 6.3676 0.3077 0.411 4.3405 6.6282 10.515 9.2945 1.4686 0.6343 2.6152 1.013 3.5091 1.2966 1.4746 0.4686 2.8165 0.4024 3.8771 0.2439 1.1827-0.1767 3.6419-1.489 4.1548-2.9267 0.513-1.438 0.513-2.6706 0.359-2.9272-0.1538-0.2567-0.5642-0.4108-1.1797-0.719z"/></svg>',
                'status' => 1,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "facebook_messenger" => [
                'label' => 'Facebook Page ID',
                'title' => 'Facebook Messenger',
                'example' => '10542163254993',
                'bg_color' => '#0075FF',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" height="909.333" viewBox="-21 -28 682.667 682" width="909.333"><path d="M545.602 84.63C485.242 30 405.125-.082 320-.082S154.758 30 94.398 84.63C33.523 139.727 0 213.133 0 291.332c0 58.578 18.863 114.742 54.633 162.94L27.14 626.188 201.06 561.94c37.828 13.81 77.805 20.81 118.94 20.81 85.125 0 165.242-30.086 225.602-84.715C606.477 442.938 640 369.53 640 291.332S606.477 139.727 545.602 84.63zM348.047 375.027l-70.738-55.344-169.203 66.965L301.71 194.086l71.594 57.168 154.875-60.047zm0 0"/></svg>',
                'status' => 1,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "viber" => [
                'label' => 'Phone Number',
                'title' => 'Viber',
                'example' => '+7485632541',
                'bg_color' => '#774D99',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M11.398.002C9.473.028 5.33.344 3.014 2.467c-1.72 1.7-2.32 4.23-2.39 7.353-.06 3.1-.13 8.95 5.5 10.54v2.42s-.038.97.602 1.17c.8.25 1.24-.5 2-1.3l1.4-1.58c3.85.32 6.8-.42 7.14-.53.78-.25 5.18-.81 5.9-6.652.74-6.03-.36-9.83-2.34-11.55l-.01-.002c-.6-.55-3-2.3-8.37-2.32 0 0-.396-.025-1.038-.016zm.067 1.697c.545-.003.88.02.88.02 4.54.01 6.71 1.38 7.22 1.84 1.67 1.43 2.528 4.856 1.9 9.892-.6 4.88-4.17 5.2-4.83 5.4-.28.1-2.88.73-6.152.52 0 0-2.44 2.94-3.2 3.7-.12.13-.26.17-.35.15-.13-.03-.17-.2-.16-.4l.02-4.02c-4.77-1.32-4.49-6.302-4.44-8.902.06-2.6.55-4.732 2-6.172 1.957-1.77 5.475-2 7.1-2.02zm.36 2.6a.299.299 0 0 0-.3.299.3.3 0 0 0 .3.3 5.631 5.631 0 0 1 4.03 1.59c1.1 1.06 1.62 2.48 1.64 4.34a.3.3 0 0 0 .3.3v-.01a.3.3 0 0 0 .3-.3 6.451 6.451 0 0 0-1.81-4.76c-1.2-1.16-2.692-1.76-4.462-1.76zm-3.954.7a.955.955 0 0 0-.615.12h-.012c-.4.24-.788.54-1.148.94-.27.32-.42.64-.46.95a1.24 1.24 0 0 0 .05.541l.02.01a13.722 13.722 0 0 0 1.2 2.6 15.383 15.383 0 0 0 2.32 3.171l.03.04.04.03.06.06a15.603 15.603 0 0 0 3.18 2.33c1.32.72 2.122 1.06 2.602 1.2V17c.14.04.268.06.398.06a1.84 1.84 0 0 0 1.102-.472c.4-.35.7-.738.93-1.148v-.01c.23-.43.15-.84-.18-1.12a13.632 13.632 0 0 0-2.15-1.54c-.5-.28-1.03-.1-1.24.17l-.45.57c-.23.28-.65.24-.65.24l-.012.01c-3.12-.8-3.95-3.96-3.95-3.96s-.04-.43.25-.65l.56-.45c.27-.22.46-.74.17-1.25a13.522 13.522 0 0 0-1.54-2.15.843.843 0 0 0-.504-.3zm4.473.9a.3.3 0 0 0 .002.6 3.78 3.78 0 0 1 2.65 1.15 3.5 3.5 0 0 1 .9 2.57.3.3 0 0 0 .3.299l.01.012a.3.3 0 0 0 .3-.301c.03-1.2-.34-2.2-1.07-3s-1.75-1.25-3.05-1.34a.3.3 0 0 0-.042 0zm.5 1.62a.305.305 0 0 0-.018.611c1 .05 1.47.55 1.53 1.58a.3.3 0 0 0 .3.29h.01a.3.3 0 0 0 .29-.32c-.07-1.34-.8-2.09-2.1-2.16a.305.305 0 0 0-.012 0z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "line" => [
                'label' => 'URL',
                'title' => 'Line',
                'example' => 'https://line.me/R/ti/p/@username',
                'bg_color' => '#1AD14C',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M19.365 9.863a.63.63 0 1 1 0 1.261H17.6v1.125h1.755a.63.63 0 0 1 .63.63.63.63 0 0 1-.63.629H16.98a.63.63 0 0 1-.627-.629v-4.77a.63.63 0 0 1 .63-.63h2.386a.63.63 0 1 1-.003 1.26H17.6v1.125h1.755zM15.5 12.88a.63.63 0 0 1-.432.596c-.064.02-.133.03-.2.03-.21 0-.39-.1-.5-.25L11.926 9.94v2.94a.63.63 0 0 1-.631.629.63.63 0 0 1-.626-.629v-4.77c0-.27.173-.5.43-.595a.58.58 0 0 1 .194-.033c.195 0 .375.104.495.254l2.462 3.33V8.108a.63.63 0 0 1 .63-.63c.345 0 .63.285.63.63v4.77zm-5.74 0a.63.63 0 0 1-.631.629.63.63 0 0 1-.627-.629v-4.77a.63.63 0 0 1 .63-.63.63.63 0 0 1 .628.63v4.77zm-2.466.63H4.917c-.345 0-.63-.285-.63-.63v-4.77c0-.345.285-.63.63-.63a.63.63 0 0 1 .63.63v4.14h1.756a.63.63 0 0 1 .629.63.63.63 0 0 1-.629.629M24 10.314c0-5.37-5.385-9.742-12-9.742S0 4.943 0 10.314c0 4.81 4.27 8.842 10.035 9.608.39.082.923.258 1.058.6.12.3.08.766.038 1.08l-.164 1.02c-.045.3-.24 1.186 1.05.645s6.916-4.078 9.436-6.975C23.176 14.393 24 12.458 24 10.314"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "mail" => [
                'label' => "Email Address",
                'title' => "Mail",
                'example' => "email@domain.com",
                'bg_color' => '#F36D23',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M339.392 258.624L512 367.744V144.896zM0 144.896v222.848l172.608-109.12zM480 80H32C16.032 80 3.36 91.904.96 107.232L256 275.264l255.04-168.032C508.64 91.904 495.968 80 480 80zM310.08 277.952l-45.28 29.824a15.98 15.98 0 0 1-17.6 0l-45.28-29.856L1.024 404.992C3.488 420.192 16.096 432 32 432h448c15.904 0 28.512-11.808 30.976-27.008L310.08 277.952z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "telegram" => [
                'label' => "Username",
                'title' => "Telegram",
                'example' => "myusername",
                'bg_color' => '#0088CC',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" height="512" viewBox="0 0 24 24" width="512"><path d="M9.417 15.18l-.397 5.584c.568 0 .814-.244 1.11-.537l2.663-2.545 5.518 4.04c1.012.564 1.725.267 1.998-.93L23.93 3.82h.001c.32-1.496-.54-2.08-1.527-1.714l-21.3 8.15c-1.453.564-1.43 1.374-.247 1.74L6.3 13.69l12.643-7.91c.595-.394 1.136-.176.69.218z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "vkontakte" => [
                'label' => "Username",
                'title' => "Vkontakte",
                'example' => "myusername",
                'bg_color' => '#45668E',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M20 13c.24-.45 1.1-1.74 1.73-2.7C23.65 7.46 24 6.86 24 6.5a.5.5 0 0 0-.5-.5H19a.5.5 0 0 0-.49.41c-.25 1.38-3.5 5.34-4 5.6-.2 0-.5-.52-.5-1.5V6.28a1.18 1.18 0 0 0-.24-.93C13.43 5 12.92 5 11.5 5 8.92 5 8 5.77 8 6.5a.46.46 0 0 0 .45.5S9 7.36 9 9.5a14.61 14.61 0 0 1-.13 2.5C7.6 11.77 5.84 8.6 5 6.32A.5.5 0 0 0 4.5 6h-4a.5.5 0 0 0-.5.59C.56 9.6 6.9 19 11 19h2c1.06 0 1.14-1.15 1.2-1.9s.1-1.1.3-1.1c.62 0 1.9 1.1 2.72 1.82S18.6 19 19 19h2.5c1.3 0 2.5 0 2.5-1 0-.38-.33-.82-2.23-3-.63-.7-1.48-1.64-1.77-2z" /></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "sms" => [
                'label' => "Phone Number",
                'title' => "SMS",
                'example' => "+7485632541",
                'bg_color' => '#FF9F00',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 495.968 495.968"><path d="M423.984 0h-352c-35.296 0-64 28.704-64 64v288c0 35.296 28.704 64 64 64v63.968a16.02 16.02 0 0 0 9.216 14.496c2.144.992 4.48 1.504 6.784 1.504 3.68 0 7.328-1.248 10.24-3.712L189.776 416h234.208c35.296 0 64-28.704 64-64V64c0-35.296-28.704-64-64-64zm-304 192c26.464 0 48 21.536 48 48s-21.536 48-48 48h-32a16.01 16.01 0 0 1-16-16 16.01 16.01 0 0 1 16-16h32a16.01 16.01 0 0 0 16-16 16.01 16.01 0 0 0-16-16c-26.464 0-48-21.536-48-48s21.536-48 48-48h32a16.01 16.01 0 0 1 16 16 16.01 16.01 0 0 1-16 16h-32a16.01 16.01 0 0 0-16 16 16.01 16.01 0 0 0 16 16zm176 80a16.01 16.01 0 0 1-16 16 16.01 16.01 0 0 1-16-16v-60.224l-1.696 3.392c-5.44 10.848-23.2 10.848-28.64 0l-1.664-3.392V272a16.01 16.01 0 0 1-16 16 16.01 16.01 0 0 1-16-16V144c0-7.424 5.12-13.888 12.32-15.584 7.136-1.728 14.656 1.792 17.984 8.416l17.696 35.392 17.696-35.36c3.296-6.688 10.624-10.144 17.984-8.448 7.232 1.696 12.32 8.16 12.32 15.584v128zm80-80c26.464 0 48 21.536 48 48s-21.536 48-48 48h-32a16.01 16.01 0 0 1-16-16 16.01 16.01 0 0 1 16-16h32c8.8 0 16-7.168 16-16s-7.2-16-16-16c-26.464 0-48-21.536-48-48s21.536-48 48-48h32a16.01 16.01 0 0 1 16 16 16.01 16.01 0 0 1-16 16h-32c-8.8 0-16 7.168-16 16s7.2 16 16 16z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "phone" => [
                'label' => "Phone Number",
                'title' => "Phone",
                'example' => "+7485632541",
                'bg_color' => '#00bb70',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 405.333 405.333"><path d="M373.333 266.88c-25.003 0-49.493-3.904-72.704-11.563-11.328-3.904-24.192-.896-31.637 6.7l-46.016 34.752c-52.8-28.18-86.592-61.952-114.39-114.368l33.813-44.928c8.512-8.512 11.563-20.97 7.915-32.64-7.723-23.36-11.648-47.872-11.648-72.832 0-17.643-14.357-32-32-32H32C14.357 0 0 14.357 0 32c0 205.845 167.488 373.333 373.333 373.333 17.643 0 32-14.357 32-32V298.88c0-17.643-14.357-32-32-32z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "wechat" => [
                'label' => "User ID",
                'title' => "WeChat",
                'example' => "User ID",
                'bg_color' => '#00C80F',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" height="512" viewBox="0 0 24 24" width="512"><path d="M12.82 9.618c-7.242 3.732-2.425 13.745 6.6 11.13.842.327 1.592.857 2.408 1.25-.21-.7-.436-1.412-.676-2.11 2.8-1.995 3.414-4.818 2.38-7.138-1.616-3.677-6.776-5.183-10.72-3.133zm2.53 3.658c-.21.655-1.156.85-1.615.353-.506-.46-.31-1.424.355-1.63.734-.3 1.582.54 1.26 1.277zm4.78.094h.014c-.257.587-1.14.725-1.575.27-.21-.192-.27-.48-.344-.733.104-.46.42-.937.93-.96.705-.098 1.336.776.975 1.422z"/><path d="M17.414 8.248c-.436-2.144-1.936-3.955-3.824-5h.027v-.001C6.917-.54-1.425 4.742.187 10.97c.433 1.848 1.71 3.397 3.262 4.43-.3.853-.585 1.706-.855 2.565L5.52 16.4c1.17.377 2.415.56 3.66.52-1.538-4.412 2.407-9.086 8.234-8.67zm-6.077-2.56c.785-.316 1.713.345 1.65 1.2L13 6.89c.008.965-1.275 1.567-1.995.913-.747-.538-.535-1.845.342-2.115zM6.932 7.134c-.172.838-1.29 1.243-1.946.68-.76-.537-.546-1.868.345-2.14.873-.338 1.865.552 1.6 1.46z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "skype" => [
                'label' => "Username",
                'title' => "Skype",
                'example' => "myusername",
                'bg_color' => '#1b96d2',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" height="512" viewBox="0 0 24 24" width="512"><path d="M23.31 14.547c1.738-7.8-5.104-14.905-13.14-13.543C5.808-1.703 0 1.356 0 6.546c0 1.207.333 2.337.912 3.31-1.615 7.828 5.283 14.82 13.31 13.366 5.675 3 11.946-2.984 9.086-8.676zm-7.638 4.7c-2.108.867-5.577.872-7.676-.227-2.993-1.596-3.525-5.19-.943-5.19 1.946 0 1.33 2.27 3.295 3.194.902.417 2.84.46 3.968-.3 1.113-.745 1.01-1.917.406-2.477-1.603-1.48-6.2-.892-8.287-3.483-.91-1.124-1.083-3.107.037-4.545 1.952-2.512 7.68-2.665 10.143-.768 2.274 1.76 1.66 4.096-.175 4.096-2.207 0-1.047-2.888-4.6-2.888-2.583 0-3.6 1.837-1.78 2.73 2.466 1.225 8.75.816 8.75 5.603-.005 1.992-1.226 3.477-3.128 4.253z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "snapchat" => [
                'label' => "Username",
                'title' => "Snapchat",
                'example' => "myusername",
                'bg_color' => '#FFFC00',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 511.976 511.976"><path d="M501.94 374.42c-68.448-11.264-99.296-81.376-100.544-84.352-.096-.192-.16-.384-.288-.576-3.488-7.104-4.32-12.96-2.4-17.408 3.744-8.768 18.08-13.312 27.552-16.32 2.752-.864 5.344-1.696 7.488-2.528 9.344-3.68 16-7.68 20.416-12.192 5.248-5.408 6.272-10.592 6.208-13.952-.16-8.192-6.432-15.488-16.448-19.04-3.36-1.376-7.232-2.112-11.136-2.112-2.656 0-6.624.352-10.368 2.112-8 3.744-15.168 5.792-21.344 6.08-3.264-.16-5.568-.864-7.072-1.536l.704-11.36c2.208-34.656 4.928-77.76-6.528-103.36-7.84-17.536-18.368-32.608-31.328-44.832-10.848-10.24-23.456-18.56-37.472-24.704-24.352-10.752-47.072-12.352-58.464-12.352h-1.28l-8.8.096c-11.36 0-34.112 1.6-58.528 12.352-13.984 6.144-26.528 14.432-37.344 24.672-12.896 12.224-23.392 27.264-31.2 44.736-11.456 25.568-8.736 68.672-6.56 103.296v.064l.704 11.36c-1.664.736-4.384 1.568-8.384 1.568-6.464 0-14.144-2.048-22.816-6.112-2.56-1.184-5.472-1.792-8.672-1.792-5.152 0-10.624 1.504-15.328 4.288-5.952 3.488-9.792 8.384-10.848 13.856-.672 3.584-.64 10.72 7.296 17.952 4.384 3.968 10.784 7.648 19.072 10.912 2.176.864 4.768 1.664 7.488 2.528 9.472 3.008 23.84 7.552 27.584 16.32 1.888 4.448 1.056 10.304-2.432 17.408-.096.192-.192.384-.256.576-.896 2.048-9.056 20.448-25.792 39.872-9.504 11.04-19.968 20.288-31.104 27.456-13.6 8.768-28.288 14.464-43.68 16.992-6.048.992-10.368 6.368-10.016 12.48.096 1.76.512 3.52 1.248 5.216v.032c2.464 5.728 8.128 10.56 17.344 14.816 11.264 5.216 28.096 9.6 50.08 13.024 1.12 2.112 2.272 7.392 3.072 11.008.832 3.84 1.696 7.808 2.944 12 1.344 4.544 4.8 9.984 13.728 9.984 3.392 0 7.264-.768 11.776-1.632 6.592-1.28 15.616-3.04 26.848-3.04 6.24 0 12.704.544 19.2 1.632 12.544 2.08 23.36 9.696 35.872 18.528 18.304 12.928 39.04 27.552 70.72 27.552.864 0 1.728-.032 2.592-.096a62.21 62.21 0 0 0 3.68.096c31.68 0 52.416-14.624 70.72-27.52l.032-.032c12.512-8.8 23.328-16.416 35.872-18.496 6.496-1.088 12.96-1.632 19.2-1.632 10.752 0 19.264 1.376 26.848 2.848 4.96.96 8.8 1.44 11.776 1.44h.576c6.528 0 11.328-3.584 13.152-9.824 1.216-4.128 2.08-7.968 2.944-11.904.736-3.392 1.952-8.832 3.04-10.944 21.984-3.456 38.816-7.808 50.08-13.024 9.184-4.256 14.848-9.088 17.312-14.784.736-1.696 1.184-3.456 1.28-5.28.352-6.048-3.968-11.456-10.016-12.416z"/></svg>',
                'status' => 0,
                'value' => '',
            ],
            "linkedin" => [
                'label' => "Username",
                'title' => "Linkedin",
                'example' => "https://linkedin.com/*",
                'bg_color' => '#0074B2',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" height="512" viewBox="0 0 24 24" width="512"><path d="M23.994 24H24v-8.802c0-4.306-.927-7.623-5.96-7.623-2.42 0-4.044 1.328-4.707 2.587h-.07V7.976H8.5V24h4.97v-7.934c0-2.1.396-4.1 2.983-4.1 2.55 0 2.587 2.384 2.587 4.243V24zM.396 7.977h4.976V24H.396zM2.882 0A2.88 2.88 0 0 0 0 2.882c0 1.592 1.3 2.9 2.882 2.9s2.882-1.318 2.882-2.9S4.472 0 2.882 0z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "twitter" => [
                'label' => "Username",
                'title' => "Twitter",
                'example' => "https://twitter.com/*",
                'bg_color' => '#65BBF2',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M512 97.248c-19.04 8.352-39.328 13.888-60.48 16.576 21.76-12.992 38.368-33.408 46.176-58.016-20.288 12.096-42.688 20.64-66.56 25.408C411.872 60.704 384.416 48 354.464 48c-58.112 0-104.896 47.168-104.896 104.992 0 8.32.704 16.32 2.432 23.936-87.264-4.256-164.48-46.08-216.352-109.792-9.056 15.712-14.368 33.696-14.368 53.056 0 36.352 18.72 68.576 46.624 87.232-16.864-.32-33.408-5.216-47.424-12.928v1.152c0 51.008 36.384 93.376 84.096 103.136-8.544 2.336-17.856 3.456-27.52 3.456-6.72 0-13.504-.384-19.872-1.792 13.6 41.568 52.192 72.128 98.08 73.12-35.712 27.936-81.056 44.768-130.144 44.768-8.608 0-16.864-.384-25.12-1.44C46.496 446.88 101.6 464 161.024 464c193.152 0 298.752-160 298.752-298.688 0-4.64-.16-9.12-.384-13.568 20.832-14.784 38.336-33.248 52.608-54.496z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "waze" => [
                'label' => "URL",
                'title' => "Waze",
                'example' => "https://www.waze.com/ul?q=XXXXXXX",
                'bg_color' => '#75cae3',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" height="512" viewBox="0 0 24 24" width="512"><path d="M17.763 11.033c-.4-.085-.804.17-.9.577-.72 3.36-6.826 3.374-7.3.015a.75.75 0 1 0-1.485.213c.72 5.035 9.198 5.057 10.26.086a.75.75 0 0 0-.576-.891zm-1.36-.733a1.73 1.73 0 0 0 1.728-1.726 1.73 1.73 0 0 0-1.728-1.727 1.73 1.73 0 0 0-1.729 1.727 1.73 1.73 0 0 0 1.729 1.726zm.228-1.727c0 .25-.457.248-.457 0 .001-.3.457-.3.457 0zM10.533 10.3a1.73 1.73 0 0 0 1.728-1.726 1.73 1.73 0 0 0-1.728-1.727 1.73 1.73 0 0 0-1.729 1.727 1.73 1.73 0 0 0 1.729 1.726zm.228-1.727c0 .25-.457.248-.457 0 0-.3.457-.3.457 0zm6.67-6.82C13.725.328 9.635.962 6.483 3.46 3.26 6.042 3.025 9.13 2.87 11.175c-.146 1.914-.275 2.066-.935 2.066-.737 0-1.418.43-1.727 1.078-.33.656-.263 1.433.175 2.032 1.1 1.483 1.9 2 3.785 3.033.314.172.696.106.944-.223-.184.412-.26.878-.2 1.37.207 2.4 3.217 3.343 4.768 1.508a2.67 2.67 0 0 0 .624-1.691c.04.356.316.647.685.67a15.07 15.07 0 0 0 1.533.021c.385-.015.678-.32.707-.695.037 1.586 1.307 2.658 2.7 2.658.078 0 .156-.003.234-.01a2.69 2.69 0 0 0 2.067-1.263 2.67 2.67 0 0 0 .23-2.398c-.04-.108-.095-.208-.15-.308.23.268.642.385.965.192 3.457-2.088 5.27-5.94 4.618-9.813-.567-3.442-2.975-6.297-6.442-7.638zM7.7 21.496a1.21 1.21 0 0 1-1.306-1.104c-.083-.825.552-1.284 1.246-1.284.6 0 1.092.347 1.162 1.088.055.66-.437 1.242-1.102 1.3zm9.238-.564c-.204.325-.54.53-.923.564-1.055.094-1.713-1.143-1.035-1.962.6-.713 1.732-.523 2.06.332v.001c.137.354.1.742-.102 1.065zm1.605-3.043c-.33.154-.524.547-.38.92-.54-.808-1.5-1.292-2.482-1.203-.73.062-1.388.407-1.855.97-.385.464-.6 1.045-.6 1.627-.044-.385-.357-.68-.766-.665a13.17 13.17 0 0 1-1.382-.019.74.74 0 0 0-.775.63c-.002-.028.003-.056.001-.084C10.08 17.76 7.6 16.96 5.95 18.137c-.322.23-.558.528-.738.856.134-.346.007-.743-.327-.926-1.83-1-2.4-1.385-3.295-2.606a.45.45 0 0 1-.036-.484c.07-.146.215-.236.38-.236 2.167 0 2.307-1.834 2.43-3.452.152-1.998.342-4.483 3.053-6.657 2.73-2.16 6.27-2.7 9.472-1.47 2.967 1.147 5.023 3.57 5.503 6.484.546 3.24-.964 6.468-3.85 8.244z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "slack" => [
                'label' => "Workplace URL",
                'title' => "Slack",
                'example' => "https://workplace.slack.com",
                'bg_color' => '#541755',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" ><path d="M8.843 12.65a2.52 2.52 0 0 0-2.521 2.521v6.306a2.521 2.521 0 1 0 5.042 0V15.17a2.52 2.52 0 0 0-2.52-2.52zM.02 15.172c0 1.393 1.13 2.523 2.523 2.523s2.523-1.13 2.523-2.523V12.65h-2.52C1.15 12.65.02 13.78.02 15.172z" class="fill-color" fill="#e91e63"/><path d="M8.846-.001C7.45-.001 6.32 1.13 6.32 2.522s1.13 2.523 2.523 2.523h2.52V2.522A2.52 2.52 0 0 0 8.846-.001zm-6.32 11.37h6.318c1.393 0 2.523-1.13 2.523-2.523s-1.13-2.523-2.523-2.523H2.525c-1.393 0-2.523 1.13-2.523 2.523s1.13 2.523 2.523 2.523z" class="fill-color" fill="#00bcd4"/><path d="M21.457 6.323a2.52 2.52 0 0 0-2.518 2.518v2.528h2.52c1.393 0 2.523-1.13 2.523-2.523s-1.13-2.523-2.523-2.523zm-8.817-3.8v6.325a2.521 2.521 0 1 0 5.042 0V2.522a2.521 2.521 0 1 0-5.042 0z" class="fill-color" fill="#4caf50"/><path d="M17.682 21.476a2.52 2.52 0 0 0-2.521-2.521h-2.52v2.523a2.521 2.521 0 0 0 5.042-.002zm3.798-8.826h-6.32c-1.393 0-2.523 1.13-2.523 2.523s1.13 2.523 2.523 2.523h6.318c1.393 0 2.523-1.13 2.523-2.523s-1.13-2.523-2.523-2.523z" class="fill-color" fill="#ff9800"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "instagram" => [
                'label' => "Username",
                'title' => "Instagram",
                'example' => "myusername",
                'bg_color' => '#df0079',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12.004 5.838a6.16 6.16 0 0 0-6.158 6.158 6.16 6.16 0 0 0 6.158 6.158 6.16 6.16 0 0 0 6.158-6.158 6.16 6.16 0 0 0-6.158-6.158zm0 10.155c-2.2 0-3.997-1.8-3.997-3.997S9.796 8 12.004 8 16 9.788 16 11.996s-1.788 3.997-3.997 3.997zM16.948.076C14.74-.027 9.27-.022 7.06.076c-1.942.1-3.655.56-5.036 1.94-2.307 2.31-2.012 5.42-2.012 9.98 0 4.668-.26 7.706 2.013 9.98 2.317 2.316 5.472 2.013 9.98 2.013 4.624 0 6.22.003 7.855-.63 2.223-.863 3.9-2.85 4.065-6.42a161.35 161.35 0 0 0 0-9.887c-.2-4.212-2.46-6.768-6.977-6.976zm3.495 20.372c-1.513 1.513-3.612 1.378-8.468 1.378-5 0-7.005.074-8.468-1.393-1.685-1.677-1.38-4.37-1.38-8.453 0-5.525-.567-9.504 4.978-9.788 1.274-.045 1.65-.06 4.856-.06l.045.03c5.33 0 9.5-.558 9.76 4.986.057 1.265.07 1.645.07 4.847-.001 4.942.093 6.96-1.394 8.453z"/><circle cx="18.406" cy="5.595" r="1.439"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "google-map" => [
                'label' => "Google Map URL",
                'title' => "Google Map",
                'example' => "https://www.google.com/maps/place/*",
                'bg_color' => '#E84A35',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 511.999 511.999"><path d="M256 0C152.786 0 68.817 85.478 68.817 190.545c0 58.77 29.724 130.103 88.35 212.017 42.902 59.948 85.178 102.702 86.957 104.494a16.69 16.69 0 0 0 11.879 4.943c4.182 0 8.367-1.558 11.61-4.683 1.783-1.717 44.166-42.74 87.15-101.86 58.672-80.7 88.42-153.007 88.42-214.912C443.18 85.478 359.2 0 256 0zm0 272.806c-50.46 0-91.51-41.052-91.51-91.51s41.052-91.51 91.51-91.51 91.51 41.052 91.51 91.51-41.053 91.51-91.51 91.51z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "signal" => [
                'label' => "Signal Group Link",
                'title' => "Signal",
                'example' => "https://signal.group/[group-id]",
                'bg_color' => '#3a76f0',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="170.667" height="170.667" viewBox="0 0 128 128"><path d="M55.4 1.4c-.3.8-.4 2.2-.2 3.1.2 1.4 1.7 1.7 8.8 1.7h8.5V3.3.5L64.2.2c-6.5-.2-8.3 0-8.8 1.2zm-11.9 2c-2.2.7-5.9 2.2-8.2 3.4-4 2.1-4.2 2.4-3.1 4.7 1.4 3.1 1.3 3.1 7.5 0 2.8-1.3 6.3-2.7 7.7-3.1 2.8-.7 2.9-.8 2-4.2-.7-2.5-.6-2.5-5.9-.8zm35.1 1.1c-.4 1.5-.6 2.8-.4 3 .2.1 3.1 1.2 6.6 2.5 3.4 1.2 6.5 2.7 6.8 3.1 1 1.7 3.1.9 4.2-1.7 1.2-2.4 1-2.6-5-5.4-9.7-4.5-11.4-4.7-12.2-1.5zm-26.2 9c-18.6 4-34.4 19.7-38.9 38.6-2.7 11.7-1.3 24.5 4 34.9l2.5 4.9-2.4 10.3c-1.4 5.6-2.4 10.4-2.3 10.5s4.9-.9 10.5-2.3l10.3-2.4 4.9 2.5c29.3 14.9 65.9-2.4 73.5-34.7 8.8-37.2-24.4-70.5-62.1-62.3zm-33.6 5.3c-3.2 3.3-5.8 6.3-5.8 6.8s1 1.5 2.1 2.3c2 1.2 2.6.8 8.1-4.7 5.4-5.3 5.9-6.2 4.9-8.1-.7-1.1-1.7-2.1-2.3-2.1s-3.7 2.6-7 5.8zm81.1-3.7c-1 1.9-.5 2.8 4.9 8.1 5.3 5.4 6.2 5.9 8.1 4.9 1.1-.7 2.1-1.7 2.1-2.3 0-1.5-11.3-12.8-12.8-12.8-.6 0-1.6 1-2.3 2.1zM6 37.2c-4.5 9.7-4.7 11.4-1.5 12.2 1.5.4 2.8.6 3 .4.1-.2 1.2-3.1 2.5-6.6 1.2-3.4 2.7-6.5 3.1-6.8 1.7-1 .9-3.1-1.7-4.2-2.4-1.2-2.6-1-5.4 5zm110.5-5l-2.4 1.4c-.1.1.8 2.7 2.2 5.6 1.4 3 2.9 6.6 3.3 8.2.6 2.4 1 2.7 3.3 2.1 1.4-.4 2.8-.8 2.9-1 .8-.8-6-17.6-7.1-17.5-.1 0-1.1.6-2.2 1.2zM.4 56.3c-.3.8-.4 4.7-.2 8.8.3 7.4.3 7.4 3.1 7.4h2.9V64c0-8.4 0-8.5-2.6-8.8C2 55 .8 55.4.4 56.3zm121.6-.9c-.1.3-.1 4.3-.1 8.8l-.1 8.3h2.9 2.8V64v-8.5l-2.7-.3c-1.6-.2-2.8-.1-2.8.2zM3.7 78.7c-2.2.6-1.8 3.1 1.8 11 1.8 4 2.2 6 1.5 8.2-.7 2.6-.5 3 1.6 3.5 3.4.9 4 .3 4.7-4.1.5-2.8 0-5-1.8-8.8-1.3-2.8-2.7-6.3-3-7.8-.7-2.7-1.3-2.9-4.8-2zm115.5 3c-.6 2.1-2 5.6-3.1 7.7-2.2 4.2-2.1 5 .9 6.6 1.7.9 2.3.4 4.3-3.7 2.6-5.2 5.1-12.2 4.5-12.8-.1-.2-1.5-.6-2.9-1-2.3-.6-2.7-.2-3.7 3.2zm-14.6 23.2c-5.2 5.2-5.7 6.1-4.7 8 .7 1.1 1.6 2.1 2 2.1 1.3 0 11-8.9 12.3-11.3.8-1.6.7-2.3-.8-3.4-1-.7-2-1.3-2.4-1.3-.3 0-3.2 2.6-6.4 5.9zM3.5 113c-2.7 11.1-.4 14.1 9.4 12.1 8.5-1.9 9-2.1 8.5-3.7l-.8-2.8c-.4-1.2-1.3-1.3-6.3-.1-3.2.7-5.9 1.3-6 1.2s.5-2.8 1.2-6c1.2-5 1.1-5.9-.1-6.3-.8-.2-2-.6-2.8-.8-1-.3-1.9 1.4-3.1 6.4zm26.5 1.7c-3.5.6-4.2 1.6-3.5 4.6.6 2.3 1 2.4 3.7 1.6 2.3-.6 3.9-.4 6.7 1 4.2 2.1 11.1 4.5 11.6 3.9.2-.1.6-1.5 1-2.9.6-2.3.3-2.7-2-3.3-1.5-.4-5-1.8-7.8-3.1-2.8-1.4-5.6-2.4-6.1-2.4l-3.6.6zm61.6.2c-.3.4-3.4 1.9-6.8 3.1l-6.6 2.5c-.2.2 0 1.5.4 3 .8 3.2 2.2 3 12-1.5 6.2-2.8 6.4-3 5.2-5.4-1.1-2.6-3.2-3.4-4.2-1.7zm-36.3 7.8c-.3.6-.3 2-.1 3 .3 1.6 1.4 1.8 8.8 1.8h8.5v-2.8-2.9l-8.4-.1c-5.7-.1-8.5.2-8.8 1z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "tiktok" => [
                'label' => "Username",
                'title' => "TikTok",
                'example' => "https://www.tiktok.com/@[username]",
                'bg_color' => '#000000',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M448 209.91a210.06 210.06 0 0 1-122.77-39.25v178.72A162.55 162.55 0 1 1 185 188.31v89.89a74.62 74.62 0 1 0 52.23 71.18V0h88a121.18 121.18 0 0 0 1.86 22.17h0A122.18 122.18 0 0 0 381 102.39a121.43 121.43 0 0 0 67 20.14z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "link" => [
                'label' => "Link",
                'title' => "Custom Link",
                'example' => "https://www.example.com/",
                'bg_color' => '#0068aa',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512.092 512.092"><path d="M312.453 199.6c-6.066-6.102-12.792-11.5-20.053-16.128a119.47 119.47 0 0 0-64.427-18.859 118.95 118.95 0 0 0-84.48 34.987L34.95 308.23C12.613 330.6.06 360.93.04 392.548c-.042 65.98 53.4 119.5 119.4 119.543a118.7 118.7 0 0 0 84.395-34.816l89.6-89.6c1.628-1.614 2.537-3.816 2.524-6.108-.027-4.713-3.87-8.5-8.583-8.484h-3.413c-18.72.066-37.273-3.53-54.613-10.58a8.53 8.53 0 0 0-9.3 1.877l-64.44 64.5c-20.006 20.006-52.442 20.006-72.448 0s-20.006-52.442 0-72.448l108.97-108.885c20-19.965 52.373-19.965 72.363 0 13.472 12.68 34.486 12.68 47.957 0a34.19 34.19 0 0 0 9.9-21.675 34.14 34.14 0 0 0-9.9-26.282zM477.06 34.993c-46.657-46.657-122.303-46.657-168.96 0l-89.515 89.43c-2.458 2.47-3.167 6.185-1.792 9.387a8.53 8.53 0 0 0 8.021 5.205h3.157a143.38 143.38 0 0 1 54.528 10.667 8.53 8.53 0 0 0 9.3-1.877l64.256-64.17c20.006-20.006 52.442-20.006 72.448 0s20.006 52.442 0 72.448L348.46 236.04l-.683.768-28 27.82c-20 19.965-52.373 19.965-72.363 0-13.472-12.68-34.486-12.68-47.957 0a34.14 34.14 0 0 0-9.899 21.845 34.13 34.13 0 0 0 9.9 26.283c9.877 9.92 21.433 18.008 34.133 23.893 1.792.853 3.584 1.536 5.376 2.304s3.67 1.365 5.46 2.048 3.67 1.28 5.46 1.792l5.035 1.365c3.413.853 6.827 1.536 10.325 2.133 4.214.626 8.458 1.025 12.715 1.195h6.485l5.12-.597c1.877-.085 3.84-.512 6.06-.512h2.9l5.888-.853 2.73-.512 4.95-1.024h.94a119.46 119.46 0 0 0 55.38-31.403l108.63-108.63c46.66-46.657 46.66-122.303.003-168.96z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
            "custom-link" => [
                'label' => "Link",
                'title' => "Custom Link",
                'example' => "https://www.example.com/",
                'bg_color' => '#F83F3C',
                'icon_color' => '#ffffff',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512.092 512.092"><path d="M312.453 199.6c-6.066-6.102-12.792-11.5-20.053-16.128a119.47 119.47 0 0 0-64.427-18.859 118.95 118.95 0 0 0-84.48 34.987L34.95 308.23C12.613 330.6.06 360.93.04 392.548c-.042 65.98 53.4 119.5 119.4 119.543a118.7 118.7 0 0 0 84.395-34.816l89.6-89.6c1.628-1.614 2.537-3.816 2.524-6.108-.027-4.713-3.87-8.5-8.583-8.484h-3.413c-18.72.066-37.273-3.53-54.613-10.58a8.53 8.53 0 0 0-9.3 1.877l-64.44 64.5c-20.006 20.006-52.442 20.006-72.448 0s-20.006-52.442 0-72.448l108.97-108.885c20-19.965 52.373-19.965 72.363 0 13.472 12.68 34.486 12.68 47.957 0a34.19 34.19 0 0 0 9.9-21.675 34.14 34.14 0 0 0-9.9-26.282zM477.06 34.993c-46.657-46.657-122.303-46.657-168.96 0l-89.515 89.43c-2.458 2.47-3.167 6.185-1.792 9.387a8.53 8.53 0 0 0 8.021 5.205h3.157a143.38 143.38 0 0 1 54.528 10.667 8.53 8.53 0 0 0 9.3-1.877l64.256-64.17c20.006-20.006 52.442-20.006 72.448 0s20.006 52.442 0 72.448L348.46 236.04l-.683.768-28 27.82c-20 19.965-52.373 19.965-72.363 0-13.472-12.68-34.486-12.68-47.957 0a34.14 34.14 0 0 0-9.899 21.845 34.13 34.13 0 0 0 9.9 26.283c9.877 9.92 21.433 18.008 34.133 23.893 1.792.853 3.584 1.536 5.376 2.304s3.67 1.365 5.46 2.048 3.67 1.28 5.46 1.792l5.035 1.365c3.413.853 6.827 1.536 10.325 2.133 4.214.626 8.458 1.025 12.715 1.195h6.485l5.12-.597c1.877-.085 3.84-.512 6.06-.512h2.9l5.888-.853 2.73-.512 4.95-1.024h.94a119.46 119.46 0 0 0 55.38-31.403l108.63-108.63c46.66-46.657 46.66-122.303.003-168.96z"/></svg>',
                'status' => 0,
                'value' => '',
                'has_desktop' => 1,
                'has_mobile' => 1
            ],
        ];
    }//end get_buttons()

    public static function get_cta_icon() {
        return [
            'chat-circle' => [
                'value' => 'chat-circle',
                'icon' => '<svg viewBox="0 0 511.63 511.63" xml:space="preserve" xmlns="http://www.w3.org/2000/svg"><path d="m301.93 327.6c30.926-13.038 55.34-30.785 73.23-53.248 17.888-22.458 26.833-46.915 26.833-73.372 0-26.458-8.945-50.917-26.84-73.376-17.888-22.459-42.298-40.208-73.228-53.249-30.93-13.039-64.571-19.556-100.93-19.556-36.354 0-69.995 6.521-100.93 19.556-30.929 13.04-55.34 30.789-73.229 53.249-17.891 22.463-26.838 46.918-26.838 73.377 0 22.648 6.767 43.975 20.28 63.96 13.512 19.981 32.071 36.829 55.671 50.531-1.902 4.572-3.853 8.754-5.852 12.566-2 3.806-4.377 7.467-7.139 10.991-2.76 3.525-4.899 6.283-6.423 8.275-1.523 1.998-3.997 4.812-7.425 8.422-3.427 3.617-5.617 5.996-6.567 7.135 0-0.191-0.381 0.24-1.143 1.287-0.763 1.047-1.191 1.52-1.285 1.431-0.096-0.103-0.477 0.373-1.143 1.42-0.666 1.048-1 1.571-1 1.571l-0.715 1.423c-0.282 0.575-0.476 1.137-0.57 1.712-0.096 0.567-0.144 1.19-0.144 1.854s0.094 1.28 0.288 1.854c0.381 2.471 1.475 4.466 3.283 5.996 1.807 1.52 3.756 2.279 5.852 2.279h0.857c9.515-1.332 17.701-2.854 24.552-4.569 29.312-7.61 55.771-19.797 79.372-36.545 17.129 3.046 33.879 4.568 50.247 4.568 36.357 0.013 70.002-6.502 100.93-19.542z"></path><path d="m491.35 338.17c13.518-19.889 20.272-41.247 20.272-64.09 0-23.414-7.146-45.316-21.416-65.68-14.277-20.362-33.694-37.305-58.245-50.819 4.374 14.274 6.563 28.739 6.563 43.398 0 25.503-6.368 49.676-19.129 72.519-12.752 22.836-31.025 43.01-54.816 60.524-22.08 15.988-47.205 28.261-75.377 36.829-28.164 8.562-57.573 12.848-88.218 12.848-5.708 0-14.084-0.377-25.122-1.137 38.256 25.119 83.177 37.685 134.76 37.685 16.371 0 33.119-1.526 50.251-4.571 23.6 16.755 50.06 28.931 79.37 36.549 6.852 1.718 15.037 3.237 24.554 4.568 2.283 0.191 4.381-0.476 6.283-1.999 1.903-1.522 3.142-3.61 3.71-6.272-0.089-1.143 0-1.77 0.287-1.861 0.281-0.09 0.233-0.712-0.144-1.852-0.376-1.144-0.568-1.715-0.568-1.715l-0.712-1.424c-0.198-0.376-0.52-0.903-0.999-1.567-0.476-0.66-0.855-1.143-1.143-1.427-0.28-0.284-0.705-0.763-1.28-1.424-0.568-0.66-0.951-1.092-1.143-1.283-0.951-1.143-3.139-3.521-6.564-7.139-3.429-3.613-5.899-6.42-7.422-8.418-1.523-1.999-3.665-4.757-6.424-8.282-2.758-3.518-5.14-7.183-7.139-10.991-1.998-3.806-3.949-7.995-5.852-12.56 23.602-13.716 42.156-30.513 55.667-50.409z"></path></svg>'
            ],
            'chat-circle-dots' => [
                'value' => 'chat-circle-dots',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" version="1.1" width="256" height="256" viewBox="0 0 256 256" xml:space="preserve"> <defs> </defs> <g style="stroke: none; stroke-width: 0; stroke-dasharray: none; stroke-linecap: butt; stroke-linejoin: miter; stroke-miterlimit: 10; fill: none; fill-rule: nonzero; opacity: 1;" transform="translate(1.4065934065934016 1.4065934065934016) scale(2.81 2.81)"> <path d="M 38.105 10.26 c -18.35 0 -33.278 12.491 -33.278 27.845 c 0 5.78 2.093 11.321 6.051 16.023 c 0.598 0.711 0.527 1.744 -0.16 2.352 l -10.718 9.469 h 38.105 c 18.35 0 33.278 -12.491 33.278 -27.844 C 71.383 22.751 56.455 10.26 38.105 10.26 z M 22.838 41.813 c -2.358 0 -4.269 -1.911 -4.269 -4.269 c 0 -2.358 1.911 -4.269 4.269 -4.269 s 4.269 1.911 4.269 4.269 C 27.107 39.902 25.196 41.813 22.838 41.813 z M 37.803 41.813 c -2.358 0 -4.269 -1.911 -4.269 -4.269 c 0 -2.358 1.911 -4.269 4.269 -4.269 c 2.358 0 4.269 1.911 4.269 4.269 C 42.072 39.902 40.161 41.813 37.803 41.813 z M 52.768 41.813 c -2.358 0 -4.269 -1.911 -4.269 -4.269 c 0 -2.358 1.911 -4.269 4.269 -4.269 s 4.269 1.911 4.269 4.269 C 57.037 39.902 55.126 41.813 52.768 41.813 z" style="stroke: none; stroke-width: 1; stroke-dasharray: none; stroke-linecap: butt; stroke-linejoin: miter; stroke-miterlimit: 10; fill: rgb(0,0,0); fill-rule: nonzero; opacity: 1;" transform=" matrix(1 0 0 1 0 0) " stroke-linecap="round"/> <path d="M 79.236 70.27 c -0.688 -0.608 -0.758 -1.641 -0.16 -2.352 c 3.958 -4.702 6.051 -10.243 6.051 -16.023 c 0 -7.322 -3.4 -13.988 -8.942 -18.964 c 0.329 1.685 0.507 3.412 0.507 5.173 c 0 17.637 -17.148 31.985 -38.228 31.985 H 26.689 c 6.107 5.904 15.117 9.65 25.159 9.65 h 38.105 L 79.236 70.27 z" style="stroke: none; stroke-width: 1; stroke-dasharray: none; stroke-linecap: butt; stroke-linejoin: miter; stroke-miterlimit: 10; fill: rgb(0,0,0); fill-rule: nonzero; opacity: 1;" transform=" matrix(1 0 0 1 0 0) " stroke-linecap="round"/> </g> </svg>'
            ],
            'chat-square-line' => [
                'value' => 'chat-square-line',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" version="1.1" width="256" height="256" viewBox="0 0 256 256" xml:space="preserve"> <defs> </defs> <g style="stroke: none; stroke-width: 0; stroke-dasharray: none; stroke-linecap: butt; stroke-linejoin: miter; stroke-miterlimit: 10; fill: none; fill-rule: nonzero; opacity: 1;" transform="translate(1.4065934065934016 1.4065934065934016) scale(2.81 2.81)"> <path d="M 53.537 32.988 H 2.208 c -1.315 0 -2.38 1.066 -2.38 2.38 v 34.564 c 0 1.315 1.066 2.38 2.38 2.38 h 6.313 v 11.781 l 12.004 -11.781 h 33.012 c 1.315 0 2.38 -1.066 2.38 -2.38 V 35.369 C 55.917 34.054 54.852 32.988 53.537 32.988 z M 44.762 58.675 H 10.984 v -3 h 33.779 V 58.675 z M 44.762 48.675 H 10.984 v -3 h 33.779 V 48.675 z" style="stroke: none; stroke-width: 1; stroke-dasharray: none; stroke-linecap: butt; stroke-linejoin: miter; stroke-miterlimit: 10; fill: rgb(0,0,0); fill-rule: nonzero; opacity: 1;" transform=" matrix(1 0 0 1 0 0) " stroke-linecap="round"/> <path d="M 87.62 8.443 H 36.291 c -1.315 0 -2.38 1.066 -2.38 2.38 v 16.165 h 19.798 c 4.621 0 8.38 3.759 8.38 8.38 v 12.399 h 7.214 l 12.004 11.781 V 47.768 h 6.313 c 1.315 0 2.38 -1.066 2.38 -2.38 V 10.823 C 90 9.509 88.934 8.443 87.62 8.443 z M 78.845 32.452 H 67.623 v -3 h 11.222 V 32.452 z M 78.845 22.452 H 45.066 v -3 h 33.779 V 22.452 z" style="stroke: none; stroke-width: 1; stroke-dasharray: none; stroke-linecap: butt; stroke-linejoin: miter; stroke-miterlimit: 10; fill: rgb(0,0,0); fill-rule: nonzero; opacity: 1;" transform=" matrix(1 0 0 1 0 0) " stroke-linecap="round"/> </g> </svg>'
            ],
            'chat-square-bubble' => [
                'value' => 'chat-square-bubble',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" version="1.1" width="480" height="480" viewBox="0 0 480 480" xml:space="preserve"> <desc>Created with Fabric.js 3.6.6</desc> <defs> </defs> <g transform="matrix(0.34 0 0 0.33 261.1 254.2)"> <g style=""> <g transform="matrix(1 0 0 1 0 0)"> <rect style="stroke: none; stroke-width: 1; stroke-dasharray: none; stroke-linecap: butt; stroke-dashoffset: 0; stroke-linejoin: miter; stroke-miterlimit: 4; fill: rgb(255,255,255); fill-rule: nonzero; opacity: 1; visibility: hidden;" x="-540" y="-540" rx="0" ry="0" width="1080" height="1080"/> </g> <g transform="matrix(8.79 0 0 8.79 -49.27 -21.48)"> <path style="stroke: none; stroke-width: 1; stroke-dasharray: none; stroke-linecap: butt; stroke-dashoffset: 0; stroke-linejoin: miter; stroke-miterlimit: 4; fill: rgb(0,0,0); fill-rule: evenodd; opacity: 1;" transform=" translate(-61.44, -56.97)" d="M 3.77 0 L 119.11 0 C 121.18756731120145 0.01093479157852618 122.86906520842147 1.6924326887985441 122.88 3.769999999999999 L 122.88 80.94 C 122.87453691595951 83.02147148070465 121.19144970590865 84.70902311306332 119.11 84.72 L 61.44 84.72 L 32.339999999999996 106.34 C 22.729999999999997 115.47 16.259999999999998 117.79 17.189999999999998 105.34 L 17.189999999999998 84.72 L 3.77 84.72 C 1.6885502940913502 84.70902311306334 0.005463084040489008 83.02147148070465 0 80.94 L 0 3.77 C 0.010934791578524467 1.6924326887985437 1.6924326887985428 0.010934791578524727 3.769999999999999 -4.440892098500626e-16 Z M 62.92 34.34 C 65.8006805844963 34.3400028412231 68.39752135081145 36.07581913845966 69.49897356621358 38.73760869571015 C 70.60042578161571 41.39939825296064 69.98943371801771 44.46262023015532 67.9510539988227 46.49813705648629 C 65.9126742796277 48.53365388281727 62.848596590399 49.14034004362791 60.18835773197965 48.03514782231174 C 57.5281188735603 46.929955600995584 55.79595410030408 44.33067774327462 55.800000000000004 41.45000000000002 C 55.79999813318926 39.562576867325305 56.55045782479452 37.75260342858289 57.88600507563849 36.41893194634966 C 59.22155232648245 35.08526046411642 61.032578734135114 34.33734539643695 62.919999999999995 34.34 Z M 90.11 34.34 C 92.99158719801962 34.33595566111827 95.5914968277246 36.06920391141345 96.69610156332025 38.73067146044862 C 97.8007062989159 41.39213900948378 97.19218512458336 44.457001414253845 95.1545932694186 46.49459326941861 C 93.11700141425383 48.53218512458337 90.05213900948377 49.1407062989159 87.3906714604486 48.03610156332025 C 84.72920391141345 46.9314968277246 82.99595566111827 44.33158719801961 83 41.44999999999999 C 83 37.52325542870306 86.18325542870306 34.34 90.11 34.34 Z M 35.72 34.34 C 38.60158719801962 34.33595566111827 41.201496827724604 36.06920391141345 42.306101563320254 38.73067146044862 C 43.4107062989159 41.39213900948378 42.80218512458336 44.457001414253845 40.7645932694186 46.49459326941861 C 38.727001414253834 48.53218512458337 35.66213900948377 49.1407062989159 33.0006714604486 48.03610156332025 C 30.339203911413442 46.9314968277246 28.605955661118266 44.33158719801961 28.61 41.44999999999999 C 28.61 37.52325542870306 31.79325542870306 34.34 35.72 34.34 Z" stroke-linecap="round"/> </g> </g> </g> </svg>'
            ],
            'chat-comment-line' => [
                'value' => 'chat-comment-line',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" version="1.1" width="640" height="640" viewBox="0 0 640 640" xml:space="preserve"> <desc>Created with Fabric.js 3.6.6</desc> <defs> </defs> <g transform="matrix(4.38 0 0 3.46 320.5 320.5)" id="Layer_1"> <path style="stroke: none; stroke-width: 1; stroke-dasharray: none; stroke-linecap: butt; stroke-dashoffset: 0; stroke-linejoin: miter; stroke-miterlimit: 4; fill: rgb(0,0,0); fill-rule: evenodd; opacity: 1;" transform=" translate(-60.91, -61.44)" d="M 55.05 97.68 l -24.9 23.88 a 3.95 3.95 0 0 1 -6.89 -2.62 V 97.68 H 10.1 A 10.16 10.16 0 0 1 0 87.58 V 10.1 A 10.18 10.18 0 0 1 10.1 0 H 111.73 a 10.16 10.16 0 0 1 10.1 10.1 V 87.58 a 10.16 10.16 0 0 1 -10.1 10.1 Z M 27.53 36.61 a 3.94 3.94 0 0 1 0 -7.87 H 94.3 a 3.94 3.94 0 1 1 0 7.87 Z m 0 25.55 a 3.94 3.94 0 0 1 0 -7.87 H 82 a 3.94 3.94 0 0 1 0 7.87 Z" stroke-linecap="round"/> </g> </svg>'
            ]
        ];
    }

    public static function get_button_size() {
        return [
            45,
            54,
            63,
            72,
            81,
            90,
            100,
        ];
    }

    public static function get_attention_effects() {
        return [
            'none' => [
                'value' => 'attention-none',
                'title' => 'None'
            ],
            'bounce' => [
                'value' => 'attention-bounce',
                'title' => 'Bounce'
            ],
            'flash' => [
                'value' => 'attention-flash',
                'title' => 'Flash'
            ],
            'gelatine' => [
                'value' => 'attention-gelatine',
                'title' => 'Gelatine'
            ],
            'pulse' => [
                'value' => 'attention-pulse',
                'title' => 'Pulse'
            ],
            'shake' => [
                'value' => 'attention-shake',
                'title' => 'Shake'
            ],
            'shockwave' => [
                'value' => 'attention-shockwave',
                'title' => 'Shockwave'
            ],
            'spin' => [
                'value' => 'attention-spin',
                'title' => 'Spin'
            ],
            'swing' => [
                'value' => 'attention-swing',
                'title' => 'Swing'
            ]
        ];
    }

    public static function get_svg_icon() {
        return [
            'prev' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M19 12H5M5 12L12 19M5 12L12 5" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>',
            'next' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M5 12H19M19 12L12 5M19 12L12 19" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>',
            'check' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M20 6L9 17L4 12" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>',
            'checkmark' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M438.6 105.4c12.5 12.5 12.5 32.7 0 45.2l-256 256c-12.5 12.5-32.7 12.5-45.2 0L9.372 278.6a31.94 31.94 0 0 1 0-45.2c12.498-12.5 32.758-12.5 45.258 0L159.1 338.7l234.3-233.3c12.5-12.52 32.7-12.52 45.2 0h0z"/></svg>',
            'close' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 512"><path d="M310.6 150.6c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0L160 210.7 54.6 105.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L114.7 256 9.4 361.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L160 301.3 265.4 406.6c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3L205.3 256 310.6 150.6z"/></svg>',
            'loader' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M222.7 32.1c5 16.9-4.6 34.8-21.5 39.8C121.8 95.6 64 169.1 64 256c0 106 86 192 192 192s192-86 192-192c0-86.9-57.8-160.4-137.1-184.1c-16.9-5-26.6-22.9-21.5-39.8s22.9-26.6 39.8-21.5C434.9 42.1 512 140 512 256c0 141.4-114.6 256-256 256S0 397.4 0 256C0 140 77.1 42.1 182.9 10.6c16.9-5 34.8 4.6 39.8 21.5z"/></svg>',
            'actions' => [
                'edit' => '<svg width="24" height="24" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M4 20H2.75C2.75 20.6904 3.30964 21.25 4 21.25V20ZM4 16L3.11612 15.1161C2.8817 15.3505 2.75 15.6685 2.75 16H4ZM17 3L17.8839 2.11612C17.3957 1.62796 16.6043 1.62796 16.1161 2.11612L17 3ZM21 7L21.8839 7.88388C22.372 7.39573 22.372 6.60427 21.8839 6.11612L21 7ZM8 20V21.25C8.33152 21.25 8.64946 21.1183 8.88388 20.8839L8 20ZM5.25 20V16H2.75V20H5.25ZM16.1161 3.88388L20.1161 7.88388L21.8839 6.11612L17.8839 2.11612L16.1161 3.88388ZM8 18.75H4V21.25H8V18.75ZM20.1161 6.11612L17.1161 9.11612L18.8839 10.8839L21.8839 7.88388L20.1161 6.11612ZM17.1161 9.11612L7.11612 19.1161L8.88388 20.8839L18.8839 10.8839L17.1161 9.11612ZM4.88388 16.8839L14.8839 6.88388L13.1161 5.11612L3.11612 15.1161L4.88388 16.8839ZM14.8839 6.88388L17.8839 3.88388L16.1161 2.11612L13.1161 5.11612L14.8839 6.88388ZM18.8839 9.11612L14.8839 5.11612L13.1161 6.88388L17.1161 10.8839L18.8839 9.11612Z" /></svg>',
                'clone' => '<svg width="24" height="24" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M17.5 12.75C16.8096 12.75 16.25 13.3096 16.25 14C16.25 14.6904 16.8096 15.25 17.5 15.25V12.75ZM8.75 6.5C8.75 7.19036 9.30964 7.75 10 7.75C10.6904 7.75 11.25 7.19036 11.25 6.5H8.75ZM12 4.25H19V1.75H12V4.25ZM19.75 5V12H22.25V5H19.75ZM19 12.75H17.5V15.25H19V12.75ZM11.25 6.5V5H8.75V6.5H11.25ZM19.75 12C19.75 12.4142 19.4142 12.75 19 12.75V15.25C20.7949 15.25 22.25 13.7949 22.25 12H19.75ZM19 4.25C19.4142 4.25 19.75 4.58579 19.75 5H22.25C22.25 3.20507 20.7949 1.75 19 1.75V4.25ZM12 1.75C10.2051 1.75 8.75 3.20507 8.75 5H11.25C11.25 4.58579 11.5858 4.25 12 4.25V1.75ZM5 11.25H12V8.75H5V11.25ZM12.75 12V19H15.25V12H12.75ZM12 19.75H5V22.25H12V19.75ZM4.25 19V12H1.75V19H4.25ZM5 19.75C4.58579 19.75 4.25 19.4142 4.25 19H1.75C1.75 20.7949 3.20507 22.25 5 22.25V19.75ZM12.75 19C12.75 19.4142 12.4142 19.75 12 19.75V22.25C13.7949 22.25 15.25 20.7949 15.25 19H12.75ZM12 11.25C12.4142 11.25 12.75 11.5858 12.75 12H15.25C15.25 10.2051 13.7949 8.75 12 8.75V11.25ZM5 8.75C3.20507 8.75 1.75 10.2051 1.75 12H4.25C4.25 11.5858 4.58579 11.25 5 11.25V8.75Z" fill=""/></svg>',
                'trash' => '<svg width="24" height="24" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M19.25 6C19.25 5.30964 18.6904 4.75 18 4.75C17.3096 4.75 16.75 5.30964 16.75 6H19.25ZM7.25 6C7.25 5.30964 6.69036 4.75 6 4.75C5.30964 4.75 4.75 5.30964 4.75 6H7.25ZM4 4.75C3.30964 4.75 2.75 5.30964 2.75 6C2.75 6.69036 3.30964 7.25 4 7.25V4.75ZM20 7.25C20.6904 7.25 21.25 6.69036 21.25 6C21.25 5.30964 20.6904 4.75 20 4.75V7.25ZM13.75 6C13.75 6.69036 14.3096 7.25 15 7.25C15.6904 7.25 16.25 6.69036 16.25 6H13.75ZM7.75 6C7.75 6.69036 8.30964 7.25 9 7.25C9.69036 7.25 10.25 6.69036 10.25 6H7.75ZM16.75 6V18H19.25V6H16.75ZM16 18.75H8V21.25H16V18.75ZM7.25 18V6H4.75V18H7.25ZM8 18.75C7.58579 18.75 7.25 18.4142 7.25 18H4.75C4.75 19.7949 6.20507 21.25 8 21.25V18.75ZM16.75 18C16.75 18.4142 16.4142 18.75 16 18.75V21.25C17.7949 21.25 19.25 19.7949 19.25 18H16.75ZM4 7.25H20V4.75H4V7.25ZM16.25 6V5H13.75V6H16.25ZM13 0.75H11V3.25H13V0.75ZM7.75 5V6H10.25V5H7.75ZM11 0.75C10.3521 0.75 9.81895 1.02903 9.43194 1.34436C9.04614 1.65869 8.74138 2.05804 8.50989 2.44944C8.05984 3.21034 7.75 4.17798 7.75 5H10.25C10.25 4.71745 10.3879 4.18509 10.6617 3.72213C10.7921 3.50161 10.9183 3.35806 11.0111 3.28248C11.1026 3.2079 11.0956 3.25 11 3.25V0.75ZM16.25 5C16.25 4.17798 15.9402 3.21034 15.4901 2.44944C15.2586 2.05804 14.9539 1.65869 14.5681 1.34436C14.181 1.02903 13.6479 0.75 13 0.75V3.25C12.9044 3.25 12.8974 3.2079 12.9889 3.28248C13.0817 3.35806 13.2079 3.50161 13.3383 3.72213C13.6121 4.18509 13.75 4.71745 13.75 5H16.25Z" /></svg>'
            ],
            'help' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.09 9C9.3251 8.33167 9.78915 7.76811 10.4 7.40913C11.0108 7.05016 11.7289 6.91894 12.4272 7.03871C13.1255 7.15849 13.7588 7.52152 14.2151 8.06353C14.6713 8.60553 14.9211 9.29152 14.92 10C14.92 12 11.92 13 11.92 13M12 17H12.01M22 12C22 17.5228 17.5228 22 12 22C6.47715 22 2 17.5228 2 12C2 6.47715 6.47715 2 12 2C17.5228 2 22 6.47715 22 12Z" stroke="#3c5668" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>',
            'trash' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M16 6V5.2C16 4.0799 16 3.51984 15.782 3.09202C15.5903 2.71569 15.2843 2.40973 14.908 2.21799C14.4802 2 13.9201 2 12.8 2H11.2C10.0799 2 9.51984 2 9.09202 2.21799C8.71569 2.40973 8.40973 2.71569 8.21799 3.09202C8 3.51984 8 4.0799 8 5.2V6M10 11.5V16.5M14 11.5V16.5M3 6H21M19 6V17.2C19 18.8802 19 19.7202 18.673 20.362C18.3854 20.9265 17.9265 21.3854 17.362 21.673C16.7202 22 15.8802 22 14.2 22H9.8C8.11984 22 7.27976 22 6.63803 21.673C6.07354 21.3854 5.6146 20.9265 5.32698 20.362C5 19.7202 5 18.8802 5 17.2V6" stroke="" stroke-width="1" stroke-linecap="round" stroke-linejoin="round"/></svg>',
            'preview' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path d="M160 256C160 185.3 217.3 128 288 128C358.7 128 416 185.3 416 256C416 326.7 358.7 384 288 384C217.3 384 160 326.7 160 256zM288 336C332.2 336 368 300.2 368 256C368 211.8 332.2 176 288 176C287.3 176 286.7 176 285.1 176C287.3 181.1 288 186.5 288 192C288 227.3 259.3 256 224 256C218.5 256 213.1 255.3 208 253.1C208 254.7 208 255.3 208 255.1C208 300.2 243.8 336 288 336L288 336zM95.42 112.6C142.5 68.84 207.2 32 288 32C368.8 32 433.5 68.84 480.6 112.6C527.4 156 558.7 207.1 573.5 243.7C576.8 251.6 576.8 260.4 573.5 268.3C558.7 304 527.4 355.1 480.6 399.4C433.5 443.2 368.8 480 288 480C207.2 480 142.5 443.2 95.42 399.4C48.62 355.1 17.34 304 2.461 268.3C-.8205 260.4-.8205 251.6 2.461 243.7C17.34 207.1 48.62 156 95.42 112.6V112.6zM288 80C222.8 80 169.2 109.6 128.1 147.7C89.6 183.5 63.02 225.1 49.44 256C63.02 286 89.6 328.5 128.1 364.3C169.2 402.4 222.8 432 288 432C353.2 432 406.8 402.4 447.9 364.3C486.4 328.5 512.1 286 526.6 256C512.1 225.1 486.4 183.5 447.9 147.7C406.8 109.6 353.2 80 288 80V80z"/></svg>',
            'lock' => '<svg width="24" height="24" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M5 21H4C4 21.5523 4.44772 22 5 22V21ZM19 21V22C19.5523 22 20 21.5523 20 21H19ZM19 11H20C20 10.4477 19.5523 10 19 10V11ZM5 11V10C4.44772 10 4 10.4477 4 11H5ZM9 11V7H7V11H9ZM9 7C9 6.49336 9.16022 5.71113 9.61153 5.0843C10.0261 4.50847 10.7259 4 12 4V2C10.0741 2 8.77386 2.82487 7.98847 3.9157C7.23978 4.95554 7 6.17331 7 7H9ZM12 4C13.2741 4 13.9739 4.50847 14.3885 5.0843C14.8398 5.71113 15 6.49336 15 7H17C17 6.17331 16.7602 4.95554 16.0115 3.9157C15.2261 2.82487 13.9259 2 12 2V4ZM15 7V11H17V7H15ZM5 22H19V20H5V22ZM20 21V11H18V21H20ZM4 11V21H6V11H4ZM8 10H5V12H8V10ZM19 10H16V12H19V10ZM16 10H8V12H16V10Z" /></svg>',
            'chevron_down' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M224 416c-8.188 0-16.38-3.125-22.62-9.375l-192-192c-12.5-12.5-12.5-32.75 0-45.25s32.75-12.5 45.25 0L224 338.8l169.4-169.4c12.5-12.5 32.75-12.5 45.25 0s12.5 32.75 0 45.25l-192 192C240.4 412.9 232.2 416 224 416z"/></svg>',
        ];
    }


    public static function get_timezone() {
        return [
            '-12' => "UTC-12:00",
            '-11.5' => "UTC-11:30",
            '-11' => "UTC-11:00",
            '-10.5' => "UTC-10:30",
            '-10' => "UTC-10:00",
            '-09.5' => "UTC-09:30",
            '-09' => "UTC-09:00",
            '-08.5' => "UTC-08:30",
            '-08' => "UTC-08:00",
            '-07.5' => "UTC-07:30",
            '-07' => "UTC-07:00",
            '-06.5' => "UTC-06:30",
            '-06' => "UTC-06:00",
            '-05.5' => "UTC-05:30",
            '-05' => "UTC-05:00",
            '-04.5' => "UTC-04:30",
            '-04' => "UTC-04:00",
            '-03.5' => "UTC-03:30",
            '-03' => "UTC-03:00",
            '-02.5' => "UTC-02:30",
            '-02' => "UTC-02:00",
            '-01.5' => "UTC-01:30",
            '-01' => "UTC-01:00",
            '-0.5' => "UTC-00:30",
            '0' => "UTC",
            '0.5' => "UTC+00:30",
            '1' => "UTC+01:00",
            '1.5' => "UTC+01:30",
            '2' => "UTC+02:00",
            '2.5' => "UTC+02:30",
            '3' => "UTC+03:00",
            '3.5' => "UTC+03:30",
            '4' => "UTC+04:00",
            '4.5' => "UTC+04:30",
            '5' => "UTC+05:00",
            '5.5' => "UTC+05:30",
            '6' => "UTC+06:00",
            '6.5' => "UTC+06:30",
            '7' => "UTC+07:00",
            '7.5' => "UTC+07:30",
            '8' => "UTC+08:00",
            '8.5' => "UTC+08:30",
            '9' => "UTC+09:00",
            '9.5' => "UTC+09:30",
            '10' => "UTC+10:00",
            '10.5' => "UTC+10:30",
            '11' => "UTC+11:00",
            '11.5' => "UTC+11:30",
            '12' => "UTC+12:00",
            '12.5' => "UTC+12:30"
        ];
    }


    public static function get_plan_prices() {
        return [
            '1_domain' => [
                '1_year' => [
                    'price' => 19,
                    'id' => 1
                ],
                'lifetime' => [
                    'price' => 49,
                    'id' => 2
                ]
            ],
            '10_domain' => [
                '1_year' => [
                    'price' => 49,
                    'id' => 3
                ],
                'lifetime' => [
                    'price' => 129,
                    'id' => 4
                ]
            ],
            '50_domain' => [
                '1_year' => [
                    'price' => 99,
                    'id' => 5
                ],
                'lifetime' => [
                    'price' => 229,
                    'id' => 6
                ]
            ]
        ];
    }


    public static function get_features_list() {
        return [
            [
                'tooltip' => esc_html__("Show different widgets for different pages based on page targeting rules. You can also show different channels on desktop and mobile", 'quick-chat-buttons'),
                'text'    => esc_html__("Multiple Widgets", 'quick-chat-buttons'),
            ],
            [
                'tooltip' => esc_html__("Choose custom color for your widget button", 'quick-chat-buttons'),
                'text'    => esc_html__("Widget Customization", 'quick-chat-buttons'),
            ],
            [
                'tooltip' => esc_html__("Select from 22 channels also upload cutom icon", 'quick-chat-buttons'),
                'text'    => esc_html__("Channel Customization", 'quick-chat-buttons'),
            ],
            [
                'tooltip' => esc_html__("Select pages where you like to show or hide Quick Chat Buttons", 'quick-chat-buttons'),
                'text'    => esc_html__("Page Targeting", 'quick-chat-buttons'),
            ],
            [
                'tooltip' => esc_html__("Select week day and time when you like to show Quick Chat Buttons", 'quick-chat-buttons'),
                'text'    => esc_html__("Days and Time Selection", 'quick-chat-buttons'),
            ],
            [
                'tooltip' => esc_html__("Show Quick Chat Buttons between selected dates and time", 'quick-chat-buttons'),
                'text'    => esc_html__("Date and Time Selection", 'quick-chat-buttons'),
            ],
            [
                'tooltip' => esc_html__("Display Quick Chat Buttons on custom location", 'quick-chat-buttons'),
                'text'    => esc_html__("Custom Location", 'quick-chat-buttons'),
            ],
        ];
    }


    public static function get_customize_widget_settings() {
        return [
            'cta_icon'             => 'chat-circle',
            'cta_text'             => 'Contact Us',
            'button_size'          => '54',
            'position'             => 'right',
            'side_position'        => '25',
            'bottom_position'      => '25',
            'icon_view'            => 'vertical',
            'icon_state'           => 'click_to_open',
            'hide_close_btn'       => 0,
            'btn_bg_color'         => '#5cc0fa',
            'btn_icon_color'       => '#ffffff',
            'attention_effect'     => 'attention-none',
            'show_chat_bubble'     => 0,
            'num_of_message'       => '1',
            'message_bg_color'     => '#ff0000',
            'message_text_color'   => '#ffffff',
            'message_border_color' => '#ffffff'
        ];
    }


    public static function get_trigger_settings() {
        return [
            'widget_status' => 1,
            'after_seconds' => 0,
            'seconds' => '0',
            'after_scroll' => 0,
            'page_scroll' => '0'
        ];
    }


    public static function get_channel_setting($setting) {
        return [
            'title' => $setting['title'],
            'bg_color' => $setting['bg_color'],
            'icon_color' => $setting['icon_color'],
            'status' => 0,
            'value' => '',
            'has_desktop' => 1,
            'has_mobile' => 1
        ];
    }


}//end class