<?php
/**
 * Events processor for PYS.
 *
 * @package QACP
 */

namespace QACP\Processors;

/**
 * Processor for PixelYourSite ajax calls.
 */
class PYS extends \QACP\Processors\Processor_Base {

	/**
	 * Associated action name.
	 *
	 * @var string action name.
	 */
	protected $_action_name = 'pys_api_event';

	/**
	 * PYS event data for using in different processing stages.
	 *
	 * @var mixed event data.
	 */
	protected $_current_processing_event_data = null;

	/**
	 * Cookies array. Used for storing the current cookies temporarily.
	 *
	 * @var mixed The cookies.
	 */
	protected $_cookies = null;

	/**
	 * Returns the action name.
	 *
	 * @return string The action name.
	 */
	public function get_action_name() {

		return $this->_action_name;
	}
	/**
	 * Class constructor
	 *
	 * @param string $p_action_name The associated action name.
	 */
	public function __construct( $p_action_name = 'pys_api_event' ) {

		$this->_action_name = $p_action_name;

		add_action( 'qacp_before_process_entry_' . $this->_action_name, array( $this, 'before_process_entry' ) );

		/**
		 * Remember to add action to the right hook in order for the processing to take place.
		 */

		add_action( 'qacp_process_one_entry_' . $this->_action_name, array( $this, 'process_entry' ), 20 );
	}
	/**
	 * Prepares a PYS event from a QACP event.
	 *
	 * @param mixed $p_entry_data Event data.
	 * @return mixed PYS SingleEvent instance.
	 */
	public function prepare_single_event( $p_entry_data ) {

		$_single_event = new \PixelYourSite\SingleEvent( '', '' );

		$_payload = array(
			'name' => $p_entry_data['event'] ?? '',
			'eventID'   => $p_entry_data['eventID'] ?? '',
			'woo_order' => $p_entry_data['wooOrder'] ?? '',
			'edd_order' => $p_entry_data['eddOrder'] ?? '',
			'pixelIds'  => $p_entry_data['ids'],
		);

		$_single_event->addParams( $p_entry_data['data'] );
		$_single_event->addPayload( $_payload );

		return $_single_event;
	}

	/**
	 * Checks conditions for entry processing.
	 *
	 * @param mixed $p_entry The entry data.
	 * @return boolean true if met, false otherwise.
	 */
	public function check_prerequisites( $p_entry ) {

		$_conditions_met = class_exists( '\PixelYourSite\SingleEvent' );

		return $_conditions_met;
	}
	/**
	 * Processes one ajax call entry.
	 * Reads the data and using PYS own functionality, sends the data to FB.
	 *
	 * @param mixed $p_entry Event data entry.
	 * @return boolean true if processed, false otherwise.
	 */
	public function process_entry( $p_entry ) {

		$_prerequisites_met = $this->check_prerequisites( $p_entry );

		if ( ! $_prerequisites_met ) {

			return false;
		}

		// Intended to fix event date time.
		//
		add_filter( 'pys_before_send_fb_server_event', array( $this, 'modify_fb_event' ), 20, 3 );

		// $p_entry is the DB row data.
		// $_ajax_call_data is all the data that was send via the AJAX call.
		// $_pys_call_data is the 'data' variable that PYS sends in the AJAX call.
		//
		$_ajax_call_data = unserialize( $p_entry['data'] );
		$_pys_call_data = $_ajax_call_data['action_data'];

		$_user_id = intval( isset( $p_entry['user_id'] ) ? $p_entry['user_id'] : 0 );
		$_user_role = sanitize_text_field( isset( $_pys_call_data['user_role'] ) ? $_pys_call_data['user_role'] : '' );
		$_cookies = $_ajax_call_data['cookie'];

		// From PYS facebook-server.php.
		//
		if ( ( isset( $_ajax_call_data['event'] ) ? $_ajax_call_data['event'] : '' ) === 'hCR' ) {

			// de mask completer registration event if it was hidden.
			//
			$_ajax_call_data['event'] = 'CompleteRegistration';
		}
		// Setting correct time.
		//
		$_pys_call_data['event_time'] = sanitize_text_field( $p_entry['event_time'] );

		// Setting _COOKIE.
		//
		$this->_cookies = $_COOKIE;
		$_COOKIE = $_cookies;

		$this->_current_processing_event_data[ $_pys_call_data['eventID'] ] = $_ajax_call_data;

		$_pys_fb_ins = \PixelYourSite\FacebookServer::instance();

		// Create event object and send it via PYS object.
		//
		$_single_event = $this->prepare_single_event( $_pys_call_data );
		$_pys_fb_ins->sendEventsNow( array( $_single_event ) );

		return true;
	}

	/**
	 * Runs before processing of one entry finished.
	 *
	 * @return void
	 */
	public function before_process_entry() {
	}

	/**
	 * Runs after processing of one entry finished.
	 *
	 * @return void
	 */
	public function after_process_entry() {

		$_COOKIE = $this->_cookies;

		remove_action( 'qacp_after_process_entry_pys', array( $this, 'after_process_entry' ) );
		remove_filter( 'pys_before_send_fb_server_event', array( $this, 'modify_fb_event' ), 20, 3 );
	}
	/**
	 * Runs after a batch processing is finished.
	 *
	 * @return void
	 */
	public function after_processing_batch() {
	}

	/**
	 * Modifies FB event time send by PYS to the correct data.
	 *
	 * @param mixed  $p_event PYS event.
	 * @param string $p_pixel_id pixel ID.
	 * @param string $p_event_id Event ID.
	 * @return mixed The modified PYS event object.
	 */
	public function modify_fb_event( $p_event, $p_pixel_id, $p_event_id ) {

		//
		// The p_event_id includes the pixel_id.
		// At this time, we support only a single pixel ID, but it seems as it would work anyway.
		// as there is not difference for this plugin purposes.
		//
		
		$_event_id_for_data = str_replace( $p_pixel_id, '', $p_event_id );
		$_event_time = $this->_current_processing_event_data[ $_event_id_for_data ]['event_time'] ?? $this->_current_processing_event_data[ $_event_id_for_data ]['current_time'] ?? '';

		$p_event->setEventTime( $_event_time );

		return $p_event;
	}

	/**
	 * Returns the data map for the AJAX call.
	 *
	 * @return array The data map.
	 */
	public function get_data_map() {

		$_data_map = array(
			'pixel' => array(
				'sanitize' => 'sanitize_text_field',
				'data_max_length' => 256,
			),
			'event' => array(
				'sanitize' => 'sanitize_text_field',
				'data_max_length' => 256,
			),
			'url' => array(
				'sanitize' => 'sanitize_text_field',
				'data_max_length' => 1024,
			),
			'data' => array(
				'sanitize' => 'sanitize_text_field',
				'fields' => array(
					'categories' => array(),
					'tags' => array(),
					'page_title' => array(),
					'post_type' => array(),
					'post_id' => array(
						'sanitize' => 'intval',
					),
					'content_ids' => array(
						'sanitize_as_array' => true,
						'sanitize' => 'intval',
					),
					'content_name' => array(),
					'content_type' => array(),
					'content_category' => array(),
					'category_name' => array(),
					'text' => array(),
					'plugin' => array(),
					'event_url' => array(),
					'user_role' => array(),
					'traffic_source' => array(),
					'event_time' => array(),
					'event_hour' => array(),
					'event_day' => array(),
					'event_month' => array(),
					'landing_page' => array(),
					'_fbp' => array(),
					'value' => array(),
					'currency' => array(),
					'product_price' => array(),
					'contents' => array(
						'sanitize_as_array' => true,
						'sanitize' => 'intval',
						'array_fields' => array(
							'id' => array(
								'sanitize' => 'intval',
							),
							'quantity' => array(
								'sanitize' => 'intval',
							),
						),
					),
				),
			),
			'ids' => array(
				'sanitize_as_array' => true,
				'sanitize' => 'sanitize_text_field',
				'array_max_length' => 30,
				'data_max_length' => 256,
			),
			'eventID' => array(
				'sanitize' => 'sanitize_text_field',
				'data_max_length' => 40,
			),
			'woo_order' => array(
				'sanitize' => 'sanitize_text_field',
				'data_max_length' => 256,
			),
			'edd_order' => array(
				'sanitize' => 'sanitize_text_field',
				'data_max_length' => 256,
			),
			'ajax_event' => array(
				'sanitize' => 'sanitize_text_field',
				'data_max_length' => 256,
			),
		);

		return $_data_map;
	}
	/**
	 * Registers the data map for the AJAX call.
	 *
	 * @return void
	 */
	public function register_data_map() {

		$_data_map = $this->get_data_map();

		$_processors = \QACP\Processors();
		$_processors->register_ajax_data_map( $this->_action_name, $_data_map );
	}

	/**
	 * Checks nonce.
	 *
	 * @return boolean true if nonce is valid, false otherwise.
	 */
	public function verify_ajax_nonce() {

		$_nonce = sanitize_text_field( wp_unslash( $_POST['ajax_event'] ?? '' ) );
		$_nonce_verified = wp_verify_nonce( $_nonce, 'ajax-event-nonce' );

		return $_nonce_verified;
	}
}
