<?php
/**
 * Options access functions.
 *
 * @package QACP
 */

namespace QACP\Options;

/**
 * General function to get option
 *
 * @param string $p_field Plugin's options field name.
 * @param string $p_default_value Default value in case field value doesn't exist.
 * @return mixed the option's value.
 */
function get_option_value( $p_field = '', $p_default_value = '' ) {

	$_options = get_options();

	$_val = isset( $_options[ $p_field ] ) ? $_options[ $p_field ] : $p_default_value;
	return $_val;
}

/**
 * Get all plugin's options
 *
 * @return array The options.
 */
function get_options() {

	$_options_prefix = get_options_prefix();
	$_options = get_option( $_options_prefix );

	if ( empty( $_options ) ) {

		$_options = array();
	}

	return $_options;
}


/**
 * Set an option.
 *
 * @param string $p_option_name option name.
 * @param string $p_option_val option value.
 * @return mixed false if no such option exists. Nothing otherwise.
 */
function set_option( $p_option_name = '', $p_option_val = '' ) {

	if ( empty( $p_option_name ) ) {

		return false;
	}

	$_options_prefix = get_options_prefix();
	$_all_options = get_options();
	$_all_options[ $p_option_name ] = $p_option_val;

	\update_option( $_options_prefix, $_all_options );
}

/**
 * Plugin's options prefix.
 *
 * @return string Plugin's option prefix.
 */
function get_options_prefix() {

	$_options_prefix = '_qacp_admin';
	return $_options_prefix;
}
/**
 * Returns true if backups are activated.
 *
 * @return bool true if backups are activated..
 */
function get_activate_backups() {

	$_val = get_option_value( 'activate_backups', '0' ) === '1';
	return $_val;
}
/**
 * Returns true if processed events should be deleted.
 *
 * @return bool true if processed events should be deleted.
 */
function get_delete_processed_calls() {

	$_val = get_option_value( 'delete_processed_calls', false ) === '1';
	return $_val;
}

/**
 * Returns true if plugin is enabled.
 *
 * @return bool true if plugin is enabled.
 */
function get_is_enabled() {

	return get_option_value( 'is_enabled', false ) === '1';
}

/**
 * Returns ajax actions to catch.
 *
 * @param boolean $p_as_array true if actions should be returns as a key => value array. false for string.
 * @return array The actions.
 */
function get_ajax_actions_to_process( $p_as_array = false ) {

	$_val = get_option_value( 'ajax_actions_to_process', '' );

	if ( $p_as_array ) {

		$_action_array = array();
		$_action_names = explode( PHP_EOL, $_val );

		if ( ! empty( $_action_names ) ) {

			foreach ( $_action_names as $action_name ) {

				$action_name = trim( $action_name );
				$_action_array[ $action_name ] = $action_name;
			}
		}

		$_val = $_action_array;
	}

	return $_val;
}

/**
 * Adds an action to the list of actions to process.
 *
 * @param string $p_action_name The action name.
 */
function remove_ajax_actions_to_process( $p_action_name = '' ) {

	$_actions = get_ajax_actions_to_process( true );
	$_index = array_search( $p_action_name, $_actions );
	unset( $_actions[ $_index ] );

	$_actions_str = implode( PHP_EOL, $_actions );

	set_option( 'ajax_actions_to_process', $_actions_str );
}

/**
 * Sets an action to the list of actions to process.
 *
 * @param string $p_action_name The action name.
 */
function set_ajax_actions_to_process( $p_action_name = '' ) {

	$_actions = get_ajax_actions_to_process( true );

	if ( ! in_array( $p_action_name, $_actions ) ) {

		$_actions[] = $p_action_name;
	}

	$_actions_str = implode( PHP_EOL, $_actions );

	set_option( 'ajax_actions_to_process', $_actions_str );
}

/**
 * Returns built-in processors.
 *
 * @return array The processors.
 */
function get_built_in_processors() {

	$_val = array(
		'pys' => 'pys_api_event',
	);

	return $_val;
}

/**
 * Sets a custom processor.
 *
 * @param string $p_processor_key The processor key.
 * @param string $p_processor_action The processor action.
 * @return void.
 */
function set_custom_processor( $p_processor_key = '', $p_processor_action = '' ) {

	$_processors = get_custom_processors();

	if ( ! isset( $_processors[ $p_processor_key ] ) ) {

		$_processors[ $p_processor_key ] = $p_processor_action;
	}

	set_option( 'custom_processors', $_processors );
}

/**
 * Removes a custom processor.
 *
 * @param string $p_processor_key The processor key.
 * @return void.
 */
function remove_custom_processor( $p_processor_key = '' ) {

	$_processors = get_custom_processors();

	if ( isset( $_processors[ $p_processor_key ] ) ) {

		unset( $_processors[ $p_processor_key ] );
	}

	set_option( 'custom_processors', $_processors );
}

/**
 * Returns selected processors.
 *
 * @return array The selected processors.
 */
function get_selected_processors() {

	$_val = get_option_value( 'selected_processors', array() );

	return $_val;
}

/**
 * Returns custom processors.
 *
 * @return array The custom processors.
 */
function get_custom_processors() {

	$_val = get_option_value( 'custom_processors', array() );

	$_decoded = is_string( $_val ) ? json_decode( $_val ) : $_val;

	return is_null( $_decoded ) ? $_val : (array) $_decoded;
}

/**
 * Returns all processors.
 *
 * @return array The processors.
 */
function get_all_active_processors() {

	$_selected_built_in_processors = get_selected_built_in_processors();
	$_custom_processors = array_keys( get_custom_processors() );

	$_all_processors = array_merge( $_custom_processors, $_selected_built_in_processors );

	return $_all_processors;
}

/**
 * Returns the cron interval.
 *
 * @return int The cron interval.
 */
function get_cron_action_interval() {

	$_options = get_options();
	$_val = intval( isset( $_options['qacp_cron_action_interval'] ) ? $_options['qacp_cron_action_interval'] : 60 );

	return $_val;
}

/**
 * Returns the number of events to process.
 *
 * @return int number of events to process.
 */
function get_cron_requests_num_to_process() {

	$_options = get_options();

	if ( ! isset( $_options['qacp_cron_requests_num_to_process'] )
		|| empty( $_options['qacp_cron_requests_num_to_process'] )
		|| ! is_numeric( $_options['qacp_cron_requests_num_to_process'] ) ) {

		return false;
	}

	$_val = intval( $_options['qacp_cron_requests_num_to_process'] );

	return $_val;
}
