<?php
/**
 * Plugin's options page functionality and creation.
 *
 * @package QACP
 */

namespace QACP\Admin;

/**
 * Initializes Options functionality.
 *
 * @return void
 */
function init() {

	if ( apply_filters( 'qacp\admin\show_admin', true ) ) {

		add_action( 'admin_enqueue_scripts', __NAMESPACE__ . '\enqueue_admin_styles' );
		add_action( 'admin_menu', __NAMESPACE__ . '\create_menu' );
		add_action( 'wp_ajax_qacp_reregister_maps', __NAMESPACE__ . '\handle_reregister_maps' );
		add_action( 'wp_ajax_qacp_add_default_actions', __NAMESPACE__ . '\handle_add_default_actions' );
	}
}

/**
 * Handles the reregistering of the maps.
 *
 * @return void
 */
function handle_add_default_actions() {

	check_ajax_referer( 'qacp_ajax_nonce', 'nonce' );

	$_qacp = qacp();
	$_qacp->add_default_actions();

	wp_send_json_success( array( 'message' => __( 'Default actions added successfully!', 'queue-ajax-calls-performantly' ) ) );
}


/**
 * Handles the reregistering of the maps.
 *
 * @return void
 */
function handle_reregister_maps() {

	check_ajax_referer( 'qacp_ajax_nonce', 'nonce' );

	$_qacp = qacp();
	$_qacp->register_data_maps();

	wp_send_json_success( array( 'message' => __( 'Maps registered successfully!', 'queue-ajax-calls-performantly' ) ) );
}


/**
 * Enqueues admin style file.
 *
 * @param string $p_hook Option page hook name.
 * @return void
 */
function enqueue_admin_styles( $p_hook ) {

	if ( 'settings_page_qacp-options-page' === $p_hook ) {

		$_qacp = \QACP::instance();
		$_plugin_version = $_qacp->get_version();

		wp_enqueue_style( 'admin-styles', plugin_dir_url( __FILE__ ) . 'assets/css/admin-styles.css', array(), $_plugin_version );

		wp_enqueue_script( 'qacp-admin-scripts', plugin_dir_url( __FILE__ ) . 'assets/js/admin-scripts.js', array( 'jquery' ), $_plugin_version, true );

		// Pass AJAX URL and nonce to the script.
		wp_localize_script(
			'qacp-admin-scripts',
			'qacp_ajax',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'qacp_ajax_nonce' ),
			)
		);
	}
}

/**
 * Registers the WP menu plugin's options page.
 *
 * @return void
 */
function create_menu() {

	// Create new submenu under 'Settings'.
	add_submenu_page( 'options-general.php', 'Queue Ajax Calls Performantly Settings', 'Queue Ajax Calls Performantly Settings', 'manage_options', 'qacp-options-page', __NAMESPACE__ . '\settings_page' );

	// Call register settings function.
	add_action( 'admin_init', __NAMESPACE__ . '\register_settings' );
}

/**
 * Sanitizes the options.
 *
 * @param array $p_input The input to sanitize.
 * @return array Sanitized input.
 */
function qacp_sanitize_options( $p_input ) {

	$_new_input = array();
	foreach ( $p_input as $key => $value ) {

		if ( is_array( $value ) ) {

			$_new_input[ $key ] = array_map( 'sanitize_text_field', $value );
		} else {
			$_new_input[ $key ] = sanitize_textarea_field( $value );
		}
	}

	return $_new_input;
}
/**
 * Registers the plugin's settings.
 */
function register_settings() {

	$_options_prefix = \QACP\Options\get_options_prefix();
	// Register our settings.
	register_setting(
		'qacp-settings-group',
		$_options_prefix,
		array(
			'sanitize_callback' => __NAMESPACE__ . '\qacp_sanitize_options',
			'type'              => 'array',
		)
	);
}

/**
 * Returns allowed HTML for wp_kses fitlering.
 *
 * @return array wp_kses allowed HTML array.
 */
function get_allowed_html() {

	$_allowed_html = array(
		'div' => array(
			'class' => array(),
			'id'    => array(),
			'style' => array(),
			'data-tip' => array(),
		),
		'input' => array(
			'type'    => array(),
			'name'    => array(),
			'value'   => array(),
			'checked' => array(),
			'class'   => array(),
			'id'      => array(),
			'style'   => array(),
		),
		'textarea' => array(
			'type'    => array(),
			'name'    => array(),
			'value'   => array(),
			'checked' => array(),
			'placeholder' => array(),
			'a' => array(
				'href'   => array(),
				'title'  => array(),
				'target' => array(),
				'class'  => array(),
				'id'     => array(),
				'style'  => array(),
				'rel'    => array(),
			),
		),
		'strong' => array(),
		'label' => array(),
		'a' => array(
			'href'   => array(),
			'title'  => array(),
			'target' => array(),
			'class'  => array(),
			'id'     => array(),
			'style'  => array(),
			'rel'    => array(),
		),
		'span' => array(
			'class' => array(),
			'id'    => array(),
			'style' => array(),
			'aria-label' => array(),
			'tabindex' => array(),
			'data-tip' => array(),
		),
		'option' => array(
			'selected' => array(),
			'value' => array(),
		),
	);

	return $_allowed_html;
}

/**
 * Creating the HTML for the settings page.
 *
 * @return void
 */
/**
 * Creating the HTML for the settings page.
 *
 * @return void
 */
function settings_page() {

	$_title         = esc_html__( 'Queue Ajax Calls Performantly Settings', 'queue-ajax-calls-performantly' );
	$_options_prefix = \QACP\Options\get_options_prefix();

	$_is_enabled_val     = \QACP\Options\get_is_enabled();
	$_is_enabled_checked = $_is_enabled_val ? 'checked' : '';

	$_qacp_activate_backups_checked = \QACP\Options\get_activate_backups() ? 'checked' : '';
	$_qacp_delete_processed_calls_checked = \QACP\Options\get_delete_processed_calls() ? 'checked' : '';

	$_cron_requests_num_to_process = \QACP\Options\get_cron_requests_num_to_process();
	$_cron_action_interval = \QACP\Options\get_cron_action_interval();

	// Retrieve ajax options to be catched and processed.
	//
	$_ajax_actions_to_process = \QACP\Options\get_ajax_actions_to_process();

	$_selected_processors = \QACP\Options\get_selected_processors();
	$_all_processors = \QACP\Options\get_custom_processors();

	// Get the plugin data for the specified plugin file.
	//
	$_qacp = \QACP::instance();
	$_plugin_version = $_qacp->get_version();

	$_description = esc_html__( 'The Queue Ajax Calls Performantly plugin (QACP) is a simple solution to allow handling Ajax calls to WordPress much more efficiently. It does so by queuing the calls using an MU plugin and bypassing loading most of the WP system. Moreover, it allows processing these calls as a batch. Most useful for non-real-time Ajax calls like tracking, page views, conversion data, and so on.', 'queue-ajax-calls-performantly' );

	$_allowed_html = get_allowed_html();

	$_plugin_url = plugin_dir_url( dirname( __DIR__, 1 ) );
	$_how_to_url = $_plugin_url . 'qacp-how-to.txt';

	?>
	<div class="wrap qacp-settings-wrap">
		<h1><?php echo esc_html( $_title ); ?></h1>
		<h2><?php echo esc_html( $_description ); ?></h2>
		<div class='general-info'>
			<p>
				<?php
				echo esc_html__( 'Plugin Version', 'queue-ajax-calls-performantly' ) . ': ' . esc_html( $_plugin_version );
				?>
				// Developed By <a href='https://profiles.wordpress.org/eddr'>eddr</a>
			</p>
		</div>

		<form method="post" action="options.php">
			<?php settings_fields( 'qacp-settings-group' ); ?>
			<?php do_settings_sections( 'qacp-settings-group' ); ?>
			
			<input type="hidden" style="display: none;" class="visually-hidden" name="_qacp_admin[custom_processors]" value="<?php echo esc_attr( json_encode( $_all_processors ) ); ?>" />

			<table class="form-table">
				<tr valign="top">
					<th scope="row"><?php esc_html_e( 'Enabled', 'queue-ajax-calls-performantly' ); ?></th>
					<td>
						<label class='visually-hidden' for='qacp_is_enabled'><?php echo esc_html__( 'Enabled', 'queue-ajax-calls-performantly' ); ?></label>
						<input type="checkbox" value="1" id='qacp_is_enabled' name="<?php echo esc_attr( $_options_prefix ); ?>[is_enabled]" <?php echo esc_attr( $_is_enabled_checked ); ?> />
					</td>
				</tr>
				<tr valign="top">
					<th scope="row"><?php esc_html_e( 'Options', 'queue-ajax-calls-performantly' ); ?></th>
					<td>
						<div>
							<div class='qacp-settings-line'>
								<div class='setting-text'>
									<label for='qacp_cron_action_interval'><?php echo esc_html__( 'Cron Action Interval (seconds)', 'queue-ajax-calls-performantly' ); ?></label>
									<p class='settings-line-desc'><?php esc_html_e( 'How long between each batch processing of calls', 'queue-ajax-calls-performantly' ); ?></p>
								</div>
								<div class='setting-value'>
									<input type="number" id='qacp_cron_action_interval' name="<?php echo esc_attr( $_options_prefix ); ?>[qacp_cron_action_interval]" value='<?php echo esc_attr( $_cron_action_interval ); ?>' />
								</div>
							</div>
							<div class='qacp-settings-line'>
								<div class='setting-text'>
									<label for='qacp_cron_requests_num_to_process'><?php echo esc_html__( 'Number of requests to process each run', 'queue-ajax-calls-performantly' ); ?></label>
									<p class='settings-line-desc'><?php esc_html_e( 'How many calls in each batch', 'queue-ajax-calls-performantly' ); ?></p>
								</div>
								<div class='setting-value'>
									<input type="number" id='qacp_cron_requests_num_to_process' name="<?php echo esc_attr( $_options_prefix ); ?>[qacp_cron_requests_num_to_process]" value='<?php echo esc_attr( $_cron_requests_num_to_process ); ?>' />
								</div>
							</div>
							<div class='qacp-settings-line'>
								<div class='setting-text'>
									<label for='qacp_activate_backups'><?php echo esc_html__( 'Keep backups of all calls', 'queue-ajax-calls-performantly' ); ?></label>
									<p class='settings-line-desc'><?php esc_html_e( 'You can find the backups under the path:', 'queue-ajax-calls-performantly' ); ?> <span style='display:inline-block;'>/qcap/backups</span></p>
								</div>
								<div class='setting-value'>
									<input type="checkbox" value="1" id='qacp_activate_backups' name="<?php echo esc_attr( $_options_prefix ); ?>[activate_backups]" <?php echo esc_attr( $_qacp_activate_backups_checked ); ?> />
								</div>
							</div>
							<div class='qacp-settings-line'>
								<div class='setting-text'>
									<label for='qacp_delete_processed_calls'><?php echo esc_html__( 'Delete processed calls', 'queue-ajax-calls-performantly' ); ?></label>
									<p class='settings-line-desc'><?php esc_html_e( 'Delete calls that were processed', 'queue-ajax-calls-performantly' ); ?></p>
								</div>
								<div class='setting-value'>
									<input type="checkbox" value="1" id='qacp_delete_processed_calls' name="<?php echo esc_attr( $_options_prefix ); ?>[delete_processed_calls]" <?php echo esc_attr( $_qacp_delete_processed_calls_checked ); ?> />
								</div>
							</div>
							<div class='qacp-settings-line qacp-actions-settings'>
								<div class='setting-text'>
									<label for='qacp_ajax_actions_to_process'><?php echo esc_html__( 'Ajax actions to catch and process', 'queue-ajax-calls-performantly' ); ?></label>
									<p class='settings-line-desc'><?php echo wp_kses( _x( 'Enter the Ajax actions that should be caught and processed, one per line in the form of a {action_name}. In order to work properly for actions without a built-in support ( see below ), a processor class should be loaded.', 'options', 'queue-ajax-calls-performantly' ), $_allowed_html ); ?></p>
									<p class='settings-line-desc'><?php echo '<a target="_blank" href=' . esc_url( $_how_to_url ) . '>Please check for more details.</a>'; ?></p>
									
								</div>
								<div class='setting-value'>
									<textarea id='qacp_ajax_actions_to_process' name="<?php echo esc_attr( $_options_prefix ); ?>[ajax_actions_to_process]" rows="5" cols="50"><?php echo esc_textarea( $_ajax_actions_to_process ); ?></textarea>
								</div>
							</div>
							<div class='qacp-settings-line'>
								<div class='setting-text'>
									<label for='qacp_selected_processors'><?php echo esc_html__( 'Enable processors', 'queue-ajax-calls-performantly' ); ?></label>
									<p class='settings-line-desc'><?php echo wp_kses( _x( 'Select to de/active these processors', 'options', 'queue-ajax-calls-performantly' ), $_allowed_html ); ?></p>
								</div>
								<div class='setting-value'>
									<select id='qacp_selected_processors' name="<?php echo esc_attr( $_options_prefix ); ?>[selected_processors][]" multiple>
										<?php

										if ( ! empty( $_all_processors ) ) {

											foreach ( $_all_processors as $processor_key => $action_name ) {

												$_selected = in_array( $processor_key, $_selected_processors ) ? 'selected="selected"' : '';

												$_option_html = "<option value='$processor_key' $_selected>$action_name</option>";

												echo wp_kses( $_option_html, $_allowed_html );
											}
										}
										?>
									</select>
								</div>
							</div>
							<div class='qacp-settings-line qacp-settings-buttons'>
								<div class='setting-text'>
									<label for='qacp_reregister_maps_button'><?php echo esc_html__( 'Reregister Maps', 'queue-ajax-calls-performantly' ); ?></label>
									<p class='settings-line-desc'><?php echo wp_kses( _x( 'Updates the processors\' data maps', 'options', 'queue-ajax-calls-performantly' ), $_allowed_html ); ?></p>
								</div>
								<div class='setting-value'>
									<button id="qacp_reregister_maps_button" class="button button-primary"><?php esc_html_e( 'Update Maps', 'queue-ajax-calls-performantly' ); ?></button>
								</div>
							</div>
							<div class='qacp-settings-line qacp-settings-buttons'>
								<div class='setting-text'>
									<label for='qacp_add_default_actions_button'><?php echo esc_html__( 'Add Default Actions', 'queue-ajax-calls-performantly' ); ?></label>
									<p class='settings-line-desc'><?php echo wp_kses( _x( 'Adds the default actions to process', 'options', 'queue-ajax-calls-performantly' ), $_allowed_html ); ?></p>
								</div>
								<div class='setting-value'>
									<button id="qacp_add_default_actions_button" class="button button-primary"><?php esc_html_e( 'Add Actions', 'queue-ajax-calls-performantly' ); ?></button>
								</div>
							</div>
						</div>
					</td>
				</tr>
			</table>

			<?php submit_button(); ?>

		</form>
	</div>
	<?php
}
