<?php
/**
 * Provides functionality for backuping data.
 *
 * @package QACP
 */

namespace QACP\DBB;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

define( 'QACP_DB_TABLE', 'qacp_queue' );

//
// Hook to Cron.
//
add_action( 'qacp_db_backup_hook', __NAMESPACE__ . '\handle_old_records' );

/**
 * Returns the QACP DB table name.
 *
 * @return string table name.
 */
function get_table_name() {

	global $wpdb;
	$_table_name = $wpdb->prefix . QACP_DB_TABLE;

	return $_table_name;
}


/**
 * Deletes old records.
 *
 * @param integer $p_months_old Minimum age of operations for deletion.
 * @param array   $p_args additional arguments.
 * @return int number of affected rows.
 */
function delete_data( $p_months_old = 1, $p_args = array() ) {

	global $wpdb;

	$_keep_deleting = true;

	$_table_name = get_table_name();
	$_number_of_rows_to_delete = 20000;
	// $_data = isset( $p_args['data'] ) ?

	$_total_deleted = 0;

	while ( $_keep_deleting ) {

		// WPCS: cache ok, db call ok.
		$_delete_result = $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}qacp_queue WHERE sent IS NOT NULL ORDER BY `id` ASC LIMIT %d;", $_number_of_rows_to_delete ), ARRAY_A ); // WPCS: cache ok, db call ok.
		$_keep_deleting = false;// $_delete_result === $_number_of_rows_to_delete;
		if ( is_numeric( $_delete_result ) ) {

			$_total_deleted += $_delete_result;
		}
	}

	// optimize.
	//

	// phpcs:ignore
	$wpdb->query( $wpdb->prepare( "OPTIMIZE TABLE {$wpdb->prefix}qacp_queue;" ), ARRAY_A ); // WPCS: db call ok.

	return $_delete_result;
}
/**
 * Returns queue data to be backup to files.
 *
 * @param array $p_args additional arguments.
 * @return array rows to backup.
 */
function get_email_log_data( $p_args = array() ) {

	global $wpdb;

	$_table_name = get_table_name();
	$_number_of_rows_to_delete = 20000;

	$results = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}qacp_queue WHERE sent IS NOT NULL ORDER BY `id` ASC LIMIT %d;", $_number_of_rows_to_delete ), ARRAY_A ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

	return $results;
}
/**
 * Writes data to a CSV file.
 *
 * @param string $p_filename file name.
 * @param array  $p_data data to write to CSV.
 * @param array  $p_args additional arguments.
 * @return void
 */
function write_to_csv( $p_filename = 'email.log.csv', $p_data = array(), $p_args = array() ) {

	$p_filename = path_is_absolute( $p_filename ) ? $p_filename : __DIR__ . "/$p_filename";
	// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
	$_fp = fopen( $p_filename, 'w' );

	foreach ( $p_data as $fields ) {

		fputcsv( $_fp, $fields );
	}
	// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
	fclose( $_fp );
}
/**
 * Writes to a CSV stream.
 *
 * @param string $p_filename file name.
 * @param array  $p_data data to write to CSV.
 * @param array  $p_args additional arguments.
 * @return object The file pointer.
 */
function write_to_csv_stream( $p_filename = 'email.log.csv', $p_data = array(), $p_args = array() ) {

	$php_memory = '0';
	$_fp = fopen( "php://temp/maxmemory:$php_memory", 'w' );

	foreach ( $p_data as $fields ) {

		fputcsv( $_fp, $fields );
	}

	return $_fp;
}
/**
 * Writes the stream to a zip file.
 *
 * @param string $p_filename file name.
 * @param [type] $p_stream data stream.
 * @param array  $p_args additional arguments.
 * @return void
 */
function write_stream_to_zip( $p_filename = 'email.log.csv.zip', $p_stream = null, $p_args = array() ) {

	$p_filename = path_is_absolute( $p_filename ) ? $p_filename : __DIR__ . "/backups/$p_filename";

	$_zipfile = $p_filename;
	$_zip = new \ZipArchive();
	$_zip->open( $_zipfile, \ZipArchive::CREATE | \ZipArchive::OVERWRITE );
	$_content = stream_get_contents( $p_stream, -1, 0 );

	$_zip->addFromString( str_replace( '.zip', '', $p_filename ), $_content );

	$_zip->close();
}

/**
 * Creates the backup.
 *
 * @return void
 */
function create_backup() {

	$_fn = 'email_' . gmdate( 'd-m-Y' ) . '.log.csv';
	$_data = get_email_log_data();
	write_to_csv( $_fn, $_data );
}

/**
 * Handles old records. Meaning backup them if required and then deletes.
 */
function handle_old_records() {

	$_data = array();
	if ( ( \QACP\Options\get_activate_backups() ) ) {

		$_fn = 'email_' . gmdate( 'd-m-Y' ) . '.log.csv';
		$_data = get_email_log_data();
		$_fp = write_to_csv_stream( $_fn, $_data );

		// creates the zip file. Default is the plugin directory.
		//
		write_stream_to_zip( 'table.bck_' . gmdate( 'd-m-Y' ) . '.csv.zip', $_fp );
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
		fclose( $_fp );
	}

	// deletes the old records.
	//
	if ( ( \QACP\Options\get_delete_processed_calls() ) ) {
		delete_data( 1, array( 'data' => $_data ) );
	}
}
