<?php
/**
 * Main class QACP file.
 *
 * @package QACP
 */

defined( 'ABSPATH' ) || die();

if ( ! class_exists( 'QACP' ) ) {

	/**
	 * The main QACP class.
	 */
	class QACP {

		/**
		 * Instances.
		 *
		 * @var array
		 */
		private static $_instances = array();
		/**
		 * Plugin directory path.
		 *
		 * @var string The path.
		 */
		private $_plugin_path;

		/**
		 * The Processors Handler.
		 *
		 * @var array The instance.
		 */
		private $_processors = null;

		/**
		 * The constructor.
		 */
		protected function __construct() {
		}

		/**
		 * Singletons should not be cloneable.
		 */
		protected function __clone() { }

		/**
		 * Singletons should not be restorable from strings.
		 *
		 * @throws \Exception Cannot unserialize a singleton.
		 */
		public function __wakeup() {

			throw new \Exception( 'Cannot unserialize a singleton.' );
		}

		/**
		 * Get the singleton instance.
		 *
		 * @return QACP
		 */
		public static function instance() {

			$_class_name = static::class;

			if ( ! isset( self::$_instances[ $_class_name ] ) ) {

				self::$_instances[ $_class_name ] = new static();
				self::$_instances[ $_class_name ]->init();
			}

			return self::$_instances[ $_class_name ];
		}

		/**
		 * Initialization.
		 *
		 * @return void
		 */
		public function init() {

			$this->_plugin_path = plugin_dir_path( __FILE__ );

			$this->define_constants();
			$this->include_files();

			$this->init_hooks();

			\QACP\MUPH\maybe_update();

			// Loading the event processors.
			//
			$this->_processors = new \QACP\Processors();

			if ( is_admin() ) {

				\QACP\Admin\init();
			}
		}

		/**
		 * Initialize hooks.
		 */
		private function init_hooks() {

			// Activation and deactivation hooks.
			//
			register_activation_hook( __DIR__ . '\qacp.php', array( $this, 'activate' ) );
			register_deactivation_hook( __DIR__ . '\qacp.php', array( $this, 'deactivate' ) );

			// WooCommerce compatibility.
			//
			add_action( 'before_woocommerce_init', array( $this, 'hpos_compatibility' ) );

			// Plugins loaded.
			//
			add_action( 'plugins_loaded', array( $this, 'plugins_loaded' ) );
		}

		/**
		 * Defines plugin's constants.
		 *
		 * @return void
		 */
		private function define_constants() {

			if ( ! defined( 'QACP_EMBEDDED' ) ) {

				define( 'QACP_EMBEDDED', false );
			}

			if ( ! defined( 'QACP_PLUGIN_PATH' ) ) {

				define( 'QACP_PLUGIN_PATH', $this->_plugin_path );
			}
		}
		/**
		 * Includes all plugin's files.
		 *
		 * @return void
		 */
		public function include_files() {

			$_dir = $this->_plugin_path;

			require_once $_dir . 'db.php';
			require_once $_dir . 'options.php';
			require_once $_dir . '/processors/class-processor-base.php';
			require_once $_dir . 'class-processors.php';
			require_once $_dir . 'process-queue.php';
			require_once $_dir . 'mu-plugin-handler.php';
			require_once $_dir . 'cron-scheduler.php';
			require_once $_dir . 'db-backup.php';

			if ( is_admin() ) {

				require_once $_dir . 'inc/admin/options-page.php';
			}
		}

		/**
		 * Basic operations upon activation.
		 *
		 * @return void
		 */
		public function activate() {

			\QACP\DB\create_table();
			\QACP\MUPH\create_mu_plugin();

			$this->register_data_maps();
		}

		/**
		 * Registers the data maps.
		 *
		 * @return void
		 */
		public function add_default_actions() {

			$_default_actions = \QACP\Options\get_built_in_processors();

			foreach ( $_default_actions as $_processor_key => $_action_name ) {

				$this->_processors->set_action( $_processor_key, $_action_name );
			}

			$this->register_data_maps();
		}

		/**
		 * Registers the data maps.
		 *
		 * @return void
		 */
		public function register_data_maps() {

			$this->_processors->register_data_maps();
		}
		/**
		 * Runs upon deactivation, removing the MU plugin file.
		 *
		 * @return void
		 */
		public function deactivate() {

			\QACP\MUPH\remove_mu_plugin();
		}

		/**
		 * Checks whether WC is activated.
		 *
		 * @return boolean
		 */
		public function is_woocommerce_activated() {

			return class_exists( 'woocommerce' );
		}


		/**
		 * HPOS compatibility declaration
		 * This plugin does not use WP default table by itself, nor any specific WC data specifically
		 */
		public function hpos_compatibility() {

			if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
			}
		}

		/**
		 * Loading ActionScheduler
		 */
		public function plugins_loaded() {

			if ( ! $this->is_woocommerce_activated()
				&& ( ! function_exists( 'as_schedule_single_action' ) ) ) {

				require_once plugin_dir_path( __FILE__ ) . '/libraries/action-scheduler/action-scheduler.php';
			}

			// Loading text domain files.
			//
			$this->load_textdomain();
		}

		/**
		 * Returns plugin's version.
		 *
		 * @return string plugin's version.
		 */
		public function get_version() {

			$_plugin_version = '1.1.3';

			return $_plugin_version;
		}

		/**
		 * Returns current plugin's directory.
		 *
		 * @return string The path.
		 */
		public function get_plugin_path() {

			return $this->_plugin_path;
		}

		/**
		 * Loads plugin's text domain files.
		 *
		 * @return void
		 */
		public function load_textdomain() {

			load_plugin_textdomain(
				'queue-ajax-calls-performantly',
				false,
				dirname( plugin_basename( __FILE__ ) ) . '/languages/'
			);
		}
	}
}
