<?php
/**
 * Processors functionality.
 *
 * @package QACP
 */

namespace QACP;

/**
 * Processors Class for handling initializing and creating object.
 */
class Processors {


	/**
	 * Action key map.
	 *
	 * @var array The action key map.
	 */
	private $_action_key_map = array();

	/**
	 * Processors that have been initialized.
	 *
	 * @var array The processors.
	 */
	private $_initialized_processors = array();

	/**
	 * Constructor.
	 */
	public function __construct() {

		$this->init_processors();
	}

	/**
	 * Returns the action key by action name.
	 *
	 * @param string $p_action_name The action name.
	 * @return string The action key.
	 */
	public function get_key_by_action( $p_action_name ) {

		return $this->_action_key_map[ $p_action_name ] ?? '';
	}
	/**
	 * Creates instances of the selected processors.
	 * It is not required to use them at all.
	 *
	 * @return array The selected processors instances.
	 */
	public function init_processors() {

		// Initializes selected built-in processors.
		//
		$_selected_processors = \QACP\Options\get_custom_processors( true );
		$_selected_processors = apply_filters( 'qacp/processors/active_processors', $_selected_processors );
		$_ajax_actions_to_process = \QACP\Options\get_ajax_actions_to_process( true );

		$_obs = array();
		if ( ! empty( $_selected_processors ) ) {

			foreach ( $_selected_processors as $_class_name => $_action_name ) {

				// If action is not activated, do no load.
				//
				if ( ! in_array( $_action_name, $_ajax_actions_to_process, true ) ) {

					continue;
				}

				$_loaded = $this->load_processor_class_file( $_class_name );

				if ( $_loaded ) {

					$_obs[ $_class_name ] = $this->generate_processor_instance_by_action_key( $_class_name );

					$this->_action_key_map[ $_action_name ] = $_class_name;
				}
			}

			$this->_initialized_processors = $_obs;
		}

		return $_obs;
	}

	/**
	 * Loads the processor class file by a processor key.
	 *
	 * @param string $p_key The processor key.
	 * @return bool true if found and loaded, false otherwise.
	 */
	public function load_processor_class_file( $p_key ) {

		// Retrieve any custom processor directories saved in the database.
		$_custom_dirs = get_option( 'qacp_custom_processor_dirs', array() );

		// Construct the filename based on the processor key.
		//
		$_processor_class_filename = 'class-' . $p_key . '.php';

		$_custom_dirs[] = plugin_dir_path( __FILE__ ) . '/processors';

		// 1. Check each custom directory first.
		//
		foreach ( $_custom_dirs as $dir ) {

			// Ensure we handle trailing slashes correctly.
			//
			$dir = untrailingslashit( $dir );
			$_filepath = $dir . '/' . $_processor_class_filename;

			if ( file_exists( $_filepath ) ) {

				require_once $_filepath;
				return true;
			}
		}

		return false;
	}

	/**
	 * Register a custom directory containing processor class files.
	 * Stores multiple directories in an array option named "qacp_custom_processor_dirs".
	 *
	 * @param string $p_directory_path Absolute path to the custom processor directory.
	 * @return bool True if the directory was added or already exists, false otherwise.
	 */
	public function register_custom_processor_directory( $p_directory_path ) {

		// Retrieve existing directories from the database (array).
		//
		$_dirs = get_option( 'qacp_custom_processor_dirs', array() );

		// Ensure it’s not already in the list.
		//
		if ( ! in_array( $p_directory_path, $_dirs, true ) ) {

			$_dirs[] = $p_directory_path;
			return update_option( 'qacp_custom_processor_dirs', $_dirs, false );
		}

		return true; // Directory already registered.
	}

	/**
	 * Unregister a previously registered custom directory of processor class files.
	 *
	 * @param string $p_directory_path Absolute path to the custom processor directory.
	 * @return bool True if the directory was removed, false otherwise.
	 */
	public function unregister_custom_processor_directory( $p_directory_path ) {

		$_dirs = get_option( 'qacp_custom_processor_dirs', array() );
		$_key  = array_search( $p_directory_path, $_dirs, true );

		if ( false !== $_key ) {

			unset( $_dirs[ $_key ] );
			$_dirs = array_values( $_dirs );
			return update_option( 'qacp_custom_processor_dirs', $_dirs, false );
		}
		return false; // Directory wasn't found.
	}

	/**
	 * Add a new action to the list processed by QACP.
	 * Format: "{p_processor_key}:{p_action_name}" per line in the option storing actions.
	 *
	 * @param string $p_action_name   The AJAX action name.
	 * @param string $p_class_name The processor's class name that will be used for loading.
	 * @return void.
	 */
	public function set_action( $p_action_name, $p_class_name = null ) {

		\QACP\Options\set_ajax_actions_to_process( $p_action_name );

		if ( ! empty( $p_class_name ) ) {

			\QACP\Options\set_custom_processor( $p_class_name, $p_action_name );
		}
	}

	/**
	 * Remove an action from the list processed by QACP.
	 *
	 * @param string $p_action_name   The AJAX action name.
	 * @param string $p_class_name The processor's class name that will be used for loading.
	 * @return void.
	 */
	public function remove_action( $p_action_name, $p_class_name ) {

		\QACP\Options\remove_ajax_actions_to_process( $p_action_name );

		if ( ! empty( $p_class_name ) ) {

			\QACP\Options\remove_custom_processor( $p_class_name );
		}
	}


	/**
	 * Returns all allowed actions' names.
	 *
	 * @return array The actions' names.
	 */
	public function get_all_actions() {

		$_actions_str = \QACP\Options\get_ajax_actions_to_process();
		$_actions = array_map( 'trim', explode( PHP_EOL, $_actions_str ) );

		$_actions = apply_filters( 'qacp/processors/actions', $_actions );

		return $_actions;
	}
	/**
	 * Identifying the proper Processor class for the action.
	 * This is based on the plugin's GUI actions list.
	 *
	 * @param string $p_action_name The action name.
	 * @return string The processor name. Empty string if none found;
	 */
	public function identify_processor( $p_action_name ) {

		/**
		 * Filter to add your own processor.
		 */
		$_processor_key = apply_filters( 'qacp/processors/identify_processor', null, $p_action_name );

		if ( empty( $_processor_key ) ) {
			// Loading the event processors.
			//
			$_actions = $this->get_all_actions();
			$_processor_key = '';

			foreach ( $_actions as $processor ) {

				list( $_current_processor_key, $_processor_action ) = array_map( 'trim', explode( ':', $processor ) );

				if ( $_processor_action === $p_action_name ) {

					$_processor_key = $_current_processor_key;
					break;
				}
			}
		}
		return $_processor_key;
	}
	/**
	 * Returns QACP processor class instance, using the action name.
	 *
	 * @param string $p_action_name The action name.
	 * @return object Processor instance.
	 */
	public function generate_processor_instance_by_action_name( $p_action_name ) {

		/**
		 * Filter to add your own processor.
		 * Currently that's the way to add your own processor.
		 */
		$_ob = apply_filters( 'qacp/processors/instance_by_action_name', null, $p_action_name );

		if ( empty( $_ob ) ) {

			$_processor_key = $this->identify_processor( $p_action_name );
			$_ob = null;

			if ( ! empty( $_processor_key ) ) {

				$_class_name = '\\QACP\\Processors\\' . $_processor_key;
				$_ob = new $_class_name();
			}
		}

		return $_ob;
	}

	/**
	 * Returns QACP processor class instance, using the action key.
	 *
	 * @param string $p_action_key The action key.
	 * @return object Processor instance.
	 */
	public function generate_processor_instance_by_action_key( $p_action_key ) {

		$_ob = null;

		if ( ! empty( $p_action_key ) ) {

			if ( isset( $this->_initialized_processors[ $p_action_key ] ) ) {

				return $this->_initialized_processors[ $p_action_key ];
			}

			$_class_name = '\\QACP\\Processors\\' . $p_action_key;
			$_ob = new $_class_name();
		}

		return $_ob;
	}

	/**
	 * Sanitize a single field.
	 *
	 * @param string $p_value The value to sanitize.
	 * @param string $p_sanitation_function The sanitization function to use.
	 * @param int    $p_max_length The maximum length of the value.
	 * @return string The sanitized value.
	 */
	public function sanitize_single_field( $p_value, $p_sanitation_function = null, $p_max_length = null ) {

		// Sanitize single value.
		//
		$_value_substring = empty( $p_max_length ) ?
							$p_value :
							substr( $p_value, 0, $p_max_length );

		$_sanitized_value = empty( $p_sanitation_function ) ?
							$_value_substring :
							call_user_func( $p_sanitation_function, $_value_substring );

		return $_sanitized_value;
	}

	/**
	 * Recursively applies a sanitization function to all elements in an array,
	 * including nested arrays.
	 *
	 * @param callable $p_callback The sanitization function to apply.
	 * @param array    $p_array The array to sanitize.
	 * @return array The sanitized array.
	 */
	public function recursive_array_map( $p_callback, $p_array ) {

		$result = array();

		foreach ( $p_array as $key => $value ) {

			if ( is_array( $value ) ) {

				// Recursively sanitize nested arrays.
				//
				$result[ $key ] = $this->recursive_array_map( $p_callback, $value );
			} else {

				// Apply sanitization function to scalar values.
				//
				$result[ $key ] = call_user_func( $p_callback, $value );
			}
		}

		return $result;
	}

	/**
	 * Registers all data maps for the initialized processors.
	 *
	 * @return void
	 */
	public function register_data_maps() {

		foreach ( $this->_initialized_processors as $p_processor_key => $processor_ob ) {

			$_data_map = $processor_ob->get_data_map();
			$_action_name = $processor_ob->get_action_name();

			$this->register_ajax_data_map( $_action_name, $_data_map );
		}
	}

	/**
	 * Register an array of AJAX $_POST data variables with sanitization functions and save them to the database.
	 *
	 * @param string $p_ajax_action_name The name of the AJAX action.
	 * @param array  $p_data_map An associative array where keys are $_POST variable names
	 *                         and values are arrays containing WP sanitization functions and max length,
	 *                         or nested arrays for complex structures.
	 * @return bool True if the option was updated successfully, false otherwise.
	 */
	public function register_ajax_data_map( $p_ajax_action_name, $p_data_map = null ) {

		// Validate that $p_data_map is an associative array.
		//
		if ( ! is_array( $p_data_map )
			|| empty( $p_data_map ) ) {

			if ( ! empty( $this->_initialized_processors[ $p_ajax_action_name ] ?? null ) ) {

				$p_data_map = $this->_initialized_processors[ $p_ajax_action_name ]->get_data_map();
			} else {

				return false;
			}
		}

		// List of known WordPress sanitization functions.
		//
		$_known_sanitization_functions = array(
			'sanitize_text_field',
			'sanitize_email',
			'sanitize_url',
			'intval',
			'absint',
			'floatval',
			'esc_attr',
			'esc_url',
			'wp_kses_post',
		);

		// Recursive validation function.
		//
		$_validate_and_format_data_map = function ( &$map ) use ( &$_validate_and_format_data_map, $_known_sanitization_functions
		) {

			foreach ( $map as $key => $value ) {

				if ( is_array( $value )
					&& isset( $value['sanitize'] ) ) {

					// Validate sanitization function.
					//
					if ( ! function_exists( $value['sanitize'] )
						|| ! in_array( $value['sanitize'], $_known_sanitization_functions, true ) ) {

						return false;
					}

					// Check if it's an array type to apply sanitation recursively.
					//
					if ( ( ( $value['sanitize_as_array'] ?? '' ) === true ) ) {

						continue; // Allow arrays of the same sanitized type.
					}

					// Check for nested fields configuration.
					//
					if ( is_array( $value['fields'] ?? '' ) ) {

						if ( ! $_validate_and_format_data_map( $value['fields'] ) ) {

							// Recursively validate subfields.
							//
							return false;
						}
					}
				} elseif ( is_array( $value ) ) {

					// Recursively validate nested arrays.
					//
					if ( ! $_validate_and_format_data_map( $value ) ) {

						return false;
					}
				} else {

					// Invalid structure.
					//
					return true;
				}
			}
			return true;
		};

		// Validate the data map.
		//
		if ( ! $_validate_and_format_data_map( $p_data_map ) ) {

			return false;
		}

		$_option_name = 'qacp_' . sanitize_key( $p_ajax_action_name );

		// Save the data map to the database.
		//
		return update_option( $_option_name, $p_data_map, false );
	}

	/**
	 * Process the $_POST data based on a registered data map.
	 *
	 * @param string $p_ajax_action_name The name of the AJAX action.
	 * @param mixed  $p_data The data to process. If non provided, will use $_POST.
	 * @return array Processed and sanitized $_POST data.
	 */
	public function process_ajax_post_data( $p_ajax_action_name, $p_data = null ) {

		// Option name using the "qacp" prefix.
		//
		$_option_name = 'qacp_' . sanitize_key( $p_ajax_action_name );

		// Retrieve the data map from the database.
		//
		$_data_map = get_option( $_option_name, array() );

		if ( empty( $_data_map )
			|| ! is_array( $_data_map ) ) {

			return array();
		}

		if ( is_null( $p_data ) ) {

			return false;
		}

		// Recursive function to process and sanitize data.
		//
		$_process_data = function ( $map, $input, $sanitation_function = null ) use ( &$_process_data ) {

			$_output = array();

			foreach ( $map as $key => $map_field_data ) {

				if ( isset( $input[ $key ] ) ) {

					// Check basic array fields exist.
					//
					if ( is_array( $map_field_data ) ) {

						$_max_length = $map_field_data['data_max_length'] ?? false;
						$_array_max_length = $map_field_data['array_max_length'] ?? false;
						$_sanitation_function = $map_field_data['sanitize'] ?? $sanitation_function ?? null;
						$_sanitized_value = null;

						if ( isset( $map_field_data['fields'] )
							&& is_array( $map_field_data['fields'] ) ) {

							// Process nested fields if the `fields` key is present.
							//
							$_sanitized_value = $_process_data(
								$map_field_data['fields'],
								$input[ $key ],
								$_sanitation_function
							);

						} elseif ( ( $map_field_data['sanitize_as_array'] ?? false )
							&& is_array( $input[ $key ] ) ) {

							$_sliced_array = empty( $_array_max_length ) ?
											$input[ $key ] : array_slice( $input[ $key ], 0, $_array_max_length );

							$_array_fields = $map_field_data['array_fields'] ?? array();

							$_sanitized_value = empty( $_sanitation_function ) ?
												$_sliced_array : $this->recursive_array_map( $_sanitation_function, $_sliced_array );

						} else {

							$_sanitized_value = $this->sanitize_single_field(
								$input[ $key ],
								$_sanitation_function,
								$_max_length
							);

						}

						$_output[ $key ] = $_sanitized_value;
					} else {

						// Just a simple field.
						//
						$_sanitized_value = $this->sanitize_single_field(
							$input[ $key ],
							$_sanitation_function,
							$_max_length
						);
						$_output[ $key ] = $_sanitized_value;
					}
				}
			}
			return $_output;
		};

		$_processed_data = $_process_data( $_data_map, $p_data );

		return $_processed_data;
	}
}
