<?php
/**
 * Plugin Name: Quarkcode NeuralCommerce Lite - Profit Analytics & Cost Intelligence Hub
 * Plugin URI: https://quarkcodeai.com/product/quarkcode-neural-commerce/
 * Description: Free WooCommerce profit analytics with AI-powered insights - Track product costs, analyze profits, monitor margins, and optimize your store performance with intelligent reporting.
 * Version: 1.0.8
 * Author: vortexplaytv
 * Author URI: https://quarkcodeai.com/
 * License: GPL-3.0+
 * License URI: http://www.gnu.org/licenses/gpl-3.0.txt
 * Text Domain: quarkcode-neuralcommerce-lite
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * WC requires at least: 8.0
 * WC tested up to: 10.4
 *
 * @package QuarkcodeNeuralCommerce
 */

namespace QuarkcodeNeuralCommerce;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Define plugin constants.
define( 'QCNC_VERSION', '1.0.7' );
define( 'QCNC_PLUGIN_FILE', __FILE__ );
define( 'QCNC_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'QCNC_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'QCNC_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
define( 'QCNC_INCLUDES_DIR', QCNC_PLUGIN_DIR . 'includes/' );
define( 'QCNC_ASSETS_URL', QCNC_PLUGIN_URL . 'assets/' );
define( 'QCNC_TEMPLATES_DIR', QCNC_PLUGIN_DIR . 'templates/' );

// Composer autoloader (if using external dependencies).
if ( file_exists( QCNC_PLUGIN_DIR . 'vendor/autoload.php' ) ) {
    require_once QCNC_PLUGIN_DIR . 'vendor/autoload.php';
}

// Custom autoloader for plugin classes.
spl_autoload_register( function ( $class ) {
    $prefix   = 'QuarkcodeNeuralCommerce\\';
    $base_dir = QCNC_INCLUDES_DIR;

    $len = strlen( $prefix );
    if ( strncmp( $prefix, $class, $len ) !== 0 ) {
        return;
    }

    $relative_class = substr( $class, $len );
    
    // Map namespaces to directories.
    $namespace_map = [
        'Admin\\'        => 'admin/',
        'Core\\'         => 'core/',
        'API\\'          => 'api/',
        'Database\\'     => 'database/',
        'Integrations\\' => 'integrations/',
        'Utilities\\'    => 'utilities/',
        'Cron\\'         => 'cron/',
        'Notifications\\' => 'notifications/',
    ];

    $directory = '';
    foreach ( $namespace_map as $namespace => $dir ) {
        if ( strpos( $relative_class, $namespace ) === 0 ) {
            $directory = $dir;
            $relative_class = substr( $relative_class, strlen( $namespace ) );
            break;
        }
    }

    // Convert class name to filename with proper snake_case handling
    // QCNC_Core -> qcnc-core
    // QCNC_Admin_Ajax -> qcnc-admin-ajax
    $class_name = str_replace( '_', '-', strtolower( $relative_class ) );
    $file = $base_dir . $directory . 'class-' . $class_name . '.php';

    if ( file_exists( $file ) ) {
        require_once $file;
    }
} );


/**
 * Check if WooCommerce is active.
 *
 * @return bool
 */
function qcnc_is_woocommerce_active() {
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
    return in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ), true ) ||
           ( is_multisite() && array_key_exists( 'woocommerce/woocommerce.php', get_site_option( 'active_sitewide_plugins', [] ) ) );
}

/**
 * Display admin notice if WooCommerce is not active.
 */
function qcnc_woocommerce_missing_notice() {
    ?>
    <div class="notice notice-error">
        <p>
            <?php
            echo wp_kses_post(
                sprintf(
                    /* translators: 1: Plugin name, 2: WooCommerce link */
                    __( '<strong>%1$s</strong> requires <strong>%2$s</strong> to be installed and activated.', 'quarkcode-neuralcommerce-lite' ),
                    'Quarkcode NeuralCommerce',
                    '<a href="https://wordpress.org/plugins/woocommerce/" target="_blank">WooCommerce</a>'
                )
            );
            ?>
        </p>
    </div>
    <?php
}

/**
 * Initialize the plugin.
 */
function qcnc_init() {
    if ( ! qcnc_is_woocommerce_active() ) {
        add_action( 'admin_notices', __NAMESPACE__ . '\\qcnc_woocommerce_missing_notice' );
        return;
    }
    // Initialize main plugin class.
    QCNC_Core::instance();
}
add_action( 'plugins_loaded', __NAMESPACE__ . '\\qcnc_init', 10 );

/**
 * Check and run database migrations on admin pages.
 */
function qcnc_check_migrations() {
    // Only run on admin pages and if WooCommerce is active
    if ( ! is_admin() || ! qcnc_is_woocommerce_active() ) {
        return;
    }

    // Check if custom costs migration is needed
    if ( ! get_option( 'qcnc_custom_costs_migrated' ) ) {
        require_once QCNC_INCLUDES_DIR . 'database/class-qcnc-database-schema.php';
        \QuarkcodeNeuralCommerce\Database\QCNC_Database_Schema::migrate_add_custom_costs();
    }
}
add_action( 'admin_init', __NAMESPACE__ . '\\qcnc_check_migrations' );
// END OF MIGRATIONS

/**
 * Plugin activation hook.
 */
function qcnc_activate() {
    if ( ! qcnc_is_woocommerce_active() ) {
        deactivate_plugins( QCNC_PLUGIN_BASENAME );
        wp_die(
            wp_kses_post( __( 'This plugin requires WooCommerce to be installed and activated.', 'quarkcode-neuralcommerce-lite' ) ),
            esc_html__( 'Plugin Activation Error', 'quarkcode-neuralcommerce-lite' ),
            [ 'back_link' => true ]
        );
    }

    require_once QCNC_INCLUDES_DIR . 'class-qcnc-activator.php';
    QCNC_Activator::activate();
}
register_activation_hook( __FILE__, __NAMESPACE__ . '\\qcnc_activate' );

/**
 * Plugin deactivation hook.
 */
function qcnc_deactivate() {
    require_once QCNC_INCLUDES_DIR . 'class-qcnc-deactivator.php';
    QCNC_Deactivator::deactivate();
}
register_deactivation_hook( __FILE__, __NAMESPACE__ . '\\qcnc_deactivate' );

/**
 * Declare HPOS compatibility.
 */
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
} );
